#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout,
                     QCheckBox, QLabel)

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.polish.main import tweak_polish
from calibre.ebooks.oeb.polish.main import CUSTOMIZATION

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/remove_unused_css.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        def label(text):
            la = QLabel(text)
            la.setWordWrap(True), l.addWidget(la), la.setMinimumWidth(450)
            l.addWidget(la)
            return la

        self.la = label(_(
            'This will remove all CSS rules that do not match any actual content.'
            ' There are a couple of additional cleanups you can enable, below:'))
        self.c = c = QCheckBox(_('Remove unused &class attributes'))
        self.c.setChecked(True)

        l.addWidget(c)
        self.la2 = label('<span style="font-size:small; font-style: italic">' + _(
            'Remove all class attributes from the HTML that do not match any existing CSS rules'))
        self.m = m = QCheckBox(_('Merge CSS rules with identical &selectors'))
        self.m.setChecked(True)

        l.addWidget(m)
        self.la3 = label('<span style="font-size:small; font-style: italic">' + _(
            'Merge CSS rules in the same stylesheet that have identical selectors.'
        ' Note that in rare cases merging can result in a change to the effective styling'
        ' of the book, so use with care.'))
        self.p = p = QCheckBox(_('Merge CSS rules with identical &properties'))
        self.p.setChecked(True)

        l.addWidget(p)
        self.la4 = label('<span style="font-size:small; font-style: italic">' + _(
            'Merge CSS rules in the same stylesheet that have identical properties.'
        ' Note that in rare cases merging can result in a change to the effective styling'
        ' of the book, so use with care.'))
        self.u = u = QCheckBox(_('Remove &unreferenced style sheets'))
        self.u.setChecked(True)

        l.addWidget(u)
        self.la5 = label('<span style="font-size:small; font-style: italic">' + _(
            'Remove stylesheets that are not referenced by any content.'
        ))

        l.addStretch(1)

        self.setMinimumSize(300,300)

    def load_settings(self, settings):
        if settings:
            customization = settings['customization']
            self.c.setChecked(customization['remove_unused_classes'])
            self.m.setChecked(customization['merge_identical_selectors'])
            self.p.setChecked(customization['merge_rules_with_identical_properties'])
            self.u.setChecked(customization['remove_unreferenced_sheets'])

    def save_settings(self):
        settings = {}
        settings['customization'] = CUSTOMIZATION.copy()
        settings['customization']['remove_unused_classes'] = self.c.isChecked()
        settings['customization']['merge_identical_selectors'] = self.m.isChecked()
        settings['customization']['merge_rules_with_identical_properties'] = self.p.isChecked()
        settings['customization']['remove_unreferenced_sheets'] = self.u.isChecked()
        return settings

class RemoveUnusedCSS(EditorAction):

    name = 'Remove Unused CSS'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        action = 'remove_unused_css'
        customization = settings.get('customization', None)
        if customization is None:
            return
        report, changed = tweak_polish(chain.current_container, {action:True}, customization=customization)

    def validate(self, settings):
        if settings.get('customization') is None:
            return _('No settings'), _('You must first configure this action')
        return True

    def config_widget(self):
        return ConfigWidget

