#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout,
                     QGroupBox, QLineEdit, QLabel)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/remove_files.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self.user_xpaths = []
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        main_label = QLabel(_('<p>Enter names of files to delete below (Comma separated list)</p>'
        ))
        main_label.setWordWrap(True)
        l.addWidget(main_label)

        self.ledit = QLineEdit()
        self.ledit.setToolTip('comma separated list of file names to remove (regex patterns)')
        l.addWidget(self.ledit)

        l.addStretch(1)

        self.setMinimumSize(300,100)

    def load_settings(self, settings):
        if settings:
            self.ledit.setText(settings['patterns'])

    def save_settings(self):
        settings = {}
        settings['patterns'] = self.ledit.text()
        return settings

class RemoveFiles(EditorAction):

    name = 'Remove Files'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        delete = set()
        patterns = [pat.strip() for pat in settings['patterns'].split(',')]
        for name, media_type in container.mime_map.items():
            for pat in patterns:
                if regex.search(pat, name):
                    delete.add(name)
        for x in delete:
            container.remove_item(x)

    def validate(self, settings):
        patterns = [pat.strip() for pat in settings['patterns'].split(',')]
        if not patterns:
            return _('No settings'), _('You must first configure this action')
        # make sure patterns are valid regexes
        for pat in patterns:
            try:
                regex.search(pat, 'random text')
            except regex._regex_core.error:
                return _('Invalid Regex'), _(f'Pattern ({pat}) is not a valid regex')
        return True

    def config_widget(self):
        return ConfigWidget

