#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QHBoxLayout,
                     QGroupBox, QCheckBox, QLabel)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.polish.split import AbortError, split, in_table
from calibre.ebooks.oeb.base import OEB_DOCS, XPNSMAP

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.common_utils import validate_xpath
from calibre_plugins.editor_chains.scope import ScopeWidget, scope_names, validate_scope, scope_is_headless

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/multisplit.py - exception when loading translations")

def multisplit(container, name, xpath, before=True, find_sibling=True):
    '''
    Split the specified file at multiple locations (all tags that match the specified XPath expression). See also: :func:`split`.
    Splitting automatically migrates all links and references to the affected
    files.

    :param before: If True the splits occur before the identified element otherwise after it.
    :param find_sibling: If True the splits will only occur if preceding_sibling is present.
    '''
    root = container.parsed(name)
    nodes = root.xpath(xpath, namespaces=XPNSMAP)
    if not nodes:
        raise AbortError(_('The expression %s did not match any nodes') % xpath)
    for split_point in nodes:
        if in_table(split_point):
            raise AbortError('Cannot split inside tables')
        if split_point.tag.endswith('}body'):
            raise AbortError('Cannot split on the <body> tag')
        if find_sibling:
            if before:
                sibling = split_point.getprevious()
                x = 'preceding_sibling'
            else:
                sibling = split_point.getnext()
                x = 'following_sibling'
            if sibling is None:
                nodes.remove(split_point)

    for i, tag in enumerate(nodes):
        tag.set('calibre-split-point', str(i))

    current = name
    all_names = [name]
    for i in range(len(nodes)):
        current = split(container, current, '//*[@calibre-split-point="%d"]' % i, before=before)
        all_names.append(current)

    for x in all_names:
        for tag in container.parsed(x).xpath('//*[@calibre-split-point]'):
            tag.attrib.pop('calibre-split-point')
        container.dirty(x)

    return all_names[1:]

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self._init_controls()

    def _init_controls(self):
        from calibre.gui2.convert.xpath_wizard import XPathEdit
        self.l = l = QVBoxLayout(self)
        self.setLayout(l)

        self.la = la = QLabel(_(
            'Specify the locations to split at, using an XPath expression (click'
            ' the wizard button for help with generating XPath expressions).'))
        la.setWordWrap(True)
        l.addWidget(la)

        self._xpath = xp = XPathEdit(self)
        xp.set_msg(_('&XPath expression:'))
        xp.setObjectName('editor-multisplit-xpath-edit')
        l.addWidget(self._xpath)

        self.sibling_chk = QCheckBox(_('Don\'t split if identified element has no preceding/following sibling'))
        self.sibling_chk.setToolTip(_('This option will prevent creating empty pages. Useful in case of re-runs of this action.'))
        self.sibling_chk.setChecked(False)
        l.addWidget(self.sibling_chk)

        self.split_after_chk = QCheckBox(_('Split after identified element (Default is before)'))
        self.split_after_chk.setChecked(False)
        l.addWidget(self.split_after_chk)

        scope_group_box = QGroupBox(_('Choose files to split'))
        l.addWidget(scope_group_box)
        scope_group_box_l = QHBoxLayout()
        scope_group_box.setLayout(scope_group_box_l)
        self.scope_widget = ScopeWidget(self, self.plugin_action,
                hide=['selected-text','styles'], show_tooltip=False,
                headless=self.plugin_action.gui is None)
        scope_group_box_l.addWidget(self.scope_widget)
        l.addWidget(scope_group_box)

        l.addStretch(1)

        self.setMinimumSize(300,400)

    @property
    def xpath(self):
        return self._xpath.xpath

    def load_settings(self, settings):
        if settings:
            self._xpath.edit.setText(settings['xpath'])
            self.split_after_chk.setChecked(settings.get('split_after', False))
            self.sibling_chk.setChecked(settings.get('find_sibling', False))
            self.scope_widget.where = settings.get('where', 'text')

    def save_settings(self):
        settings = {}
        settings['xpath'] = self.xpath
        settings['split_after'] = self.split_after_chk.isChecked()
        settings['find_sibling'] = self.sibling_chk.isChecked()
        settings['where'] = self.scope_widget.where
        return settings

class SplitMultipleLocations(EditorAction):

    name = 'Split At Multiple Locations'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        names = scope_names(chain, settings.get('where', 'text'))
        names_before_split = [name for name in names if name in list(container.manifest_items_of_type(OEB_DOCS))]
        for x in names_before_split:
            for y in ['nav.xhtml', 'titlepage.xhtml']:
                if x.endswith(y):
                    names_before_split.remove(x)
        for name in names_before_split:
            try:
                before = not settings.get('split_after', False)
                find_sibling = settings.get('find_sibling', False)
                changed = multisplit(container, name, settings['xpath'], before=before, find_sibling=find_sibling)
            except AbortError:
                if DEBUG:
                    prints(f'No splitting at: {name}')

        all_names = [name for name, media_type in chain.current_container.mime_map.items() if media_type in OEB_DOCS]
        for x in all_names:
            container.dirty(x)

    def validate(self, settings):
        xpath = settings.get('xpath', '')
        if not xpath:
            return _('No xpath'), _('You must enter xpath expression')
        res = validate_xpath(xpath)
        if res is not True:
            return res
        scope_ok = validate_scope(settings.get('where', 'text'))
        if scope_ok is not True:
            return scope_ok
        return True

    def config_widget(self):
        return ConfigWidget

    def is_headless(self, settings):
        where = settings.get('where', 'text')
        return scope_is_headless(where)
