#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import re

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QGroupBox,
                     QLineEdit, QCheckBox, QLabel, QFrame)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.editor.smarts.html import (edit_block, select_tag, find_closing_tag,
                            find_closest_containing_tag, find_closest_containing_block_tag)
from calibre.ebooks.oeb.base import OEB_DOCS
from calibre.gui2.tweak_book import editor_name

from calibre_plugins.editor_chains.actions.base import EditorAction


def mark_tag(editor, mark_text, block_tag=False):
    if block_tag:
        find_closest = find_closest_containing_block_tag
    else:
        find_closest = find_closest_containing_tag
    editor.highlighter.join()
    cursor = editor.textCursor()
    block, offset = cursor.block(), cursor.positionInBlock()
    tag = find_closest(block, offset)

    if tag is not None:
        if tag.name == 'body':
            ntag = find_closest(block, offset + 1)
            if ntag is not None and ntag.name != 'body':
                tag = ntag
            elif offset > 0:
                ntag = find_closest(block, offset - 1)
                if ntag is not None and ntag.name != 'body':
                    tag = ntag
        closing_tag = find_closing_tag(tag)
        if closing_tag is None:
            if DEBUG:
                prints( _('Editor Chains: '
                'There is an unclosed %s tag. You should run the Fix HTML tool'
                ' before trying to rename tags.') % tag.name,)
        with edit_block(cursor):
            text = select_tag(cursor, tag)
            text = re.sub(r'^(<\s*[a-zA-Z0-9]+)', r'\1 {}'.format(mark_text), text)
            cursor.insertText(text)
    else:
        if DEBUG:
            prints(_('Editor Chains: No suitable tag was found to rename'))


class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        main_label = QLabel(_('<p>Add an attribute to the tag where the editor cursor is currently at. '
                              'This is useful for marking several tags using a shortcut keyboard. '
                              'You can locate these tags later using the attribute you marked them with '
                              'through "Tag Actions", and batch apply whatever modifications you want.</p>'
                              '<p><b>Note:</b> The cursor can be anywhere as long as it is inside the tag.</p>'
        ))
        main_label.setWordWrap(True)
        l.addWidget(main_label)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)

        group_box = QGroupBox(_('Mark with attribute'))
        group_box_l = QVBoxLayout()
        group_box.setLayout(group_box_l)
        l.addWidget(group_box)
        self.ledit = QLineEdit()
        self.ledit.setToolTip(_('Write and attribute to mark the tag with e.g. data-ec="rename"'))
        group_box_l.addWidget(self.ledit)

        self.block_chk = QCheckBox(_('Mark the closest (block) tag'))
        self.block_chk.setChecked(False)
        l.addWidget(self.block_chk)

        l.addStretch(1)

        self.setMinimumSize(500,300)

    def load_settings(self, settings):
        if settings:
            self.ledit.setText(settings['mark_text'])
            self.block_chk.setChecked(settings['block'])


    def save_settings(self):
        settings = {}
        settings['mark_text'] = self.ledit.text()
        settings['block'] = self.block_chk.isChecked()
        return settings

class MarkCursorTag(EditorAction):

    name = 'Mark Cursor Tag'
    _is_builtin_ = True
    headless = False

    def run(self, chain, settings, *args, **kwargs):

        container = chain.current_container
        current_editor = chain.gui.central.current_editor
        if current_editor:
            name = editor_name(current_editor)
            if not name or container.mime_map[name] not in OEB_DOCS:
                if DEBUG:
                    print('Editor Chains: No tag found at cursor')
                    return

            mark_tag(current_editor.editor, settings['mark_text'], settings['block'])
            chain.boss.commit_all_editors_to_container()

    def validate(self, settings):
        if not settings:
            return _('Settings Error'), _('You must configure this action')
        if not settings['mark_text']:
            return _('No attribute'), _('You must enter an attribute to mark the tag with')
        return True

    def config_widget(self):
        return ConfigWidget
