#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QGroupBox,
                     QLineEdit, QCheckBox, QLabel)

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.polish.toc import (get_book_language, lang_as_iso639_1, find_inline_toc,
                                           get_toc, toc_to_html, serialize, set_guide_item, translate)

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/insert_inline_toc.py - exception when loading translations")

def create_inline_toc(container, title=None, spine_index=0):
    '''
    Create an inline (HTML) Table of Contents from an existing NCX Table of Contents.

    :param title: The title for this table of contents.
    :param spine_index: spine index of insert table of contents.
    '''
    lang = get_book_language(container)
    default_title = 'Table of Contents'
    if lang:
        lang = lang_as_iso639_1(lang) or lang
        default_title = translate(lang, default_title)
    title = title or default_title
    toc = get_toc(container)
    if len(toc) == 0:
        return None
    toc_name = find_inline_toc(container)

    name = toc_name
    html = toc_to_html(toc, container, name, title, lang)
    raw = serialize(html, 'text/html')
    if name is None:
        name, c = 'toc.xhtml', 0
        while container.has_name(name):
            c += 1
            name = 'toc%d.xhtml' % c
        container.add_file(name, raw, spine_index=spine_index)
    else:
        with container.open(name, 'wb') as f:
            f.write(raw)
    set_guide_item(container, 'toc', title, name, frag='calibre_generated_inline_toc')
    return name

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        self.insert_at_back_chk = QCheckBox(_('Insert TOC at back of book (default is front)'))
        self.insert_at_back_chk.setChecked(False)
        l.addWidget(self.insert_at_back_chk)

        self.gb = gb = QGroupBox('Change title of inserted TOC')
        gb_l = QVBoxLayout()
        gb.setLayout(gb_l)
        gb.setCheckable(True)
        gb.setChecked(False)
        l.addWidget(gb)
        self.title_ledit = QLineEdit()
        gb_l.addWidget(self.title_ledit)

        l.addStretch(1)

        self.setMinimumSize(350,100)

    def load_settings(self, settings):
        if settings:
            if settings.get('insert_at_back', False):
                self.insert_at_back_chk.setChecked(True)
            title = settings.get('title')
            if title:
                self.gb.setChecked(True)
                self.title_ledit.setText(title)

    def save_settings(self):
        settings = {}
        settings['insert_at_back'] = self.insert_at_back_chk.isChecked()
        if self.gb.isChecked():
            settings['title'] = self.title_ledit.text()
        return settings

class InsertInlineTOC(EditorAction):

    name = 'Insert Inline TOC'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        title = settings.get('title', '').strip()
        if not title: title = None
        if settings.get('insert_at_back', False):
            spine_index = len(list(container.spine_iter))
        else:
            spine_index = 0
        name = create_inline_toc(container, title=title, spine_index=spine_index)
        if name is None:
            report = _(
                'Cannot create an inline Table of Contents as this book has no existing'
                ' Table of Contents. You must first create a Table of Contents using the'
                ' Edit Table of Contents tool.')
        else:
            report = _('Inline TOC inserted succesfully')

    def validate(self, settings):
        return True

    def config_widget(self):
        return ConfigWidget

