#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, re
import tempfile
from collections import defaultdict

from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QHBoxLayout, QGridLayout,
                     QCheckBox, QIcon, QGroupBox, QLabel, QLineEdit, QFrame, QPushButton)

from calibre import prints
from calibre.constants import DEBUG
from css_parser.css import CSSStyleSheet, CSSStyleRule
from calibre.gui2.convert.xpath_wizard import XPathEdit
from calibre.ebooks.oeb.base import OEB_STYLES, OEB_DOCS, XPNSMAP, etree, barename
from calibre.ebooks.oeb.polish.css import merge_identical_properties
from calibre.ebooks.oeb.polish.container import clone_container
from calibre.ebooks.oeb.polish.utils import link_stylesheets

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.common_utils import safe_name, validate_xpath
from calibre_plugins.editor_chains.css import get_style_maps

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/inline_styles.py - exception when loading translations")

def effective_style(style_map, elem, names):
    d = {}
    for name in sorted(names):
        val = style_map.get(elem, {}).get(name)
        if val:
            val = val.cssText
        d[name] = val
    return d

def append_to_class(tag, value):
    cls_val = tag.attrib.get('class', '').split()
    if value not in cls_val:
        cls_val.append(value)
    cls_val = ' '.join(cls_val)
    tag.attrib['class'] = cls_val

def style_hash(style):
    res = ''
    for name in sorted(style.keys()):
        res += style.getProperty(name).cssText + ';'
    return res.rstrip(';')

def rule_hash(rule):
    res = ''
    res += rule.selectorText + ' {'
    res += style_hash(rule.style)
    res += '}'
    return res

def all_class_names(container):
    klasses = set()
    for name, media_type in container.mime_map.items():
        if media_type in OEB_DOCS:
            root = container.parsed(name)
            res = root.xpath('//*/@class')
            for x in res:
                for cls in x.split():
                    klasses.add(cls)
    return klasses

def remove_class(elem, cls):
    classes = elem.attrib.get('class', '').split()
    for klass in classes:
        if klass == cls:
            classes.remove(klass)
    elem.attrib['class'] = ' '.join(classes)

class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self._init_controls()

    def _init_controls(self):

        l = QVBoxLayout()
        self.setLayout(l)

        label_1 = QLabel(_(
            '<b>Note:</b> This action will check all inline styles, '
            'to determine whether moving the styles to CSS will affect '
            'the formatting of the elements.\n'
            'This is done on best effort basis. If the action determines '
            'that some of the styles cannot be moved, it will halt the '
            'conversion of all other styles, unless you choose otherwise '
            'in the settings below.'
        ))
        label_1.setWordWrap(True)
        l.addWidget(label_1)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)

        self.allow_partial_chk = QCheckBox(_('Allow partial conversion of styles'))
        self.allow_partial_chk.setChecked(False)
        l.addWidget(self.allow_partial_chk)

        gl = QGridLayout()
        l.addLayout(gl)
        name_l = QLabel(_('Stylesheet name'))
        self.name_ledit = QLineEdit()
        self.name_ledit.setToolTip(_('Name of the stylesheet that will be created'))
        base_l = QLabel(_('Class base name'))
        self.base_ledit = QLineEdit()
        self.base_ledit.setToolTip(_(
            'The base name for the classes that will created for the '
            'migrated styles.\n'
            'The same name will be used for classes plus a number suffix.'
        ))
        gl.addWidget(name_l, 0, 0, 1, 1)
        gl.addWidget(self.name_ledit, 0, 1, 1, 1)
        gl.addWidget(base_l, 1, 0, 1, 1)
        gl.addWidget(self.base_ledit, 1, 1, 1, 1)

        gb = QGroupBox(_('Use XPath to exclude element styles from conversion'))
        gb_l = QVBoxLayout()
        gb.setLayout(gb_l)
        l.addWidget(gb)
        self._xpath = xp = XPathEdit(self)
        xp.set_msg(_('&XPath expression:'))
        xp.setObjectName('ec-inlinestyles-xpath-edit')
        gb_l.addWidget(self._xpath)

        l.addStretch(1)

        self.setMinimumSize(300,400)

    def load_settings(self, settings):
        if settings:
            self.allow_partial_chk.setChecked(settings['allow_partial'])
            self.name_ledit.setText(settings['stylesheet_name'])
            self.base_ledit.setText(settings['base_name'])
            self._xpath.edit.setText(settings.get('xpath_to_exclude', ''))

    def save_settings(self):
        settings = {}
        settings['allow_partial'] = self.allow_partial_chk.isChecked()
        settings['stylesheet_name'] = self.name_ledit.text()
        settings['base_name'] = self.base_ledit.text()
        xpath = self._xpath.edit.text().strip()
        if xpath:
            settings['xpath_to_exclude'] = xpath
        return settings

class InlineStyles(EditorAction):

    name = 'Inline Styles To CSS'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        xpath_to_exclude = settings.get('xpath_to_exclude')
        sheet_name = settings.get('stylesheet_name', 'inline.css')
        if not sheet_name.lower().endswith('.css'):
            sheet_name += '.css'
        base_name = settings.get('base_name', 'inline')
        allow_partial = settings.get('allow_partial', False)
        style_map, pseudo_style_map = get_style_maps(container)
        dirty = set()
        cls_elems_map = defaultdict(list)
        cls_rule_map = {}
        hash_lookup = {}
        elem_tuples = []
        all_klasses = all_class_names(container)
        # add the base_name so that new classes will start with number 1 appended to class
        all_klasses.add(base_name)
        sheet = CSSStyleSheet()

        # First pass:
        # (1) find elements with style attribute
        # (2) calculate and store effective style values for style properties (effective_pre)
        # (3) convert inline styles to css
        for name, media_type in container.mime_map.items():
            if media_type in OEB_DOCS:
                root = container.parsed(name)
                tree = etree.ElementTree(root)
                # Exclude style elements in svg or math elements
                exclude = root.xpath('//*[@style and local-name()="svg"]') + root.xpath('//*[@style and local-name()="math"]')
                if xpath_to_exclude:
                    exclude += root.xpath(xpath_to_exclude, namespaces=XPNSMAP)
                elems = root.xpath('//*[@style]')
                elems = [x for x in elems if x not in exclude]
                if len(elems):
                    dirty.add(name)
                    for elem in elems:
                        elem_path = tree.getelementpath(elem)
                        #tag_name = barename(elem.tag)
                        inline_style = container.parse_css(elem.get('style'), is_declaration=True)
                        hash_ = style_hash(inline_style)
                        cls = hash_lookup.get(hash_)
                        effective_pre = effective_style(style_map, elem, inline_style.keys())
                        if cls is None:
                            cls = safe_name(base_name, all_klasses)
                            all_klasses.add(cls)
                            rule = CSSStyleRule()
                            rule.cssText = f'.{cls} {{{inline_style.cssText}}}'
                            sheet.insertRule(rule)
                            hash_lookup[hash_] = cls
                            cls_rule_map[cls] = rule
                        append_to_class(elem, cls)
                        elem_tuples.append((name, elem_path, elem, cls, inline_style, rule, effective_pre))
                        cls_elems_map[cls].append(elem)

        sheet_name = container.make_name_unique(sheet_name)
        new_sheet = container.add_file(sheet_name, sheet.cssText)
        link_stylesheets(container, list(dirty), [new_sheet])

        # Second pass:
        # (1) calculate effective styles after converting inline styles to css (effective_post)
        # (2) compare effective_post to effective_pre for each element
        # (3) if comparison yields unmatched values, roll back depending on the settings
        # (4) otherwise, keep on and remove inline styles
        convert_ok = []
        convert_no = []
        style_map, pseudo_style_map = get_style_maps(container, include_inline_styles=False)
        for name, elem_path, elem, cls, inline_style, rule, effective_pre in elem_tuples:
            effective_post = effective_style(style_map, elem, inline_style.keys())
            if effective_post != effective_pre:
                if DEBUG:
                    prints(f'Editor Chains: Inline CSS: Moving styles changed properties for elem > {etree.tostring(elem)}')
                convert_no.append((name, elem_path, elem, cls, rule))
            else:
                convert_ok.append((name, elem_path, elem, cls, rule))

        if allow_partial or not convert_no:
            for name, elem_path, elem, cls, rule in convert_ok:
                del elem.attrib['style']
        else:
            remove_class(elem, cls)

        if convert_no:
            for name, elem_path, elem, cls, rule in convert_no:
                remove_class(elem, cls)

            if allow_partial:
                for cls, elements in cls_elems_map.items():
                    remove_rule = True
                    for element in elements:
                        if element in [elem for name, elem_path, elem, cls, rule in convert_ok]:
                            # cls is used in a rule, keep corresponding css rule
                            remove_rule = False
                            break
                    if remove_rule:
                        rule = cls_rule_map[cls]
                        if DEBUG:
                            prints(f'Editor Chains: Inline CSS: Removing rule (sticking with inline) > {rule.cssText}')
                        sheet.deleteRule(rule)

        if convert_ok and (allow_partial or not convert_no):
            container.open(new_sheet, 'wb').write(sheet.cssText)
        else:
            container.remove_item(new_sheet)

        for name in dirty:
            container.dirty(name)

    def validate(self, settings):
        if not settings:
            return _('Settings Error'), _('You must configure this action')
        xpath = settings.get('xpath_to_exclude', '')
        if xpath:
            res = validate_xpath(xpath)
            if res is not True:
                return res
        if not settings.get('stylesheet_name'):
            return _('No Stylesheet'), _('You must specify a stylesheet name')
        if not settings.get('stylesheet_name'):
            return _('No class name'), _('You must specify a class name')
        return True

    def config_widget(self):
        return ConfigWidget



