#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.html_transform_rules import unwrap_tag
from calibre.ebooks.oeb.polish.pretty import pretty_xml_tree
from calibre.ebooks.oeb.base import barename, OEB_DOCS

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/flatten_toc.py - exception when loading translations")

def flatten_toc(container):
    ncx_changed = False
    nav_changed = False

    # epub v2.0
    name = 'toc.ncx'
    for name, media_type in container.mime_map.items():
        if name.endswith('toc.ncx'):
            root = container.parsed(name)
            for node in root.xpath(f'//*[local-name()="navPoint"]'):
                subnodes = node.xpath(f'//*[local-name()="navPoint"]')
                if len(subnodes):
                    ncx_changed = True
                    parent = node.getparent()
                    idx = parent.index(node) + 1
                    for subnode in reversed(node):
                        if barename(subnode.tag) == "navPoint":
                            parent.insert(idx, subnode)
            if ncx_changed:
                pretty_xml_tree(root)
                container.dirty(name)

        # epub v3.0
        if name.endswith('nav.xhtml'):
            root = container.parsed(name)
            for node in root.xpath(f'//*[local-name()="li"]'):
                subnodes = node.xpath(f'//*[local-name()="ol"]')
                if len(subnodes):
                    nav_changed = True
                    parent = node.getparent()
                    idx = parent.index(node) + 1
                    for subnode in reversed(node):
                        if barename(subnode.tag) == "ol":
                            parent.insert(idx, subnode)
                            unwrap_tag(subnode)
            if nav_changed:
                pretty_xml_tree(root)
                container.dirty(name)

    return ncx_changed or nav_changed

class FlattenTOC(EditorAction):

    name = 'Flatten TOC'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        changed = flatten_toc(chain.current_container)

    def validate(self, settings):
        return True
