#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2024, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import traceback

from calibre.utils.localization import ngettext
from polyglot.builtins import iteritems
from calibre.ebooks.oeb.polish.download import (
    download_external_resources, get_external_resources, replace_resources,
)

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/external_resources.py - exception when loading translations")


class ExternalResources(EditorAction):

    name = 'Download External Resources'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        try:
            resources = get_external_resources(container)
        except Exception as err:
            print(err, traceback.format_exc())
            return
        if not resources:
            print(_('No external resources were found in this book.'))
            return
        try:
            downloaded = download_external_resources(container, resources)
        except Exception as err:
            print(_('Failed to download external resources.'))
            print(err, traceback.format_exc())
            return

        replacements, failures = downloaded
        if failures:
            tb = [f'{url}\n\t{err}\n' for url, err in iteritems(failures)]
            det_msg='\n'.join(tb)
            if not replacements:
                print(_(
                    'Failed to download external resources.'), det_msg)
                return
            else:
                print(_(
                    'Warning: Failed to download some external resources.'), det_msg)

        #t = ngettext(
            #'Successfully processed the external resource', 'Successfully processed {} external resources', len(replacements)).format(len(replacements))
        #if failures:
            #t += '<br>' + ngettext('Could not download one image', 'Could not download {} images', len(failures)).format(len(failures))
        #self.success.setText('<p style="text-align:center">' + t)

        try:
            ret = replace_resources(container, resources, replacements)
        except Exception as err:
            print(_('Failed to replace external resources'))
            print(err, traceback.format_exc())
            return

    def validate(self, settings):
        return True
