#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QGroupBox,
                     QComboBox, QLabel, QFrame)

from calibre import prints
from calibre.constants import DEBUG
from calibre.customize.ui import output_profiles
from calibre.ebooks.oeb.normalize_css import normalizers
from calibre.ebooks.oeb.base import OEB_DOCS, OEB_STYLES, XPNSMAP, serialize
from calibre.ebooks.oeb.stylizer import FONT_SIZE_NAMES
from css_parser.css import CSSStyleRule, CSSPageRule, CSSPageRule
from calibre import force_unicode

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/expand_styles.py - exception when loading translations")

def list_profiles():
    for x in output_profiles():
        yield x.short_name

def list_profiles():
    l = []
    for x in output_profiles():
        l.append(x.short_name)
    return l

def get_profile(name, default='default'):
    for x in output_profiles():
        if x.short_name == name:
            return x
    for x in output_profiles():
        if x.short_name == default:
            return x

def profile_name_from_short_name(short_name):
    for x in output_profiles():
        if x.short_name == short_name:
            return x.name

def profile_short_name_from_name(description):
    for x in output_profiles():
        if x.name == description:
            return x.short_name

def expand_style(style, expand_fontsize=False, profile_name='default'):
    changed = False
    delete_prop = set()
    for prop in style:
        name = prop.name
        normalizer = normalizers.get(name, None)
        #is_important = prop.priority == 'important'
        if normalizer is not None:
            changed = True
            delete_prop.add(name)
            for name, val in normalizer(name, prop.propertyValue).items():
                style.setProperty(name, val, priority=prop.priority)
    for name in delete_prop:
        style.removeProperty(name)
    size_changed = False
    if expand_fontsize:
        size_changed = normalize_fontsize(style, profile_name=profile_name)
    changed = changed or size_changed
    return style, changed

def expand_stylesheet(stylesheet, expand_fontsize=False, profile_name='default'):
    sheet_changed = False
    for rule in stylesheet:
        if isinstance(rule, CSSStyleRule) or isinstance(rule, CSSPageRule):
            style, changed = expand_style(rule.style,
                    expand_fontsize=expand_fontsize, profile_name=profile_name)
            if changed:
                sheet_changed = True
    return stylesheet, sheet_changed

def normalize_fontsize(style, profile_name='default'):
    changed = False
    profile = get_profile(profile_name)
    size = style['font-size']
    if 'font-size' in style:
        if size == 'normal':
            size = 'medium'
        if size == 'smallest':
            size = 'xx-small'
        if size in FONT_SIZE_NAMES:
            style['font-size'] = f"{profile.fnames[size] / float(profile.fbase):.1f}rem"
    #if '-epub-writing-mode' in style:
        #for x in ('-webkit-writing-mode', 'writing-mode'):
            #style[x] = style.get(x, style['-epub-writing-mode'])
    changed = size != style['font-size']
    return changed

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        main_label = QLabel(_('<p>This action expands the shorthand form of various CSS properties like margin, padding, border ....etc</p>'
                              '<p>It operates on CSS sheet, sytle tags and inline styles.</p>'
        ))
        main_label.setWordWrap(True)
        l.addWidget(main_label)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)

        self.fontsize_gb = fontsize_gb = QGroupBox('Normalize Font Size')
        self.fontsize_gb.setToolTip(_(
            'This will convert shorthand values like small, normal ... etc to numeric values.\n'
            'The conversion result might vary depending on the reader profile selected below.'
        ))
        fontsize_gb_l = QVBoxLayout()
        fontsize_gb.setLayout(fontsize_gb_l)
        fontsize_gb.setCheckable(True)
        fontsize_gb.setChecked(False)
        l.addWidget(fontsize_gb)
        self.profile_combo = QComboBox()
        profile_descriptions = [ profile_name_from_short_name(desc) for desc in list_profiles()]
        self.profile_combo.addItems(profile_descriptions)
        default = profile_name_from_short_name('default')
        if default:
            self.profile_combo.setCurrentText('default')
        fontsize_gb_l.addWidget(self.profile_combo)

        l.addStretch(1)

        self.setMinimumSize(350,250)

    def load_settings(self, settings):
        if settings:
            if settings.get('expand_fontsize', False):
                self.fontsize_gb.setChecked(True)
                profile = settings.get('profile', '')
                description = profile_name_from_short_name(profile)
                if profile:
                    self.profile_combo.setCurrentText(description)

    def save_settings(self):
        settings = {}
        if self.fontsize_gb.isChecked():
            settings['expand_fontsize'] = True
            description = self.profile_combo.currentText()
            settings['profile'] = profile_short_name_from_name(description)
        return settings

class ExpandStyles(EditorAction):

    name = 'Expand Styles'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        expand_fontsize = settings.get('expand_fontsize', False)
        profile_name = settings.get('profile_name', 'default')
        for name, media_type in container.mime_map.items():
            if media_type in OEB_STYLES:
                # A stylesheet. Parsed stylesheets are css_parser CSSStylesheet
                # objects.
                sheet, changed = expand_stylesheet(container.parsed(name),
                            expand_fontsize=expand_fontsize, profile_name=profile_name)
                if changed:
                    container.dirty(name)  # Tell the container that we have changed the stylesheet
            elif media_type in OEB_DOCS:
                # A HTML file. Parsed HTML files are lxml elements
                root = container.parsed(name)
                style_tags = root.xpath('//h:style', namespaces=XPNSMAP)
                for style_tag in style_tags:
                    if style_tag.text and style_tag.get('type', None) in {None, 'text/css'}:
                        # We have an inline CSS <style> tag, parse it into a
                        # stylesheet object
                        sheet = container.parse_css(style_tag.text)
                        sheet, changed = expand_stylesheet(sheet,
                                expand_fontsize=expand_fontsize, profile_name=profile_name)
                        if changed:
                            #style_tag.text = serialize(sheet, 'text/css', pretty_print=True)
                            #ValueError: All strings must be XML compatible: Unicode or ASCII, no NULL bytes or control characters
                            style_tag.text = sheet.cssText.decode(encoding='utf-8')
                            container.dirty(name)  # Tell the container that we have changed the stylesheet
                for elem in root.xpath('//*[@style]'):
                    # Process inline style attributes
                    block = container.parse_css(elem.get('style'), is_declaration=True)
                    style, changed = expand_style(block,
                            expand_fontsize=expand_fontsize, profile_name=profile_name)
                    if changed:
                        elem.set('style', force_unicode(block.getCssText(separator=' '), 'utf-8'))
                        container.dirty(name)

    def validate(self, settings):
        if settings.get('expand_fontsize', False):
            profile = settings.get('profile', '')
            if profile not in list_profiles():
                return _('Missing Profile'), _(f'Profile ({profile}) is not available')
        return True

    def config_widget(self):
        return ConfigWidget

