#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QHBoxLayout, QRadioButton,
                     QGroupBox, QPushButton)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2.toc.main import XPathDialog
from calibre.ebooks.oeb.polish.toc import from_xpaths, from_links, from_files, commit_toc
from calibre.utils.config import JSONConfig

from calibre_plugins.editor_chains.actions.base import EditorAction

te_prefs = JSONConfig('toc-editor')

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/create_toc.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self.user_xpaths = []
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        self.opts_box = QGroupBox(_('Options:'))
        self.opts_box_l = opts_box_l = QVBoxLayout()
        self.opts_box.setLayout(self.opts_box_l)
        l.addWidget(self.opts_box)

        self.major_headings_opt = QRadioButton(_('Create TOC From Major Headings'))
        self.major_headings_opt.setChecked(True)
        opts_box_l.addWidget(self.major_headings_opt)

        self.all_headings_opt = QRadioButton(_('Create TOC From All Headings'))
        opts_box_l.addWidget(self.all_headings_opt)

        self.files_opt = QRadioButton(_('Create TOC From Files'))
        opts_box_l.addWidget(self.files_opt)

        self.links_opt = QRadioButton(_('Create TOC From Links'))
        opts_box_l.addWidget(self.links_opt)

        user_xpaths_layout = QHBoxLayout()
        self.user_xpaths_opt = QRadioButton(_('Create TOC From User Xpaths'))
        self.user_xpaths_opt.setEnabled(False)
        self.user_xpaths_button = QPushButton(_('Add user xpaths'))
        self.user_xpaths_button.clicked.connect(self._on_user_xpaths_opt_chosen)
        user_xpaths_layout.addWidget(self.user_xpaths_opt, 1)
        user_xpaths_layout.addWidget(self.user_xpaths_button)
        opts_box_l.addLayout(user_xpaths_layout)

        l.addStretch(1)

        self.setMinimumSize(300,300)

    def _on_user_xpaths_opt_chosen(self):
        d = XPathDialog(self, te_prefs)
        for idx, w in enumerate(d.widgets):
            try:
                w.edit.setText(self.user_xpaths[idx])
            except IndexError:
                pass
        if d.exec_() == d.Accepted and d.xpaths:
            self.user_xpaths = d.xpaths
            self.user_xpaths_opt.setEnabled(True)
            self.user_xpaths_opt.setChecked(True)
            self.user_xpaths_button.setText(_('Edit user xpaths'))

    def load_settings(self, settings):
        if settings:
            toc_src = settings['toc_src']
            if toc_src == 'from_major_headers':
                self.major_headings_opt.setChecked(True)
            elif toc_src == 'from_all_headers':
                self.all_headings_opt.setChecked(True)
            elif toc_src == 'from_files':
                self.files_opt.setChecked(True)
            elif toc_src == 'from_links':
                self.links_opt.setChecked(True)
            elif toc_src == 'from_user_xpaths':
                self.user_xpaths_opt.setEnabled(True)
                self.user_xpaths_opt.setChecked(True)
                self.user_xpaths_button.setText(_('Edit User Xpaths'))
                self.user_xpaths = settings['user_xpaths']

    def save_settings(self):
        settings = {}
        if self.major_headings_opt.isChecked():
            settings['toc_src'] = 'from_major_headers'
        if self.all_headings_opt.isChecked():
            settings['toc_src'] = 'from_all_headers'
        if self.files_opt.isChecked():
            settings['toc_src'] = 'from_files'
        if self.links_opt.isChecked():
            settings['toc_src'] = 'from_links'
        if self.user_xpaths_opt.isChecked():
            settings['toc_src'] = 'from_user_xpaths'
            settings['user_xpaths'] = self.user_xpaths
        return settings

class CreateTOC(EditorAction):

    name = 'Create TOC'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        toc_src = settings['toc_src']
        if toc_src == 'from_files':
            toc = from_files(container)
        elif toc_src == 'from_links':
            toc = from_links(container)
        elif toc_src == 'from_major_headers':
            toc = from_xpaths(container, ['//h:h1', '//h:h2', '//h:h3'])
        elif toc_src == 'from_all_headers':
            toc = from_xpaths(container, ['//h:h1', '//h:h2', '//h:h3','//h:h4', '//h:h5', '//h:h6'])
        elif toc_src == 'from_user_xpaths':
            toc = from_xpaths(container, settings['user_xpaths'])
        commit_toc(container, toc)

    def validate(self, settings):
        if not settings:
            return _('No settings'), _('You must first configure this action')
        if settings['toc_src'] == 'from_user_xpaths':
            if not settings['user_xpaths']:
                return _('Xpaths missing'), _('You must first configure user xpaths')
        return True

    def config_widget(self):
        return ConfigWidget
