#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QHBoxLayout,
                     QCheckBox, QLabel, QSpinBox)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/compress_images.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        self.la = la = QLabel(_(
            'You can compress the images in this book losslessly, reducing the file size of the book,'
            ' without affecting image quality. Typically image size is reduced by 5 - 15%.'))
        la.setWordWrap(True)
        la.setMinimumWidth(250)
        l.addWidget(la), l.addSpacing(30)

        self.enable_lossy = el = QCheckBox(_('Enable &lossy compression of JPEG images'))
        el.setToolTip(_('This allows you to change the quality factor used for JPEG images.\nBy lowering'
                        ' the quality you can greatly reduce file size, at the expense of the image looking blurred.'))
        l.addWidget(el)
        self.h2 = h = QHBoxLayout()
        l.addLayout(h)
        self.jq = jq = QSpinBox(self)
        jq.setMinimum(0), jq.setMaximum(100), jq.setValue(80), jq.setEnabled(False)
        jq.setToolTip(_('The compression quality, 1 is high compression, 100 is low compression.\nImage'
                        ' quality is inversely correlated with compression quality.'))
        el.toggled.connect(jq.setEnabled)
        self.jql = la = QLabel(_('Compression &quality:'))
        la.setBuddy(jq)
        h.addWidget(la), h.addWidget(jq)

        l.addStretch(1)

        self.setMinimumSize(300,200)

    def load_settings(self, settings):
        if settings:
            self.enable_lossy.setChecked(settings['enable_lossy'])
            if settings['enable_lossy']:
                self.jq.setValue(settings['value'])


    def save_settings(self):
        settings = {}
        settings['enable_lossy'] = self.enable_lossy.isChecked()
        settings['value'] = self.jq.value()
        return settings

class CompressImages(EditorAction):

    name = 'Compress Images'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        if settings.get('enable_lossy', False):
            jpeg_quality = settings['value']
        else:
            jpeg_quality = None
        if chain.gui:
            from calibre.gui2.tweak_book.polish import CompressImagesProgress
            d = CompressImagesProgress(jpeg_quality=jpeg_quality, parent=chain.gui)
            if d.exec() != d.Accepted:
                pass
        else:
            from calibre.ebooks.oeb.polish.images import compress_images
            res = compress_images(container, jpeg_quality=jpeg_quality)

    def validate(self, settings):
        return True

    def config_widget(self):
        return ConfigWidget

