#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import re, io, sys
import inspect

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit, QPushButton, QDialog, QSize,
                     QFontMetrics)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.gui2.tweak_book.widgets import Dialog
from calibre.gui2.complete2 import EditWithComplete
from calibre.gui2.tweak_book.editor.text import TextEdit

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.modules import compile_code


try:
    load_translations()
except NameError:
    prints("ActionChains::actions/actions/code.py - exception when loading translations")

CODE_EDITOR_TEMPLATE = '''
def run(chain, settings):
    # Enter you code here
    pass
'''

class CodeEditor(Dialog):

    def __init__(self, parent, plugin_action, action, name, title):
        self.template = CODE_EDITOR_TEMPLATE
        self.language = 'python'
        self.show_name = False
        self.action = action
        Dialog.__init__(self, name, title, parent=parent)

    def setup_ui(self):
        self.l = l = QVBoxLayout(self)
        self.h = h = QHBoxLayout()
        l.addLayout(h)

        self.la1 = la = QLabel(_('M&odule name:'))
        h.addWidget(la)
        self.fb = fb = EditWithComplete(self)
        la.setBuddy(fb)
        h.addWidget(fb, stretch=10)
        if not self.show_name:
            self.la1.hide()
            self.fb.hide()

        self.la3 = la = QLabel(_('&Code:'))
        self.source_code = TextEdit(self)
        self.source_code.load_text(self.template, self.language)
        la.setBuddy(self.source_code)
        l.addWidget(la), l.addWidget(self.source_code)

        self.la2 = la = QLabel()
        la.setOpenExternalLinks(True)
        l.addWidget(la)

        l.addWidget(self.bb)

    def sizeHint(self):
        fm = QFontMetrics(self.font())
        return QSize(fm.averageCharWidth() * 120, 600)

    @property
    def mod_name(self):
        return self.fb.text().strip()

    @property
    def source(self):
        return self.source_code.toPlainText()

    def load_settings(self, settings):
        if settings:
            self.source_code.setPlainText(settings['source'] or CODE_EDITOR_TEMPLATE)

    def save_settings(self):
        settings = {}
        settings['source'] = self.source
        return settings

    def accept(self):
        source = self.source
        try:
            mod = compile_code(source, self.mod_name)
        except Exception as err:
            return error_dialog(self, _('Invalid Python code'), _(
                'The code you created is not valid Python code, with error: %s') % err, show=True)

        try:
            func = mod['run']
        except Exception as err:
            return error_dialog(self, _('Function not defined'), _(
                'You must define a functino called "run"') % err, show=True)
        self.settings = self.save_settings()
        # validate settings
        is_valid = self.action.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(
                self,
                msg,
                details,
                show=True
            )
            return
        Dialog.accept(self)

class CodeAction(EditorAction):

    name = 'Run Python Code'
    _is_builtin = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        source = settings['source']
        mod = compile_code(source, 'module')
        run = mod.get('run')
        return run(chain, settings)

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action'))
        source = settings['source']
        mod = compile_code(source, 'module')
        validate_func = mod.get('validate', None)
        if validate_func:
            return validate_func(settings)
        return True

    def config_widget(self):
        return CodeEditor

    def is_headless(self, settings):
        source = settings['source']
        mod = compile_code(source, 'module')
        run_in_ac = mod.get('is_headless', True)
        return run_in_ac

    def supported_formats(self, settings):
        source = settings['source']
        mod = compile_code(source, 'module')
        fmts = mod.get('supported_formats', ['epub','azw3'])
        return fmts
