#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QGroupBox, QComboBox, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.chains import Chain
import calibre_plugins.editor_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/chain_caller.py - exception when loading translations")

def validate_stack(chain_name, stack=[]):
    '''
    Retrun true only after making sure all called chains are valid, and no
    infinite recursion is happening
    '''
    if chain_name in stack:
        return _('Recursive chain calls'), _(f'When calling chains circualr recursion found in this stack: {stack + [chain_name]}')
    stack.append(chain_name)
    all_chain_names = [chain_config['menuText'] for chain_config in cfg.get_chains_config() if chain_config['menuText']]
    chain_config = cfg.get_chain_config(chain_name)
    chain_settings = chain_config.get('chain_settings', {})
    chain_links = chain_settings.get('chain_links',[])
    if len(chain_links) == 0:
        return True
    for chain_link in chain_links:
        action_name = chain_link['action_name']
        action_settings = chain_link.get('action_settings')
        if action_name == 'Chain Caller':
            target_chain_name = action_settings.get('chain_name')
            if not target_chain_name in all_chain_names:
                return _('Chain unavailable'), _(f'Chain ({target_chain_name}) is not available. stack({stack})')
            is_stack_valid = validate_stack(target_chain_name, stack)
            if is_stack_valid is not True:
                return is_stack_valid
    popped = stack.pop(-1)
    return True

def validate(settings, chains_config=[], parent_chain_name=None):
    if not chains_config:
        chains_config = cfg.get_chains_config()
    if not settings:
        return (_('Settings Error'), _('You must configure this action before running it'))
    target_chain_name = settings['chain_name']
    all_chain_names = [chain_config['menuText'] for chain_config in chains_config if chain_config['menuText']]
    if target_chain_name not in all_chain_names:
        return (_('Cahin Error'), _(f'Called chain ({target_chain_name}) is not currently available.'))
    # Validate the chain to make sure no infinite recursion occur
    stack = []
    if parent_chain_name: stack = [parent_chain_name]
    is_stack_valid = validate_stack(target_chain_name, stack=stack)
    if is_stack_valid is not True:
        return is_stack_valid
    return True

def run_in_ac(chain_name, plugin_action):
    '''
    Retrun true only after making sure all called chains can run in action chains
    '''
    all_chain_names = [chain_config['menuText'] for chain_config in cfg.get_chains_config() if chain_config['menuText']]
    chain_config = cfg.get_chain_config(chain_name)
    chain_settings = chain_config.get('chain_settings', {})
    chain_links = chain_settings.get('chain_links',[])
    for chain_link in chain_links:
        action_name = chain_link['action_name']
        action_settings = chain_link.get('action_settings')
        action = plugin_action.actions.get(action_name)
        if not action:
            return False
        if action_name == 'Chain Caller':
            target_chain_name = action_settings.get('chain_name')
            if not target_chain_name in all_chain_names:
                return False
            res = run_in_ac(target_chain_name, plugin_action)
            if not res:
                return False
        else:
            res = action.is_headless(action_settings)
            if not res:
                return False
    return True

class ChainCallerConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.chain_name = chain_name
        self.chains_config = chains_config
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)
        
        chain_groupbox = QGroupBox(_('Select chain'))
        l.addWidget(chain_groupbox)
        chain_groupbox_l = QVBoxLayout()
        chain_groupbox.setLayout(chain_groupbox_l)
        self.chain_combo = QComboBox()
        chain_groupbox_l.addWidget(self.chain_combo)
        
        #all_chain_names = [chain_config['menuText'] for chain_config in cfg.get_chains_config()]
        all_chain_names = [chain_config['menuText'] for chain_config in self.chains_config if chain_config['menuText']]
        try:
            all_chain_names.remove(self.chain_name)
        except:
            pass
        self.chain_combo.addItems(all_chain_names)
        self.chain_combo.setCurrentIndex(-1)
        
        l.addStretch(1)

        self.setMinimumSize(300,300)

    def load_settings(self, settings):
        if settings:
            idx = self.chain_combo.findText(settings['chain_name'])
            self.chain_combo.setCurrentIndex(idx)

    def save_settings(self):
        settings = {}
        settings['chain_name'] = self.chain_combo.currentText()
        return settings

    def validate(self, settings):
        return validate(settings, chains_config=self.chains_config, parent_chain_name=self.chain_name)

class ChainCaller(EditorAction):

    name = 'Chain Caller'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        target_chain_config = cfg.get_chain_config(settings['chain_name'])
        chain_vars = copy.deepcopy(chain.chain_vars)
        target_chain = Chain(self.plugin_action, target_chain_config, chain.tool,
                             gui=chain.gui, boss=chain.boss, add_savepoint=False,
                             book_path=chain.book_path, container=chain.current_container,
                             chain_vars=chain_vars, is_chain_caller=True,
                             report=chain.report)
        target_chain.run()
        del target_chain

    def validate(self, settings):
        return validate(settings, chains_config=[])

    def config_widget(self):
        return ChainCallerConfigWidget

    def is_headless(self, settings):
        return run_in_ac(settings['chain_name'], self.plugin_action)
