#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

class EditorAction(object):

    # Actions must have a unique name attribute.
    name = 'Editor Action'
    headless = False

    '''
    This is the base class for all actions
    '''
    def __init__(self, plugin_action):
        '''
        All actions are intialized at startup
        The are re-initialized on library change, and on adding or modifying custom modules
        '''
        self.plugin_action = plugin_action

    def run(self, chain, settings, *args, **kwargs):
        '''
        This is the method that contain the logic of the action you want.
        It is called whenever a chain is activated.
        The settings is a dictionary with options configured for the specific
        action in the chain dialog.
        chain is the instance running the chain actions, it has info
        used by some actions (e.g. start time for each chain action)
        You can obtain current container by using: chain.current_container
        To access the boss object use: chain.boss
        To access gui object use: chain.gui
        Note: chain.gui & chain.boss are set to None if the plugin is
        run in batch mode from Action Chains.
        '''
        raise NotImplementedError

    def config_widget(self):
        '''
        If you want your action to have settings dialog, implement this method
        This should return a Qwidget (not dialog) with the following methods:
        [*] __init__(self, plugin_action)
        [*] save_settings(settings)
                This method is used to save the settings from the widget
                it should return a dictionary containing all the settings
        [*] load_settings(self, settings)
                This method is used to load the saved settings into the
                widget
        '''
        return None

    def validate(self, settings):
        '''
        Validate settings dict. This is called when you press the OK button in config dialog.
        changes are applied only if the this methods returns True.
        It is called also when verifying the chain validity on multiple occasions: startup,
        library change, chain dialog initialization .....
        If the setting in the dictionary are not valid, return a tuple
        of two strings (message, details), these will be displayed as a warning dialog to
        the user and the process will be aborted.
        '''
        return True

    def default_settings(self):
        '''
        default settings to be used if no settings are configured
        '''
        return {}

#    def on_modules_update(self):
#        '''
#        This method is called every time the modules editor is edited/updated
#        Implement only if you want to process custom functions related to your action
#        '''
#        raise NotImplementedError

    def is_headless(self, settings):
        '''
        This method should examine the action and its settings to determine
        whether it is suitable for running in batch mode from Action Chains plugin
        or from the command line
        '''
        return self.headless

    def supported_formats(self, settings):
        '''
        Returns a list of formats this action support. Currently used only
        in to asses formats when running from Action Chains.
        '''
        return ['epub','azw3']
