#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout,
                     QCheckBox, QLabel)

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.polish.replace import rationalize_folders, rename_files
from calibre.gui2.tweak_book.widgets import RationalizeFolders
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.base import EditorAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/arrange_into_folders.py - exception when loading translations")

class ConfigDialog(RationalizeFolders):

    def __init__(self, parent, plugin_action, action, name, title):
        RationalizeFolders.__init__(self)
        self.plugin_action = plugin_action
        self.action = action

    def set_folder_map(self, fmap):
        ans = {}
        for typ, x in self.TYPE_MAP:
            w = getattr(self, '%s_folder' % typ)
            v = fmap[typ]
            if v:
                v += '/'
            w.setText(v)
        return ans

    def load_settings(self, settings):
        if settings:
            self.set_folder_map(settings['folder_map'])

    def save_settings(self):
        settings = {}
        settings['folder_map'] = self.folder_map
        return settings

    def accept(self):
        self.settings = self.save_settings()
        # validate settings
        is_valid = self.action.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(
                self,
                msg,
                details,
                show=True
            )
            return
        RationalizeFolders.accept(self)

class ArrangeIntoFolders(EditorAction):

    name = 'Arrange Into Folders'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        folder_map = settings['folder_map']
        name_map = rationalize_folders(container, folder_map)
        rename_files(container, name_map)

    def validate(self, settings):
        if not settings:
            return _('No settings'), _('You must first configure this action')
        return True

    def supported_formats(self, settings):
        return ['epub']

    def config_widget(self):
        return ConfigDialog

