#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
import os
import regex

from qt.core import (QApplication, Qt, QWidget, QHBoxLayout, QVBoxLayout,
                     QGroupBox, QToolButton, QCheckBox, QLineEdit,
                     QRadioButton)

from calibre import prints
from calibre.constants import DEBUG, iswindows
from calibre.gui2 import error_dialog, choose_files, choose_dir

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.common_utils import (get_file_path,
    DragDropComboBox, get_icon)
from calibre.ebooks.oeb.base import OEB_DOCS, OEB_STYLES
from calibre.ebooks.oeb.polish.utils import link_stylesheets

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/add_files.py "
           "- exception when loading translations")

def add_file_to_container(container, filename, target_name=None,
            if_name_exists='skip', link_css=False,
            link_html_to_css=False):

    file_path = get_file_path(filename)
    ext = os.path.splitext(file_path)[-1].lstrip('.').lower()
    if not target_name:
        target_name = os.path.basename(file_path)

    modify_name_if_needed = False
    if container.has_name(target_name):
        if if_name_exists == 'skip':
            return
        elif if_name_exists == 'rename':
            modify_name_if_needed = True
        elif if_name_exists == 'overwrite':
            container.remove_item(target_name)

    with open(file_path, 'rb') as f:
        data = f.read()
        added_name = container.add_file(
            target_name, data, modify_name_if_needed=modify_name_if_needed)

        if ext == 'css' and link_css:
            all_text = list(container.manifest_items_of_type(OEB_DOCS))
            for x in all_text:
                for y in ['nav.xhtml','titlepage.xhtml']:
                    if x.endswith(y):
                        all_text.remove(x)
            link_stylesheets(container, all_text, [added_name])
            for x in all_text:
                container.dirty(x)
        elif ext in ['html','xhtml'] and link_html_to_css:
            sheets = list(container.manifest_items_of_type(OEB_STYLES))
            link_stylesheets(container, [added_name], sheets)
            container.dirty(added_name)


class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.mode = 'file'
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        self.file_box = QGroupBox(_('Choose file(s):'))
        l.addWidget(self.file_box)
        self.file_box_layout = QVBoxLayout()
        self.file_box.setLayout(self.file_box_layout)
        hl1 = QHBoxLayout()
        self.file_box_layout.addLayout(hl1)
        self.file_opt = QRadioButton(_('Select a file to add'))
        hl1.addWidget(self.file_opt)
        self.dir_opt = QRadioButton(_('Select a folder to add'))
        hl1.addWidget(self.dir_opt)
        self.file_opt.setChecked(True)
        for x in [self.file_opt, self.dir_opt]:
            x.toggled.connect(self._mode_toggled)
        file_layout = self.file_layout = QHBoxLayout()
        self.file_box_layout.addLayout(file_layout)
        self.file_combo = DragDropComboBox(self, drop_mode='file')
        file_layout.addWidget(self.file_combo)
        self.choose_file_button = QToolButton(self)
        self.choose_file_button.setToolTip(_('Choose file'))
        self.choose_file_button.setIcon(get_icon('document_open.png'))
        self.choose_file_button.clicked.connect(self._choose_file)
        file_layout.addWidget(self.choose_file_button)

        self.name_groupbox = QGroupBox(_(
            'Filename and path inside ebook (optional)'))
        self.name_groupbox.setToolTip(_(
            'Determine the filename and path inside the ebook. '
            'e.g. Images/image-1.png.\n'
            'If not set, filename will not be modified '
            'and the target file will reside in the root of the container'))
        name_l = QVBoxLayout()
        self.name_groupbox.setLayout(name_l)
        self.name_groupbox.setCheckable(True)
        self.name_groupbox.setChecked(False)
        l.addWidget(self.name_groupbox)
        self.target_exists_le = QLineEdit()
        name_l.addWidget(self.target_exists_le)

        self.exists_groupbox = QGroupBox(_(
            'What to do if same filename exists in ebook.'))
        exists_l = QVBoxLayout()
        self.exists_groupbox.setLayout(exists_l)
        l.addWidget(self.exists_groupbox)
        self.skip_opt = QRadioButton(_(
            'Don\'t add the file'))
        self.skip_opt.setChecked(True)
        exists_l.addWidget(self.skip_opt)
        self.overwrite_opt = QRadioButton(_(
            'Overwrite existing file'))
        exists_l.addWidget(self.overwrite_opt)
        self.modify_opt = QRadioButton(_(
            'Add and modify name'))
        exists_l.addWidget(self.modify_opt)


        self.css_groupbox = QGroupBox(_('CSS Options'))
        css_l = QVBoxLayout()
        self.css_groupbox.setLayout(css_l)
        l.addWidget(self.css_groupbox)

        self.css_link_chk = QCheckBox(_(
            'Link added css file to html files in book.'))
        self.css_link_chk.setChecked(False)
        css_l.addWidget(self.css_link_chk)

        self.html_link_chk = QCheckBox(_(
            'Link added XHTML file to css sheets already in container'))
        self.html_link_chk.setChecked(False)
        css_l.addWidget(self.html_link_chk)

        l.addStretch(1)

        self.setMinimumSize(350,500)

    def _mode_toggled(self):
        if self.file_opt.isChecked():
            self.mode = 'file'
        else:
            self.mode = 'dir'
        self.name_groupbox.setVisible(self.file_opt.isChecked())
        self.file_combo.setCurrentText('')

    def _choose_file(self):
        if self.mode == 'file':
            chosen_files = choose_files(
                None, _('Select file dialog'), _('Select a file'),
                all_files=True, select_only_single_file=True)
            if not chosen_files:
                return

            file_path = chosen_files[0]

        else:
            file_path = choose_dir(self, 'choose direcotry',
                    _('Choose direcotry'))

        if iswindows:
            file_path = os.path.normpath(file_path)

        self.block_events = True
        existing_index = self.file_combo.findText(file_path, Qt.MatchExactly)
        if existing_index >= 0:
            self.file_combo.setCurrentIndex(existing_index)
        else:
            self.file_combo.insertItem(0, file_path)
            self.file_combo.setCurrentIndex(0)
        self.block_events = False

    def load_settings(self, settings):
        if settings:
            is_file = settings.get('is_file', True)
            self.file_opt.setChecked(is_file)
            self.dir_opt.setChecked(not is_file)
            filename = settings.get('filename')
            if filename:
                self.file_combo.setCurrentText(filename)
            if_name_exists = settings.get('if_name_exists')
            if if_name_exists == 'skip':
                self.skip_opt.setChecked(True)
            elif if_name_exists == 'overwrite':
                self.overwrite_opt.setChecked(True)
            elif if_name_exists == 'rename':
                self.modify_opt.setChecked(True)
            self.css_link_chk.setChecked(settings['link_css'])
            self.html_link_chk.setChecked(
                settings.get('link_html_to_css', False))
            if is_file:
                target_name = settings.get('target_name', '')
                if target_name:
                    self.name_groupbox.setChecked(True)
                    self.target_exists_le.setText(target_name)

    def save_settings(self):
        settings = {}
        settings['is_file'] = self.file_opt.isChecked()
        settings['filename'] = self.file_combo.currentText().strip()
        if self.skip_opt.isChecked():
            settings['if_name_exists'] = 'skip'
        elif self.overwrite_opt.isChecked():
            settings['if_name_exists'] = 'overwrite'
        elif self.modify_opt.isChecked():
            settings['if_name_exists'] = 'rename'
        settings['link_css'] = self.css_link_chk.isChecked()
        settings['link_html_to_css'] = self.html_link_chk.isChecked()
        if self.file_opt.isChecked() and self.name_groupbox.isChecked():
            settings['target_name'] = self.target_exists_le.text()
        return settings


def tree_names(tree, include_dirs=False):
    l = []
    for parent, dirs, files in os.walk(tree):
        iter_over = files
        if include_dirs: iter_over += dirs
        for x in iter_over:
            full_path = os.path.join(parent, x)
            l.append(full_path)
    return l

class AddFiles(EditorAction):

    name = 'Add Files'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container

        is_file = settings.get('is_file', True)
        filename = settings.get('filename')
        if_name_exists = settings.get('if_name_exists', 'skip')
        link_css = settings.get('link_css', False)
        link_html_to_css = settings.get('link_html_to_css', False)

        target_name = settings.get('target_name')

        if is_file:
            add_file_to_container(
                container,
                filename,
                target_name=target_name,
                if_name_exists=if_name_exists,
                link_css=link_css,
                link_html_to_css=link_html_to_css
            )
        else:
            tree = filename
            for fpath in tree_names(tree):
                relpath = os.path.relpath(fpath, tree)
                add_file_to_container(
                    container,
                    fpath,
                    target_name=relpath,
                    if_name_exists=if_name_exists,
                    link_css=link_css,
                    link_html_to_css=link_html_to_css
                )

    def validate(self, settings):
        if not settings:
            return (
                _('Settings Error'),
                _('You must configure this action before running it')
            )
        filename = settings.get('filename')
        if filename:
            file_path = get_file_path(filename)
            if not os.access(file_path, os.R_OK):
                return (_
                        ('Invalid path'),
                        _(f'Path: {file_path} is not a valid path')
                )
        else:
            return (
                _('Missing file/folder'),
                _('You must choose a valid file/folder')
            )
        if settings.get('is_file', True):
            if not os.path.isfile(file_path):
                return (
                    _('File error'),
                    _(f'Path: {filename} does not belong to a file')
                )
            target_name = settings.get('target_name')
            if not target_name:
                target_name = os.path.basename(file_path)
            targets = [target_name]
        else:
            if not os.path.isdir(file_path):
                return (
                    _('Directory error'),
                    _(f'Path: {filename} does not belong to a directory')
                )
            targets = [os.path.relpath(x, file_path) for x in
                       tree_names(file_path)]
            if not targets:
                return (
                    _('Empty directory'),
                    _(f'Directory: {filename} does not contain any file')
                )
        for target_name in targets:
            if '..' in target_name:
                    return (
                        _('Invalid target name'),
                        _(f'Target name ({target_name}) '
                          'contains .. which is not allowed')
                    )
            for x in target_name.split('/'):
                if not regex.search(r'^[\w\d\-\._]+$', x):
                    return (
                        _('Invalid target name'),
                        _(f'Target name ({target_name}) contains '
                          'illegal characters')
                    )
        return True

    def config_widget(self):
        return ConfigWidget

