#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.actions.tag_actions import TagActions
from calibre_plugins.editor_chains.actions.style_actions import StyleActions
from calibre_plugins.editor_chains.actions.search_replace import SearchReplace
from calibre_plugins.editor_chains.actions.chain_caller import ChainCaller
from calibre_plugins.editor_chains.actions.code import CodeAction
from calibre_plugins.editor_chains.actions.mark_cursor_tag import MarkCursorTag
from calibre_plugins.editor_chains.actions.insert_before_cursor_tag import InsertBeforeCursorTag
from calibre_plugins.editor_chains.actions.insert_after_cursor_tag import InsertAfterCursorTag
from calibre_plugins.editor_chains.actions.embed_fonts import EmbedFonts
from calibre_plugins.editor_chains.actions.subset_fonts import SubsetFonts
from calibre_plugins.editor_chains.actions.remove_unused_css import RemoveUnusedCSS
from calibre_plugins.editor_chains.actions.smarten import SmartenPunctuation
from calibre_plugins.editor_chains.actions.upgrade_internals import UpgradeInternals
from calibre_plugins.editor_chains.actions.transform_styles import TransformStyles
from calibre_plugins.editor_chains.actions.transform_html import TransformHTML
from calibre_plugins.editor_chains.actions.fix_html import FixHTML
from calibre_plugins.editor_chains.actions.pretty_all import BeautifyAll
from calibre_plugins.editor_chains.actions.create_toc import CreateTOC
from calibre_plugins.editor_chains.actions.arrange_into_folders import ArrangeIntoFolders
from calibre_plugins.editor_chains.actions.multisplit import SplitMultipleLocations
from calibre_plugins.editor_chains.actions.add_files import AddFiles
from calibre_plugins.editor_chains.actions.compress_images import CompressImages
from calibre_plugins.editor_chains.actions.remove_files import RemoveFiles
from calibre_plugins.editor_chains.actions.insert_inline_toc import InsertInlineTOC
from calibre_plugins.editor_chains.actions.flatten_toc import FlattenTOC
from calibre_plugins.editor_chains.actions.remove_unused_images import RemoveUnusedImages
from calibre_plugins.editor_chains.actions.expand_styles import ExpandStyles
from calibre_plugins.editor_chains.actions.inline_styles import InlineStyles
from calibre_plugins.editor_chains.actions.run_command import RunCommand
#from calibre_plugins.editor_chains.actions.external_resources import ExternalResources

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/__int__.py - exception when loading translations")


def get_imported_actions(plugin_action):
    all_imported_actions = {}
    for plugin_name, imported_resources in plugin_action.imported_resources.items():
        imported_actions = imported_resources.get('actions', [])
        for imported_action in imported_actions:
            name = imported_action.name
            # must define a name attribute, must be set and not clash with builtin names
            # which can be imported into the module manager by custom actions
            if name in ['', 'Editor Action']:
                continue
            imported_action._source_plugin = plugin_name
            all_imported_actions[name] = imported_action          
    return all_imported_actions

def get_user_actions(plugin_action):
    user_actions = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[EditorAction]):
        name = cls.name
        # must define a name attribute, must be set and not clash with builtin names
        # which can be imported into the module manager by custom actions
        if name in ['', 'Editor Action']:
            continue
        user_actions[name] = cls            
    return user_actions

def get_all_actions(plugin_action):

    headless_mode = plugin_action.gui is None
    builtin_actions = OrderedDict()
    non_headless = []
    
    _builtin_actions = [
        TagActions,
        StyleActions,
        SearchReplace,
        ChainCaller,
        CodeAction,
        MarkCursorTag,
        InsertBeforeCursorTag,
        InsertAfterCursorTag,
        AddFiles,
        RemoveFiles,
        EmbedFonts,
        SubsetFonts,
        RemoveUnusedCSS,
        SmartenPunctuation,
        UpgradeInternals,
        TransformStyles,
        TransformHTML,
        FixHTML,
        BeautifyAll,
        CreateTOC,
        ArrangeIntoFolders,
        InsertInlineTOC,
        FlattenTOC,
        SplitMultipleLocations,
        CompressImages,
        RemoveUnusedImages,
#        ExternalResources,
        ExpandStyles,
        InlineStyles,
        RunCommand
    ]

    for action_cls in _builtin_actions:
        builtin_actions[action_cls.name] = action_cls

    _user_actions = get_user_actions(plugin_action)
    
    _imported_actions = get_imported_actions(plugin_action)

    all_actions = OrderedDict()
    user_actions = OrderedDict()
    imported_actions = OrderedDict()
    
    for action_name, action_cls in builtin_actions.items():
        if headless_mode and not action_cls.headless:
            non_headless.append(action_name)
            continue
        action = action_cls(plugin_action)
        all_actions[action_name] = action

    # Note: imported actions can either be class or instantiated object
    for action_name, action_obj in _imported_actions.items():
        if headless_mode and not action_obj.headless:
            non_headless.append(action_name)
            continue
        # dont override builtin actions
        if action_name in builtin_actions.keys():
            continue
        try:
            if isinstance(action_obj, EditorAction):
                action = action_obj
            elif issubclass(action_obj, EditorAction):
                action = action_obj(plugin_action)
            all_actions[action_name] = action
            imported_actions[action_name] = action
        except TypeError as e:
            # TypeError: issubclass() arg 1 must be a class
            import traceback
            if DEBUG:
                prints(f'Editor Chains: Error intializing imported action: Un-reconized object: {action_obj}\n{traceback.format_exc()}')
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Editor Chains: Error intializing imported action: {action_name}\n{traceback.format_exc()}')

    for action_name, action_cls in _user_actions.items():
        if headless_mode and not action_cls.headless:
            non_headless.append(action_name)
            continue
        # dont override builtin actions
        if action_name in builtin_actions.keys():
            continue
        try:
            action = action_cls(plugin_action)
            all_actions[action_name] = action
            user_actions[action_name] = action
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Editor Chains: Error intializing user action: {action_name}\n{traceback.format_exc()}')

    if headless_mode:
        print(f'Editor Chains: actions that does not run headless: {non_headless}')
    return all_actions, builtin_actions, user_actions, imported_actions
