#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QHBoxLayout, QVBoxLayout,
                     QGroupBox, QComboBox, QCheckBox, QLabel, QLineEdit,
                     QRadioButton, QToolButton, QPushButton, QIcon)

from collections import defaultdict
import copy
import os
import shutil

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import question_dialog, error_dialog
from calibre.gui2.tweak_book.save import send_message
from calibre.gui2.widgets2 import Dialog
from calibre.ptempfile import PersistentTemporaryDirectory

from calibre_plugins.action_chains.actions.base import ChainAction

from calibre_plugins.editor_chains.headless import editor_chains_action, editor_wrapper
from calibre_plugins.editor_chains.common_utils import get_icon
from calibre_plugins.editor_chains.export_import import (
    export_chains, chains_config_from_archive)
import calibre_plugins.editor_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("EditorChains::action_chains.py - exception when loading translations")

def copy_to_format(
    gui,
    book_ids,
    source_fmt,
    destination_fmt,
    remove_source=False,
    confirm_overwrite=True
    ):
    tdir = PersistentTemporaryDirectory('_ec_copy_format')
    db = gui.current_db
    if confirm_overwrite:
        if not overwrite_confirmation(gui, book_ids, source_fmt, destination_fmt):
            return
    
    for book_id in book_ids:
        fmts_string = db.formats(book_id, index_is_id=True)
        if fmts_string:
            available_fmts = [ fmt.strip() for fmt in fmts_string.split(',') ]
        else:
            available_fmts = []
        if source_fmt.upper() in available_fmts:
            path = os.path.join(tdir, str(book_id) + '.' + destination_fmt)
            with open(path, 'wb') as f:
                db.copy_format_to(book_id, source_fmt, f, index_is_id=True)
                db.new_api.add_format(book_id, destination_fmt, path)
            if remove_source:
                db.remove_format(book_id, source_fmt, index_is_id=True, notify=False)
    try:
        shutil.rmtree(tdir)
    except:
        traceback.print_exc()

def overwrite_confirmation(gui, book_ids, source_fmt, destination_fmt):
    db = gui.current_db
    l = []
    for book_id in book_ids:
        fmts_string = db.formats(book_id, index_is_id=True)
        if fmts_string:
            available_fmts = [ fmt.strip() for fmt in fmts_string.split(',') ]
        else:
            available_fmts = []
        if not source_fmt.upper() in available_fmts:
            continue
        if destination_fmt.upper() in available_fmts:
            l.append(f'{db.title(book_id, index_is_id=True)} ({book_id})')
    if l:
        msg = _('Copy to format will overwrite already existing files '
                'for some books. Are you sure you want to overwrite them?')
        details = '\n'.join(l)
        msg = _('Some books have the same format you are trying to copy. '
                'Are you sure you want to overwrite existing formats?')
        return question_dialog(gui, _('Are you sure?'), msg, det_msg=details)
    return True

#==========================

class ChooseChainNameDialog(Dialog):
    def __init__(self, parent, msg, names):
        self.msg = msg
        self.names = names
        Dialog.__init__(
            self,
            'Chain Name Dialog',
            'ec-choose-chain-dialog',
            parent)

    def setup_ui(self):
        self.setWindowTitle(_('Choose chain to import'))
        l = QVBoxLayout()
        self.setLayout(l)

        self.lbl = QLabel(self.msg)
        l.addWidget(self.lbl)
        self.chain_combo = QComboBox()
        self.chain_combo.addItems(self.names)
        l.addWidget(self.chain_combo)
        l.addStretch(1)

        l.addWidget(self.bb)

    def accept(self):
        self.chain_name = self.chain_combo.currentText()
        Dialog.accept(self)


def to_chain_config(chains_data):
    cfg.migrate_plugin_prefs_if_required(chains_data, commit=False)
    chain_config = chains_data.get(cfg.STORE_MENUS_NAME, {}).get(cfg.KEY_MENUS, [])[0]
    return chain_config

def from_chain_config(chain_config, chain_name=None):
    if chain_name:
        chain_config['menuText'] = chain_name
    return {
        cfg.STORE_MENUS_NAME :{
            cfg.KEY_MENUS: [chain_config]
        },
        cfg.KEY_SCHEMA_VERSION: cfg.DEFAULT_SCHEMA_VERSION
    }

class ConfigWidget(QWidget):
    def __init__(
        self,
        plugin_action,
        chain_name,
        chains_config,
        *args,
        **kwargs
    ):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.icon_cache = plugin_action.icon_cache
        self.chain_config = {
            'menuText': chain_name,
            'active': True,
            'image': '',
            'chain_settings': {
                'chain_links': []
            }
        }
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)
        
        chain_groupbox = QGroupBox(_('Select editor chain'))
        l.addWidget(chain_groupbox)
        chain_groupbox_l = QVBoxLayout()
        chain_groupbox.setLayout(chain_groupbox_l)
        hl1 = QHBoxLayout()
        chain_groupbox_l.addLayout(hl1)
        self.existing_chain_opt = QRadioButton(
            _('Select an existing chain'))
        self.existing_chain_opt.setChecked(True)
        self.chain_combo = QComboBox()
        hl1.addWidget(self.existing_chain_opt)
        hl1.addWidget(self.chain_combo)
        hl2 = QHBoxLayout()
        chain_groupbox_l.addLayout(hl2)
        self.new_chain_opt = QRadioButton(_('Create a new chain'))
        self.create_chain_button = QPushButton(_('Create/Edit'))
        self.create_chain_button.setIcon(QIcon(I('gear.png')))
        self.create_chain_button.clicked.connect(
            self._on_create_chain_button_clicked)

        self.chain_name_ledit = QLineEdit()
        self.chain_name_ledit.setToolTip(
            _('Enter the chain name here. '
              'This is only necessary if you want to export the chain.'))
        self.chain_name_ledit.setText(self.chain_config['menuText'])

        self.import_button = QToolButton()
        self.import_button.setIcon(get_icon('images/import.png'))
        self.import_button.setToolTip(_('Import chains ...'))
        self.import_button.clicked.connect(self.import_chain)

        self.export_button = QToolButton()
        self.export_button.setIcon(get_icon('images/export.png'))
        self.export_button.setToolTip(_('Export chains ...'))
        self.export_button.clicked.connect(self.export_chain)

        hl2.addWidget(self.new_chain_opt)
        hl2.addWidget(self.chain_name_ledit)
        hl2.addWidget(self.import_button)
        hl2.addWidget(self.export_button)
        hl2.addWidget(self.create_chain_button)

        fmts_groupbox = QGroupBox(_('Format:'))
        l.addWidget(fmts_groupbox)
        self.fmts_layout = QHBoxLayout()
        fmts_groupbox.setLayout(self.fmts_layout)
        self.fmts_combo = QComboBox()
        self.fmts_layout.addWidget(self.fmts_combo)
        self.fmts_combo.addItems(['epub','azw3'])
        
        self.backup_opts_groupbox = backup_opts_groupbox = QGroupBox(
            _('Backup format before running editor chain'))
        backup_opts_groupbox_l = QVBoxLayout()
        backup_opts_groupbox.setLayout(backup_opts_groupbox_l)
        backup_opts_groupbox.setCheckable(True)
        backup_opts_groupbox.setChecked(True)
        l.addWidget(backup_opts_groupbox)

        dest_l = QHBoxLayout()
        backup_opts_groupbox_l.addLayout(dest_l)        
        dest_label = QLabel(_('Destination format:'))
        self.dest_fmt_ledit = QLineEdit()
        self.dest_fmt_ledit.setText('EC_BACKUP')
        self.dest_fmt_ledit.setToolTip(
            _('Extension used for backup e.g. epub_backup'))
        dest_l.addWidget(dest_label)
        dest_l.addWidget(self.dest_fmt_ledit)

        self.confirm_overwrite_chk = QCheckBox(
            _('Popup dialog to confirm before '
              'overwriting previously existing backups'))
        self.confirm_overwrite_chk.setChecked(False)
        backup_opts_groupbox_l.addWidget(self.confirm_overwrite_chk)
        
        try:
            all_chain_names = editor_wrapper.get_compatible_chains()
        except:
            import traceback
            if DEBUG:
                prints(traceback.format_exc())
            all_chain_names = []

        self.chain_combo.addItems(all_chain_names)
        self.chain_combo.setCurrentIndex(-1)
        if not all_chain_names:
            self.new_chain_opt.setChecked(True)

        self.read_only_chk = QCheckBox(
            _('Read only chain (will not save any changes to book format.)'))
        self.read_only_chk.setChecked(False)
        l.addWidget(self.read_only_chk)
        
        l.addStretch(1)
        self._refresh_buttons()

        self.setMinimumSize(600,350)

    def _on_create_chain_button_clicked(self):
        from calibre_plugins.editor_chains.gui.actions_dialog import ActionsDialog
        chain_name = self.chain_name_ledit.text()
        self.chain_config['menuText'] = chain_name
        d = ActionsDialog(self, editor_chains_action, [self.chain_config], chain_name)
        if d.exec_() == d.Accepted:
            self.chain_config['chain_settings'] = d.chain_settings
            icon = d.image_combobox.current_text()
            if icon:
                self.chain_config['image'] = icon
            self.new_chain_opt.setChecked(True)
        self._refresh_buttons()

    def _refresh_buttons(self):
        self.export_button.setEnabled(len(self.chain_config.get('chain_settings', {}).get('chain_links', [])) != 0)
        icon = self.chain_config.get('image')
        if icon:
            self.create_chain_button.setIcon(get_icon(icon))

    def import_chain(self):
        chains_config = chains_config_from_archive(self.gui, self.plugin_action)
        if not chains_config: return
        names = [chain_config['menuText'] for chain_config in chains_config]
        count = len(names)
        if count == 0:
            return
        elif count == 1:
            chain_config = chains_config[0]
        elif count > 1:
            d = ChooseChainNameDialog(
                self,
                _('Zip file cotains multiple chains. Choose the name of the chain you want to import.'),
                names
            )
            if d.exec() == d.Accepted:
                chain_name = d.chain_name
                chain_config = cfg.get_chain_config(chain_name, chains_config)
            else:
                return
        self.chain_name_ledit.setText(chain_config['menuText'])
        self.chain_config = chain_config
        self.new_chain_opt.setChecked(True)
        self._refresh_buttons()

    def export_chain(self):
        chain_name = self.chain_name_ledit.text()
        if not chain_name:
            return error_dialog(
                self,
                _('Chain must have a name'),
                _('You must enter a chain name before pressing export.'),
                show=True
            )
        if len(self.chain_config.get('chain_settings', {}).get('chain_links', [])) == 0:
            return error_dialog(
                self,
                _('No chain to import'),
                _('You must first create a chain before pressing export.'),
                show=True
            )
        chain_config = copy.deepcopy(self.chain_config)
        chain_config['menuText'] = chain_name
        if chain_name:
            export_chains(self.plugin_action, self.gui, [chain_config])

    def load_settings(self, settings):
        if settings:
            opt = settings.get('opt', 'existing')
            if opt == 'existing':
                self.existing_chain_opt.setChecked(True)
                idx = self.chain_combo.findText(settings['chain_name'])
                self.chain_combo.setCurrentIndex(idx)
            elif opt == 'new':
                self.new_chain_opt.setChecked(True)
                self.chain_config = to_chain_config(settings['chains_data'])
                self.chain_name_ledit.setText(self.chain_config.get('menuText', ''))
            self.fmts_combo.setCurrentText(settings['fmt'])
            if settings.get('backup'):
                self.backup_opts_groupbox.setChecked(True)
                self.confirm_overwrite_chk.setChecked(
                    settings.get('confirm_overwrite', False))
                self.dest_fmt_ledit.setText(settings.get('destination_fmt', ''))
            else:
                self.backup_opts_groupbox.setChecked(False)
            self.read_only_chk.setChecked(settings.get('read_only', False))
        self._refresh_buttons()

    def save_settings(self):
        settings = {}
        if self.existing_chain_opt.isChecked():
            settings['opt'] = 'existing'
            settings['chain_name'] = self.chain_combo.currentText()
        elif self.new_chain_opt.isChecked():
            settings['opt'] = 'new'
            #settings['chains_data'] = self.chain_config
            settings['chains_data'] = from_chain_config(
                self.chain_config,
                chain_name=self.chain_name_ledit.text())
        settings['fmt'] = self.fmts_combo.currentText()
        if self.backup_opts_groupbox.isChecked():
            settings['backup'] = True
            settings['confirm_overwrite'] = self.confirm_overwrite_chk.isChecked()
            settings['destination_fmt'] = self.dest_fmt_ledit.text()
        else:
            settings['backup'] = False
        settings['read_only'] = self.read_only_chk.isChecked()
        return settings

class Book:
    pass

class RunEditorChainAction(ChainAction):

    name = 'Run Editor Chain'
    support_scopes = True

    def run(self, gui, settings, chain):
        db = gui.current_db
        tdir = PersistentTemporaryDirectory('_ec_ac')
        book_ids = chain.scope().get_book_ids()
        fmt = settings['fmt'].upper()
        opt = settings.get('opt', 'existing')
        if opt == 'existing':
            chain_name = settings['chain_name']
            chain_config = cfg.get_chain_config(chain_name)
        elif opt == 'new':
            chain_config = to_chain_config(settings['chains_data'])
        books = []
        read_only = settings.get('read_only', False)
        for book_id in book_ids:
            book_path = self.get_book_path(gui, book_id, fmt)
            if book_path:
                book = Book()
                book.path = book_path
                book.id = book_id
                books.append(book)
        if settings.get('backup') and not read_only:
            confirm_overwrite = settings.get('confirm_overwrite', False)
            destination_fmt = settings['destination_fmt']
            copy_to_format(
                gui,
                [book.id for book in books],
                fmt, destination_fmt,
                remove_source=False,
                confirm_overwrite=confirm_overwrite)
        action_chain_vars = getattr(chain, 'chain_vars', {})
        for book in books:
            debug_string = f'Running for ({book.id}) {db.title(book.id, index_is_id=True)}'
            # copy format to temporary file {
            path_to_temp_book = os.path.join(tdir, str(book_id) + '.' + fmt.lower())
            with open(path_to_temp_book, 'wb') as f:
                db.copy_format_to(book.id, fmt, f, index_is_id=True)
            # }
            editor_wrapper.run_editor_chain(
                chain_config,
                path_to_temp_book,
                book.id,
                chain_vars={'_ac_chain_vars': action_chain_vars},
                debug_string=debug_string)
            if not read_only:
                #db.new_api.add_format(book.id, fmt, path_to_temp_book)
                with open(path_to_temp_book, 'rb') as rf:
                    with open(book.path, 'wb') as wf:
                        wf.write(rf.read())
                # Update size for format
                x = db.new_api.format_metadata(book.id, fmt.upper(), update_db=True)
        if not read_only:
            db.update_last_modified([book.id for book in books])
        try:
            shutil.rmtree(tdir)
        except:
            traceback.print_exc()

    def get_book_path(self, gui, book_id, fmt):
        db = gui.current_db
        fmts_string = db.formats(book_id, index_is_id=True)
        if fmts_string:
            available_fmts = [ fmt.strip().upper() for fmt in fmts_string.split(',') ]
        else:
            available_fmts = []
        if fmt.upper() in available_fmts:
            path_to_book = db.format_abspath(book_id, fmt, index_is_id=True)
            return path_to_book
        else:
            return None

    def validate(self, settings):
        opt = settings.get('opt', 'existing')
        if opt == 'existing':
            chain_name = settings.get('chain_name')
            if not chain_name:
                return (
                    _('No chain'),
                    _('No editor chain is selected')
                )
            chain_config = cfg.get_chain_config(chain_name)
        elif opt == 'new':
            chain_config = to_chain_config(settings['chains_data'])
            chain_settings = chain_config['chain_settings']
            chain_links = chain_settings.get('chain_links',[])
            if len(chain_links) == 0:
                return (
                    _('No actions'),
                    _('Chain must have at least one action')
                )
        fmt = settings.get('fmt')
        if not fmt:
            return (
                _('No format'),
                _('No format is selected')
            )
        
        if settings.get('backup', False):
            destination_fmt = settings.get('destination_fmt', '')
            if not destination_fmt:
                return (
                    _('No backup format'),
                    _('You must set a backup format or turnoff backup option')
                )
        
        is_valid = editor_wrapper.validate_editor_chain(chain_config)
        if is_valid is not True:
            return is_valid
        chain_supported_formats = editor_wrapper.get_chain_supported_formats(chain_config)
        if fmt not in chain_supported_formats:
            return (
                _('Unsupported format'),
                _(f"Editor Chain ({chain_config['menuText']}) does not support ({fmt}) format")
            )
        return True

    def config_widget(self):
        return ConfigWidget
