#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import defaultdict
import copy
import importlib
import os

from calibre import prints
from calibre.constants import iswindows, isosx, DEBUG
from calibre.customize.ui import initialized_plugins, builtin_names

from calibre_plugins.editor_chains.common_utils import call_method
from calibre_plugins.editor_chains.actions import get_all_actions
from calibre_plugins.editor_chains.chains import Chain
from calibre_plugins.editor_chains.modules import UserModules


try:
    load_translations()
except NameError:
    prints("EditorChains::action.py - exception when loading translations")

def get_plugin_resources(plugin):
    from calibre_plugins.editor_chains.actions.base import EditorAction
    res = {}
    try:
        main = importlib.import_module(plugin.__class__.__module__+'.editor_chains')
    except ModuleNotFoundError:
        pass
    except ImportError:
        import traceback
        traceback.print_exc()
    else:
        actions = []
        for k,v in vars(main).items():
            if isinstance(v, type) and issubclass(v, EditorAction):
                actions.append(v)
            if k == 'on_modules_update' and callable(v):
                res['on_modules_update'] = v
        if actions:
            res['actions'] = actions
    return res

class EditorChainsPluginAction(object):

    def __init__(self, gui=None, boss=None, tool=None, resources_dir=None, icon_cache=None):
        self.gui = gui
        self.boss = boss
        self.tool = tool
        if not resources_dir:
            from calibre.utils.config import config_dir
            resources_dir = os.path.join(config_dir, 'resources/images')
        self.resources_dir = resources_dir
        if icon_cache is None:
            from calibre_plugins.editor_chains.gui import IconCache
            icon_cache = IconCache(self)
        self.icon_cache = icon_cache

        # these variables are used to block events when chains and events are running
        self.chainStack = []
        # counter for how many times a chain has run
        self.chain_iterations = defaultdict(lambda: 0)

        self.imported_resources = self._get_plugins_resources()
        self.on_modules_update()

    def on_modules_update(self):
        self.user_modules = UserModules()
        self.actions, self.builtin_actions, self.user_actions, self.imported_actions  = get_all_actions(self)
        self.call_actions_method('on_modules_update', self.user_modules)
        self._call_plugins_on_modules_update(self.user_modules)

    def _get_plugins_resources(self):
        '''
        Look in plugins for action chains resources (e.g. actions, events, scopes)
        Resources should be defined in action_chains.py module in the root of the plugin.
        resources will be a dictionary as follows
        {
            'actions': [],
            'events': [],
            'template_functions: []
        }
        '''
        if DEBUG:
            prints('Editor chains: _get_plugins_resources(): start')
        all_resources = {}

        # Collect resources from plugins
        for plugin in initialized_plugins():
            if plugin.name in builtin_names:
                pass
            else:
                try:
                    resources = get_plugin_resources(plugin)
                except:
                    resources = {}
                    import traceback
                    traceback.print_exc()
                if resources:
                    all_resources[plugin.name] = resources

        if DEBUG:
            prints('Editor Chains: _get_plugins_resources(): finished')
        return all_resources

    def _call_plugins_on_modules_update(self, user_modules):
        '''
        This method makes the module editor available for plugins that
        implement on_modules_update(). It passes user_modules instance
        which allows plugins to access different objects and classes
        defined in the module editor.
        '''
        if DEBUG:
            prints('Editor chains: _call_plugins_on_modules_update(): start')
        for plugin_name, resources in self.imported_resources.items():
            if plugin_name == 'Editor Chains': continue
            on_modules_update = resources.get('on_modules_update')
            if on_modules_update:
                try:
                    if DEBUG:
                        prints(f'Editor Chains: Running on_modules_update() for plugin: {plugin_name}')
                    on_modules_update(user_modules)
                except Exception as e:
                    if DEBUG:
                        prints(f'Editor Chains: Error running on_modules_update from plugin: {plugin_name}')
                        import traceback
                        print(traceback.format_exc())
        if DEBUG:
            prints('Editor Chains: _call_plugins_on_modules_update(): finished')

    def run_chain(self, chain_config, check_conditions=True, print_errors=True):
        chain = Chain(self, chain_config, self.tool, gui=self.gui, boss=self.boss, chain_vars={})
        chain.run(check_conditions=check_conditions,
                  print_errors=print_errors)
        del chain

    def call_actions_method(self, meth_name, *args):
        '''
        loop over all actions and call the supplied method if present
        '''
        call_method(self.actions, meth_name, *args)
