﻿using System;
using System.Collections.Generic;
using System.Text;

using System.IO;

namespace ConvertIMPGUI
{
    public class CLRFGenerator
    {
        UInt16 LRF_VERSION    = 0x08;
        UInt16 LRF_PSUEDOKEY   = 0x0A;
        UInt16 LRF_HEADSUBFILE = 0x0C;
        UInt16 LRF_SUBFILECOUNT   = 0x10;
        UInt16 LRF_SUBFILEOFFSET  = 0x18;
        UInt16 LRF_DIRECTION   = 0x24  ;
        UInt16 LRF_DIRECTION_FORWARDS  = 0x01;
        UInt16 LRF_DIRECTION_BACKWARDS  = 0x10;
        UInt16 LRF_UNK1     = 0x26;
        UInt16 LRF_UNK2     = 0x2A;
        UInt16 LRF_UNK3     = 0x2C;
        UInt16 LRF_UNK4     = 0x2E;
        UInt16 LRF_ALTSUBFILE  = 0x44;        /* doesn't seem to be used */
        UInt16 LRF_UNK5     = 0x48;
        UInt16 LRF_INFOLEN  = 0x4C;
        UInt16 LRF_IMGLEN   = 0x50;
        UInt16 LRF_UNK6     = 0x4E;
        byte[] fontname = new byte[] {22, 0,(byte)'I',0,(byte)'W',0,(byte)'A',0,0x0E,0x66,(byte)'-',0,
            0x2D,0x4E,0x30,0x7D,(byte)'N',0,(byte)'-',0,(byte)'e',0,(byte)'b',0};

        public CIMPFile impFile = null;
        int pages = 0;
        UInt16 global_subfile = 0x32;

        SubFile book;
        SubFile head;
        SubFile tail;
        UInt32 font_id;
        UInt32 margins_id;
        UInt32 page_box_id;
        StringBuilder infoFile = new StringBuilder();

        private UInt32 READ_LE_DWORD(byte[] pv, int o)
        {
            return (UInt32)(pv[o + 0] + (pv[o + 1] << 8) + (pv[o + 2] << 16) + (pv[o + 3] << 24));
        }

        private UInt16 READ_LE_WORD(byte[] pv, int o)
        {
            return (UInt16)(pv[o + 0] + (pv[o + 1] << 8));
        }

        private void WRITE_LE_DWORD(ref byte[] p, int o, UInt32 x)
        {
            p[o + 0] = (byte)(x & 0xFF);
            p[o + 1] = (byte)((x >> 8) & 0xFF);
            p[o + 2] = (byte)((x >> 16) & 0xFF);
            p[o + 3] = (byte)((x >> 24) & 0xFF);
        }

        private void WRITE_LE_WORD(ref byte[] p, int o, UInt16 x)
        {
            p[o + 0] = (byte)(x & 0xFF);
            p[o + 1] = (byte)((x >> 8) & 0xFF);
        }

        #region new_subfile
        private SubFile new_subfile(UInt16 type)
        {
            SubFile q = new SubFile();
            q.id = global_subfile++;
            q.type = type;
            return q;
        }
        #endregion

        #region new_tag
        private Tag new_tag(byte tagid, int len, byte[] data, UInt32 val)
        {
            Tag q = new Tag();
            q.tagid = tagid;
            if (len <= 16)
            {
                q.len = len;
                if (data != null)
                {
                    for (int i = 0; i < len; i++)
                        q.shortdata[i] = data[i];
                }
                else
                {
                    for (int i = 0; i < len; i++)
                        q.shortdata[i] = (byte)val;
                }
            }
            else
            {
                q.len = 0;
                q.data = "";
                for (int i = 0; i < len; i++)
                    q.data += (char)data[i];
            }

            return q;
        }
        #endregion

        #region add_tag_to_subfile
        private void add_tag_to_subfile(SubFile q, Tag r)
        {
            Tag p;

            if (q.taglist != null)
            {
                p = q.taglist;
                while (p.next != null)
                    p = p.next;
                p.next = r;
            }
            else
                q.taglist = r;
        }
        #endregion

        #region StartNewBook
        public void StartNewBook()
        {
            SubFile fontrec = new SubFile();
            SubFile margins = new SubFile();
            SubFile page_box = new SubFile();

            pages = 0;

            head = new_subfile(0x1C);
            add_tag_to_subfile(head, new_tag(0x75,sizeof(UInt16), null,2));
            add_tag_to_subfile(head, new_tag(0x76,sizeof(UInt16),null,0));
            add_tag_to_subfile(head, new_tag(0x77,sizeof(UInt16),null,1));

            byte[] tmp78 = new byte[] { 0,0,0,0,0x16,0xf5,0,0, 0x1,0x30};
            add_tag_to_subfile(head, new_tag(0x78,10,tmp78,0));

            add_tag_to_subfile(head, new_tag(0x79,sizeof(UInt16),null,2));
            add_tag_to_subfile(head, new_tag(0x7A,sizeof(UInt16),null,0x10));
            add_tag_to_subfile(head, new_tag(0xDA,sizeof(UInt16),null,2));

            tail = head;

            SubFile temp = new_subfile(0x1E);
            temp.dataflags = 0x51;
            temp.data = "" + (char)0 + (char)0 + (char)0 + (char)0;
            tail.next = temp;
            tail = tail.next;

            /* Global font record for the file */
            fontrec = new_subfile(0x0b);
            add_tag_to_subfile(fontrec,new_tag(0x76,sizeof(UInt16),null,0x0));
            add_tag_to_subfile(fontrec,new_tag(0x77,sizeof(UInt16),null,0x1));
            add_tag_to_subfile(fontrec,new_tag(0x79,sizeof(UInt16),null,0x1));
            add_tag_to_subfile(fontrec,new_tag(0x7A,sizeof(UInt16),null,0x0));
            add_tag_to_subfile(fontrec,new_tag(0x11,sizeof(UInt16),null,0x64));
            add_tag_to_subfile(fontrec,new_tag(0x12,sizeof(UInt16),null,0xFFF6));
            add_tag_to_subfile(fontrec,new_tag(0x13,sizeof(UInt16),null,0x0));
            add_tag_to_subfile(fontrec,new_tag(0x14,sizeof(UInt16),null,0x0));
            add_tag_to_subfile(fontrec,new_tag(0x15,sizeof(UInt16),null,0x190));
            add_tag_to_subfile(fontrec,new_tag(0x16,fontname.Length,fontname,0));
            add_tag_to_subfile(fontrec,new_tag(0x17,sizeof(UInt32),null,0));
            add_tag_to_subfile(fontrec,new_tag(0x18,sizeof(UInt32),null,0xff));
            add_tag_to_subfile(fontrec,new_tag(0x19,sizeof(UInt16),null,0x19));
            add_tag_to_subfile(fontrec,new_tag(0x1a,sizeof(UInt16),null,0));
            add_tag_to_subfile(fontrec,new_tag(0x1b,sizeof(UInt16),null,0x8C));
            add_tag_to_subfile(fontrec,new_tag(0x1c,sizeof(UInt16),null,0x0a));
            add_tag_to_subfile(fontrec,new_tag(0x1d,sizeof(UInt16),null,0));
            add_tag_to_subfile(fontrec,new_tag(0x1e,sizeof(UInt16),null,0));
            add_tag_to_subfile(fontrec,new_tag(0xf1,sizeof(UInt16),null,2));
            add_tag_to_subfile(fontrec,new_tag(0xf2,sizeof(UInt32),null,0));
            add_tag_to_subfile(fontrec,new_tag(0x3c,sizeof(UInt16),null,1));
            add_tag_to_subfile(fontrec,new_tag(0x3d,sizeof(UInt16),null,1));
            add_tag_to_subfile(fontrec,new_tag(0x3e,sizeof(UInt16),null,0));
            add_tag_to_subfile(fontrec,new_tag(0x75,sizeof(UInt16),null,1));
            tail.next = fontrec;
            tail = tail.next;
            font_id = fontrec.id;

            book = new_subfile(0x01);
            /* will be filled in later */
            tail.next = book;
            tail = tail.next;

            add_tag_to_subfile(head, new_tag(0x7B,sizeof(UInt32),null,book.id));
           

            margins = new_subfile(0x05);
            /* Margins */
            add_tag_to_subfile(margins,new_tag(0x21,sizeof(UInt16),null,5));
            add_tag_to_subfile(margins,new_tag(0x22,sizeof(UInt16),null,0x35));
            add_tag_to_subfile(margins,new_tag(0x23,sizeof(UInt16),null,5));
            add_tag_to_subfile(margins,new_tag(0x24,sizeof(UInt16),null,0x2a));
            add_tag_to_subfile(margins,new_tag(0x2C,sizeof(UInt16),null,0x2a));
            add_tag_to_subfile(margins,new_tag(0x25,sizeof(UInt16),null,0x2a2));
            add_tag_to_subfile(margins,new_tag(0x26,sizeof(UInt16),null,0x204));
            add_tag_to_subfile(margins,new_tag(0x27,sizeof(UInt16),null,0x3a));
            add_tag_to_subfile(margins,new_tag(0x28,sizeof(UInt16),null,0x35));
            add_tag_to_subfile(margins,new_tag(0x35,sizeof(UInt16),null,0x34));
            add_tag_to_subfile(margins,new_tag(0x2b,sizeof(UInt16),null,0));
            add_tag_to_subfile(margins,new_tag(0x2a,sizeof(UInt16),null,1));
            add_tag_to_subfile(margins,new_tag(0xda,sizeof(UInt16),null,2));

            byte[] sixbytes = new byte[] { 0, 0, 0, 0, 0, 0};
            sixbytes[0] = 1;
            add_tag_to_subfile(margins,new_tag(0x29,6,sixbytes,0));

            tail.next = margins;
            tail = tail.next;
            margins_id = margins.id;

            page_box = new_subfile(0x07);
            add_tag_to_subfile(page_box,new_tag(0x31,sizeof(UInt16),null,600));
            add_tag_to_subfile(page_box,new_tag(0x32,sizeof(UInt16),null,800));
            add_tag_to_subfile(page_box,new_tag(0x33,sizeof(UInt16),null,0x12));
            add_tag_to_subfile(page_box,new_tag(0x34,sizeof(UInt32),null,0xFF));
            add_tag_to_subfile(page_box,new_tag(0x35,sizeof(UInt16),null,0x34));
            add_tag_to_subfile(page_box,new_tag(0x36,sizeof(UInt16),null,0));
            add_tag_to_subfile(page_box,new_tag(0x37,sizeof(UInt32),null,0));
            add_tag_to_subfile(page_box,new_tag(0x2e,sizeof(UInt16),null,1));
            add_tag_to_subfile(page_box,new_tag(0x38,sizeof(UInt16),null,0));
            add_tag_to_subfile(page_box,new_tag(0x39,sizeof(UInt16),null,0));

            sixbytes = new byte[] { 0, 0, 0, 0, 0, 0};
            sixbytes[0] = 1;
            add_tag_to_subfile(page_box,new_tag(0x29,6,sixbytes,0));

            tail.next = page_box;
            tail = tail.next;
            page_box_id = page_box.id;
        }
        #endregion

        #region provide_metadata
        public void provide_metadata()
        {
            string bookid = impFile.BookProperties.ID;
            infoFile = new StringBuilder();
            infoFile.Append((char)0xFEFF);
            infoFile.Append("<?xml version=\"1.0\" encoding=\"UTF-16\" ?>\n");
            infoFile.Append("<Info version=\"1.0\" >\n");
            infoFile.Append("  <BookInfo>\n");
            infoFile.Append("    <Title>" + impFile.BookProperties.Title + "</Title>\n");
            infoFile.Append("    <Author>" + impFile.BookProperties.FirstName + "</Author>\n");
            infoFile.Append("    <BookID>" + bookid + "</BookID>\n");
            infoFile.Append("    <Publisher>ConvertIMPGUI</Publisher>\n");
            infoFile.Append("    <Label></Label>\n");
            infoFile.Append("    <Category>" + impFile.BookProperties.Bookshelf_Category + "</Category>\n");
            infoFile.Append("    <Classification></Classification>\n");
            infoFile.Append("  </BookInfo>\n");
            infoFile.Append("  <DocInfo>\n");
            infoFile.Append("    <Language>en</Language>\n");
            infoFile.Append("    <Creator>ConvertIMPGUI</Creator>\n");
            infoFile.Append("    <CreationDate>" + DateTime.Now.ToString("yyyy-MM-dd") + "</CreationDate>\n");
            infoFile.Append("    <Producer>ConvertIMPGUI</Producer>\n");
            infoFile.Append("    <Page>" + pages + "</Page>\n");
            infoFile.Append("  </DocInfo>\n");
            infoFile.Append("</Info>\n");
        }
        #endregion

        #region convert_to_utf16
        private string convert_to_utf16(string file)
        {
            byte[] p, q;
            int i, c;

            if (file.Length < 2)
                return null;
            p = ASCIIEncoding.ASCII.GetBytes(file);
            if (READ_LE_WORD(p, 0) == 0xFEFF)
                return null;
            q = new byte[p.Length * 2 + 2];
            WRITE_LE_WORD(ref q, 0, 0xFEFF);
            int pOffset = 0;
            int qOffset = 2;
            for (i = 0; i < file.Length; i++)
            {
                c = p[pOffset++];
                WRITE_LE_WORD(ref q, qOffset, (UInt16)c);
                qOffset += 2;
            }

            return ASCIIEncoding.ASCII.GetString(q);
        }
        #endregion

        #region add_txt_to_book
        public void add_txt_to_book(string txtFileName)
        {
            StreamReader sr = new StreamReader(txtFileName);
            string textfile = sr.ReadToEnd();
            sr.Close();

            string newtext = "";
            int page_len;
            byte[] pagefiles = new byte[1000 * 4 + 2];
            int nfile = 0;
            int chapterize = 0;
            byte[] page_ptr;
            SubFile page, box, para, physical_pages;

            textfile = convert_to_utf16(textfile);

            page_len = 0;

            WRITE_LE_DWORD(ref pagefiles, nfile * 4 + 2, (byte)font_id);

            nfile++;

            do
            {
                page_ptr = ASCIIEncoding.ASCII.GetBytes(textfile);
                page_len = textfile.Length;

                para = make_one_long_para(page_ptr, page_len);
                tail.next = para;
                tail = tail.next;

                /* Font Size */
                add_tag_to_subfile(para, new_tag(0x03, sizeof(UInt32), null, font_id));
                WRITE_LE_DWORD(ref pagefiles, nfile * 4 + 2, para.id);
                nfile++;

                /* the para goes into a bounding box */
                box = new_subfile(0x06);
                add_tag_to_subfile(box, new_tag(0x03, sizeof(UInt32), null, page_box_id));

                string boxdat = "" + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0;
                byte[] p = ASCIIEncoding.ASCII.GetBytes(boxdat);
                p[0] = 0x03;
                p[1] = 0xF5;
                WRITE_LE_DWORD(ref p, 2, para.id);
                boxdat = ASCIIEncoding.ASCII.GetString(p);
                box.data = boxdat;

                tail.next = box;
                tail = tail.next;

                WRITE_LE_DWORD(ref pagefiles, nfile * 4 + 2, box.id);
                nfile++;

                page = new_subfile(0x02);
                WRITE_LE_WORD(ref pagefiles, 0, (UInt16)nfile);

                /* page points back to book */
                add_tag_to_subfile(page, new_tag(0x7C, sizeof(UInt32), null, book.id));

                add_tag_to_subfile(page, new_tag(0x0b, nfile * 4 + 2, pagefiles, 0));
                add_tag_to_subfile(page, new_tag(0x03, sizeof(UInt32), null, margins_id));

                string pagedat = "" + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0;
                p = ASCIIEncoding.ASCII.GetBytes(pagedat);
                p[0] = 0x03;
                p[1] = 0xF5;
                WRITE_LE_DWORD(ref p, 2, box.id);
                pagedat = ASCIIEncoding.ASCII.GetString(p);
                page.data = pagedat;

                tail.next = page;
                tail = tail.next;

                pages++;

                physical_pages = new_subfile(0x1a);
                physical_pages.dataflags = 0x82;

                string laydat = "";
                for (int iii = 0; iii < 24; iii++)
                    laydat += "" + (char)0;
                p = ASCIIEncoding.ASCII.GetBytes(laydat);
                int pOffset = 0;
                WRITE_LE_DWORD(ref p, pOffset, 1);
                pOffset += sizeof(UInt32);
                WRITE_LE_DWORD(ref p, pOffset, box.id);
                pOffset += sizeof(UInt32);
                laydat = ASCIIEncoding.ASCII.GetString(p);
                physical_pages.data = laydat;

                tail.next = physical_pages;
                tail = tail.next;

                add_tag_to_subfile(page, new_tag(0x02, sizeof(UInt32), null, physical_pages.id));
            } while (false);
        }
        #endregion

        #region make_one_long_para
        private SubFile make_one_long_para(byte[] page_ptr, int byteCount)
        {
            SubFile psub;
            byte[] ppara;
            byte[] p, q;
            char s;
            UInt16 c;
            UInt32 j;
            int qOffset = 0;
            int pOffset = 0;

            psub = new_subfile(0x0A);
            ppara = new byte[byteCount + 6 + 4 + 10];
            p = ppara;
            q = page_ptr;

            c = READ_LE_WORD(q, qOffset);
            if (c == 0xFEFF)
                qOffset += 2;
            p[pOffset++] = 0xA1;
            p[pOffset++] = 0xF5;
            WRITE_LE_DWORD(ref p, pOffset, 0);
            pOffset += sizeof(UInt32);

            for (j = 0; j < byteCount / 2; j++)
            {
                c = READ_LE_WORD(q, qOffset);
                qOffset += 2;
                if ((char)c == '\r' || c == 0)
                    continue;
                if ((char)c == '\n')
                    c = 0xF5D2;
                WRITE_LE_WORD(ref p, pOffset, c);
                pOffset += 2;
            }

            p[pOffset++] = 0xA2;
            p[pOffset++] = 0xF5;

            int pparaLength = pOffset - ppara.Length;
            if (pparaLength > 64)
            {
                ppara = compress_sptr(ppara, pparaLength);
                psub.dataflags |= 0x100;
            }

            psub.data = ASCIIEncoding.ASCII.GetString( ppara);
            return psub;
        }
        #endregion

        #region compress_sptr
        private byte[] compress_sptr(byte[] bytes, int len)
        {
            int newlen = len + ((len / 1000) + 1) + 12 + 4;
            byte[] outBuf = new byte[newlen + sizeof(UInt32)];

            WRITE_LE_DWORD(ref outBuf, 0, (UInt16)len);

            MemoryStream ms = new MemoryStream();
            System.IO.Compression.DeflateStream ds = new System.IO.Compression.DeflateStream(ms, System.IO.Compression.CompressionMode.Compress, true);
            ds.Write(bytes, 0, len);
            ds.Flush();

            ms.Seek(0, SeekOrigin.Begin);
            ms.Read(outBuf, 0, newlen);

            return outBuf;
        }
        #endregion

        #region write_book_to_file
        public void write_book_to_file(string outfile)
        {
            string newmeta = "";
            byte[] bookpageslist;
            SubFile pagenums;
            string pagenumsdata = "";
            char[] pagetext = new char[10];
            int i;
            SubFile ps;
            byte[] p, q;
            int pOffset = 0;
            int qOffset = 0;

            p = new byte[10];
            WRITE_LE_DWORD(ref p, pOffset, 1);
            pOffset += 4;

            q = new byte[6];
            WRITE_LE_WORD(ref q, qOffset, 1);
            qOffset += 2;

            ps = head;
            i = 0;

            while (ps != null)
            {
                if (ps.type == 2)
                {
                    WRITE_LE_DWORD(ref p, pOffset, ps.id);
                    pOffset += 4;
                    WRITE_LE_WORD(ref p, pOffset, 1);
                    pOffset += 2;
                    WRITE_LE_DWORD(ref q, qOffset, ps.id);
                    qOffset += 4;
                    i++;
                }
                ps = ps.next;
            }

            pagenums = new_subfile(0x1A);
            pagenums.dataflags = 0x81;
            pagenums.data = ASCIIEncoding.ASCII.GetString(p);
            tail.next = pagenums;
            tail = tail.next;

            add_tag_to_subfile(book, new_tag(0x02,sizeof(UInt32),null,pagenums.id));
            add_tag_to_subfile(book, new_tag(0x5C,pages*4 + 2,q,0));

            write_lrf_file(outfile, infoFile.ToString(), null, head);
        }
        #endregion

        #region write_lrf_file
        public void write_lrf_file(string outfile, string metaText, byte[] image, SubFile subfilelist)
        {
            byte[] lrfheader = new byte[0x54];
            UInt32 running;
            int nsubfiles;
            int newlen;
            int needed;
            string dataptr;
            int fd = -1;
            int error = -1;
            int zstatus;
            byte[] align = new byte[16];
            SubFile subp;

            for (int i = 0; i < 16; i++)
                align[i] = 0;

            for (int i = 0; i < 0x54; i++)
                lrfheader[i] = 0;

            running = (UInt32)lrfheader.Length;
            lrfheader[0] = (byte)'L';
            lrfheader[2] = (byte)'R';
            lrfheader[4] = (byte)'F';

            WRITE_LE_WORD(ref lrfheader, LRF_VERSION, 0x3E7);
            WRITE_LE_WORD(ref lrfheader, LRF_PSUEDOKEY, 0x30);

            lrfheader[LRF_DIRECTION] = (byte)LRF_DIRECTION_FORWARDS;

            WRITE_LE_WORD(ref lrfheader, LRF_UNK1, 800*2);
            WRITE_LE_WORD(ref lrfheader, LRF_UNK2, 600);
            WRITE_LE_WORD(ref lrfheader, LRF_UNK3, 800);

            lrfheader[LRF_UNK4] = 0x18;

            WRITE_LE_WORD(ref lrfheader, LRF_UNK5, 0x1536);

            lrfheader[LRF_UNK6] = 0x14;

            dataptr = null;

            do
            {
                BinaryWriter bw = new BinaryWriter(File.Create(outfile));
                bw.Seek((int)running, SeekOrigin.Begin);

                newlen = metaText.Length + ((metaText.Length / 1000) + 1) + 12 + 4;
                dataptr = new string((char)0, newlen + sizeof(UInt32));
                byte[] bdataptr = ASCIIEncoding.ASCII.GetBytes(dataptr);
                WRITE_LE_DWORD(ref bdataptr, 0, (UInt16)metaText.Length);
                byte[] cc = compress_sptr(ASCIIEncoding.ASCII.GetBytes(metaText), metaText.Length);
                newlen += sizeof(UInt32);

                bw.Write(cc, 0, newlen);

                running += (UInt32)newlen;

                WRITE_LE_DWORD(ref lrfheader, LRF_INFOLEN, (UInt16)newlen);

                needed = (int)((16 - (running & 0xF)) & 0x0F);

                bw.Write(align, 0, needed);

                running += (UInt32)needed;

                subp = subfilelist;
                nsubfiles = 0;

                while (subp != null)
                {
                    int size;
                    byte[] tagbuffer = new byte[18];
                    Tag tagp;

                    nsubfiles++;
                    if (subp.type == 0x1C)
                        WRITE_LE_WORD(ref lrfheader, LRF_HEADSUBFILE, (UInt16)subp.id);
                    if (subp.type == 0x1E)
                        WRITE_LE_WORD(ref lrfheader, LRF_ALTSUBFILE, (UInt16)subp.id);

                    tagbuffer[0] = 0x0;
                    tagbuffer[1] = 0xF5;

                    WRITE_LE_DWORD(ref tagbuffer, 2, (UInt16)subp.id);
                    WRITE_LE_DWORD(ref tagbuffer, 6, subp.type);

                    bw.Write(tagbuffer, 0, 8);

                    size = 8;

                    tagp = subp.taglist;

                    while (tagp != null)
                    {
                        int len = 2;

                        tagbuffer[0] = tagp.tagid;
                        tagbuffer[1] = 0xF5;

                        if (tagp.len <= 16)
                        {
                            Array.Copy(tagp.shortdata, 0, tagbuffer, 2, tagp.len);
                            len += tagp.len;
                        }

                        bw.Write(tagbuffer, 0, len);

                        size += len;

                        if (tagp.data != null)
                        {
                            bw.Write(ASCIIEncoding.ASCII.GetBytes(tagp.data), 0, tagp.data.Length);
                            size += tagp.data.Length;
                        }
                        tagp = tagp.next;
                    }

                    if (tagp != null)
                        break;

                    if (subp.data != null)
                    {
                        tagbuffer[0] = 0x54;
                        tagbuffer[1] = 0xF5;
                        WRITE_LE_WORD(ref tagbuffer, 2, subp.dataflags);
                        tagbuffer[4] = 0x04;
                        tagbuffer[5] = 0xF5;
                        WRITE_LE_WORD(ref tagbuffer, 6, (UInt16)subp.data.Length);
                        tagbuffer[10] = 0x05;
                        tagbuffer[11] = 0xF5;

                        bw.Write(tagbuffer, 0, 12);

                        size += 12;

                        bw.Write(ASCIIEncoding.ASCII.GetBytes(subp.data), 0, subp.data.Length);

                        size += subp.data.Length;

                        tagbuffer[0] = 0x06;
                        tagbuffer[1] = 0xF5;

                        bw.Write(tagbuffer, 0, 2);
                        size += 2;
                    }

                    tagbuffer[0] = 0x01;
                    tagbuffer[1] = 0xF5;
                    bw.Write(tagbuffer, 0, 2);
                    size += 2;

                    subp.location = running;
                    running += (UInt32)size;
                    subp.size = size;
                    subp = subp.next;
                }

                if (subp != null)
                    break;

                needed = (UInt16)((16 - (running & 0xF)) & 0x0F);

                bw.Write(align, 0, needed);

                running += (UInt32)needed;

                WRITE_LE_DWORD(ref lrfheader, LRF_SUBFILECOUNT, (UInt16)nsubfiles);
                WRITE_LE_DWORD(ref lrfheader, LRF_SUBFILEOFFSET, running);

                subp = subfilelist;

                while (subp != null)
                {
                    byte[] subrec = new byte[16];

                    WRITE_LE_DWORD(ref subrec, 12, 0);
                    WRITE_LE_DWORD(ref subrec, 0, subp.id);
                    WRITE_LE_DWORD(ref subrec, 4, subp.location);
                    WRITE_LE_DWORD(ref subrec, 8, (UInt16)subp.size);

                    bw.Write(subrec, 0, 16);

                    running += (UInt16)16;

                    subp = subp.next;
                }

                if (subp != null)
                    break;

                bw.Seek(0, SeekOrigin.Begin);
                bw.Write(lrfheader, 0, lrfheader.Length);
                bw.Close();
            } while (false);
        }
        #endregion
    }

    #region Class Tag
    public class Tag
    {
        public Tag next;
        public byte tagid;
        public string data;
        public int len;
        public byte[] shortdata = new byte[16];
    }
    #endregion

    #region Class SubFile
    public class SubFile
    {
        public SubFile next;
        public int size;
        public UInt32 location;
        public UInt32 id;
        public UInt16 type;
        public UInt16 dataflags;
        public string data;
        public Tag taglist;
    }
    #endregion
}
