﻿using System;
using System.Collections.Generic;
using System.Text;

using System.IO;
using System.Windows.Forms;

namespace ConvertIMPGUI
{
    public class CIMPFile
    {
        #region Class encoded_string_t
        class encoded_string_t
        {
            public int offset;
            public int length;
        }
        #endregion

        #region Class bitStream
        class bitStream
        {
            public int bitCount = 0;
            public int bitBuffer = 0;
        }
        #endregion

        public bool REB1200 = false;
        public int numberOfPages = 1;

        private IMPHeader _header = new IMPHeader();
        private IMPBookProperties _bookProperties = new IMPBookProperties();
        private List<IMPTableContentsItem> _tableOfContents = new List<IMPTableContentsItem>();
        private List<IMPRESFile> _resFiles = new List<IMPRESFile>();
        private List<SWIndex> _swIndexes = new List<SWIndex>();
        private List<IMGRunItem> _imgRuns = new List<IMGRunItem>();
        private List<IMPStyleItem> _styles = new List<IMPStyleItem>();
        private List<IMPExtendedStyleItem> _extStyles = new List<IMPExtendedStyleItem>();
        private List<IMPStyleRunItem> _styleRuns = new List<IMPStyleRunItem>();
        private List<IMPPPic> _pPics = new List<IMPPPic>();
        private List<IMPPcZ1> _Pcz1 = new List<IMPPcZ1>();
        private List<IMPPcZ1> _PcZ1 = new List<IMPPcZ1>();
        private List<IMPTableItem> _tables = new List<IMPTableItem>();
        private List<IMPTableRowGroup> _tableRows = new List<IMPTableRowGroup>();
        private List<IMPTableCellGroup> _tableRowCells = new List<IMPTableCellGroup>();
        private string IMPResourceFile = "";

        private int EOF = -1;
        public string IMPFilePath = "";

        #region Constructors
        public CIMPFile() { }

        public CIMPFile(string impFile)
        {
            IMPFilePath = impFile;
        }
        #endregion

        #region Properties

        #region NumberOfPages
        public int NumberOfPages
        {
            get { return numberOfPages; }
        }
        #endregion

        #region Header
        public IMPHeader Header
        {
            get { return _header; }
            set { _header = value; }
        }
        #endregion

        #region BookProperties
        public IMPBookProperties BookProperties
        {
            get { return _bookProperties; }
            set { _bookProperties = value; }
        }
        #endregion

        #region TableOfContents
        public List<IMPTableContentsItem> TableOfContents
        {
            get { return _tableOfContents; }
            set { _tableOfContents = value; }
        }
        #endregion

        #region RESFiles
        public List<IMPRESFile> RESFiles
        {
            get { return _resFiles; }
            set { _resFiles = value; }
        }
        #endregion

        #region SWIndexes
        public List<SWIndex> SWIndexes
        {
            get { return _swIndexes; }
            set { _swIndexes = value; }
        }
        #endregion

        #region ImageRuns
        public List<IMGRunItem> ImageRuns
        {
            get { return _imgRuns; }
            set { _imgRuns = value; }
        }
        #endregion

        #region TextStyles
        public List<IMPStyleItem> TextStyles
        {
            get { return _styles; }
            set { _styles = value; }
        }
        #endregion

        #region TextExtendedStyles
        public List<IMPExtendedStyleItem> TextExtendedStyles
        {
            get { return _extStyles; }
            set { _extStyles = value; }
        }
        #endregion

        #region StyleRuns
        public List<IMPStyleRunItem> StyleRuns
        {
            get { return _styleRuns; }
            set { _styleRuns = value; }
        }
        #endregion

        #region PPics
        public List<IMPPPic> PPics
        {
            get { return _pPics; }
            set { _pPics = value; }
        }
        #endregion

        #region Pcz1
        public List<IMPPcZ1> Pcz1
        {
            get { return _Pcz1; }
            set { _Pcz1 = value; }
        }
        #endregion

        #region PcZ1
        public List<IMPPcZ1> PcZ1
        {
            get { return _PcZ1; }
            set { _PcZ1 = value; }
        }
        #endregion

        #region Tables
        public List<IMPTableItem> Tables
        {
            get { return _tables; }
            set { _tables = value; }
        }
        #endregion

        #region TableRows
        public List<IMPTableRowGroup> TableRows
        {
            get { return _tableRows; }
            set { _tableRows = value; }
        }
        #endregion

        #region TableRowCells
        public List<IMPTableCellGroup> TableRowCells
        {
            get { return _tableRowCells; }
            set { _tableRowCells = value; }
        }
        #endregion

        #region Content
        public string Content
        {
            get { return IMPResourceFile; }
        }
        #endregion

        #endregion

        #region Private Methods used int Reversing IMP File

        #region ReadInt
        private int ReadInt(ref BinaryReader br, int count)
        {
            int nResult = 0;
            byte[] buffer = br.ReadBytes(count);

            for (int i = 0; i < count; i++)
                nResult = (nResult << 8) + buffer[i];

            return nResult;
        }
        #endregion

        #region ReadString
        private string ReadString(ref BinaryReader br)
        {
            string strResult = "";

            char ch = ' ';

            while (ch != '\0')
            {
                strResult += ch;
                ch = br.ReadChar();
            }

            return strResult.Trim();
        }
        #endregion

        #region GetInt
        private int GetInt(byte[] buffer, int start, int count)
        {
            int nResult = 0;

            for (int i = start; i < start + count; i++)
                nResult = (nResult << 8) + buffer[i];

            return nResult;
        }
        #endregion

        #region GetString
        private string GetString(byte[] buffer, int start, int count)
        {
            string strResult = "";

            for (int i = start; i < start + count; i++)
                strResult += "" + (char)buffer[i];

            return strResult.Trim();
        }
        #endregion

        #region GetString2
        private string GetString2(byte[] buffer, int start, ref int count)
        {
            string strResult = "";

            int i = start;
            char ch = ' ';

            while (ch != '\0')
            {
                strResult += ch;
                ch = (char)buffer[i];
                i++;
            }

            count = i;
            return strResult.Trim();
        }
        #endregion

        #region ReadOffset
        private int ReadOffset(byte[] buffer)
        {
            return GetInt(buffer, 10, 4);
        }
        #endregion

        #region ReadSW
        private void ReadSW(int offsetToIndex, byte[] buffer)
        {
            int swCount = 0;

            if (!REB1200)
                swCount = (buffer.Length - offsetToIndex) / 18;
            else
                swCount = (buffer.Length - offsetToIndex) / 16;

            int offset = offsetToIndex;

            for (int j = 0; j < swCount; j++)
            {
                SWIndex swIndex = new SWIndex();

                if (REB1200)
                {
                    swIndex.sequenceNumber = GetInt(buffer, offset, 2);
                    offset += 2;
                    swIndex.lengthOfIndex = GetInt(buffer, offset, 4);
                    offset += 4;
                    swIndex.offsetToStartOfIndex = GetInt(buffer, offset, 4);
                    offset += 4;
                    offset += 2;
                    swIndex.fileType = GetString(buffer, offset, 4);
                    offset += 4;
                }
                else
                {
                    swIndex.sequenceNumber = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    int len1 = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    int len2 = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    int offset1 = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    int offset2 = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    offset += 2;
                    offset += 2;
                    swIndex.fileType = GetString(buffer, offset, 4);
                    offset += 4;

                    swIndex.lengthOfIndex = len1 * 65536 + len2;
                    swIndex.offsetToStartOfIndex = offset1 * 65536 + offset2;
                }

                _swIndexes.Add(swIndex);
            }
        }
        #endregion

        #region ReadImRn
        private void ReadImRn(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);
            int imrn_count = (indexOffset - 32) / 32;

            if (!REB1200)
            {
                imrn_count = (indexOffset - 32 + 2) / 36;
            }

            int offset = 32;

            for (int j = 0; j < imrn_count; j++)
            {
                if (REB1200)
                {
                }
                else
                {
                    IMGRunItem item = new IMGRunItem();

                    offset += 8;
                    item.imageWidth = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    item.imageHeight = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    offset += 4;
                    offset += 2;
                    offset += 2;
                    int offset1 = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    int offset2 = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    item.dataOffset = offset2 * 65536 + offset1;

                    item.unknown1 = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    item.unknown2 = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    item.imageFileType = "" + (char)buffer[offset + 3];
                    item.imageFileType += "" + (char)buffer[offset + 2];
                    item.imageFileType += "" + (char)buffer[offset + 1];
                    item.imageFileType += "" + (char)buffer[offset];
                    offset += 4;
                    item.imageID = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;

                    if (j != imrn_count - 1)
                        offset += 2;

                    _imgRuns.Add(item);
                }
            }
        }
        #endregion

        #region ReadStyle
        private void ReadStyle(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int data_indexcount = (offsetToIndex - 32) / 48;

            if (REB1200)
                data_indexcount = (offsetToIndex - 32) / 46;

            int offset = 32;

            for(int j = 0; j < data_indexcount; j++)
            {
                IMPStyleItem item = new IMPStyleItem();

                offset += 2; // UNKNOWN

                // Element
                if (REB1200)
                    item.element = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                else
                    item.element = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                offset += 2;

                offset += 2; // UNKNOWN

                // Font Family
                if (REB1200)
                    item.fontFamily = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                else
                    item.fontFamily = string.Format("0x{0:X2}{1:X2}", buffer[offset + 1], buffer[offset]);
                if (item.fontFamily == "0x0014")
                    item.fontFamilyName = "Serif";
                else if (item.fontFamily == "0x0015")
                    item.fontFamilyName = "Sans-Serif";
                else if (item.fontFamily == "0x0003")
                    item.fontFamilyName = "Small Font";
                else if (item.fontFamily == "0x0004")
                    item.fontFamilyName = "Mono Space";
                else
                    item.fontFamilyName = "Default";
                offset += 2;

                // Font Style
                if (REB1200)
                    item.style = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                else
                    item.style = string.Format("0x{0:X2}{1:X2}", buffer[offset + 1], buffer[offset]);
                int style = 0;
                if (REB1200)
                    style = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    style = (buffer[offset + 1] << 8) + buffer[offset];
                if (style == 1)
                    item.style = "font-weight:bold";
                else if (style == 2)
                    item.style = "font-style:italic";
                else if (style == 3)
                    item.style = "font-weight:bold; font-style:italic";
                else if (style == 4)
                    item.style = "text-decoration:underline";
                else if (style == 5)
                    item.style = "font-weight:bold;text-decoration:underline";
                else if (style == 6)
                    item.style = "font-style:italic;text-decoration:underline";
                else if (item.style == "0x0080")
                    item.style = "text-decoration:underline";
                else if (item.style == "0x0000")
                    item.style = "";
                offset += 2;

                // Font Size
                if (REB1200)
                    item.fontRealSize = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.fontRealSize = (buffer[offset + 1] << 8) + buffer[offset];

                if (item.fontRealSize == 1)
                    item.fontSize = "xx-small";
                if (item.fontRealSize == 2)
                    item.fontSize = "x-small";
                if (item.fontRealSize == 3)
                    item.fontSize = "small";
                if (item.fontRealSize == 4)
                    item.fontSize = "medium";
                if (item.fontRealSize == 5)
                    item.fontSize = "large";
                if (item.fontRealSize == 6)
                    item.fontSize = "x-large";
                if (item.fontRealSize == 7)
                    item.fontSize = "xx-large";

                offset += 2;

                // Text Align
                if (REB1200)
                    item.textAlign = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                else
                    item.textAlign = string.Format("0x{0:X2}{1:X2}", buffer[offset + 1], buffer[offset]);
                if (item.textAlign == "0x0000")
                    item.textAlign = "Left";
                else if (item.textAlign == "0xFFFE")
                    item.textAlign = "Left";
                else if (item.textAlign == "0xFFFF")
                    item.textAlign = "Right";
                else if (item.textAlign == "0x0001")
                    item.textAlign = "Center";
                else if (item.textAlign == "0xFFFD")
                    item.textAlign = "Justified";
                else
                    item.textAlign = "Left";

                offset += 2;

                // Text Color
                item.textColorRed = (int)buffer[offset++];
                item.textColorGreen = (int)buffer[offset++];
                item.textColorBlue = (int)buffer[offset++];
                item.textColor = System.Drawing.Color.FromArgb(item.textColorRed, item.textColorGreen, item.textColorBlue);

                // Text BGColor
                item.textBGColorRed = (int)buffer[offset++];
                item.textBGColorGreen = (int)buffer[offset++];
                item.textBGColorBlue = (int)buffer[offset++];
                item.textBGColor = System.Drawing.Color.FromArgb(item.textBGColorRed, item.textBGColorGreen, item.textBGColorBlue);

                // Margin Top  if 0xFFFF then not defined make 0
                if (REB1200)
                    item.marginTop = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.marginTop = (buffer[offset + 1] << 8) + buffer[offset];
                if (item.marginTop == (0xFFFF))
                    item.marginTop = 0;
                offset += 2;

                // Text Indent
                if (REB1200)
                    item.textIndent = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.textIndent = (buffer[offset + 1] << 8) + buffer[offset];
                offset += 2;

                // Margin Right
                if (REB1200)
                    item.marginRight = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.marginRight = (buffer[offset + 1] << 8) + buffer[offset];
                offset += 2;

                // Margin Left
                if (REB1200)
                    item.marginLeft = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.marginLeft = (buffer[offset + 1] << 8) + buffer[offset];
                offset += 2;

                offset += 2; // UNKNOWN

                // OEB Column Number
                if (REB1200)
                    item.oebColumnNumber = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.oebColumnNumber = (buffer[offset + 1] << 8) + buffer[offset];
                offset += 2;

                if (REB1200)
                    offset += 14; // UNKNOWN
                else
                    offset += 16; // UNKNOWN

                _styles.Add(item);
            }
        }
        #endregion

        #region ReadExtendedStyle
        private void ReadExtendedStyle(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int dataLength = buffer.Length - 32;
            int indexes = (buffer.Length - indexOffset) / 14;

            if (REB1200)
                indexes = (buffer.Length - indexOffset) / 12;

            int offset = 32;

            int Res1A = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res1B = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res1C = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;

            int Res11 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res12 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res13 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res14 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;

            int Res21 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res22 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res23 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res24 = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;

            int Res2A = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res2B = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;
            int Res2C = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3]; ;
            offset += 4;

            IMPExtendedStyleItem item = new IMPExtendedStyleItem();
            item.Resource1 = string.Format("0x{0:X4} 0x{1:X4} 0x{2:X4}", Res1A, Res1B, Res1C);
            item.OrphanPull = string.Format("0x{0:X4} 0x{1:X4} 0x{2:X4} 0x{3:X4}", Res11, Res12, Res13, Res14);
            item.WidowPush = string.Format("0x{0:X4} 0x{1:X4} 0x{2:X4} 0x{3:X4}", Res21, Res22, Res23, Res24);
            item.Resource2 = string.Format("0x{0:X4} 0x{1:X4} 0x{2:X4}", Res2A, Res2B, Res2C);
            _extStyles.Add(item);
        }
        #endregion

        #region ReadStyleRun
        private void ReadStyleRun(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);
            int data_indexcount = (offsetToIndex - 32) / 8;

            int offset = 32;

            for (int j = 0; j < data_indexcount; j++)
            {
                IMPStyleRunItem item = new IMPStyleRunItem();

                if (REB1200)
                {
                    item.Offset = GetInt(buffer, offset, 4);
                    offset += 4;
                    item.StyleIndex = GetInt(buffer, offset, 4);
                    offset += 4;
                }
                else
                {
                    item.Offset = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                    item.StyleIndex = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                }

                _styleRuns.Add(item);
            }
        }
        #endregion

        #region ReadPPic
        private void ReadPPic(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int dataLength = buffer.Length - 32;
            int indexes = (buffer.Length - indexOffset) / 14;

            if (REB1200)
                indexes = (buffer.Length - indexOffset) / 12;

            int ppic_count = (indexOffset - 32) / 18;

            int offset = 32;
            for (int j = 0; j < ppic_count; j++)
            {
                IMPPPic item = new IMPPPic();

                offset += 2;

                item.CountOfCellAndTableBorders = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                offset += 4;
                offset += 4;
                item.CountOfImages = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                offset += 4;
                offset += 4;

                _pPics.Add(item);
            }
        }
        #endregion

        #region ReadPcZ1
        private void ReadPcZ1(string fileType, int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int dataLength = buffer.Length - 32;
            int indexes = (buffer.Length - indexOffset) / 14;

            string view = "";
            if (fileType == "Pcz1")
                view = "Small View";
            else
                view = "Large View";

            int const1 = (buffer[0] >> 8) + buffer[1];
            int indexCount = 0;
            int count = 0;

            if (const1 == 2)
            {
                indexCount = (buffer.Length - offsetToIndex) / 16;
                count = offsetToIndex - 32;
                if (!REB1200)
                    indexCount = (buffer.Length - offsetToIndex) / 18;

                int offset = 32;
                for (int j = 0; j < indexCount; j++)
                {
                    IMPPcZ1 item = new IMPPcZ1();
                    item.View = view;
                    item.ID = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    offset += 2;

                    item.Length = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;

                    offset += 4;

                    item.Offset = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;

                    offset += 2;

                    if (fileType == "Pcz1")
                        _Pcz1.Add(item);
                    else
                        _PcZ1.Add(item);
                }
            }
        }
        #endregion

        #region ReadTabl
        private void ReadTabl(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int dataLength = buffer.Length - 32;
            int indexes = 0;

            if (REB1200)
                indexes = (buffer.Length - indexOffset) / 12;
            else
                indexes = (buffer.Length - indexOffset) / 14;

            int offset = 32;

            while (true)
            {
                IMPTableItem item = new IMPTableItem();

                if (REB1200)
                    item.Align = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                else
                    item.Align = string.Format("0x{0:X2}{1:X2}", buffer[offset + 1], buffer[offset]);
                if (item.Align == "0xFFFA")
                    item.AlignType = "";
                if (item.Align == "0x0000")
                    item.AlignType = "center";
                if (item.Align == "0xFFFF")
                    item.AlignType = "right";
                if (item.Align == "0xFFFE")
                    item.AlignType = "left";
                if (item.Align == "0xFFFD")
                    item.AlignType = "justified";
                offset += 2;

                string s = "";
                if (REB1200)
                    s = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                else
                    s = string.Format("0x{0:X2}{1:X2}", buffer[offset + 1], buffer[offset]);

                if (s == "0xFFFF")
                    item.Width = -1;
                else if (s == "0xFFA5")
                    item.Width = 90;
                else
                {
                    if (REB1200)
                    {
                        item.Width = (buffer[offset] << 8) + buffer[offset + 1];
                        item.Width = (int)(((decimal)item.Width / (decimal)0xFFFF) * 100M);
                    }
                    else
                    {
                        item.Width = (buffer[offset + 1] << 8) + buffer[offset];
                        item.Width = (int)(((decimal)item.Width / (decimal)0xFFFF) * 100M);
                    }
                }
                offset += 2;

                if (REB1200)
                    item.Border = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.Border = (buffer[offset + 1] << 8) + buffer[offset];
                if (item.Border == 0xFFFF)
                    item.Border = 2;
                else
                    item.Border = 1;
                offset += 2;

                if (REB1200)
                    item.CellSpacing = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.CellSpacing = (buffer[offset + 1] << 8) + buffer[offset];
                if (item.CellSpacing == 0xFFFF)
                    item.CellSpacing = -1;
                offset += 2;

                if (REB1200)
                    item.CellPadding = (buffer[offset] << 8) + buffer[offset + 1];
                else
                    item.CellPadding = (buffer[offset + 1] << 8) + buffer[offset];
                if (item.CellPadding == 0xFFFF)
                    item.CellPadding = -1;
                offset += 2;

                if (REB1200)
                    s = string.Format("0x{0:X2}{1:X2}{2:X2}{3:X2}", buffer[offset + 3], buffer[offset + 2], buffer[offset + 1], buffer[offset]);
                else
                {
                    offset += 2;
                    s = string.Format("0x{0:X2}{1:X2}{2:X2}{3:X2}", buffer[offset], buffer[offset + 1], buffer[offset + 2], buffer[offset + 3]);
                }
                if (s == "0xFFFFFFFF")
                    item.CaptionPresent = false;
                else
                    item.CaptionPresent = true;
                offset += 4;

                if (REB1200)
                    item.CaptionLength = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                else
                    item.CaptionLength = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                offset += 4;

                offset += 2;

                if (REB1200)
                    item.ListStyleType = string.Format("0x{0:X2}{1:X2}", buffer[offset + 1], buffer[offset]);
                else
                    item.ListStyleType = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                offset += 2;

                if (REB1200)
                    item.TRowID = (buffer[offset + 1] << 8) + buffer[offset];
                else
                    item.TRowID = (buffer[offset] << 8) + buffer[offset + 1];
                offset += 2;

                if (!REB1200)
                    offset += 2;

                _tables.Add(item);

                if (offset + 24 > indexOffset)
                    break;
            }
        }
        #endregion

        #region ReadTRow
        private void ReadTRow(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int dataLength = buffer.Length - 32;
            int indexes = 0;

            if (REB1200)
                indexes = (buffer.Length - indexOffset) / 12;
            else
                indexes = (buffer.Length - indexOffset) / 18;

            int offset = indexOffset;
            List<IMPResource> resources = new List<IMPResource>();

            for (int i = 0; i < indexes; i++)
            {
                IMPResource item = new IMPResource();

                if (REB1200)
                {
                    item.id = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    item.length = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                    offset += 4;
                    item.offset = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                    offset += 4;
                    offset += 2;
                }
                else
                {
                    item.index = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    offset += 2;
                    item.length = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                    offset += 4;
                    item.offset = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 2;
                    offset += 4;
                }

                resources.Add(item);
            }

            offset = 32;

            if (REB1200)
            {
                foreach (IMPResource res in resources)
                {
                    IMPTableRowGroup rowGroup = new IMPTableRowGroup();
                    rowGroup.RowID = res.id;

                    offset = res.offset;

                    while (offset < res.offset + res.length)
                    {
                        IMPTableRowItem item = new IMPTableRowItem();

                        item.TRowID = res.id;

                        item.RowType = string.Format("0x{0:X2}{1:X2}{2:X2}{3:X2}", buffer[offset], buffer[offset + 1], buffer[offset + 2], buffer[offset + 3]);
                        if (item.RowType == "0xFFFAFFFA")
                            item.RowType = "table row";
                        else
                            item.RowType = "list row";
                        offset += 4;

                        item.Border = (buffer[offset] << 8) + buffer[offset + 1];
                        if (item.Border == 0xFFFF)
                            item.Border = 2;
                        else
                            item.Border = 1;
                        offset += 2;

                        item.TCelID = (buffer[offset] << 8) + buffer[offset + 1];
                        offset += 2;

                        item.Offset = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                        offset += 4;

                        item.Length = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                        offset += 4;

                        rowGroup.Rows.Add(item);
                    }

                    _tableRows.Add(rowGroup);
                }
            }
            else
            {
                for (int j = 0; j < (indexOffset - 32) / 18; j++)
                {
                    IMPResource res = null;
                    foreach (IMPResource r in resources)
                    {
                        if (r.offset <= offset)
                        {
                            res = r;
                        }
                    }
                    IMPTableRowItem item = new IMPTableRowItem();

                    offset = res.offset;

                    item.TRowID = res.index;

                    offset += 2;

                    item.val1 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val2 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val3 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val4 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val5 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val6 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val7 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val8 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val9 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val10 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val11 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val12 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val13 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val14 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val15 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val16 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val17 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val18 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val19 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val20 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val21 = GetInt(buffer, offset, 2);
                    offset += 2;

                    item.val22 = GetInt(buffer, offset, 2);
                    offset += 2;

                    //_tableRows.Add(item);

                    if (offset + 18 >= indexOffset)
                        break;
                }
            }
        }
        #endregion

        #region ReadTCel
        private void ReadTCel(int offsetToIndex, byte[] buffer)
        {
            int indexOffset = GetInt(buffer, 10, 4);

            int dataLength = buffer.Length - 32;

            int indexes = 0;
            if (REB1200)
                indexes = (buffer.Length - indexOffset) / 12;
            else
                indexes = (buffer.Length - indexOffset) / 18;

            int offset = indexOffset;
            List<IMPResource> resources = new List<IMPResource>();

            for (int i = 0; i < indexes; i++)
            {
                IMPResource item = new IMPResource();
                if (!REB1200)
                {
                    item.index = (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                    item.length = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                    offset += 4; // Constant
                    item.offset = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 6;
                }
                else
                {
                    item.index = (buffer[offset] << 8) + buffer[offset + 1];
                    offset += 2;
                    item.length = GetInt(buffer, offset, 4); // (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                    item.offset = GetInt(buffer, offset, 4); //(buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                    offset += 4;
                    offset += 2;
                }
                resources.Add(item);
            }

            offset = 32;

            foreach (IMPResource res in resources)
            {
                IMPTableCellGroup cellGroup = new IMPTableCellGroup();
                cellGroup.CellID = res.id;

                offset = res.offset;

                while (offset < res.offset + res.length)
                {
                    IMPTableCellItem item = new IMPTableCellItem();

                    item.TCelID = res.index;

                    System.Diagnostics.Debug.WriteLine("Offset: " + offset);
                    System.Diagnostics.Debug.WriteLine("Length: " + res.length);

                    if (REB1200)
                    {
                        #region REB 1200
                        offset += 6;

                        item.CellType = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                        if (item.CellType == "0xFFFA")
                            item.CellType = "table cell";
                        else
                            item.CellType = "list cell";
                        offset += 2;

                        item.VerticalAlign = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                        if (item.VerticalAlign == "0xFFFA")
                            item.VerticalAlign = "middle";
                        else if (item.VerticalAlign == "0xFFFC")
                            item.VerticalAlign = "top";
                        else if (item.VerticalAlign == "0xFFFB")
                            item.VerticalAlign = "bottom";
                        else
                            item.VerticalAlign = "";
                        offset += 2;

                        item.Width = (buffer[offset] << 8) + buffer[offset + 1];
                        offset += 2;

                        item.Height = (buffer[offset] << 8) + buffer[offset + 1];
                        offset += 2;

                        item.Offset = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                        offset += 4;

                        item.Length = (buffer[offset] << 24) + (buffer[offset + 1] << 16) + (buffer[offset + 2] << 8) + buffer[offset + 3];
                        offset += 4;

                        string s = string.Format("0x{0:X2}{1:X2}{2:X2}{3:X2}", buffer[offset], buffer[offset + 1], buffer[offset + 2], buffer[offset + 3]);

                        if (s == "0xFFFFFFFF")
                        {
                            item.BGColorRed = 255;
                            item.BGColorGreen = 255;
                            item.BGColorBlue = 255;
                            item.BGColor = System.Drawing.Color.White;
                            offset += 4;
                        }
                        else
                        {
                            offset += 1;
                            item.BGColorRed = buffer[offset];
                            offset += 1;
                            item.BGColorGreen = buffer[offset];
                            offset += 1;
                            item.BGColorBlue = buffer[offset];
                            offset += 1;

                            item.BGColor = System.Drawing.Color.FromArgb(item.BGColorRed, item.BGColorGreen, item.BGColorBlue);
                        }
                        #endregion
                    }
                    else
                    {
                        #region EBW 1150
                        offset += 6; // Consts

                        item.CellType = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                        if (item.CellType == "0x7FC0")
                            item.CellType = "table cell";
                        else
                            item.CellType = "list cell";
                        offset += 2;

                        item.VerticalAlign = string.Format("0x{0:X2}{1:X2}", buffer[offset], buffer[offset + 1]);
                        if (item.VerticalAlign == "0xE001")
                            item.VerticalAlign = "middle";
                        else if (item.VerticalAlign == "0xE003")
                            item.VerticalAlign = "top";
                        else if (item.VerticalAlign == "0xE002")
                            item.VerticalAlign = "bottom";
                        else
                            item.VerticalAlign = "";
                        offset += 2;

                        item.Width = (buffer[offset + 1] << 8) + buffer[offset];
                        offset += 2;

                        item.Height = (buffer[offset + 1] << 8) + buffer[offset];
                        offset += 2;

                        item.Offset = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                        offset += 4;

                        item.Length = (buffer[offset + 3] << 24) + (buffer[offset + 2] << 16) + (buffer[offset + 1] << 8) + buffer[offset];
                        offset += 4;

                        string s = string.Format("0x{0:X2}{1:X2}{2:X2}{3:X2}", buffer[offset], buffer[offset + 1], buffer[offset + 2], buffer[offset + 3]);

                        if (s == "0xFFFFFFFF")
                        {
                            item.BGColorRed = 255;
                            item.BGColorGreen = 255;
                            item.BGColorBlue = 255;
                            item.BGColor = System.Drawing.Color.White;
                            offset += 4;
                        }
                        else
                        {
                            offset += 1;
                            item.BGColorRed = buffer[offset];
                            offset += 1;
                            item.BGColorGreen = buffer[offset];
                            offset += 1;
                            item.BGColorBlue = buffer[offset];
                            offset += 1;

                            item.BGColor = System.Drawing.Color.FromArgb(item.BGColorRed, item.BGColorGreen, item.BGColorBlue);
                        }
                        #endregion
                    }

                    cellGroup.Cells.Add(item);
                }
                _tableRowCells.Add(cellGroup);
            }
        }
        #endregion

        #region ReadBPgzBPgZ
        private void ReadBPgzBPgZ(int offsetToIndex, byte[] buffer)
        {
            int offset = 32;

            while (true)
            {
                IMPBPgzItem item = new IMPBPgzItem();

                if (offset + 32 >= buffer.Length)
                    break;
            }
        }
        #endregion

        #region ReadPNG
        static void ReadPNG(string fileName, int offsetToIndex, byte[] buffer)
        {
            BinaryWriter bw = new BinaryWriter(File.Create(@"C:\Temp\eBook\" + fileName + ".PNG"));

            bw.Write(buffer, 32, offsetToIndex - 32);

            bw.Close();
        }
        #endregion

        #region ResourceHasID
        public bool ResourceHasID(IMGRunItem imgItem, ref byte[] buffer)
        {
            int indexOffset = ReadOffset(buffer);

            int dataIndex = (buffer.Length - indexOffset) / 14;

            int offset = indexOffset;

            for (int j = 0; j < dataIndex; j++)
            {
                int imageid = (buffer[offset + 1] << 8) + buffer[offset];

                if (imageid == imgItem.imageID)
                    return true;

                offset += 14;
            }

            return false;
        }
        #endregion

        #region Wrap
        private int Wrap(int value, int limit)
        {
            return (((value) < (limit)) ? (value) : ((value) - (limit)));
        }
        #endregion

        #region BitFileGetBitsInt
        private int BitFileGetBitsInt(ref bitStream stream, byte[] buffer, ref int bufferOffset, ref int bits, int count)
        {
            return BitFileGetBitsLE(ref stream, buffer, ref bufferOffset, ref bits, count);
        }
        #endregion

        #region BitFileGetBitsLE
        private int BitFileGetBitsLE(ref bitStream stream, byte[] buffer, ref int bufferOffset, ref int bits, int count)
        {
            UInt16 bytes = 0;
            byte shifts = 0;
            int offset = 0;
            int remaining = count;
            int returnValue = 0;

            while (remaining >= 8)
            {
                returnValue = BitFileGetChar(ref stream, buffer, ref bufferOffset);

                bytes = (byte)returnValue;
                remaining -= 8;
                offset++;
            }

            if (remaining != 0)
            {
                shifts = (byte)(8 - remaining);

                while (remaining > 0)
                {
                    returnValue = BitFileGetBit(ref stream, buffer, ref bufferOffset);

                    if (returnValue == EOF)
                        return EOF;

                    bytes <<= 1;
                    bytes |= (byte)(returnValue & 0x01);
                    remaining--;
                }
            }

            byte byte2 = 0;
            if (bytes >= 0xFF)
            {
                bits = bytes;
            }
            else
            {
                shifts = (byte)bytes;
                bytes = byte2;
                byte2 = shifts;
                string s = string.Format("0x{0:X}{1:X}", bytes, byte2);
                bits = Convert.ToUInt16(s, 16);
            }

            return count;
        }
        #endregion

        #region BitFileGetBit
        private int BitFileGetBit(ref bitStream stream, byte[] buffer, ref int offset)
        {
            if (offset == buffer.Length)
                return EOF;

            int returnValue = 0;

            if (stream.bitCount == 0)
            {
                returnValue = (int)buffer[offset++];

                stream.bitCount = 8;
                stream.bitBuffer = returnValue;
            }

            stream.bitCount--;

            returnValue = (stream.bitBuffer >> stream.bitCount);
            return (returnValue & 0x01);
        }
        #endregion

        #region BitFileGetChar
        private int BitFileGetChar(ref bitStream stream, byte[] buffer, ref int offset)
        {
            if (offset == buffer.Length)
                return EOF;

            int returnValue = (int)buffer[offset++];

            if (stream.bitCount == 0)
            {
                return returnValue;
            }

            byte tmp = (byte)((byte)returnValue >> stream.bitCount);
            tmp |= (byte)((stream.bitBuffer) << (8 - (stream.bitCount)));
            stream.bitBuffer = returnValue;

            returnValue = tmp;

            return returnValue;
        }
        #endregion

        #region ReadDATA_FRK
        private string ReadDATA_FRK(string fileName, int offsetToIndex, byte[] buffer)
        {
            int offset = 0;

            int OFFSET_BITS = 14;
            int LENGTH_BITS = 3;
            int WINDOW_SIZE = (1 << OFFSET_BITS);
            byte c = 0;
            int i = 0, nextChar = 1;
            encoded_string_t code = new encoded_string_t();
            int uncoded = 1;
            int MAX_UNCODED = 2;
            int MAX_CODED = ((1 << LENGTH_BITS) + MAX_UNCODED);
            string strFile = "";
            bitStream fIn = new bitStream();

            byte[] slidingWindow = new byte[WINDOW_SIZE];
            byte[] uncodedLookahead = new byte[WINDOW_SIZE];

            for (int x = 0; x < WINDOW_SIZE; x++)
                slidingWindow[x] = (byte)' ';

            for (int x = 0; x < WINDOW_SIZE; x++)
                uncodedLookahead[x] = (byte)' ';

            while (true)
            {
                if (offset == buffer.Length)
                    break;

                c = (byte)BitFileGetBit(ref fIn, buffer, ref offset);

                if (c == uncoded)
                {
                    if (offset == buffer.Length)
                        break;

                    c = (byte)BitFileGetChar(ref fIn, buffer, ref offset);

                    if (c == 0x0e || c == 0x0f || c == 0x15 || c == 0x16 || c == 0xA5)
                    {
                        strFile += "" + (char)0x0D;
                        strFile += "" + (char)0x0A;
                        strFile += "" + (char)0x7C;

                        if (c == 0x0F)
                        {
                            strFile += "" + (char)0x5E;
                            strFile += "" + (char)0x7C;
                        }
                    }
                    else
                    {
                        if (c == 0x14 || c == 0x0A || c == 0x0B)
                        {
                            strFile += "" + (char)0x0D;
                            strFile += "" + (char)0x0A;

                            if (c == 0x14)
                            {
                                strFile += "___";
                                strFile += "___";
                                strFile += "___";
                                strFile += "___";
                            }
                            else
                            {
                                if (c == 0x0B)
                                {
                                    strFile += "" + (char)0x0D;
                                    strFile += "" + (char)0x0A;
                                }
                            }
                        }
                        else
                        {
                            if (c == 0x13)
                            {
                                strFile += "_";
                            }
                            else if (c == 0xD5)
                            {
                                strFile += "'";
                            }
                            else
                            {
                                strFile += "" + (char)c;
                            }
                        }
                    }

                    if (c == 0x0D)
                    {
                        strFile += "" + (char)0x0A;
                    }

                    slidingWindow[nextChar] = c;
                    nextChar = Wrap(((int)nextChar + 1), WINDOW_SIZE);
                }
                else
                {
                    code.offset = 0;
                    code.length = 0;

                    int coffset = 0;
                    int clength = 0;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref coffset, OFFSET_BITS) == EOF)
                        break;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref clength, LENGTH_BITS) == EOF)
                        break;

                    if (coffset == 0 && clength == 0)
                        break;

                    code.offset = coffset;
                    code.length = clength;
                    code.length += MAX_UNCODED + 1;

                    for (i = 0; i < code.length; i++)
                    {
                        c = slidingWindow[Wrap((code.offset + i), WINDOW_SIZE)];
                        if (code.offset + code.length > nextChar)
                        {
                            if (i >= nextChar - code.offset)
                            {
                                c = uncodedLookahead[i - nextChar + code.offset];
                            }
                        }
                        if (c == 0x0e || c == 0x0f || c == 0x15 || c == 0x16 || c == 0xA5)
                        {
                            strFile += "" + (char)0x0D;
                            strFile += "" + (char)0x0A;
                            strFile += "" + (char)0x7C;
                            if (c == 0x0f)
                            {
                                strFile += "" + (char)0x5E;
                                strFile += "" + (char)0x7C;
                            }
                        }
                        else
                        {
                            if (c == 0x14 || c == 0x0A || c == 0x0B)
                            {
                                strFile += "" + (char)0x0D;
                                strFile += "" + (char)0x0A;
                                if (c == 0x14)
                                {
                                    strFile += "___";
                                    strFile += "___";
                                    strFile += "___";
                                    strFile += "___";
                                }
                                else
                                {
                                    if (c == 0x0B)
                                    {
                                        strFile += "" + (char)0x0D;
                                        strFile += "" + (char)0x0A;
                                    }
                                }
                            }
                            else
                            {
                                if (c == 0x13)
                                {
                                    strFile += "_";
                                }
                                else
                                {
                                    strFile += "" + (char)c;
                                }
                            }
                        }
                        if (c == 0x0d)
                        {
                            strFile += "" + (char)0x0A;
                        }
                        uncodedLookahead[i] = c;
                    }

                    /* write out decoded string to sliding window */
                    for (i = 0; i < code.length; i++)
                    {
                        slidingWindow[(nextChar + i) % WINDOW_SIZE] =
                            uncodedLookahead[i];
                    }

                    nextChar = Wrap((nextChar + code.length), WINDOW_SIZE);
                }
            }

            return strFile;
        }
        #endregion

        #region SwapHTMLCharacter
        private string SwapHTMLCharacter(byte c, ref int imageIndex, ref int compressedIndex, ref int uncompressedIndex, ref bool bOpenTable, ref bool bOpenSpan, ref bool bOpenStyle, ref IMPStyleItem currentStyle)
        {
            string ret = "";

            foreach (IMPStyleRunItem styleRun in _styleRuns)
            {
                if (styleRun.Offset == uncompressedIndex)
                {
                    currentStyle = _styles[styleRun.StyleIndex];

                    if (bOpenStyle)
                        ret += "</span>";

                    if (currentStyle != null)
                        ret += "<span style=\"text-align:" + currentStyle.textAlign.ToLower() + ";font-family:" + currentStyle.fontFamilyName + ";font-size:" + currentStyle.fontSize + ";" + currentStyle.style + "\">";
                    else
                        ret += "<span>";
                    bOpenStyle = true;

                    break;
                }
            }

            if (c < 0x20)
            {
                #region Swap For HTML
                if (c == 0x0D)
                    ret += "\r\n<br>\r\n";
                else if (c == 0x0F)
                {
                    foreach (IMGRunItem img in _imgRuns)
                    {
                        if (img.dataOffset == uncompressedIndex)
                        {
                            ret += "\r\n<br>\r\n<img src=\"" + img.imageFileName + "\"><br>\r\n";
                            break;
                        }
                    }
                }
                else if (c == 0x14)
                {
                    ret += "\r\n<hr>\r\n";
                }
                else if (c == 0x0B)
                {
                    if (bOpenTable)
                    {
                        bOpenTable = false;
                        ret += "\r\n</td></tr></table>\r\n";
                    }

                    if (bOpenSpan)
                        ret += "\r\n</p>\r\n";

                    bOpenSpan = true;
                    if (currentStyle != null)
                        ret += "\r\n<p style=\"text-align:" + currentStyle.textAlign.ToLower() + ";font-family:" + currentStyle.fontFamilyName + ";font-size:" + currentStyle.fontSize + ";" + currentStyle.style + "\">\r\n";
                    else
                        ret += "\r\n<p>\r\n";
                }
                else if (c == 0x0E)
                {
                    if (bOpenTable)
                        ret += "\r\n</td></tr></table>\r\n";
                    bOpenTable = true;
                    System.Diagnostics.Debug.WriteLine("Start Table Offset = " + uncompressedIndex + "  Compressed Index = " + compressedIndex);
                    ret += "\r\n<table><tr><td>";
                }
                else if (c == 0x13)
                {
                    System.Diagnostics.Debug.WriteLine("End Table Cell Offset = " + uncompressedIndex + "  Compressed Index = " + compressedIndex);
                    ret += "\r\n</td><td>\r\n";
                }
                else if (c == 0x0A)
                {
                    if (bOpenTable)
                    {
                        bOpenTable = false;
                        ret += "\r\n</tr></table>\r\n";
                    }

                    if (bOpenSpan)
                    {
                        bOpenSpan = false;
                        ret += "\r\n</p>\r\n";
                    }

                    numberOfPages++;
                    ret += "\r\n<div style=\"page-break-before:always;\"></div>\r\n";
                }
                else
                {
                    System.Diagnostics.Debug.WriteLine(string.Format("Character Unknown {0:X2}", c));
                    ret += "";
                }
                #endregion
            }
            else
            {
                if (c == 32)
                    ret += " ";
                else if (c == 0x8E)
                    ret += "&eacute;";
                else if (c == 0xA0)
                    ret += "&nbsp;";
                else if (c == 0xA5)
                    ret += "&bull;";
                else if (c == 0xA8)
                    ret += "&reg;";
                else if (c == 0xA9)
                    ret += "&copy;";
                else if (c == 0xAA)
                    ret += "&trade;";
                else if (c == 0xAE)
                    ret += "&AElig;";
                else if (c == 0xC7)
                    ret += "&laquo;";
                else if (c == 0xC8)
                    ret += "&raquo;";
                else if (c == 0xC9)
                    ret += "&hellip;";
                else if (c == 0xD0)
                    ret += "&ndash;";
                else if (c == 0xD1)
                    ret += "&mdash;";
                else if (c == 0xD2)
                    ret += "&ldquo;";
                else if (c == 0xD3)
                    ret += "&rdquo;";
                else if (c == 0xD4)
                    ret += "&lsquo;";
                else if (c == 0xD5)
                    ret += "&rsquo;";
                else if (c == 0xE1)
                    ret += "&middot;";
                else
                {
                    if (c > 128)
                    {
                        System.Diagnostics.Debug.WriteLine(string.Format("Character Unknown {0} {1}", (int)c, (char)c));
                    }
                    ret += "" + (char)c;
                }
            }

            return ret;
        }
        #endregion

        #region FormatDATAToHTML
        private string FormatDATAToHTML(string fileName, int offsetToIndex, byte[] buffer)
        {
            numberOfPages = 0;

            IMPStyleItem currentStyle = null;
            bool bOpenStyle = false;
            bool bOpenTable = false;
            bool bOpenSpan = false;
            int imageIndex = 0;
            int offset = 0;
            int uncompressedIndex = 0;

            int OFFSET_BITS = 14;
            int LENGTH_BITS = 3;
            int WINDOW_SIZE = (1 << OFFSET_BITS);
            byte c = 0;
            int i = 0, nextChar = 1;
            encoded_string_t code = new encoded_string_t();
            int uncoded = 1;
            int MAX_UNCODED = 2;
            int MAX_CODED = ((1 << LENGTH_BITS) + MAX_UNCODED);
            string strFile = "";
            bitStream fIn = new bitStream();

            byte[] slidingWindow = new byte[WINDOW_SIZE];
            byte[] uncodedLookahead = new byte[WINDOW_SIZE];

            for (int x = 0; x < WINDOW_SIZE; x++)
                slidingWindow[x] = (byte)' ';

            for (int x = 0; x < WINDOW_SIZE; x++)
                uncodedLookahead[x] = (byte)' ';

            while (true)
            {
                if (offset == buffer.Length)
                    break;

                c = (byte)BitFileGetBit(ref fIn, buffer, ref offset);

                if (c == uncoded)
                {
                    if (offset == buffer.Length)
                        break;

                    c = (byte)BitFileGetChar(ref fIn, buffer, ref offset);

                    strFile += SwapHTMLCharacter(c, ref imageIndex, ref offset, ref uncompressedIndex, ref bOpenTable, ref bOpenSpan, ref bOpenStyle, ref currentStyle);
                    uncompressedIndex++;

                    slidingWindow[nextChar] = c;
                    nextChar = Wrap(((int)nextChar + 1), WINDOW_SIZE);
                }
                else
                {
                    code.offset = 0;
                    code.length = 0;

                    int coffset = 0;
                    int clength = 0;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref coffset, OFFSET_BITS) == EOF)
                        break;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref clength, LENGTH_BITS) == EOF)
                        break;

                    if (coffset == 0 && clength == 0)
                        break;

                    code.offset = coffset;
                    code.length = clength;
                    code.length += MAX_UNCODED + 1;

                    for (i = 0; i < code.length; i++)
                    {
                        c = slidingWindow[Wrap((code.offset + i), WINDOW_SIZE)];
                        if (code.offset + code.length > nextChar)
                        {
                            if (i >= nextChar - code.offset)
                            {
                                c = uncodedLookahead[i - nextChar + code.offset];
                            }
                        }

                        strFile += SwapHTMLCharacter(c, ref imageIndex, ref offset, ref uncompressedIndex, ref bOpenTable, ref bOpenSpan, ref bOpenStyle, ref currentStyle);
                        uncompressedIndex++;

                        uncodedLookahead[i] = c;
                    }

                    /* write out decoded string to sliding window */
                    for (i = 0; i < code.length; i++)
                    {
                        slidingWindow[(nextChar + i) % WINDOW_SIZE] =
                            uncodedLookahead[i];
                    }

                    nextChar = Wrap((nextChar + code.length), WINDOW_SIZE);
                }
            }

            return strFile;
        }
        #endregion

        #region SwapLRSCharacter
        private string SwapLRSCharacter(byte c, ref int imageIndex, ref int compressedIndex, ref int uncompressedIndex, ref bool bOpenTable, ref bool bOpenSpan, ref bool bOpenStyle, ref IMPStyleItem currentStyle)
        {
            string ret = "";

            if (c < 0x20)
            {
                #region Swap For HTML
                if (c == 0x0D)
                    ret += "\r\n<CR>\r\n";
                else if (c == 0x0F)
                {
                }
                else if (c == 0x14)
                {
                    ret += "\r\n______________________________________\r\n";
                }
                else if (c == 0x0B)
                {
                    if (bOpenTable)
                    {
                        bOpenTable = false;
                        ret += "\r\n</P>\r\n<CR>";
                    }
                    
                    if (bOpenSpan)
                        ret += "\r\n</P>\r\n<CR>";

                    bOpenSpan = true;
                    ret += "\r\n<P>\r\n";
                }
                else if (c == 0x0E)
                {
                    if (bOpenTable)
                        ret += "\r\n</P>\r\n<CR>";
                    ret += "\r\n<P>";
                }
                else if (c == 0x13)
                {
                    ret += "\r\n</P>\r\n<CR>";
                }
                else if (c == 0x0A)
                {
                    if (bOpenTable)
                    {
                        bOpenTable = false;
                        ret += "\r\n</P>\r\n<CR>";
                    }

                    if (bOpenSpan)
                    {
                        bOpenSpan = false;
                        ret += "\r\n</P>\r\n<CR>";
                    }

                    numberOfPages++;
                    ret += "\r\n<CR><CR>\r\n";
                }
                else
                {
                    System.Diagnostics.Debug.WriteLine(string.Format("Character Unknown {0:X2}", c));
                    ret += "";
                }
                #endregion
            }
            else
            {
                if (c == 32)
                    ret += " ";
                else
                {
                    if (c > 128)
                    {
                        System.Diagnostics.Debug.WriteLine(string.Format("Character Unknown {0} {1}", (int)c, (char)c));
                    }
                    ret += "" + (char)c;
                }
            }

            return ret;
        }
        #endregion

        #region FormatDATAToLRS
        private string FormatDATAToLRS(string fileName, int offsetToIndex, byte[] buffer)
        {
            numberOfPages = 0;

            IMPStyleItem currentStyle = null;
            bool bOpenStyle = false;
            bool bOpenTable = false;
            bool bOpenSpan = false;
            int imageIndex = 0;
            int offset = 0;
            int uncompressedIndex = 0;

            int OFFSET_BITS = 14;
            int LENGTH_BITS = 3;
            int WINDOW_SIZE = (1 << OFFSET_BITS);
            byte c = 0;
            int i = 0, nextChar = 1;
            encoded_string_t code = new encoded_string_t();
            int uncoded = 1;
            int MAX_UNCODED = 2;
            int MAX_CODED = ((1 << LENGTH_BITS) + MAX_UNCODED);
            string strFile = "";
            bitStream fIn = new bitStream();

            byte[] slidingWindow = new byte[WINDOW_SIZE];
            byte[] uncodedLookahead = new byte[WINDOW_SIZE];

            for (int x = 0; x < WINDOW_SIZE; x++)
                slidingWindow[x] = (byte)' ';

            for (int x = 0; x < WINDOW_SIZE; x++)
                uncodedLookahead[x] = (byte)' ';

            while (true)
            {
                if (offset == buffer.Length)
                    break;

                c = (byte)BitFileGetBit(ref fIn, buffer, ref offset);

                if (c == uncoded)
                {
                    if (offset == buffer.Length)
                        break;

                    c = (byte)BitFileGetChar(ref fIn, buffer, ref offset);

                    strFile += SwapLRSCharacter(c, ref imageIndex, ref offset, ref uncompressedIndex, ref bOpenTable, ref bOpenSpan, ref bOpenStyle, ref currentStyle);
                    uncompressedIndex++;

                    slidingWindow[nextChar] = c;
                    nextChar = Wrap(((int)nextChar + 1), WINDOW_SIZE);
                }
                else
                {
                    code.offset = 0;
                    code.length = 0;

                    int coffset = 0;
                    int clength = 0;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref coffset, OFFSET_BITS) == EOF)
                        break;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref clength, LENGTH_BITS) == EOF)
                        break;

                    if (coffset == 0 && clength == 0)
                        break;

                    code.offset = coffset;
                    code.length = clength;
                    code.length += MAX_UNCODED + 1;

                    for (i = 0; i < code.length; i++)
                    {
                        c = slidingWindow[Wrap((code.offset + i), WINDOW_SIZE)];
                        if (code.offset + code.length > nextChar)
                        {
                            if (i >= nextChar - code.offset)
                            {
                                c = uncodedLookahead[i - nextChar + code.offset];
                            }
                        }

                        strFile += SwapLRSCharacter(c, ref imageIndex, ref offset, ref uncompressedIndex, ref bOpenTable, ref bOpenSpan, ref bOpenStyle, ref currentStyle);
                        uncompressedIndex++;

                        uncodedLookahead[i] = c;
                    }

                    /* write out decoded string to sliding window */
                    for (i = 0; i < code.length; i++)
                    {
                        slidingWindow[(nextChar + i) % WINDOW_SIZE] =
                            uncodedLookahead[i];
                    }

                    nextChar = Wrap((nextChar + code.length), WINDOW_SIZE);
                }
            }

            return strFile;
        }
        #endregion

        #endregion

        #region ReadForDisplay
        public void ReadForDisplay()
        {
            int index = 0;
            byte[] buffer = null;
            BinaryReader br = new BinaryReader(File.OpenRead(IMPFilePath));

            _header = new IMPHeader();
            _bookProperties = new IMPBookProperties();
            _tableOfContents = new List<IMPTableContentsItem>();
            _resFiles = new List<IMPRESFile>();
            _swIndexes = new List<SWIndex>();
            _imgRuns = new List<IMGRunItem>();
            _styles = new List<IMPStyleItem>();
            _extStyles = new List<IMPExtendedStyleItem>();
            _styleRuns = new List<IMPStyleRunItem>();
            _pPics = new List<IMPPPic>();
            _Pcz1 = new List<IMPPcZ1>();
            _PcZ1 = new List<IMPPcZ1>();
            _tables = new List<IMPTableItem>();
            _tableRows = new List<IMPTableRowGroup>();
            _tableRowCells = new List<IMPTableCellGroup>();

            buffer = br.ReadBytes(48);

            #region Header
            _header.version = GetInt(buffer, index, 2);
            index += 2;
            _header.constant = GetString(buffer, index, 8);
            index += 8;
            index += 8; // Skip Unknown
            _header.fileCount = GetInt(buffer, index, 2);
            index += 2;
            _header.directoryLength = GetInt(buffer, index, 2);
            index += 2;
            _header.bytesRemaining = GetInt(buffer, index, 2);
            index += 2;
            index += 8; // Skip Unknown
            _header.compression = (GetInt(buffer, index, 4) == 1);
            index += 4;
            _header.encryption = (GetInt(buffer, index, 4) == 1);
            index += 4;
            _header.impType = GetInt(buffer, index, 2);
            index += 2;
            _header.zoomState = GetInt(buffer, index, 2);
            index += 2;
            index += 4; // Skip Unknown

            _header.impType = _header.zoomState >> 4;
            _header.zoomState = _header.zoomState - (_header.impType * 16);

            REB1200 = (_header.impType == 1 || _header.impType == 0);

            if (_header.impType == 1 || _header.impType == 0)
                _header.impTypeDevice = "REB 1200/GEB 2150";
            else
                _header.impTypeDevice = "EBW 1150/GEB 1150";
            #endregion

            #region Book Properties
            index = 0;
            buffer = br.ReadBytes(_header.bytesRemaining - 24);

            _bookProperties.ID = GetString2(buffer, index, ref index);
            _bookProperties.Bookshelf_Category = GetString2(buffer, index, ref index);
            _bookProperties.Subcategory = GetString2(buffer, index, ref index);
            _bookProperties.Title = GetString2(buffer, index, ref index);
            _bookProperties.LastName = GetString2(buffer, index, ref index);
            _bookProperties.MiddleName = GetString2(buffer, index, ref index);
            _bookProperties.FirstName = GetString2(buffer, index, ref index);

            _bookProperties.Length = _bookProperties.ID.Length +
                _bookProperties.Bookshelf_Category.Length +
                _bookProperties.Subcategory.Length +
                _bookProperties.Title.Length +
                _bookProperties.LastName.Length +
                _bookProperties.MiddleName.Length +
                _bookProperties.FirstName.Length + 7 +
                _header.directoryLength;

            buffer = br.ReadBytes(_header.directoryLength);
            _bookProperties.ResFile = GetString(buffer, 0, _header.directoryLength);
            #endregion

            br.Close();
        }
        #endregion

        #region ReadIMP
        public void ReadIMP(ref Label lblText, ref ProgressBar pbProgress)
        {
            int index = 0;
            byte[] buffer = null;
            BinaryReader br = new BinaryReader(File.OpenRead(IMPFilePath));

            _header = new IMPHeader();
            _bookProperties = new IMPBookProperties();
            _tableOfContents = new List<IMPTableContentsItem>();
            _resFiles = new List<IMPRESFile>();
            _swIndexes = new List<SWIndex>();
            _imgRuns = new List<IMGRunItem>();
            _styles = new List<IMPStyleItem>();
            _extStyles = new List<IMPExtendedStyleItem>();
            _styleRuns = new List<IMPStyleRunItem>();
            _pPics = new List<IMPPPic>();
            _Pcz1 = new List<IMPPcZ1>();
            _PcZ1 = new List<IMPPcZ1>();
            _tables = new List<IMPTableItem>();
            _tableRows = new List<IMPTableRowGroup>();
            _tableRowCells = new List<IMPTableCellGroup>();

            buffer = br.ReadBytes(48);

            #region Header
            _header.version = GetInt(buffer, index, 2);
            index += 2;
            _header.constant = GetString(buffer, index, 8);
            index += 8;
            index += 8; // Skip Unknown
            _header.fileCount = GetInt(buffer, index, 2);
            index += 2;
            _header.directoryLength = GetInt(buffer, index, 2);
            index += 2;
            _header.bytesRemaining = GetInt(buffer, index, 2);
            index += 2;
            index += 8; // Skip Unknown
            _header.compression = (GetInt(buffer, index, 4) == 1);
            index += 4;
            _header.encryption = (GetInt(buffer, index, 4) == 1);
            index += 4;
            _header.impType = GetInt(buffer, index, 2);
            index += 2;
            _header.zoomState = GetInt(buffer, index, 2);
            index += 2;
            index += 4; // Skip Unknown

            _header.impType = _header.zoomState >> 4;
            _header.zoomState = _header.zoomState - (_header.impType * 16);

            REB1200 = (_header.impType == 1 || _header.impType == 0);

            if (_header.impType == 1 || _header.impType == 0)
                _header.impTypeDevice = "REB 1200/GEB 2150";
            else
                _header.impTypeDevice = "EBW 1150/GEB 1150";
            #endregion

            #region Book Properties
            index = 0;
            buffer = br.ReadBytes(_header.bytesRemaining - 24);

            _bookProperties.ID = GetString2(buffer, index, ref index);
            _bookProperties.Bookshelf_Category = GetString2(buffer, index, ref index);
            _bookProperties.Subcategory = GetString2(buffer, index, ref index);
            _bookProperties.Title = GetString2(buffer, index, ref index);
            _bookProperties.LastName = GetString2(buffer, index, ref index);
            _bookProperties.MiddleName = GetString2(buffer, index, ref index);
            _bookProperties.FirstName = GetString2(buffer, index, ref index);

            _bookProperties.Length = _bookProperties.ID.Length +
                _bookProperties.Bookshelf_Category.Length +
                _bookProperties.Subcategory.Length +
                _bookProperties.Title.Length +
                _bookProperties.LastName.Length +
                _bookProperties.MiddleName.Length +
                _bookProperties.FirstName.Length + 7 +
                _header.directoryLength;

            buffer = br.ReadBytes(_header.directoryLength);
            _bookProperties.ResFile = GetString(buffer, 0, _header.directoryLength);
            #endregion

            int imp_offset_TOC = 48 + _bookProperties.Length;

            #region Table Of Contents
            for (int i = 0; i < _header.fileCount; i++)
            {
                if (_header.version == 1)
                {
                }
                else if (_header.version == 2)
                {
                    buffer = br.ReadBytes(20);

                    index = 0;

                    IMPTableContentsItem item = new IMPTableContentsItem();
                    item.fileName = GetString(buffer, index, 4);
                    index += 4;
                    index += 4;
                    item.fileSize = GetInt(buffer, index, 4);
                    index += 4;
                    item.fileType = GetString(buffer, index, 4);
                    index += 4;
                    index += 4;

                    _tableOfContents.Add(item);
                }
            }
            #endregion

            pbProgress.Value = 0;
            pbProgress.Minimum = 0;
            pbProgress.Maximum = _tableOfContents.Count;
            lblText.Text = "Processing IMP";
            Application.DoEvents();

            foreach (IMPTableContentsItem tocItem in _tableOfContents)
            {
                IMPRESFile resFile = new IMPRESFile();

                #region Read Header
                if (_header.version == 1)
                {
                }
                else if (_header.version == 2)
                {
                    buffer = br.ReadBytes(20);
                }
                #endregion

                lblText.Text = "Processing IMP " + resFile.fileType;
                pbProgress.Value = pbProgress.Value + 1;
                Application.DoEvents();

                resFile.fileName = tocItem.fileName;
                resFile.fileSize = tocItem.fileSize;
                resFile.fileType = tocItem.fileType;
                resFile.buffer = br.ReadBytes(resFile.fileSize);

                int toc_offsetIndex = ReadOffset(resFile.buffer);

                if (!REB1200)
                    tocItem.indexes = (resFile.fileSize - toc_offsetIndex) / 14;
                else
                    tocItem.indexes = (resFile.fileSize - toc_offsetIndex) / 12;

                #region File Type Parsing
                switch (resFile.fileType)
                {
                    case "    ":
                        if (!_header.compression)
                            IMPResourceFile = ASCIIEncoding.ASCII.GetString(resFile.buffer);
                        break;
                    case "!!cm":
                        // TODO
                        break;
                    case "!!sw":
                        ReadSW(toc_offsetIndex, resFile.buffer);
                        break;
                    case "AncT":
                        // TODO
                        break;
                    case "AnTg":
                        // TODO
                        break;
                    case "BGcl":
                        // TODO
                        break;
                    case "BPgz":
                        // TODO
                        break;
                    case "BPgZ":
                        // TODO
                        break;
                    case "Devm":
                        // TODO
                        break;
                    case "eLnk":
                        // TODO
                        break;
                    case "ESts":
                        ReadExtendedStyle(toc_offsetIndex, resFile.buffer);
                        break;
                    case "HfPz":
                        // TODO
                        break;
                    case "HfPZ":
                        // TODO
                        break;
                    case "Hyp2":
                        // TODO
                        break;
                    case "Lnks":
                        // TODO
                        break;
                    case "Mrgn":
                        // TODO
                        break;
                    case "Pc31":
                        // TODO
                        break;
                    case "Pcz0":
                    case "PcZ0":
                        // TODO
                        break;
                    case "Pcz1":
                    case "PcZ1":
                        ReadPcZ1(resFile.fileType, toc_offsetIndex, resFile.buffer);
                        break;
                    case "pInf":
                        // TODO
                        break;
                    case "PPic":
                        ReadPPic(toc_offsetIndex, resFile.buffer);
                        break;
                    case "Styl":
                        ReadStyle(toc_offsetIndex, resFile.buffer);
                        break;
                    case "StRn":
                        ReadStyleRun(toc_offsetIndex, resFile.buffer);
                        break;
                    case "StR2":
                        // TODO
                        break;
                    case "Tabl":
                        ReadTabl(toc_offsetIndex, resFile.buffer);
                        break;
                    case "TCel":
                        ReadTCel(toc_offsetIndex, resFile.buffer);
                        break;
                    case "TRow":
                        ReadTRow(toc_offsetIndex, resFile.buffer);
                        break;
                    case "ImRn":
                        ReadImRn(toc_offsetIndex, resFile.buffer);
                        break;
                    case "GIF ":
                    case "JPEG":
                    case "PIC2":
                    case "PNG ":
                    case "PICT":
                        break;
                    case "HRle":
                        // TODO
                        break;
                    case "BPos":
                        // TODO
                        break;
                    case "MRPs":
                        // TODO
                        break;
                    case "Ano2":
                        // TODO
                        break;
                    case "Hlts":
                        // TODO
                        break;
                    case "BTok":
                        // TODO
                        break;
                    case "BMks":
                        // TODO
                        break;
                    case "TGNt":
                        // TODO
                        break;
                    case "Form":
                        // TODO
                        break;
                    case "FItm":
                        // TODO
                        break;
                    case "FIDt":
                        // TODO
                        break;
                    case "FrDt":
                        // TODO
                        break;
                }
                #endregion

                _resFiles.Add(resFile);
            }

            br.Close();

            foreach (IMGRunItem item in _imgRuns)
            {
                foreach (IMPRESFile f in this.RESFiles)
                    if (f.fileType.Trim() == item.imageFileType.Trim())
                    {
                        if (ResourceHasID(item, ref f.buffer))
                        {
                            item.dataFile = f;
                            break;
                        }
                    }
            }
        }
        #endregion

        #region ReverseResourceFile
        public void ReverseResourceFile()
        {
            try
            {
                foreach (IMPRESFile file in _resFiles)
                {
                    if (file.fileType == "")
                    {
                        int indexOffset = ReadOffset(file.buffer);
                        IMPResourceFile = ReadDATA_FRK(file.fileName, indexOffset, file.buffer);
                    }
                }
            }
            catch (Exception exc)
            {
                System.Diagnostics.Debug.WriteLine(exc.ToString());
            }
        }
        #endregion

        #region ReverseResourceFileToHTML
        public string ReverseResourceFileToHTML()
        {
            try
            {
                foreach (IMPRESFile file in _resFiles)
                {
                    if (file.fileType == "")
                    {
                        int indexOffset = ReadOffset(file.buffer);
                        return FormatDATAToHTML(file.fileName, indexOffset, file.buffer);
                    }
                }
            }
            catch (Exception exc)
            {
                System.Diagnostics.Debug.WriteLine(exc.ToString());
            }

            return "";
        }
        #endregion

        #region ReverseResourceFileToLRS
        public string ReverseResourceFileToLRS()
        {
            try
            {
                foreach (IMPRESFile file in _resFiles)
                {
                    if (file.fileType == "")
                    {
                        int indexOffset = ReadOffset(file.buffer);
                        return FormatDATAToLRS(file.fileName, indexOffset, file.buffer);
                    }
                }
            }
            catch (Exception exc)
            {
                System.Diagnostics.Debug.WriteLine(exc.ToString());
            }

            return "";
        }
        #endregion

        #region GetImage
        public MemoryStream GetImage(int index, ref IMPRESFile file)
        {
            int indexOffset = ReadOffset(file.buffer);

            int dataIndex = (file.buffer.Length - indexOffset) / 14;

            int offset = indexOffset;

            for (int j = 0; j < dataIndex; j++)
            {
                int imageid = (file.buffer[offset + 1] << 8) + file.buffer[offset];

                if (imageid == index)
                {
                    offset += 4;
                    int len1 = (file.buffer[offset + 1] << 8) + file.buffer[offset];
                    offset += 2;
                    int len2 = (file.buffer[offset + 1] << 8) + file.buffer[offset];
                    offset += 2;
                    int offset1 = (file.buffer[offset + 1] << 8) + file.buffer[offset];
                    offset += 2;
                    int offset2 = (file.buffer[offset + 1] << 8) + file.buffer[offset];
                    offset += 2;

                    int len = (len2 * 65536) + len1;
                    int offset3 = (offset2 * 65536) + offset1;

                    MemoryStream ms = new MemoryStream();

                    ms.Write(file.buffer, offset3, len);
                    ms.Seek(0, SeekOrigin.Begin);

                    return ms;
                }

                offset += 14;
            }

            return new MemoryStream();
        }
        #endregion

        #region ChangeBookProperties
        public void ChangeBookProperties(IMPBookProperties newProp)
        {
            byte[] buffer = null;

            int newBPLength = newProp.Bookshelf_Category.Length;
            newBPLength += newProp.FirstName.Length;
            newBPLength += newProp.ID.Length;
            newBPLength += newProp.LastName.Length;
            newBPLength += newProp.MiddleName.Length;
            newBPLength += newProp.Subcategory.Length;
            newBPLength += newProp.Title.Length;
            newBPLength += 7 + 24;

            BinaryReader br = new BinaryReader(File.OpenRead(IMPFilePath));
            MemoryStream ms = new MemoryStream();

            int offset = 0;

            #region IMP Header
            buffer = new byte[2];
            // IMP Version
            buffer = br.ReadBytes(2);
            ms.Write(buffer, 0, 2);
            offset += 2;

            buffer = new byte[8];
            // Constant
            buffer = br.ReadBytes(8);
            ms.Write(buffer, 0, 8);
            offset += 8;

            buffer = new byte[8];
            // UNKNOWN
            buffer = br.ReadBytes(8);
            ms.Write(buffer, 0, 8);
            offset += 8;

            buffer = new byte[2];
            // Files in RES Directory
            buffer = br.ReadBytes(2);
            ms.Write(buffer, 0, 2);
            offset += 2;

            buffer = new byte[2];
            // Directory Length (RES File Length)
            buffer = br.ReadBytes(2);
            ms.Write(buffer, 0, 2);
            offset += 2;

            // Bytes Remaining
            buffer = br.ReadBytes(2);

            buffer = new byte[] { (byte)(newBPLength >> 8), (byte)newBPLength };
            ms.Write(buffer, 0, 2);
            offset += 2;

            buffer = new byte[4];
            // UNKNOWN
            buffer = br.ReadBytes(4);
            ms.Write(buffer, 0, 4);
            offset += 4;

            buffer = new byte[4];
            // UNKNOWN
            buffer = br.ReadBytes(4);
            ms.Write(buffer, 0, 4);
            offset += 4;

            buffer = new byte[4];
            // Compressed
            buffer = br.ReadBytes(4);
            ms.Write(buffer, 0, 4);
            offset += 4;

            buffer = new byte[4];
            // Encrypted
            buffer = br.ReadBytes(4);
            ms.Write(buffer, 0, 4);
            offset += 4;

            buffer = new byte[4];
            // Zoom State
            buffer = br.ReadBytes(4);
            ms.Write(buffer, 0, 4);
            offset += 4;

            buffer = new byte[4];
            // UNKNOWN
            buffer = br.ReadBytes(4);
            ms.Write(buffer, 0, 4);
            offset += 4;

            #endregion

            offset = 0;
            #region Write New Properties
            // Book ID
            _bookProperties.ID = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.ID);
            ms.Write(buffer, 0, newProp.ID.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.ID.Length + 1;

            // Bookshelf Category
            _bookProperties.Bookshelf_Category = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.Bookshelf_Category);
            ms.Write(buffer, 0, newProp.Bookshelf_Category.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.Bookshelf_Category.Length + 1;

            // Subcategory
            _bookProperties.Subcategory = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.Subcategory);
            ms.Write(buffer, 0, newProp.Subcategory.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.Subcategory.Length + 1;

            // Book Title
            _bookProperties.Title = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.Title);
            ms.Write(buffer, 0, newProp.Title.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.Title.Length + 1;

            // Last Name
            _bookProperties.LastName = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.LastName);
            ms.Write(buffer, 0, newProp.LastName.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.LastName.Length + 1;

            // Middle Name
            _bookProperties.MiddleName = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.MiddleName);
            ms.Write(buffer, 0, newProp.MiddleName.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.MiddleName.Length + 1;

            // First Name
            _bookProperties.FirstName = ReadString(ref br);
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.FirstName);
            ms.Write(buffer, 0, newProp.FirstName.Length);
            ms.WriteByte((byte)'\0');
            offset += newProp.FirstName.Length + 1;

            // RES File Name
            _bookProperties.ResFile = ASCIIEncoding.ASCII.GetString(br.ReadBytes(_header.directoryLength));
            buffer = new byte[201];
            buffer = ASCIIEncoding.ASCII.GetBytes(newProp.ResFile);
            ms.Write(buffer, 0, newProp.ResFile.Length);
            offset += newProp.ResFile.Length;
            #endregion

            #region Write Remaining Bytes to new Stream
            byte[] restOfFile = br.ReadBytes((int)br.BaseStream.Length - (int)br.BaseStream.Position + 1);
            ms.Write(restOfFile, 0, restOfFile.Length);
            #endregion

            br.Close();

            ms.Seek(0, SeekOrigin.Begin);

            BinaryWriter bw = new BinaryWriter(File.Create(IMPFilePath));
            bw.Write(ms.ToArray());
            bw.Close();
        }
        #endregion
    }
}
