﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;

using System.IO;

namespace ConvertIMPGUI
{
    public class CBBeBFile
    {
        #region Constants
        /* Primary LRF header offsets */
        public static int LRF_SIGNATURE1 = 0x00;	// UTF-16LE "LRF\000"
        public static int LRF_VERSION = 0x08;	// file version (short)
        public static int LRF_PSUEDOKEY = 0x0A;	// XOR key (short)
        public static int LRF_ROOT_ID = 0x0C;	// (int)
        public static int LRF_OBJECT_COUNT = 0x10;	// (long) count of objects
        public static int LRF_OBJECTTREE_OFFSET = 0x18;// (long) offset to object table

        /* Secondary LRF header offsets */
        public static int LRF_UNK0 = 0x20;		// (int)
        public static int LRF_DIRECTION = 0x24;	// (byte)
        public static int LRF_DIRECTION_FORWARDS = 0x01;
        public static int LRF_DIRECTION_BACKWARDS = 0x10;
        public static int LRF_UNK1 = 0x26;		// (short)
        public static int LRF_UNK2 = 0x2A;		// (short)
        public static int LRF_UNK3 = 0x2C;		// (short)
        public static int LRF_UNK4 = 0x2E;		// (byte)
        public static int LRF_SIGNATURE2 = 0x30;	// 5 int's (some kind of signature)
        public static int LRF_TOC_ID = 0x44;		// (int)
        public static int LRF_TOC_OFFSET = 0x48;	// (int)
        public static int LRF_INFOLEN = 0x4C;	// Compressed metaData size (short)

        // Next two only walid if version >= 800
        public static int LRF_UNK7 = 0x4E;		// (short)
        public static int LRF_THUMBNAIL_LENGTH = 0x50;// GIF thumbnail size (int)

        public static int LRF_UNK6 = 0x4E;

        public static byte OBJECT_TYPE_INVALID_00 = 0x00;
        public static byte OBJECT_TYPE_PageTree = 0x01;
        public static byte OBJECT_TYPE_Page = 0x02;
        public static byte OBJECT_TYPE_Header = 0x03;
        public static byte OBJECT_TYPE_Footer = 0x04;
        public static byte OBJECT_TYPE_PageAtr = 0x05;
        public static byte OBJECT_TYPE_Block = 0x06;
        public static byte OBJECT_TYPE_BlockAtr = 0x07;
        public static byte OBJECT_TYPE_MiniPage = 0x08;
        public static byte OBJECT_TYPE_BlockList = 0x09;
        public static byte OBJECT_TYPE_Text = 0x0a;
        public static short FLAG_COMPRESSED = 0x0100;
        public static short FLAG_SCRAMBLED = 0x0200;
        public static int FLAG_ENCRYPTED = 0x8000;
        public static byte OBJECT_TYPE_TextAtr = 0x0b;
        public static byte OBJECT_TYPE_Image = 0x0c;
        public static byte OBJECT_TYPE_Canvas = 0x0d;
        public static byte OBJECT_TYPE_ParagraphAtr = 0x0e;
        public static byte OBJECT_TYPE_Invalid_0f = 0x0f;
        public static byte OBJECT_TYPE_Invalid_10 = 0x10;
        public static byte OBJECT_TYPE_ImageStream = 0x11;
        public static byte OBJECT_TYPE_Import = 0x12;
        public static byte OBJECT_TYPE_Button = 0x13;
        public static byte OBJECT_TYPE_Window = 0x14;
        public static byte OBJECT_TYPE_PopUpWin = 0x15;
        public static byte OBJECT_TYPE_Sound = 0x16;
        public static byte OBJECT_TYPE_PlaneStream = 0x17;
        public static byte OBJECT_TYPE_Invalid_18 = 0x18;
        public static byte OBJECT_TYPE_Font = 0x19;
        public static byte OBJECT_TYPE_ObjectInfo = 0x1a;
        public static short FLAG_PAGE_NUMBERS = 0x0081;
        public static short FLAG_PAGE_LAYOUT = 0x0082;
        public static byte OBJECT_TYPE_Invalid_1b = 0x1b;
        public static byte OBJECT_TYPE_BookAtr = 0x1c;
        public static byte OBJECT_TYPE_SimpleText = 0x1d;
        public static byte OBJECT_TYPE_TOC = 0x1e;
        public static short FLAG_TOC_51 = 0x0051;
        public static byte OBJECT_TYPE_Invalid_1f = 0x1f;
        #endregion

        #region Member Variables
        public short globalBBeBObject;
        private byte[] buf;
        public ArrayList objectList = new ArrayList();

        int pages = 0;
        BBeBObject bookFile = null;
        int margins_id = 0;
        int page_box_id = 0;
        int font_id = 0;
        private byte[] metaData = null;
        private byte[] thumbnail = null;
        private byte[] fontname = null;

        private StringBuilder bookFileName = new StringBuilder();
        private StringBuilder bookTitle = new StringBuilder();
        private StringBuilder bookAuthor = new StringBuilder("Unknown Author");
        private StringBuilder bookID = new StringBuilder();
        private StringBuilder bookPublisher = new StringBuilder("Unknown Publisher");
        private StringBuilder bookLabel = new StringBuilder();
        private StringBuilder bookCategory = new StringBuilder("Unknown");
        private StringBuilder bookClassification = new StringBuilder("Unknown");
        private StringBuilder bookFreeText = new StringBuilder();
        private StringBuilder bookIconName = new StringBuilder();
        private StringBuilder bookCoverName = new StringBuilder();
        private StringBuilder docLanguage = new StringBuilder("en");
        private StringBuilder docCreator = new StringBuilder("Unknown Creator");
        private StringBuilder docCreationDate = new StringBuilder();
        private StringBuilder docFileName = new StringBuilder();
        private Hashtable configTags = new Hashtable(32);

        public int currentPosition = 0;

        byte[] textBuffer = null;
        int textOffset = 0;
        byte[] outBuf = new byte[65536];
        public int outBufOffset = 0;
        #endregion

        #region Constructor
        public CBBeBFile()
        {
            globalBBeBObject = 0x32;
            buf = new byte[16 * 1024 * 1024];
        }
        #endregion

        #region BindBook
        public void BindBook(string htmlFile, CIMPFile impFile)
        {
            bookFileName = new StringBuilder(htmlFile);
            bookTitle = new StringBuilder(impFile.BookProperties.Title);
            bookAuthor = new StringBuilder(impFile.BookProperties.FirstName);
            bookID = new StringBuilder(impFile.BookProperties.ID);
            bookPublisher = new StringBuilder("");
            bookLabel = new StringBuilder("");
            bookCategory = new StringBuilder(impFile.BookProperties.Bookshelf_Category);
            bookClassification = new StringBuilder("");
            bookFreeText = new StringBuilder("");
            bookIconName = new StringBuilder("");
            bookCoverName = new StringBuilder(impFile.BookProperties.Title + " - " + impFile.BookProperties.FirstName);
            docLanguage = new StringBuilder("en");
            docCreator = new StringBuilder("ConvertIMP GUI");
            docCreationDate = new StringBuilder(DateTime.Now.Millisecond.ToString());
            docFileName = new StringBuilder(htmlFile.Replace(".html", ".lrf"));

            StartNewBook();
            ParseHTML(bookFileName.ToString());
            metaData = CreateMetaData();
            WriteBookToFile(docFileName.ToString(), bookIconName.ToString());
            Write(docFileName.ToString());
        }
        #endregion

        #region StartNewBook
        public void StartNewBook()
        {
            CBBeBFile _this = this;
            pages = 0;

            BBeBObject head = new BBeBObject(ref _this, CBBeBFile.OBJECT_TYPE_BookAtr); // Root Object
            head.addTag(0x75, 0x0002);
            head.addTag(0x76, 0x0000);
            head.addTag(0x77, 0x0001);

            byte[] tmp78 = new byte[] { 0, 0, 0, 0, 0x16, (byte)0xf5, 0, 0, 0x1, 0x30 };
            head.addTag(0x78, tmp78);

            head.addTag(0x79, 0x0002);
            head.addTag(0x7a, 0x0010);
            head.addTag(0xda, 0x0002);

            byte[] empty = new byte[] { 0x00, 0x00, 0x00, 0x00 };
            BBeBObject temp = new BBeBObject(ref _this, CBBeBFile.OBJECT_TYPE_TOC, CBBeBFile.FLAG_TOC_51, empty); // special ?

            // UTF-16LE string containing font name
            byte[] fontname = new byte[] {22, 0,
			   (byte)'I',0,
			   (byte)'W',0,
			   (byte)'A',0,
			   0x0E,0x66,
			   (byte)'-',0,
			   0x2D,0x4E,
			   0x30,0x7D,
			   (byte)'N',0,
			   (byte)'-',0,
			   (byte)'e',0,
			   (byte)'b',0};

            // Create Global font record for the file
            BBeBObject fontRecord = new BBeBObject(ref _this, CBBeBFile.OBJECT_TYPE_TextAtr);
            fontRecord.addTag(0x76, 0x0000);
            fontRecord.addTag(0x77, 0x0001);
            fontRecord.addTag(0x79, 0x0001);
            fontRecord.addTag(0x7a, 0x0000);
            fontRecord.addTag(0x11, 0x0064);
            fontRecord.addTag(0x12, 0xfff6);
            fontRecord.addTag(0x13, 0x0000);
            fontRecord.addTag(0x14, 0x0000);
            fontRecord.addTag(0x15, 0x0190);
            fontRecord.addTag(0x16, fontname);
            fontRecord.addTagInt(0x17, 0);
            fontRecord.addTagInt(0x18, 0x00ff);
            fontRecord.addTag(0x19, 0x0019);
            fontRecord.addTag(0x1a, 0x0000);
            fontRecord.addTag(0x1b, 0x008c);
            fontRecord.addTag(0x1c, 0x000a);
            fontRecord.addTag(0x1d, 0x0000);
            fontRecord.addTag(0x1e, 0x0000);
            fontRecord.addTag(0xf1, 0x0002);
            fontRecord.addTagInt(0xf2, 0);
            fontRecord.addTag(0x3c, 0x0001);
            fontRecord.addTag(0x3d, 0x0001);
            fontRecord.addTag(0x3e, 0x0000);
            fontRecord.addTag(0x75, 0x0001);

            font_id = fontRecord.id;

            // Fill this one in later need id now for head
            bookFile = new BBeBObject(ref _this, CBBeBFile.OBJECT_TYPE_PageTree);

            head.addTagInt(0x7b, bookFile.id);

            // Margins
            BBeBObject margins = new BBeBObject(ref _this, CBBeBFile.OBJECT_TYPE_PageAtr);
            margins.addTag(0x21, 0x0005);	// 5
            margins.addTag(0x22, 0x0035);	// 53
            margins.addTag(0x23, 0x0005);	// 5
            margins.addTag(0x24, 0x002a);	// 42
            margins.addTag(0x2c, 0x002a);	// 42
            margins.addTag(0x25, 0x02a2);	// 674
            margins.addTag(0x26, 0x0204);	// 516
            margins.addTag(0x27, 0x003a);	// 58
            margins.addTag(0x28, 0x0035);	// 53
            margins.addTag(0x35, 0x0034);	// 52
            margins.addTag(0x2b, 0x0000);
            margins.addTag(0x2a, 0x0001);
            margins.addTag(0xda, 0x0002);

            byte[] sixBytes = new byte[] { 0x01, 0x00, 0x00, 0x00, 0x00, 0x00 };
            margins.addTag(0x29, sixBytes);

            margins_id = margins.id;

            BBeBObject pageBox = new BBeBObject(ref _this, CBBeBFile.OBJECT_TYPE_BlockAtr);
            pageBox.addTag(0x31, 600);
            pageBox.addTag(0x32, 800);
            pageBox.addTag(0x33, 0x0012);
            pageBox.addTagInt(0x34, 0x00ff);
            pageBox.addTag(0x35, 0x0034);
            pageBox.addTag(0x36, 0x0000);
            pageBox.addTagInt(0x37, 0);
            pageBox.addTag(0x2e, 0x0001);
            pageBox.addTag(0x38, 0x0000);
            pageBox.addTag(0x39, 0x0000);
            pageBox.addTag(0x29, sixBytes);

            page_box_id = pageBox.id;
        }
        #endregion

        #region ParseHTML
        public void ParseHTML(string aHTMLFileName)
        {
            HtmlParser htmlParser = new HtmlParser(this, aHTMLFileName);
            htmlParser.parsePages();
        }
        #endregion

        #region outAppend
        public void outAppend(int c)
        {
            if ((outBufOffset + 2) > outBuf.Length)
            {
                byte[] newBuf = new byte[outBuf.Length + 8192];
                System.Array.Copy(outBuf, newBuf, outBuf.Length);
                outBuf = newBuf;
            }
            outBuf[outBufOffset++] = (byte)(c & 0x00ff);
            outBuf[outBufOffset++] = (byte)((c >> 8) & 0x00ff);
        }
        #endregion

        #region CreateMetaData
        public byte[] CreateMetaData()
        {
            if (bookTitle.Length == 0)
            {
                System.Diagnostics.Debug.WriteLine("WARNING: Book Title not supplied, setting to 'UNKNOWN'.");
                bookTitle.Append("UNKNOWN");
            }
            if (bookID.Length == 0)
            {
                bookID.Append("LBBB").Append(DateTime.Now.Millisecond.ToString());
            }
            if (bookID.Length > 16)
            {
                System.Diagnostics.Debug.WriteLine("BookID is longer than 16 characters. It will be trimmed.");
                bookID = new StringBuilder(bookID.ToString().Substring(0, 16));
            }
            if (bookLabel.Length == 0)
            {
                System.Diagnostics.Debug.WriteLine("WARNING: Book Label not supplied, setting to 'UNKNOWN'.");
                bookLabel.Append("UNKNOWN");
            }
            if (docCreationDate.Length == 0)
            {
                docCreationDate.Append(DateTime.Now.Millisecond.ToString());
            }
            if (docCreationDate.Length > 15)
            {
                System.Diagnostics.Debug.WriteLine("Document creation data is longer than 15 characters. It will be trimmed.");
                docCreationDate = new StringBuilder(bookID.ToString().Substring(0, 15));
            }

            StringBuilder s = new StringBuilder(4096);
            s.Append((char)0xFEFF);
            s.Append("<?xml version=\"1.0\" encoding=\"UTF-16\" ?>\n");
            s.Append("<Info version=\"1.0\" >\n");
            s.Append("  <BookInfo>\n");
            s.Append("    <Title>").Append(bookTitle).Append("</Title>\n");
            s.Append("    <Author>").Append(bookAuthor).Append("</Author>\n");
            s.Append("    <BookID>").Append(bookID).Append("</BookID>\n");
            s.Append("    <Publisher>").Append(bookPublisher).Append("</Publisher>\n");
            s.Append("    <Label>").Append(bookLabel).Append("</Label>\n");
            s.Append("    <Category>").Append(bookCategory).Append("</Category>\n");
            s.Append("    <Classification>").Append(bookClassification).Append("</Classification>\n");
            if (bookFreeText.Length != 0)
            {
                s.Append("    <FreeText>").Append(bookFreeText).Append("</FreeText>\n");
            }
            s.Append("  </BookInfo>\n");
            s.Append("  <DocInfo>\n");
            s.Append("    <Language>").Append(docLanguage).Append("</Language>\n");
            s.Append("    <Creator>").Append(docCreator).Append("</Creator>\n");
            s.Append("    <CreationDate>").Append(docCreationDate).Append("</CreationDate>\n");
            s.Append("    <Producer>Legally Blind Book Binder 2.0.1</Producer>\n");
            s.Append("    <Page>").Append(pages).Append("</Page>\n");
            s.Append("  </DocInfo>\n");
            s.Append("</Info>\n");

            //	System.out.println(s);

            return System.Text.ASCIIEncoding.UTF8.GetBytes(s.ToString());
        }
        #endregion

        #region WriteBookToFile
        public void WriteBookToFile(string lrfFile, string iconFileName)
        {
            CBBeBFile _this = this;
            byte[] pageNumberData = new byte[(6 * pages) + 4];
            byte[] bookPagesList = new byte[(4 * pages) + 2];
            pageNumberData[0] = bookPagesList[0] = (byte)(pages & 0x00ff);
            pageNumberData[1] = bookPagesList[1] = (byte)((pages >> 8) & 0x00ff);

            int pnIndex = 4;
            int bpIndex = 2;
            for (int i = 0; i < objectList.Count; i++)
            {
                BBeBObject subFile = (BBeBObject)objectList[i];
                if (subFile.type == OBJECT_TYPE_Page)
                {
                    pageNumberData[pnIndex] = bookPagesList[bpIndex] = (byte)(subFile.id & 0x00ff);
                    pageNumberData[pnIndex + 1] = bookPagesList[bpIndex + 1] = (byte)((subFile.id >> 8) & 0x00ff);

                    // Set to say 1 page per page (a kludge for now)
                    pageNumberData[pnIndex + 4] = 1;
                    pageNumberData[pnIndex + 5] = 0;

                    pnIndex += 6;
                    bpIndex += 4;
                }
            }

            BBeBObject pageNumbers = new BBeBObject(ref _this, OBJECT_TYPE_ObjectInfo, FLAG_PAGE_NUMBERS, pageNumberData);

            bookFile.addTagInt(0x02, pageNumbers.id);
            bookFile.addTag(0x5c, bookPagesList);

            addLRF(metaData, thumbnail);
        }
        #endregion

        #region Write
        public void Write(string lrfFile)
        {
            BinaryWriter br = new BinaryWriter(File.Create(lrfFile));
            br.Write(buf, 0, getPutPosition());
            br.Close();
        }
        #endregion

        #region putByte
        public void putByte(int aValue)
        {
            buf[currentPosition++] = (byte)aValue;
        }

        public void putByte(int offset, int aValue)
        {
            buf[offset] = (byte)aValue;
        }

        public void putBytes(byte[] aValue)
        {
            for (int i = 0; i < aValue.Length; i++)
                buf[currentPosition++] = aValue[i];
        }

        public void putBytes(byte[] aValue, int offset, int length)
        {
            for (int i = 0; i < length; i++)
                buf[currentPosition++] = aValue[i];
        }

        public void putShort(byte[] buf, int offset, int aValue)
        {
            buf[offset + 0] = (byte)(aValue & 0x00ff);
            buf[offset + 1] = (byte)((aValue >> 8) & 0x00ff);
        }

        public void putShort(int aValue)
        {
            buf[currentPosition++] = (byte)aValue;
        }

        public void putShort(int offset, int aValue)
        {
            buf[offset] = (byte)aValue;
        }

        public void putChar(byte[] buf, int offset, char aValue)
        {
            buf[offset + 0] = (byte)(aValue & 0x00ff);
            buf[offset + 1] = (byte)((aValue >> 8) & 0x00ff);
        }

        public  void putChar(char aValue)
        {
            buf[currentPosition++] = (byte)aValue;
        }

        public  void putChar(int offset, char aValue)
        {
            buf[offset] = (byte)aValue;
        }

        public  void putInt(byte[] buf, int offset, int aValue)
        {
            buf[offset + 0] = (byte)(aValue & 0x00ff);
            buf[offset + 1] = (byte)((aValue >> 8) & 0x00ff);
            buf[offset + 2] = (byte)((aValue >> 16) & 0x00ff);
            buf[offset + 3] = (byte)((aValue >> 24) & 0x00ff);
        }

        public  void putInt(int aValue)
        {
            buf[currentPosition++] = (byte)aValue;
        }

        public  void putInt(int offset, int aValue)
        {
            buf[offset] = (byte)aValue;
        }

        public  void setPutPosition(int position)
        {
            currentPosition = position;
        }

        public  int getPutPosition()
        {
            return currentPosition;
        }

        public  void alignPutPosition(int alignment)
        {
            int pos = currentPosition;
            int mod = pos % alignment;
            if (mod != 0)
            {
                currentPosition = pos + (alignment - mod);
            }
        }
        #endregion

        #region addLRF
        private void addLRF(byte[] metaData, byte[] thumbnail)
        {
            // Put magic marker at front
            putChar(0, 'L');
            putChar(2, 'R');
            putChar(4, 'F');

            // Put LRF verion in file
            putShort(LRF_VERSION, 999);	// 0x3E7

            // Put in an encryption key (place holder for now)
            putShort(LRF_PSUEDOKEY, 0x30);

            // Set book direction to forwards
            putByte(LRF_DIRECTION, LRF_DIRECTION_FORWARDS);

            // Put Dimensions
            putShort(LRF_UNK1, 800 * 2);
            putShort(LRF_UNK2, 600);
            putShort(LRF_UNK3, 800);

            // Various unknown (but critical) values
            putByte(LRF_UNK4, 0x18);

            putShort(LRF_TOC_OFFSET, 0x1536);	// XXX is this legit?

            putByte(LRF_UNK6, 0x14);

            // Set position to just past header
            setPutPosition(0x54);

            // XML file containing metadata
            int compressedLength = putCompressedBytes(metaData);
            putShort(LRF_INFOLEN, compressedLength + 4);

            // Thumbnail (60x80)
            putInt(LRF_THUMBNAIL_LENGTH, thumbnail.Length);
            putBytes(thumbnail);

            // 16-byte align before placing first BBeBObject into file.
            alignPutPosition(16);

            // Put BBeBObjects into file
            for (int i = 0; i < objectList.Count; i++)
            {
                ((BBeBObject)objectList[i]).addToBuffer();
            }

            // 16-byte align before placing BBeBObject tree into file
            alignPutPosition(16);

            // Put BBEBObject info into LRF header
            // And yes, these are technically 64 bits each...
            putInt(LRF_OBJECT_COUNT, objectList.Count);
            putInt(LRF_OBJECTTREE_OFFSET, getPutPosition());

            // Put BBeBObject tree into file
            for (int i = 0; i < objectList.Count; i++)
            {
                BBeBObject o = (BBeBObject)objectList[i];
                putInt(o.id);
                putInt(o.location);
                putInt(o.size);
                putInt(0);
            }
        }
        #endregion

        #region putCompressedBytes
        private int putCompressedBytes(byte[] bytes)
        {
            putInt(bytes.Length);

            byte[] outBytes = new byte[bytes.Length + ((bytes.Length / 1000) + 1) + 12 + 4];

            MemoryStream outFileStream = new MemoryStream();
            MemoryStream inFileStream = new MemoryStream();
            inFileStream.Write(bytes, 0, bytes.Length);
            inFileStream.Seek(0, SeekOrigin.Begin);

            System.IO.Compression.DeflateStream ds = new System.IO.Compression.DeflateStream(inFileStream, System.IO.Compression.CompressionMode.Compress);
            ds.Write(bytes, 0, bytes.Length);
            ds.Flush();

            int ii = (int)inFileStream.Position;
            inFileStream.Read(outBytes, 0, ii);
            int compressedDataLength = ii;

            Array.Copy(outBytes, 0, buf, currentPosition, ii);

            currentPosition += compressedDataLength;

            return compressedDataLength;

            //MemoryStream ms = new MemoryStream();
            //System.IO.Compression.DeflateStream ds = new System.IO.Compression.DeflateStream(ms, System.IO.Compression.CompressionMode.Compress);
            //ds.Write(bytes, 0, bytes.Length);
            //ds.Flush();

            ////ds.Close();

            //int compressedDataLength = (int)ms.Length;

            //ms.Seek(0, SeekOrigin.Begin);

            //ms.Read(outBytes, 0, (int)ms.Length);

            //Array.Copy(outBytes, 0, buf, currentPosition, (int)ms.Length);

            //currentPosition += (int)ms.Length;

            //return (int)ms.Length;
        }
        #endregion

        public void CopyStream(System.IO.Stream input, System.IO.Stream output)
        {
            byte[] buffer = new byte[2000];
            int len;
            while ((len = input.Read(buffer, 0, 2000)) > 0)
            {
                output.Write(buffer, 0, len);
            }
            output.Flush();
        }

        #region addOutBufAsTextPage
        public void addOutBufAsTextPage()
        {
            CBBeBFile _this = this;
            // If we have more than start page / end page
            if (outBufOffset != 8)
            {
                byte[] outBytes = outBuf;
                int flags = 0;
                if (outBufOffset > 64)
                {
                    // Allocate a buffer to compress into
                    outBytes = new byte[outBuf.Length];

                    // Stash uncompressed size
                    putInt(outBytes, 0, outBufOffset);

                    // Deflate text
                    MemoryStream inFileStream = new MemoryStream();
                    System.IO.Compression.DeflateStream ds = new System.IO.Compression.DeflateStream(inFileStream, System.IO.Compression.CompressionMode.Compress, true);
                    ds.Write(outBuf, 0, outBuf.Length);
                    ds.Flush();

                    byte[] buffer = new byte[inFileStream.Length];
                    inFileStream.Seek(0, SeekOrigin.Begin);
                    inFileStream.Read(buffer, 0, (int)inFileStream.Length);

                    outBufOffset = (int)inFileStream.Length;

                    Array.Copy(buffer, 0, buf, currentPosition, (int)inFileStream.Length);

                    currentPosition += outBufOffset;

                    //MemoryStream ms = new MemoryStream();
                    //System.IO.Compression.DeflateStream ds = new System.IO.Compression.DeflateStream(ms, System.IO.Compression.CompressionMode.Compress);
                    //ds.Write(outBuf, 0, outBufOffset);
                    //ds.Flush();
                    ////ds.Close();

                    //ms.Seek(0, SeekOrigin.Begin);
                    //outBufOffset = (int)ms.Length + 4;
                    //flags = FLAG_COMPRESSED;

                    //ms.Read(outBuf, 0, outBufOffset);
                }

                addTextPage(new BBeBObject(ref _this, OBJECT_TYPE_Text, flags, outBytes, outBufOffset));
            }

            // Ready for next page
            outBufOffset = 0;
        }
        #endregion

        #region addTextPage
        protected void addTextPage(BBeBObject text)
        {
            CBBeBFile _this = this;

            // font size?? para.addTag( 0x11, 80);
            text.addTagInt(0x03, font_id);

            /* The text goes into a bounding box */
            byte[] boxData = new byte[6];
            putShort(boxData, 0, 0x00f503);
            putInt(boxData, 2, text.id);

            BBeBObject box = new BBeBObject(ref _this, OBJECT_TYPE_Block, 0, boxData);
            box.addTagInt(0x03, page_box_id);

            byte[] pageFiles = new byte[14];
            putShort(pageFiles, 0, 3);
            putInt(pageFiles, 2, font_id);
            putInt(pageFiles, 6, text.id);
            putInt(pageFiles, 10, box.id);

            byte[] pageData = new byte[6];
            putShort(pageData, 0, 0x00f503);
            putInt(pageData, 2, box.id);

            BBeBObject page = new BBeBObject(ref _this, OBJECT_TYPE_Page, 0, pageData);
            page.addTagInt(0x7c, bookFile.id);
            page.addTag(0x0b, pageFiles);
            page.addTagInt(0x03, margins_id);


            // We make the layout big so paragraphs can break over display pages
            byte[] layoutData = new byte[24];
            putInt(layoutData, 0, 1);
            putInt(layoutData, 4, box.id);
            BBeBObject physicalPages = new BBeBObject(ref _this, OBJECT_TYPE_ObjectInfo, FLAG_PAGE_LAYOUT, layoutData);
            page.addTagInt(0x02, physicalPages.id);

            pages++;
        }
        #endregion
    }
}
