/* 
 * Copyright (C) 2005 and 2006, Scott Turner scotty1024@mac.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.io.File;
import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.zip.Deflater;
import java.util.zip.Inflater;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserFactory;
import org.xmlpull.v1.XmlPullParserException;
import java.awt.image.BufferedImage;
import javax.imageio.ImageIO;
import javax.imageio.IIOImage;
import javax.imageio.stream.ImageOutputStream;
import javax.imageio.ImageWriter;
import javax.imageio.ImageWriteParam;
import org.jpedal.PdfDecoder;
import org.jpedal.objects.PdfPageData;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

public class BBeBook extends JPanel implements ActionListener {

    static final boolean verbose = false;


    /* Primary LRF header offsets */
    static final int LRF_SIGNATURE1 = 0x00;	// UTF-16LE "LRF\000"
    static final int LRF_VERSION = 0x08;	// file version (short)
    static final int LRF_PSUEDOKEY = 0x0A;	// XOR key (short)
    static final int LRF_ROOT_ID = 0x0C;	// (int)
    static final int LRF_OBJECT_COUNT = 0x10;	// (long) count of objects
    static final int LRF_OBJECTTREE_OFFSET = 0x18;// (long) offset to object table

    /* Secondary LRF header offsets */
    static final int LRF_UNK0 = 0x20;		// (int)
    static final int LRF_DIRECTION = 0x24;	// (byte)
    static final int LRF_DIRECTION_FORWARDS = 0x01;
    static final int LRF_DIRECTION_BACKWARDS = 0x10;
    static final int LRF_UNK1 = 0x26;		// (short)
    static final int LRF_UNK2 = 0x2A;		// (short)
    static final int LRF_UNK3 = 0x2C;		// (short)
    static final int LRF_UNK4 = 0x2E;		// (byte)
    static final int LRF_SIGNATURE2 = 0x30;	// 5 int's (some kind of signature)
    static final int LRF_TOC_ID = 0x44;		// (int)
    static final int LRF_TOC_OFFSET = 0x48;	// (int)
    static final int LRF_INFOLEN = 0x4C;	// Compressed metaData size (short)

    // Next two only walid if version >= 800
    static final int LRF_UNK7 = 0x4E;		// (short)
    static final int LRF_THUMBNAIL_LENGTH = 0x50;// GIF thumbnail size (int)

    static final int LRF_UNK6 = 0x4E;

    static final byte OBJECT_TYPE_INVALID_00	= 0x00;
    static final byte OBJECT_TYPE_PageTree	= 0x01;
    static final byte OBJECT_TYPE_Page		= 0x02;
    static final byte OBJECT_TYPE_Header	= 0x03;
    static final byte OBJECT_TYPE_Footer	= 0x04;
    static final byte OBJECT_TYPE_PageAtr	= 0x05;
    static final byte OBJECT_TYPE_Block		= 0x06;
    static final byte OBJECT_TYPE_BlockAtr	= 0x07;
    static final byte OBJECT_TYPE_MiniPage	= 0x08;
    static final byte OBJECT_TYPE_BlockList	= 0x09;
    static final byte OBJECT_TYPE_Text		= 0x0a;
    static final short FLAG_COMPRESSED	= 0x0100;
    static final short FLAG_SCRAMBLED	= 0x0200;
    static final int FLAG_ENCRYPTED	= 0x8000;
    static final byte OBJECT_TYPE_TextAtr	= 0x0b;
    static final byte OBJECT_TYPE_Image		= 0x0c;
    static final byte OBJECT_TYPE_Canvas	= 0x0d;
    static final byte OBJECT_TYPE_ParagraphAtr	= 0x0e;
    static final byte OBJECT_TYPE_Invalid_0f	= 0x0f;
    static final byte OBJECT_TYPE_Invalid_10	= 0x10;
    static final byte OBJECT_TYPE_ImageStream	= 0x11;
    static final byte OBJECT_TYPE_Import	= 0x12;
    static final byte OBJECT_TYPE_Button	= 0x13;
    static final byte OBJECT_TYPE_Window	= 0x14;
    static final byte OBJECT_TYPE_PopUpWin	= 0x15;
    static final byte OBJECT_TYPE_Sound		= 0x16;
    static final byte OBJECT_TYPE_PlaneStream	= 0x17;
    static final byte OBJECT_TYPE_Invalid_18	= 0x18;
    static final byte OBJECT_TYPE_Font		= 0x19;
    static final byte OBJECT_TYPE_ObjectInfo	= 0x1a;
    static final short FLAG_PAGE_NUMBERS = 0x0081;
    static final short FLAG_PAGE_LAYOUT = 0x0082;
    static final byte OBJECT_TYPE_Invalid_1b	= 0x1b;
    static final byte OBJECT_TYPE_BookAtr	= 0x1c;
    static final byte OBJECT_TYPE_SimpleText	= 0x1d;
    static final byte OBJECT_TYPE_TOC		= 0x1e;
    static final short FLAG_TOC_51 = 0x0051;
    static final byte OBJECT_TYPE_Invalid_1f	= 0x1f;
    /*
F500: *ObjectStart
F501: *ObjectEnd
F502: *ObjectInfoLink
F503: *Link
F504: *StreamSize
F505: *StreamStart
F506: *StreamEnd
F50B: *ContainedObjectsList

F511: FontSize
F512: FontWidth
F513: FontEscapement
F514: FontOrientation
F515: FontWeight
F516: FontFacename
F517: TextColor
F518: TextBgColor
F519: WordSpace
F51A: LetterSpace
F51B: BaseLineSkip
F51C: LineSpace
F51D: ParIndent
F51E: ParSkip

F525: *PageHeight
F526: *PageWidth

F531: BlockWidth
F532: BlockHeight
F533: BlockRule

F541: *MiniPageHeight
F542: *MiniPageWidth
F546: *LocationY
F547: *LocationX
F548: ?
F549: PutSound
F54A: *ImageRect
F54B: *ImageSize
F54C: *ImageStream
F54D: ?
F54E: ?

F551: *CanvasWidth
F552: *CanvasHeight
F554: *StreamFlags

F559: *FontFileName

F55B: ViewPoint
F55C: *PageList
F55D: *FontFaceName

F56C: *JumpTo

F573: RuledLine
F575: RubyAlign
F576: RubyOverhang
F577: EmpDotsPosition
F578: EmpDotsCode
F579: EmpLinePosition
F57A: EmpLineMode
F57B: *ChildPageTree
F57C: *ParentPageTree

F581: Italic
F582: Italic


F5A1: BeginPage
F5A2: EndPage
F5A5: KomaGaiji
F5A6: KomaEmpDotChar?
F5A7: BeginButton
F5A8: EndButton
F5A9: BeginRuby
F5AA: EndRuby
F5AB: *BeginRubyBase
F5AC: *EndRubyBase
F5AD: *BeginRubyText
F5AE: *EndRubyText

F5B1: KomaYokomoji 
F5B2: ?
F5B3: Tate
F5B4: Tate
F5B5: Nekase
F5B6: Nekase
F5B7: BeginSup
F5B8: EndSup
F5B9: BeginSub
F5BA: EndSub
F5BB: ?
F5BC: ?
F5BD: ?
F5BE: ?

F5C1: BeginEmpLine
F5C2: ?
F5C3: BeginDrawChar
F5C4: EndDrawChar
F5C5: ?
F5C6: ?
F5C7: ?
F5C8: KomaAutoSpacing
F5C9: ?
F5CA: Space
F5CB: ?
F5CC: ?

F5D1: KomaPlot
F5D2: EOL
F5D4: Wait
F5D6: SoundStop
F5D7: MoveObj
F5D8: *BookFont
F5D9: KomaPlotText
F5DD: CharSpace

F5F1: LineWidth
F5F2: LineColor
F5F3: FillColor
F5F4: LineMode
F5F5: MoveTo
F5F6: LineTo
F5F7: DrawBox
F5F8: DrawEllipse
    */

    private short globalBBeBObject;
    private ByteBuffer buf;

    /**
     * List of BBeBObject instances added to this BBeBook
     */
    private ArrayList objectList = new ArrayList();

    int pages = 0;
    BBeBObject bookFile = null;
    int margins_id = 0;
    int page_box_id = 0;
    int font_id = 0;
    private byte[] metaData = null;
    private byte[] thumbnail = null;
    private byte[] fontname = null;

    private StringBuffer bookFileName = new StringBuffer();
    private StringBuffer bookTitle = new StringBuffer();
    private StringBuffer bookAuthor = new StringBuffer("Unknown Author");
    private StringBuffer bookID = new StringBuffer();
    private StringBuffer bookPublisher = new StringBuffer("Unknown Publisher");
    private StringBuffer bookLabel = new StringBuffer();
    private StringBuffer bookCategory = new StringBuffer("Unknown");
    private StringBuffer bookClassification = new StringBuffer("Unknown");
    private StringBuffer bookFreeText = new StringBuffer();
    private StringBuffer bookIconName = new StringBuffer();
    private StringBuffer bookCoverName = new StringBuffer();
    private StringBuffer docLanguage = new StringBuffer("en");
    private StringBuffer docCreator = new StringBuffer("Unknown Creator");
    private StringBuffer docCreationDate = new StringBuffer();
    private StringBuffer docFileName = new StringBuffer();
    private HashMap configTags = new HashMap(32);

    public static void main(String[] args)
	throws Exception
    {
	BBeBook book = new BBeBook();
	if (args.length == 0) {
	    book.makeFrame();
	} else {
	    book.makeBook(args[0]);
	}
    }

    private JTextField tfBookFileName = new JTextField(32);
    private JTextField tfBookTitle = new JTextField(32);
    private JTextField tfBookAuthor = new JTextField(32);
    private JTextField tfBookID = new JTextField(16);
    private JTextField tfBookPublisher = new JTextField(32);
    private JTextField tfBookLabel = new JTextField(32);
    private JTextField tfBookCategory = new JTextField(32);
    private JTextField tfBookClassification = new JTextField(32);
    private JTextField tfBookFreeText = new JTextField(32);
    private JTextField tfBookIconName = new JTextField(32);
    private JTextField tfBookCoverName = new JTextField(32);
    private JTextField tfDocLanguage = new JTextField("en");
    private JTextField tfDocCreator = new JTextField(32);
    private JTextField tfDocCreationDate = new JTextField(32);
    private JTextField tfDocFileName = new JTextField(32);
    private JLabel statusLabel = new JLabel("           ");
    private JLabel statusLabel2 = new JLabel("           ");
    private JFrame frame;
    private JButton bindButton = new JButton("Bind Book");
    private JButton clearButton = new JButton("Bind Book");
    private JButton bookButton = new JButton("Choose");
    private JButton iconButton = new JButton("Choose");

    private void addComponent(Component cc,
GridBagLayout gb,
GridBagConstraints c, int gridX, int gridY) {
c.gridx = gridX;
c.gridy = gridY;
	gb.setConstraints(cc, c);
	add(cc);
    }

    public void makeFrame() {
        //Make sure we have nice window decorations.
        JFrame.setDefaultLookAndFeelDecorated(true);

        //Create and set up the window.
	frame = new JFrame("BBeBook");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        // set us as the content pane.
	setOpaque(true); //content panes must be opaque
        frame.setContentPane(this);

	frame.getContentPane().setPreferredSize(new Dimension(500, 400));
	GridBagConstraints c = new GridBagConstraints();
//	c.fill = GridBagConstraints.NONE;
	c.fill = GridBagConstraints.HORIZONTAL;
	c.anchor = GridBagConstraints.EAST;

	GridBagLayout gridBag = new GridBagLayout();
	setLayout(gridBag);

	tfBookFileName.addActionListener(this);
	int gridY = 0;
	addComponent(new JLabel("Book Filename: "), gridBag, c, 0, gridY);
	addComponent(tfBookFileName, gridBag, c, 1, gridY);
	bookButton.addActionListener(this);
	addComponent(bookButton, gridBag, c, 2, gridY);

gridY++;

	tfBookTitle.addActionListener(this);
	addComponent(new JLabel("Book Title: "), gridBag, c, 0, gridY);
	addComponent(tfBookTitle, gridBag, c, 1, gridY);

gridY++;

	tfBookAuthor.addActionListener(this);
	addComponent(new JLabel("Book Author: "), gridBag, c, 0, gridY);
	addComponent(tfBookAuthor, gridBag, c, 1, gridY);

gridY++;

	tfBookID.addActionListener(this);
	addComponent(new JLabel("Book ID: "), gridBag, c, 0, gridY);
	addComponent(tfBookID, gridBag, c, 1, gridY);

gridY++;

	tfBookPublisher.addActionListener(this);
	addComponent(new JLabel("Book Publisher: "), gridBag, c, 0, gridY);
	addComponent(tfBookPublisher, gridBag, c, 1, gridY);

gridY++;

	tfBookLabel.addActionListener(this);
	addComponent(new JLabel("Book Label: "), gridBag, c, 0, gridY);
	addComponent(tfBookLabel, gridBag, c, 1, gridY);

gridY++;

	tfBookCategory.addActionListener(this);
	addComponent(new JLabel("Book Category: "), gridBag, c, 0, gridY);
	addComponent(tfBookCategory, gridBag, c, 1, gridY);

gridY++;

	tfBookClassification.addActionListener(this);
	addComponent(new JLabel("Book Classification: "), gridBag, c, 0, gridY);
	addComponent(tfBookClassification, gridBag, c, 1, gridY);

gridY++;

	tfBookFreeText.addActionListener(this);
	addComponent(new JLabel("Book Memo: "), gridBag, c, 0, gridY);
	addComponent(tfBookFreeText, gridBag, c, 1, gridY);

gridY++;

	tfBookIconName.addActionListener(this);
	addComponent(new JLabel("Icon Name: "), gridBag, c, 0, gridY);
	addComponent(tfBookIconName, gridBag, c, 1, gridY);
	iconButton.addActionListener(this);
	addComponent(iconButton, gridBag, c, 2, gridY);

gridY++;

	tfBookCoverName.addActionListener(this);
	addComponent(new JLabel("Cover Name: "), gridBag, c, 0, gridY);
	addComponent(tfBookCoverName, gridBag, c, 1, gridY);

gridY++;

	tfDocLanguage.addActionListener(this);
	addComponent(new JLabel("Document Language: "), gridBag, c, 0, gridY);
	addComponent(tfDocLanguage, gridBag, c, 1, gridY);

gridY++;

	tfDocCreator.addActionListener(this);
	addComponent(new JLabel("Document Creator: "), gridBag, c, 0, gridY);
	addComponent(tfDocCreator, gridBag, c, 1, gridY);

gridY++;

	tfDocCreationDate.addActionListener(this);
	addComponent(new JLabel("Document Creation Date: "), gridBag, c, 0, gridY);
	addComponent(tfDocCreationDate, gridBag, c, 1, gridY);

gridY++;

	tfDocFileName.addActionListener(this);
	addComponent(new JLabel("Document Filename: "), gridBag, c, 0, gridY);
	addComponent(tfDocFileName, gridBag, c, 1, gridY);

gridY++;

	bindButton = new JButton("Bind Book");
	bindButton.addActionListener(this);
	addComponent(bindButton, gridBag, c, 0, gridY);

	clearButton = new JButton("Clear Fields");
	clearButton.addActionListener(this);
	addComponent(clearButton, gridBag, c, 1, gridY);

gridY++;

	addComponent(statusLabel, gridBag, c, 0, gridY);
	//	addComponent(statusLabel2);

        //Display the window.
        frame.pack();
        frame.setVisible(true);
    }

    private BookFilter bookFilter = new BookFilter();
    private IconFilter iconFilter = new IconFilter();

    public void actionPerformed(ActionEvent evt) {
	//System.out.println("Event fired: " + evt);
	statusLabel.setText("");
	if (evt.getSource() == bindButton) {
	    // Bind book
	    bindBook();
	} else if (evt.getSource() == clearButton) {
	    // clear fields
	    clearBook();
	} else if (evt.getSource() == bookButton) {
	    JFileChooser chooser = new JFileChooser();
	    chooser.setFileFilter(bookFilter);
	    int returnVal = chooser.showOpenDialog(getParent());

	    if(returnVal == JFileChooser.APPROVE_OPTION) {
		tfBookFileName.setText(chooser.getSelectedFile().getName());
	    }
	} else if (evt.getSource() == iconButton) {
	    JFileChooser chooser = new JFileChooser();
	    chooser.setFileFilter(iconFilter);
	    int returnVal = chooser.showOpenDialog(getParent());

	    if(returnVal == JFileChooser.APPROVE_OPTION) {
		tfBookIconName.setText(chooser.getSelectedFile().getName());
	    }
	}
    }

    private void bindBook() {
	bookFileName.setLength(0);
	bookFileName.append(tfBookFileName.getText());
	bookTitle.setLength(0);
	bookTitle.append(tfBookTitle.getText());
	bookAuthor.setLength(0);
	bookAuthor.append(tfBookAuthor.getText());
	bookID.setLength(0);
	bookID.append(tfBookID.getText());
	bookPublisher.setLength(0);
	bookPublisher.append(tfBookPublisher.getText());
	bookLabel.setLength(0);
	bookLabel.append(tfBookLabel.getText());
	bookCategory.setLength(0);
	bookCategory.append(tfBookCategory.getText());
	bookClassification.setLength(0);
	bookClassification.append(tfBookClassification.getText());
	bookFreeText.setLength(0);
	bookFreeText.append(tfBookFreeText.getText());
	bookIconName.setLength(0);
	bookIconName.append(tfBookIconName.getText());
	bookCoverName.setLength(0);
	bookCoverName.append(tfBookCoverName.getText());
	docLanguage.setLength(0);
	docLanguage.append(tfDocLanguage.getText());
	docCreator.setLength(0);
	docCreator.append(tfDocCreator.getText());
	docCreationDate.setLength(0);
	docCreationDate.append(tfDocCreationDate.getText());
	docFileName.setLength(0);
	docFileName.append(tfDocFileName.getText());

	if (bookFileName.length() == 0) {
	    statusLabel.setText("No Book Filename");
	    frame.show();
	    return;
	}
	if (docFileName.length() == 0) {
	    statusLabel.setText("No Document Filename");
	    frame.show();
	    return;
	}

	startNewBook();

	if (bookFileName.toString().trim().toLowerCase().endsWith(".html")) {
	    parseHTML(bookFileName.toString());
	} else if (bookFileName.toString().trim().toLowerCase().endsWith(".pdf")) {
	    //addPDFToBook(bookFileName.toString());
	    parsePDF(bookFileName.toString());
	} else {
	    addTextToBook(bookFileName.toString());
	}

	metaData = createMetaData();
	writeBookToFile(docFileName.toString(), bookIconName.toString()); 

	try {
	    write(docFileName.toString());
	} catch (IOException e) {
	    statusLabel.setText("Error writing document: " + e.getMessage());
	    frame.show();
	}
    }

    private void clearBook() {
	tfBookFileName.setText("");
	tfBookTitle.setText("");
	tfBookAuthor.setText("");
	tfBookID.setText("");
	tfBookPublisher.setText("");
	tfBookLabel.setText("");
	tfBookCategory.setText("");
	tfBookClassification.setText("");
	tfBookFreeText.setText("");
	tfBookIconName.setText("");
	tfBookCoverName.setText("");
	tfDocLanguage.setText("en");
	tfDocCreator.setText("");
	tfDocCreationDate.setText("");
	tfDocFileName.setText("");
    }

    private void makeBook(String aConfigFileName)
	throws IOException
    {
	if (aConfigFileName.trim().toLowerCase().endsWith(".pdf")) {
	    bookTitle.append(aConfigFileName);
	    bookID.append(aConfigFileName);
	    docFileName.append(aConfigFileName + ".lrf");
	    bookIconName.append("image.gif");
	    bookLabel.append(aConfigFileName);

	    startNewBook();

	    //addPDFToBook(bookFileName.toString());
	    parsePDF(aConfigFileName);

	    metaData = createMetaData();
	    writeBookToFile(docFileName.toString(), bookIconName.toString()); 
	    write(docFileName.toString());
	} else {
	    parseConfig(aConfigFileName);

	    if (bookFileName.length() == 0) {
		System.err.println("No <File> in config file.");
		System.exit(1);
	    }
	    if (docFileName.length() == 0) {
		System.err.println("No <Output> in config file.");
		System.exit(1);
	    }

	    startNewBook();

	    if (bookFileName.toString().trim().toLowerCase().endsWith(".html")) {
		//addPDFToBook(bookFileName.toString());
		parseHTML(bookFileName.toString());
	    } else if (bookFileName.toString().trim().toLowerCase().endsWith(".pdf")) {
		//addPDFToBook(bookFileName.toString());
		parsePDF(bookFileName.toString());
	    } else {
		addTextToBook(bookFileName.toString());
	    }

	    metaData = createMetaData();
	    writeBookToFile(docFileName.toString(), bookIconName.toString()); 
	    write(docFileName.toString());
	}
    }

    BBeBook() {
	configTags.put("File", bookFileName);          
	configTags.put("Title", bookTitle);          
	configTags.put("Author", bookAuthor);	    
	configTags.put("BookID", bookID);	    
	configTags.put("Publisher", bookPublisher);	    
	configTags.put("Label", bookLabel);	    
	configTags.put("Category", bookCategory);	    
	configTags.put("Classification", bookClassification); 
	configTags.put("FreeText", bookFreeText);	    
	configTags.put("Icon", bookIconName);
	configTags.put("Cover", bookCoverName);
	configTags.put("Language", docLanguage);	    
	configTags.put("Creator", docCreator);	    
	configTags.put("CreationDate", docCreationDate);   
	configTags.put("Output", docFileName);          

	// Magic.. Ask Sony why all LRF files start here
	globalBBeBObject = 0x32;

	// Grab a buffer that is hopefully big enough for the book
	buf = ByteBuffer.allocate(16 * 1024 * 1024);

	// All BBeB's are Little Endian
	buf.order(ByteOrder.LITTLE_ENDIAN);
    }

    void parseConfig(String configFileName) {
	try {
	    XmlPullParserFactory factory = XmlPullParserFactory.newInstance(System.getProperty(XmlPullParserFactory.PROPERTY_NAME), null);
	    factory.setFeature(XmlPullParser.FEATURE_PROCESS_NAMESPACES, true);
	    
	    XmlPullParser xpp = factory.newPullParser();

	    xpp.setInput ( new FileReader(new File(configFileName)));

	    int holderForStartAndLength[] = new int[2];
	    String name = null;

	    for (int eventType = xpp.getEventType(); eventType != XmlPullParser.END_DOCUMENT; eventType = xpp.next()) {

		if (eventType == XmlPullParser.START_TAG) {
		    name = xpp.getName();
		} else if ((eventType == XmlPullParser.TEXT) && !xpp.isWhitespace()) {
		    char ch[] = xpp.getTextCharacters(holderForStartAndLength);
		    if (holderForStartAndLength[1] != 0) {
			StringBuffer config = (StringBuffer)configTags.get(name);
			if (config != null) {
			    config.setLength(0);
			    config.append(ch, holderForStartAndLength[0], holderForStartAndLength[1]);
			    if (verbose) {
				System.out.println(name + " = " + config);
			    }
			}
		    }
		}
	    }
	} catch (Exception e) {
	    System.err.println("Error parsing metadata: " + e.getMessage());
	    e.printStackTrace();
	}
    }

    protected void write(String name)
	throws IOException
    {
	FileOutputStream out = new FileOutputStream( new File(name));
	out.write(buf.array(), 0, getPutPosition());
	out.close();
    }

    /* These methods are explicitly named so you don't need to cast constants */
    void putByte(int aValue) {
	buf.put((byte)aValue);
    }

    void putByte(int offset, int aValue) {
	buf.put(offset, (byte)aValue);
    }

    void putBytes(byte[] aValue) {
	buf.put(aValue);
    }

    void putBytes(byte[] aValue, int offset, int length) {
	buf.put(aValue, offset, length);
    }

    int putCompressedBytes(byte[] aValue) {
	// Put original size
	putInt(aValue.length);

	// Allocate a buffer to compress into
        byte[] out = new byte[aValue.length + ((aValue.length / 1000) + 1) + 12 + 4];
	Deflater compresser = new Deflater();
	compresser.setInput(aValue);
	compresser.finish();

	int compressedDataLength = compresser.deflate(out);

	buf.put(out, 0, compressedDataLength);

	return compressedDataLength;
    }

    void putShort(byte[] buf, int offset, int aValue) {
	buf[offset+0] = (byte)(aValue & 0x00ff);
	buf[offset+1] = (byte)((aValue >> 8) & 0x00ff);
    }

    void putShort(int aValue) {
	buf.putShort((short)aValue);
    }

    void putShort(int offset, int aValue) {
	buf.putShort(offset, (short)aValue);
    }

    void putChar(byte[] buf, int offset, char aValue) {
	buf[offset+0] = (byte)(aValue & 0x00ff);
	buf[offset+1] = (byte)((aValue >> 8) & 0x00ff);
    }

    void putChar(char aValue) {
	buf.putChar(aValue);
    }

    void putChar(int offset, char aValue) {
	buf.putChar(offset, aValue);
    }

    void putInt(byte[] buf, int offset, int aValue) {
	buf[offset+0] = (byte)(aValue & 0x00ff);
	buf[offset+1] = (byte)((aValue >> 8) & 0x00ff);
	buf[offset+2] = (byte)((aValue >> 16) & 0x00ff);
	buf[offset+3] = (byte)((aValue >> 24) & 0x00ff);
    }

    void putInt(int aValue) {
	buf.putInt(aValue);
    }

    void putInt(int offset, int aValue) {
	buf.putInt(offset, aValue);
    }

    void setPutPosition(int position) {
	buf.position(position);
    }

    int getPutPosition() {
	return buf.position();
    }

    void alignPutPosition(int alignment) {
	int pos = buf.position();
	int mod = pos % alignment;
	if (mod != 0) {
	    buf.position(pos + (alignment - mod));
	}
    }

    void startNewBook() {
	pages = 0;

	BBeBObject head = new BBeBObject(OBJECT_TYPE_BookAtr); // Root Object
	head.addTag(0x75, 0x0002);
	head.addTag(0x76, 0x0000);
	head.addTag(0x77, 0x0001);

	byte tmp78[] = {0,0,0,0,0x16,(byte)0xf5,0,0, 0x1,0x30};
        head.addTag(0x78, tmp78);

	/*
	head.addTagInt(0x78, 0);
	head.addTagInt(0x16, 0x30010000);
	*/
	head.addTag(0x79, 0x0002);
	head.addTag(0x7a, 0x0010);
	head.addTag(0xda, 0x0002);

	byte[] empty = {0x00, 0x00, 0x00, 0x00};
        BBeBObject temp = new BBeBObject(OBJECT_TYPE_TOC, FLAG_TOC_51, empty); // special ?

	// UTF-16LE string containing font name
	byte[] fontname = {22, 0,	// 11 chars
			   'I',0,
			   'W',0,
			   'A',0,
			   0x0E,0x66,
			   '-',0,
			   0x2D,0x4E,
			   0x30,0x7D,
			   'N',0,
			   '-',0,
			   'e',0,
			   'b',0};

	// Create Global font record for the file
	BBeBObject fontRecord = new BBeBObject(OBJECT_TYPE_TextAtr);
	fontRecord.addTag(0x76, 0x0000);
	fontRecord.addTag(0x77, 0x0001);
	fontRecord.addTag(0x79, 0x0001);
	fontRecord.addTag(0x7a, 0x0000);
	fontRecord.addTag(0x11, 0x0064);
	fontRecord.addTag(0x12, 0xfff6);
	fontRecord.addTag(0x13, 0x0000);
	fontRecord.addTag(0x14, 0x0000);
	fontRecord.addTag(0x15, 0x0190);
	fontRecord.addTag(0x16, fontname);
	fontRecord.addTagInt(0x17, 0);
	fontRecord.addTagInt(0x18, 0x00ff);
	fontRecord.addTag(0x19, 0x0019);
	fontRecord.addTag(0x1a, 0x0000);
	fontRecord.addTag(0x1b, 0x008c);
	fontRecord.addTag(0x1c, 0x000a);
	fontRecord.addTag(0x1d, 0x0000);
	fontRecord.addTag(0x1e, 0x0000);
	fontRecord.addTag(0xf1, 0x0002);
	fontRecord.addTagInt(0xf2, 0);
	fontRecord.addTag(0x3c, 0x0001);
	fontRecord.addTag(0x3d, 0x0001);
	fontRecord.addTag(0x3e, 0x0000);
	fontRecord.addTag(0x75, 0x0001);

	font_id = fontRecord.id;

	// Fill this one in later need id now for head
	bookFile = new BBeBObject(OBJECT_TYPE_PageTree);

	head.addTagInt(0x7b, bookFile.id);

	// Margins
	BBeBObject margins = new BBeBObject(OBJECT_TYPE_PageAtr);
	margins.addTag(0x21, 0x0005);	// 5
	margins.addTag(0x22, 0x0035);	// 53
	margins.addTag(0x23, 0x0005);	// 5
	margins.addTag(0x24, 0x002a);	// 42
	margins.addTag(0x2c, 0x002a);	// 42
	margins.addTag(0x25, 0x02a2);	// 674
	margins.addTag(0x26, 0x0204);	// 516
	margins.addTag(0x27, 0x003a);	// 58
	margins.addTag(0x28, 0x0035);	// 53
	margins.addTag(0x35, 0x0034);	// 52
	margins.addTag(0x2b, 0x0000);
	margins.addTag(0x2a, 0x0001);
	margins.addTag(0xda, 0x0002);

	byte[] sixBytes = {0x01, 0x00, 0x00, 0x00, 0x00, 0x00};
	margins.addTag(0x29, sixBytes);

	margins_id = margins.id;

	BBeBObject pageBox = new BBeBObject(OBJECT_TYPE_BlockAtr);
	pageBox.addTag(0x31, 600);
	pageBox.addTag(0x32, 800);
	pageBox.addTag(0x33, 0x0012);
	pageBox.addTagInt(0x34, 0x00ff);
	pageBox.addTag(0x35, 0x0034);
	pageBox.addTag(0x36, 0x0000);
	pageBox.addTagInt(0x37, 0);
	pageBox.addTag(0x2e, 0x0001);
	pageBox.addTag(0x38, 0x0000);
	pageBox.addTag(0x39, 0x0000);
	pageBox.addTag(0x29, sixBytes);

	page_box_id = pageBox.id;
    }

    void writeBookToFile(String aFileName, String thumbnailFileName) {
	if (metaData == null) {
	    System.err.println("No MetaData.");
	    System.exit(1);
	}

	thumbnail = readWholeFile(thumbnailFileName);

	// Update "Page" tag in metaData with value of pages
	/*
	  sprintf(pagetext,"%d",pages);
	  newmeta = update_tag(metadata, "Page", pagetext);
	  free(metadata); metadata = newmeta;
	*/

	byte[] pageNumberData = new byte[(6 * pages) + 4]; 
	byte[] bookPagesList = new byte[(4 * pages) + 2];
	pageNumberData[0] = bookPagesList[0] = (byte)(pages & 0x00ff);
	pageNumberData[1] = bookPagesList[1] = (byte)((pages >> 8) & 0x00ff);

	int pnIndex = 4;
	int bpIndex = 2;
	for (int i = 0; i < objectList.size(); i++) {
	    BBeBObject subFile = (BBeBObject)objectList.get(i);
	    if (subFile.type == OBJECT_TYPE_Page) {
		pageNumberData[pnIndex] = bookPagesList[bpIndex] = (byte)(subFile.id & 0x00ff);
		pageNumberData[pnIndex+1] = bookPagesList[bpIndex+1] = (byte)((subFile.id >> 8) & 0x00ff);

		// Set to say 1 page per page (a kludge for now)
		pageNumberData[pnIndex+4] = 1;
		pageNumberData[pnIndex+5] = 0;

		pnIndex += 6;
		bpIndex += 4;
	    }
	}

	BBeBObject pageNumbers = new BBeBObject( OBJECT_TYPE_ObjectInfo, FLAG_PAGE_NUMBERS, pageNumberData);

	bookFile.addTagInt( 0x02, pageNumbers.id);
	bookFile.addTag( 0x5c, bookPagesList);

	addLRF(metaData, thumbnail);
    }

    void addLRF(byte[] metaData, byte[] thumbnail) {
	// Put magic marker at front
	putChar(0, 'L');
	putChar(2, 'R');
	putChar(4, 'F');

	// Put LRF verion in file
	putShort(LRF_VERSION, 999);	// 0x3E7

	// Put in an encryption key (place holder for now)
	putShort(LRF_PSUEDOKEY, 0x30);
   
	// Set book direction to forwards
	putByte(LRF_DIRECTION, LRF_DIRECTION_FORWARDS); 

	// Put Dimensions
	putShort(LRF_UNK1, 800*2);
	putShort(LRF_UNK2, 600);
	putShort(LRF_UNK3, 800);

	// Various unknown (but critical) values
	putByte(LRF_UNK4, 0x18);

	putShort(LRF_TOC_OFFSET, 0x1536);	// XXX is this legit?

	putByte(LRF_UNK6, 0x14);

	// Set position to just past header
	setPutPosition(0x54);

        // XML file containing metadata
	int compressedLength = putCompressedBytes(metaData);
        putShort(LRF_INFOLEN, compressedLength + 4);

	// Thumbnail (60x80)
	putInt(LRF_THUMBNAIL_LENGTH, thumbnail.length);
	putBytes(thumbnail);

        // 16-byte align before placing first BBeBObject into file.
	alignPutPosition(16);

        // Put BBeBObjects into file
	for (int i = 0; i < objectList.size(); i++) {
	    ((BBeBObject)objectList.get(i)).addToBuffer();
	}

        // 16-byte align before placing BBeBObject tree into file
	alignPutPosition(16);

        // Put BBEBObject info into LRF header
	// And yes, these are technically 64 bits each...
	putInt(LRF_OBJECT_COUNT, objectList.size());
        putInt(LRF_OBJECTTREE_OFFSET, getPutPosition());

        // Put BBeBObject tree into file
	for (int i = 0; i < objectList.size(); i++) {
	    BBeBObject object = (BBeBObject)objectList.get(i);
            putInt(object.id);
            putInt(object.location);
            putInt(object.size);
            putInt(0);
        }
    }

    /* Convert to UTF16, or do nothing if its already the case */
    byte[] convertToUTF16LE(byte[] in) {
	if ((in.length < 2) ||
	    ((in[0] == 0xff) && (in[1] == 0xfe))) {
	    return in;
	}

	byte[] out = new byte[(in.length * 2) + 2];
	out[0] = (byte)0x00ff;
	out[1] = (byte)0x00fe;

	for (int i = 0; i < in.length; i++) {
	    out[(i + 1) * 2] = in[i];
	}

	return out;
    }

    byte[] readWholeFile(String aFileName) {
	File file = new File(aFileName);

	int fileSize = (int)file.length();
	byte[] buf = new byte[fileSize];

	try {
	    InputStream in = new FileInputStream(file);

	    if (in.read(buf) != buf.length) {
		System.err.println("Error reading file: " + file);
		System.exit(1);
	    }
	    in.close();
	    in = null;

	    return buf;
	} catch (Exception e) {
	    System.err.println("Error loading LRF file: " + file + " message: " + e.getMessage());
	    e.printStackTrace();
	    System.exit(1);
	}
	return null;
    }

    
    byte[] textBuffer = null;
    int textOffset = 0;
    byte[] outBuf = new byte[65536];
    int outBufOffset = 0;

    void outAppend(int c) {
	if ((outBufOffset + 2) > outBuf.length) {
	    byte[] newBuf = new byte[outBuf.length + 8192];
	    System.arraycopy( outBuf, 0, newBuf, 0, outBuf.length);
	    outBuf = newBuf;
	    System.out.println("Increased outBuf size");
	}
	outBuf[outBufOffset++] = (byte)(c & 0x00ff);
	outBuf[outBufOffset++] = (byte)((c >> 8) & 0x00ff);
    }

    BBeBObject getNextPage() {
	if (textOffset >= textBuffer.length) {
	    // All done
	    return null;
	}

	// Empty buffer
	outBufOffset = 0;

	// Place start of Page marker
	outAppend(0xf5a1);
	outAppend(0x0000);
	outAppend(0x0000);

	// Convert ASCII to UTF-16LE and along the way we strip \r and null
	// and convert \n to 0xf5d2
	int lineCount = 0;
	int nls = 0;
	while ((textOffset < textBuffer.length) && (lineCount < 200) && !((nls == 6) && (lineCount > 30))) {
	    int b = textBuffer[textOffset++] & 0x00ff;
	    if ((b == 13) || (b == 0)) {
		// Skip \r and null values
		continue;
	    } else if (b == 10) {
		// Place EOL marker
		outAppend(0xf5d2);
		lineCount++;
		nls++;
	    } else if (b == '<') {
		if (((textOffset + 1) < textBuffer.length) &&
		    ((textBuffer[textOffset+1] & 0x00ff) == '>')) {
		    if (((textBuffer[textOffset] & 0x00ff) == 'i') ||
			((textBuffer[textOffset] & 0x00ff) == 'I')) {
			outAppend(0xf581);
			textOffset += 2;
		    } else if (((textBuffer[textOffset] & 0x00ff) == 'b') ||
			       ((textBuffer[textOffset] & 0x00ff) == 'B')) {
			outAppend('B');
		outAppend(0xf5d2);
			outAppend(0xf593);
			textOffset += 2;
		    }
		} else if (((textOffset + 2) < textBuffer.length) &&
			   ((textBuffer[textOffset] & 0x00ff) == '/') &&
			   ((textBuffer[textOffset+2] & 0x00ff) == '>')) {
		    if (((textBuffer[textOffset+1] & 0x00ff) == 'i') ||
			((textBuffer[textOffset+1] & 0x00ff) == 'I')) {
			outAppend(0xf582);
			textOffset += 3;
		    } else if (((textBuffer[textOffset+1] & 0x00ff) == 'b') ||
			       ((textBuffer[textOffset+1] & 0x00ff) == 'B')) {
			outAppend(0xf584);
			textOffset += 3;
		    }
		} else {
		    outAppend(b);
		}
	    } else {
		outAppend(b);
		nls = 0;
	    }
	}

	// End of Page marker
	outAppend(0xf5a2);

	byte[] out = outBuf;
	int flags = 0;
	if (outBufOffset > 64) {
	    // Allocate a buffer to compress into
	    out = new byte[outBuf.length];

	    // Stash uncompressed size
	    putInt( out, 0, outBufOffset);

	    // Deflate text
	    Deflater compresser = new Deflater();
	    compresser.setInput(outBuf, 0, outBufOffset);
	    compresser.finish();
	    outBufOffset = compresser.deflate(out, 4, out.length - 4) + 4;
	    flags = FLAG_COMPRESSED;
	}

	return new BBeBObject(OBJECT_TYPE_Text, flags, out, outBufOffset);
    }

    void addTextToBook(String aFileName) {
	// Load text for book
	textBuffer = readWholeFile(aFileName);

	while (true) {
	    BBeBObject text = getNextPage();
	    if (text == null) {
		// All done
		break;
	    }
	    addTextPage(text);
	}
    }

    protected void addTextPage(BBeBObject text) {
	// font size?? para.addTag( 0x11, 80);
	text.addTagInt( 0x03, font_id);

	/* The text goes into a bounding box */
	byte[] boxData = new byte[6];
	putShort( boxData, 0, 0x00f503);
	putInt( boxData, 2, text.id);

	BBeBObject box = new BBeBObject(OBJECT_TYPE_Block, 0, boxData);
	box.addTagInt(0x03, page_box_id);

	// add_tag_to_subfile(box,new_tag(0x31,sizeof(WORD),0,600));
	// add_tag_to_subfile(box,new_tag(0x32,sizeof(WORD),0,800));
	// add_tag_to_subfile(box,new_tag(0x33,sizeof(WORD),0,0x22));

	byte[] pageFiles = new byte[14];
	putShort(pageFiles, 0, 3);
	putInt(pageFiles, 2, font_id);
	putInt(pageFiles, 6, text.id);
	putInt(pageFiles, 10, box.id);

	byte[] pageData = new byte[6];
	putShort( pageData, 0, 0x00f503);
	putInt( pageData, 2, box.id);

	BBeBObject page = new BBeBObject(OBJECT_TYPE_Page, 0, pageData);
	page.addTagInt(0x7c, bookFile.id);
	page.addTag(0x0b, pageFiles);
	page.addTagInt(0x03, margins_id);


	// We make the layout big so paragraphs can break over display pages
	byte[] layoutData = new byte[24];
	putInt(layoutData, 0, 1);
	putInt(layoutData, 4, box.id);
	BBeBObject physicalPages = new BBeBObject(OBJECT_TYPE_ObjectInfo, FLAG_PAGE_LAYOUT, layoutData);
	page.addTagInt(0x02, physicalPages.id);

	pages++;
    }

    private LinkedList pageTexts = new LinkedList();
    private LinkedList pageBoxes = new LinkedList();

    protected void startPage() {
	pageTexts.clear();
	pageBoxes.clear();
    }

    protected void addTextToPage(String chars, int x, int y, int w, int h) {
	outBufOffset = 0;
	for (int i = 0; i < chars.length(); i++) {
	    outAppend(chars.charAt(i));
	}

	BBeBObject textObject = new BBeBObject(OBJECT_TYPE_Text, 0, outBuf, outBufOffset);
	// font size?? para.addTag( 0x11, 80);
	textObject.addTagInt( 0x03, font_id);

	/* The text goes into a bounding box */
	byte[] boxData = new byte[6];
	putShort( boxData, 0, 0x00f503);
	putInt( boxData, 2, textObject.id);

	BBeBObject box = new BBeBObject(OBJECT_TYPE_Block, 0, boxData);
	//	box.addTagInt(0x03, page_box_id);
	box.addTag(0x31,w);
	box.addTag(0x32,h);
	box.addTag(0x33,0x34);

	// add_tag_to_subfile(box,new_tag(0x31,sizeof(WORD),0,600));
	// add_tag_to_subfile(box,new_tag(0x32,sizeof(WORD),0,800));
	// add_tag_to_subfile(box,new_tag(0x33,sizeof(WORD),0,0x22));

	pageTexts.add(textObject);
	pageBoxes.add(box);
    }

    protected void addImageToPage() {
    }

    protected void endPage() {
	byte[] pageFiles = new byte[2 + 4 + (pageTexts.size() * 8)];
	putShort(pageFiles, 0, 3);
	putInt(pageFiles, 2, font_id);

	byte[] pageData = new byte[ 2 + (pageTexts.size() * 4)];
	putShort(pageData, 0, 0x00f503);

	byte[] layoutData = new byte[4 + (pageTexts.size() * 4)];
	putInt(layoutData, 0, pageTexts.size());

	int offsetPF = 6;
	int offsetPD = 2;
	int offsetLD = 4;
	Iterator texts = pageTexts.iterator();
	Iterator boxes = pageBoxes.iterator();
	while (texts.hasNext()) {
	    putInt(pageFiles, offsetPF, ((BBeBObject)texts.next()).id);
	    offsetPF += 4;
	    int boxid = ((BBeBObject)boxes.next()).id;
	    putInt(pageFiles, offsetPF, boxid);
	    offsetPF += 4;
	    putInt(pageData, offsetPD, boxid);
	    offsetPD += 4;
	    putInt(layoutData, offsetLD, boxid);
	    offsetLD += 4;
	}

	// We make the layout big so paragraphs can break over display pages
	BBeBObject physicalPages = new BBeBObject(OBJECT_TYPE_ObjectInfo, FLAG_PAGE_LAYOUT, layoutData);
	BBeBObject page = new BBeBObject(OBJECT_TYPE_Page, 0, pageData);
	page.addTagInt(0x02, physicalPages.id);
	page.addTag(0x0b, pageFiles);
	page.addTagInt(0x03, margins_id);

	//	page.addTagInt(0x07, id);
	page.addTag(0x22, 0x34);
	page.addTag(0x24, 0x37);
	page.addTag(0x25, 800);
	page.addTag(0x26, 600);
	page.addTag(0x35, 0x34);

	page.addTagInt(0x7c, bookFile.id);

	/*
  <Object ID="0x00000034" offset = "16830" size = "324" name = "OBJECT_TYPE_Page">
    <Tag ID= "0xf502" name= "*ObjectInfoLink" length= "4">
0x00003a22
    </Tag>
    <Tag ID= "0xf50b" name= "*ContainedObjectsList" length= "0">
      <0b_5c count = "33">
        0x00000323
        0x00001b74
        0x0000032a
        0x00001b78
        0x00001b79
        0x00000324
        0x00001b77
        0x00001b73
        0x00000321
        0x00001b72
        0x00000326
        0x00003a1e
        0x00003a1d
        0x00000329
        0x000030f3
        0x0000031f
        0x000030f1
        0x00001b7a
        0x00001b7c
        0x000030f2
        0x00000322
        0x00001b71
        0x000039d4
        0x00001b75
        0x00000327
        0x00001b76
        0x00000320
        0x00000328
        0x00000325
        0x0000032b
        0x00001b7b
        0x000030f0
        0x00001b7d
      </0b_5c>
    </Tag>
    <Tag ID= "0xf503" name= "*Link" length= "4">
0x00003a1c
    </Tag>
    <Tag ID= "0xf507" name= "Unknown_07" length= "4">
0x000039d4
    </Tag>
    <Tag ID= "0xf522" name= "Unknown_22" length= "2">
0x0034
    </Tag>
    <Tag ID= "0xf524" name= "Unknown_24" length= "2">
0x0037
    </Tag>
    <Tag ID= "0xf525" name= "*PageHeight" length= "2">
0x0299
    </Tag>
    <Tag ID= "0xf526" name= "*PageWidth" length= "2">
0x01e0
    </Tag>
    <Tag ID= "0xf535" name= "Unknown_35" length= "2">
0x0034
    </Tag>
    <Tag ID= "0xf57c" name= "*ParentPageTree" length= "4">
0x0000004d
    </Tag>
  <Object ID="0x000039d4" offset = "2527901" size = "58" name = "OBJECT_TYPE_Header">
    <Tag ID= "0xf535" name= "Unknown_35" length= "2">
0x0034
    </Tag>
    <Tag ID= "0xf534" name= "Unknown_34" length= "4">
0x000000ff
    </Tag>
    <Tag ID= "0xf536" name= "Unknown_36" length= "2">
0x0000
    </Tag>
    <Tag ID= "0xf537" name= "Unknown_37" length= "4">
0x00000000
    </Tag>
    <Tag ID= "0xf52e" name= "Unknown_2E" length= "2">
0x0001
    </Tag>
    <Stream flags="0x0000"/>
    <Stream length="10"/>
49f5000000001f030000
  </Object>
*/
	pages++;
    }

    void addPDFToBook(String aPDFFileName) {
	try {
	    PdfDecoder decode_pdf = new PdfDecoder(false);
	    decode_pdf.setDefaultDisplayFont("SansSerif");
	    decode_pdf.enableHighQualityMode();

	    String[] nameInPDF={"TimesNewRoman"};//,"helvetica","arial"}; //$NON-NLS-1$
	    decode_pdf.setSubstitutedFontAliases("Times New Roman", nameInPDF); //$NON-NLS-1$
	    decode_pdf.setThumbnaiImageTransparency(false);
	    decode_pdf.openPdfFile(aPDFFileName);
	    
	    int pdfPages = decode_pdf.getPageCount();

	    //page data so we can choose portrait or landscape
	    PdfPageData pdfPageData = decode_pdf.getPdfPageData();

	    //	    System.out.println("Width: " + pageData.getCropBoxWidth(1) + " height: " +pageData.getCropBoxHeight(1));

	    for (int i = 0; i < pdfPages; i++) {
		int pageNo = i + 1;
		int width = pdfPageData.getCropBoxWidth(pageNo);
		int height = pdfPageData.getCropBoxHeight(pageNo);

		float scale = 1.0f;
		if (width > 600) {
		    scale = 600.0f / width;
		}
		if (height > 800) {
		    scale = 800.0f / height;
		}

		decode_pdf.setExtractionMode(PdfDecoder.FINALIMAGES +
					     PdfDecoder.RAWCOMMANDS +
					     PdfDecoder.RAWIMAGES +
					     PdfDecoder.RENDERIMAGES +
					     PdfDecoder.RENDERTEXT +
					     PdfDecoder.TEXT +
					     PdfDecoder.TEXTCOLOR,
					     72,
					     scale);

		addBufferedImage(decode_pdf.getPageAsImage(pageNo));

		System.out.print("Processed PDF page " + pages + " of " + pdfPages + "\r");
	    }
	} catch (Exception e) {
	    e.printStackTrace();
	}
    }

    protected void addBufferedImage(BufferedImage bufferedImage) {
	int x = bufferedImage.getWidth();
	int y = bufferedImage.getHeight();

	// Convert to JPEG
	byte[] imageBytes = null;
	try {
	    ByteArrayOutputStream baos = new ByteArrayOutputStream();
	    if (false) {
		ImageIO.write(bufferedImage, "jpg", baos);
	    } else {
		ImageOutputStream imageOut = ImageIO.createImageOutputStream(baos);
		ImageWriter imageWriter = (ImageWriter)ImageIO.getImageWritersBySuffix("jpeg").next();
		imageWriter.setOutput(imageOut);
		ImageWriteParam param = imageWriter.getDefaultWriteParam();
		param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
		param.setCompressionQuality(0.40f);
		imageWriter.write(null, new IIOImage(bufferedImage, null, null), param);
	    }
	    imageBytes = baos.toByteArray();
	} catch (Exception e) {
	    System.out.println("Error rendering image");
	    return;
	}

	BBeBObject margins = new BBeBObject(OBJECT_TYPE_PageAtr);
	margins.addTag(0x21, 0);
	margins.addTag(0x22, 0);
	margins.addTag(0x23,(800 - y)/2);
	margins.addTag(0x24,(600 - x)/2);
	margins.addTag(0x2C,(600 - x)/2);
	margins.addTag(0x25,y);
	margins.addTag(0x26,x);
	margins.addTag(0x27,0);
	margins.addTag(0x28,0);
	margins.addTag(0x35,0x41);
	margins.addTag(0x2b,2);
	margins.addTag(0x2a,1);
	margins.addTag(0xda,1);
	byte[] sixbytes = new byte[6];
	margins.addTag(0x29,sixbytes);

	BBeBObject image = new BBeBObject(OBJECT_TYPE_ImageStream, 0x11, imageBytes); 

	/*
	// Allocate a buffer to compress into
	byte[] out = new byte[imageBytes.length];


	putInt( out, 0, imageBytes.length);

	Deflater compresser = new Deflater();
	compresser.setInput(imageBytes, 0, imageBytes.length);
	compresser.finish();
	int outSize = compresser.deflate(out, 4, out.length - 4) + 4;

	BBeBObject image = new BBeBObject(OBJECT_TYPE_ImageStream, 0x113, out, outSize);
	*/
	BBeBObject imageHolder = new BBeBObject(OBJECT_TYPE_Image);
	byte[] imageRect = new byte[8];
	putShort(imageRect, 0, 0);
	putShort(imageRect, 2, 0);
	putShort(imageRect, 4, x);
	putShort(imageRect, 6, y);
	imageHolder.addTag(0x4a, imageRect);
	byte[] imageSize = new byte[4];
	putShort(imageSize, 0, x);
	putShort(imageSize, 2, y);
	imageHolder.addTag(0x4b, imageSize);
	imageHolder.addTagInt(0x4c, image.id);

	BBeBObject blockAttr = new BBeBObject(OBJECT_TYPE_BlockAtr);
	blockAttr.addTag(0x31, 0x204);
	blockAttr.addTag(0x32, 0x64);
	blockAttr.addTag(0x33,0x12);
	blockAttr.addTagInt(0x34,0x00FF);
	blockAttr.addTag(0x35,0x34);
	blockAttr.addTag(0x36,0);
	blockAttr.addTagInt(0x37,0);
	blockAttr.addTag(0x2e,1);
	blockAttr.addTag(0x38,0);
	blockAttr.addTag(0x39,0);
	blockAttr.addTag(0x3a,0);

	byte[] sixbytes2 = new byte[6];
	sixbytes2[0] = 1;
	blockAttr.addTag(0x29,sixbytes2);

	byte[] boxData = new byte[6];
	putShort(boxData, 0, 0xf503);
	putInt(boxData, 2, imageHolder.id);
	BBeBObject box = new BBeBObject(OBJECT_TYPE_Block, 0, boxData);
	box.addTagInt(0x03, blockAttr.id);
	box.addTag(0x31, x);
	box.addTag(0x32, y);
	box.addTag(0x33, 0x22);

	byte[] pageData = new byte[6];
	putShort(pageData, 0, 0xf503);
	putInt(pageData, 2, box.id);

	byte[] pageFiles = new byte[18];
	putShort(pageFiles, 0, 4);
	putInt(pageFiles,  2, box.id);
	putInt(pageFiles,  6, imageHolder.id);
	putInt(pageFiles, 10, blockAttr.id);
	putInt(pageFiles, 14, image.id);

	BBeBObject page = new BBeBObject(OBJECT_TYPE_Page, 0, pageData);
	page.addTagInt(0x7c, bookFile.id);
	page.addTag(0x0b, pageFiles);
	page.addTagInt(0x03, margins.id);

	pages++;

	byte[] layoutData = new byte[24];
	putInt(layoutData, 0, 1);
	putInt(layoutData, 4, box.id);
	BBeBObject physicalPages = new BBeBObject(OBJECT_TYPE_ObjectInfo, 0x0082, layoutData);
	page.addTagInt(0x02, physicalPages.id);
    }

    private void parseHTML(String aHTMLFileName) {
	try {
	    HtmlParser htmlParser = new HtmlParser(this, aHTMLFileName);
	    htmlParser.parsePages();
	} catch (IOException e) {
	    e.printStackTrace();
	}
    }

    private void parsePDF(String aPDFFileName) {
	try {
	    PageParser pageParser = new PageParser(this, aPDFFileName);
	    pageParser.parsePages();
	} catch (IOException e) {
	    e.printStackTrace();
	}
    }

    protected void addOutBufAsTextPage() {
	// If we have more than start page / end page
	if (outBufOffset != 8) {
	    byte[] out = outBuf;
	    int flags = 0;
	    if (outBufOffset > 64) {
		// Allocate a buffer to compress into
		out = new byte[outBuf.length];

		// Stash uncompressed size
		putInt( out, 0, outBufOffset);

		// Deflate text
		Deflater compresser = new Deflater();
		compresser.setInput(outBuf, 0, outBufOffset);
		compresser.finish();
		outBufOffset = compresser.deflate(out, 4, out.length - 4) + 4;
		flags = FLAG_COMPRESSED;
	    }

	    addTextPage(new BBeBObject(OBJECT_TYPE_Text, flags, out, outBufOffset));
	}

	// Ready for next page
	outBufOffset = 0;
    }

    byte[] createMetaData() {
	if (bookTitle.length() == 0) {
	    System.err.println("WARNING: Book Title not supplied, setting to 'UNKNOWN'.");
	    bookTitle.append("UNKNOWN");
	}
	if (bookID.length() == 0) {
	    bookID.append("LBBB").append(System.currentTimeMillis());
	}
	if (bookID.length() > 16) {
	    System.err.println("BookID is longer than 16 characters. It will be trimmed.");
	    bookID.setLength(16);
	}
	if (bookLabel.length() == 0) {
	    System.err.println("WARNING: Book Label not supplied, setting to 'UNKNOWN'.");
	    bookLabel.append("UNKNOWN");
	}
	if (docCreationDate.length() == 0) {
	    docCreationDate.append(new java.sql.Date(System.currentTimeMillis()).toString());
	}
	if (docCreationDate.length() > 15) {
	    System.err.println("Document creation data is longer than 15 characters. It will be trimmed.");
	    docCreationDate.setLength(15);
	}

	StringBuffer s = new StringBuffer(4096);
	s.append((char)0xFEFF);
	s.append("<?xml version=\"1.0\" encoding=\"UTF-16\" ?>\n");
	s.append("<Info version=\"1.0\" >\n");
	s.append("  <BookInfo>\n");
	s.append("    <Title>").append(bookTitle).append("</Title>\n");
	s.append("    <Author>").append(bookAuthor).append("</Author>\n");
	s.append("    <BookID>").append(bookID).append("</BookID>\n");
	s.append("    <Publisher>").append(bookPublisher).append("</Publisher>\n");
	s.append("    <Label>").append(bookLabel).append("</Label>\n");
	s.append("    <Category>").append(bookCategory).append("</Category>\n");
	s.append("    <Classification>").append(bookClassification).append("</Classification>\n");
	if (bookFreeText.length() != 0) {
	    s.append("    <FreeText>").append(bookFreeText).append("</FreeText>\n");
	}
	s.append("  </BookInfo>\n");
	s.append("  <DocInfo>\n");
	s.append("    <Language>").append(docLanguage).append("</Language>\n");
	s.append("    <Creator>").append(docCreator).append("</Creator>\n");
	s.append("    <CreationDate>").append(docCreationDate).append("</CreationDate>\n");
	s.append("    <Producer>Legally Blind Book Binder 2.0.1</Producer>\n");
	s.append("    <Page>").append(pages).append("</Page>\n");
	s.append("  </DocInfo>\n");
	s.append("</Info>\n");

	//	System.out.println(s);

	try {
	    return s.toString().getBytes("UTF-16LE");
	} catch (java.io.UnsupportedEncodingException e) {
	    System.err.println("Unlnown encoding");
	    System.exit(1);
	    return null;
	}
    }

    class BBeBObject {
	short type;     // Object type
	short id;      	// Unique object ID
	short dataFlags;
	byte[] data;
	int dataLength;
	ArrayList tags;
	int location; 	// Where Object was placed in file
	int size;	// Size of Object in file

	BBeBObject(int aType) {
	    id = globalBBeBObject++;
	    type = (short)aType;

	    dataFlags = 0;
	    location = 0;
	    size = 0;
	    data = null;
	    dataLength = 0;
	    tags = new ArrayList(64);

	    // Record certain id's in LRF Header
            if (type == OBJECT_TYPE_BookAtr) {
                putShort(LRF_ROOT_ID, id);
	    } else if (type == OBJECT_TYPE_TOC) {
                putShort(LRF_TOC_ID, id);
	    }

	    objectList.add(this);
	}

        BBeBObject(int aType, int aDataFlags, byte[] aData) {
	    this(aType);
	    dataFlags = (short)aDataFlags;
	    data = aData;
	    dataLength = data.length;
	}

        BBeBObject(int aType, int aDataFlags, byte[] aData, int aLength) {
	    this(aType);
	    dataFlags = (short)aDataFlags;
	    data = aData;
	    dataLength = aLength;
	}

	void addTag(int aID, int aValue) {
	    byte[] value = new byte[2+2];

	    value[0] = (byte)(aID & 0x00ff);
	    value[1] = (byte)0xf5;
	    putShort(value, 2, aValue);

	    tags.add(value);
	}

	void addTagInt(int aID, int aValue) {
	    byte[] value = new byte[2+4];

	    value[0] = (byte)(aID & 0x00ff);
	    value[1] = (byte)0xf5;
	    putInt(value, 2, aValue);

	    tags.add(value);
	}

	void addTag(int aID, byte[] aValue) {
	    byte[] value = new byte[2+aValue.length];

	    value[0] = (byte)(aID & 0x00ff);
	    value[1] = (byte)0xf5;
	    System.arraycopy( aValue, 0, value, 2, aValue.length);

	    tags.add(value);
	}

	void addToBuffer() {
	    // Record position of sub file record in file
	    location = getPutPosition();

	    // Put in Object Type TAG
	    putShort(0xf500);
	    putInt(id);
	    putShort(type);

	    // Put in tags
	    for (int i = 0; i < tags.size(); i++) {
		putBytes((byte[])tags.get(i));
	    }

	    // Write data block if we have one
	    if (data != null) {
		putShort(0xf554);	// dataflags
		putShort(dataFlags);
		putShort(0xf504);	// length
		putInt(dataLength);
		putShort(0xf505);	// data
		putBytes(data, 0, dataLength);
		putShort(0xf506);	// end of data
	    }

	    // Write end of sub file marker
	    putShort(0xf501);	// end of Sub File

	    // Record size of sub file data written
	    size = getPutPosition() - location;
	}
    }
}
