import json
import pickle
import platform
import shutil
import subprocess
import sys
import zipfile
from pathlib import Path

from calibre.constants import is64bit, ismacos, iswindows
from calibre.utils.config import config_dir
from calibre_plugins.grauthornotes.config import prefs

PLUGIN_PATH = Path(config_dir).joinpath('plugins/GR Author Notes.zip')
PY_VERSION = '.'.join(platform.python_version_tuple()[:2])
LIBS_PATH = Path(config_dir).joinpath(f"plugins/gr_author_notes-libs-py{PY_VERSION}")

def load_json_or_pickle(filepath, is_json):
    with zipfile.ZipFile(PLUGIN_PATH) as zf:
        with zf.open(filepath) as f:
            if is_json:
                return json.load(f)
            else:
                return pickle.load(f)

def install_libs(notif):
    if (reinstall := False if LIBS_PATH.exists() else True):
        for old_path in LIBS_PATH.parent.glob('gr_author_notes-libs-py*'):
            old_path.rename(LIBS_PATH)

    for pkg, value in load_json_or_pickle('data/modules.json', True).items():
        pip_install(pkg, value['version'], value['compiled'],
                    reinstall=reinstall, notif=notif)
    
    if LIBS_PATH not in sys.path:
        sys.path.insert(0, str(LIBS_PATH))

def pip_install(pkg, pkg_version=None, compiled=False,
                url=None, reinstall=False, notif=None):
    if pkg_version:
        folder = LIBS_PATH.joinpath(
            f"{pkg.replace('-', '_')}-{pkg_version}.dist-info")
    else:
        folder = LIBS_PATH.joinpath(pkg.replace('-', '_').lower())

    if not folder.exists() or (reinstall and compiled):
        if notif:
            notif.put((0, f'Installing {pkg}'))
        args = pip_args(pkg, pkg_version, compiled, url)
        for arg in args:
            arg.replace("\'", "\"")
        print(args)
        if iswindows:
            subprocess.run(args, check=True, capture_output=True,
                           creationflags=subprocess.CREATE_NO_WINDOW)
        else:
            subprocess.run(args, check=True, capture_output=True)


def pip_args(pkg, pkg_version, compiled, url):
    py = 'python3'
    if iswindows:
        py = 'py' if shutil.which('py') else 'python'
    elif ismacos:
        # stupid macOS loses PATH when calibre is not launched in terminal
        if platform.machine() == 'arm64':
            py = '/opt/homebrew/bin/python3'
        else:
            py = '/usr/local/bin/python3'
        if not Path(py).exists():
            py = '/usr/bin/python3'  # command line tools
    libs_str = str(LIBS_PATH)
    args = [py, '-m', 'pip', 'install', '-U', '-t', libs_str, '--no-deps']
    if compiled:
        args.extend(['--python-version', PY_VERSION])
        if iswindows:
            args.append('--platform')
            if is64bit:
                args.append('win_amd64')
            else:
                args.append('win32')
    if url:
        args.append(url)
    elif pkg_version:
        args.append(f'{pkg}=={pkg_version}')
    else:
        args.append(pkg)
    return args