#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

# target script

#----------------------------------------------------------------------------
# pluginMenu.py   testModule.py     hTranslate.py
#----------------------------------------------------------------------------
import sys
import re
import os
from pathlib import Path
from subprocess import Popen
from PyQt5.QtWidgets import (QApplication, QMainWindow, QPushButton,  QHBoxLayout, 
                            QVBoxLayout,  QFrame,  QTabWidget,  QWidget, QStatusBar, 
                            QFileDialog, QSplitter, QProgressBar, QTextEdit, QMessageBox)
from PyQt5.QtGui import QIcon,  QFont,  QColor,  QCursor,  QImage,  QPixmap
from PyQt5.QtCore import Qt  
from PyQt5.Qsci import *

import hFileClassE as hFc
import hConstant as hC
import hCsvUtil as hCsU
import hTranslateOneStep as hTr
import hTabConfig as hTc
import hSigilBkUtil as hBk
import hThumbsPreview as hTp
import hXhtmlTemplate as hXt
import hMargin as hM
import hFontClass as hFo
from hQTableWidgetE import *
import hMiscClass as hMc 


class MenuWin(QMainWindow):
    def __init__(self, bk,  oTabConf_DOCX, oTabConf_CSV,  oTabDocx_DOCX, oTabDocx_CSV):
        super().__init__()
        
        self.bk= bk
        self.bkUtil= hBk.BkUtil()
        self.bkUtil.errorMsgSignal.connect(self.processErrBkUtil)
        
        self.saveXhtml= ''
        self.pageLoaded= False      # if True value from Sigil already loaded in misc page table

        self.oTabConf_DOCX= oTabConf_DOCX
        self.oTabConf_DOCX.setFileProperties('')
        self.oTabConf_CSV= oTabConf_CSV
        self.oTabConf_CSV.setFileProperties('')
        self.oTabDocx_DOCX= oTabDocx_DOCX
        self.oTabDocx_DOCX.setFileProperties('')
        self.oTabDocx_CSV= oTabDocx_CSV
        self.oTabDocx_CSV.setFileProperties('')

        self.xhtmlCodeSave= ''     
        self.txtCodeSave= ''
        self.htmCodeSave= '' 

        if hC.DEBUG:
            self.iconPath= 'images/'
        else:
            self.iconPath= os.path.join(self.bk._w.plugin_dir, self.bk._w.plugin_name, 'images/')    
#            print(self.iconPath)
            
        self.tabConfParamLst= []
        self.tabDocParamLst= []
        self.initializeUI()
        self.show()

        self.setStyleSheet("""QToolTip { 
                                   background-color: #ffffcc; 
                                   color: black; 
                                   border: black solid 1px
                                   }""")
        
    def initializeUI(self):
        self.setGeometry(100, 100, 900, 850)
        self.setWindowTitle('docxTranslat')
        self.setWindowIcon(QIcon(self.iconPath+ 'docxTranslatIcon15.png'))
#        self.createMenu()
        self.createStatusBar()
        #--- btn
        frameBtn= QFrame()
        frameBtn.setFrameStyle(QFrame.Box | QFrame.Sunken)     
        startWordBtn= QPushButton('WinWord open')
        startWordBtn.setIcon(QIcon(self.iconPath+ 'wordOpen.png'))
        startWordBtn.clicked.connect(lambda: self.docxMngr('open'))
        startWordBtn.setToolTip(hC.startWordBtn)
        clearLogBtn= QPushButton('clear log')
        clearLogBtn.clicked.connect(self.initLogTextEdit)
        clearLogBtn.setToolTip(hC.initLogTextEditBtn)
        clearLogBtn.setIcon(QIcon(self.iconPath+ 'clearLog.png'))      
        endBtn= QPushButton('exit')
        endBtn.setIcon(QIcon(self.iconPath+ 'cross.png'))        
        endBtn.clicked.connect(self.close)
        #
        btnLayout= QHBoxLayout()
        btnLayout.addWidget(startWordBtn)
        btnLayout.addStretch()
        btnLayout.addWidget(clearLogBtn)
        btnLayout.addWidget(endBtn)
        frameBtn.setLayout(btnLayout)        
        #---    create tab bar 
        frameTabBar= QFrame()
        frameTabBar.setFrameStyle(QFrame.Box | QFrame.Sunken)
        self.tabBar = QTabWidget()   
        self.tabBar.currentChanged.connect(self.tabBarChanged)
        #
#        self.tabBar.setCurrentIndex(1)
        tabBarLayout= QHBoxLayout()
        tabBarLayout.addWidget(self.tabBar) 
        frameTabBar.setLayout(tabBarLayout)        
        # csv progress bar
        self.csvProgBar= QProgressBar()
        self.csvProgBar.setAlignment(Qt.AlignHCenter)
        self.csvProgBar.hide()
        tabBarLayout= QHBoxLayout()
        tabBarLayout.addWidget(self.csvProgBar)        
        #---    create QTextEdit to use as a log area for csv file creation
        frameTextEdit= QFrame() 
        frameTextEdit.setFrameStyle(QFrame.Box | QFrame.Sunken)
        self.logTextEdit= QTextEdit()
        self.setLogtextEditColor(hC.blackColor,  hC.whiteColor)
        self.logTextEdit.setFont(QFont('COURIER NEW', 11))
        self.initLogTextEdit()
        logTextEditLayout= QVBoxLayout()
        logTextEditLayout.setContentsMargins(2, 2, 2, 2)
        logTextEditLayout.addWidget(QLabel('log area'))
        logTextEditLayout.addWidget(self.logTextEdit)  
        frameTextEdit.setLayout(logTextEditLayout)  
        #--- font class to use with combos in hTabConfig.py tabName= "Docx to translate"
        self.currentFontFamily= hC.defaultFontFamily
        self.currentFontSize= hC.defaultFontSize        
        self.fontObj= hFo.MyFonts()
        self.fontObj.getFamilyLst()   
        #--- add tabs 
        self.tabBarChangedCreation= True
        hTc.addTabConfig(self)
        hTc.addTabDocument(self)
        hTc.addTabTranslationOneStep(self)
        hTc.addTabImageTools(self)
        hTc.addTabManifest(self)        
        hTc.addTabSigilSpine(self)
        hTc.addMiscPages(self)
        hTc.addBkTestTab(self)
        #
        self.tabBarChangedCreation= False
#        self.tabBarChanged(3)  
        self.tabBarChanged(0)
        self.tabBar.setCurrentIndex(0)
        # splitter to redim tabBar and textEdit
        mainSplitter= QSplitter(Qt.Vertical)      
        mainSplitter.addWidget(frameTabBar)
        mainSplitter.addWidget(frameTextEdit)
        # set height of splitter's components
        mainSplitter.setStretchFactor(0, 3)     # 0 --> frameTabBar
        mainSplitter.setStretchFactor(1, 2)     # 1 --> frameTextEdit
        #
        self.initLogTextEdit()      # reset bk errors in log area
        #
        mainLayout= QVBoxLayout()
        mainLayout.addWidget(frameBtn)
        mainLayout.addWidget(mainSplitter) 
        mainLayout.addLayout(tabBarLayout)
        mainFrame= QWidget()
        mainFrame.setLayout(mainLayout)
        #
        self.setCentralWidget(mainFrame)        

    # --------------------------------------------------------------
    #   MENU
    # --------------------------------------------------------------    
    def createMenu(self):
        # create menu bar
        menuBar= self.menuBar()

        # create menu header (header => .addMenu)
        wordMenu= menuBar.addMenu('Word file')
        
        # create menu item (action => .addAction
        chooseDocxAct= wordMenu.addAction('Choose')
        chooseDocxAct.setStatusTip('Choose .docx file to load in Sigil')
        chooseDocxAct.triggered.connect(lambda: self.docxMngr('choose'))
        #
        openDocxAct= wordMenu.addAction('Open')
        openDocxAct.setStatusTip('Open .docx file to load in Sigil')
        openDocxAct.triggered.connect(lambda: self.docxMngr('open')) 
        #
        wordMenu.addSeparator()
        endAct= wordMenu.addAction('End')
        endAct.triggered.connect(self.close) 
    
    # status bar
    def createStatusBar(self):
        self.setStatusBar(QStatusBar(self))

    # --------------------------------------------------------------
    #   MENU functions
    # -------------------------------------------------------------- 
    #   menu actions
    def docxMngr(self,  oper):
        dialog = QFileDialog()
        docxFileName, _= dialog.getOpenFileName(self,'Choose File','', hC.filterDocx)
        if docxFileName:
            if oper== 'choose':
                self.oTabConf_DOCX.setFileProperties(docxFileName)
            elif oper== 'open':
                docxFileName= docxFileName.replace("/", "\\")
                commandString = self.winWordExeLineE.text()+ ' '+ '"'+ docxFileName+ '"'               
                Popen(commandString)        # asynchronous   


    # --------------------------------------------------------------
    #   TABS functions
    # -------------------------------------------------------------- 
    
    def tabBarChanged(self,  index):
        if self.tabBarChangedCreation:
            # during tabs creation no operations on them allowed
            return 0
        #
        if index== hC.TAB_CONFIG:
            self.saveLoadSettings('load')
        if index== hC.TAB_BAR_NR_SIGIL:
            self.refreshSpineTabTable()
        elif index== hC.TAB_BAR_NR_MANIFEST:
            self.refreshManifestTabTable()
        elif index== hC.TAB_IMAGES_TOOLS:
            self.thumbPreviewRefresh()

    def loadSettingsFromConfigFile(self, configFilePath, fieldLst):
        configFile= hFc.File()
        configFile.setFileProperties(configFilePath)
        fileContent= configFile.read()
        fileContentLst= fileContent.splitlines()
        for line in fileContentLst:
            splitLineLst= line.split('=')    # [0]= field name, [1]= field value
            field= splitLineLst[0]
            value= splitLineLst[1].replace('"',  '')
            if field== fieldLst[0]:
                self.winWordExeLineE.setText(value)
            elif field== fieldLst[1]:
                self.editorExeLineE.setText(value) 
            elif field== fieldLst[2]:
                self.workingDirPathLineE.setText(value)
            elif field== fieldLst[3]: 
                self.docxSplitStyleLineE.setText(value)
            elif field== fieldLst[4]:
                self.docxNameChapStrLineE.setText(value)
            elif field== fieldLst[5]:
                self.currentFontFamily= value
                self.defaultFontFamilyComboBox.setCurrentText(value)
            elif field== fieldLst[6]:            
                self.currentFontSize= value                
                self.defaultFontSizeComboBox.setCurrentText(value)
                
    def saveSettings(self, configFilePath, fieldLst):
        fileContent= ''
        fileContent= fileContent+ fieldLst[0]+ '='+ self.winWordExeLineE.text()+ '\n'
        fileContent= fileContent+ fieldLst[1]+ '='+ self.editorExeLineE.text()+ '\n'
        fileContent= fileContent+ fieldLst[2]+ '='+ self.workingDirPathLineE.text()+ '\n'
        fileContent= fileContent+ fieldLst[3]+ '='+ self.docxSplitStyleLineE.text()+ '\n'
        fileContent= fileContent+ fieldLst[4]+ '='+ self.docxNameChapStrLineE.text()+ '\n'
        fileContent= fileContent+ fieldLst[5]+ '='+ self.defaultFontFamilyComboBox.currentText()+ '\n'
        fileContent= fileContent+ fieldLst[6]+ '='+ self.defaultFontSizeComboBox.currentText()
        #
        configFile= hFc.File()
        configFile.setFileProperties(configFilePath) 
        configFile.write(fileContent)    

    def saveLoadSettings(self, oper):
        fieldLst= ['winWordExePath', 'editorExePath', 'workingDir', 'docxSplitStyle',  'docxNameChapStr', 'fontFamily',  'fontSize']
        #
        appPath= os.path.dirname(__file__)+ '\\'
        configFilePath= appPath+ hC.configFile
#        print (f"saveLoad settings {configFilePath}")
        if oper== 'save':        
            self.saveSettings(configFilePath, fieldLst)
            msg= 'ok saving settings into file:\n'+ configFilePath
            self.logTextEditMsg(msg, hC.OK)
        elif oper== 'load':
            if os.path.exists(configFilePath): 
                # load from config file
                self.loadSettingsFromConfigFile(configFilePath, fieldLst)  
                msg= 'ok loading settings from file:\n'+ configFilePath
                self.logTextEditMsg(msg, hC.OK)
            else:
                msg= 'Config file\n'+ configFilePath+ '\nnot found. Restored default settings from hConstant.py' 
                self.logTextEditMsg(msg, hC.OK)                
                # read default from hConstant.py
                self.restoreDefaultTabConfig()


    def baseConfigTabConfig(self,  operation,  filter= ''):
        dialog = QFileDialog()
        if operation== 'd':
            # search for directory
            dirName = dialog.getExistingDirectory(self, "Open Directory", "", QFileDialog.ShowDirsOnly)
            if dirName:
                self.workingDirPathLineE.setText(dirName+ '/')
        else:
            # search for file
            fileName, ok = dialog.getOpenFileName(self,'Load File','', filter)
            if fileName:
                if operation== 'w':
                    self.winWordExeLineE.setText(fileName)
                elif operation== 'e':    
                    self.editorExeLineE.setText(fileName)

    def restoreDefaultTabConfig(self):
        self.workingDirPathLineE.setText(hC.workingDir)        
        self.winWordExeLineE.setText(hC.winWordExePath) 
        self.editorExeLineE.setText(hC.editorExePath) 
        self.docxSplitStyleLineE.setText(hC.CSV_STYLE_TO_SEARCH) 
        self.docxNameChapStrLineE.setText(hC.CHAPTER)  
        self.defaultFontFamilyComboBox.setCurrentText(hC.defaultFontFamily)              
        self.defaultFontSizeComboBox.setCurrentText(hC.defaultFontSize)
  
        
    def chooseFileTabDoc(self, operation, filter,  tab):  
        dialog = QFileDialog() 
        fileName, ok = dialog.getOpenFileName(self,'Choose File','', filter)
        if fileName:
            if tab== 'tabDoc':
                if operation== 'docx':
                    self.oTabDocx_DOCX.setFileProperties(fileName)
                    self.configWordDocLineEdit.setText(fileName)
                    csvFileName= self.workingDirPathLineE.text()+ self.oTabDocx_DOCX.onlyName+ '.csv'
                    # reset grid with value from csv
                    self.resetCsvValueTable() 
                    if os.path.exists(csvFileName):
                        self.configCsvLineEdit.setText(csvFileName) 
                        rc= hCsU.loadCsvFile(csvFileName, hC.CSV_TABLE_NUM_COL, self.csvValueTable)                
                        if rc== -1:
                            self.logTextEditMsg('wrong number of columns\nin .csv file '+ csvFileName+ \
                            '\nfor .docx file '+ fileName,  False)
                            self.configWordDocLineEdit.setText('')
                            self.configCsvLineEdit.setText('') 
                        else:
                            # reset fields start page, end page, file HTM, TXT, XHTML
                            self.configHtmFileLineEdit.setText('')        
#                            self.configTxtFileLineEdit.setText('')     
#                            self.configXhtmlFileLineEdit.setText('') 
                            self.configWordStartPageLineEdit.setText('')  
                            self.configWordEndPageLineEdit.setText('')                             
                    else:
                        msg= 'no .csv file for selected file '+ fileName.upper()+ '\ncreate .csv file in tab config'
                        self.logTextEditMsg(msg,  False)
                        self.configWordDocLineEdit.setText('')
                        self.resetParamTabDoc('tabDoc')
            elif tab== 'tabConf':
                self.oTabConf_DOCX.setFileProperties(fileName)
                self.tabConfigDocxLineEdit.setText(fileName)
                csvFileName= self.workingDirPathLineE.text()+ self.oTabConf_DOCX.onlyName+ '.csv'
                self.tabConfigCsvLineEdit.setText(csvFileName) 
                self.createCsvBtn.setEnabled(True)
                
    def createCsvTabDoc(self):
        FILE_DOCX= self.tabConfigDocxLineEdit.text()
        WORKING_DIR= self.workingDirPathLineE.text()
        #
        if WORKING_DIR== '' or not os.path.exists(WORKING_DIR):
            self.logTextEditMsg('no working dir or wrong path',  False)
            return -1
        #
        self.setCursor(QCursor(Qt.WaitCursor))
        #
        self.initLogTextEdit()
        self.crCsv= hCsU.CreateCsvFile()
        self.crCsv.csvLineAdded.connect(self.addLineToLog)
        self.crCsv.maxParagraphsSignal.connect(self.setTabBar)
        self.crCsv.progressTabBarSignal.connect(self.progressProcess)
        rc= self.crCsv.docx2Csv(FILE_DOCX, WORKING_DIR, self.docxSplitStyleLineE.text()) 
        if rc != '':
            self.logTextEditMsg(rc,  False)
        else:
            self.addLineToLog('\nNO ERROR IN .CSV FILE CREATION')
        #
        self.unsetCursor()
        #
        
    def resetParamTabDoc(self,  tab):
        if tab== 'tabDoc':
            for lineEdit, lbl in self.tabDocParamLst:
                lineEdit.setText('')
                lbl.setStyleSheet("color: "+ hC.blackColor)
            self.oTabDocx_DOCX.setFileProperties('')
            self.resetCsvValueTable()
            #
            self.startTranslationBtn.setEnabled(False)
            self.uploadSigilBtn.setEnabled(False)
            self.resetFontDefault()
#            self.initLogTextEdit()
        elif tab== 'tabConf':
            self.oTabConf_DOCX.setFileProperties('')
            self.tabConfigDocxLineEdit.setText('')
            self.tabConfigCsvLineEdit.setText('')
            self.createCsvBtn.setEnabled(False)
            self.initLogTextEdit()

    def resetFontDefault(self):
        self.fontComboBox.setCurrentText(self.defaultFontFamilyComboBox.currentText())
        self.sizeComboBox.setCurrentText(self.defaultFontSizeComboBox.currentText())

    def resetCsvValueTable(self):
        self.csvValueTable.clear()
        for rowN in range(self.csvValueTable.rowCount(), 0,  -1):
            self.csvValueTable.removeRow(rowN)

    def startTranslationOneStepTabDoc(self):
        if self.isTabConfOk(self.tabConfParamLst) and self.isTabDocOk(self.tabDocParamLst):
            self.initLogTextEdit()
            # set labels in translate TAB
            self.htmTextEditLbl.setText(self.configHtmFileLineEdit.text())
            #
            self.htmCodeSave= ''            
            #
            self.setCursor(QCursor(Qt.WaitCursor))
            # 
            self.translateFontFamily= self.fontComboBox.currentText()
            self.translateFontSize= self.sizeComboBox.currentText()   
            #
            self.translateOneStep= hTr.Translate(
                                [self.configWordDocLineEdit.text(),  
                                self.configHtmFileLineEdit.text(),              
                                self.configWordStartPageLineEdit.text(), 
                                self.configWordEndPageLineEdit.text(), 
                                self.workingDirPathLineE.text(), 
                                ],  self.bk, self.translateFontFamily, self.translateFontSize, self.searchChapterTitle)            
            # signal from translate object
            self.translateOneStep.htmFileReadySignal.connect(self.loadTranslatedHtmFile)       
            #
            self.translateOneStep.extractHtmFile()
            #
            self.uploadSigilBtn.setEnabled(True)
            if not self.translateOneStep.error:
                self.addLineToLog('\nTRANSLATION PROCESS SUCCESSFULLY COMPLETED')
                self.addLineToLog('you can:\na) see results in tab "Preview translation"\nb) load in Sigil with button "load"')
                self.uploadSigilBtn.setEnabled(True)
                self.startTranslationBtn.setEnabled(False)
            
            self.unsetCursor()
            #
#            self.startTranslationBtn.setEnabled(False)


    def uploadDocInSigil(self):
        self.uploadSigilBtn.setEnabled(False)
        htmFile= hFc.File()
        htmFile.setFileProperties(self.configHtmFileLineEdit.text())
        manifestId= htmFile.baseName
        fileContent= htmFile.readByte()
        fileContent= fileContent.decode('utf-8')
        mimeType= htmFile.mime
        loadIntoSpine= True
        askIfFileAlreadyIn= True
        rcOk= self.bkUtil.bkLoadFileIntoSigil(self.bk, manifestId, fileContent, mimeType, loadIntoSpine, askIfFileAlreadyIn)
        if rcOk:        
            # upload images, if any
            onlyName= Path(htmFile.baseNameAndPath).stem
            imgDirEN= onlyName+ '_files'
            imgDirIT= onlyName+ '_file'
            workingDir= self.workingDirPathLineE.text()
            imgMsg= ''
            imgDir= ''
            if os.path.exists(workingDir+ imgDirIT):
                imgDir= imgDirIT
            if os.path.exists(workingDir+ imgDirEN): 
                imgDir= imgDirEN
            if imgDir:    
                imgFileLst= os.listdir(workingDir+ imgDir)
#                print('uploadDocInSigil ', ' imgFileLst ',  imgFileLst)
                if not hC.DEBUG:
                    for n, file in enumerate(imgFileLst):
                        filePath= workingDir+ imgDir+ '/'+ file
                        self.loadImageInWordFileToSigil(filePath, onlyName)
                    nImg= str(n+ 1)
                    imgMsg= ' and '+ nImg+ ' images'
            self.logTextEditMsg('OK loading '+ manifestId+ imgMsg,  hC.OK)
            self.startTranslationBtn.setEnabled(False)


    def loadImageInWordFileToSigil(self, image,  onlyName):
        oImageFile= hFc.File()
        oImageFile.setFileProperties(image)
#        print(f"loadImageInWordFileToSigil onlyName {onlyName} baseName {oImageFile.baseName}")
        manifestId= onlyName+ '_'+ oImageFile.baseName
#        print(f"loadImageInWordFileToSigil manifestId {manifestId} ")
        fileContent= oImageFile.read()
        mimeType= oImageFile.mime
        loadIntoSpine= False
        askIfFileAlreadyIn= False
        #
#        print('load image ', manifestId)
        self.bkUtil.bkLoadFileIntoSigil(self.bk, manifestId, fileContent, mimeType, loadIntoSpine, askIfFileAlreadyIn)


    def loadTranslatedHtmFile(self, fileContent):
        if not self.translateOneStep.error:
            self.htmTextEdit.setPlainText(fileContent)      # view as is i.e. html
            self.htmCodeSave= fileContent                   # to review same way in  tabTranslationHtmPreview()
            self.addLineToLog('OK extraction of '+ self.configHtmFileLineEdit.text()+ ' from '+ \
                                self.oTabDocx_DOCX.baseName)
        else:
            self.logTextEditMsg('error extracting html from docx\n'+ fileContent[1:],  False)

#    def loadTranslatedTxtFile(self, fileContent): 
#        if not self.translate.error:
#            self.txtTextEdit.setPlainText(fileContent)
#            self.txtCodeSave= fileContent  
#            self.addLineToLog('OK creation of '+ self.configTxtFileLineEdit.text()+ ' from '+ \
#                                self.configHtmFileLineEdit.text())
#        else:
#            self.logTextEditMsg('error creating txt from html\n'+ fileContent[1:],  False)
#
#    def loadTranslatedXhtmlFile(self,  fileContent):
#        if not self.translate.error:
#            self.xhtmlTextEdit.setPlainText(fileContent)
#            self.xhtmlCodeSave= fileContent 
#            self.addLineToLog('OK creation of '+ self.configXhtmlFileLineEdit.text()+ ' from '+ \
#                                self.configTxtFileLineEdit.text())
#        else:
#            self.logTextEditMsg('error creating xhtml from txt\n'+ fileContent[1:],  False)        


    def selectedTableCell(self, row,  column):  
        onlyNameDocx= self.oTabDocx_DOCX.onlyName
        csvFile= self.configCsvLineEdit.text()
        if onlyNameDocx!= "" and csvFile!= "":
            valueLst= []
            for col in range (hC.CSV_TABLE_NUM_COL):
                valueLst.append(self.csvValueTable.item(row, col).text())
            #
            startPage= valueLst[hC.CSV_TABLE_COL_START_P]
            endPage= valueLst[hC.CSV_TABLE_COL_END_P]
            filePrefix= valueLst[hC.CSV_TABLE_COL_PREFIX]
            #
            if startPage== '' or endPage== '' or filePrefix== '':
                self.logTextEditMsg('missing start page or end page or prefix',  False)
            else:
                self.searchChapterTitle= valueLst[hC.CSV_TABLE_COL_TITLE] #used by hTranslateOneStep.py
                workingDir= self.workingDirPathLineE.text()
                chapter= self.docxNameChapStrLineE.text()
#                outputFileBaseName= workingDir+ onlyNameDocx+ hC.CHAPTER+ filePrefix
                outputFileBaseName= workingDir+ chapter+ filePrefix+ '_'+ onlyNameDocx
                #
                self.configHtmFileLineEdit.setText(outputFileBaseName+ '.htm')		
#                self.configTxtFileLineEdit.setText(outputFileBaseName+ '.txt')
#                self.configXhtmlFileLineEdit.setText(outputFileBaseName+ '.xhtml')
                self.configWordStartPageLineEdit.setText(startPage)
                self.configWordEndPageLineEdit.setText(endPage)
                self.startTranslationBtn.setEnabled(True)
                self.uploadSigilBtn.setEnabled(False)
        else:
            self.logTextEditMsg('both .docx AND .csv file names required to set export parameters',  False)

    # --------------------------------------------------------------
    #   tab Translation functions
    # --------------------------------------------------------------
    def tabTranslationPreview(self,  file):
        if file== 'htm':
            label= self.htmTextEditLbl.text()
            if os.path.exists(label):
                btnCaption= self.htmPreviewBtn.text()
                self.htmTextEdit.clear()
                if btnCaption== 'Preview':
                    self.htmPreviewBtn.setText('html view')
                    self.htmPreviewBtn.setIcon(QIcon(self.iconPath+ 'html.png'))
                    self.htmTextEdit.setHtml(self.htmCodeSave)
                else:
                    self.htmPreviewBtn.setText('Preview')
                    self.htmPreviewBtn.setIcon(QIcon(self.iconPath+ 'preview.png'))
                    self.htmTextEdit.setPlainText(self.htmCodeSave)
            else:
                self.logTextEditMsg(f"file '{label}' doesn't exists", False)
        elif file== 'xhtml':
            label= self.xhtmlTextEditLbl.text()
            if os.path.exists(self.xhtmlTextEditLbl.text()):
                btnCaption= self.xhtmlPreviewBtn.text()
                self.xhtmlTextEdit.clear()
                if btnCaption== 'Preview':
                    self.xhtmlPreviewBtn.setText('xhtml file')
                    self.xhtmlPreviewBtn.setIcon(QIcon(self.iconPath+ 'html.png'))
                    self.xhtmlTextEdit.setHtml(self.xhtmlCodeSave)
                else:
                    self.xhtmlPreviewBtn.setText('Preview')
                    self.xhtmlPreviewBtn.setIcon(QIcon(self.iconPath+ 'preview.png'))
                    self.xhtmlTextEdit.setPlainText(self.xhtmlCodeSave)                
            else:
                self.logTextEditMsg(f"file '{label}' doesn't exists", False)

 
    def tabTranslationEdit(self,  file):
        commandString= self.editorExeLineE.text()+ ' '
        if file== 'htm':
            label= self.htmTextEditLbl.text()
            if os.path.exists(label):
                commandString= commandString+ label
                Popen(commandString)
            else:
                self.logTextEditMsg(f"file '{label}' doesn't exists", False)

                
    #   font functions
    def currentTextChangedFont(self, locSizeComboBox, locSampleLbl, locLineEdit,  family):
        self.currentFontFamily= family
        defaultFontStyle= self.fontObj.getDefaultFontStyle(family)
        pointsLst= self.fontObj.getPoints(family, defaultFontStyle)
        
        size= locSizeComboBox.currentText()
        if not size:
            size= self.currentFontSize
        locSizeComboBox.clear()
        locSizeComboBox.addItems(pointsLst)
        locSizeComboBox.setCurrentText(size)
        self.setFontChangeFont(family, locLineEdit)
        #   sample text
        self.setSampleText(locSampleLbl, self.currentFontFamily, self.currentFontSize)

    def currentTextChangedSize(self, locSampleLbl,  initFlag,  fontSize):
####        if initFlag:
####            self.currentFontSize= self.currentFontSize
####            initFlag= False
####        else:
        if not fontSize:
            fontSize= self.currentFontSize
        self.currentFontSize= fontSize
        #   sample text
        self.setSampleText(locSampleLbl, self.currentFontFamily, self.currentFontSize)

    def setFontChangeFont(self, family, locLineEdit):
        # font
        font = QFont(family, 10)
        # setting font of self.fontComboBox line edit
        locLineEdit.setFont(font) 

    def setSampleText(self, locSampleLbl, family, fontSize):      
        sampleTextStyle= '''
        border-bottom:1px solid black;
        background-color:white
        '''       
        font = QFont(family, int(fontSize))
        locSampleLbl.setFont(font)
        #
        locSampleLbl.setStyleSheet(sampleTextStyle) 
        

    def setFontsInCombo(self,  locCombo):
        for index in range(locCombo.count()):        
            font = QFont(locCombo.itemText(index), 12)
            locCombo.setItemData(index, font, Qt.FontRole)   

    def initSampleLabel(self, locSampleLbl):
        locSampleLbl.setFixedSize(210, 70)
        locSampleLbl.setAlignment(Qt.AlignCenter) 
          
    # --------------------------------------------------------------
    #   tab Sigil MANIFEST functions
    # --------------------------------------------------------------
    def refreshManifestTabTable(self):
        self.manifestTblW.clear()
        self.manifestTblW.setRowCount(0)         
        if hC.DEBUG:
            manifestLst= []
            manifestLst=[('id129', 'Text/La_banda_Sacco_(La_memoria)_(It_split_000.html',	'mimetType'),	
                         ('id128',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_001.html',	'mimetType'),	
                         ('id127',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_002.html',	'mimetType'),	
                         ('id126',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_003.html',	'mimetType'),	
                         ('id125',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_004.html',	'mimetType'),	
                         ('id124',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_005.html',	'mimetType'),	
                         ('id123',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_006.html',	'mimetType'),	
                         ('id122',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_007.html',	'mimetType'),	
                         ('id121',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_008.html',	'mimetType'),	
                         ('id120',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_009.html',	'mimetType'),	
                         ('id119',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_010.html',	'mimetType'),	
                         ('id118',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_011.html',	'mimetType'),	
                         ('id117',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_012.html',	'mimetType'),	
                         ('id116',	 'Text/La_banda_Sacco_(La_memoria)_(It_split_013.html',	'mimetType')]
        else:
            try:
                manifestLst= self.bkUtil.bkManifestList(self.bk)
            except:
                err= '*refreshManifestTabTable'+ str(sys.exc_info()[0])+ '\n'+ str(sys.exc_info()[1])+ '\n'+ str(sys.exc_info()[2])
                self.logTextEditMsg(err,  hC.ERR)
        if manifestLst:       
            self.manifestTblW.tableSetHeader(self.manifestTblW, hC.manifestTableHeadLst)
            self.manifestTblW.manifestTableFromList( self.manifestTblW, manifestLst, TABLE_NOT_EDITABLE)


    # --------------------------------------------------------------
    #   tab Sigil SPINE functions
    # --------------------------------------------------------------
    def refreshSpineTabTable(self):
            self.spineTblW.clear()
            self.spineTblW.setRowCount(0)           
            if hC.DEBUG:
                self.dictSpine= {} 
                
                self.dictSpine=  {'titlepage': 'Text/titlepage.xhtml',
                 'id129': 'Text/La_banda_Sacco_(La_memoria)_(It_split_000.html',
                 'id128': 'Text/La_banda_Sacco_(La_memoria)_(It_split_001.html',
                 'id127': 'Text/La_banda_Sacco_(La_memoria)_(It_split_002.html',
                 'id126': 'Text/La_banda_Sacco_(La_memoria)_(It_split_003.html',
                 'id125': 'Text/La_banda_Sacco_(La_memoria)_(It_split_004.html',
                 'id124': 'Text/La_banda_Sacco_(La_memoria)_(It_split_005.html',
                 'id123': 'Text/La_banda_Sacco_(La_memoria)_(It_split_006.html',
                 'id122': 'Text/La_banda_Sacco_(La_memoria)_(It_split_007.html',
                 'id121': 'Text/La_banda_Sacco_(La_memoria)_(It_split_008.html',
                 'id120': 'Text/La_banda_Sacco_(La_memoria)_(It_split_009.html',
                 'id119': 'Text/La_banda_Sacco_(La_memoria)_(It_split_010.html',
                 'id118': 'Text/La_banda_Sacco_(La_memoria)_(It_split_011.html',
                 'id117': 'Text/La_banda_Sacco_(La_memoria)_(It_split_012.html',
                 'id116': 'Text/La_banda_Sacco_(La_memoria)_(It_split_013.html',
                 'id115': 'Text/La_banda_Sacco_(La_memoria)_(It_split_014.html',
                 'id114': 'Text/La_banda_Sacco_(La_memoria)_(It_split_015.html',
                 'id113': 'Text/La_banda_Sacco_(La_memoria)_(It_split_016.html',
                 'id112': 'Text/La_banda_Sacco_(La_memoria)_(It_split_017.html',
                 'id111': 'Text/La_banda_Sacco_(La_memoria)_(It_split_018.html',
                 'id110': 'Text/La_banda_Sacco_(La_memoria)_(It_split_019.html',
                 'id19': 'Text/La_banda_Sacco_(La_memoria)_(It_split_020.html',
                 'id18': 'Text/La_banda_Sacco_(La_memoria)_(It_split_021.html',
                 'id17': 'Text/La_banda_Sacco_(La_memoria)_(It_split_022.html',
                 'id16': 'Text/La_banda_Sacco_(La_memoria)_(It_split_023.html',
                 'id15': 'Text/La_banda_Sacco_(La_memoria)_(It_split_024.html',
                 'id14': 'Text/La_banda_Sacco_(La_memoria)_(It_split_025.html',
                 'id13': 'Text/La_banda_Sacco_(La_memoria)_(It_split_026.html',
                 'id12': 'Text/La_banda_Sacco_(La_memoria)_(It_split_027.html',
                 'id11': 'Text/La_banda_Sacco_(La_memoria)_(It_split_028.html'}                   
            else:
                self.dictSpine= self.bkUtil.bkMakeSpineDict(self.bk)
            try:
                self.spineTblW.tableSetHeader(self.spineTblW, hC.tableHeadLst)
            except:
                err= '*refreshSpineTabTable'+ str(sys.exc_info()[0])+ '\n'+ str(sys.exc_info()[1])+ '\n'+ str(sys.exc_info()[2])
                self.logTextEditMsg(err,  hC.ERR)
            self.spineTblW.tableValueFromDict( self.spineTblW, self.dictSpine, TABLE_NOT_EDITABLE)
            
    def spineTabRemoveRow(self):
        if self.spineTblW.rowCount()== 1:
            self.logTextEditMsg("Last document in Sigil. Can't delete it", False)
        else:    
            currRow= self.spineTblW.currentRow()
            manifestIdToDelete= self.spineTblW.item(currRow,  0).text()
            confirmMsg = QMessageBox() # create not_found_msg object to avoid it
                                      # referenced before assignment error
            confirmMsg= QMessageBox.question(self, "Delete SPINE file ",
            		"Are you sure you want to delete "+ manifestIdToDelete+ "?",
            		QMessageBox.Yes | QMessageBox.No, QMessageBox.Yes)
            if confirmMsg == QMessageBox.Yes:            
                # remove from interface
                self.spineTblW.removeRow(currRow)
                # remove from Spine AND from Manifest
                self.bkUtil.bkDeleteFile(self.bk, manifestIdToDelete)

    def spineSaveToSigil(self):
        spineDict= self.spineTblW.fromTableToDict(self.spineTblW)
        if spineDict:
            keyLst= list(spineDict.keys())
            newSpineTupleList= []
            for key in keyLst:
                newSpineTupleList.append((key, 'yes'))
            setSpine= self.bkUtil.bkSetSpine(self.bk, newSpineTupleList)  
            if setSpine:
                self.logTextEditMsg(hC.OPERATION_SUCCESS, True)

 
    # --------------------------------------------------------------
    #   log textEdit functions
    # -------------------------------------------------------------- 
    def processErrBkUtil(self, errMsg):
        self.logTextEditMsg(errMsg,  False)    
    
    def initLogTextEdit(self):
        self.logTextEdit.clear()
        self.setLogtextEditColor(hC.blackColor,  hC.whiteColor)
        self.csvProgBar.hide()

    def setLogtextEditColor(self, foreColor,  backColor):
        self.logTextEdit.setTextColor(QColor(foreColor))
        self.logTextEdit.setStyleSheet('background-color:'+ backColor)

    def addLineToLog(self, line):
        self.logTextEdit.append(line)
        self.logTextEdit.repaint()

    def logTextEditMsg(self, msg, log):
        '''
            log= True => ok msg     log= False => error message
        '''        
        self.initLogTextEdit()
        if log:
            self.setLogtextEditColor(hC.blackColor, hC.whiteColor)
            self.addLineToLog(msg)
        else:    
            self.setLogtextEditColor(hC.whiteColor, hC.errorColor)
            self.addLineToLog('EXECUTION ERROR:\n'+ msg)

    # --------------------------------------------------------------
    #   csv progress bar functions
    # --------------------------------------------------------------
    def setTabBar(self,paragraph ):
#        print (f"paragraph {paragraph}  ")
        self.csvProgBar.setRange(0, paragraph)
        self.csvProgBar.setTextVisible(False)
        self.csvProgBar.show()
        
    def progressProcess(self, value):
        self.csvProgBar.setValue(value)
                
    # --------------------------------------------------------------
    #   check configuration before starting translation
    # -------------------------------------------------------------- 
    def isTabConfOk(self, locCheckLst):
        checkOk= True
        for textField, label in locCheckLst:
            value= textField.text()
            if not os.path.exists(value):   # value= '' or no path
                label.setStyleSheet("color: "+ hC.redColor)
                checkOk= False
            else:
                label.setStyleSheet("color: "+ hC.blackColor)
        if not checkOk:           
            self.logTextEditMsg('incomplete configuration\n'+ \
                                'choose .docx file and select a row in the grid',  False) 
        return checkOk

    def isTabDocOk(self, locCheckLst):
        checkOk= True
        for i,  (textField, label) in list(enumerate(locCheckLst)):
            value= textField.text()
            if i== hC.ST_PAGE_POS or i==hC.EN_PAGE_POS:
                # check not empty and numeric  .isnumeric('') -> False
                if not value.isnumeric():
                    checkOk= False
                    label.setStyleSheet("color: "+ hC.redColor)
                else:
                    label.setStyleSheet("color: "+ hC.blackColor)
            elif i== hC.DOCX_FIELD_POS or i== hC.CSV_FIELD_POS:
                # check on path
                if not os.path.exists(value):
                    checkOk= False  
                    label.setStyleSheet("color: "+ hC.redColor)
                else:
                    label.setStyleSheet("color: "+ hC.blackColor)
            elif i== hC.HTM_FIELD_POS:  # or i== hC.TXT_FIELD_POS or i== hC.XHTML_FIELD_POS:
                # the file may not exists but its directory component of the pathname must
                if not os.path.dirname(value):
                    checkOk= False 
                    label.setStyleSheet("color: "+ hC.redColor)
                else:
                    label.setStyleSheet("color: "+ hC.blackColor)
        if not checkOk:            
             self.logTextEditMsg('incomplete configuration\n'+ \
                'choose .docx file and select a row in the grid',  False) 
        return checkOk       

    # --------------------------------------------------------------
    #   image tool TAB functions
    # -------------------------------------------------------------- 

    def loadAndPreviewImage(self):
        '''
            choose image and preview it on screen. Redim with ctrl+ mouse wheel
        '''       
        imageFile, _ = QFileDialog.getOpenFileName(self, "Open Image", "",
            "JPG Files (*.jpg );;PNG Files (*.png);;GIF Files (*.gif)")
        if imageFile:
            # dimensioni
            imgFile= hFc.File()
            imgFile.setFileProperties(imageFile)
            self.width= imgFile.pixmapWidth
            self.height= imgFile.pixmapHeight
            # set labels image path and image dim in pixel on image tools TAB
            self.imgPathLbl.setText(imgFile.baseNameAndPath)        
            dim= self.width+ 'x'+ self.height+ 'px | '+ imgFile.fileSize+ ' KB'   # widthxheight px | ... KB'
            self.imgDimInPixel.setText(dim)
            # used to re-dim image
            self.currentWidth= int(self.width)
            self.currentHeight= int(self.height)
            self.step= 0             
            #--- defined in image tool TAB
            self.imgLoaded = QPixmap(imageFile)
#            self.imageLoadedLbl.setPixmap(self.imgLoaded.scaled(self.currentWidth, self.currentHeight, 
#                                    Qt.KeepAspectRatio))  #, Qt.SmoothTransformation))
            self.imageLoadedLbl.setPixmap(self.imgLoaded.scaled(300, 300, 
                                        Qt.KeepAspectRatio))   

            self.btnLoadImage.setEnabled(True)
        
    def redimWheelEvent(self, direction):
        '''
            if direction > 0
                wheel rotated forwards away from the user => image enlarges
            else
                wheel rotated backwards toward the user   => image shriks
        https://stackoverflow.com/questions/41738554/python-pyqt4-how-to-detect-control-press-mouse-scroll        
        '''
        increment= -1
        if direction > 0:
            increment= 1
        #
        self.step= self.step+ increment
        if self.step > hC.MAX_MOUSE_STEP:
            self.step= hC.MAX_MOUSE_STEP
        if self.step < -hC.MAX_MOUSE_STEP:
            self.step= -hC.MAX_MOUSE_STEP
        #
        self.changeImageSize(self.step)
 
    def changeImageSize(self, step):       
        w= self.currentWidth+ step* hC.ENLARGE_FACTOR
        h= self.currentHeight+ step* hC.ENLARGE_FACTOR
        #
        self.imageLoadedLbl.setPixmap(self.imgLoaded.scaled(w, h, Qt.KeepAspectRatio))

    def loadImageToSigil(self):
        image= self.imgPathLbl.text()
        if image:
            oImageFile= hFc.File()
#            print(f"image {image}")
            oImageFile.setFileProperties(image)
            manifestId= oImageFile.baseName
            fileContent= oImageFile.read()
            mimeType= oImageFile.mime
            loadIntoSpine= False
            askIfFileAlreadyIn= True
            #
            rcOk= self.bkUtil.bkLoadFileIntoSigil(self.bk, manifestId, fileContent, mimeType, loadIntoSpine, askIfFileAlreadyIn)
            if rcOk:
                self.logTextEditMsg('OK loading '+ manifestId,  hC.OK)
                self.thumbPreviewRefresh()
 
    def resetImageInToolTab(self):
        self.imageLoadedLbl.clear()
        self.imageLoadedLbl.setText('image')
        self.imgPathLbl.setText('image path')
        self.imgDimInPixel.setText('widthxheight px | ... KB')
        self.btnChooseImage.setEnabled(True)
        self.btnLoadImage.setEnabled(False)
        self.creatThumbXhtmlBtn.setEnabled(False)


    # --------------------------------------------------------------
    #   thumbnails management
    # --------------------------------------------------------------
    def clickedTableView(self,  item):
        # init loaded and preview image area
        manifestId= item.data()[1]
        self.thumbManifest= manifestId
        fileContent= self.bkUtil.bkFileContentFromManifestId(self.bk,  manifestId)
        if fileContent:
            self.resetImageInToolTab
            #
            self.imgLoaded = QPixmap()
            self.imgLoaded.loadFromData(fileContent)
            self.currentWidth= self.imgLoaded.width()
            self.currentHeight= self.imgLoaded.height()
            self.step= 0          
            #
            self.imageLoadedLbl.setPixmap(self.imgLoaded.scaled(300, 300, 
                                        Qt.KeepAspectRatio))  #, Qt.SmoothTransformation)) 
       
            # when loading an image from thumbnails disable 'choose image' and 'load image' button
            self.creatThumbXhtmlBtn.setEnabled(True)
            self.btnChooseImage.setEnabled(False)
            self.btnLoadImage.setEnabled(False)  
            # set labels image path and image dim in pixel on image tools TAB
            self.imgPathLbl.setText('Images/'+ manifestId)   
            dim= str(self.currentWidth)+ 'x'+ str(self.currentHeight)+ 'px'   # widthxheight px | ... KB'
            self.imgDimInPixel.setText(dim)


    def thumbPreviewRefresh(self):
        self.thumbTableView.setModel(self.model)
        self.model.clearTable(self.thumbTableView)
####        if hC.DEBUG:
####            imageFile= hFc.File()
####            for n, fn in enumerate(glob.glob("C:\\Users\\Sergio\\Pictures\\Saved Pictures\\*.gif")):
####                    imageFile.setFileProperties(fn)
####                    data= imageFile.read()
####                    image = QImage()
####                    image.loadFromData(data)
####                    item = hTp.preview(n, fn, image)
####                    self.model.previews.append(item)
####        else:
        imgDataDict= self.bkUtil.bkSearchManifestImage(self.bk)
        if imgDataDict:
            keyLst= list(imgDataDict.keys())   
            for n,  fn in enumerate(keyLst):
                try:
                    image = QImage()
                    image.loadFromData(imgDataDict[fn])
                    item = hTp.preview(n, fn, image)
                    self.model.previews.append(item)    
                    self.model.layoutChanged.emit()     
                    self.thumbTableView.resizeRowsToContents()
                    self.thumbTableView.resizeColumnsToContents()                        
                except:
                    err= '*thumbPreviewRefresh\n'+ str(sys.exc_info()[0])+ '\n'+ str(sys.exc_info()[1])+ '\n'+ str(sys.exc_info()[2])
                    self.logTextEditMsg(err,  hC.ERR)
                            

    def createXhtmlFromThumb(self):
        fileContent= hXt.imgXhtmlPagTemplate.replace("BASENAME", self.thumbManifest)   # thumbManifest set in clickedTableView()
        manifestId= 'IMG_'+ self.thumbManifest[0:-3]+ 'xhtml'
        mimeType= "application/xhtml+xml"
        loadIntoSpine= True      # xhtml with image href recorded in Spine
        askIfFileAlreadyIn= True
        try:
            self.bkUtil.bkLoadFileIntoSigil(self.bk, manifestId, fileContent, mimeType, loadIntoSpine, askIfFileAlreadyIn)
            msg= 'successfully created xhtml file for '+ manifestId
            self.logTextEditMsg(msg,  hC.OK)
        except:
            err= '*createXhtmlFromThumb'+ str(sys.exc_info()[0])+ '\n'+ str(sys.exc_info()[1])+ '\n'+ str(sys.exc_info()[2])
            self.logTextEditMsg(err,  hC.ERR)

    #-------------------------------------------------------
    # misc pages TAB functions
    #  https://stackoverflow.com/questions/55665317/pyqt5-qtableview-how-to-disable-user-interaction-selection-while-keeping-the-de
    #-------------------------------------------------------
        
    def getFileFromBk(self, manifestId,  index):
        if hC.DEBUG:
            if index== 6:        # TOC
                return False
            # in debug mode always gets template from hXhtmlTemplate
            fileContent= hXt.pagXhtmlTempLst[index]
            if not self.pageLoaded:
                # searches values in page from sigil to load in miscPageTblW table
                valuesToLoadLst= self.getValueFromPage(fileContent,  index)  
                for groupLst in valuesToLoadLst:
                    self.miscPageTblW.addXTableFromList(self.miscPageTblW, groupLst, hXt.pagXhtmlAddIsModLst[index]) 
                self.pageLoaded= True
                self.setAddDelBtnStatus(index)
                return True
            else:
                return False        # add new block (if btn enabled)
        else:
            fileContent= ''
            if self.bkUtil.bkSearchManifestId(self.bk, manifestId):
                fileContent= self.bkUtil.bkFileContentFromManifestId(self.bk, manifestId)
            if fileContent:    
                if not self.pageLoaded:
                    # searches values in page from sigil to load in miscPageTblW table
                    valuesToLoadLst= self.getValueFromPage(fileContent,  index)  
####                    if index== 6:
####                        return False        # no values to load for TOC page
                    for groupLst in valuesToLoadLst:
                        self.miscPageTblW.addXTableFromList(self.miscPageTblW, groupLst, hXt.pagXhtmlAddIsModLst[index]) 
                    self.pageLoaded= True
                    self.setAddDelBtnStatus(index)
                    return True
                else:
                    return False        # add new block (if btn enabled)
            else:
                # no file in Sigil: gets template from hXhtmlTemplate
                return False

    def getValueFromPage(self, fileContent,  index):
        '''
            return a list of sublist, one sublist for each group of data to insert in the table.
            Each sublist is made of tuplas (one for each line of the group).
            Each tupla has two terms: (line name, value)
            Example for page alsoBy:
            [ [('title', 'titolo 1')], [('title', 'titolo 2')], [('title', 'titolo 3')] ]
        '''
        locTuplasGuideLst= hXt.pagXhtmlAddGuideLst[index]
        fieldsLst= []
        for field, value in locTuplasGuideLst:
            fieldsLst.append(field)
        #
        # get values from page
        # index= 0 ['author', 'title', 'subTitle', 'publisher', 'choose'] 
        # index= 1 ['title', 'subtitle']
        valueLst= []
        valueGroupLst= []
        if index== 0 or index==1:
            for field in fieldsLst:
                if field != 'choose':
                    filterStart= 'class="'+ field+ '">'
                    filterEnd= '</p>'
                else:
                    filterStart= 'src="'
                    filterEnd= '" height='
                start= fileContent.find(filterStart)
                end= fileContent.find(filterEnd,  start)
                # in some cases (eg. title page and image not loaded) start or end or both have value -1 (filter not found)
                if start != -1 and end != -1:
                    value= fileContent[start+ len(filterStart):end]
                else:
                    value= ''
                ##############################
                htmlCharReverseLst= list(hC.htmlCharReverseDict.keys())
                value= self.specialCharReplace(htmlCharReverseLst, value)
                ##############################
                
                valueLst.append(value)
            valueGroupLst.append(valueLst)    
        #
        # get values from page
        # index= 2 ['copy', 'emptyLine'] 
        # index= 3 ['dedica', 'emptyLine']
        # index= 4 ['title']
        # index= 5 ['epigraph', 'author', 'emptyLine']
        if index== 2 or index== 3 or index== 4 or index== 5:
            offset= 0
            loop= True
            fieldNr= len(locTuplasGuideLst)
            while loop:
                fieldCounter= 0
                for field in fieldsLst:
                    filterStart= 'class="'+ field+ '">'
                    filterEnd= '</p>'
                    start= fileContent.find(filterStart,  offset)
                    end= fileContent.find(filterEnd,  start)
                    if start== -1 or end== -1: 
                        loop= False
                        break
                    else:
                        value= fileContent[start+ len(filterStart):end]
                        ############################
                        htmlCharReverseLst= list(hC.htmlCharReverseDict.keys())
                        value= self.specialCharReplace(htmlCharReverseLst, value)                        
                        ############################
                        valueLst.append(value)
                        offset= end
                        fieldCounter+= 1
                        if fieldCounter== fieldNr:
                            valueGroupLst.append(valueLst)
                            valueLst= []
                            fieldCounter= 0            
        if index== 6:
            startSearch= '<a href="half.xhtml">'
            endSearch= '</a>'
            titleTag= hMc.BetweenFilters(fileContent, startSearch, endSearch)
            title= titleTag.search()            
            valueToLoadLst= [[('title', title)]]
            return valueToLoadLst
        #
        # for aech group merges guide (field names in locTuplasGuideLst) with values (valueGroupLst)
        valuesToLoadLst= []
        for valueGroup in valueGroupLst:
            groupLst= []
            for n, tupla in enumerate (locTuplasGuideLst):
                field= tupla[0]
                value= valueGroup[n]
                t= (field, value)
                groupLst.append(t)
            valuesToLoadLst.append(groupLst)            
        #             
        return valuesToLoadLst            

    def specialCharReplace(self, htmlCharReverseLst, value):
        htmlCharReverseLst= list(hC.htmlCharReverseDict.keys())
        for k in htmlCharReverseLst:    
            if k in value:
                value= value.replace(k, hC.htmlCharReverseDict[k]) 
        return value

    def addBlockToMiscPageTable(self):        
        index= self.miscPagesCombo.currentIndex()
        manifestId= self.miscPagesCombo.currentText()+ '.xhtml'
        if index== 6:
            manifestId= 'custom'+ manifestId    #customTOC.xhtml
        if not self.getFileFromBk(manifestId,  index):
            # no file in Sigil get template from  hXhtmlTemplate.py
            if index== 5 or index== 4 or index== 3 or index== 2 or index== 1 or index== 0:
                self.createInputLines(index)
            if index== 6:
                if hC.DEBUG:
                    msg=  '<h3>No file Toc.xhtml in debug mode.<h3>'
                    QMessageBox.warning(self, 'Warning',msg, QMessageBox.Ok, QMessageBox.Ok)
                else:
                    self.createInputLines(index)

 
 
    def createInputLines(self, index):
        self.miscPageTblW.addXTableFromList(self.miscPageTblW, hXt.pagXhtmlAddGuideLst[index], 
                hXt.pagXhtmlAddIsModLst[index]) 
        self.setAddDelBtnStatus(index)
                
                
    def setAddDelBtnStatus(self, index):
        status= hXt.pagXhtmlBtnAddEnableLst[index]
        self.addRowMiscPageBtn.setEnabled(status)
        self.delRowMiscPageBtn.setEnabled(status)
        self.miscPagesCombo.setEnabled(False)
    

    def clearMiscPageTable(self):
        self.miscPageTblW.clearTable(self.miscPageTblW)
        self.addRowMiscPageBtn.setEnabled(True)
        self.delRowMiscPageBtn.setEnabled(True)
        self.miscPagesCombo.setEnabled(True)
        # also clear page preview QTextEdit
        self.saveXhtml= ''
        self.miscPagePreviewClicked()
        # clear log area
        self.initLogTextEdit()
        # reset flag page value already loaded
        self.pageLoaded= False
        # btn caption to ' preview'
        btnCaption= self.miscPagePreviewBtn.text()
        if btnCaption != ' preview':
            self.miscPagePreviewBtn.setText(' preview')
            self.miscPagePreviewBtn.setIcon(QIcon(self.iconPath+ 'preview.png'))

        
    def deleteBlockMiscPage(self):
        pageIndex= self.miscPagesCombo.currentIndex()
        blockElementNr= len(hXt.pagXhtmlAddGuideLst[pageIndex])
        currRow= self.miscPageTblW.currentRow()
        mod= currRow% blockElementNr
        if blockElementNr== 3:
            if mod== 2:
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
            elif mod==1:
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow()+1)
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())            
            elif mod==0:
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow()+1)
                self.miscPageTblW.removeRow(self.miscPageTblW.currentRow()+1) 
                if self.miscPageTblW.rowCount()== 1:
                    self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
        elif blockElementNr== 2:       
            self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())
            self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())  
        elif blockElementNr== 1: 
            self.miscPageTblW.removeRow(self.miscPageTblW.currentRow())

                
    def signalPublisherImgReceived (self, myLst):  
        row= myLst[0]
        col= myLst[1]
        fileName= myLst[2]
        #
        item= QTableWidgetItem(fileName)
        self.miscPageTblW.setItem(row, col, item ) 
        item.setFlags(item.flags() ^ Qt.ItemIsEditable)     # no changes allowed on image fiels
        
        
    def updateXHTML(self):
        # get data from table 
        tableDict= self.miscPageTblW.fromTableToDictRepeatedKey(self.miscPageTblW)        
        if tableDict:
            xhtmlToAdd= ''
            xhtmlImg= ''
            keyLst= list(tableDict.keys())
            for n, key in enumerate(keyLst):
                if key != 'choose':
                    value= tableDict[key]
                    key= (key if key.find('|') < 0 else key[0:key.find('|')])      # remove |... from key
                    xhtmlToAdd= xhtmlToAdd+ '<p id="row'+ str(n)+ '" class="'+ key +'">'+ value+ '</p>\n'
                else:
                    if tableDict[key]:
                        xhtmlImg= '<img src="../Images/'+ tableDict[key]+ '" height="70" width="50"/>'   
            # get page index from combo
            index= self.miscPagesCombo.currentIndex()
            if index== 5 or index== 4 or index== 3 or index== 2 or index== 1 or index== 0:    
                # get xhtml template
                xhtml= hXt.pagXhtmlTempLst[index]
                # get cede between <div> </div>  and code between <span> </span>
                regexDivLst= re.findall(hXt.regexDivDiv,  xhtml)
                regexSpanLst= re.findall(hXt.regexSpanSpan,  xhtml) 
                # replace code in xhtml template
                xhtml= xhtml.replace(regexDivLst[0], '\n'+  xhtmlToAdd+  '\n')
                if regexSpanLst:
                    xhtml= xhtml.replace(regexSpanLst[0], '\n'+ xhtmlImg+  '\n')                        
            if index== 6:
                # xhtmlToAdd is like <p id="row0" class="title">TITLE</p>
                filter= '(?<=\>)(.*?)(?=\<)'
                lst= re.findall(filter, xhtmlToAdd) 
                title= lst[0]
                xhtml= self.tocElaboration(title)
            #
            #   NEW.  Filter unwanted characters in user's input (like &) in all 
            #   pages but TOC
            #
            if index != 6:
                htmlCharLst= list(hC.htmlCharDict.keys())
                for char in htmlCharLst:
                    if char in xhtml:
                        xhtml= xhtml.replace(char, hC.htmlCharDict[char])
                        break 
            #
            self.displaysXhtml(xhtml)
            
            
    def displaysXhtml(self, xhtml):
        #
        # displays xhtml 
        self.xhtmlTEdit.setPlainText(xhtml) 
        self.saveXhtml= xhtml
        # btn caption to ' preview'
        btnCaption= self.miscPagePreviewBtn.text()
        if btnCaption != ' preview':
            self.miscPagePreviewBtn.setText(' preview')
            self.miscPagePreviewBtn.setIcon(QIcon(self.iconPath+ 'preview.png'))

    def tocElaboration(self,  title):
        manifestId= 'customTOC.xhtml'
        id= self.bkUtil.bkSearchManifestId(self.bk, manifestId)
        if not id:
            table= ''
            for manifestId, linear in self.bk.getspine():
                fileContent= self.bk.readfile(manifestId)
                h3Tag= hMc.BetweenFilters(fileContent, '<h3>', '</h3>')
                h3TagContent= h3Tag.search()
                if h3TagContent != '':
                    h3Lst= h3TagContent.split('<br/>')
                    hrefStart= '<a href="'+ manifestId+ '">'
                    hrefEnd= '</a>'
                    table+= '\n<tr>'
                    if len(h3Lst) > 1:
                        cell0= h3Lst[0]
                        cell1= h3Lst[1]
                    else:
                        cell0= ''
                        cell1= h3Lst[0]
                    table= table+ '\n'+ '<td>'+ hrefStart+ cell0+ hrefEnd+ '</td>'
                    table= table+ '<td>'+ hrefStart+ cell1+ hrefEnd+ '</td>'+ '\n'
                    table+= '</tr>'
            table+= '\n'
            #
            fileContent= hXt.TOCpage
            fileContent= fileContent.replace('TABLE_CONTENT', table)
            fileContent= fileContent.replace('HALF_TITLE_PAGE', title)	
        else:
            # works on customTOC.xhtml already existing
            fileContent= self.bk.readfile(manifestId)
            startSearch= '<a href="half.xhtml">'
            endSearch= '</a>'
            titleTag= hMc.BetweenFilters(fileContent, startSearch, endSearch)
            titleToChange= titleTag.search()
            # work also with titleToChange= ''
            toReplace= startSearch+ titleToChange
            replaceWith= startSearch+ title
            fileContent= fileContent.replace(toReplace, replaceWith, 1)
####            print(50*'#####################################')
####            print('DEBUG fileContent')
####            print (fileContent)
####            print(50*'#####################################')
        #
        return (fileContent)         
        

    def miscPagePreviewClicked(self):
        if self.xhtmlTEdit.toPlainText():
            btnCaption= self.miscPagePreviewBtn.text()
            if btnCaption== ' preview':
                self.miscPagePreviewBtn.setText(' xhtml view')
                self.miscPagePreviewBtn.setIcon(QIcon(self.iconPath+ 'html.png'))
                self.xhtmlTEdit.setHtml(self.saveXhtml)          
            else:
                self.miscPagePreviewBtn.setText(' preview')
                self.miscPagePreviewBtn.setIcon(QIcon(self.iconPath+ 'preview.png'))
                self.xhtmlTEdit.toHtml()
                self.xhtmlTEdit.setPlainText(self.saveXhtml)            


    def loadMiscPageIntoSigil(self):
        if self.saveXhtml:
            manifestId= self.miscPagesCombo.currentText()+ '.xhtml'
            index= self.miscPagesCombo.currentIndex()
            if index== 6:
                manifestId= 'custom'+ manifestId    #customTOC.xhtml
            loadIntoSpine= True
            askIfFileAlreadyIn= True
            retVal= self.bkUtil.bkLoadFileIntoSigil(self.bk, manifestId, self.saveXhtml, "application/xhtml+xml", loadIntoSpine, askIfFileAlreadyIn)
            if retVal:
                msg= 'successfully created xhtml file for '+ manifestId
                self.logTextEditMsg(msg,  hC.OK)


    def miscPageTblWClicked(self, row,  col):
#        print(f"row {row}   col {col}")
        pass

    #-------------------------------------------------------
    #   tab bk_test functions
    #-------------------------------------------------------

    def initLexer(self):
        fontBase= QFont("Courier New", 11)
        fontBold= QFont("Courier New", 11, QFont.Bold)
        #
        self.lexer.setFont(fontBase, -1)      # same font for all styles
        #
        self.lexer.setColor(QColor(hC.blue), self.lexer.Keyword)
        self.lexer.setFont(fontBold, self.lexer.Keyword)
        #
        self.lexer.setColor(QColor(hC.darkGrey), self.lexer.DoubleQuotedString)
        self.lexer.setColor(QColor(hC.darkGrey), self.lexer.SingleQuotedString)
        self.lexer.setColor(QColor(hC.darkGreen), self.lexer.TripleSingleQuotedString) 
        self.lexer.setColor(QColor(hC.darkGreen), self.lexer.TripleDoubleQuotedString)
        self.lexer.setColor(QColor(hC.fuchsia), self.lexer.ClassName)
        self.lexer.setFont(fontBold, self.lexer.ClassName)
        self.lexer.setColor(QColor(hC.fuchsia), self.lexer.FunctionMethodName)
        self.lexer.setColor(QColor(hC.redColor), self.lexer.Number)

    def setEditorProperties(self):
        # Editor customization
        self.editor.setUtf8(True)
        #
        self.editor.setSelectionToEol(True)
        self.editor.setAutoIndent(True)
        self.editor.setWrapMode(QsciScintilla.WrapCharacter)
        self.editor.setIndentationsUseTabs(True)
        self.editor.setTabWidth(4)
        self.editor.setIndentationGuides(True)
        self.editor.setCaretWidth(1)        
        self.editor.setWrapIndentMode(QsciScintilla.WrapIndentSame)
        
    def setMargins(self):
        # number margin
        marginNr= hM.MARGIN_0_NR
        self.editor.setMarginType(marginNr, QsciScintilla.NumberMargin)
        self.editor.setMarginWidth(marginNr, '00000000')       
        # symbol margin 1
        marginNr= 1
        self.editor.setMarginType(marginNr, QsciScintilla.SymbolMarginColor)
        self.editor.setMarginWidth(marginNr, "00")        
        self.editor.setMarginBackgroundColor(marginNr, QColor(hC.whiteColor))      

    def loadPythonScript(self):
        fileName, _ = QFileDialog.getOpenFileName(self, "Load py script", "", "Python Files (*.py )")
        if fileName:
            scriptFile= hFc.File()
            scriptFile.setFileProperties(fileName)
            self.pythonScript= scriptFile.read()
            #
            script= self.pythonScript
            self.editor.setText(script)
            

    def savePythonScript(self):
        bufLen= self.getEditorLen()
        text= self.getEditorText(bufLen)
        if text:
            fileName, _ = QFileDialog.getSaveFileName(self,'Save py script', "", 'Python Files (*.py )')
            if fileName:
                scriptFile= hFc.File()
                scriptFile.setFileProperties(fileName)
                self.pythonScript= scriptFile.write(text)

        
    def runPythonScript(self):
        bufLen= self.getEditorLen()
        text= self.getEditorText(bufLen)
        if text:
            try:
                exec(text)
            except:
                err= '*runPythonScript\n'+ str(sys.exc_info()[0])+ '\n'+ str(sys.exc_info()[1])+ '\n'+ str(sys.exc_info()[2])    
                print(err)

    def getEditorLen(self):
        SEND= self.editor.SendScintilla
        #
        lenBytes= SEND(self.editor.SCI_GETLENGTH)     # length in bytes
        bufLen= lenBytes
        return bufLen
 
    def getEditorText(self, bufLen):
        SEND= self.editor.SendScintilla
        buffer= bytearray(bufLen*' ',  'utf-8')
        SEND(self.editor.SCI_GETTEXTRANGE,  0,  bufLen,  buffer)
        text= buffer.decode('utf-8')
        text= re.sub('\x00', '', text) # strip null \x00  
        # buffer contains something like bytearray(b'riga1\r\nriga2\r\n\r\nriga3\r\n')
        # after decoding it, the following statements removes all \r instances from the
        # string in order to avoid not wanted blank line
        text= re.sub('\r', '',  text)
        return text
 
#------------------------------------------------------------------------------------
#   APPLICATION START
#------------------------------------------------------------------------------------
def start(bk):
    # crea interfaccia operatore con PyQt5
    app= QApplication(sys.argv)

    oTabConf_DOCX= hFc.File()
    oTabConf_CSV= hFc.File()
    oTabDocx_DOCX= hFc.File()
    oTabDocx_CSV= hFc.File()    
    
#    FILE_DOCX_EXE= hFc.File()
    oMenu= MenuWin(bk, oTabConf_DOCX, oTabConf_CSV,  oTabDocx_DOCX, oTabDocx_CSV) 
    
    app.exec_()     


if not hC.DEBUG:                
    #------------------------------------------------------------------------------------
    #   SIGIL MAIN LOOP    
    #------------------------------------------------------------------------------------
    def run(bk):  

        start(bk)
        # Setting the proper Return value is important.
        # 0 - means success
        # anything else means failure
        return 0
else:
    if __name__ == '__main__':
        bk= 'pippo'
        start(bk)
        
    

