#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QAbstractTableModel, QModelIndex,
                     QIcon, QImage, QBrush, pyqtSignal, QStyle)

from calibre import prints
from calibre.gui2 import error_dialog
from calibre.constants import DEBUG

from calibre_plugins.last_modified.common_utils import get_icon, custom_date_columns
import calibre_plugins.last_modified.config as cfg

from calibre.constants import numeric_version
if numeric_version >= (5,99,0):
    QT_CHECKED = Qt.CheckState.Checked.value
    QT_UNCHECKED = Qt.CheckState.Unchecked.value
else:
    QT_CHECKED = Qt.Checked
    QT_UNCHECKED = Qt.Unchecked

try:
    load_translations()
except NameError:
    prints("Last Modified::gui/models.py - exception when loading translations")

DOWN    = 1
UP      = -1

class MenusModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, gui, dates_config):
        QAbstractTableModel.__init__(self)
        self.dates_config = dates_config
        self.gui = gui
        self.db = gui.current_db
        self.col_map = ['active', 'date_column', 'settings', 'comments', 'errors']
        self.editable_columns = ['date_column','comments']
        self.hidden_cols = []
        self.optional_cols = []
        self.col_min_width = {
            'active': 10,
            'date_column': 200,
            'settings': 30,
            'comments': 300,
            'errors': 300
        }
        all_headers = ['', _('Date Column'), _('Settings'), _('comments'), _('errors')]
        self.headers = all_headers
        self.custom_date_columns = custom_date_columns(self.db)

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.dates_config)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.dates_config):
            return None
        date_config = self.dates_config[row]
        col_name = self.col_map[col]
        value = date_config.get(col_name, '')
        error = ''
        if col_name == 'errors':
            date_column = date_config['date_column']
            if date_column and ( date_column not in custom_date_columns(self.db) ):
                error = 'column "{}" is not available'.format(date_config['date_column'])            

        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name in ['active','settings']:
                pass
            elif col_name == 'errors':
                if error:
                    return error
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return QIcon(get_icon('dialog_error.png'))
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                return error

            elif col_name == 'comment':
                tooltip = date_config.get('comment')
                return tooltip

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                is_checked = date_config[col_name]
                state = QT_CHECKED if is_checked else QT_UNCHECKED
                return state

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        date_config = self.dates_config[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name in ['active','settings']:
                pass
            else:
                date_config[col_name] = val
            done = True

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                state = True if (value == QT_CHECKED) else False
                date_config[col_name] = state
            return True
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            date_config = self.dates_config[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
            if col_name == 'active':
                flags |=  Qt.ItemFlag.ItemIsUserCheckable
        return flags

    def button_state(self, index):
        visible = False
        enabled = True
        row, col = index.row(), index.column()
        date_config = self.dates_config[row]
        active = date_config.get('active')
        date_column = date_config.get('date_column')
        if date_column and active:
            visible = True
        return visible, enabled

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            date_config = {}
            date_config['active'] = True
            date_config['date_column'] = ''
            date_config['settings'] = {}
            date_config['errors'] = ''
            self.dates_config.insert(row + i, date_config)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.dates_config.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        for row in srows:
            pop = self.dates_config.pop(row)
            self.dates_config.insert(row+direction, pop)
        self.layoutChanged.emit()

    def validate(self):
        is_valid = True
        for row, date_config in enumerate(self.dates_config):
            date_column = date_config['date_column']
            if date_column and not (date_column in self.custom_date_columns):
                is_valid = False
        return is_valid
