#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import defaultdict
from functools import partial

from qt.core import Qt, QApplication, QToolButton

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import open_url, error_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.db.listeners import EventType
from calibre.utils.date import now

import calibre_plugins.last_modified.config as cfg
from calibre_plugins.last_modified.gui.dialogs import LastModifiedDialog
from calibre_plugins.last_modified.common_utils import (set_plugin_icon_resources, get_icon, KeyboardConfigDialog,
                                        create_menu_item, create_menu_action_unique, custom_date_columns)

try:
    load_translations()
except NameError:
    prints("Last Modified::action.py - exception when loading translations")


class LastModifiedHibernation(object):

    def __init__(self, plugin_action):
        self.plugin_action = plugin_action
        self.counter = 0

    def __enter__(self):
        if self.counter == 0:
            self.plugin_action.hibernation_cache = defaultdict(set)
            self.plugin_action.hibernation_mode = True
            if DEBUG:
                prints('Last Modified Plugin: Entering hibernation mode.')
        else:
            if DEBUG:
                prints('Last Modified Plugin: Hibernation mode is already on.')
        self.counter += 1
        return None

    def __exit__(self, *args):
        self.counter -= 1
        # if the counter > 0, means there are more hibernation jobs running
        if self.counter == 0:
            if DEBUG:
                prints('Last Modified Plugin: Exiting hibernation mode.')
            self.plugin_action.end_hibernation()
            self.plugin_action.hibernation_mode = False
        else:
            if DEBUG:
                prints('Last Modified Plugin: Some hibernation jobs still running. Not exiting.')

class LastModifiedAction(InterfaceAction):

    name = 'Last Modified'
    action_spec = ('Last Modified', None, None, None)
    popup_type = QToolButton.InstantPopup
    action_type = 'current'
    dont_add_to = frozenset(['context-menu-device'])

    def genesis(self):
        # Read the plugin icons and store for potential sharing with the config widget
        icon_names = ['images/'+i for i in cfg.PLUGIN_ICONS]
        icon_resources = self.load_resources(icon_names)
        set_plugin_icon_resources(self.name, icon_resources)

        # Assign our menu to this action and an icon
        self.qaction.setIcon(get_icon('images/'+cfg.PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.show_last_modified_dialog)

    def initialization_complete(self):
        self.gui.add_db_listener(self.event_listener)
        self.book_created_cache = {}
        self.book_created_cache_last_updated = None
        self.date_col_update_cache = {}
        self.date_col_update_cache_last_updated = None
        # Used by other plugin to temporarily turn on/off this plugin
        self.is_enabled = True
        #
        self.reset_hibernation_data()

    def reset_hibernation_data(self):
        self.hibernate = LastModifiedHibernation(self)
        self.hibernation_mode = False
        self.hibernation_cache = defaultdict(set)

    def library_changed(self, db):
        self.clean_book_created_cache_if_old(max_duration=0)
        self.clean_date_col_update_cache_if_old(max_duration=0)
        self.reset_hibernation_data()

    def sutting_down(self):
        self.clean_book_created_cache_if_old(max_duration=0)
        self.clean_date_col_update_cache_if_old(max_duration=0)
        self.end_hibernation()
        
    def event_lookup(self, val):
        if val == EventType.metadata_changed:
            return 'metadata_changed'
        elif val == EventType.format_added:
            return 'format_added'
        elif val == EventType.formats_removed:
            return 'formats_removed'
        elif val == EventType.items_renamed:
            return 'items_renamed'
        elif val == EventType.items_removed:
            return 'items_removed'
        elif val == EventType.book_edited:
            return 'book_edited'
        elif val == EventType.book_created:
            return 'book_created'

    def event_listener(self, db, event_type, event_data):
        if not self.is_enabled:
            return
        if not db.library_id == self.gui.current_db.library_id:
            return

        event_name = self.event_lookup(event_type)
        if not event_name:
            return
            
        elif event_name == 'metadata_changed':
            field, book_ids = event_data
            if field == 'cover':
                event_name = 'cover_modified'
        # Modify event_data
        elif event_name == 'format_added':
            book_id, fmt = event_data
            event_data = set([book_id]), fmt
        elif event_name == 'book_edited':
            book_id, fmt = event_data
            event_data = set([book_id]), fmt
        elif event_name == 'book_created':
            book_id = event_data[0]
            event_data = (set([book_id]),)

        self.update_all_date_columns(event_name, event_data, hibernation_mode=self.hibernation_mode)
                
    def get_date_columns_to_update(self, event_name, event_data):
        cols = set()
        library_config = cfg.get_library_config(self.gui.current_db)
        dates_config = library_config[cfg.KEY_LAST_MODIFIED_ENTRIES]
        for date_config in dates_config:
            active = date_config['active']
            date_column = date_config['date_column']
            settings = date_config['settings']
            if active and (date_column in custom_date_columns(self.gui.current_db)):
                if event_name == 'metadata_changed':
                    if settings.get(event_name, False):
                        field, book_ids = event_data
                        metadata_settings = settings.get('metadata_settings', {})
                        opt = metadata_settings.get('metadata_opt', 'all')
                        exclude_fields = ['last_modified']
                        if metadata_settings.get('exclude_date_cols', True):
                            exclude_fields += list(custom_date_columns(self.gui.current_db))
                        if opt == 'all':
                            if not field in exclude_fields:
                                cols.add(date_column)
                        elif opt == 'include':
                            include_fields = [ x.strip() for x in metadata_settings.get('include', '').split(',') ]
                            if field in include_fields:
                                if not field in exclude_fields:
                                    cols.add(date_column)
                        elif opt == 'exclude':
                            exclude_fields = exclude_fields + [ x.strip() for x in metadata_settings.get('exclude', '').split(',') ]
                            if not field in exclude_fields:
                                cols.add(date_column)
                else:
                    if settings.get(event_name, False):
                        cols.add(date_column)
        return cols   

    def update_all_date_columns(self, event_name, event_data, hibernation_mode=False):
        timestamp = now()
        cols = self.get_date_columns_to_update(event_name, event_data)
        if event_name in ['metadata_changed','cover_modified']:
            field, book_ids = event_data
        elif event_name == 'format_added':
            book_ids = event_data[0]
        elif event_name == 'formats_removed':
            book_ids = event_data[0].keys()      
        elif event_name == 'items_renamed':
            book_ids = event_data[1]    
        elif event_name == 'items_removed':
            book_ids = event_data[1]          
        elif event_name == 'book_edited':
            book_ids = event_data[0]
        elif event_name == 'book_created':
            book_ids = event_data[0]

        if not book_ids:
            return

        # We drop events in two cases
        # [1] Adding books fire a lot of db events (metada_changed for every, cover modified, format_added)
        #     This leads to ThreadViolation exception. To mitigage the situation, drop all these events if
        #     the book is recently created, and only proceed with the book_created event
        # [2] If date column is automatically updated by this plugin, we drop the event resulting from this
        # {
        self.clean_book_created_cache_if_old()
        self.clean_date_col_update_cache_if_old()

        if event_name == 'book_created':
            for book_id in book_ids:
                self.add_to_book_created_cache(book_id, timestamp)
        elif event_name in ['metadata_changed','cover_modified','format_added']:
            if len(book_ids) == 1:
                book_id = list(book_ids)[0]
                if self.is_book_recently_created(book_id):
                    return
            if event_name == 'metadata_changed':
                # If date column is updated manually by the user, remove it from cols to avoid infinite loop
                if field in cols:
                    cols = cols.difference(set([field]))
                # Drop metadata_changed event for automatically updated date columns
                if field in custom_date_columns(self.gui.current_db):
                    if self.is_date_col_recently_updated(book_ids, field):
                        return
                
        #}}}

        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            cache = self.gui.current_db.new_api
            for col in cols:
                self.update_date_column(cache, col, timestamp, book_ids, hibernation_mode)
            #FIXME: The commented out code hangs calibre when batch editing metadata
            #self.gui.library_view.model().refresh_ids(book_ids)
        except:
            import traceback
            traceback.print_exc()
        finally:
            QApplication.restoreOverrideCursor()

    def update_date_column(self, cache, col, timestamp, book_ids, hibernation_mode=False):
        if hibernation_mode:
            self.hibernation_cache[col] |= set(book_ids)
        else:
            id_map = {book_id: timestamp for book_id in book_ids}
            cache.set_field(col, id_map)
            if DEBUG:
                prints('Last Modified Plugin: Update column ({}) for book_ids: {}'.format(col, book_ids))
        self.add_to_date_col_update_cache(book_ids, col, timestamp)


    def add_to_book_created_cache(self, book_id, timestamp):
        self.book_created_cache[str(book_id)] = timestamp
        self.book_created_cache_last_updated = timestamp

    def clean_book_created_cache_if_old(self, max_duration=20):
        if self.book_created_cache_last_updated:
            duration = now() - self.book_created_cache_last_updated
            if duration.seconds > max_duration:
                self.book_created_cache = {}
                self.book_created_cache_last_updated = None

    def is_book_recently_created(self, book_id, max_duration=5):
        ctime = self.book_created_cache.get(str(book_id))
        if ctime:
            duration = now() - ctime
            if duration.seconds > max_duration:
                return False
            else:
                return True
        else:
            return False

    def add_to_date_col_update_cache(self, book_ids, date_col, timestamp):
        key = ','.join(sorted([str(book_id) for book_id in book_ids]))+'|'+date_col
        self.date_col_update_cache[key] = timestamp
        self.date_col_update_cache_last_updated = timestamp

    def clean_date_col_update_cache_if_old(self, max_duration=3):
        if self.date_col_update_cache_last_updated:
            duration = now() - self.date_col_update_cache_last_updated
            if duration.seconds > max_duration:
                self.date_col_update_cache = {}
                self.date_col_update_cache_last_updated = None

    def is_date_col_recently_updated(self, book_ids, date_col, max_duration=3):
        key = ','.join(sorted([str(book_id) for book_id in book_ids]))+'|'+date_col
        ctime = self.date_col_update_cache.get(key)
        if ctime:
            duration = now() - ctime
            if duration.seconds > max_duration:
                return False
            else:
                return True
        else:
            return False

    def show_last_modified_dialog(self):
        d = LastModifiedDialog(self.gui)
        if d.exec_():
            d.save_settings()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    # Hibernation mode: Allow plugins to put Last Modified to hibernation.
    # This happens if plugins are doing operations that emit a lot of consecutive db events.
    # Hibernatin mode is done by setting attribute self.hibernation_mode = True
    # When in hibernation, Last Modified will listen to and cache events without
    # acting on them. When it comes out of hibernation, it will set the cached
    # date fields {{{{{

    def end_hibernation(self):
        try:
            cache = self.gui.current_db.new_api
            timestamp = now()
            for col, book_ids in self.hibernation_cache.items():
                id_map = {book_id: timestamp for book_id in book_ids}
                cache.set_field(col, id_map)
                self.add_to_date_col_update_cache(book_ids, col, timestamp)
                if DEBUG:
                    prints('Last Modified Plugin: Update column ({}) for book_ids: {} (Hibernation cache)'.format(col, book_ids))
        finally:
            self.hibernation_mode = False
            # Reset cache
            self.hibernation_cache = defaultdict(set)
    #}}}}}
