#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
import copy

from qt.core import (Qt, QWidget, QVBoxLayout, QHBoxLayout, QGridLayout, QLabel, QPushButton,
                     QToolButton, QAction, QIcon, QSizePolicy, QSpacerItem, QFrame,
                     QGroupBox, QCheckBox, QLineEdit, QRadioButton, QToolButton)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, info_dialog, question_dialog
from calibre.gui2.widgets2 import Dialog

from calibre_plugins.last_modified.common_utils import get_icon, PrefsViewerDialog
from calibre_plugins.last_modified.gui.views import MenusTable
from calibre_plugins.last_modified.gui.models import MenusModel, UP, DOWN

import calibre_plugins.last_modified.config as cfg

try:
    load_translations()
except NameError:
    prints("Last Modified::gui/dialogs.py - exception when loading translations")

class FieldsDialog(Dialog):

    def __init__(self, gui, settings={}):
        self.gui = gui
        self.db = gui.current_db
        self.settings = settings
        Dialog.__init__(self, 'Fields Dialog', 'last-modified-fields-dialog', self.gui)

    def setup_ui(self):
        l = self.l = QVBoxLayout(self)
        self.setLayout(l)

        self.exclude_date_cols_chk = QCheckBox(_('Exclude all custom date columns'))
        self.exclude_date_cols_chk.setChecked(True)
        l.addWidget(self.exclude_date_cols_chk)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)
                
        fields_group_box = QGroupBox(_('Columns options'))
        fields_group_box_l = QVBoxLayout()
        fields_group_box.setLayout(fields_group_box_l)
        l.addWidget(fields_group_box)
        
        self.all_opt = QRadioButton(_('update when metadata changes for all columns'))
        self.all_opt.setChecked(True)
        fields_group_box_l.addWidget(self.all_opt)

        self.include_opt = QRadioButton(_('update when metadata changes only for columns specified (comma separated list)'))
        self.include_edit = QLineEdit()
        fields_group_box_l.addWidget(self.include_opt)
        fields_group_box_l.addWidget(self.include_edit)

        self.exclude_opt = QRadioButton(_('update when metadata changes for all columns except specified (comma separated list)'))
        self.exclude_edit = QLineEdit()
        fields_group_box_l.addWidget(self.exclude_opt)
        fields_group_box_l.addWidget(self.exclude_edit)

        l.addStretch(1)

        l.addWidget(self.bb)
        
        self.load_settings(self.settings)

    def load_settings(self, settings):
        if settings:
            self.exclude_date_cols_chk.setChecked(settings.get('exclude_date_cols', True))
            if settings['metadata_opt'] == 'all':
                self.all_opt.setChecked(True)
            elif settings['metadata_opt'] == 'include':
                self.include_opt.setChecked(True)
                self.include_edit.setText(settings['include'])
            elif settings['metadata_opt'] == 'exclude':
                self.exclude_opt.setChecked(True)
                self.exclude_edit.setText(settings['exclude'])

    def save_settings(self):
        settings = {}
        settings['exclude_date_cols'] = self.exclude_date_cols_chk.isChecked()
        if self.all_opt.isChecked():
            settings['metadata_opt'] = 'all'
        elif self.include_opt.isChecked():
            settings['metadata_opt'] = 'include'
            settings['include'] = self.include_edit.text()
        elif self.exclude_opt.isChecked():
            settings['metadata_opt'] = 'exclude'
            settings['exclude'] = self.exclude_edit.text()
        return settings

    def accept(self):
        self.settings = self.save_settings()
        Dialog.accept(self)

class SettingsDialog(Dialog):

    def __init__(self, parent, gui, date_config):
        self.gui = gui
        self.db = gui.current_db
        self.date_config = date_config
        self.metadata_settings = {}
        Dialog.__init__(self, _('Settings'), 'date-column-settings', parent)
        self.load_settings(date_config.get('settings', {}))

    def setup_ui(self):
        l = self.l = QVBoxLayout()
        self.setLayout(l)
                
        events_group_box = QGroupBox('Update on these events')
        events_group_box_l = QGridLayout()
        events_group_box.setLayout(events_group_box_l)
        l.addWidget(events_group_box)
        
        self.metadata_changed_chk = QCheckBox(_('Book Metadata Changed'))
        events_group_box_l.addWidget(self.metadata_changed_chk, 0, 0, 1, 1)
        self.metadata_settings_button = QToolButton()
        self.metadata_settings_button.setIcon(get_icon('gear.png'))
        events_group_box_l.addWidget(self.metadata_settings_button, 0, 1, 1, 1)
        self.format_added_chk = QCheckBox(_('Book Format Added'))
        events_group_box_l.addWidget(self.format_added_chk, 1, 0, 1, 1)
        self.formats_removed_chk = QCheckBox(_('Book Format(s) Removed'))
        events_group_box_l.addWidget(self.formats_removed_chk, 2, 0, 1, 1)
        self.cover_modified_chk = QCheckBox(_('Book Cover Modified'))
        events_group_box_l.addWidget(self.cover_modified_chk, 3, 0, 1, 1)
        self.item_renamed_chk = QCheckBox(_('Category Item Renamed'))
        events_group_box_l.addWidget(self.item_renamed_chk, 4, 0, 1, 1)
        self.item_removed_chk = QCheckBox(_('Category Item Removed'))
        events_group_box_l.addWidget(self.item_removed_chk, 5, 0, 1, 1)
        self.book_edited_chk = QCheckBox(_('Book Modified By Calibre Editor'))
        events_group_box_l.addWidget(self.book_edited_chk, 6, 0, 1, 1)
        self.book_created_chk = QCheckBox(_('Book Created'))
        events_group_box_l.addWidget(self.book_created_chk, 7, 0, 1, 1)

        self.metadata_settings_button.clicked.connect(self._on_metadata_button_clicked)

        l.addStretch(1)
        l.addWidget(self.bb)

    def load_settings(self, settings):
        if settings:
            self.metadata_changed_chk.setChecked(settings.get('metadata_changed', False))
            self.metadata_settings = settings.get('metadata_settings', {})
            self.format_added_chk.setChecked(settings.get('format_added', False))
            self.formats_removed_chk.setChecked(settings.get('formats_removed', False))
            self.cover_modified_chk.setChecked(settings.get('cover_modified', False))
            self.item_renamed_chk.setChecked(settings.get('items_renamed', False))
            self.item_removed_chk.setChecked(settings.get('items_removed', False))
            self.book_edited_chk.setChecked(settings.get('book_edited', False))
            self.book_created_chk.setChecked(settings.get('book_created', False))

    def save_settings(self):
        settings = {}
        settings['metadata_changed'] = self.metadata_changed_chk.isChecked()
        settings['format_added'] = self.format_added_chk.isChecked()
        settings['formats_removed'] = self.formats_removed_chk.isChecked()
        settings['cover_modified'] = self.cover_modified_chk.isChecked()
        settings['items_renamed'] = self.item_renamed_chk.isChecked()
        settings['items_removed'] = self.item_removed_chk.isChecked()
        settings['book_edited'] = self.book_edited_chk.isChecked()
        settings['book_created'] = self.book_created_chk.isChecked()
        settings['metadata_settings'] = self.metadata_settings
        return settings     

    def _on_metadata_button_clicked(self):
        d = FieldsDialog(self.gui, self.metadata_settings)
        if d.exec_() == d.Accepted:
            self.metadata_settings = d.settings
    
    def accept(self):
        self.settings = self.save_settings()
        Dialog.accept(self)

class LastModifiedDialog(Dialog):

    def __init__(self, gui):
        self.gui = gui
        self.db = gui.current_db
        self.library_config = cfg.get_library_config(self.db)
        self.dates_config = self.library_config.get(cfg.KEY_LAST_MODIFIED_ENTRIES, [])
        Dialog.__init__(self, 'Last Modified Dialog', 'last-modified-dialog', self.gui)

    def setup_ui(self):
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        heading_layout = QHBoxLayout()
        layout.addLayout(heading_layout)
        heading_label = QLabel(_('&Select and configure date columns:'), self)
        heading_layout.addWidget(heading_label)
        # Add hyperlink to a help url at the right.
        help_label = QLabel(_('<a href="https://www.mobileread.com/forums/showpost.php?p=4146122&postcount=1">Help</a>'), self)
        help_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
        help_label.setOpenExternalLinks(True)
        help_label.setAlignment(Qt.AlignRight)
        heading_layout.addWidget(help_label)

        # Add a horizontal layout containing the table and the buttons next to it
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        # Create a table the user can edit the data values in
        self._table = MenusTable(self)
        heading_label.setBuddy(self._table)
        table_layout.addWidget(self._table)
        
        # Create table model
        _model = MenusModel(self.gui, copy.deepcopy(self.dates_config))
        self._table.set_model(_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)

        # restore table state
        state = self.library_config[cfg.KEY_GPREFS][cfg.KEY_LAST_MODIFIED_TABLE_STATE]
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        table_layout.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        button_layout.addWidget(move_up_button)
        spacerItem = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add a new row'))
        add_button.setIcon(QIcon(I('plus.png')))
        button_layout.addWidget(add_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(QIcon(I('minus.png')))
        button_layout.addWidget(delete_button)
        spacerItem3 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)

        hl = QHBoxLayout()
        layout.addLayout(hl)

        hl.addStretch(1)
                
        view_prefs_button = QPushButton(_('&View library preferences...'), self)
        view_prefs_button.setToolTip(_(
                    'View data stored in the library database for this plugin'))
        view_prefs_button.clicked.connect(self._view_prefs)
        hl.addWidget(view_prefs_button)

        layout.addWidget(self.bb)

        self._on_table_selection_change()

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # save table state
        self.library_config[cfg.KEY_GPREFS][cfg.KEY_LAST_MODIFIED_TABLE_STATE] = self._table.get_state()
        cfg.set_library_config(self.db, self.library_config)

    def save_settings(self):
        self.save_table_state()

        self.library_config[cfg.KEY_LAST_MODIFIED_ENTRIES] = self._table.model().dates_config
        cfg.set_library_config(self.db, self.library_config)

    def _view_prefs(self):
        d = PrefsViewerDialog(self.gui, cfg.PREFS_NAMESPACE)
        d.exec_()

    def accept(self):
        is_valid = self._table.model().validate()
        if is_valid is not True:
            message = _('Some date columns are not currently available. '
                        'Are you sure you want to proceed with these settings anyway?')
            if not question_dialog(self, _('Are you sure?'), message, show_copy_button=False):
                return

        dates_config = self._table.model().dates_config

        # Remove error keys from config
        for date_config in dates_config:
            try:
                del date_config['errors']
            except:
                pass

        self.library_config[cfg.KEY_LAST_MODIFIED_ENTRIES] = dates_config
        cfg.set_library_config(self.db, self.library_config)
        
        self.save_table_state()
        Dialog.accept(self)

    def reject(self):
        self.save_table_state()
        Dialog.reject(self)
