#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright (c) 2019-2022 John Crew
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
for commercial and non-commercial purposes are permitted provided that the
following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

"""
This plugin will insert an image into an xhtml section and formats the section
using SVG.

The plugin determines the width and height of the selected image to set SVG
values to ensure the image is displayed correctly in the ebook.

Further modifications may be made to remove the flag importPILFlag since PIL is
supplied with Sigil. 
However, this is retained for standalone testing at present.
"""

global importPILFlag	#test this flag to check whether PIL loaded before using the class dlgImage

import tkinter as tk				#Needed for custom dialog box using tk. commands
import tkinter.ttk as ttk	#Needed for ttk. commands
import re 							#Needed the function natural_sort(l)
from tkinter import *				#Needed for root = Tk()
from tkinter import messagebox as tkMessageBox		#Needed for messagebox
import GenUtils						#Needed to centerWindow()
from UpdateChecker import CheckForUpdates, SetUpdateTimes
from io import BytesIO				#Needed to convert the image data returned by bk.readfile()
import ntpath						#Needed to extract filename from full path
import os
import subprocess 					#Needed to copy text to clipboard
import platform
import sys

from tkinter import filedialog		#Needed to open file dialog box
try:
	from sigil_bs4 import BeautifulSoup, Tag
except:
	from bs4 import BeautifulSoup, Tag
	
########################### DEBUG TEMP CODE BEGINS ###################

import inspect  #Needed for def lineno()

def lineno():
	#Useage: dbgPrint(lineno() + ....)
	LineNumber= inspect.currentframe().f_back.f_lineno
	return "AT LINE NUMBER "+str(LineNumber)+": "

dbgState=True

def dbgPrint(*args, **kwargs):
	if dbgState==True:
		outputStr=""
		for arg in args:
			if isinstance(arg, str):
				outputStr=outputStr+arg+" "
			elif isinstance(arg, int):
				outputStr=outputStr+" "+str(arg)
			else: # float
				outputStr=outputStr+" " +str(arg)
		print("DEBUG " + outputStr)

########################### DEBUG TEMP CODE ENDS ###################

try:
	from PIL import Image, ImageTk
except ImportError:
	print("Cannot import PIL library")
	importPILFlag=False
	print()
else:
	importPILFlag=True

class dlgInsertImagePage():
	dlgTop=""
	def __init__(self, parent, bk, DefaultFilePath, prefs):
		self.top = tk.Toplevel(parent)
		dlgInsertImagePage.dlgTop=self.top 		#Needed for external program to wait until this dialog window is destroyed
		self.bk1=bk
		self.pictureDict={}	#Clear dictionary to hold href and corresponding id for pictures UPDATED
		self.DefaultFilePath=DefaultFilePath
		self.prefs = prefs
		self.top.bind('<Escape>', self.EscapePressed)
		
		self.imageList = list()  #Create list to store the images in the ePub file so that it can be sorted

	def natural_sort(self, l): 	#borrowed from https://stackoverflow.com/questions/4836710/does-python-have-a-built-in-function-for-string-natural-sort
		convert = lambda text: int(text) if text.isdigit() else text.lower() 
		alphanum_key = lambda key: [ convert(c) for c in re.split('([0-9]+)', key) ] 
		return sorted(l, key = alphanum_key)
		
	def ShowInsertImageDlg(self):
		"""
		Called by an external module.
		Parent is the window that owns this dialog.
		bk is the parameter used in the Sigil run(bk) function.
		"""
		if self.bk1.launcher_version() >= 20160102:
			self.epubver = self.bk1.epub_version()
		else:
			self.epubver = BeautifulSoup(self.bk1.get_opf(), 'lxml').find('package')['version']
			
		self.InternalImageSelected=False		#Initialise flag for selecting image
										#Set to True only if an image is selected for changing
		self.ExternalImageLoaded = False		#Initialise flag for selecting a loaded image
										#Set to True only if an image is selected for loading
		self.Filename=""					#Name of external file that has been selected
		#self.top.grab_set() 				#Make dialog box modal
		self.top.title("Insert Image into ebook with SVG wrapping")	#Title for dialog box
		self.dlgframe = ttk.Frame(self.top, padding="15 15 12 12")
		self.dlgframe.grid(column=0, row=0, sticky=(N, W, E, S))
		self.dlgframe.columnconfigure(0, weight=1)
		self.dlgframe.rowconfigure(0, weight=1)

############################ Canvas ##########################################

		self.CanvasFrame=ttk.Frame(self.dlgframe, padding="5 5 5 5", relief=SUNKEN)
		self.CanvasFrame.grid(column=1, row=0, sticky=(N, W, E, S))
		self.CanvasFrame.columnconfigure(0, weight=1, minsize=400)
		self.CanvasFrame.rowconfigure(0, weight=1, minsize=300)
		self.canvas = Canvas(self.CanvasFrame)
		self.canvas.grid(column=0, row=0, sticky=(N,W,E,S))

############################# Left frame ####################################

		LeftFrame=ttk.Frame(self.dlgframe,padding="5 5 5 5", relief=SUNKEN)
		LeftFrame.grid(column=0, row=0, sticky=(N, W, E, S))
############################# Image List box and its label ##################

		tk.Label(LeftFrame, text="Select image:").grid(column=0, row=0, sticky=W)

		self.lbox = Listbox(LeftFrame, height=10, width=50, selectmode=SINGLE)
		self.lbox.grid(column=0, columnspan=3, row=1, sticky=(N,S,E,W))
		scroller = ttk.Scrollbar(LeftFrame, orient=VERTICAL, command=self.lbox.yview)
		scroller.grid(column=3, row=1, sticky=(N,S,W))
		self.lbox.configure(yscrollcommand=scroller.set)
		#Calls pictureChosen() when user double clicks a list box item
		#self.lbox.bind("<Double-Button-1>", self.pictureChosen)
		#Calls pictureChosen() when user single clicks a list box item	
		self.lbox.bind('<<ListboxSelect>>',self.pictureChosen)	
##UPDATED 07/07/2018===================================================================================
		for (id, href, mime) in self.bk1.image_iter():
			href=href.replace('Images/','') #Remove 'Images/'
			self.imageList.append(href)
			self.pictureDict[href] = id		#Store ids with picture name
		
		self.imageList=self.natural_sort(self.imageList) #Sort the list of images
		##Append the name and extension of each image in the list to the list box
		for item in self.imageList:
			self.lbox.insert(END, item)			
##END UPDATE ===========================================================================
		self.lbox.select_set(0) #Set focus on the first item in the listbox
		self.lbox.activate(0)
		self.lbox.event_generate("<<ListboxSelect>>")
		self.lbox.config(exportselection=False)  #Do not set to false before lbox.event_generate
		
############################# Combo box for insert position #################
		tk.Label(LeftFrame, text="").grid(column=0, row=2, sticky=W) # Blank row
		tk.Label(LeftFrame, text="Select position").grid(column=0, row=3, sticky=W)

		self.cbChoices=('At the start', 'At the end', 'After:', 'Before:')
		self.cbPosition=ttk.Combobox(LeftFrame, state="readonly", values=self.cbChoices,exportselection=0, width=15)
		self.cbPosition.grid(column=0, row=5,  sticky = (W))
		self.cbPosition.set('At the start')
		self.cbPosition.bind("<<ComboboxSelected>>", self.newPosSelected)
		
############################# Combo box for displaying xhtml files #################			
		tk.Label(LeftFrame, text="").grid(column=0, row=6, sticky=W) # Blank row
		self.xhtmlList=[]
		for (id, href) in self.bk1.text_iter():
			self.xhtmlList.append(href) 
			self.xhtmlList=self.natural_sort(self.xhtmlList)
		
		self.cbXhtmlChoices = StringVar()
		self.cbxhtml=ttk.Combobox(LeftFrame, state="readonly", textvariable=self.cbXhtmlChoices, exportselection=0, width=30)
		self.cbxhtml['values'] = self.xhtmlList
		self.cbxhtml.grid(column=0, columnspan=3, row=7,  sticky = (E, W))
		self.cbxhtml.current(0)	#Displays first item in the combobox
		self.cbxhtml.grid_forget()

############################# Labels and Buttons  ########################################
		#tk.Label(LeftFrame, text="").grid(column=0, row=8, sticky=W)	#Blank row
		tk.Button(LeftFrame, text="Import external image", command=self.LoadFilePressed, width = 20).grid(column=0, row=9, sticky=(E,W))
		tk.Button(LeftFrame, text="Default folder", command=self.SetDirectory, width = 20).grid(column=1, row=9, sticky=(E,W))
		tk.Label(LeftFrame, text="").grid(column=0, row=10, sticky=W)	#Blank row
		self.insertBtn = tk.Button(LeftFrame, text="Insert", command=self.Insert, width = 15)
		self.insertBtn.grid(column=0, row=30, sticky=(E,W))
		#Get standard background colour for button
		self.SysButtonColour = self.insertBtn.cget("background")
		self.insertBtn.configure(bg=self.SysButtonColour, fg='black')
		self.copyBtn = tk.Button(LeftFrame, text="Copy", command=self.writeToClipboard, width = 15)
		self.copyBtn.grid(column=1, row=30, sticky=(E, W))
		tk.Label(LeftFrame, text="").grid(column=0, row=35, sticky=W)	#Blank row
		tk.Button(LeftFrame, text="Close", command=self.Close, width = 15).grid(column=0, row=40, sticky=(E, W))

		GenUtils.centerWindow(self.top)

############################# Methods  ########################################

	def newPosSelected(self, event):
		#Called when the selection in cbPosition is changed
		if self.cbPosition.current()==0 or self.cbPosition.current()==1:
			#Hide the xhtml combo box
			self.cbxhtml.grid_forget()
		else:
			#Show the xhtml combo box
			self.cbxhtml.grid(column=0, columnspan=3, row=6,  sticky = (E, W))
## Dec 2021
	def CopyToClipboard(self, txt):
		"""
		Puts the text in txt on the clipboard
		Based on code from:
		https://www.programcreek.com/python/?CodeExample=copy+to+clipboard
		"""

		if platform.system() == 'Darwin':
			command = ["pbcopy"]
		elif platform.system() == 'Windows':
			command=["clip"]
		elif platform.system() == "Linux":
			command = ["xsel", "-b", "-i"]
		
		env = dict(os.environ).copy()
		encoding = "utf-8"
		env["PYTHONIOENCODING"] = encoding
		
		if sys.version_info >= (3, 6):
			extra = {"encoding": encoding}
		else:
			extra = {}
		
		proc = subprocess.Popen(
			command,
			stdin=subprocess.PIPE,
			shell=False,
			env=env,
			universal_newlines=True,
			close_fds=True,
			**extra
		)
		proc.communicate(input=txt, timeout=0.1)

	def writeToClipboard(self):
		"""
		This method is called when the user clicks the Copy button.
		If an image in the ePub has been selected it will copy it to the clipboard with SVG tags.
		Alternatively if an image has been selected for import by clicking the import button
		then this method will import that image and then copy it to the clipboard with SVG tags.
		The user can then paste the image and SVG tags into any part of their ePub
		"""
		if self.ExternalImageLoaded == True:
			self.ImportImageFile() #resets self.ExternalImageLoaded to False if invalid file
		if (not self.InternalImageSelected) and (not self.ExternalImageLoaded):
			tkMessageBox.showerror(title='WARNING', message='AAAYou must select an image from the list\nor load a file\nor press Close', icon='warning')
			return
		if (self.ExternalImageLoaded == True) or (self.InternalImageSelected == True):
		
			width, height = self.img.size
			strWidth=str(width)
			strHeight=str(height)
				
			xml = '<div>\n'
			xml += '	<svg \n'
			xml += '	xmlns="http://www.w3.org/2000/svg"\n'
			xml += '	height="100%"  width="100%"\n'
			xml += '	preserveAspectRatio="xMidYMid meet" version="1.1"\n'
			xml += '	viewBox="0 0 ' + strWidth + ' ' + strHeight + '"\n' 
			xml += '	xmlns:xlink="http://www.w3.org/1999/xlink">\n'
			xml += '	<image width="' + strWidth + '" height="' + strHeight + '" xlink:href="../Images/' + self.pictureName + '"/>\n'
			xml += '	</svg>\n'
			xml += '</div>\n'
			
			self.CopyToClipboard(xml)
			
			tkMessageBox.showinfo(title='COPIED', message='Your image has been copied to the clipboard')
			
			self.img.close
			
			self.top.destroy()

	def SetDirectory(self):
		dirname = filedialog.askdirectory(parent=self.top, initialdir="/",title='Please select a directory')
		if len(dirname ) > 0:
			self.DefaultFilePath = dirname
			self.prefs['DefaultFilePath']  = self.DefaultFilePath
			bk1.savePrefs(self.prefs)

	def pictureChosen(self, event):
		"""
		This method is called when the user clicks the name of a picture in the
		list box. It loads the picture and displays it
		"""
		self.ExternalImageLoaded = False
		#widget references the widget that sent the (click) event; saves using self.lbox
		widget = event.widget	
		#get index number of the selected picture in listbox
		selectedPicture=widget.curselection()
		#get text for selected picture from listbox
		self.pictureName = widget.get(selectedPicture[0])
		bkImage = self.bk1.readfile(self.pictureDict[self.pictureName])	#Look up ID for picture name in dictionary and read into bkImagen
		img_temp = Image.open(BytesIO(bkImage))	#Replaced the above with code from DiapDealer
		self.img = img_temp.copy()
		size = 400, 400
		img_temp.thumbnail(size, Image.LANCZOS)
		self.photo = ImageTk.PhotoImage(img_temp)
		
		canvasImageId = self.canvas.create_image(0, 0, anchor=NW, image=self.photo)
		#indicate an image has been selected from the ePub images
		self.InternalImageSelected=True	

	def LoadFilePressed(self):
		"""
		This method is called when the user clicks the button to import an
		external file
		It displays a file dialog to allow the user to select an image to
		load, opens the file and displays the image. It does not put the image
		into the ebook.
		"""
		FILEOPENOPTIONS = dict(title='Choose an image file', initialdir=self.DefaultFilePath, initialfile = '', filetypes=[('Image files', ('.bm', '.bmp', '.gif', '.jpeg', '.jpg', '.png', '.tiff', '.tif')), ('All files', ('*.*'))])
		filehandle = filedialog.askopenfile(mode='rb', **FILEOPENOPTIONS)
		if filehandle != None:
			self.Filename=filehandle.name
			img_temp = Image.open(filehandle.name)
			self.img = img_temp.copy()
			size = 400, 400
			img_temp.thumbnail(size, Image.LANCZOS)
			self.photo = ImageTk.PhotoImage(img_temp)	
					
			canvasImageId = self.canvas.create_image(0, 0, anchor=NW, image=self.photo)
			#indicate an image has been loaded from an external file
			self.ExternalImageLoaded = True 
			self.insertBtn.configure(bg='#000000', fg='#b7f731')
			self.copyBtn.configure(bg='#000000', fg='#b7f731')
			#Remove highlight from currently selected item in the listbox
			self.lbox.selection_clear(0, END )
		else: #print the name of the file if it cannot be opened
			print("Cannot open file: ", filehandle.name)
			self.insertBtn.configure(bg=self.SysButtonColour, fg='black')
			
	def ImportImageFile(self):
		"""
		This method is called by the Insert() method
		It imports the external file that has been selected into the ePub,
		putting the image in the Images section of the ePub
		"""
		self.InternalImageSelected=False
		try:
			f=open(self.Filename, 'rb')	#open externally selected image file...
		except IOError:
			print("Cannot open ", self.Filename)
		else:
			#Use the file name as the base for the uid etc
			self.pictureName=ntpath.basename(self.Filename)
			self.pictureName=self.pictureName.replace(" ", "_")
			#extract filename and extension from path
			fName = self.pictureName[:self.pictureName.index(".")]
			ext= self.pictureName[self.pictureName.index(".")+len("."):]
			ext=ext.lower()	#convert extension to lower case
			if ext == "jpg" : ext="jpeg"
			if ext == "bm" : ext="bmp"
			if ext == "tif" : ext="tiff"
			if ext not in['bm', 'bmp', 'gif', 'jpeg', 'png', 'tiff']:
				print(ext + " is an invalid file extension")
				self.ExternalImageLoaded = False
				return

			bkImage=f.read()	#Load image into memory...
			f.close()
			uid= self.pictureName
			basename  = self.pictureName
			mime ="image/" + ext
			#Now check whether the image file already exists in the ePub
			for (id, href, mime) in self.bk1.image_iter():
				if (id==uid):
					reply=tkMessageBox.askquestion("WARNING", "This image is already present in the ePub.\nDo you want to overwrite it?")
					if reply=="no":
						self.ExternalImageLoaded = False
						return
					else:
						self.bk1.deletefile(id)
			self.bk1.addfile(uid, basename, bkImage, mime)

	def EscapePressed(self, event):
		'''Called when Escape key is pressed'''
		self.Close()
		
	def Close(self):
		"""
		Called when the user clicks the "Close" button
		It destroys the dialog window
		"""
		self.top.destroy()

	def InsertSVGPage(self):
		"""
		Called by Insert(self)
		It inserts the image in the ePub file in a new xhtml section
		It checks whether the <name>.xhtml is already present in the eBook;
		if so, it deletes the section.
		It then determines the width and height of the selected image.
		Then constructs the xhtml text for the new page, incorporating the 
		values for the width and height of the selected image, setting 
		preserveAspectRatio to xMidYMid
		"""

		pName = self.pictureName
		#todo If there is not a "." in pName then the next line will hang.

		if pName.find(".") !=-1:	#If extension is present...
			pName = pName[:pName.index(".")] #..then remove it
			
		width, height = self.img.size
		strWidth=str(width)
		strHeight=str(height)

		
		for (id, href) in self.bk1.text_iter():
			if id==pName or href=='Text/SVG_'+pName+ '.xhtml':	#If the section already exists
				reply=tkMessageBox.askquestion("WARNING", "Do you want to delete the current page named "+ pName+".xhtml?")
				if reply=="yes":
					self.bk1.deletefile(id)		#then delete it
				else:
					print("Present xhtml page has been retained.")
					return
		
		print ("EPUB Version: " + self.epubver)
		xml  = '<?xml version="1.0" encoding="UTF-8" standalone="no" ?>\n'
		if self.epubver.startswith('2'):
			xml += '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n'
		else:
			xml += '<!DOCTYPE html>\n'

		xml += '<html xmlns="http://www.w3.org/1999/xhtml">\n'
		xml += '<head>\n'
		xml += '  <title></title>\n'
		xml += '<style type="text/css">\n'
		xml += '@page {padding: 0; margin:0}\n'
		xml += 'body  {text-align: center; padding:0; margin: 0; }\n'
		xml += '</style>\n'  
		xml += '</head>\n'
		xml += '\n'
		xml += '<body>\n'
		xml += '  <div>\n'
		xml += '	<svg \n'
		xml += '	xmlns="http://www.w3.org/2000/svg"\n'
		xml += '	height="100%"  width="100%"\n'
		xml += '	preserveAspectRatio="xMidYMid meet" version="1.1"\n' 
		xml += '	viewBox="0 0 ' + strWidth + ' ' + strHeight + '"\n' 
		xml += '	xmlns:xlink="http://www.w3.org/1999/xlink">\n'
		xml += '	<image width="' + strWidth + '" height="' + strHeight + '" xlink:href="../Images/' + self.pictureName + '"/>\n'
		xml += '	</svg>\n'
		xml += '  </div>\n'
		xml += '</body>\n'
		xml += '</html>\n'   			
	
		uid	   = 'SVG_'+pName
		basename  = 'SVG_'+pName+'.xhtml'
		mime = 'application/xhtml+xml'
		
		if self.epubver.startswith('2'):
			self.bk1.addfile(uid, basename, xml, mime)	#Add new xhtml section to the epub2 eBook
		else:
			self.bk1.addfile(uid, basename, xml, mime, 'svg')	#Add new xhtml section to the epub3 eBook

		#values for cbPosition are ('At the start', 'At the end', 'After:', 'Before:')
		insertPos=self.cbPosition.current()
		#Get index of item selected in the xhtml combo box
		indexBeforeAfter=self.cbxhtml.current()
		if insertPos == 0:	#The position 'At the start' is selected
			location = 0
		elif insertPos == 1:	#The position 'At the end' is selected
			location = -1
		elif insertPos == 2:	#The position 'After:' is selected
			location = indexBeforeAfter+1
		elif insertPos == 3:	#The position 'Before:' is selected
			location = indexBeforeAfter
			
		#If  location=0 then inserts at start; location=-1 appends to the end
		self.bk1.spine_insert_before(location, uid, None, None)

		"""
		self.cbXhtmlList contains a list of images for combobox cbXhtm
		Insert image in appropriate place in list and update combobox cbXhtm
		"""		
		
		#If image is to be inserted at the end of the ePub's text section (ie -1) ...
		if location == -1:	
			#then set location to the number of items in the ePub's text section
			location = len(self.cbXhtmlList)
			
		#Update the list box
		#Remove highlight from currently selected item in the listbox
		self.lbox.selection_clear(0, END )
	
		self.lbox.delete(0,END) #Empty the list box
		#If an external image is selected
		if self.ExternalImageLoaded == True:
			#insert the name of the image at the end of the listbox
			#self.lbox.insert(END, self.pictureName)
			self.imageList.append(self.pictureName)

		
		self.imageList=self.natural_sort(self.imageList) #Sort the list of images for the listbox
		##Append the name and extension of each image in the list to the list box
		for item in self.imageList:
			self.lbox.insert(END, item)	
			
		#get the index of this image in the listbox
		i = self.lbox.get(0, END).index(self.pictureName)
		#and highlight the new image in the listbox
		self.lbox.selection_set(i)
		self.lbox.activate(i)
	
#Update the combobox
		
		#if an image in the ePub has been selected
		if self.InternalImageSelected==True:
			#delete its name from the combo box cbxhtml list otherwise
			#a duplicate name appears in the combobox
			try:
				self.cbXhtmlList.remove(basename)
			#If item is not in list then ignore the exception generated
			except:
				pass
		
		#Insert the name of SVG file in the list for the combo box cbxhtml...
		self.cbXhtmlList.insert(location, basename)

		#...and insert the new SVG file name in the cbxhtml combo box list		
		self.cbXhtmlList=self.natural_sort(self.cbXhtmlList) #Sort the list of images
		self.cbxhtml.selection_clear(0, END )
		self.cbxhtml['values'] = self.cbXhtmlList

		#Restore the Insert button to its normal colour
		self.insertBtn.configure(bg=self.SysButtonColour, fg='black')
		
	def Insert(self):
		"""
		Called when the user clicks the "Insert" button
		This uses the flag self.InternalImageSelected to check whether an existing image
		has been selected and self.ExternalImageLoaded to check whether an external
		image has been loaded.
		If not, it shows a warning message.
		If so, it adds a SVG page to the ePub, closes the image and destroys
		the dialog box.
		"""
		if (not self.InternalImageSelected) and (not self.ExternalImageLoaded):
			tkMessageBox.showerror(title='WARNING', message='You must select an image from the list\nor load a file\nor press Close', icon='warning')
			return
			
		self.insertBtn.configure(bg=self.SysButtonColour, fg='black')
		self.copyBtn.configure(bg=self.SysButtonColour, fg='black')
			
		if self.ExternalImageLoaded == True:
			self.ImportImageFile() #resets self.ExternalImageLoaded to False if invalid file
		if (self.ExternalImageLoaded == True) or (self.InternalImageSelected == True):
			self.InsertSVGPage()
		else:
			print("Cannot load xhtml file")
		self.img.close

def run(bk):
	root = tk.Tk()
	root.withdraw()

	global importPILFlag
	if importPILFlag:

		prefs = bk.getPrefs()
		#prefs.defaults['DefaultFilePath'] = 'D:\Media\My Pictures\Pictures'
		prefs.defaults['DefaultFilePath'] = 'C:'
			
		#Specify the url for the web page that contains the plugin eg:
		url = "https://www.mobileread.com/forums/showthread.php?t=283333"
		CheckForUpdates(root, prefs, url)
			
		imageDialog = dlgInsertImagePage(root, bk, prefs['DefaultFilePath'], prefs)
		imageDialog.ShowInsertImageDlg()
		root.wait_window(dlgInsertImagePage.dlgTop)
		root.destroy()
		bk.savePrefs(prefs)
	else:
		print("PIL not found")
	
	root.mainloop()

	return (0)

def main():
	print ("I reached main when I should not have\n")
	return -1
	
if __name__ == "__main__":
	sys.exit(main())