#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
import sys, re, os
from sigil_bs4 import BeautifulSoup, Tag
from quickparser import QuickXHTMLParser
from compatibility_utils import unquoteurl
from plugin_utils import Qt, QtCore, QtGui, QtWidgets
from plugin_utils import loadUi, PluginApplication, iswindows

# required for loading the .ui and .qm files from the plugin folder
plugin_dir = os.path.dirname(os.path.abspath(__file__))

#decode xml encoded strings (from navprocessor.py)
def xmldecode(data):
    if data is None:
        return ''
    newdata = data
    newdata = newdata.replace('&quot;', '"')
    newdata = newdata.replace('&gt;', '>')
    newdata = newdata.replace('&lt;', '<')
    newdata = newdata.replace('&amp;', '&')
    return newdata

# returns landmarks (from navprocessor.py)
def getLandmarks(navdata):
    ''' returns ordered list of tuples (epubtype, href, title)
    href is unquoted (percent encodings removed)
    title has been xml decoded/unescaped '''
    landmarks = []
    
    qp = QuickXHTMLParser()
    qp.setContent(navdata)
    title = ""
    nav_type = None
    href = None
    epubtype = None
    for txt, tp, tname, ttype, tattr in qp.parse_iter():
        if txt is not None:
            if ".a." in tp or tp.endswith(".a"):
                title = title + txt
            else:
                title = ""
        else:
            if tname == "nav" and ttype == "begin":
                nav_type = tattr.get("epub:type", None)
                continue
            if tname == "nav" and ttype == "end":
                nav_type = None
                continue

            if nav_type is not None and nav_type == "landmarks": 
                if tname == "a" and ttype == "begin":
                    href = tattr.get("href", "")
                    href = unquoteurl(href)
                    epubtype = tattr.get("epub:type", None)
                    continue
                if tname == "a" and ttype == "end":
                    if epubtype is not None:
                        title = xmldecode(title)
                        landmarks.append((epubtype, href, title))
                    title = ""
                    epubtype = None
                    href=None
                    continue
    return landmarks


# taken from KevinH's epub3tizer plugin
_guide_epubtype_map = {
     'acknowledgements'   : 'acknowledgments',
     'other.afterword'    : 'afterword',
     'other.appendix'     : 'appendix',
     'other.backmatter'   : 'backmatter',
     'bibliography'       : 'bibliography',
     'text'               : 'bodymatter',
     'other.chapter'      : 'chapter',
     'colophon'           : 'colophon',
     'other.conclusion'   : 'conclusion',
     'other.contributors' : 'contributors',
     'copyright-page'     : 'copyright-page',
     'copyright'     : 'copyright-page',
     'cover'              : 'cover',
     'dedication'         : 'dedication',
     'other.division'     : 'division',
     'epigraph'           : 'epigraph',
     'other.epilogue'     : 'epilogue',
     'other.errata'       : 'errata',
     'other.footnotes'    : 'footnotes',
     'foreword'           : 'foreword',
     'other.frontmatter'  : 'frontmatter',
     'glossary'           : 'glossary',
     'other.halftitlepage': 'halftitlepage',
     'other.imprint'      : 'imprint',
     'other.imprimatur'   : 'imprimatur',
     'index'              : 'index',
     'other.introduction' : 'introduction',
     'other.landmarks'    : 'landmarks',
     'other.loa'          : 'loa',
     'loi'                : 'loi',
     'lot'                : 'lot',
     'other.lov'          : 'lov',
     'notes'              : '',
     'other.notice'       : 'notice',
     'other.other-credits': 'other-credits',
     'other.part'         : 'part',
     'other.preamble'     : 'preamble',
     'preface'            : 'preface',
     'other.prologue'     : 'prologue',
     'other.rearnotes'    : 'rearnotes',
     'other.subchapter'   : 'subchapter',
     'title-page'         : 'titlepage',
     'toc'                : 'toc',
     'other.volume'       : 'volume',
     'other.warning'      : 'warning'
}


epubtype_title_map = {
'acknowledgments'        : 'Acknowledgments',
'afterword'        : 'Afterword',
'annotation'        : 'Annotation',
'appendix'        : 'Appendix',
'assessment'        : 'Assessment',
'backmatter'        : 'Back Matter',
'bibliography'        : 'Bibliography',
'bodymatter'        : 'Body Matter',
'chapter'        : 'Chapter',
'colophon'        : 'Colophon',
'conclusion'        : 'Conclusion',
'contributors'        : 'Contributors',
'copyright-page'        : 'Copyright Page',
'cover'        : 'Cover',
'dedication'        : 'Dedication',
'division'        : 'Division',
'epigraph'        : 'Epigraph',
'epilogue'        : 'Epilogue',
'errata'        : 'Errata',
'footnotes'        : 'Footnotes',
'foreword'        : 'Foreword',
'frontmatter'        : 'Front Matter',
'glossary'        : 'Glossary',
'halftitlepage'        : 'Half Title Page',
'imprimatur'        : 'Imprimatur',
'imprint'        : 'Imprint',
'index'        : 'Index',
'introduction'        : 'Introduction',
'landmarks'        : 'Landmarks',
'loa'        : 'List of Audio Clips',
'loi'        : 'List of Illustrations',
'lot'        : 'List of Tables',
'lov'        : 'List of Video Clips',
'notice'        : 'Notice',
'other-credits'        : 'Other Credits',
'page-list'        : 'Page List',
'part'        : 'Part',
'preamble'        : 'Preamble',
'preface'        : 'Preface',
'prologue'        : 'Prologue',
'qna'        : 'Questions and Answers',
'rearnotes'        : 'Rear Notes',
'revision-history'        : 'Revision History',
'subchapter'        : 'Subchapter',
'titlepage'        : 'Title Page',
'toc'        : 'Table of Contents',
'volume'        : 'Volume',
'warning'        : 'Warning'
}


# credit: DiapDealer 
# https://www.mobileread.com/forums/showpost.php?p=3575920&postcount=229        
def find_python3lib_dir(appdir):
    extralibdir = None
    # Determine python3lib path per platform
    if sys.platform.startswith('darwin'):
        extralibdir = os.path.join(appdir, '../python3lib')
    elif sys.platform.startswith('win'):
        extralibdir = os.path.join(appdir, 'python3lib')
    else:
        for path in sys.path:
            if 'plugin_launchers/python' in path:
                extralibdir = os.path.join(path, '../../python3lib')
                break
    return extralibdir


# define GUI
class GUI(QtWidgets.QDialog):

    def __init__(self, bk, landmarks, guide_items, cover_image_props_id, cover_image_meta_id, dirty):

        # get variables
        self.bk = bk
        self.landmarks = landmarks
        self.guide_items = guide_items
        self.cover_image_props_id = cover_image_props_id
        self.cover_image_meta_id = cover_image_meta_id
        self.dirty = dirty

        # initialize class
        super(GUI, self).__init__()

        # load gui file
        loadUi(os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'main.ui'), self)

        # hide "What is this?" icon [?]
        #self.setWindowFlags(Qt.WindowType.WindowCloseButtonHint)

        # define connects (A “Close” button defined with the RejectRole)
        self.buttonBox.rejected.connect(self.close)

        # add change asterisk, if files were changed
        if dirty:
            self.setWindowTitle('Semantics*')

        #==================
        # define table
        #==================

        # define header
        if landmarks != {}:
            self.header = ['Name', 'Landmark', 'Title', 'Guide', 'Title']
        else:
            self.header = ['Name', 'Guide', 'Title']

        # define table model
        self.QSModel = QtGui.QStandardItemModel()
        if landmarks != {}:
            self.QSModel.setColumnCount(5)
        else:
            self.QSModel.setColumnCount(3)
        self.QSModel.setHorizontalHeaderLabels(self.header)
        row = self.QSModel.rowCount()

        # define table view
        self.tableView.setModel(self.QSModel)
        try:
            self.tableView.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeMode.ResizeToContents)
            self.tableView.verticalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeMode.ResizeToContents)
        except:
            self.tableView.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
            self.tableView.verticalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)

        #==================
        # populate table
        #==================

        # epub3 files
        if landmarks != {}:

            #----------------------------------------------------------------------------------
            # add landmarks and equivalent guide items to the table
            #----------------------------------------------------------------------------------
            print('\n\nName\tLandmark\tTitle\tGuide\tTitle')
            for landmark in landmarks:
                epub_type, title = landmarks[landmark]
                row = self.QSModel.rowCount()
                self.QSModel.insertRow(row)
                self.QSModel.setData(self.QSModel.index(row, 0), landmark)
                self.QSModel.setData(self.QSModel.index(row, 1), epub_type)
                self.QSModel.setData(self.QSModel.index(row, 2), title)
                if landmark in guide_items:
                    guide_type, guide_title = guide_items[landmark]
                    self.QSModel.setData(self.QSModel.index(row, 3), guide_type)
                    self.QSModel.setData(self.QSModel.index(row, 4), guide_title)
                    print('{}\t{}\t{}\t{}\t{}'.format(landmark, epub_type, title, guide_type, guide_title))
                else:
                    print('{}\t{}\t{}\t\t'.format(landmark, epub_type, title))

            #--------------------------------------------------------------------
            # add guide items without equivalent landmarks
            #--------------------------------------------------------------------
            for guide_item in guide_items:
                if guide_item not in landmarks:
                    guide_type, guide_title = guide_items[guide_item]
                    row = self.QSModel.rowCount()
                    self.QSModel.insertRow(row)
                    self.QSModel.setData(self.QSModel.index(row, 0), guide_item)
                    self.QSModel.setData(self.QSModel.index(row, 3), guide_type)
                    self.QSModel.setData(self.QSModel.index(row, 4), guide_title)
                    print('{}\t{}\t{}\t{}\t{}'.format(guide_item, '', '', guide_type, guide_title))

            #----------------------------------------------------------
            # add cover image information to table
            #----------------------------------------------------------
            if cover_image_props_id:
                cover_image_name = os.path.basename(bk.id_to_href(cover_image_props_id))
                row = self.QSModel.rowCount()
                self.QSModel.insertRow(row)
                self.QSModel.setData(self.QSModel.index(row, 0), cover_image_name)
                self.QSModel.setData(self.QSModel.index(row, 1), 'cover-image')
                self.QSModel.setData(self.QSModel.index(row, 2), 'Cover image')
                if cover_image_meta_id:
                    self.QSModel.setData(self.QSModel.index(row, 3), 'cover')
                    self.QSModel.setData(self.QSModel.index(row, 4), 'Cover image')
                    print('{}\t{}\t{}\t{}\t{}'.format(cover_image_name, 'cover-image', 'Cover image', 'cover', 'Cover image'))
                else:
                    print('{}\t{}\t{}\t\t'.format(cover_image_name, 'cover-image', 'Cover image'))

            # only EPUB2 cover image found
            if not cover_image_props_id and cover_image_meta_id:
                cover_image_name = os.path.basename(bk.id_to_href(cover_image_meta_id))
                row = self.QSModel.rowCount()
                self.QSModel.insertRow(row)
                self.QSModel.setData(self.QSModel.index(row, 0), cover_image_name)
                self.QSModel.setData(self.QSModel.index(row, 3), 'cover')
                self.QSModel.setData(self.QSModel.index(row, 4), 'Cover image')
                print('{}\t\t\t{}\t{}'.format(cover_image_name, 'cover-image', 'Cover image'))

        else:
            # epub2 files
            if guide_items != {}:

                #-----------------------------------------
                # add guide items to table
                #-----------------------------------------
                print('\n\nName\tGuide\tTitle')
                for guide_item in guide_items:
                    guide_type, guide_title = guide_items[guide_item]
                    row = self.QSModel.rowCount()
                    self.QSModel.insertRow(row)
                    self.QSModel.setData(self.QSModel.index(row, 0), guide_item)
                    self.QSModel.setData(self.QSModel.index(row, 1), guide_type)
                    self.QSModel.setData(self.QSModel.index(row, 2), guide_title)
                    print('{}\t{}\t{}'.format(guide_item, guide_type, guide_title))

                #-------------------------------------------------------
                # add cover image information to table
                #-------------------------------------------------------
                if cover_image_meta_id is not None:
                    cover_image_name = os.path.basename(bk.id_to_href(cover_image_meta_id))
                    row = self.QSModel.rowCount()
                    self.QSModel.insertRow(row)
                    self.QSModel.setData(self.QSModel.index(row, 0), cover_image_name)
                    self.QSModel.setData(self.QSModel.index(row, 1), 'cover')
                    self.QSModel.setData(self.QSModel.index(row, 2), 'Cover image')
                    print('{}\t{}\t{}'.format(cover_image_name, 'cover', 'Cover image'))

        # show dialog box
        self.show()

# # dark mode support 
# def dark_palette(bk, app):
    # supports_theming = (bk.launcher_version() >= 20200117)
    # if not supports_theming:
        # return
    # if bk.colorMode() != "dark":
        # return
    # try:
        # from PyQt6.QtGui import QColor, QPalette
        # from PyQt6.QtWidgets import QStyleFactory
    # except ModuleNotFoundError:
        # from PyQt5.QtGui import QColor, QPalette
        # from PyQt5.QtWidgets import QStyleFactory
    # except ImportError:
        # return

    # p = QPalette()
    # sigil_colors = bk.color
    # dark_color = QColor(sigil_colors("Window"))
    # disabled_color = QColor(127,127,127)
    # dark_link_color = QColor(108, 180, 238)
    # text_color = QColor(sigil_colors("Text"))
    # p.setColor(p.Window, dark_color)
    # p.setColor(p.WindowText, text_color)
    # p.setColor(p.Base, QColor(sigil_colors("Base")))
    # p.setColor(p.AlternateBase, dark_color)
    # p.setColor(p.ToolTipBase, dark_color)
    # p.setColor(p.ToolTipText, text_color)
    # p.setColor(p.Text, text_color)
    # p.setColor(p.Disabled, p.Text, disabled_color)
    # p.setColor(p.Button, dark_color)
    # p.setColor(p.ButtonText, text_color)
    # p.setColor(p.Disabled, p.ButtonText, disabled_color)
    # p.setColor(p.BrightText, Qt.red)
    # p.setColor(p.Link, dark_link_color)
    # p.setColor(p.Highlight, QColor(sigil_colors("Highlight")))
    # p.setColor(p.HighlightedText, QColor(sigil_colors("HighlightedText")))
    # p.setColor(p.Disabled, p.HighlightedText, disabled_color)

    # app.setStyle(QStyleFactory.create("Fusion"))
    # app.setPalette(p)

def run(bk):

    #---------------------------------------------------------------------
    # get epub version (requires Sigil-0.9.3 or higher)
    #---------------------------------------------------------------------
    try:
        epubversion = bk.epub_version()
    except:
        print('This plugin requires at least Sigil 0.9.3.\nClick OK to close.')
        return -1

    #=============================
    # custom functions
    #=============================
    
    #----------------------------------------------------------------------------------
    # required by generateNCX() and generateGuideEntries()
    #----------------------------------------------------------------------------------
    python3lib = find_python3lib_dir(bk._w.appdir)

    #-------------------------------------
    # get the cover id
    #-------------------------------------
    def get_cover_id():
        ''' returns the EUB2 cover image id '''
        ps = bk.qp
        ps.setContent(bk.getmetadataxml())
        res = []
        coverid = None
        for text, tagprefix, tagname, tagtype, tagattr in ps.parse_iter():
            if text is None:
                if tagname == "meta" and tagattr.get("name",'') == "cover":
                    coverid = tagattr["content"]
                    break
        return(coverid)

    #-----------------------------------
    # get cover fallback id
    #-----------------------------------
    cover_fallback_id = None
    def get_cover_fallback_id(cover_landmark_id):
        ''' identifies cover image candidates via the cover html page '''
        cover_fallback_id = None
        if not cover_landmark_id:
            return cover_fallback_id
        else:
            cover_page_html = bk.readfile(cover_landmark_id)
            cover_image_href = re.search('(href|src)="(\.\.\/images\/[^"]+)"', cover_page_html, re.IGNORECASE)
            # future Sigil versions might not normalize folder names
            if not cover_image_href:
                cover_image_href = re.search('(href|src)="([^"]+\.(gif|jpg|jpeg|png|svg))"', cover_page_html, re.IGNORECASE)
            if cover_image_href:
                cover_fallback_id = bk.href_to_id(cover_image_href.group(2).replace('../', ''))
                return cover_fallback_id

    #-------------------------------------------------------------
    # generate EPUB2 cover metadata entry
    #-------------------------------------------------------------
    def fix_cover_meta_id(cover_fallback_id):
        ''' adds or replaces an image metadata entry '''
        metadata = bk.getmetadataxml()
        new_cover_meta = '<meta name="cover" content="{}" />'.format(cover_fallback_id)

        # replace existing cover image metadata entry
        meta_pattern = re.compile(r'<meta\s+[^>]*name\s*=\s*"cover"[^>]*>')
        cover_meta = meta_pattern.search(metadata)
        if cover_meta:
            new_cover_metadata = metadata.replace(cover_meta.group(0), new_cover_meta)
            print('Existing EPUB2 cover image metadata entry updated: ', new_cover_meta, ' ✅')

        # add new  cover image metadata entry
        else:
            new_cover_metadata = metadata.replace('</metadata>', new_cover_meta + '\n</metadata>')
            print('Missing EPUB2 cover image metadata entry added: ', new_cover_meta, ' ✅')

        # update metadata section
        bk.setmetadataxml(new_cover_metadata)

    #--------------------------------------------
    # get epub3 cover image
    #--------------------------------------------
    def get_epub3_cover_id():
        ''' returns the epub3 cover image '''
        cover_image_props_id = None
        for  id, href, mt, properties, fallback, mo in bk.manifest_epub3_iter():
            if properties == 'cover-image':
                cover_image_props_id = id
                break 
        return cover_image_props_id

    #----------------------------------------------------------------------
    # bk.getnavid() for older versions < 20190716
    #----------------------------------------------------------------------
    def getnavid():
        ''' returns the manifest id of the navigation document '''
        if bk.epub_version == "2.0":
            return None
        for id in sorted(bk._w.id_to_mime):
            mime = bk._w.id_to_mime[id]
            if mime == "application/xhtml+xml":
                properties = bk._w.id_to_props[id]
                if properties is not None and "nav" in properties:
                    return id
        return None

    #------------------------------------------------------------------------------------------------
    # auxiliary function that returns dtb:uid and docTitle for generateNCX
    #------------------------------------------------------------------------------------------------

    def get_ncx_meta():
        ''' returns dtb:uid and docTitle '''
        dtb_uid = None
        docTitle = ''

        # parse .opf file
        print('parsing opf')

        if python3lib:
            # Prepend python3lib directory to sys.path
            if os.path.exists(python3lib) and os.path.isdir(python3lib):

                # import Opf_Parser()
                sys.path.insert(0, os.path.abspath(python3lib))
                from opf_newparser import Opf_Parser
                from ncxgenerator import generateNCX
                op = Opf_Parser(bk.get_opf())
                
                # get uid
                ver, uid, tattr = op.package
                uid_list = []
                uid_list.append(uid)
                
                # get docTitle and dtb_uid
                for mname, mcontent, keylist, vallist in op.get_metadata():
                    #print(mname, mcontent, keylist, vallist)
                    if mname == 'dc:title': 
                        docTitle = mcontent
                    if mname == 'dc:identifier': 
                        if  keylist == ['id'] and vallist == uid_list:
                            dtb_uid = mcontent
                            break
        else:
            print('Internal error: python3lib not found!')

        return(dtb_uid, docTitle)    

    #=======================
    # set preferences
    #=======================
    prefs = prefs = bk.getPrefs()
    if prefs == {}:
        prefs['fix_cover'] = False
        prefs['add_landmarks'] = False
        prefs['add_guides'] = False
        #prefs['add_ncx'] = False
        bk.savePrefs(prefs)

    #======================
    # get preferences
    #======================
    fix_cover = prefs.get('fix_cover', False)
    add_landmarks = prefs.get('add_landmarks', False)
    add_guides = prefs.get('add_guides', False)
    add_ncx = prefs.get('add_ncx', False)
    debug = prefs.get('debug', False)

    # display preferences
    if debug: print('[DEBUG] Preferences:\nfix_cover: {}\nadd_landmarks: {}\nadd_guides: {}\n'.format(fix_cover, add_landmarks,add_guides))

    #===================
    # get spine ids
    #===================
    spine_ids = []
    for spine_item in bk.spine_iter():
        id, linear, href = spine_item
        spine_ids.append(id)

    #=====================
    # define variables
    #=====================
    guide_items = {}
    landmarks = {}
    cover_image_meta_id = None
    cover_image_props_id = None
    cover_guide_id = None
    dirty = False

    #=====================
    # get EPUB2 data
    #=====================
    for type, title, href in bk.getguide():
        basename = os.path.basename(href)
        # get html cover page id for get_cover_fallback_id()
        if type == 'cover':
            cover_guide_id = bk.href_to_id(href.split('#')[0]) 
        guide_items[basename] = (type, title)

    #-------------------------------------------------
    # get cover image metadata entry
    #-------------------------------------------------
    cover_image_meta_id = get_cover_id()
    if cover_image_meta_id:
        if epubversion.startswith("3"):
            cover_fallback_id = cover_image_meta_id
        # make sure the image id actually exists in the manifest
        if not bk.id_to_href(cover_image_meta_id):
            print('\nThe EPUB2 cover image referenced by the id "{}" is not in the manifest! ⚠'.format(cover_image_meta_id))
            cover_image_meta_id = None
            cover_fallback_id = None
    else:
        print('\nEPUB2 cover image NOT found! ⚠)')

    #------------------------------------------
    # get cover image fallback id
    #------------------------------------------
    if not cover_image_meta_id and cover_guide_id:
        cover_fallback_id = get_cover_fallback_id(cover_guide_id)

    if debug: print('\n[DEBUG] Guide items: ', guide_items, cover_guide_id, cover_image_meta_id, cover_fallback_id)

    #--------------------------------------------------------------------
    # fix/add EPUB 2 cover image metadata entry
    #---------------------------------------------------------------------
    if not cover_image_meta_id and cover_fallback_id:
        new_cover_meta = '<meta name="cover" content="{}" />'.format(cover_fallback_id)
        cover_image_file = os.path.basename(bk.id_to_href(cover_fallback_id))
        print('\nThe EPUB2 cover image metadata entry is missing or invalid!\n({} is most likely the cover image.)'.format(cover_image_file))

        if fix_cover:
            fix_cover_meta_id(cover_fallback_id)
            cover_image_meta_id = cover_fallback_id
            if debug: print('[DEBUG] *** EPUB2 cover fix #1:  cover_fallback_id > cover_image_meta_id ***')
            dirty = True
        else:
            print('You might want to add the following entry to content.opf:', new_cover_meta, ' ⚠')

    #===================
    # get EPUB3 data
    #===================
    if epubversion.startswith("3"):
        landmarks = {}
        cover_image_props_id = None
        cover_landmark_id = None

        #=======================
        # cover image handling
        #=======================

        #---------------------------------------------
        # get epub3 cover image item
        #---------------------------------------------
        cover_image_props_id = get_epub3_cover_id()

        #--------------------------------------------
        # fix broken epub2 metadata
        #--------------------------------------------
        if not cover_image_meta_id and cover_image_props_id:
            cover_image_meta_id = cover_image_props_id
            fix_cover_meta_id(cover_image_meta_id)
            if debug: print('[DEBUG] *** EPUB2 cover fix #2: props id => meta id ***')
            dirty = True


        #================================
        # landmarks handling
        #================================

        if bk.launcher_version() >= 20190716:
            #----------------------------------------------------------------------------------------------
            # use faster NavProcessor bundled with sigil 0.9.1.6 and higher
            #----------------------------------------------------------------------------------------------
            if debug: print('\n[DEBUG] Using external NavProcessor for nav parsing...')
            
            #-------------------------------------------------------------
            # get nav_id, nav_basename and nav_data
            #-------------------------------------------------------------
            nav_id = bk.getnavid()
            if nav_id:
                nav_basename = os.path.basename(bk.id_to_href(nav_id))
                nav_data = bk.readfile(nav_id)

                #------------------------------
                # get landmarks
                #------------------------------
                from navprocessor import NavProcessor
                np = NavProcessor(nav_data)
                landmark_list = np.getLandmarks()
                # if debug: print('\n[DEBUG] Landmark list: ', landmark_list)
            else:
                print('NAV landmark not found! Can\'t continue.\nClick OK to close.')
                return -1

        else:
            if debug: print('\n[DEBUG] Using embedded NavProcessor for nav doc parsing...')
            #---------------------------------------------------------------
            # get nav_id, nav_basename and nav_data
            #---------------------------------------------------------------
            nav_id = getnavid()
            if nav_id:
                nav_basename = os.path.basename(bk.id_to_href(nav_id))
                nav_data = bk.readfile(nav_id)

                #------------------------------
                # get landmarks
                #------------------------------
                landmark_list = getLandmarks(nav_data)
                if debug: print('\n[DEBUG] Landmark list: ', landmark_list)
                
            else:
                print('NAV landmark not found! Can\'t continue.\nClick OK to close.')
                return -1

        #-----------------------------------------------
        # store landmarks in dictionary
        #-----------------------------------------------
        for epub_type, href, title in landmark_list:

            # get basename
            basename = os.path.basename(href)

            #check for hrefs without filenames
            if basename.startswith('#'):
                # assume that fragment ids without a file name refer to nav.xhtml
                basename = nav_basename + href

            # check for cover page landmark
            if epub_type == 'cover':
                cover_landmark_id = bk.href_to_id(href.replace('../', '').split('#')[0])

            # merge landmarks with multiple epub type attributes
            if not basename in landmarks:
                landmarks[basename] = (epub_type, title)
            else:
                old_epub_type, title = landmarks[basename]
                if epub_type != old_epub_type:
                    epub_type = old_epub_type + ' ' + epub_type
                    landmarks[basename] = (epub_type, title)

        #=========================
        # fallback cover image handling
        #=========================

        #------------------------------------------
        # get cover image fallback id
        #------------------------------------------
        if not cover_image_props_id and cover_landmark_id:
            cover_fallback_id = get_cover_fallback_id(cover_landmark_id)

        #---------------------------------------------------------------------------------
        # add missing EPUB 3 "cover-image" properties attribute
        #---------------------------------------------------------------------------------
        if not cover_image_props_id and cover_fallback_id:
            cover_image_file = os.path.basename(bk.id_to_href(cover_fallback_id))
            print('\nEPUB3 cover image NOT found! ⚠\n({} is most likely the cover image.)'.format(cover_image_file))

            # add cover-image properties attribute to manifest
            if fix_cover == True:
                bk.set_manifest_epub3_attributes(cover_fallback_id, properties='cover-image')
                print('EPUB3 cover-image properties attribute added to {}. ✅'.format(cover_image_file))
                cover_image_props_id = cover_fallback_id

                # fix EPUB2 cover metadata entry
                if not cover_image_meta_id:
                    cover_image_meta_id = cover_fallback_id
                    fix_cover_meta_id(cover_image_meta_id)
                    if debug: print('[DEBUG] *** EPUB2 cover fix #3: epub3 fallback id => cover meta id ***')

                # set dirty flag
                dirty = True

            else:
                print('You might want to add an EPUB3 "cover-image" properties attribute to {}. ⚠'.format(cover_image_file))
                if cover_fallback_id and not cover_image_meta_id:
                    print('Missing EPUB2 cover image metadata entry ignored. ⚠')


        #=========================
        # add landmarks
        #=========================

        #---------------------------------------------------------------------------------
        # add missing EPUB3 landmarks
        #---------------------------------------------------------------------------------
        if landmarks != {}:

            # look for guide items without landmarks equivalent
            missing_landmarks = [k for k in guide_items if not k in landmarks]

            if missing_landmarks != []:
                print('\nThere are missing EPUB3 landmarks. ⚠')
                if debug: print('[DEBUG] Missing landmarks: {}'.format(missing_landmarks))

                # add missing landmarks
                if add_landmarks:
                    guide_items_found = False
                    print('\nTrying to map EPUB2 guide items to EPUB3 landmarks...\n')
                    
                    #-----------------------------
                    # get nav soup
                    #-----------------------------
                    nav_soup = BeautifulSoup(nav_data, 'html.parser')
                    nav_landmarks = nav_soup.find('nav', {'epub:type' : 'landmarks'}) 

                    for missing_landmark in missing_landmarks:
                        guide_type, guide_title = guide_items[missing_landmark]

                        # get id and href for landmark entries
                        manifest_id = bk.basename_to_id(missing_landmark.split('#')[0])
                        href = bk.id_to_href(manifest_id)
                        if '#' in missing_landmark:
                            href += '#' + missing_landmark.split('#')[1]

                        # bk.basename_to_id() will fail for xml files
                        if not manifest_id:
                            href = 'Text/' + missing_landmark
                            if href.split('#')[0] in bk._w.href_to_id:
                                manifest_id = bk._w.href_to_id[href]

                        # make sure the file is in the spine
                        if manifest_id in spine_ids:

                            if guide_type in _guide_epubtype_map:
                                guide_items_found = True

                                # map guide type to landmark type
                                epub_type = _guide_epubtype_map[guide_type]

                                # epubcheck doesn't like nav items without strings
                                if guide_title == '':
                                    if epub_type in epubtype_title_map:
                                        guide_title = epubtype_title_map[epub_type]
                                    else:
                                        guide_title = epub_type

                                # fallback cover handling
                                if epub_type == 'cover':
                                    cover_landmark_id = manifest_id

                                # add new landmarks list item
                                if href:
                                    new_li = nav_soup.new_tag('li')
                                    new_a = Tag(builder=nav_soup.builder, name='a', attrs={'epub:type' : epub_type, 'href' : '../' + href})
                                    new_a.string = guide_title
                                    new_li.append(new_a)
                                    nav_landmarks.ol.append(new_li)
                                    print('New EPUB3 landmark added: ', str(new_li))

                                    # add landmark to landmarks dictionary
                                    landmarks[missing_landmark] = (epub_type, guide_title)

                            else:
                                print('Unknown EPUB2 guide type found: ', guide_type)

                        else:
                            print('Couldn\'t add EPUB3 landmark for {}: NOT IN SPINE. ⚠'.format(os.path.basename(href)))

                    #------------------------------------------------
                    # update existing landmark entries
                    #------------------------------------------------
                    if guide_items_found:
                        nav_data = str(nav_soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  "))
                        bk.writefile(nav_id, nav_data)
                        print('\nUpdated landmarks section written to nav. ✅')

                        # set dirty flag
                        dirty = True
                    else:
                        print('\nNo suitable EPUB2 guide items found.')

                else:
                    print('\nMissing EPUB3 landmarks ignored. ⚠')

            else:
                if debug: print('\n[DEBUG] No missing EPUB3 landmarks found.')

        #---------------------------------------------------------------------------------
        # add new EPUB3 landmarks section
        #---------------------------------------------------------------------------------
        else:
            print('\nNo EPUB3 landmarks found. ⚠')

            if add_landmarks:
                #-------------------------------------------
                # add landmarks from guide
                #-------------------------------------------
                if guide_items != {}:

                    #-----------------------------
                    # get nav soup
                    #-----------------------------
                    nav_soup = BeautifulSoup(nav_data, 'html.parser')
                    nav_landmarks = nav_soup.find('nav', {'epub:type' : 'landmarks'}) 

                    # add new landmarks section
                    new_nav = Tag(builder=nav_soup.builder, name='nav', attrs={'epub:type' : 'landmarks', 'id' : 'landmarks', 'hidden' : ''})
                    nav_soup.body.append(new_nav)
                    new_h1 = nav_soup.new_tag('h1')
                    new_h1.string = 'Landmarks'
                    new_nav.append(new_h1)
                    new_ol = nav_soup.new_tag('ol')
                    new_nav.append(new_ol)

                    # add missing namespace
                    if 'xmlns:epub' not in nav_soup.html.attrs:
                        nav_soup.html['xmlns:epub'] = 'http://www.idpf.org/2007/ops'
                        print('Missing "xmlns:epub" namespace added.')

                    # add new landmarks
                    for guide_item in guide_items:
                        guide_type, guide_title = guide_items[guide_item]

                        # get id and href for landmark entries
                        manifest_id = bk.basename_to_id(guide_item.split('#')[0])
                        href = bk.id_to_href(manifest_id)
                        if '#' in guide_item:
                            href += '#' + guide_item.split('#')[1]

                        # bk.basename_to_id() will fail for xml files
                        if not manifest_id:
                            href = 'Text/' + guide_item
                            if href.split('#')[0] in bk._w.href_to_id:
                                manifest_id = bk._w.href_to_id[href]

                        # make sure document is in the spine
                        if manifest_id in spine_ids:

                            if guide_type in _guide_epubtype_map:
                                # map guide type to landmark type
                                epub_type = _guide_epubtype_map[guide_type]

                                # epubcheck doesn't like guide items without strings
                                if guide_title == '':
                                    if epub_type in epubtype_title_map:
                                        guide_title = epubtype_title_map[epub_type]
                                    else:
                                        guide_title = epub_type

                                # fallback cover handling
                                if epub_type == 'cover':
                                    cover_landmark_id = manifest_id

                                # add new landmarks list item
                                if href:
                                    new_li = nav_soup.new_tag('li')
                                    new_a = Tag(builder=nav_soup.builder, name='a', attrs={'epub:type' : epub_type, 'href' : '../' + href})
                                    new_a.string = guide_title
                                    new_li.append(new_a)
                                    new_ol.append(new_li)
                                    print('New landmark added: ', str(new_li))

                                    # add landmark to landmarks dictionary
                                    landmarks[guide_item] = (epub_type, guide_title)

                                    # set dirty flag
                                    landmarks_added = True
                            else:
                                print('Unknown EPUB2 guide type found: ', guide_type)

                        else:
                            print('Couldn\'t add EPUB3 landmark for {}: NOT IN SPINE. ⚠'.format(os.path.basename(href)))

                    #---------------------------------------------------
                    # add new landmark section to nav
                    #---------------------------------------------------
                    nav_data = str(nav_soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  "))
                    bk.writefile(nav_id, nav_data)
                    print('\nNew EPUB3 landmarks section added to nav. ✅')

                    # set dirty flag
                    dirty = True
            else:
                print('Missing EPUB3 landmarks ignored. ⚠')

        # print all items 
        if debug: print('\n[DEBUG] Landmarks: ', nav_basename, landmarks, cover_landmark_id, cover_image_props_id, cover_fallback_id)

        #=============================
        # add missing EPUB3 guide items
        #=============================
        if guide_items == {}:
            print('\nNo EPUB2 guide items found. ⚠')

            if add_guides and landmarks != {}:

                if python3lib is not None:

                    # Prepend python3lib directory to sys.path
                    if os.path.exists(python3lib) and os.path.isdir(python3lib):

                        # import generateGuideEntries()
                        sys.path.insert(0, os.path.abspath(python3lib))
                        from ncxgenerator import generateGuideEntries

                        # generate guide items from landmarks
                        if bk.launcher_version() < 20190927:
                            guide = generateGuideEntries(nav_data, nav_basename)
                        else:
                            # the syntax changed in Sigil 1.0
                            # generateGuideEntries(nav_data, nav_bkpath, opfdir):
                            opfdir = bk.get_startingdir(bk.get_opfbookpath())
                            nav_bkpath = bk.id_to_bookpath(bk.getnavid())
                            guide = generateGuideEntries(nav_data, nav_bkpath, opfdir)
                        
                        if debug: print('\n[DEBUG] generateGuideEntries: ', guide, '\n')

                        if guide != []:
                            # generateGuideEntries() returns [(type, href, TITLE)])
                            # but bk.setguide() expects [(type, TITLE, href) and type != None
                            new_guide =  [(t[0], t[2], t[1]) for t in guide if t[0]]

                            # add guide item
                            bk.setguide(new_guide)

                             # set dirty flag
                            dirty = True
                            print('New EPUB2 guide section added. ✅')

                            # populate guide_items{}
                            for type, title, href in new_guide:
                                basename = os.path.basename(href)
                                guide_items[basename] = (type, title)

                        else:
                            print('\nEPUB2 Guide items couldn\'t be generated.')

                    else:
                        print('\nInternal error: couldn\'t add python3lib directory to sys.path!')

                else:
                    print('Internal error: python3lib not found!')

        #=============================
        # add/update NCX
        #=============================
        if add_ncx:
            if python3lib:
        
                # get required NCX metadata values
                dtbuid, docTitle = get_ncx_meta()
                if debug: print('\ndtb:uid: {}\ndocTitle: {}'.format(dtbuid, docTitle))
                
                # generate NCX data
                if dtbuid:
                    sys.path.insert(0, os.path.abspath(python3lib))
                    from ncxgenerator import generateNCX
                    ncxdata = generateNCX(nav_data, nav_basename, docTitle, dtbuid)
                else:
                    ncxdata = None
                    print('\nInternal error: dtbuid not found.')
                
                # add/update toc.ncx
                ncx_id = bk.gettocid()
                if ncx_id:
                    # update NCX
                    bk.writefile(ncx_id, ncxdata)
                    print('\nNCX updated. ✅')
                    dirty = True
                else:
                    # add NCX
                    try:
                        bk.addfile('toc.cx', 'toc.cx', ncxdata, 'application/x-dtbncx+xml')
                        print('\nNCX added. ✅')
                        dirty = True
                    except:
                        print('Not supported by this version. ❌')

            else:
                print('Internal error: python3lib not found!')
        else:
            #if debug: print('NCX not updated.')
            pass

    #==================
    # display GUI
    #==================
    if landmarks != {} or guide_items != {}:
        if debug:
            print('\n[DEBUG] Parameters:\n\nlandmarks: {}\nguide_items: {}\ncover_image_props_id: {}\ncover_image_meta_id: {}\nDirty: {}'.format(landmarks, guide_items, cover_image_props_id, cover_image_meta_id, dirty))

        mdp = True if iswindows else False
        app_icon = os.path.join(plugin_dir, 'app.ico')
        app = PluginApplication(sys.argv, bk, app_icon=app_icon, match_dark_palette=mdp,
                                load_qtbase_translations=False, load_qtplugin_translations=False)
        window = GUI(bk, landmarks, guide_items, cover_image_props_id, cover_image_meta_id, dirty)
        #window.show()
        app.exec()

    else:
        print('\nThis book contains neither landmarks nor guide items! ⚠')

        if cover_image_props_id or cover_image_meta_id:
            if cover_image_meta_id:
                cover_file_name = os.path.basename(bk.id_to_href(cover_image_meta_id))
                print('\nEPUB2 cover image found: {} [{}] ✅'.format(cover_file_name, cover_image_meta_id)) 
            if cover_image_props_id:
                cover_file_name = os.path.basename(bk.id_to_href(cover_image_props_id))
                print('\nEPUB3 cover image found: {} [{}] ✅'.format(cover_file_name, cover_image_props_id)) 

    print('\nDone. Click OK to close.')

    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
