#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals, division, absolute_import, print_function
import xml.etree.ElementTree as ET
import os, os.path, sys, codecs, re, locale, tempfile, json, socket
import subprocess
from subprocess import Popen, PIPE
from os.path import expanduser
from zipfile import ZipFile
from datetime import datetime, timedelta

if sys.platform.startswith('win'):
    os_encoding = locale.getpreferredencoding()
else:
    os_encoding = 'utf-8'

try:
    from sigil_bs4 import BeautifulSoup
except:
    from bs4 import BeautifulSoup

# encode/escape text to make it xml safe (from navprocessor.py)
def xmlencode(data):
    ''' escapes text to make it xml safe '''
    if data is None:
        return ''
    newdata = data
    newdata = newdata.replace('&', '&amp;')
    newdata = newdata.replace('<', '&lt;')
    newdata = newdata.replace('>', '&gt;')
    newdata = newdata.replace('"', '&quot;')
    return newdata

# code provided by DiapDealer
def string_to_date(datestring):
    ''' converts dates to strings '''
    return datetime.strptime(datestring, "%Y-%m-%d %H:%M:%S.%f")

# code provided by DiapDealer
def is_connected():
    ''' tests Internet connectivity '''
    try:
        sock = socket.create_connection(('8.8.8.8', 53), 1)
        sock.close()
        return True
    except:
        pass

#----------------------------------
# detect Python version
#----------------------------------
PY2 = sys.version_info[0] == 2
if PY2:
    import Tkinter as tkinter
    import tkFileDialog as tkinter_filedialog
    from urllib import urlopen
else:
    import tkinter
    import tkinter.filedialog as tkinter_filedialog
    from urllib.request import urlopen

#----------------------------------
# detect operating system
#----------------------------------
isosx = sys.platform.startswith('darwin')
islinux = sys.platform.startswith('linux')
iswindows = sys.platform.startswith('win32')

#----------------------------------
# detect UI language
#----------------------------------
def GetUserDefaultUILanguage():
    ''' returns the UI language '''
    if iswindows:
        import ctypes
        return locale.windows_locale[ctypes.windll.kernel32.GetUserDefaultUILanguage()][0:2]
    else:
        language, codepage = locale.getdefaultlocale()
        if language:
            return language[0:2]
        else:
            return 'en'

#------------------------------------
# display LT file selection dialog
#------------------------------------
def GetFileName(bk):
    ''' displays the LanguageTool file selection dialog '''
    # requires Sigil 0.9.8 or higher or PyQt5
    try:
        home = expanduser('~')
        from plugin_utils import QtWidgets
        from plugin_utils import PluginApplication, iswindows
        mdp = True if iswindows else False
        app = PluginApplication(sys.argv, bk, match_dark_palette=mdp)
        w = QtWidgets.QWidget()
        file_path, filter = QtWidgets.QFileDialog.getOpenFileName(w,'Select LanguageTool Java file', home, 'LT (languagetool-commandline.jar)')
    except:
        root = tkinter.Tk()
        root.withdraw()
        filetypes = (('Java archive files', 'languagetool-commandline.jar'),("All files", "*.*"))
        file_path = tkinter_filedialog.askopenfilename(filetypes = filetypes, title='Select LanguageTool Java file')
    return file_path


#------------------------------------
# create temp folder
#------------------------------------
from contextlib import contextmanager
@contextmanager
def make_temp_directory():
    ''' creates a temporary directory '''
    import tempfile
    import shutil
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)

#-------------------------
# simple Java wrapper
#-------------------------
def ltWrapper(*args):
    ''' LanguageTool Java wrapper '''
    import subprocess
    startupinfo = None

    # stop the windows console popping up every time the prog is run
    if iswindows:
        startupinfo = subprocess.STARTUPINFO()
        startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        startupinfo.wShowWindow = subprocess.SW_HIDE

    process = subprocess.Popen(list(args), stdout=subprocess.PIPE, stderr=subprocess.PIPE, startupinfo=startupinfo)
    ret = process.communicate()
    returncode = process.returncode
    return ret, returncode

#-------------------------------------
# remove all html tags and entities
#-------------------------------------
def DeleteHTMLTags(html):
    ''' removes all html tags and entities '''
    zwj = u'\u200d'
    cleaned_html = html

    # replace XML declaration
    match = re.search(r'<\?xml[^>]+>', cleaned_html)
    if match is not None:
        match_found = match.group(0)
        replacement = zwj * len(match_found)
        cleaned_html = cleaned_html.replace(match_found, replacement)

    # replace doctype
    match = re.search(r'<!DOCTYPE[^>]+>', cleaned_html)
    if match is not None:
        match_found = match.group(0)
        if not '\n' in match_found:
            replacement = zwj * len(match_found)
        else:
            string_list = list(match_found)
            for letter in range(0, len(match_found)):
                if string_list[letter] not in ['\r', '\n', ' ']:
                    string_list[letter] = zwj
            replacement = "".join(string_list)
        cleaned_html = cleaned_html.replace(match_found, replacement)

    # replace script sections
    match = re.search('<script[^>]*>.*</script>', cleaned_html, re.MULTILINE|re.DOTALL)
    if match is not None:
        match_found = match.group(0)
        if not '\n' in match_found:
            replacement = zwj * len(match_found)
        else:
            string_list = list(match_found)
            for letter in range(0, len(match_found)):
                if string_list[letter] not in ['\r', '\n', ' ']:
                    string_list[letter] = zwj
            replacement = "".join(string_list)
        cleaned_html = cleaned_html.replace(match_found, replacement)

    # replace standard HTML tags
    matches = re.finditer(r'''</?([A-Za-z][^\s>/]*)(?:=\s*(?:"[^"]*"|'[^']*'|[^\s>]+)|[^>])*(?:>|$)''', cleaned_html)
    for match in matches:
        match_found = match.group(0)
        replacement = ' ' * len(match_found)
        cleaned_html = cleaned_html.replace(match_found, replacement)

    # replace entities
    matches = re.finditer(r'&[^;]+;', cleaned_html)
    for match in matches:
        match_found = match.group(0)
        # non-breaking space
        if match_found in ['&#160;', '&nbsp;', '&#xa0;']:
            replacement = u'\u200D' * (len(match_found) -1) + u'\u00A0'
        # en dash
        elif match_found in ['&#8211;', '&ndash;', '&#x2013;']:
            replacement = u'\u200D' * int(len(match_found) / 2) + u'\u2013' + u'\u200D' * 3
        # em dash
        elif match_found in ['&#8212;', '&mdash;', '&#x2014;']:
            replacement = u'\u200D' * int(len(match_found) / 2) + u'\u2014' + u'\u200D' * 3
        # ampersand
        elif match_found in ['&amp;', '&#38;']:
            replacement = u'\u200D' * 2 + u'&' + u'\u200D' * 2
        # all other entities
        else:
            replacement = ' ' * len(match_found)
        cleaned_html = cleaned_html.replace(match_found, replacement)

    assert len(html) == len(cleaned_html), 'Internal HTML filter error.'

    return cleaned_html

#=================
# main routine
#=================
def run(bk):
    ''' main routine '''

    # initialize variables
    LT_version = ''
    userRulesPath = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'user-rules.xml')
    uiLanguage = GetUserDefaultUILanguage()

    #--------------------------------------------
    # Get language from metadata
    #--------------------------------------------
    temp_dir = bk._w.ebook_root
    dc_language = 'en'
    orig_lang = 'en'
    language = BeautifulSoup(bk.get_opf(), 'lxml').find('dc:language')
    if language:
        orig_lang = language.text
        if orig_lang != '':
            dc_language = orig_lang
    else:
        print('language not found!')

    #---------------------------------------------------------------------------------------
    # change two letter language codes to four letter language codes for better LT results
    #---------------------------------------------------------------------------------------
    lang_dict = {
        'ast' : 'ast-ES', 'be' : 'be-BY', 'br' : 'br-FR' , 'da' : 'da-DK' , 'de' : 'de-DE',
        'el' : 'el-GR', 'gl' : 'gl-ES', 'ja' : 'ja-JP', 'km' : 'km-KH', 'pl' : 'pl-PL', 'ro' : 'ro-RO',
        'ru' : 'ru-RU', 'sk' : 'sk-SK', 'sl' : 'sl-SI', 'ta' : 'ta-IN', 'tl' : 'tl-PH', 'uk' : 'uk-UA'
    }
    if dc_language in lang_dict:
        dc_language = lang_dict[dc_language]

    #-------------------------------------------------------------
    # make sure that the language is actually supported by LT
    #-------------------------------------------------------------
    suported_language_codes = ['ast-ES', 'be-BY', 'br-FR', 'ca-ES', 'ca-ES-valencia', 'da-DK', 'de', 'de-AT',
        'de-CH', 'de-DE', 'de-DE-x-simple-language', 'el-GR', 'en', 'en-AU', 'en-CA', 'en-GB', 'en-NZ', 'en-US',
        'en-ZA', 'eo', 'es', 'fa', 'fr', 'gl-ES', 'it', 'ja-JP', 'km-KH', 'nl', 'pl-PL', 'pt', 'pt-AO', 'pt-BR',
        'pt-MZ', 'pt-PT', 'ro-RO', 'ru-RU', 'sk-SK', 'sl-SI', 'sr', 'sr-BA', 'sr-HR', 'sr-ME', 'sr-RS', 'sv',
        'ta-IN', 'tl-PH', 'uk-UA', 'zh-CN']

    if not dc_language in suported_language_codes:
        dc_language = dc_language[0:2]
        if not dc_language in suported_language_codes:
            print('Language code not supported!', "'{}' is not a language code known to LanguageTool. Supported language codes are: {}.".format(orig_lang, ', '.join(suported_language_codes)))
            return -1

    #-----------------------------------
    # handle preferences
    #-----------------------------------
    prefs = bk.getPrefs()

    # set some meaningful defaults
    if prefs == {}:

        # use Sigil GUI language as the default
        if bk.launcher_version() >= 20170115:
            uiLanguage = bk.sigil_ui_lang[0:2]

        # add language language specific defaults to disable the spellchecker and whitespace errors
        if uiLanguage == 'fr':
            prefs['disabledRules'] = 'FR_SPELLING_RULE'
        elif uiLanguage == 'it':
            prefs['disabledRules'] = 'MORFOLOGIK_RULE_IT_IT'
        elif uiLanguage == 'de':
            prefs['disabledRules'] = 'GERMAN_SPELLER_RULE,GERMAN_WORD_REPEAT_BEGINNING_RULE'
        elif uiLanguage == 'es':
            prefs['disabledRules'] = 'MORFOLOGIK_RULE_ES'
        elif uiLanguage == 'nl':
            prefs['disabledRules'] = 'MORFOLOGIK_RULE_NL_NL'
        elif uiLanguage == 'pl':
            prefs['disabledRules'] = 'MORFOLOGIK_RULE_PL_PL'
        elif uiLanguage == 'sr':
            prefs['disabledRules'] = 'MORFOLOGIK_RULE_SR_EKAVIAN'
        else:
            prefs['disabledRules'] = 'MORFOLOGIK_RULE_EN_US,MORFOLOGIK_RULE_EN_GB,ENGLISH_WORD_REPEAT_BEGINNING_RULE'

        # save defaults
        prefs['disabledRules'] += ',WHITESPACE_RULE,COMMA_PARENTHESIS_WHITESPACE,SENTENCE_WHITESPACE'
        #prefs['disabledCategories'] = 'REDUNDANCY'
        prefs['last_time_checked'] = str(datetime.now() - timedelta(days=7))
        prefs['update_check'] = False
        prefs['allFiles'] = False
        prefs['clipboard_copy'] = False
        bk.savePrefs(prefs)

    #---------------------------------
    # get preferences
    #---------------------------------
    prefs = bk.getPrefs()

    # path to the ngram-index directory (--languagemodel)
    ngramIndexDir = prefs.get('ngramIndexDir', None)
    if ngramIndexDir:
        if not os.path.isdir(os.path.join(ngramIndexDir, dc_language[0:2])):
            print('\n*** ngram directory {} not found! ***\n'.format(os.path.join(ngramIndexDir, dc_language[0:2])))
            ngramIndexDir = None

    # path to the word2vecDir directory (--word2vecmodel)
    word2vecDir = prefs.get('word2vecDir', None)
    if word2vecDir:
        if not os.path.isdir(os.path.join(word2vecDir, dc_language[0:2])):
            print('\n*** word2vec directory {} not found! ***\n'.format(os.path.join(word2vecDir, dc_language[0:2])))
            word2vecDir = None

    # check allFiles preference
    allFiles = prefs.get('allFiles', False)

    # IDs of rules to be enabled, comma-separated (-e)
    enabledRules = prefs.get('enabledRules', None)
    if enabledRules and enabledRules != '' :
        enabledRules = enabledRules.replace(' ', '')

    # IDs of rules to be disabled, comma-separated (-d)
    disabledRules = prefs.get('disabledRules', 'WHITESPACE_RULE,COMMA_PARENTHESIS_WHITESPACE,SENTENCE_WHITESPACE')
    if disabledRules and disabledRules != '':
        disabledRules = disabledRules.replace(' ', '')
        disabledRulesList = disabledRules.split(',')
    else:
        disabledRulesList = []

    # IDs of categories to be enabled, comma-separated (--enablecategories)
    enabledCategories = prefs.get('enabledCategories', None)
    if enabledCategories and enabledCategories != '':
        enabledCategories = enabledCategories.replace(' ', '')

    # IDs of categories to be disabled, comma-separated to be disabled (--disablecategories)
    disabledCategories = prefs.get('disabledCategories', None)
    if disabledCategories and disabledCategories != '':
        disabledCategories = disabledCategories.replace(' ', '')
        disabledCategoriesList = disabledCategories.split(',')
    else:
        disabledCategoriesList = []

    # If true, only the rules and categories whose IDs are specified with enabledRules or enabledCategories are enabled. (--enabledonly)
    enabledOnly = prefs.get('enabledOnly', False)
    if enabledOnly == True:
        enabledOnly = '-eo'

    # Get LanguageTool path
    ltPath = prefs.get('ltPath', None)
    if not ltPath or not os.path.isfile(ltPath):
        ltPath = GetFileName(bk)
        if ltPath:
            prefs['ltPath'] = ltPath
            bk.savePrefs(prefs)

    # clipboard copy setting
    clipboard_copy = prefs.get('clipboard_copy', False)

    # for debugging
    debug = prefs.get('debug', False)

    # enables update check
    update_check = prefs.get('update_check', False)

    # check interval  
    check_interval  = prefs.get('check_interval ', 7)

    # for update check
    last_time_checked = prefs.get('last_time_checked', str(datetime.now() - timedelta(days=7)))

    #-------------------------------------------------------
    # define LT command line parameters
    #-------------------------------------------------------
    args = ['java', '-Xmx1024m', '-Xms512m', '-jar', ltPath, '-c', 'utf-8']

    # path to the ngram-index directory (--languagemodel)
    if ngramIndexDir:
        args.extend(['--languagemodel', ngramIndexDir])

    # path to the word2vecDir directory (--word2vecmodel)
    if word2vecDir:
        args.extend(['--word2vecmodel', word2vecDir])

    # IDs of rules to be enabled, comma-separated (-e)
    if not enabledRules and ngramIndexDir:
        args.extend(['-e', 'CONFUSION_RULE'])

    if enabledRules and not ngramIndexDir:
        args.extend(['-e', enabledRules.replace(' ', '')])

    # IDs of rules to be disabled, comma-separated (-d)
    if disabledRules and not ngramIndexDir:
        args.extend(['-d', disabledRules.replace(' ', '')])

    # IDs of categories to be enabled, comma-separated (--enablecategories)
    if enabledCategories and not ngramIndexDir:
        args.extend(['--enablecategories', enabledCategories.replace(' ', '')])

    # IDs of categories to be disabled, comma-separated to be disabled (--disablecategories)
    if disabledCategories and not (ngramIndexDir or word2vecDir):
        args.extend(['--disablecategories', disabledCategories.replace(' ', '')])

    # If true, only the rules and categories whose IDs are specified with
    # enabledRules or enabledCategories are enabled. (--enabledonly)
    if (ngramIndexDir and not enabledOnly) or enabledOnly:
        args.extend(['-eo'])

    # check for optional userrules.xml file
    if os.path.isfile(userRulesPath) and not ngramIndexDir:
        args.extend(['--rulefile', userRulesPath])

    # select json output
    args.extend(['--json', '-l', dc_language])

    #============================
    # check files
    #============================
    if ltPath and os.path.isfile(ltPath):

        # ------------------------------
        # get list of file names
        # ------------------------------

        # get all selected xhtml files, if any
        selected_files = []
        for file_name in list(bk.selected_iter()):
            if bk.id_to_mime(file_name[1]) == 'application/xhtml+xml':
                selected_files.append((file_name[1], bk.id_to_href(file_name[1])))

        # select files to be processed
        if selected_files != [] and not allFiles:
            # only selected files
            file_list = selected_files
        else:
            # ALL files
            file_list = list(bk.text_iter())
            print('Processing ALL files...\n')

        # create a temp folder
        with make_temp_directory() as new_temp_dir:

            # write header to clipboard file
            if clipboard_copy and not debug:
                clipboard_messages = '{}\t{}\t{}\t{}\t{}\t{}\t{}\t{}\n'.format('File Name', 'Line', 'Context', 'Category', 'Rule', 'Short Message', 'Message', 'Suggestions')

            if ngramIndexDir:
                print('N-gram mode enabled!\n')

            if word2vecDir:
                print('word2vec mode enabled!\n')

            #--------------------------------
            # process all selected files
            #--------------------------------
            for (html_id, href) in file_list:
                if  bk.launcher_version() >= 20190927:
                    file_name = os.path.join(bk.id_to_bookpath(html_id))
                else:
                    file_name = os.path.abspath(os.path.join(temp_dir, 'OEBPS', href))
                basename = os.path.basename(href)
                print('Working on ' +  basename + '...')
                html = bk.readfile(html_id)

                # remove carriage returns in the Sigil Windows version
                if iswindows:
                    html = html.replace('\r', '')

                # get line breaks
                line_breaks = list(re.finditer('\n', html))

                # remove html tags
                cleaned_html = DeleteHTMLTags(html)

                # create temp file
                file_name = os.path.join(new_temp_dir, basename)
                with open(file_name, 'wb') as f:
                    f.write(cleaned_html.encode('utf8'))

                #--------------------------------------------------------------------
                # run LanguageTool
                #-------------------------------------------------------------------
                run_args = []
                run_args.extend(args)
                run_args.append(file_name)
                result, returncode = ltWrapper(*run_args)
                stdout = result[0].decode('utf-8')
                stderr = result[1].decode(os_encoding)
                if stderr != '' and debug:
                    bk.add_result('info', None, None, 'DEBUG: ' + stderr)
                
                #------------------------------------
                # check for Java errors
                #------------------------------------
                if stderr.find('java.lang.StackOverflowError') != -1 or stderr.find('Exception in thread') != -1:
                    print('LanguageTool Java error.\n', stderr)
                    return -1

                if returncode == 0:

                    # process LT JSON
                    parsed_json = (json.loads(stdout))

                    # get software name & version
                    software_name = parsed_json['software']['name']
                    software_version = parsed_json['software']['version']
                    LT_version = 'v' + software_version

                    # get error messages
                    matches = parsed_json['matches']
                    print(len(matches), 'error(s) found.')

                    # parse error messages
                    for match in matches:

                        # get message, short message and replacements
                        message = match['message']
                        shortMessage = match['shortMessage']
                        replacements = match['replacements']

                        # format suggestions
                        if len(replacements) == 0:
                            suggestions = ''
                        elif len(replacements) == 1:
                            suggestions = 'Suggestion: ' +  replacements[0]['value']
                        else:
                            suggestions = ''
                            for replacement in replacements:
                                suggestions += replacement['value'] + ', '
                            suggestions = 'Suggestions: ' + suggestions[0:len(suggestions)-2]

                        # get error offset and length (error_length = context_length)
                        error_offset = match['offset']
                        # error_length = match['length']

                        # format context
                        context_text = match['context']['text']
                        context_offset = match['context']['offset']
                        context_length = match['context']['length']
                        typo = context_text[context_offset:context_offset+context_length].strip()
                        textbefore = context_text[0:context_offset].strip()
                        textafter = context_text[context_offset + context_length:].strip()
                        context = re.sub('[\t ]{2,}', ' ', textbefore + ' ■' + typo + '■ ' + textafter)

                        # get sentence and type
                        # sentence = match['sentence']
                        # type = match['type']['typeName']

                        # get rule and category ids
                        rule_id = match['rule']['id']
                        rule_description = match['rule']['description']
                        rule_category_id = match['rule']['category']['id']
                        rule_category_name = match['rule']['category']['name']

                        # format message
                        lt_message = xmlencode('Context: {} {}:{} {} {} {}'.format(context, rule_category_id, rule_id, shortMessage, message, suggestions))

                        # get colunn offset (requires Sigi 0.9.7 or higher)
                        if bk.launcher_version() >= 20160909:
                            coffset = error_offset
                        else:
                            coffset = -1

                        # get line number
                        last_offset = 0
                        line_number = 0
                        for line_break in line_breaks:
                            line_number += 1
                            if error_offset > last_offset and error_offset < line_break.end():
                                break
                            else:
                                last_offset = line_break.end()


                        #----------------------------------------
                        # add error message to validation pane
                        #---------------------------------------
                        message_type = 'warning'

                        # sigil 1.x (and higher) supports custom folders and requires book paths
                        if  bk.launcher_version() >= 20190927:
                            basename = os.path.join(bk.id_to_bookpath(html_id))

                        # LanguageTool ignores some disabled internal rules and categories
                        if rule_id not in disabledRulesList and rule_category_id not in disabledCategoriesList:
                            if coffset != -1:
                                bk.add_extended_result(message_type, basename, line_number, coffset, lt_message)
                            else:
                                bk.add_result(message_type, basename, line_number, lt_message)

                            # copy error messages to the clipboard
                            if clipboard_copy and not debug:
                                clipboard_messages += '{}\t{}\t{}\t{}\t{}\t{}\t{}\t{}\n'.format(basename, line_number, context, rule_category_id, rule_id, shortMessage, message, suggestions)
                        else:
                            if debug:
                                message_type = 'info'
                                                
                                if coffset != -1:
                                    bk.add_extended_result(message_type, basename, line_number, coffset, 'DEBUG: IGNORED: ' + lt_message)
                                else:
                                    bk.add_result(message_type, basename, line_number, 'DEBUG: IGNORED: ' + lt_message)

                else:
                    # LanguageTool failed; display error message(s)
                    bk.add_result('error', None, None, 'Command line: ' +  xmlencode(' '.join(run_args)))
                    bk.add_result('error', None, None, 'Java error message: ' +  xmlencode(stdout + '\n' + stderr))
                    break

            # copy results to the clipboard
            if clipboard_copy:
                try:
                    from plugin_utils import PluginApplication
                    app = PluginApplication(sys.argv, bk)
                    if debug:
                        #app.clipboard().setText(cleaned_html)
                        app.clipboard().setText(xmlencode(' '.join(run_args)) + '\n\n' + cleaned_html)
                        bk.add_result('info', None, None, 'Command line: ' +  xmlencode(' '.join(run_args)))
                    else:
                        app.clipboard().setText(clipboard_messages)
                except:
                    pass


            # check for LT updates
            if update_check and 'SNAPSHOT' not in LT_version and LT_version != '':

                # make sure we have an Internet connection
                if is_connected():

                    # compare current date against last update check date
                    time_delta = (datetime.now() - string_to_date(last_time_checked)).days
                    if time_delta >= check_interval:

                        if debug:
                            bk.add_result('info', None, None, 'DEBUG: Checked for update. Time delta: ' + str(time_delta))

                        # update time stamp
                        prefs['last_time_checked'] = str(datetime.now())
                        bk.savePrefs(prefs)

                        try:
                            github_url = 'https://api.github.com/repos/languagetool-org/languagetool/tags'
                            response = urlopen(github_url).read().decode('utf-8')
                            parsed_json = json.loads(response)
                            latest_version = parsed_json[0]['name']
                            if latest_version != LT_version:
                                bk.add_result('info', None, None, 'New LangugeTool version available: ' + latest_version)
                        except:
                            pass

                    else:
                        if debug:
                            bk.add_result('info', None, None, 'DEBUG: Update check skipped. Time delta: ' + str(time_delta))

                else:
                    bk.add_result('info', None, None, 'Update skipped: no Internet.')

        return 0
    else:
        print('LanguageTool path not selected or not found!')
        return -1

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
