#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys, os, re, urllib, socket
import xml.etree.ElementTree as ET

from plugin_utils import Qt, QtCore, QtGui, QtWidgets
from plugin_utils import loadUi, PluginApplication, iswindows
from datetime import datetime

# required for loading the .ui and .qm files from the plugin folder
plugin_dir = os.path.dirname(os.path.abspath(__file__))

try:
    from sigil_bs4 import BeautifulSoup
except:
    from bs4 import BeautifulSoup

# roman numerals converter
def int_to_roman(input):
    """ Convert an integer to a Roman numeral. """

    if not isinstance(input, type(1)):
        raise TypeError("expected integer, got %s" % type(input))
    if not 0 < input < 4000:
        raise ValueError("Argument must be between 1 and 3999")
    ints = (1000, 900,  500, 400, 100,  90, 50,  40, 10,  9,   5,  4,   1)
    nums = ('M',  'CM', 'D', 'CD','C', 'XC','L','XL','X','IX','V','IV','I')
    result = []
    for i in range(len(ints)):
        count = int(input / ints[i])
        result.append(nums[i] * count)
        input -= ints[i] * count
    return ''.join(result)

# code provided by DiapDealer
def is_connected():
    try:
        sock = socket.create_connection(('8.8.8.8', 53), 1)
        sock.close()
        return True
    except:
        pass

# code provided by DiapDealer
def string_to_date(datestring):
    return datetime.strptime(datestring, "%Y-%m-%d %H:%M:%S.%f")

# main GUI
class GUI(QtWidgets.QDialog):
    
    def __init__(self, bk):
        self.bk = bk
        super(GUI, self).__init__()
        loadUi(os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'addids.ui'), self)

        # define connects
        self.buttonBox.accepted.connect(self.accept)
        self.accepted.connect(self.get_values)
        # OK button pressed, save values
        self.buttonBox.rejected.connect(self.reject)
        # Cancel button pressed, set Cancel value
        self.rejected.connect(self.cancel_button)

        # populate text boxes with preference values
        if 'tag' in prefs:
            self.tagEntry.setText(prefs['tag'])
        if 'attribute' in prefs:
            self.attributeEntry.setText(prefs['attribute'])
        if 'value' in prefs:
            self.valueEntry.setText(prefs['value'])
        if 'id' in prefs:
            self.idEntry.setText(prefs['id'])
        if 'counter' in prefs:
            self.counterEntry.setText(prefs['counter'])
        if 'prefix' in prefs:
            self.titleEntry.setText(prefs['prefix'])
        # set check box values
        if 'zeros' in prefs:
            self.zerosCheckBox.setChecked(prefs['zeros'])
        if 'roman' in prefs:
            self.romanCheckBox.setChecked(prefs['roman'])
        if 'title' in prefs:
            self.titleCheckBox.setChecked(prefs['title'])

        self.show()

    # save options
    def get_values(self):
        global Cancel
        Cancel = False
        prefs['tag'] = self.tagEntry.text().strip()
        prefs['attribute'] = self.attributeEntry.text().strip()
        prefs['value'] = self.valueEntry.text().strip()
        prefs['id'] = self.idEntry.text().strip()
        prefs['counter'] = self.counterEntry.text().strip()
        prefs['prefix'] = self.titleEntry.text().strip()
        prefs['zeros'] = self.zerosCheckBox.isChecked()
        prefs['roman'] = self.romanCheckBox.isChecked()
        prefs['title'] = self.titleCheckBox.isChecked()
        self.bk.savePrefs(prefs)

    # set cancel flag
    def cancel_button(self):
        global Cancel
        Cancel = True
        self.close

def run(bk):
    # get the UI language of Sigil
    try:
        sigil_lang = bk.sigil_ui_lang[0:2]
    except:
        sigil_lang = 'en'

    # PyQt requires Sigil 0.9.8
    if not bk.launcher_version() >= 20170115:
        # can't use Qtranslator()
        if sigil_lang == 'fr':
            print('Ce greffon requiert au moins Sigil 0.9.8.\n\nCliquez sur OK pour fermer.')
        elif sigil_lang == 'it':
            print('Questo plugin richiede almeno Sigil 0.9.8.\n\nPer chiudere, fare clic su OK.')
        elif sigil_lang == 'de':
            print('Dieses Plugin erfordert mindestens Sigil 0.9.8.\n\nKlicken Sie zum Schließen auf "OK".')
        elif sigil_lang == 'es':
            print('Este complemento necesita al menos Sigil 0.9.8.\n\nHaga clic en Aceptar para cerrar.')
        elif sigil_lang == 'nl':
            print('Deze plugin vereist minimaal Sigil 0.9.8.\n\nKlik op OK om te sluiten.')
        else:
            print('This plugin requires at least Sigil 0.9.8.\n\nClick OK to close.')
        return -1

   # get preferences
    global prefs
    prefs = bk.getPrefs()
    if prefs == {}:
        prefs['id'] = 'id'
        prefs['counter'] = '1'
 
     # get update check preference
    if 'update_check' in prefs:
        update_check = prefs['update_check']
    else:
        update_check = True
        prefs['update_check'] = True
    
    # get last update check date
    if update_check:
        if 'last_time_checked' in prefs:
            last_time_checked = string_to_date(prefs['last_time_checked'])
        else:
            last_time_checked = datetime.now()
            prefs['last_time_checked'] = str(last_time_checked)
            bk.savePrefs(prefs)

    # define dialog box
    mdp = True if iswindows else False
    app_icon = os.path.join(plugin_dir, 'app.ico')
    app = PluginApplication(sys.argv, bk, app_icon=app_icon, match_dark_palette=mdp,
                            load_qtbase_translations=False, load_qtplugin_translations=False)

    # FIGS localization
    if sigil_lang in ['fr', 'it', 'de', 'es', 'nl']:
        # load plugin translations
        translator = QtCore.QTranslator()
        translator.load(sigil_lang + '.qm', os.path.join(bk._w.plugin_dir, bk._w.plugin_name))
        app.installTranslator(translator)
    else:
        translator = QtCore.QTranslator()
        translator.load('en.qm', os.path.join(bk._w.plugin_dir, bk._w.plugin_name))

    # display GUI
    window = GUI(bk)
    app.exec()

    # terminate if Cancel button was pressed
    if Cancel == True:
        print(translator.translate('Messages', 'Terminated by user.'), '\n')
        print(translator.translate('Messages', 'Click OK to close.'))
        return -1

    # get parameters
    tag = attribute = value = id = None
    if prefs['tag'] != '':
        tag = re.sub('[<|>| ]', '', prefs['tag']) 
    if prefs['attribute'] != '':
        attribute = prefs['attribute']
    if prefs['value'] != '':
        value = prefs['value']
    if 'id' in prefs:
        id = prefs['id']
    if prefs['prefix'] != '':
        prefix = prefs['prefix']
    counter = int(prefs['counter']) - 1
    initial_counter = counter
    zeros = prefs['zeros']
    roman = prefs['roman']
    title = prefs['title']
    prefix = prefs['prefix']

    # main routine
    if tag is not None or (attribute is not None and value is not None):

        # get selected files
        selected_files = []
        for file_name in list(bk.selected_iter()):
            if bk.id_to_mime(file_name[1]) == 'application/xhtml+xml':
                selected_files.append((file_name[1], bk.id_to_href(file_name[1]))) 

        # get all files
        all_files = list(bk.text_iter())

        if selected_files != []:
            print(translator.translate('Messages', 'Processing selected files...'), '\n')
            file_list = selected_files
        else:
            print(translator.translate('Messages', 'Processing all files...'), '\n')
            file_list = all_files
        
        # process file list
        for (html_id, href) in file_list:
            html = bk.readfile(html_id)
            
            # load html code into BeautifulSoup
            soup = BeautifulSoup(html, 'html.parser')
            orig_soup = str(soup)
            
            # search by tag, attribute and value
            if tag and attribute and value:
                all_tags = soup.find_all(tag, {attribute : value})
            # search by attribute and value
            elif attribute and value:
                all_tags = soup.find_all(attrs={attribute : value})
            # search only by tag
            else:
                all_tags = soup.find_all(tag)

            # process matches
            for index, each_tag in enumerate(all_tags, start = 1):
                counter += 1
                if roman:
                    each_tag['id'] = id + int_to_roman(counter)
                    if title:
                        each_tag['title'] = (prefix + ' ' + int_to_roman(counter)).strip()
                else:
                    if zeros:
                        each_tag['id'] = id + "{0:0>4}".format(counter)
                    else:
                        each_tag['id'] = id + str(counter).strip()
                    if title:
                        each_tag['title'] = (prefix + ' ' + str(counter)).strip()
            
            # update html if the code was changed
            if str(soup) != orig_soup:
                try:
                    bk.writefile(html_id, str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
                except:
                    bk.writefile(html_id, str(soup))
                print(translator.translate('Messages', '{} updated.').format(os.path.basename(href)))
        
        if counter:
            print('\n' + translator.translate('Messages', '{} match(es) found.').format(str(counter - initial_counter)), '\n')
        else:
            print(translator.translate('Messages', 'No matches found.'), '\n')

    # no values entered in dialog box
    else:
        print(translator.translate('Messages', 'No value entered.'), '\n')

    # done
    print(translator.translate('Messages', 'Click OK to close.'))

    # run plugin version check
    if is_connected() and update_check and (datetime.now() - last_time_checked).days >=1:

            href = 'https://www.mobileread.com/forums/showpost.php?p=3184441&postcount=1'
            _latest_pattern = re.compile(r'Current Version:\s*&quot;([^&]*)&')
            plugin_xml_path = os.path.abspath(os.path.join(bk._w.plugin_dir, 'AddIDs', 'plugin.xml'))
            plugin_version = ET.parse(plugin_xml_path).find('.//version').text
            try:
                latest_version = None
                response = urllib.request.urlopen(href)
                m = _latest_pattern.search(response.read().decode('utf-8', 'ignore'))
                if m:
                    latest_version = (m.group(1).strip())
                    if latest_version and latest_version != plugin_version:
                        print('\n*** ' + translator.translate('Messages', 'A newer plugin version is available, version {}.').format(latest_version) + '***')
            except:
                pass

            # update last time checked time stamp
            last_time_checked = str(datetime.now())
            prefs['last_time_checked'] = last_time_checked
            bk.savePrefs(prefs)

    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
