# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2023 DaltonST'
__my_version__ = "1.0.235"  # Decode url encoding; multiple instances of Notes Viewer as a new option

import copy

from qt.core import (Qt, QVBoxLayout, QWidget, QLabel, QLineEdit, QGroupBox,
                                    QDialog, QPushButton, QFont, QScrollArea, QGridLayout,
                                    QCheckBox, QApplication, QHBoxLayout, QSize, QComboBox,
                                    QButtonGroup, QRadioButton, QFrame)

from calibre.constants import DEBUG
from calibre.gui2 import gprefs, error_dialog

from polyglot.builtins import iteritems

TEXT_DEFAULT_WIDTH =  800
TEXT_DEFAULT_HEIGHT = 500
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(500, 300)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        #~ if DEBUG: self.geom = None
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result=None):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class NotesViewerCustomizationDialog(SizePersistedDialog):

    def __init__(self,maingui,nv_state_dict,js_icon,nv_prefs,convert_string_to_dict,return_from_notes_viewer_customization_to_save_prefs):

        parent = None
        unique_pref_name = 'Job_Spy:NotesViewerCustomizationDialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.maingui = maingui
        self.nv_state_dict = nv_state_dict
        self.comments_datatype_list = self.nv_state_dict['comments_datatype_list']
        self.nv_prefs = nv_prefs
        self.prefs_changes_dict = {}
        self.convert_string_to_dict = convert_string_to_dict
        self.return_from_notes_viewer_customization_to_save_prefs = return_from_notes_viewer_customization_to_save_prefs

        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint )
        self.setWindowIcon(js_icon)
        title = "JS+:GUI Tool:  Notes Viewer Customization"
        self.tool_name = title
        self.setWindowTitle(title)
        self.setSizeGripEnabled(True)
        self.setModal(True)
        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(11)
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignLeft)
        self.setLayout(self.layout_top)
        #-----------------------------------------------------
        self.miscellaneous_groupbox = QGroupBox('')
        self.miscellaneous_groupbox.setMaximumWidth(TEXT_DEFAULT_WIDTH)
        self.layout_top.addWidget(self.miscellaneous_groupbox)
        #-----------------------------------------------------
        self.layout_miscellaneous_groupbox = QVBoxLayout()
        self.layout_miscellaneous_groupbox.setSpacing(0)
        self.layout_miscellaneous_groupbox.setAlignment(Qt.AlignLeft)
        self.miscellaneous_groupbox.setLayout(self.layout_miscellaneous_groupbox)
        #-----------------------------------------------------
        self.layout_miscellaneous = QHBoxLayout()
        self.layout_miscellaneous.setAlignment(Qt.AlignLeft)
        self.layout_miscellaneous_groupbox.addLayout(self.layout_miscellaneous)

        self.default_label = QLabel("Default Custom Column: ")
        self.default_label.setFont(font)
        self.default_label.setToolTip("<p style='white-space:wrap'>For the Current Library, the Custom Column with which Notes Viewer will always start.  This Custom Column is unable to be Hidden.")
        self.default_label.setMaximumWidth(300)
        self.default_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_miscellaneous.addWidget(self.default_label)

        self.custom_column_combobox = QComboBox()
        self.custom_column_combobox.setEditable(False)
        self.custom_column_combobox.setFrame(True)
        self.custom_column_combobox.setDuplicatesEnabled(True)
        self.custom_column_combobox.setMaxVisibleItems(25)
        self.custom_column_combobox.setMinimumWidth(300)
        self.custom_column_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.custom_column_combobox.setFont(font)
        self.custom_column_combobox.setPlaceholderText("#Columns")
        self.custom_column_combobox.setToolTip("<p style='white-space:wrap'>For the Current Library, the Custom Column with which Notes Viewer will always start.  This Custom Column is unable to be Hidden.<br><br>Long-Text/Comments Custom Columns.\n\nUse the Arrow Keys to move up and down the List of Custom Columns that may be set to the default for this Library.")
        self.layout_miscellaneous.addWidget(self.custom_column_combobox)

        self.custom_column_combobox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.custom_column_combobox.addItem("[No Default Set]")

        for cc in self.comments_datatype_list:
            self.custom_column_combobox.addItem(cc)
        #END FOR

        self.layout_miscellaneous.addSpacing(10)

        self.default_in_all_libraries_checkbox = QCheckBox("Default for &All Calibre Libraries?")
        self.default_in_all_libraries_checkbox.setFont(font)
        self.default_in_all_libraries_checkbox.setToolTip("<p style='white-space:wrap'>Should this Custom Column also be the Default Custom Column for all Calibre Libraries?\
        <br><br>To check or uncheck this box, use the keyboard spacebar, or click with a mouse.")
        self.default_in_all_libraries_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.layout_miscellaneous.addWidget(self.default_in_all_libraries_checkbox)

        self.default_in_all_libraries_checkbox.setFocus()

        #-----------------------------------------------------

        self.layout_miscellaneous_more = QHBoxLayout()
        self.layout_miscellaneous_more.setAlignment(Qt.AlignCenter)
        self.layout_miscellaneous_groupbox.addLayout(self.layout_miscellaneous_more)

        self.lock_current_custom_column_checkbox = QCheckBox("'Lock' checkbox value at NV Start")
        self.lock_current_custom_column_checkbox.setFont(font)
        self.lock_current_custom_column_checkbox.setToolTip("<p style='white-space:wrap'>'Locked' checked/unchecked state for the current Library at Notes Viewer startup?\
        <br><br>To check or uncheck this box, use the keyboard spacebar, or click with a mouse.")
        self.lock_current_custom_column_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.layout_miscellaneous_more.addWidget(self.lock_current_custom_column_checkbox)

        self.layout_miscellaneous_more.addSpacing(20)

        self.default_state_for_auto_jump_checkbox = QCheckBox("'AutoJump' checkbox value at NV Start")
        self.default_state_for_auto_jump_checkbox.setFont(font)
        self.default_state_for_auto_jump_checkbox.setToolTip("<p style='white-space:wrap'>'AutoJump' checked/unchecked state for the current Library at Notes Viewer startup?\
        <br><br>To check or uncheck this box, use the keyboard spacebar, or click with a mouse.")
        self.default_state_for_auto_jump_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.layout_miscellaneous_more.addWidget(self.default_state_for_auto_jump_checkbox)

        #-----------------------------------------------------

        self.layout_search_results_sorting_params = QHBoxLayout()
        self.layout_search_results_sorting_params.setAlignment(Qt.AlignCenter)
        self.layout_miscellaneous_groupbox.addLayout(self.layout_search_results_sorting_params)

        self.sort_label = QLabel("Search Results Sort Default:")
        self.sort_label.setFont(font)
        #~ self.sort_label.setMaximumWidth(200)
        self.sort_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_search_results_sorting_params.addWidget(self.sort_label)

        self.layout_search_results_sorting_params.addSpacing(20)

        self.sort_title_radio = QRadioButton('Title')  # these text values in .lower() are used when saving preferences; do not translate them.
        self.sort_title_radio.setFont(font)
        self.sort_title_radio.setToolTip("<p style='white-space:wrap'>Default to 'Sort by Title'")
        self.sort_title_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.sort_author_radio = QRadioButton('Author')
        self.sort_author_radio.setFont(font)
        self.sort_author_radio.setToolTip("<p style='white-space:wrap'>Default to 'Sort by Author'")
        self.sort_author_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.sort_bookid_radio = QRadioButton('BookID')
        self.sort_bookid_radio.setFont(font)
        self.sort_bookid_radio.setToolTip("<p style='white-space:wrap'>Default to 'Sort by BookID'")
        self.sort_bookid_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.sort_snippet_radio = QRadioButton('Snippet')
        self.sort_snippet_radio.setFont(font)
        self.sort_snippet_radio.setToolTip("<p style='white-space:wrap'>Default to 'Sort by Snippet'")
        self.sort_snippet_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.sort_qbuttongroup = QButtonGroup(self.layout_search_results_sorting_params)
        self.sort_qbuttongroup.setExclusive(True)

        self.sort_qbuttongroup.addButton(self.sort_title_radio)
        self.sort_qbuttongroup.addButton(self.sort_author_radio)
        self.sort_qbuttongroup.addButton(self.sort_bookid_radio)
        self.sort_qbuttongroup.addButton(self.sort_snippet_radio)

        self.layout_search_results_sorting_params.addWidget(self.sort_title_radio)
        self.layout_search_results_sorting_params.addWidget(self.sort_author_radio)
        self.layout_search_results_sorting_params.addWidget(self.sort_bookid_radio)
        self.layout_search_results_sorting_params.addWidget(self.sort_snippet_radio)

        self.sort_bookid_radio.setChecked(True)

        self.layout_search_results_sorting_params.addSpacing(20)

        self.sort_descending_checkbox = QCheckBox("Sort Descending?")
        self.sort_descending_checkbox.setFont(font)
        self.sort_descending_checkbox.setToolTip("<p style='white-space:wrap'>Sort Search Results (by the Specified Column) Descending instead of Ascending (the Default)?")
        self.sort_descending_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.layout_search_results_sorting_params.addWidget(self.sort_descending_checkbox)

        self.sort_descending_checkbox.setChecked(False)

        #-----------------------------------------------------

        self.layout_multiple_instances = QHBoxLayout()
        self.layout_multiple_instances.setAlignment(Qt.AlignCenter)
        self.layout_miscellaneous_groupbox.addLayout(self.layout_multiple_instances)

        self.instances_label = QLabel("NV Instances Simultaneously:")
        self.instances_label.setFont(font)
        #~ self.instances_label.setMaximumWidth(200)
        self.instances_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_multiple_instances.addWidget(self.instances_label)

        self.instances_1_radio = QRadioButton('1')  # these text values in .lower() are used when saving preferences; do not translate them.
        self.instances_1_radio.setFont(font)
        self.instances_1_radio.setToolTip("<p style='white-space:wrap'>Allow only 1 instance of Notes Viewer to exist.")
        self.instances_1_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.instances_2_radio = QRadioButton('2')
        self.instances_2_radio.setFont(font)
        self.instances_2_radio.setToolTip("<p style='white-space:wrap'>Allow up to 2 instances of Notes Viewer open simultaneously.")
        self.instances_2_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.instances_3_radio = QRadioButton('3')
        self.instances_3_radio.setFont(font)
        self.instances_3_radio.setToolTip("<p style='white-space:wrap'>Allow up to 3 instances of Notes Viewer open simultaneously.")
        self.instances_3_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.instances_qbuttongroup = QButtonGroup(self.layout_multiple_instances)
        self.instances_qbuttongroup.setExclusive(True)

        self.instances_qbuttongroup.addButton(self.instances_1_radio)
        self.instances_qbuttongroup.addButton(self.instances_2_radio)
        self.instances_qbuttongroup.addButton(self.instances_3_radio)

        self.layout_multiple_instances.addWidget(self.instances_1_radio)
        self.layout_multiple_instances.addWidget(self.instances_2_radio)
        self.layout_multiple_instances.addWidget(self.instances_3_radio)

        self.instances_1_radio.setChecked(True)

        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------

        self.layout_top.addSpacing(2)

        self.frame_1 = QFrame()
        self.frame_1.setFrameShape(QFrame.HLine)
        self.frame_1.setFrameShadow(QFrame.Sunken)
        self.frame_1.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_1)

        self.layout_top.addSpacing(2)

        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignLeft)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(TEXT_DEFAULT_WIDTH,TEXT_DEFAULT_HEIGHT)

        self.layout_top.addWidget(self.scroll_area_frame)
        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_top.addWidget(self.scroll_widget)
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setSpacing(0)
        self.layout_frame.setAlignment(Qt.AlignCenter)

        self.scroll_widget.setLayout(self.layout_frame)
        #-----------------------------------------------------
        self.custom_columns_groupbox = QGroupBox('Long-Text/Comments Custom Columns:')
        self.custom_columns_groupbox.setMaximumWidth(TEXT_DEFAULT_WIDTH)
        self.custom_columns_groupbox.setToolTip("<p style='white-space:wrap'>Customize each Custom Column to behave as you wish within Notes Viewer.'")
        self.layout_frame.addWidget(self.custom_columns_groupbox)

        self.layout_custom_columns_groupbox = QVBoxLayout()
        self.layout_custom_columns_groupbox.setSpacing(0)
        self.layout_custom_columns_groupbox.setAlignment(Qt.AlignLeft)
        self.custom_columns_groupbox.setLayout(self.layout_custom_columns_groupbox)

        self.layout_frame_pushbuttons = QHBoxLayout()
        self.layout_frame_pushbuttons.setSpacing(0)
        self.layout_frame_pushbuttons.setAlignment(Qt.AlignLeft)
        self.layout_custom_columns_groupbox.addLayout(self.layout_frame_pushbuttons)

        self.clear_all_checkboxes_pushbutton = QPushButton("&Show All", self)
        self.clear_all_checkboxes_pushbutton.clicked.connect(self.clear_all_checkboxes)
        self.clear_all_checkboxes_pushbutton.setMaximumWidth(300)
        self.layout_frame_pushbuttons.addWidget(self.clear_all_checkboxes_pushbutton)

        self.check_all_checkboxes_pushbutton = QPushButton("&Hide All", self)
        self.check_all_checkboxes_pushbutton.clicked.connect(self.check_all_checkboxes)
        self.check_all_checkboxes_pushbutton.setMaximumWidth(300)
        self.layout_frame_pushbuttons.addWidget(self.check_all_checkboxes_pushbutton)

        self.layout_frame_pushbuttons.addSpacing(247)

        self.column_header_label = QLabel("Always Use a Specific Text Format")
        self.column_header_label.setFont(font)
        self.column_header_label.setMinimumWidth(300)
        self.column_header_label.setMaximumWidth(300)
        self.column_header_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_frame_pushbuttons.addWidget(self.column_header_label)
        #-----------------------------------------------------
        self.custom_columns_layout = QGridLayout()
        self.layout_custom_columns_groupbox.addLayout(self.custom_columns_layout)
        #-----------------------------------------------------
        #-----------------------------------------------------
        font.setPointSize(8)
        font.setBold(True)
        #-----------------------------------------------------
        #-----------------------------------------------------
        is_valid,error_msg = self.create_custom_column_choices(font)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.layout_frame.addSpacing(10)

        self.layout_save_pushbuttons = QHBoxLayout()
        self.layout_save_pushbuttons.setSpacing(0)
        self.layout_save_pushbuttons.setAlignment(Qt.AlignCenter)
        self.layout_frame.addLayout(self.layout_save_pushbuttons)

        self.exit_pushbutton = QPushButton("Save && E&xit", self)
        self.exit_pushbutton.setDefault(True)
        self.exit_pushbutton.clicked.connect(self.exit_dialog)
        self.exit_pushbutton.setMaximumWidth(300)
        self.layout_save_pushbuttons.addWidget(self.exit_pushbutton)

        self.resize_dialog()

        if not is_valid:
            error_dialog(self.maingui, _(self.tool_name),_(error_msg), show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_custom_column_choices(self,font):

        library_options_dict = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_CUSTOMIZATION_BY_LIBRARY']

        library_options_dict,is_valid = self.convert_string_to_dict(library_options_dict)

        if not is_valid or not(isinstance(library_options_dict,dict)):
            msg = "Preferences dictionary loaded, but it is invalid.  Cannot use it."
            return msg

        self.library_options_dict = library_options_dict

        self.library_id = self.maingui.current_db.new_api.backend.library_id
        if DEBUG: print("library_id: ", self.library_id)
        if self.library_id in self.library_options_dict:
            library_val = self.library_options_dict[self.library_id]
            if len(library_val) == 7:
                cc_settings_dict, default_column, lock_state, autojump_state, sort_column, sort_descending, max_instances_allowed = library_val
                if DEBUG: print("Current Version of library_val found...")  # , str(library_val))
                if DEBUG: print("lock_state,autojump_state,sort_column, sort_descending as saved previously: ", lock_state, autojump_state, sort_column, sort_descending, max_instances_allowed)
            elif len(library_val) == 6:
                max_instances_allowed = "1"
                cc_settings_dict, default_column, lock_state, autojump_state, sort_column, sort_descending = library_val
                if DEBUG: print("Prior Version of library_val found...")  # , str(library_val))
                if DEBUG: print("lock_state,autojump_state,sort_column, sort_descending as saved previously: ", lock_state, autojump_state, sort_column, sort_descending)
            elif len(library_val) == 4:
                cc_settings_dict, default_column, lock_state, autojump_state = library_val             # current:   cc_settings_dict, default_column, lock_state, autojump_state = library_val
                sort_column = None
                sort_descending = False
                max_instances_allowed = "1"
                if DEBUG: print("\n\nOlder Current library_val found: ", str(library_val),"\n\n")
                if DEBUG: print("lock_state and autojump_state as saved previously: ", lock_state, autojump_state)
            else:
                cc_settings_dict, default_column = library_val                                                         # legacy:     cc_settings_dict, default_column = library_val
                lock_state = False
                autojump_state = False
                sort_column = None
                sort_descending = False
                max_instances_allowed = "1"
                if DEBUG: print("\n\nLegacy library_val found: ", str(library_val),"\n\n")
                if DEBUG: print("No lock_state and autojump_state were saved previously; defaulted to False: ", lock_state, autojump_state)
        else:
            cc_settings_dict = {}
            default_column = None
            lock_state = False
            autojump_state = False
            sort_column = None
            sort_descending = False
            max_instances_allowed = 1
            if DEBUG: print("\n\nNo library_val found whatsoever...everything defaulted to None or False.\n\n")

        if default_column is not None:
            if default_column in self.comments_datatype_list:
                self.custom_column_combobox.setCurrentText(default_column)

        #~ ---------------------------------
        self.lock_current_custom_column_checkbox.setChecked(lock_state)
        self.default_state_for_auto_jump_checkbox.setChecked(autojump_state)
        self.sort_descending_checkbox.setChecked(sort_descending)
        if sort_column is None:
            self.sort_bookid_radio.setChecked(True)
        elif sort_column == "title":
            self.sort_title_radio.setChecked(True)
        elif sort_column == "author":
            self.sort_author_radio.setChecked(True)
        elif sort_column == "bookid":
            self.sort_bookid_radio.setChecked(True)
        elif sort_column == "snippet":
            self.sort_snippet_radio.setChecked(True)
        else:
            self.sort_bookid_radio.setChecked(True)
        #~ ---------------------------------
        if max_instances_allowed == "1":
            self.instances_1_radio.setChecked(True)
        elif max_instances_allowed == "2":
            self.instances_2_radio.setChecked(True)
        elif max_instances_allowed == "3":
            self.instances_3_radio.setChecked(True)
        else:
            max_instances_allowed = "1"
            self.instances_1_radio.setChecked(True)

        #~ ---------------------------------

        self.qt_qcheckbox_objects_dict = {}
        self.qt_qradio_plain_objects_dict = {}
        self.qt_qradio_md_objects_dict = {}
        self.qt_qradio_html_objects_dict = {}
        self.qt_qradio_guess_objects_dict = {}

        font.setPointSize(10)

        r = 0

        try:
            for cc in self.comments_datatype_list:
                try:
                    option_qlabel = QLabel(cc)
                    option_qlabel.setFont(font)
                    option_qlabel.setToolTip("<p style='white-space:wrap'>Custom Column to customize for Notes Viewer")
                    option_qlabel.setEnabled(False)
                    option_qlabel.setMinimumWidth(300)
                    option_qlabel.setMaximumWidth(300)
                except  Exception as e:
                    msg = "option_qlabel error: " + str(e)
                    if DEBUG: print(msg)
                    return False,msg

                try:
                    option_qcheckbox = QCheckBox("Hide?")
                    option_qcheckbox.setFont(font)
                    option_qcheckbox.setTristate(False)
                    option_qcheckbox.setToolTip("<p style='white-space:wrap'>Hide this Custom Column from Notes Viewer?<br><br>To check or uncheck this box, use the keyboard spacebar, or click with a mouse. <br><br>Use the tab button and arrow keys to move elsewhere.")
                except  Exception as e:
                    msg = "option_qcheckbox error: " + str(e)
                    if DEBUG: print(msg)
                    return False,msg

                try:
                    option_hlayout = QHBoxLayout()
                    option_hlayout.setAlignment(Qt.AlignLeft)
                    option_hlayout.setSpacing(30)

                    option_plain_radio = QRadioButton('Plain')
                    option_plain_radio.setToolTip("<p style='white-space:wrap'>Do not 'guess'; assume the Custom Column text is always:  Plain.<br><br>To check or uncheck this button, use the keyboard spacebar, or click with a mouse. <br><br>Use the tab button and arrow keys to move elsewhere.")
                    option_md_radio = QRadioButton('Markdown')
                    option_md_radio.setToolTip("<p style='white-space:wrap'>Do not 'guess'; assume the Custom Column text is always:  Markdown.<br><br>To check or uncheck this button, use the keyboard spacebar, or click with a mouse. <br><br>Use the tab button and arrow keys to move elsewhere.")
                    option_html_radio = QRadioButton('HTML')
                    option_html_radio.setToolTip("<p style='white-space:wrap'>Do not 'guess'; assume the Custom Column text is always:  HTML.<br><br>To check or uncheck this button, use the keyboard spacebar, or click with a mouse. <br><br>Use the tab button and arrow keys to move elsewhere.")
                    option_guess_radio = QRadioButton('Guess')
                    option_guess_radio.setToolTip("<p style='white-space:wrap'>'Guess'; assume the Custom Column text has a random text format.<br><br>To check or uncheck this button, use the keyboard spacebar, or click with a mouse. <br><br>Use the tab button and arrow keys to move elsewhere.")

                    option_plain_radio.setFont(font)
                    option_md_radio.setFont(font)
                    option_html_radio.setFont(font)
                    option_guess_radio.setFont(font)

                    option_qbuttongroup = QButtonGroup(option_hlayout)
                    option_qbuttongroup.setExclusive(True)

                    option_qbuttongroup.addButton(option_plain_radio)
                    option_qbuttongroup.addButton(option_md_radio)
                    option_qbuttongroup.addButton(option_html_radio)
                    option_qbuttongroup.addButton(option_guess_radio)

                except  Exception as e:
                    msg = "option_qbuttongroup error: " + str(e)
                    if DEBUG: print(msg)
                    return False,msg

                option_hlayout.addWidget(option_qcheckbox)
                option_hlayout.addWidget(option_plain_radio)
                option_hlayout.addWidget(option_md_radio)
                option_hlayout.addWidget(option_html_radio)
                option_hlayout.addWidget(option_guess_radio)

                #~ ---------------------------------------------------------------------
                if not cc in cc_settings_dict:  #  cc_settings_dict = library_options_dict[self.library_id]
                    pass
                else:
                    settings_dict = cc_settings_dict[cc]
                    qcb_state = settings_dict['OPTION_QCHECKBOX']   # qcb_state = option_qcheckbox.isChecked()
                    qpr_state = settings_dict['OPTION_PLAIN_RADIO']
                    qmr_state = settings_dict['OPTION_MD_RADIO']
                    qhr_state = settings_dict['OPTION_HTML_RADIO']
                    qgr_state = settings_dict['OPTION_GUESS_RADIO']

                    option_qcheckbox.setChecked(bool(qcb_state))
                    option_plain_radio.setChecked(bool(qpr_state))
                    option_md_radio.setChecked(bool(qmr_state))
                    option_html_radio.setChecked(bool(qhr_state))
                    option_guess_radio.setChecked(bool(qgr_state))

                    if not option_plain_radio.isChecked():
                        if not option_md_radio.isChecked():
                            if not option_html_radio.isChecked():
                                option_guess_radio.setChecked(True)
                #~ ---------------------------------------------------------------------

                self.custom_columns_layout.addWidget(option_qlabel,r,0)
                self.custom_columns_layout.addLayout(option_hlayout,r,2)

                self.qt_qcheckbox_objects_dict[cc]         =    option_qcheckbox
                self.qt_qradio_plain_objects_dict[cc]      =    option_plain_radio
                self.qt_qradio_md_objects_dict[cc]         =    option_md_radio
                self.qt_qradio_html_objects_dict[cc]       =    option_html_radio
                self.qt_qradio_guess_objects_dict[cc]      =    option_guess_radio

                r = r + 1

            #END FOR
        except  Exception as e:
            if DEBUG: print("create_custom_column_choices -- error: ", str(e))
            return False,str(e)

        self.update()
        QApplication.instance().processEvents()

        return True,None
    #---------------------------------------------------------------------------------------------------------------------------------------
    def clear_all_checkboxes(self):
        for cc,option_qcheckbox in iteritems(self.qt_qcheckbox_objects_dict):
            option_qcheckbox.setChecked(False)
            option_qcheckbox.update()
        #END FOR
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def check_all_checkboxes(self):
        for cc,option_qcheckbox in iteritems(self.qt_qcheckbox_objects_dict):
            option_qcheckbox.setChecked(True)
            option_qcheckbox.update()
        #END FOR
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_all_customization_settings(self):

        if DEBUG: print("library_id: ", self.library_id)
        cc_settings_dict = {}

        default_column = self.custom_column_combobox.currentText()  # cannot be hidden...
        n_total_columns = len(self.comments_datatype_list)
        n_total_hidden = 0

        for custom_column in self.comments_datatype_list:

            settings_dict = {}

            option_qcheckbox = self.qt_qcheckbox_objects_dict[custom_column]
            if custom_column == default_column:
                option_qcheckbox.setChecked(False)
            qcb_state = option_qcheckbox.isChecked()
            settings_dict['OPTION_QCHECKBOX'] = qcb_state

            option_plain_radio = self.qt_qradio_plain_objects_dict[custom_column]
            qrp_state = option_plain_radio.isChecked()
            settings_dict['OPTION_PLAIN_RADIO'] = qrp_state

            option_md_radio = self.qt_qradio_md_objects_dict[custom_column]
            qrm_state = option_md_radio.isChecked()
            settings_dict['OPTION_MD_RADIO'] = qrm_state

            option_html_radio = self.qt_qradio_html_objects_dict[custom_column]
            qrh_state = option_html_radio.isChecked()
            settings_dict['OPTION_HTML_RADIO'] = qrh_state

            option_guess_radio = self.qt_qradio_guess_objects_dict[custom_column]
            qgr_state = option_guess_radio.isChecked()
            settings_dict['OPTION_GUESS_RADIO'] = qgr_state

            cc_settings_dict[custom_column] = settings_dict

            del settings_dict
        #END FOR

        if n_total_hidden == n_total_columns:
            error_msg = "ERROR: You cannot hide All Custom Columns.<br><br>Nor can you hide the Default Custom Column if it has been specified."
            error_dialog(self.maingui, _(self.tool_name),_(error_msg), show=True)
            return False

        lock_state = self.lock_current_custom_column_checkbox.isChecked()
        autojump_state  = self.default_state_for_auto_jump_checkbox.isChecked()

        checked_button = self.sort_qbuttongroup.checkedButton()
        if checked_button is None:
            self.sort_bookid_radio.setChecked(True)
            checked_button = self.sort_bookid_radio
        text = checked_button.text()
        sort_column = text.replace("&","")
        sort_column = sort_column.lower()

        sort_descending = self.sort_descending_checkbox.isChecked()

        is_one = self.instances_1_radio.isChecked()
        is_two = self.instances_2_radio.isChecked()
        is_three = self.instances_3_radio.isChecked()
        if is_one:
            max_instances_allowed = 1
        elif is_two:
            max_instances_allowed = 2
        elif is_three:
            max_instances_allowed = 3
        else:
            max_instances_allowed = 1
        max_instances_allowed = str(max_instances_allowed)

        self.library_options_dict[self.library_id] = cc_settings_dict, default_column, lock_state, autojump_state, sort_column, sort_descending, max_instances_allowed

        k = 'GUI_TOOLS_NOTES_VIEWER_CUSTOMIZATION_BY_LIBRARY'
        v = str(self.library_options_dict)
        self.prefs_changes_dict[k] = v

        if self.default_in_all_libraries_checkbox.isChecked() and default_column.startswith("#"):
            k = 'GUI_TOOLS_NOTES_VIEWER_DEFAULT_CUSTOM_COLUMN_GLOBAL'
            v = default_column
        else:
            k = 'GUI_TOOLS_NOTES_VIEWER_DEFAULT_CUSTOM_COLUMN_GLOBAL'
            v = ""
        self.prefs_changes_dict[k] = v

        return True
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def exit_dialog(self):
        is_saveable = self.save_all_customization_settings()
        if not is_saveable:
            return
        self.dialog_closing()
        self.return_from_notes_viewer_customization_to_save_prefs(self.prefs_changes_dict)
        self.close()
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------

#~ # end of ris_tags_custom_columns_select_dialog.py