# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.210"  #Enhancements to GUI Tool 'Copy User Category to Target Library'.

import os,sys,apsw,ast,copy,json

from qt.core import Qt, QApplication, QDialog, QVBoxLayout, QFont, QWidget, QSize, QPushButton, QComboBox, QCheckBox

from calibre import isbytestring
from calibre.constants import preferred_encoding,filesystem_encoding, DEBUG
from calibre.gui2 import gprefs, question_dialog, info_dialog, error_dialog
from calibre.utils.config_base import to_json, from_json
from calibre.utils.serialize import json_dumps, json_loads

from polyglot.builtins import as_unicode, iteritems, unicode_type

from calibre_plugins.job_spy.config import prefs

ADD = 'add'
CHANGE = 'change'

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(100, 100)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
            self.save_dialog_geometry()
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class CopyUserCategoriesDialog(SizePersistedDialog):

    def __init__(self,gui,icon,source_user_cat_dict,source_field_metadata_dict,target_library_list):

        parent = gui
        unique_pref_name = 'job_spy:copy_user_categories_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.gui = gui

        self.source_user_cat_dict = source_user_cat_dict
        self.source_field_metadata_dict = source_field_metadata_dict
        self.target_library_list = target_library_list

        self.setWindowTitle('JS+ GUI Tool: Copy User Category to Target Library')
        self.setWindowIcon(icon)

        self.layout = QVBoxLayout()
        self.layout.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout)

        self.setToolTip("<p style='white-space:wrap'>Frequently back up your metadata.db file of your Target Library.")

        font = QFont()
        font.setBold(False)
        font.setPointSize(11)

        self.user_categories_combobox = QComboBox()
        self.user_categories_combobox.setMaximumWidth(500)
        self.user_categories_combobox.setEditable(False)
        self.user_categories_combobox.setFont(font)
        self.user_categories_combobox.setToolTip("<p style='white-space:wrap'>These are the available User Categories in your Source (Current) Library.")
        self.layout.addWidget(self.user_categories_combobox)

        self.user_categories_combobox.setObjectName("user_categories_combobox")   # required for .findChildren()

        self.user_categories_combobox.addItem('Choose a User Category')

        self.user_cat_list = []

        #~ for k,v in self.source_user_cat_dict.iteritems():
        for k,v in iteritems(self.source_user_cat_dict):
            self.user_cat_list.append(k)
        #END FOR

        self.user_cat_list.sort()

        for cat in self.user_cat_list:
            self.user_categories_combobox.addItem(cat)
        #END FOR

        self.user_categories_combobox.setMaxVisibleItems(len(self.source_user_cat_dict))

        lastused = prefs['GUI_TOOLS_USER_CATEGORIES_COPY_LAST_SOURCE_USER_CAT']
        i = self.user_categories_combobox.findText(lastused,Qt.MatchExactly)
        if i >= 0:
            self.user_categories_combobox.setCurrentIndex(i)

        self.layout.addStretch(1)

        self.target_libraries_combobox = QComboBox()
        self.target_libraries_combobox.setMaximumWidth(500)
        self.target_libraries_combobox.setEditable(False)
        self.target_libraries_combobox.setFont(font)
        self.target_libraries_combobox.setToolTip("<p style='white-space:wrap'>These are all of your 'remembered' Calibre Libraries that you may select from to indicate which will be your Target Library.")
        self.layout.addWidget(self.target_libraries_combobox)

        self.target_libraries_combobox.setObjectName("target_libraries_combobox")   # required for .findChildren()

        self.target_libraries_combobox.addItem('Choose a Target Library')
        for lib in self.target_library_list:
            self.target_libraries_combobox.addItem(lib)
        #END FOR

        lastused = prefs['GUI_TOOLS_USER_CATEGORIES_COPY_LAST_TARGET_LIBRARY']
        i = self.target_libraries_combobox.findText(lastused,Qt.MatchExactly)
        if i >= 0:
            self.target_libraries_combobox.setCurrentIndex(i)

        self.layout.addStretch(4)

        self.push_button_copy_user_categories = QPushButton("Copy the Selected User Category to the Selected Target Library", self)
        self.push_button_copy_user_categories.setMaximumWidth(500)
        self.push_button_copy_user_categories.clicked.connect(self.copy_user_categories)
        self.push_button_copy_user_categories.setFont(font)
        self.push_button_copy_user_categories.setDefault(False)
        self.push_button_copy_user_categories.setToolTip("<p style='white-space:wrap'>This pushbutton will start the copy process.  It will also save the current size and location of this tool's window.")
        self.layout.addWidget(self.push_button_copy_user_categories)

        self.layout.addStretch(1)

        self.push_button_copy_user_categories = QPushButton("Copy All User Categories to the Selected Target Library", self)
        self.push_button_copy_user_categories.setMaximumWidth(500)
        self.push_button_copy_user_categories.clicked.connect(self.copy_all_user_categories)
        self.push_button_copy_user_categories.setFont(font)
        self.push_button_copy_user_categories.setDefault(False)
        self.push_button_copy_user_categories.setToolTip("<p style='white-space:wrap'>This pushbutton will start the copy process.  It will also save the current size and location of this tool's window.")
        self.layout.addWidget(self.push_button_copy_user_categories)

        self.skip_questions_messages_checkbox = QCheckBox("Copy All: Suppress Questions && Information Messages?")
        self.skip_questions_messages_checkbox.setToolTip("<p style='white-space:wrap'>Do not offer any questions and messages during the mass-copying operation (only)?")
        self.layout.addWidget(self.skip_questions_messages_checkbox)

        self.suppress_questions_messages = False

        self.layout.addStretch(3)
        #------------------------------------------------------------------------------------------
        self.push_button_cancel = QPushButton("Exit")
        self.push_button_cancel.setMaximumWidth(500)
        self.push_button_cancel.clicked.connect(self.reject)
        self.push_button_cancel.setDefault(True)
        self.push_button_cancel.setToolTip("<p style='white-space:wrap'>Exit.")
        self.layout.addWidget(self.push_button_cancel)
        #------------------------------------------------------------------------------------------
        self.resize_dialog()
        self.save_dialog_geometry()
    #-----------------------------------------------------------------------------------------
    def copy_all_user_categories(self):

        self.suppress_questions_messages = self.skip_questions_messages_checkbox.isChecked()
        all_user_categories_list = []
        all_user_categories_index_dict = {}
        max = self.user_categories_combobox.count()
        i = 0
        last_usercat = None
        while i < max:
            r = self.user_categories_combobox.itemText(i)
            if r.startswith("Choose"):
                pass
            else:
                all_user_categories_list.append(r)
                all_user_categories_index_dict[r] = i
            i = i + 1
        #END WHILE

        selected_target = self.target_libraries_combobox.currentText()
        if selected_target == 'Choose a Target Library':
            return error_dialog(self.gui, _('JS+ GUI Tool: Copy All User Categories to Target Library'),_('Choose a Target Library.'), show=True)

        if isbytestring(selected_target):
            selected_target = selected_target.decode(filesystem_encoding)
        selected_target = selected_target.replace(os.sep, '/')
        if not os.path.exists(selected_target):
            return error_dialog(self.gui, _('JS+ GUI Tool: Copy All User Categories to Target Library'),_('Sorry, but the selected Target Library is not a valid directory path.'), show=True)

        prefs['GUI_TOOLS_USER_CATEGORIES_COPY_LAST_TARGET_LIBRARY'] = selected_target
        prefs

        self.selected_target = selected_target

        category_results_dict = {}

        if not self.suppress_questions_messages:
            do_question = True
        else:
            do_question = False

        can_continue = True
        if do_question:
            if question_dialog(self.gui, "JS+ GUI Tool: Copy All User Categories to Target Library", "Copy All Qualified User Categories to the Selected Target Library?"):
                can_continue = True
            else:
                can_continue = False

        msg = ""

        if not can_continue:
            msg = "Nothing Done."
            if DEBUG: print(msg)
            return   info_dialog(self.gui, "JS+ GUI Tool: All User Categories to Target Library", msg).show()
        else:
            for usercat in all_user_categories_list:
                i = all_user_categories_index_dict[usercat]
                self.user_categories_combobox.setCurrentIndex(i)
                self.user_categories_combobox.update()
                QApplication.instance().processEvents()
                selected_usercat = self.user_categories_combobox.currentText()
                if not usercat == selected_usercat:
                    if DEBUG: print("Program Error: not usercat == selected_usercat ", usercat, selected_usercat)
                    continue
                prefs['GUI_TOOLS_USER_CATEGORIES_COPY_LAST_SOURCE_USER_CAT'] = usercat
                prefs
                okay,error_msg = self.copy_control(usercat,selected_target)
                if not okay:
                    category_results_dict[usercat] = error_msg
                    if DEBUG: print(usercat, "  ", error_msg)
                else:
                    msg = "Successfully Copied"
                    category_results_dict[usercat] = msg
                    if DEBUG: print(usercat, "  ", msg)
            #END FOR

            s = ""
            for usercat,msg in category_results_dict.items():
                s = s + usercat + ": " + msg + "\n"
            #END FOR
            s = s[0:400]  #could overextend the message box below the bottom of the window, so bottom buttons cannot be seen at all.
            if DEBUG: print(s)
            info_dialog(self.gui, "JS+ GUI Tool: All User Categories to Target Library", s).show()
            del category_results_dict
            del all_user_categories_list
    #-----------------------------------------------------------------------------------------
    def copy_user_categories(self):

        self.save_dialog_geometry()

        selected_usercat = self.user_categories_combobox.currentText()
        if selected_usercat == 'Choose a User Category':
            return error_dialog(self.gui, _('JS+ GUI Tool: Copy User Category to Target Library'),_('Choose a User Category.'), show=True)

        selected_target = self.target_libraries_combobox.currentText()
        if selected_target == 'Choose a Target Library':
            return error_dialog(self.gui, _('JS+ GUI Tool: Copy User Category to Target Library'),_('Choose a Target Library.'), show=True)

        if isbytestring(selected_target):
            selected_target = selected_target.decode(filesystem_encoding)
        selected_target = selected_target.replace(os.sep, '/')
        if not os.path.exists(selected_target):
            return error_dialog(self.gui, _('JS+ GUI Tool: Copy User Category to Target Library'),_('Sorry, but the selected Target Library is not a valid directory path.'), show=True)

        prefs['GUI_TOOLS_USER_CATEGORIES_COPY_LAST_SOURCE_USER_CAT'] = selected_usercat
        prefs['GUI_TOOLS_USER_CATEGORIES_COPY_LAST_TARGET_LIBRARY'] = selected_target
        prefs

        self.selected_target = selected_target

        if question_dialog(self.gui, "JS+ GUI Tool: Copy User Category to Target Library", "Copy the Selected User Category to the Selected Target Library?"):
            okay,error_msg = self.copy_control(selected_usercat,selected_target)
            if okay:
                msg = "Copy was Successful: <br><br>" + selected_usercat + "<br>to:<br>" + selected_target
                msg = msg + "<br><br>Recommended: 'Quick Switch' to the Target Library and confirm the validity of the copied UC."
                info_dialog(self.gui, "JS+ GUI Tool: Copy User Category to Target Library", msg).show()
                if DEBUG: print(msg)
                return
            else:
                if DEBUG: print(error_msg)
                return error_dialog(self.gui, _('JS+ GUI Tool: Copy User Category to Target Library'),_(error_msg), show=True)
        else:
            msg = "Nothing Done."
            info_dialog(self.gui, "JS+ GUI Tool: Copy User Category to Target Library", msg).show()
            if DEBUG: print(msg)
            return
    #-----------------------------------------------------------------------------------------
    def copy_control(self,selected_usercat,selected_target):

        self.selected_usercat = selected_usercat

        okay = True
        error_msg = ""

        dbconn_error,target_fm_source_error = self.get_target_preference_key_data(selected_target)
        if dbconn_error:
            okay = False
            error_msg = "Database Connection Error.  Cannot Connect to the Target Library."
            return okay,error_msg
        if target_fm_source_error:
            okay = False
            error_msg = "The Target Library has no prerequisite 'field metadata' (yet) to allow User Categories to be copied into it.<br><br>Switch to the Target Library, review it, and then restart Calibre.  Switch back to the Source Library, and retry copying the UC to the new Target Library.  "
            return okay,error_msg

        okay,error_msg = self.update_target_control(selected_usercat)

        try:
            del selected_usercat
            del selected_target
        except:
            pass

        self.delete_old_objects()

        return okay,error_msg
    #-----------------------------------------------------------------------------------------
    def delete_old_objects(self):
        try:
            del self.original_target_user_categories_dict
            del self.original_target_field_metadata_dict
        except:
            pass
        try:
            del self.revised_target_user_categories_dict
            del self.revised_target_field_metadata_dict
        except:
            pass
        try:
            del self.uc_comparison_was_successful
            del self.target_fm_changes_required
        except:
            pass
    #-----------------------------------------------------------------------------------------
    def get_target_preference_key_data(self,selected_target):
    #-----------------------------------------------------------------------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_target_library(selected_target)
        if not is_valid:
             error_dialog(self.gui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Target Library.'), show=True)
             return True,False
        #-------------------------------------
        mysql = "SELECT key,val FROM preferences WHERE key = 'field_metadata' "
        my_cursor.execute(mysql)
        target_field_metadata_list = my_cursor.fetchall()
        self.target_fm_exists = False   # on a brand new library created by Calibre from its default structure, and with no books or any other configuration, FM will not exist in table preferences...
        if not target_field_metadata_list:
            target_field_metadata_list = []
        for row in target_field_metadata_list:
            key,val = row     # key = 'field_metadata'
            val = self.json__raw_to_object(val)
            val = as_unicode(val)
            self.original_target_field_metadata_dict = ast.literal_eval(val)
            self.target_fm_exists = True
            break
        #END FOR

        if not self.target_fm_exists:
            my_db.close()
            return False,True
        #-------------------------------------
        mysql = "SELECT key,val FROM preferences WHERE key = 'user_categories' "
        my_cursor.execute(mysql)
        target_user_cat_uc_list = my_cursor.fetchall()   #all of them...not just selected_usercat...original structure which is a list with dicts...
        if not target_user_cat_uc_list:
            target_user_cat_uc_list = []
        for row in target_user_cat_uc_list:
            key,val = row     # key = 'user_categories'
            val = self.json__raw_to_object(val)
            val = as_unicode(val)
            self.original_target_user_categories_dict = ast.literal_eval(val)
            break
        #END FOR
        #-------------------------------------
        #-------------------------------------
        #need the target custom columns to ensure source cc's in the source user category exist in the target...
        mysql = "SELECT label,datatype FROM custom_columns"
        my_cursor.execute(mysql)
        target_custom_column_list = my_cursor.fetchall()
        if not target_custom_column_list:
            target_custom_column_list = []
        self.target_custom_column_dict = {}
        for row in target_custom_column_list:
            label,datatype = row
            if datatype == "comments" or datatype == "composite":   # invalid datatypes for user categories.
                continue
            self.target_custom_column_dict[label] = datatype
        #END FOR
        del target_custom_column_list
        #-------------------------------------
        my_db.close()

        return False,False
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def update_target_control(self,selected_usercat):
        okay = True
        error_msg = ""
        #-------------------------------
        uc_preexists,uc_action_required,okay,error_msg = self.synthesize_revised_target_user_category_key_value(selected_usercat)
        if not okay:
            okay = False
            return okay,error_msg
        #-------------------------------
        if uc_preexists:  #to preexist, then the FM *must* already have its value within it; nothing to do to FM...only UC possibly needed revision...
            pass
        else:
            okay,error_msg = self.synthesize_revised_target_field_metadata_key_value(selected_usercat)
            if not okay:
                okay = False
                return okay,error_msg
        #-------------------------------
        if not self.uc_comparison_was_successful:  # double-check...
            okay = False
            error_msg = "[UTC1] Program Error: self.uc_comparison_was_successful is ambiguous.  Copy Aborted."
            return okay,error_msg
        #-------------------------------
        msg = "Please review the following facts about the requested copy and then indicate whether to proceed or not."
        msg = msg + "<br><br>Selected User Category:                                     " + selected_usercat
        msg = msg + "<br><br>The UC already exists in the Target Library:        " + as_unicode(uc_preexists)
        msg = msg + "<br><br>The action required for the Target UC:               " + uc_action_required
        msg = msg + "<br><br>Any #names in the UC also exist in the Target?  " + as_unicode(self.uc_comparison_was_successful)  # just a teaching factlet for the user...
        msg = msg + "<br><br><br>Do you wish to continue with the actual copy?"

        if not self.suppress_questions_messages:
            if question_dialog(self.gui, "JS+ GUI Tool: Copy User Category to Target Library", msg):
                pass
            else:
                msg = "Nothing Done."
                info_dialog(self.gui, "JS+ GUI Tool: Copy User Category to Target Library", msg).show()
                okay = True
                error_msg = ""
                return okay,error_msg
        #-------------------------------
        okay,error_msg = self.do_actual_updates_of_target_table_preferences()
        return okay,error_msg
    #-----------------------------------------------------------------------------------------
    def synthesize_revised_target_user_category_key_value(self,selected_usercat):
        uc_action_required = 'none'
        uc_preexists = False
        okay = True
        error_msg = ""
        self.target_fm_changes_required = None

        #~ ----------------------------------------
        #~ self.source_user_cat_dict
        #~ self.source_field_metadata_dict
        #~ self.original_target_field_metadata_dict
        #~ self.original_target_user_categories_dict
        #~ self.revised_target_field_metadata_dict
        #~ self.revised_target_user_categories_dict
        #~ ----------------------------------------

        uc_preexists,okay,error_msg = self.compare_existing_target_user_category_to_source(selected_usercat)

        if not okay:
            uc_preexists = True
            uc_action_required = 'none'
            okay = False
            if error_msg == "":
                error_msg = "[SRTUCKV1] Program Error in self.compare_existing_target_user_category_to_source: "
            return uc_preexists,uc_action_required,okay,error_msg
        elif uc_preexists:
                uc_action_required = CHANGE
        else:
            uc_action_required = ADD

        okay,error_msg = self.edit_target_user_category_preferences_val(selected_usercat,uc_action_required)

        return uc_preexists,uc_action_required,okay,error_msg
    #-----------------------------------------------------------------------------------------
    def edit_target_user_category_preferences_val(self,selected_usercat,uc_action_required):
        okay = False
        error_msg = ""

        #~ ----------------------------------------
        #~ self.source_user_cat_dict
        #~ self.source_field_metadata_dict
        #~ self.original_target_field_metadata_dict
        #~ self.original_target_user_categories_dict
        #~ self.revised_target_field_metadata_dict
        #~ self.revised_target_user_categories_dict
        #~ ----------------------------------------

        if uc_action_required == ADD:
            self.target_fm_changes_required = True
        else:
            self.target_fm_changes_required = False

        # an @selected_usercat is only used for field_metadata, not user_categories...

        self.revised_target_user_categories_dict = copy.deepcopy(self.original_target_user_categories_dict)
        self.revised_target_user_categories_dict[selected_usercat] = self.source_user_cat_dict[selected_usercat]

        okay = True
        error_msg = ""

        #~ ----------------------------------------
        return okay,error_msg
    #-----------------------------------------------------------------------------------------
    def compare_existing_target_user_category_to_source(self,selected_usercat):
        okay = True
        error_msg = ""

        self.uc_comparison_was_successful = False

        #~ ----------------------------------------
        #~ self.source_user_cat_dict
        #~ self.original_target_user_categories_dict
        #~ self.revised_target_user_categories_dict
        #~ ----------------------------------------

        # an '@selected_usercat' is only used for field_metadata, not user_categories...

        if selected_usercat in self.original_target_user_categories_dict:
            uc_preexists = True
        else:
            uc_preexists = False

        self.revised_target_user_categories_dict = copy.deepcopy(self.original_target_user_categories_dict)

        self.revised_target_user_categories_dict[selected_usercat] = self.source_user_cat_dict[selected_usercat]

        #~ ----------------------------------------
        source_elements = self.source_user_cat_dict[selected_usercat]
        #~ if DEBUG: print("source_elements",source_elements)
        source_elements = as_unicode(source_elements)
        source_elements_list = ast.literal_eval(source_elements)
        if not isinstance(source_elements_list,list):
            okay = False
            error_msg = "[CETUCTS1]  Data Error: elements of the Source UC are not a valid list.  Copy Aborted. "
            return uc_preexists,okay,error_msg

        for row in source_elements_list:  # each row is a little list with 3 items
            #~ if DEBUG: print("single source_element: ", as_unicode(row))
            val,column,ignore = row
            if column.startswith("#"):
                c = column.replace("#","").strip()
                if not c in self.target_custom_column_dict:
                    okay = False
                    error_msg = "Source User Category uses a Custom Column that does not exist in the Target Library: " + column   + "  Copy Aborted. "
                    return uc_preexists,okay,error_msg
                else:
                    pass
                    #~ if DEBUG: print("Source Custom Column also exists in the Target Library:", column)
        #END FOR

        self.uc_comparison_was_successful = True

        return uc_preexists,okay,error_msg
    #-----------------------------------------------------------------------------------------
    def synthesize_revised_target_field_metadata_key_value(self,selected_usercat):
        okay = True
        error_msg = ""
        #~ ----------------------------------------
        #~ self.source_user_cat_dict
        #~ self.source_field_metadata_dict
        #~ self.original_target_field_metadata_dict
        #~ self.original_target_user_categories_dict
        #~ self.revised_target_field_metadata_dict
        #~ self.revised_target_user_categories_dict
        #~ ----------------------------------------

        self.revised_target_field_metadata_dict = self.original_target_field_metadata_dict

        # @selected_usercat is only used for field_metadata, not user_categories...
        fm_selected_usercat = "@" + selected_usercat

        if not fm_selected_usercat in self.revised_target_field_metadata_dict:
            if fm_selected_usercat in self.source_field_metadata_dict:
                self.target_fm_changes_required = True
                self.revised_target_field_metadata_dict[fm_selected_usercat] = self.source_field_metadata_dict[fm_selected_usercat]
            else:
                #~ which is exactly why JS+ does not use gprefs for its own preferences; gprefs does not write and commit changes instantly and reliably to the physical db in real-time...
                okay = False
                error_msg = "Calibre Error: A User Category was just renamed, but Calibre has not been restarted since then. Restart Calibre so it will update an important internal table, and then retry your copy. Nothing done."
                if DEBUG: print(error_msg)
                return okay,error_msg
        else:
            self.target_fm_changes_required = False

        del fm_selected_usercat

        return okay,error_msg
    #-----------------------------------------------------------------------------------------
    def do_actual_updates_of_target_table_preferences(self):
        okay = True
        error_msg = ""
        #~ ----------------------------------------
        #~ self.source_user_cat_dict
        #~ self.source_field_metadata_dict
        #~ self.original_target_field_metadata_dict
        #~ self.original_target_user_categories_dict
        #~ self.revised_target_field_metadata_dict
        #~ self.revised_target_user_categories_dict
        #~ ----------------------------------------
        if not isinstance(self.target_fm_changes_required,bool):
            okay = False
            error_msg = "[DAUOFTP1] Program Error: self.target_fm_changes_required is not a valid boolean. Nothing done."
            if DEBUG: print(error_msg)
            return okay,error_msg

        if not isinstance(self.revised_target_user_categories_dict,dict):
            okay = False
            error_msg = "[DAUOFTP2] Data Error: revised_target_user_categories_dict is not a valid dict. Nothing done."
            if DEBUG: print(error_msg)
            return okay,error_msg

        if not len(self.revised_target_user_categories_dict) > 0:
            okay = False
            error_msg = "[DAUOFTP3] Data Error: revised_target_user_categories_dict is empty.  Nothing done."
            if DEBUG: print(error_msg)
            return okay,error_msg

        if not isinstance(self.revised_target_user_categories_dict,dict):
            msg = '[ERROR 0] UTF8 Decoding Error for revised_target_user_categories_dict Prior to JSONize'
            return False,error_msg

        #~ ----------------------------------------
        #~ must mimic what gprefs would end up with in table preferences for the fm and uc keys for their val: pretty-print indented, and unicode now having no u' anywhere.
        #~ ----------------------------------------
        try:
            #give json a real, valid dict to start...
            # User Categories Only
            if DEBUG: print("[1] UC  before any json: self.revised_target_user_categories_dict: ", as_unicode(self.revised_target_user_categories_dict))
            self.revised_target_user_categories_dict = self.json__to_raw(self.revised_target_user_categories_dict)
            if DEBUG: print("[2] UC after json__to_raw: self.revised_target_user_categories_dict: ", as_unicode(self.revised_target_user_categories_dict))
        except Exception as e:
            if DEBUG: print(error_msg)
            okay = False
            error_msg = '[DAUOFTP4] JSON self.json__to_raw exception for self.revised_target_user_cat_dict: ' + as_unicode(e)
            return okay,error_msg
        #~ ----------------------------------------
        #~ ----------------------------------------
        if self.target_fm_changes_required:
            if not isinstance(self.revised_target_field_metadata_dict,dict):
                okay = False
                error_msg = "[DAUOFTP5] Data Error: self.revised_target_field_metadata_dict is not a valid dict. Nothing done."
                if DEBUG: print(error_msg)
                return okay,error_msg
            if not len(self.revised_target_field_metadata_dict) > 20:   # FM always has all of the standard columns plus some other internally-used items...not counting custom columns and UCs...
                okay = False
                error_msg = "[DAUOFTP6] Data Error: self.revised_target_field_metadata_dict is invalid.  Nothing done."
                if DEBUG: print(error_msg)
                return okay,error_msg
            try:
               #give json a real, valid dict to start...
               # Field Metadata Only
               self.revised_target_field_metadata_dict = self.json__to_raw(self.revised_target_field_metadata_dict)
            except Exception as e:
                if DEBUG: print(error_msg)
                okay = False
                error_msg = '[DAUOFTP7] JSON self.json__to_raw exception for self.revised_target_field_metadata_dict: ' + as_unicode(e)
                return okay,error_msg
        #~ ----------------------------------------
        #~ ----------------------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_target_library(self.selected_target)
        if not is_valid:
            okay = False
            error_msg = 'Database Connection Error.  Cannot Connect to the Target Library.'
            if DEBUG: print(error_msg)
            return okay,error_msg
        #~ ----------------------------------------
        try:
            my_cursor.execute("begin")
            mysql = "UPDATE preferences SET val = ? WHERE key = 'user_categories'   "
            my_cursor.execute(mysql,([self.revised_target_user_categories_dict]))
            if self.target_fm_changes_required:
                mysql = "UPDATE preferences SET val = ? WHERE key = 'field_metadata'  "
                my_cursor.execute(mysql,([self.revised_target_field_metadata_dict]))
            my_cursor.execute("commit")
        except Exception as e:
            okay = False
            error_msg = "Target Update Failure.  Reason: " + as_unicode(e)
            if DEBUG: print(error_msg)
        #~ ----------------------------------------
        my_db.close()
        #~ ----------------------------------------
        return okay,error_msg
    #-----------------------------------------------------------------------------------------
    def json__raw_to_object(self, raw):
        if not isinstance(raw, unicode_type):
            raw = raw.decode(preferred_encoding)
        return json_loads(raw)
    #-----------------------------------------------------------------------------------------
    def json__to_raw(self, val):
        return json_dumps(val, indent=2, default=to_json)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def apsw_connect_to_target_library(self,selected_target):

        self.selected_target = selected_target

        path = selected_target
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", as_unicode(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#END OF copy_user_categories_dialog.py