# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.194"  #WrapMode

import os,sys,apsw,ast

from qt.core import (Qt, QDialog, QLabel,  QFont, QWidget, QApplication,
                                       QIcon, QGroupBox, QMargins, QScrollArea,
                                       QDialogButtonBox, QSize, QPushButton,
                                       QVBoxLayout, QHBoxLayout,
                                       QTextEdit, QTextOption)

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.gui2 import gprefs, error_dialog
from calibre.utils.config import JSONConfig

from polyglot.builtins import as_unicode, iteritems

from calibre_plugins.job_spy.config import prefs

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(500, 200)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent, Qt.WindowSystemMenuHint|Qt.WindowMinimizeButtonHint )
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class FormatSpyDialog(SizePersistedDialog):

    def __init__(self,maingui,parent):
        unique_pref_name = 'Job_Spy:formatspy_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.maingui = maingui
        self.guidb = self.maingui.library_view.model().db
        self.original_guidb = self.guidb

        mytitle = 'JS+ GUI Tool:  FormatSpy'
        self.setWindowTitle(_(mytitle))

        layout = QVBoxLayout(self)
        self.setLayout(layout)
        #--------------------------------------------------
        self.book_qtextedit =  QTextEdit("")
        self.book_qtextedit.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.book_qtextedit.setReadOnly(True)
        self.book_qtextedit.clear()
        self.book_qtextedit.setHtml("")
        #--------------------------------------------------
        layout.addWidget(self.book_qtextedit)
        #-----------------------------------------------------

        self.bottom_buttonbox = QDialogButtonBox()
        #~ self.bottom_buttonbox.rejected.connect(self.reject)
        layout.addWidget(self.bottom_buttonbox)

        self.push_button_refresh_formatspy = QPushButton(" ", self)
        self.push_button_refresh_formatspy.setText("Refresh Current Book")
        self.push_button_refresh_formatspy.setToolTip("<p style='white-space:wrap'>The metadata displayed will be refreshed using the currently selected book.  This button is rarely needed.\
                                                                                                                                <br><br>Left or Right Clicking or Doubling-Clicking the mouse on a book will automatically refresh the displayed metadata,\
                                                                                                                                as will moving the cursor via the keyboard to a new book and then pressing the Enter/Return key (if that action is not assigned elsewhere).")
        self.push_button_refresh_formatspy.clicked.connect(self.refresh_formatspy)
        self.bottom_buttonbox.addButton(self.push_button_refresh_formatspy,QDialogButtonBox.AcceptRole)

        self.push_button_save_and_exit = QPushButton(" ", self)
        self.push_button_save_and_exit.setText("Exit FormatSpy")
        self.push_button_save_and_exit.setToolTip("<p style='white-space:wrap'>Save current FormatSpy window settings and then exit.")
        self.push_button_save_and_exit.clicked.connect(self.save_and_exit)
        self.bottom_buttonbox.addButton(self.push_button_save_and_exit,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(True)
        #-----------------------------------------------------
        self.refresh_formatspy()
        #-----------------------------------------------------
        self.resize_dialog()

        self.update()

        self.maingui.library_view.activated.connect(self.refresh_formatspy)
        self.maingui.library_view.clicked.connect(self.refresh_formatspy)
        self.maingui.library_view.doubleClicked.connect(self.refresh_formatspy)
        #~ self.maingui.library_view.entered.connect(self.refresh_formatspy)  #mouse tracking is not enabled...
        self.maingui.library_view.pressed.connect(self.refresh_formatspy)

    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_custom_columns_listing_dialog_geometry(self):
        self.dialog_closing(None)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_and_exit(self):
        self.dialog_closing(None)
        try:
            self.maingui.library_view.activated.disconnect(self.refresh_formatspy)
            self.maingui.library_view.clicked.disconnect(self.refresh_formatspy)
            self.maingui.library_view.doubleClicked.disconnect(self.refresh_formatspy)
            #~ self.maingui.library_view.entered.disconnect(self.refresh_formatspy)  #mouse tracking is not enabled...
            self.maingui.library_view.pressed.disconnect(self.refresh_formatspy)
        except:
            pass
        self.close()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_formatspy(self):
        #~ Note:  mi = self.maingui.library_view.model().db.get_metadata.........   avoids complications when changing libraries causing guidb changes, proper disconnection of events, and user not properly closing this dialog before changing the library.
        #~ Note: if the user changes the library **without** first closing this dialog, then the final book metadata from the prior library will still be shown until an event causes a refresh in the new library.  Worth the simplification that affords.
        try:
            cix = self.maingui.library_view.currentIndex()
            if cix.isValid():
                bookid = self.maingui.library_view.model().id(cix)
                mi = self.maingui.library_view.model().db.get_metadata(bookid, index_is_id=True, get_user_categories=False)
                timestamp = as_unicode(mi.timestamp)
                authorsx = mi.authors
                if isinstance(authorsx,list):
                    n_authors = as_unicode(len(authorsx))
                elif isinstance(authorsx,tuple):
                    authorsx = list(authorsx)
                n_authors = as_unicode(len(authorsx))
                title = mi.title
                #---------------------------
                my_db,my_cursor,is_valid = self.apsw_connect_to_library()
                if not is_valid:
                     return error_dialog(self.gui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
                #-------------------------------------
                mysql = "SELECT id,path FROM books WHERE id = ?"
                my_cursor.execute(mysql,([bookid]))
                tmp_rows = my_cursor.fetchall()
                for row in tmp_rows:
                    id,path = row
                #END FOR
                del tmp_rows
                #-------------------------------------
                mysql = "SELECT book,data FROM conversion_options WHERE book = ?"
                my_cursor.execute(mysql,([bookid]))
                tmp_rows = my_cursor.fetchall()
                if not tmp_rows:
                    tmp_rows = []
                if len(tmp_rows) > 0:
                    has_conversion_options = True
                else:
                    has_conversion_options = False
                #END FOR
                del tmp_rows
                #-------------------------------------
                mysql = "SELECT id,format,uncompressed_size,name FROM data WHERE book = ? ORDER BY id"
                my_cursor.execute(mysql,([bookid]))
                data_rows = my_cursor.fetchall()
                if not data_rows:
                    data_rows = []
                #---------------------------
                my_db.close()
                #---------------------------
                #---------------------------
                details = "<pre>"
                #---------------------------
                details = details + "<b>Book ID:</b>  " + as_unicode(bookid) + "<br><br>"
                #---------------------------
                details = details + "<b>DateTime Added:</b>  " + timestamp + "<br><br>"
                #---------------------------
                details = details + "<b>Library Path:</b>  '" + as_unicode(path) + "'<br><br>"
                #---------------------------
                details = details + "<b>Formats in Import/Polishing/Modification/Conversion Sequence:</b><br>"
                seq = 0
                for row in data_rows:
                    id,format,uncompressed_size,name = row
                    seq = seq + 1
                    format = format + "        "
                    format = format[0:6]
                    uc = as_unicode(uncompressed_size)
                    uc = uc + "            "
                    uc = uc[0:10]
                    details = details + "#" + as_unicode(seq) + ":  "  + format + " - Raw File Size: " + uc + " - File Name:  " + name + "<br>"
                #END FOR
                del data_rows
                #---------------------------
                details = details + "<br>"
                #---------------------------
                details = details + "<b>Author Count:</b>   " + as_unicode(n_authors) + "<br>"
                details = details + "<b>Author(s):</b>   "
                s = ""
                for author in authorsx:
                    s = s + author + " & "
                #END FOR
                s = s[0:-3].strip()
                details = details + s + "<br><br>"
                #---------------------------
                details = details + "<b>Title:</b>  " + title + "<br><br>"
                #---------------------------
                if not has_conversion_options:
                    details = details + "<b>Conversion Options Exist?</b>  " + as_unicode(has_conversion_options) + "<br><br>"
                else:
                    details = details + "<b>Conversion Options:</b><br><br>"
                #---------------------------
                if has_conversion_options:
                    conversion_options = self.guidb.conversion_options(bookid, 'PIPE')
                    conversion_options,is_valid = self.convert_string_to_dict(conversion_options)
                    if is_valid:
                        tmp_list = []
                        #~ for k,v in conversion_options.iteritems():
                        for k,v in iteritems(conversion_options):
                            row = k,v
                            tmp_list.append(row)
                        #END FOR
                        tmp_list.sort()
                        for row in tmp_list:
                            k,v = row
                            details = details + "<b>" + as_unicode(k) + ":</b>  " + as_unicode(v) + "<br>"
                        #END FOR
                        del tmp_list
                    else:
                        details = details + "<b>Conversion Options:</b>  " + "Could not be displayed...<br>"

                    del conversion_options
                else:
                    pass
                #---------------------------
                details = details + "</pre>"
                #---------------------------
                self.book_qtextedit.setHtml(details)
                #---------------------------

                t = "<p style='white-space:wrap'>The 'Format Sequence' <b>is highly influenced</b> by Polishing, the 'Modify EPUB' plug-in, and Conversion of a pre-existing format. \
                                                                        <br><br>For example, if you originally import an 'EPUB', and then immediately convert it to 'PDF', the first Sequence will be 'EPUB' \
                                                                        and the second Sequence will be 'PDF'.  If you then Polish the 'EPUB' format, the 'EPUB' will be assigned a Sequence of #2, and the 'PDF \
                                                                        will be assigned a Sequence of #1.  \
                                                                        <br><br>If you wish to retain the implicit 'Original Format Sequence', you must immediately Polish and otherwise finish reconverting and modifying the original format <b>before</b> converting \
                                                                        that original format to additional formats.  Or, reconvert all of the additional formats <b>after</b> the original format has been changed. \
                                                                        <br><br>One guaranteed method of retaining the original format type is to use a Calibre 'File-Type Plug-in' designed to add the 'original file name', including its file extension, as metadata.\
                                                                        <br><br>Another guaranteed method is to track the exact date and time that every format (plus almost all other metadata) was created or changed by installing the 'Audit Log Plug-in'."

                self.book_qtextedit.setToolTip(t)
                #---------------------------
                del mi
                del details
                del authorsx

            else:
                if DEBUG: print("cix is NOT valid...")
                return
            #--------------------------------------
            #--------------------------------------
        except Exception as e:
            if DEBUG: print("Exception in: class FormatSpyDialog, def refresh_formatspy :", as_unicode(e))
            return

#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_connect_to_library(self):

        my_db = self.maingui.library_view.model().db

        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", as_unicode(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
#---------------------------------------------------------------------------------------------------------------------------------------
    def convert_string_to_dict(self,sdict):
        sdict = as_unicode(sdict)
        try:
            d = ast.literal_eval(sdict)
            del sdict
            if not isinstance(d,dict):
                if DEBUG: print("[1] conversion_options is not formatted properly as a valid dictionary.")
                return d,False
            else:
               return d,True
        except:
            if DEBUG: print("[2] conversion_options is not formatted properly as a valid dictionary.")
            return d,False
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#END OF formatspy_dialog.py