# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import)
from calibre_plugins.savetoformat.gui.prompt import PromptDialog
from calibre.gui2.tools import convert_single_ebook
from multiprocessing import Lock
import os
from calibre.ebooks import BOOK_EXTENSIONS

__license__ = 'GPL 3'
__copyright__ = '2015, Darau, blė <darau.ble@gmail.com>'
__docformat__ = 'restructuredtext en'

from calibre_plugins.savetoformat import ActionSaveToFormat
from calibre_plugins.savetoformat.gui.config import prefs

try:
    from PyQt5.Qt import QMenu, QMessageBox
except:
    from PyQt4.Qt import QMenu, QMessageBox

from calibre.gui2 import Dispatcher, choose_dir, error_dialog
from calibre.gui2.actions import InterfaceAction

class SaveToFormatAction(InterfaceAction):

    name = 'Save To Format'
    action_spec = (_('Save To Format'), None, None, None)
    
    def genesis(self):
        self.mixin = SaveToFormatMixin(self.gui)
        self.qaction.setIcon(get_icons('icons/savetoformat.png'))
        self._setDefaultAction()
        self.menu = QMenu()
        #self.prompt = PromptDialog(self.gui)
        self.load_menu()
        
    def load_menu(self):
        self.menu.clear()
        self.menu.addAction(_('Save to formats...'), self.save_to_formats)
        self.menu.addAction(_('Save favourite formats to favourite location'), self.save_to_formats_default)
        self.menu.addAction(_('Settings...'), self.show_configuration)
        self.menu.addAction(_('About...'), self.about)
        self.qaction.setMenu(self.menu)
    
    '''
    Save (and convert) last used formats with configured
    settings to configured location.
    '''
    def save_to_formats_default(self):
        if self.mixin.running is True:
            return None
        
        print('Default saving')
        context = SaveToFormatContext(self.gui)
        context.setPath(prefs['defdir'])
        context.cleanup = prefs['autodelete']
        self._save_to_formats(context, prefs['formats'])
   
    '''
    Save (and convert) chosen formats to chosen location.
    '''
    def save_to_formats(self):
        if self.mixin.running:
            return None
        
        from collections import Counter
        c = Counter()
        for fmt in BOOK_EXTENSIONS:
            c[fmt] += 1
        d = PromptDialog(c, 'Choose formats to save', parent=self.gui)
        if d.exec_() != d.Accepted:
            return None
        
        self.changeConfig()
        
        path = choose_dir(self.gui, 'save to format dialog', _('Choose destination directory'))
        if not path: return
        
        context = SaveToFormatContext(self.gui)
        # TODO: patestuoti
        context.setPath(os.path.abspath(path).replace('/', os.sep)+os.sep)
        # TODO: pridėti varnelę
        context.cleanup = prefs['autodelete']
        
        formatsToConvert = d.selected_formats
        
        self._save_to_formats(context, formatsToConvert)
        
    '''
    Save books to particular formats with given context settings
    '''    
    def _save_to_formats(self, context, formatsToConvert):
        context.rows = self.gui.library_view.selectionModel().selectedRows()
        ids = [self.gui.library_view.model().id(r) for r in context.rows]
        metadata = self.gui.library_view.model().metadata_for(ids)
        imetadata = iter(metadata)
        
        fmtConvert = {}
        for convFmt in formatsToConvert:
            fmtConvert[convFmt] = []
        
        for idItem in ids:
            mi = next(imetadata)
            for convFmt in formatsToConvert:
                if convFmt.upper() not in mi.formats:
                    fmtConvert[convFmt].append(idItem)
            print('Formats?..', idItem, mi.formats)
        print('Conversion map', fmtConvert)
        
        context.formatsToConvert = fmtConvert
        self.mixin.startAutoConversion(context)
    '''
    Standard Calibre's method to show configuration window.
    '''
    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
    
    '''
    "About" window display.
    '''
    def about(self):
        text = get_resources('additional/about.txt')
        qb = QMessageBox(self.gui)
        qb.setText(text.decode('utf-8') % ActionSaveToFormat.version)
        qb.setWindowTitle('About "Save To Format"')
        qb.setIconPixmap(get_icons('icons/savetoformat.png').pixmap(128, 128))
        qb.show()
   
    '''
    Apply configuration, when changed.
    ''' 
    def changeConfig(self):
        from calibre_plugins.savetoformat.gui.config import prefs
        self._setDefaultAction()
    
    '''
    Set default action when plugin's icon is clicked.
    '''
    def _setDefaultAction(self):
        if (prefs['defaction'] == 'PROMPT'):
            self.qaction.triggered.connect(self.save_to_formats)
        elif (prefs['defaction'] == 'FAVOURITE'):
            self.qaction.triggered.connect(self.save_to_formats_default)
    '''
    Not used currently. Still in debugging progress.

    def _runningError(self):
        error_dialog(self.gui, _('Process is still running'),
                _('Previous saving process is still running. Please wait until it completes.'),
                show=True)
    '''

'''
Context class for passing of saving/conversion information.
'''
class SaveToFormatContext(object):
    
    def __init__(self, gui):
        self.gui = gui
        self.rows = None
        self.formatsToConvert = None
        self._path = None
        self.autoConvert = prefs['autoconvert']
        self.autoDelete = prefs['autodelete']
    
    '''
    Set path from directory dialogue and perform
    check, if path is not the library path.
    '''
    def setPath(self, path):
        lpath = self.gui.library_view.model().db.library_path.replace('/',
                os.sep)+os.sep
        if path.startswith(lpath):
            return error_dialog(self.gui, _('Not allowed'),
                    _('You are trying to save files into the Calibre '
                      'library. This can cause corruption of your '
                      'library. Save to disk is meant to export '
                      'files from your calibre library elsewhere.'), show=True)
        else:
            self._path = path
    
    '''
    Return previously set path.
    '''
    def getPath(self):
        return self._path

'''
"Mixin" class to perform actual work.
'''
class SaveToFormatMixin(object):

    def __init__(self, gui):
        self.running = False
        self.gui = gui
        self._context = None
        self._convCnt = 0
        self._convDoneCnt = 0
        self._saveCnt = 0
        self._saveDoneCnt = 0
    
    def genesis(self):
        '''
        Genesis must always be called before using an SaveToFormatMixin object.
        '''
    
    '''
    Starting the autoconversion, when desired
    books are selected.
    '''
    def startAutoConversion(self, context):
        self.running = True
        self._context = context
        
        self._convCnt = 0
        self._convDoneCnt = 0
        
        for convFmt in self._context.formatsToConvert:
            self._convCnt += len(self._context.formatsToConvert[convFmt])
        
        # Nothing to convert, go directly to saving.
        if self._convCnt == 0 or self._context.autoConvert is False:
            self._startSaving()
            return
        
        previous = self.gui.library_view.currentIndex()
        for convFmt in self._context.formatsToConvert:
            #print 'Convert format:', convFmt
            #print 'Ids to convert:', self._context.formatsToConvert[convFmt]
            if self._context.formatsToConvert[convFmt] != []:
                jobs, changed, bad = convert_single_ebook(self.gui, self.gui.library_view.model().db, self._context.formatsToConvert[convFmt], True, convFmt)
                if jobs == []: continue
                self.gui.iactions['Convert Books'].queue_convert_jobs(jobs, changed, bad, self._context.formatsToConvert[convFmt], previous, self._endAutoConversion, rows_are_ids=True)
    
    '''
    Event handler of finished conversions. Counts all
    executed conversion and compares to conversion lists.
    Mutex is requred, as conversion jobs are performed
    simultaneously.
    '''
    def _endAutoConversion(self, job):
        #print '_endAutoConversion called'
        lock = Lock()
        with lock:
            self._convDoneCnt += 1
            if job.failed:
                self.gui.job_exception(job, dialog_title=_('Conversion failed'))
                return
            self.gui.iactions['Convert Books'].book_converted(job)
            self.gui.status_bar.show_message(job.description + ' ' + _('finished'), 5000)
            #print '_convDoneCnt, convCnt:', self._convDoneCnt, self._convCnt
            if self._convDoneCnt == self._convCnt:
                # Now it is time to pass control to saving job.
                self._startSaving()
    
    '''
    Called after all conversions are done
    (or are skipped). Calls internal Calibre's Saver.
    '''
    def _startSaving(self):
        #print '_startSaving called. Yippy!'
        #from calibre.gui2.save import Saver
        from calibre_plugins.savetoformat.formatsaver import FormatSaver
        from calibre.library.save_to_disk import config
        opts = config().parse()
        
        formatCsv = ''
        for fmt in self._context.formatsToConvert:
            formatCsv += ', '+fmt
        print('Saving formats csv list:', formatCsv[2:])

        opts.formats = formatCsv[2:]
        book_ids = set(map(self.gui.library_view.model().id, self._context.rows))
        self._saver =FormatSaver(book_ids, self.gui.library_view.model().db, opts, self._context.getPath(),
                parent=self.gui, pool=self.gui.spare_pool())
        self._saver.end_signal.connect(self.endSignalHandler)
        #self._signal_cnt = 0
        
        self.gui.status_bar.show_message(_('Starting to save books'), 3000)
    
    def endSignalHandler(self):
        print("Saver end signal caught")
        if self._context.autoDelete is True:
            self._startCleanup()
        else:
            self.running = False
        #print "do end one signal raised", self._signal_cnt
        #print "saver pd cancelled", self._saver.parent
        #self._signal_cnt += 1
    
    '''
    Handler to catch finished Saver's execution.
    '''
    def _endSaving(self, path, failures, error):
        #print 'Saving ended! Time to cleanup!'
        if error:
            return error_dialog(self.gui, _('Error while saving'),
                    _('There was an error while saving.'),
                    error, show=True)
            self.running = False
        if failures:
            failures = ['%s\n\t%s'%
                    (title, '\n\t'.join(err.splitlines())) for title, err in
                    failures]
            for failure in failures:
                print('%s' % failure)
            self.gui.status_bar.show_message(_('There were some failures during saving'), 5000)
        else:
            self.gui.status_bar.show_message(_('Saving to ') + path + _(' done successfully'), 5000)
        
        if self._context.autoDelete is True:
            self._startCleanup()
        else:
            self.running = False
    
    '''
    Called after saving if cleanup of auto-converted files
    is configured. No job is triggered, removing is done
    right here, instantly.
    '''
    def _startCleanup(self):
        self.gui.status_bar.show_message(_('Starting to cleanup auto converted files... '), 3000)
        
        for fmt in self._context.formatsToConvert:
            for bid in self._context.formatsToConvert[fmt]:
                self.gui.library_view.model().db.remove_format(bid, fmt,
                        index_is_id=True, notify=False)
                
        self.gui.library_view.model().refresh_ids(self._context.formatsToConvert[fmt])
        self.gui.library_view.model().current_changed(self.gui.library_view.currentIndex(),
                self.gui.library_view.currentIndex())
        
        if self._context.formatsToConvert[fmt] or len(self._context.formatsToConvert[fmt]):
            self.gui.tags_view.recount()
            
        self.gui.status_bar.show_message(_('Cleanup of auto converted files performed successfully!'), 5000)
        self.running = False
        
