import json
import re
import time
import unicodedata
from collections.abc import Iterable, Mapping
from queue import Empty, Queue
from threading import Event
from typing import Callable

import mechanize

from calibre.ebooks.BeautifulSoup import BeautifulSoup
from calibre.ebooks.metadata.book.base import Metadata
from calibre.ebooks.metadata.sources.base import Source
from calibre.utils.logging import Log


def levenshtein_distance(s1, s2):
    # Source: https://stackoverflow.com/a/32558749/61061.
    if len(s1) > len(s2):
        s1, s2 = s2, s1

    distances = range(len(s1) + 1)
    for i2, c2 in enumerate(s2):
        distances_ = [i2 + 1]
        for i1, c1 in enumerate(s1):
            if c1 == c2:
                distances_.append(distances[i1])
            else:
                distances_.append(1 + min((distances[i1], distances[i1 + 1], distances_[-1])))
        distances = distances_
    return distances[-1]


def remove_control_characters(text: str) -> str:
    unicode_categories = {
        'C',  # Control
        'M',  # Mark
    }

    return ''.join(filter(lambda ch: unicodedata.category(ch)[0] not in unicode_categories, text))


def remove_optional_hebrew_characters(text: str) -> str:
    optional_characters = {'א', 'ה', 'ו', 'י', '"', "'"}
    return ''.join(filter(lambda ch: ch not in optional_characters, text))


def replace_symbols(text: str) -> str:
    unicode_categories = {
        'S',  # Symbol
    }
    text = ''.join(map(lambda ch: ' ' if unicodedata.category(ch)[0] in unicode_categories else ch, text))
    return ' '.join(text.split())


def replace_punctuation(text: str) -> str:
    unicode_categories = {
        'P',  # Punctuation
    }
    text = ''.join(map(lambda ch: ' ' if unicodedata.category(ch)[0] in unicode_categories else ch, text))
    return ' '.join(text.split())


def normalize(text: str) -> str:
    return replace_symbols(
        remove_control_characters(
            unicodedata.normalize('NFC', text)))


def normalize_for_match(text: str) -> str:
    return replace_punctuation(
        replace_symbols(
            remove_control_characters(
                unicodedata.normalize('NFC', text))))


def normalize_for_search(text: str) -> str:
    forbidden_characters = {'?'}
    return ''.join(filter(lambda ch: ch not in forbidden_characters, normalize(text)))


def aggressive_normalize(text: str) -> str:
    return remove_optional_hebrew_characters(normalize(text))


def author_match(left: str, right: str) -> bool:
    '''
    'לוסי מוד מונטגומרי'
    'ל. מ. מונטגומרי'
    'ל.מ. מונטגומרי'
    'ל"מ מונטגומרי'
    "ג'. ר. ר. טולקין", "ג'ון רונלד רעואל טולקין"
    'ש"י עגנון', 'שמואל יוסף עגנון'
    'ד"ר ליאת יקיר', 'ליאת יקיר'
    '''
    TITLES = ['ד"ר ', 'דוקטור ', 'פרופ. ', 'פרופסור ']
    for title in TITLES:
        left = left.removeprefix(title)
        right = right.removeprefix(title)

    left_parts = [part for part in re.split(r'[, ."]', left) if part]
    right_parts = [part for part in re.split(r'[, ."]', right) if part]

    # Match last name
    if left_parts[-1] != right_parts[-1]:
        return False

    if all(any(lp.startswith(rp) for lp in left_parts[:-1]) for rp in right_parts[:-1]):
        return True

    if all(any(rp.startswith(lp) for lp in left_parts[:-1]) for rp in right_parts[:-1]):
        return True

    return False


def token_match(left: str, right: str) -> bool:
    if left == right:
        return True

    left_regex = r'^.*\b' + r'\b.*\b'.join(map(re.escape, left.split())) + r'\b.*$'
    if re.match(left_regex, right):
        return True

    right_regex = r'^.*\b' + r'\b.*\b'.join(map(re.escape, right.split())) + r'\b.*$'
    if re.match(right_regex, left):
        return True

    return False


def title_match(title: str, product_title: str,
                normalize_func: Callable[[str], str]) -> bool:
    title = normalize_func(title)
    product_title = normalize_func(product_title)

    if token_match(title, product_title):
        return True

    print(f'"{title}" does not match "{product_title}"')
    return False


def authors_match(authors: Iterable[str], product_authors: Iterable[str],
                  normalize_func: Callable[[str], str]) -> bool:
    authors = list(map(normalize_func, authors))
    product_authors = list(map(normalize_func, product_authors))

    if any(any(token_match(author, product_author)
               for product_author in product_authors)
           for author in authors):
        return True

    if any(any(author_match(author, product_author)
               for product_author in product_authors)
           for author in authors):
        return True

    print(f'{authors} not in {product_authors}')
    return False


def partial_queries(query: str, min_length: int = 2) -> list[str]:
    queries: Iterable[str] = [query[i:j] for i in range(len(query)) for j in range(i + min_length, len(query) + 1)]
    queries = map(lambda s: s.strip(), queries)
    queries = filter(lambda s: len(s) >= min_length, queries)
    queries = set(queries)
    queries: list[str] = sorted(queries, key=len, reverse=True)

    return queries


def partial_spans(query: str,
                  normalize_func: Callable[[str], str],
                  min_length: int = 2) -> list[str]:
    queries = query.split()
    queries = map(lambda s: normalize_func(s), queries)
    queries = map(lambda s: s.strip(), queries)
    queries = list(queries)
    queries = [' '.join(queries[:i]) for i in range(min_length - 1, len(queries) + 1)]
    queries = list(queries)

    if len(queries) == 1 and queries[0] == query:
        return []

    return queries


def split_queries(query: str,
                  normalize_func: Callable[[str], str],
                  min_length: int = 2) -> list[str]:
    queries = re.split(r'[-:]', query)
    queries = map(lambda s: normalize_func(s), queries)
    queries = map(lambda s: s.strip(), queries)
    queries = filter(lambda s: len(s) >= min_length, queries)
    queries = list(queries)

    if len(queries) == 1 and queries[0] == query:
        return []

    return queries


class Evrit(Source):
    name = 'Evrit'
    description = 'Get metadata information from Evrit.'
    capabilities = frozenset(['identify', 'cover'])
    author = 'Hebrew Reader <hebrew.reader.calibre@gmail.com>'
    version = (1, 4, 0)
    can_get_multiple_covers = False
    touched_fields = frozenset(
        ['title', 'authors', 'tags', 'publisher', 'comments',
         'pubdate', 'rating', 'identifier:evrit', 'languages'])
    supports_gzip_transfer_encoding = True
    cached_cover_url_is_reliable = True
    has_html_comments = True

    def get_book_url(self, identifiers: Mapping[str, str]) \
            -> tuple[str, str, str] | None:
        evrit_id = identifiers.get('evrit', None)
        if evrit_id:
            url = f'https://www.e-vrit.co.il/Product/{evrit_id}'
            return 'evrit', evrit_id, url
        return None

    def get_book_url_name(self, idtype: str, idval: str, url: str) -> str:
        return 'עברית'

    def identify(self, log: Log, result_queue: Queue, abort: Event,
                 title: str = None, authors: Iterable[str] = None,
                 identifiers: Mapping[str, str] = None, timeout: int = 10) -> None:
        end_time = time.time() + timeout
        if identifiers is None:
            identifiers = {}
        # Evrit id exists, read detail page
        evrit_id = identifiers.get('evrit', None)
        evrit_ids_with_relevance = dict()
        if evrit_id:
            evrit_ids_with_relevance[evrit_id] = 0
        else:
            evrit_ids_with_relevance = self.search_for_evrit_id(log, title, authors, abort, timeout)
        best_relevance = min(evrit_ids_with_relevance.values())
        ids_with_best_relevance = [k for k in evrit_ids_with_relevance
                                   if evrit_ids_with_relevance[k] == best_relevance]
        for evrit_id in ids_with_best_relevance:
            timeout = int(end_time - time.time())
            if abort.is_set() or timeout <= 0:
                break
            self.retrieve_evrit_detail(log, evrit_id, best_relevance, result_queue, abort, timeout)

    def search_for_evrit_id(self, log: Log, title: str, authors: Iterable[str],
                            abort: Event, timeout: int) -> dict[str, int]:
        end_time = time.time() + timeout
        authors = [normalize(author) for author in authors]
        queries: list[str] = [normalize_for_search(title)]
        queries += split_queries(title, normalize_for_search)
        queries += partial_spans(title, normalize_for_search)
        queries += partial_queries(normalize_for_search(title))
        seen_queries = set()
        evrit_ids_with_relevance = dict()
        for query in queries:
            # Skip repeated queries.
            if query in seen_queries:
                continue
            seen_queries.add(query)
            fetcher: Callable[[Log, str, int], list[tuple[str, list[str], str]]]
            for fetcher in [self.get_product_values_from_book_teller,
                            self.get_product_values_from_autocomplete]:
                timeout = int(end_time - time.time())
                if abort.is_set() or timeout <= 0:
                    break
                product_values = fetcher(log, query, timeout)

                for normalize_func in [normalize, normalize_for_match, aggressive_normalize]:
                    for product_title, product_authors, product_id in product_values:
                        relevance = levenshtein_distance(title, product_title)
                        if (relevance < 3 or
                            title_match(title, product_title, normalize_func)) and \
                                authors_match(authors, product_authors, normalize_func):
                            if product_id in evrit_ids_with_relevance:
                                relevance = min(evrit_ids_with_relevance[product_id], relevance)
                            evrit_ids_with_relevance[product_id] = relevance
                if evrit_ids_with_relevance:
                    log.info(evrit_ids_with_relevance)
                    return evrit_ids_with_relevance

        return evrit_ids_with_relevance

    def get_product_values_from_book_teller(self, log: Log, query: str, timeout: int) \
            -> list[tuple[str, list[str], str]]:
        unicode_categories = {
            'P',  # Punctuation
        }
        query = ''.join(map(lambda ch: ' ' if unicodedata.category(ch)[0] in unicode_categories else ch, query))
        query = ' '.join(query.split())

        url = 'https://www.e-vrit.co.il/api/ProductApi' \
              '/GetBookTellerProductsByQuery' \
              f'?query={query}'
        log.info(url)
        response = self.browser.open_novisit(url, timeout=timeout)
        results = json.loads(response.read())

        def extract_product_values(result: Mapping) -> tuple[str, list[str], str]:
            log.info(result)
            product_name: str = result['ProductName']
            product_title, product_authors = product_name.rsplit(' / ', 1)
            product_authors = product_authors.split(', ')
            product_id = str(result['ProductID'])
            log.info(product_title, product_authors, product_id)
            return product_title, product_authors, product_id

        product_values = list(map(extract_product_values, results))
        return product_values

    def get_product_values_from_autocomplete(self, log: Log, query: str, timeout: int) \
            -> list[tuple[str, list[str], str]]:
        query_param = json.dumps(query, ensure_ascii=False)
        url = f'https://www.e-vrit.co.il/Menu/GetAutoCompleteSearchItems?keyWord={query_param}'
        log.info(url)
        response = self.browser.open_novisit(url, timeout=timeout)
        soup = BeautifulSoup(response.read())
        books = soup.find('div', class_='show-Books')
        if not books:
            return []
        results = books.find_all('div', class_='item-result')

        def extract_product_values(result) -> tuple[str, list[str], str]:
            product_id = result.get('data-itemid')
            details = result.find('div', class_='details-result')
            product_title = details.find(class_='spnName').text
            product_authors = details.find(class_='spnExtraName').text.split(', ')

            log.info(product_title, product_authors, product_id)
            return product_title, product_authors, product_id

        product_values = list(map(extract_product_values, results))
        return product_values

    def get_tags_from_product_page(self, log: Log, evrit_id: str, abort: Event, timeout: int) -> list[str]:
        if abort.is_set():
            return []
        try:
            url = f'https://www.e-vrit.co.il/Product/{evrit_id}'
            log.info(url)
            response = self.browser.open_novisit(url, timeout=timeout)
            soup = BeautifulSoup(response.read(), features='lxml')
            tags_content = soup.find('div', class_='group-tags__content')
            if tags_content is None:
                return []
            tags = [tag_link.text for tag_link in tags_content.find_all('a', class_='content__name')]
            return tags
        except:
            log.exception()

    def retrieve_evrit_detail(self, log: Log, evrit_id: str, relevance: int,
                              result_queue: Queue, abort: Event, timeout: int) -> None:
        if abort.is_set():
            return
        try:
            url = 'https://www.e-vrit.co.il/api/ProductApi/GetCustomerProductsDetails'
            log.info(url)
            data = {
                'CustomerReviewID': 0,
                'PageNumber': 1,
                'Items': [{
                    'ProductID': int(evrit_id),
                    'CustomerID': 0,
                    'ReviewID': 0,
                }]
            }
            HEADERS = {
                'Content-Type': 'application/json; charset=UTF-8',
            }
            request = mechanize.Request(url, data=json.dumps(data), headers=HEADERS, timeout=timeout)
            response = self.browser.open_novisit(request)
            results = json.loads(response.read())
            for book_data in results:
                log.info(book_data)
                title = remove_control_characters(book_data['Name'])
                authors = [remove_control_characters(author_data['Name']) for author_data in book_data['AuthorList']]

                mi = Metadata(title, authors)
                mi.identifiers = {'evrit': evrit_id}
                mi.languages = ['Hebrew']

                year = book_data['PublishYear']
                month = book_data['PublishMonth']
                if month is None or month == 0:
                    month = 1
                pubdate = f'{year if year else 0:04}-{month:02}'
                if pubdate:
                    try:
                        from calibre.utils.date import parse_date, utcnow
                        default = utcnow().replace(day=15)
                        mi.pubdate = parse_date(pubdate, assume_utc=True, default=default)
                    except:
                        log.exception('Failed to parse pubdate %r' % pubdate)

                tags = []
                if 'CategoryList' in book_data:
                    tags = [category_data['Name'] for category_data in book_data['CategoryList']]
                tags += self.get_tags_from_product_page(log, evrit_id, abort, timeout)
                mi.tags = list(set(tags))

                if 'LongDescription' in book_data:
                    mi.comments = book_data['LongDescription']

                if 'PublisherList' in book_data:
                    mi.publisher = ', '.join(
                        [publisher_details['Name'] for publisher_details in book_data['PublisherList']])

                if 'AvgReviews' in book_data and book_data['AvgReviews'] > 0:
                    mi.rating = book_data['AvgReviews']
                else:
                    mi.rating = None

                if 'Image' in book_data:
                    image_path = book_data['Image']
                    image_path = str.replace(image_path.lower(), '/image_', '/')
                    mi.has_evrit_cover = f'https://images-evrit.yit.co.il/{image_path}'
                    self.cache_identifier_to_cover_url(mi.identifiers['evrit'], mi.has_evrit_cover)
                else:
                    mi.has_evrit_cover = None

                mi.source_relevance = relevance
                log.info(mi.source_relevance)
                result_queue.put(mi)
        except:
            log.exception()

    def download_cover(self, log: Log, result_queue: Queue,
                       abort: Event, title: str = None,
                       authors: Iterable[str] = None,
                       identifiers: Mapping[str, str] = None,
                       timeout: int = 30,
                       get_best_cover: bool = False) -> None:
        if identifiers is None:
            identifiers = {}
        cached_url = self.get_cached_cover_url(identifiers)
        if cached_url is None:
            log.info('No cached cover found, running identify')
            rq = Queue()
            self.identify(log, rq, abort, title=title,
                          authors=authors, identifiers=identifiers)
            if abort.is_set():
                return
            results = []
            while True:
                try:
                    results.append(rq.get_nowait())
                except Empty:
                    break
            results.sort(
                key=self.identify_results_keygen(
                    title=title, authors=authors, identifiers=identifiers)
            )
            for mi in results:
                cached_url = self.get_cached_cover_url(mi.identifiers)
                if cached_url is not None:
                    break
        if cached_url is None:
            log.info('No cover found')
            return

        if abort.is_set():
            return

        log('Downloading cover from:', cached_url)
        self.download_image(cached_url, timeout, log, result_queue)

    def get_cached_cover_url(self, identifiers: Mapping[str, str] = None) -> str:
        if identifiers is None:
            identifiers = {}
        url = None
        evrit_id = identifiers.get('evrit', None)
        if evrit_id is not None:
            url = self.cached_identifier_to_cover_url(evrit_id)
        return url


def identifier_test(key: str, value: str) -> Callable[[Metadata], bool]:
    from calibre import prints

    def fn(mi: Metadata) -> bool:
        identifier = mi.get_identifiers().get(key, None)
        if identifier == value:
            return True
        prints(f'Identifier test failed. Expected: \'{key}:{value}\' found {mi.get_identifiers()}')
        return False

    return fn


def test() -> None:
    # To run these test use:
    #   calibre-customize -b evrit && calibre-debug -e evrit\__init__.py
    from calibre.ebooks.metadata.sources.test import (
        test_identify_plugin, title_test, authors_test, tags_test
    )

    test_identify_plugin(
        Evrit.name, [
            ({  # Identify by identifier.
                 'identifiers': {'evrit': '3785'},
             }, [title_test('חפצים חדים', exact=True),
                 authors_test(['גיליאן פלין']),
                 tags_test(['ספרי ביכורים', 'ספרים שעובדו לסדרות', 'מתח ופעולה',
                            'ספרים שעובדו לסרטים', 'ספרים עם טוויסט',
                            'ספרות גותית', 'נפש', 'דרום ארה"ב', 'בלשיות']),
                 ]),
            ({  # Identify by title & author.
                 'title': 'חפצים חדים',
                 'authors': ['גיליאן פלין']
             }, [identifier_test('evrit', '3785'),
                 title_test('חפצים חדים', exact=True),
                 authors_test(['גיליאן פלין'])]),
            ({  # Identify by partial title & author.
                 'title': 'חפצים',
                 'authors': ['גיליאן פלין']
             }, [title_test('חפצים חדים', exact=True),
                 authors_test(['גיליאן פלין'])]),
            ({  # Multi-author books.
                 'title': 'רוזאנה',
                 'authors': ['פר ואלו', 'מאי שובאל']
             }, [title_test('רוזאנה', exact=True),
                 authors_test(['מאי שובאל', 'פר ואלו'])]),
            ({  # Single author is enough to identify multi-author books.
                 'title': 'רוזאנה',
                 'authors': ['מאי שובאל']
             }, [title_test('רוזאנה', exact=True),
                 authors_test(['מאי שובאל', 'פר ואלו'])]),
            ({  # Author name has unicode control character.
                 'title': 'קלרה והשמש',
                 'authors': ['קאזואו אישיגורו']
             }, [title_test('קלרה והשמש', exact=True),
                 authors_test(['קאזואו אישיגורו'])]),
            ({  # Author with alternative spelling: פורשאו vs. פורשו.
                 'title': 'היקום הקוונטי',
                 'authors': ['בריאן קוקס', "ג'ף פורשאו"]
             }, [identifier_test('evrit', '26730'),
                 authors_test(['בריאן קוקס', "ג'ף פורשו"])]),
            ({  # Title with question mark.
                 'title': 'איך זה קרה?',
                 'authors': ['מייקל קוריטה']
             }, [identifier_test('evrit', '13500')]),
            ({  # Partial title.
                 'title': 'עקצן איש העקרב',
                 'authors': ['אדם בליד']
             }, [title_test('השרביט והחרב 18 - עקצן איש העקרב', exact=True),
                 identifier_test('evrit', '25936')]),
            ({  # Partial title with alternative spelling: טופר vs. טפר.
                 'title': 'טופר הקוף הענקי',
                 'authors': ['אדם בליד']
             }, [title_test('השרביט והחרב 8 - טפר הקוף הענקי', exact=True),
                 identifier_test('evrit', '12092')]),
            ({  # Title with extra hyphen.
                 'title': 'יום הפלישה - D-DAY',
                 'authors': ["ג'יילס מילטון"]
             }, [title_test('יום הפלישה D-DAY', exact=True),
                 identifier_test('evrit', '25283')]),
            ({  # Title with punctuation.
                 'title': "פ' זה פחד",
                 'authors': ["קן ברואן"]
             }, [title_test("פ' זה פחד", exact=True),
                 identifier_test('evrit', '3332')]),
            ({  # Partial title: extra token in product.
                 'title': "ארו שאה ונקטר האלמוות",
                 'authors': ["רשני צ'קשי"]
             }, [title_test("ארו שאה 5 ונקטר האלמוות", exact=True),
                 identifier_test('evrit', '26897')]),
            ({  # Partial author name: extra token in product.
                 'title': "קיצור תולדות האהבה",
                 'authors': ["ליאת יקיר"]
             }, [authors_test(['ד"ר ליאת יקיר']),
                 identifier_test('evrit', '25314')]),
            ({  # Not all authors found.
                 'title': "איליאדה",
                 'authors': ["הומרוס", "שאול טשרניחובסקי"]
             }, [authors_test(['הומרוס']),
                 identifier_test('evrit', '26106')]),
            ({  # Support titles with '/'.
                 'title': 'אף אחד לא מושלם',
                 'authors': ["ג'ים בנטון"]
             }, [title_test('יומני הסודי ביותר - ספר 3 / אף אחד לא מושלם. אבל אני הכי קרובה לזה'),
                 authors_test(["ג'ים בנטון"])]),
            ({  # Support titles with '/'.
                 'title': 'אן מאבונלי',
                 'authors': ['לוסי מוד מונטגומרי']
             }, [title_test('האסופית 2 - אן מאבונלי'),
                 authors_test(['ל. מ. מונטגומרי'])]),
            ({  # Title with subtitles.
                 'title': 'הארנק והחרב: המהפכה המשפטית ושברה',
                 'authors': ['דניאל פרידמן']
             }, [title_test('הארנק והחרב'),
                 identifier_test('evrit', '2374')]),
            ({  # Title with Geresh.
                 'title': "להגן על ג'ייקוב",
                 'authors': ['ויליאם לאנדיי']
             }, [identifier_test('evrit', '2592')]),
            ({  # Colon replaced with Dash.
                 'title': "אמון: סבא שלי היה יורה בי",
                 'authors': ["ג'ניפר טגה", 'ניקולה זלמייר']
             }, [title_test('אמון - סבא שלי היה יורה בי'),
                 identifier_test('evrit', '3556')]),
            ({  # Title with Gershayim.
                 'title': 'העי"ג',
                 'authors': ['רואלד דאל']
             }, [title_test('העי"ג'),
                 identifier_test('evrit', '13352')]),
            ({  # Bug - was matched with a disney book
                 'title': 'אומץ לב',
                 'authors': ['אריה אבנרי']
             }, [title_test('אומץ לב'),
                 authors_test(['אריה אבנרי']),
                 identifier_test('evrit', '3901')]),
            ({  # Can distinguish between title which is a prefix of the requested one:
                 # 'זה הולך לכאוב בחגים' vs. 'זה הולך לכאוב'
                 'title': 'זה הולך לכאוב בחגים',
                 'authors': ['אדם קיי']
             }, [title_test('זה הולך לכאוב בחגים'),
                 authors_test(['אדם קיי']),
                 identifier_test('evrit', '17252')]),
            ({  # Mismatching titles (missing letter in product title)
                 'title': 'אל המקום הטוב בעולם ובחזרה',
                 'authors': ['עדי ברוקס']
             }, [title_test('אל המקום הטוב בעולם וחזרה'),
                 authors_test(['עדי ברוקס']),
                 identifier_test('evrit', '20316')]),
            # EXISTS MULTIPLE:
            # איליאדה - הומרוס
            # 6866, 26106
            # חוות החיות - ג'ורג' אורוול
            # 13892, 24405
            #
            # {'title': 'מראות מצחיקות', 'authors': ['עליזה מיוחס אלראי'], 'identifiers': {}, 'timeout': 30}
            #   מראות מצחיקות ['עליזה אלראי'] 190
            # {'title': 'ותכתבו אהובתנו', 'authors': ['אמונה אלון'], 'identifiers': {}, 'timeout': 30}
            # {'title': 'המטרה: תל-אביב', 'authors': ['רם אורן'], 'identifiers': {}, 'timeout': 30}
            # {'title': 'קרוב', 'authors': ['דרור בורשטיין'], 'identifiers': {}, 'timeout': 30}
            #   קרוב ['דרור בורשטיין'] 333
            # {'ProductID': 7271, 'ProductImage': 'Images/Products/covers_2018/image_perfect_master.jpg', 'ProductName': "יומני הסודי ביותר - ספר 3 / אף אחד לא מושלם. אבל אני הכי קרובה לזה / ג'ים בנטון"}
            # {'title': 'אן מאבונלי', 'authors': ['לוסי מוד מונטגומרי'], 'identifiers': {}, 'timeout': 30}
            # האסופית 2 - אן מאבונלי ['ל. מ. מונטגומרי'] 1255

        ], fail_missing_meta=False
    )


def test_short() -> None:
    # To run these test use:
    #   calibre-customize -b evrit && calibre-debug -e evrit\__init__.py
    from calibre.ebooks.metadata.sources.test import (
        test_identify_plugin, title_test, authors_test
    )

    test_identify_plugin(
        Evrit.name, [
            ({  # Mismatching titles (missing letter in product title)
                 'title': 'אל המקום הטוב בעולם ובחזרה',
                 'authors': ['עדי ברוקס']
             }, [title_test('אל המקום הטוב בעולם וחזרה'),
                 authors_test(['עדי ברוקס']),
                 identifier_test('evrit', '20316')]),
        ], fail_missing_meta=False
    )


if __name__ == '__main__':
    # from calibre.utils.matcher import Matcher, CScorer
    #
    # items = ['זה הולך לכאוב בחגים', 'זה הולך לכאוב', 'זה הולך לכאוב מחר', 'מתי זה הולך לכאוב']
    # # items = ['זה הולך לכאוב בחגים', 'זה: הולך לכאוב בחגים', 'זה - הולך לכאוב בחגים', ]
    # m = Matcher(items)
    # needle = 'זה: לכאוב מ'
    # print(m(needle))
    #
    # f = fuzzy_title(needle)
    # print(f)
    #
    # s = CScorer(items, level3=' :-')
    # print(list(s(f)))
    #
    # print([levenshteinDistance(needle, item) for item in items])
    test()
