# ebook.py 0.3 2006-10-21
# Copyright (C) 2006 Igor Skochinsky <skochinsky@mail.ru>
#
# This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with this program; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

import struct, time, sys, os
from ctypes import *

reqFileOpen     = 0x10
reqFileClose    = 0x11
reqGetFileSize  = 0x12
reqSetFileSize  = 0x13
reqFileRead     = 0x16
reqFileWrite    = 0x17
reqGetFileInfo  = 0x18
reqFileCreate   = 0x1A
reqFileDelete   = 0x1B
reqFileRename   = 0x1C
reqDirEnumStart = 0x33
reqDirEnumStop  = 0x34
reqDirEnumNext  = 0x35
reqGetProperty  = 0x101
reqUpdateChangeMode = 0x300
reqUpdateDeletePartition = 0x301
reqUpdateCreatePartition = 0x302
reqUpdateCreatePartitionWithImage = 0x303
reqUpdateGetPartitionSize = 0x304 

#DO NOT change the following line if you don't know what you're doing
#overwriting files on the Reader is HIGHLY RISKY and might void your warranty
enableWriting = 1

def debug(msg):
    #print >>sys.stderr, "[DEBUG]", msg
    pass

def timetostr(t):
    try:
        s = time.ctime(t)
    except:
        s = str(t)
    return s

def _complain_ifclosed(closed):
    if closed:
        raise ValueError, "I/O operation on closed file"

class EbookFile:
    def __init__(self, ebook):
        self.ebook = ebook
        self.handle = None
        self.closed = True

    def open(self, filename, mode="r"):
        info = self.ebook.GetFileInfo(filename)
        #dbg="Filename: %s\n"%path+"\tsize: %d\n"%info[0]+"\ttype: %d\n"%info[1]+"\tcreate time: %s\n"%timetostr(info[2])+"\twrite time: %s\n"%timetostr(info[3])+"\tflags: %d\n"% info[4]
        if "w" in mode:
            if info:
                self.ebook.FileDelete(filename)
            self.ebook.FileCreate(filename)
            info = self.ebook.GetFileInfo(filename)
            if not info:
                raise os.error, "Cannot create file '%s'"%filename
        elif not info:
            raise os.error, "File '%s' does not exist"%filename
        
        #if info[1]!=1:
        #    raise os.error, "'%s' is not a file"%filename

        self.size = info[0]
        self.ctime = info[2]
        self.wtime = info[3]
        self.flags = info[4]
        self.pos = 0
        self.handle = self.ebook.FileOpen(filename, "w" in mode)
        self.filename = filename
        if not self.handle:
            raise os.error, "Could not open file '%s'"%filename
        self.closed = False
                

    def seek(self, pos, mode = 0):
        """Set the file's current position.

        The mode argument is optional and defaults to 0 (absolute file
        positioning); other values are 1 (seek relative to the current
        position) and 2 (seek relative to the file's end).

        There is no return value.
        """
        _complain_ifclosed(self.closed)
        if mode == 1:
            pos += self.pos
        elif mode == 2:
            pos += self.size
        self.pos = max(0, pos)

    def tell(self):
        """Return the file's current position."""
        _complain_ifclosed(self.closed)
        return self.pos

    def read(self, n = -1):
        """Read at most size bytes from the file
        (less if the read hits EOF before obtaining size bytes).

        If the size argument is negative or omitted, read all data until EOF
        is reached. The bytes are returned as a string object. An empty
        string is returned when EOF is encountered immediately.
        """
        _complain_ifclosed(self.closed)
        if n < 0:
            newpos = sys.maxint
        else:
            newpos = self.pos+n
        left = newpos-self.pos
        r = ""
        while left>0:
            toread = left
            if toread>0x8000:
                toread=0x8000
            #debug("Reading %d at %08X"%(toread, self.pos))
            chunk = self.ebook.FileRead(self.handle, self.pos, toread)
            if chunk is None or len(chunk)==0: break
            self.pos += len(chunk)
            left -= len(chunk)
            r += chunk
        #self.pos = newpos
        debug("Total read: %d (of %d)"%(len(r), n))
        return r

    def write(self, s):
        """Write a string to the file.
        There is no return value.
        """
        _complain_ifclosed(self.closed)
        if not s: return
        # Force s to be a string or unicode
        if not isinstance(s, basestring):
            s = str(s)
        spos = self.pos
        slen = len(s)
        written = self.ebook.FileWrite(self.handle, self.pos, s)
        if written != slen:
            raise os.error, "Error writing %d bytes to file '%s'"%(slen, filename)
        self.pos = spos + slen
        if self.pos > self.size:
           self.size = spos + slen

    def close(self):
        if not self.closed and self.handle:
            self.ebook.FileClose(self.handle)
            self.closed = True

def cksum(fname):
    o = os.popen("cksum "+fname)
    s = o.read()
    res = s.split(' ')
    o.close()
    if len(res)!=3:
        raise os.error, "Error in chksum (return string: %s)"%s
    return int(res[0]), int(res[1])

def cksum2(s):
    import psyco
    psyco.full()

    crctab = [0x0,
	0x04c11db7, 0x09823b6e, 0x0d4326d9, 0x130476dc, 0x17c56b6b,
	0x1a864db2, 0x1e475005, 0x2608edb8, 0x22c9f00f, 0x2f8ad6d6,
	0x2b4bcb61, 0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd,
	0x4c11db70, 0x48d0c6c7, 0x4593e01e, 0x4152fda9, 0x5f15adac,
	0x5bd4b01b, 0x569796c2, 0x52568b75, 0x6a1936c8, 0x6ed82b7f,
	0x639b0da6, 0x675a1011, 0x791d4014, 0x7ddc5da3, 0x709f7b7a,
	0x745e66cd, 0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
	0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5, 0xbe2b5b58,
	0xbaea46ef, 0xb7a96036, 0xb3687d81, 0xad2f2d84, 0xa9ee3033,
	0xa4ad16ea, 0xa06c0b5d, 0xd4326d90, 0xd0f37027, 0xddb056fe,
	0xd9714b49, 0xc7361b4c, 0xc3f706fb, 0xceb42022, 0xca753d95,
	0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1, 0xe13ef6f4,
	0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d, 0x34867077, 0x30476dc0,
	0x3d044b19, 0x39c556ae, 0x278206ab, 0x23431b1c, 0x2e003dc5,
	0x2ac12072, 0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16,
	0x018aeb13, 0x054bf6a4, 0x0808d07d, 0x0cc9cdca, 0x7897ab07,
	0x7c56b6b0, 0x71159069, 0x75d48dde, 0x6b93dddb, 0x6f52c06c,
	0x6211e6b5, 0x66d0fb02, 0x5e9f46bf, 0x5a5e5b08, 0x571d7dd1,
	0x53dc6066, 0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
	0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e, 0xbfa1b04b,
	0xbb60adfc, 0xb6238b25, 0xb2e29692, 0x8aad2b2f, 0x8e6c3698,
	0x832f1041, 0x87ee0df6, 0x99a95df3, 0x9d684044, 0x902b669d,
	0x94ea7b2a, 0xe0b41de7, 0xe4750050, 0xe9362689, 0xedf73b3e,
	0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2, 0xc6bcf05f,
	0xc27dede8, 0xcf3ecb31, 0xcbffd686, 0xd5b88683, 0xd1799b34,
	0xdc3abded, 0xd8fba05a, 0x690ce0ee, 0x6dcdfd59, 0x608edb80,
	0x644fc637, 0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb,
	0x4f040d56, 0x4bc510e1, 0x46863638, 0x42472b8f, 0x5c007b8a,
	0x58c1663d, 0x558240e4, 0x51435d53, 0x251d3b9e, 0x21dc2629,
	0x2c9f00f0, 0x285e1d47, 0x36194d42, 0x32d850f5, 0x3f9b762c,
	0x3b5a6b9b, 0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
	0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623, 0xf12f560e,
	0xf5ee4bb9, 0xf8ad6d60, 0xfc6c70d7, 0xe22b20d2, 0xe6ea3d65,
	0xeba91bbc, 0xef68060b, 0xd727bbb6, 0xd3e6a601, 0xdea580d8,
	0xda649d6f, 0xc423cd6a, 0xc0e2d0dd, 0xcda1f604, 0xc960ebb3,
	0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7, 0xae3afba2,
	0xaafbe615, 0xa7b8c0cc, 0xa379dd7b, 0x9b3660c6, 0x9ff77d71,
	0x92b45ba8, 0x9675461f, 0x8832161a, 0x8cf30bad, 0x81b02d74,
	0x857130c3, 0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640,
	0x4e8ee645, 0x4a4ffbf2, 0x470cdd2b, 0x43cdc09c, 0x7b827d21,
	0x7f436096, 0x7200464f, 0x76c15bf8, 0x68860bfd, 0x6c47164a,
	0x61043093, 0x65c52d24, 0x119b4be9, 0x155a565e, 0x18197087,
	0x1cd86d30, 0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
	0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088, 0x2497d08d,
	0x2056cd3a, 0x2d15ebe3, 0x29d4f654, 0xc5a92679, 0xc1683bce,
	0xcc2b1d17, 0xc8ea00a0, 0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb,
	0xdbee767c, 0xe3a1cbc1, 0xe760d676, 0xea23f0af, 0xeee2ed18,
	0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4, 0x89b8fd09,
	0x8d79e0be, 0x803ac667, 0x84fbdbd0, 0x9abc8bd5, 0x9e7d9662,
	0x933eb0bb, 0x97ffad0c, 0xafb010b1, 0xab710d06, 0xa6322bdf,
	0xa2f33668, 0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4]

    crc = 0L
    for c in s:
        i = ((crc >> 24) ^ ord(c))&0xFF
        crc = (crc << 8) ^ crctab[i]
    l = len(s)
    while l:
        i = ((crc >> 24) ^ l)&0xFF
        crc = (crc << 8) ^ crctab[i]
        l >>= 8
    return (~crc&0xFFFFFFFF)

class Ebook:
    def __init__(self):
        self.dll = windll.ebookUsb
        self.sendProc = getattr(self.dll,"_UsbSendProc@16")
        self.receiveProc = getattr(self.dll,"_UsbReceiveProc@12")
        self.bufFreeProc = getattr(self.dll,"_UsbBuffFree@4")
        res = getattr(self.dll,"_UsbInitCheck@0")()
        self.connected = False
        self.present = False
        if res!=0:
            debug("checkConnect err=%d"%res)
            return
        else:
            debug("Device is present")
            self.present = True
        proto = create_string_buffer(8)
        res = getattr(self.dll,"_UsbGetProtcolVer@8")(1,proto)
        if res!=0:
            debug("UsbGetProtcolVer err=%d"%res)
            self.present = False
            return
        else:
            debug("Protocol: %s"%proto.value)

    def connect(self, unlockKey="-1"):
        if not self.present:
            debug("Device is not connected!")
            return

        res = getattr(self.dll,"_UsbUnlockDevice@4")(unlockKey)
        if res!=0:
            debug("UsbUnlockDevice err=%d"%res)
            #ignore unlock errors, connect will fail if unlock is needed
            #return
        else:
            debug("Unlocked OK")

        res = getattr(self.dll,"_UsbConnect@0")()
        if res!=0:
            debug("UsbConnect err=%d"%res)
            self.connected = False
            return
        else:
            debug("Connected OK")
            self.connected = True


    def disconnect(self):
        if not self.connected: return
        res = getattr(self.dll,"_UsbDisConnect@0")()
        if res!=0:
            debug("UsbDisConnect err=%d"%res)
        else:
            debug("Disconnected OK")
        self.connected = False

    def requestReceive(self, reqNo, extradata, readsize):
        request = struct.pack("<IIII",reqNo, 0, 0, len(extradata))+extradata
        answer = c_int(0)
        debug("Request: %02X, extra len: %d" % (reqNo, len(extradata)))
        res = self.receiveProc(request, readsize, byref(answer))
        if res!=0:
            debug("Error from UsbReceiveProc: %d!"%res)
            return
        pa = answer.value
        if pa:
            ans_unp = struct.unpack("<IIII",string_at(pa, 16))
            debug("Answer: "+str(ans_unp))
            ret_len = ans_unp[3]
            #print "pa:",pa
            res = string_at(pa+16, ret_len)
            self.bufFreeProc(pa)
            return res

    def requestSend(self, reqNo, extradata, senddata):
        request = struct.pack("<IIII",reqNo, 0, 0, len(extradata))+extradata
        send_len = len(senddata)
        buf = struct.pack("<IIII",0x10005, 0, 0, send_len)+senddata
        bytessent = c_int(0)
        debug("Request: %02X, extra len: %d, send len: %d" % (reqNo, len(extradata), send_len))
        res = self.sendProc(request, buf, send_len, byref(bytessent))
        if res!=0:
            debug("Error from UsbReceiveProc: %d!"%res)
            return
        debug("Sent %d/%d bytes"%(bytessent.value,send_len))
        return bytessent.value

    def FileOpen(self, path, write=False):
        mode = 0
        if write: mode=1
        extra = struct.pack("<II", mode, len(path))+path
        res = self.requestReceive(reqFileOpen, extra, 4)
        if res:
            handle = struct.unpack("<I",res)[0]
            debug("Opened '%s', handle: %02X"%(path, handle))
            return handle

    def FileClose(self, handle):
        extra = struct.pack("<I", handle)
        self.requestReceive(reqFileClose, extra, 0)

    def FileCreate(self, path):
        extra = struct.pack("<I", len(path))+path
        self.requestReceive(reqFileCreate, extra, 0)

    def FileDelete(self, path):
        extra = struct.pack("<I", len(path))+path
        self.requestReceive(reqFileDelete, extra, 0)

    def FileRename(self, path1, path2):
        extra = struct.pack("<III", len(path1), len(path2))+path1+path2
        self.requestReceive(reqFileRename, extra, 0)

    def FileRead(self, handle, offset, size):
        extra = struct.pack("<IQI", handle, offset, size)
        res = self.requestReceive(reqFileRead, extra, size)
        return res

    def FileWrite(self, handle, offset, data):
        extra = struct.pack("<IQI", handle, offset, len(data))
        res = self.requestSend(reqFileWrite, extra, data)
        return res

    def GetFileInfo(self, path):
        extra = struct.pack("<I", len(path))+path
        res = self.requestReceive(reqGetFileInfo, extra, 0x18)
        if res:
            info = struct.unpack("<QIIII",res)
            dbg="Filename: %s\n"%path+"\tsize: %d\n"%info[0]+"\ttype: %d\n"%info[1]+"\tcreate time: %s\n"%timetostr(info[2])+"\twrite time: %s\n"%timetostr(info[3])+"\tflags: %d\n"% info[4]
            debug(dbg)
            return info

    def UpdateChangeMode(self, mode):
        if mode:
            mode=1
        else:
            mode=0
        extra = struct.pack("<I",mode)
        res = self.requestReceive(reqUpdateChangeMode, extra, 0)
        return res

    def UpdateGetPartitionSize(self, name):
        extra = struct.pack("<32s", name)
        res = self.requestReceive(reqUpdateGetPartitionSize, extra, 0xC)
        if res:
            info = struct.unpack("<III",res)
            dbg="Device: %d, start sector: 0x%x, image size: 0x%x"%info
            debug(dbg)
            return info
        else:
            print "UpdateGetPartitionSize: could not get info for partition %s"%name
        
    def UpdateCreatePartitionWithImage(self, name, fname, info, crc, datalen):
        #info = self.UpdateGetPartitionSize(name)
        if not info or len(info)!=3:
            print "UpdateCreatePartitionWithImage: partition info error"
            return
        
        debug("Calculated cksum: %d"%crc)
        extra = struct.pack("<32sIIII32s", name, info[0], info[1], info[2], datalen, str(crc))
        #print "Sending request: ",extra
        
        data = file(fname,"rb").read()
        if len(data)!=datalen:
            print "UpdateCreatePartitionWithImage: datalen doesn't match file length"
            return
        
        if info[2]<datalen:
            print "UpdateCreatePartitionWithImage: data is bigger than partition!"
            return                    
        
        if not raw_input("Ready to write %d bytes to partition %s. Are you SURE you want to proceed? [N]"%(datalen,name)).lower().startswith("y"):
            return

        res = self.requestSend(reqUpdateCreatePartitionWithImage, extra, data)
        return res
    
    def open(self, path, mode="r"):
        f = EbookFile(self)
        f.open(path,mode)
        return f

    def isdir(self,path):
        if not path.endswith('/'): path+='/'
        info = self.GetFileInfo(path)
        return (info and info[1]==2)

    #ripped almost verbatim from os.path
    def walk(self, top, topdown=True, onerror=None):
        from posixpath import join
        try:
            names = self.listdir(top)
        except os.error, err:
            if onerror is not None:
                onerror(err)
            return

        dirs, nondirs = [], []
        for name in names:
            if self.isdir(join(top, name)):
                dirs.append(name)
            else:
                nondirs.append(name)

        if topdown:
            yield top, dirs, nondirs
        for name in dirs:
            path = join(top, name)
            if not (path.startswith('/dev/') or path.startswith('/proc/')): #skip dirs inside /dev and /proc, these can lead to recursion
                for x in walk(path, topdown, onerror):
                    yield x
        if not topdown:
            yield top, dirs, nondirs

    def listdir(self, path):
        #if path.startswith('/dev/') or path.startswith('/proc/'):
        #    return
        if not path.endswith('/'): path+='/'
        if not self.isdir(path):
            return []
        extra = struct.pack("<I", len(path))+path
        enumHandle = self.requestReceive(reqDirEnumStart, extra, 4)
        names = []
        if enumHandle:
            while True:
                nextFile = self.requestReceive(reqDirEnumNext, enumHandle, 0)
                if nextFile:
                    #dword type, dword name_len, char[] name
                    typ = struct.unpack("<I", nextFile[:4])[0]
                    name = nextFile[8:]
                    names.append(name)
                else:
                    self.requestReceive(reqDirEnumStop, enumHandle, 0)
                    return names
        else:
            #print "Could not list directory!"
            raise os.error, "Could not list directory '%s'"%path

def Usage():
    print "Usage: ebook.py cmd [params]"
    print "  ls <dir> [-R]: list device directory <dir> [recursively]"
    print "  get <path>: download <path> from the device to current directory"
    print "  cat <path>: dump <path> from the device to the console."
    if enableWriting:
        print "  put <localfile> <devicefile>: upload file <localfile> to the <devicefile> at the device."
        print "  del <devicefile>: delete <devicefile> from device."
        print "  um <mode>: change update mode (normal/recovery)"
        print "  pinfo <name>: show info about MTD partition <name> (only in recovery mode)"
        print "  pwrite <name> <file>: write to MTD partition <name> from file <file> (only in recovery mode)"
        print "  BE VERY CAREFUL when uploading!"

def forcedir(path):
    if os.path.isdir(path):
        return
    if os.path.exists(path):
        os.remove(path)
    os.mkdir(path)

def download(ebook, path, targetdir, limit):
    from posixpath import basename, join
    print "%s -> %s"%(path, targetdir),
    if path.endswith("/"): path=path[:-1]
    base = basename(path)
    if ebook.isdir(path):
        print 
        base = basename(path)
        newdir = os.path.join(targetdir, base)
        forcedir(newdir)
        files = ebook.listdir(path)
        for f in files:
            download(ebook, join(path,f), newdir)
    else:
        outf = file(os.path.join(targetdir, base),"wb")
        f = b.open(path)
        try:
            if not limit:
                limit = sys.maxint
            left = limit
            while left>0:
                toread = left
                if toread>0x8000:
                    toread=0x8000
                #debug("Reading %d at %08X"%(toread, self.pos))
                chunk = f.read(toread)
                if chunk is None or len(chunk)==0: break
                left -= len(chunk)
                outf.write(chunk)
            print "%d bytes"%f.tell()
        finally:
            outf.close()
            f.close()
	
	
def downloadToConsole(ebook, path, targetdir, limit):
    from posixpath import basename, join
    print "%s -> %s"%(path, targetdir),
    if path.endswith("/"): path=path[:-1]
    base = basename(path)
    if ebook.isdir(path):
        print "Can't cat directory";
    else:
        print
        f = b.open(path)
        try:
            if not limit:
                limit = sys.maxint
            left = limit
            while left>0:
                toread = left
                if toread>0x8000:
                    toread=0x8000
                #debug("Reading %d at %08X"%(toread, self.pos))
                chunk = f.read(toread)
                if chunk is None or len(chunk)==0: break
                left -= len(chunk)
                print chunk
            print "%d bytes"%f.tell()
        finally:
            f.close()

def upload(ebook, local, remote):
    print "Uploading %s as %s"%(local, remote),
    if ebook.isdir(remote):
        print "Remote path is a directory; please specify a filename"
        return
    else:
        f = b.open(remote,"w")
        try:
            f.write(file(local,"rb").read())
            print "%d bytes"%f.tell()
        finally:
            f.close()

def print_dir(ebook, path, recurse):
    files = ebook.listdir(path)
    files.sort()
    from posixpath import join
    dirs = []
    if not path.endswith("/"): path+="/"
    print path+":"
    adddirs = recurse and not (path.startswith('/dev') or path.startswith('/proc'))
    for f in files:
        fpath = join(path,f)
        info = ebook.GetFileInfo(fpath)
        if info:
            if info[1]==2:
                if adddirs: dirs.append(fpath)
                typ='drwxrwxr-x '
            else:
                typ='-rw-rw-r-- '
            print typ+"%13d %s %s"%(info[0], time.strftime("%b %d %Y %H:%M",time.gmtime(info[3])), f)
        #print "Filename: %s\n"%fpath+"\tsize: %d\n"%info[0]+"\ttype: %d\n"%info[1]+"\tcreate time: %s\n"%time.ctime(info[2])+"\twrite time: %s\n"%time.ctime(info[3])+"\tflags: %d\n"% info[4]
    if recurse: print
    if len(dirs):
        for d in dirs: print_dir(ebook, d, recurse)

def myint(s):
    if len(s)>1 and s[0]=='0' and (s[1]=='x' or s[1]=='X'):
        return int(s[2:],16)
    else:
        return int(s)

print "Sony Reader utility 0.3 (c) 2006 Igor Skochinsky"

if len(sys.argv)>2:
    b = Ebook()
    try:
        b.connect()
        if b.connected:
            cmd = sys.argv[1]
            if cmd=="ls":
                d = sys.argv[2]
                recurse = len(sys.argv)>3 and sys.argv[3]=="-R"
                print_dir(b, d, recurse)
            elif cmd=="get":
                filename = sys.argv[2]
                limit = None
                if len(sys.argv)>3: limit = myint(sys.argv[3])
                download(b, filename, ".", limit)
            elif cmd=="cat":
				filename = sys.argv[2]
				limit = None
				if len(sys.argv)>3: limit = myint(sys.argv[3])
				downloadToConsole(b, filename, ".", limit)
            elif cmd=="put" and enableWriting and len(sys.argv)>3:
                localfile, remotefile = sys.argv[2:4]
                upload(b, localfile, remotefile)
            elif cmd=="del" and enableWriting and len(sys.argv)>2:
                remotefile = sys.argv[2]
                b.FileDelete(remotefile)
            elif cmd=="um" and enableWriting:
                mode = sys.argv[2]
                if mode=="normal":
                    res = b.UpdateChangeMode(0)
                elif mode=="recovery":
                    res = b.UpdateChangeMode(1)
                else:
                    print "Unknown mode"
                    usage()
            elif cmd=="pinfo":
                name = sys.argv[2]
                info = b.UpdateGetPartitionSize(name)
                if info:
                    print "Info for partition %s"%name
                    print "Device: %d, start sector: 0x%x, image size: 0x%x"%info
                else:
                    print "Error getting info for partition %s!"%name
            elif cmd=="pwrite" and enableWriting and len(sys.argv)>3:
                name = sys.argv[2]
                if len(sys.argv)==4:
                    info = b.UpdateGetPartitionSize(name)
                else:
                    info = myint(sys.argv[4]), myint(sys.argv[5]), myint(sys.argv[6])
                if info:
                    print "Info for partition %s"%name
                    print "Device: %d, start sector: 0x%x, image size: 0x%x"%info
                    fname = sys.argv[3]
                    #data = file(fname,"rb").read()
                    ck = cksum(fname)
                    res = b.UpdateCreatePartitionWithImage(name, fname, info, ck[0], ck[1])
                else:
                    print "Error getting info for partition %s!"%name
            else:
                Usage()
        else:
            print "Device is not connected to PC"
    finally:
        b.disconnect()
else:
    Usage()