# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.196"  #"View Pie Chart": Legend changed to make font size variable based on number of legend values to paint.

from qt.core import Qt,QRect,QPoint,QSize,QImage,QColor,QPainter,QWidget,QDialog,QVBoxLayout,QFont

from itertools import cycle

DEFAULT_COLORS = [0x3366cc, 0xdc3912, 0xff9900, 0x109618, 0x990099,
                                  0x0099c6, 0xdd4477, 0x66aa00, 0xb82e2e, 0x316395,
                                  0x994499, 0x22aa99, 0xaaaa11, 0x6633cc, 0x16d620,
                                  0xFF6666, 0x66CCFF, 0xFF33FF, 0x666600, 0xFF00CC,
                                  0x0033FF, 0x000033, 0x99FF33, 0x9999FF, 0x006600,
                                  0xFFFFCC]

#--------------------------------------------------------------------------------------------
class Chart(object):
    def __init__(self, data, colors=None):
        self.data = data
        self.colors = colors

        self._ref_col = 0
        self._ref_isv = True

    def drawLegend(self, painter, rectangle):
        SPACE = 2

        font = QFont()
        font.setBold(False)

        n_num_legend_rows = len(self.data.rows) # Version 1.0.196:  "View Pie Chart": Legend changed to make font size variable based on number of legend values to paint.
        if n_num_legend_rows < 13:
            font.setPointSize(12)
        elif n_num_legend_rows < 25:
            font.setPointSize(11)
        elif n_num_legend_rows < 35:
            font.setPointSize(9)
        elif n_num_legend_rows < 40:
            font.setPointSize(8)
        else:
            font.setPointSize(7)    #no higher than 7 or not enough room in legend for all 50...

        painter.setFont(font)

        font_metrics = painter.fontMetrics()
        size = font_metrics.xHeight() * 4

        y = SPACE
        x0 = SPACE
        x1 = x0 + size + SPACE * 3


        w = rectangle.width() - size - SPACE
        tw = w - x1

        painter.save()
        painter.translate(rectangle.x(), rectangle.y())

        color = self._icolors()
        for i, column in enumerate(self._fetchLegendData()):
            if (y + size + SPACE * 2) >= (rectangle.y() + rectangle.height()) and i < (len(self.data.columns) - 1):
                painter.drawText(x1, y, tw, size, Qt.AlignLeft | Qt.AlignVCenter, "!!!")
                y += size + SPACE
                break
            text = font_metrics.elidedText(column, Qt.ElideRight, tw)
            painter.fillRect(x0, y, size, size, QColor(next(color)))
            painter.drawText(x1, y, tw, size, Qt.AlignLeft | Qt.AlignVCenter , text)
            y += size + SPACE

        painter.setPen(Qt.lightGray)
        painter.drawRect(0, 0, w, y)
        painter.restore()

    def _fetchLegendData(self):
        for i, column in enumerate(self.data.columns):
            if i != self._ref_col:
                yield column

    def _icolors(self):
        if self.colors is None:
            return cycle(DEFAULT_COLORS)
        return cycle(self.colors)
#--------------------------------------------------------------------------------------------
class PieChart(Chart):
    def draw(self, painter, rectangle):
        painter.save()
        painter.translate(rectangle.x(), rectangle.y())

        # Calculate Values
        vtotal = float(sum(row[not self._ref_col] for row in self.data.rows))
        values = [row[not self._ref_col] / vtotal for row in self.data.rows]

        # Draw Characters
        start_angle = 90 * 16
        start_angle = int(start_angle)
        for color, v in zip(self._icolors(), values):
            span_angle = v * -360.0 * 16
            span_angle = int(span_angle)
            painter.setPen(Qt.white)
            painter.setBrush(QColor(color))
            painter.drawPie(rectangle, start_angle, span_angle)
            start_angle += span_angle

        painter.restore()

    def _fetchLegendData(self):
        for row in self.data.rows:
            yield row[self._ref_col]
#--------------------------------------------------------------------------------------------
class ViewerPieChart(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        self.graph = None

    def setGraph(self, func):
        self.graph = func
        self.update()

    def paintEvent(self, event):
        painter = QPainter(self)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing)

        if self.graph is not None:
            self.graph.draw(painter, QRect(0, 0, event.rect().width(), event.rect().height()))

        painter.end()
#--------------------------------------------------------------------------------------------
class ViewerLegend(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        self.graph = None

    def setGraph(self, func):
        self.graph = func
        self.update()

    def paintEvent(self, event):
        painter = QPainter(self)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing)

        if self.graph is not None:
            self.graph.drawLegend(painter, QRect(0, 0, event.rect().width(), event.rect().height()))

        painter.end()
#--------------------------------------------------------------------------------------------
class DataTable(object):
    def __init__(self):
        self.columns = []
        self.rows = []

    def addColumn(self, label):
        self.columns.append(label)

    def addRow(self, row):
        self.rows.append(row)
#--------------------------------------------------------------------------------------------
#END OF qpainter_charts.py