# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.237"  # Per Library Tweak for 'Default Author Link', default_author_link.  Notes Viewer: Ctrl+C conflict; Ctrl+Z now for Create Instance.

import os,sys
import re, sre_constants
import copy
from functools import partial
import subprocess, webbrowser

from qt.core import (QAbstractItemView, QAction, QApplication, QButtonGroup, QCheckBox, QColor, QComboBox,
                                    QDialog, QDialogButtonBox, QFont, QFrame, QGridLayout,QHBoxLayout, QHeaderView,
                                    QIcon, QInputDialog, QKeySequence, QLabel, QLineEdit, QMargins, QMenu, QModelIndex, QPushButton, QRadioButton,
                                    QScrollArea, QSize, QStandardItem, QStandardItemModel, QTableView,
                                    QTextBrowser, QTextEdit, QTextOption, QTimer, QVBoxLayout, QWidget, Qt, pyqtSignal)

from calibre.constants import DEBUG, ismacos, iswindows
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import gprefs, error_dialog, info_dialog, question_dialog
from calibre.library.comments import comments_to_html
from calibre.utils.config_base import tweaks
from calibre.utils.html2text import html2text

from polyglot.builtins import as_unicode

from calibre_plugins.job_spy.config import (NOTES_VIEW_INSTANCE_SPECIFIC_PREFS_KEYS_SET,
                                                                        NOTES_VIEW_PARENT_INSTANCE_SPECIFIC_PREFS_KEYS_SET)

MD = 'md'
HTML = 'html'
PLAIN = 'plain'
MAX_AUTHOR_TITLE_LENGTH = 85
color_white = QColor("white")
tool_name = "Notes Viewer"
ellipsis = '…'
REGEX_DEFAULT = ""
CURRENTCHANGED = "currentChanged"
OTHER = "other"
SORT_BY_TITLE = 1
SORT_BY_AUTHOR = 2
SORT_BY_BOOKID = 3
SORT_BY_SNIPPET = 4
SORT_VALUE_HEADING = "Sort Value: "

#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(200, 200)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent, Qt.WindowSystemMenuHint|Qt.WindowMinimizeButtonHint )
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
class NotesViewerDialog(SizePersistedDialog):

    child_active_signal = pyqtSignal(int)

    def __init__(self,maingui,parent,js_icon,prefs,instance_number,
                        return_from_notes_viewer_to_save_prefs,
                        return_from_notes_viewer_customization_to_save_prefs,
                        return_from_notes_viewer_to_cancel_html_edits_and_restart):
        unique_pref_name = 'Job_Spy:notesviewer_dialog'
        if instance_number > 1:
            unique_pref_name = unique_pref_name + str(instance_number)
            self.instance_parent = parent
        else:
            self.instance_parent = maingui
        #self.instance_parent is used to directly call the child's parent's instance's functions; parent is Qt's parent for window handling.
        SizePersistedDialog.__init__(self, parent=maingui, unique_pref_name=unique_pref_name)
        #-----------------------------------------------------
        self.maingui = maingui
        self.gui = maingui
        self.guidb = self.maingui.library_view.model().db
        self.original_guidb = self.guidb
        self.js_icon = js_icon
        self.nv_prefs = prefs
        self.prefs_changes_dict = {}     #only keys that have changed in instance
        self.instance_number = instance_number   # 1 is the parent of 2 and 3
        self.return_from_notes_viewer_to_save_prefs = return_from_notes_viewer_to_save_prefs
        self.return_from_notes_viewer_customization_to_save_prefs = return_from_notes_viewer_customization_to_save_prefs
        self.return_from_notes_viewer_to_cancel_html_edits_and_restart = return_from_notes_viewer_to_cancel_html_edits_and_restart
        #-----------------------------------------------------
        if self.instance_number == 1:
            self.instance_parent = parent  # self.maingui
            if DEBUG: print("#1:  self.instance_parent: ", str(self.instance_parent))
        else:
            self.instance_parent = parent  # instance #1's calibre_plugins.job_spy.notes_viewer_dialog.NotesViewerDialog object
            if DEBUG: print("child: self.instance_parent: ", str(self.instance_parent))
        #-----------------------------------------------------
        self.is_dark_mode = QApplication.instance().is_dark_theme
        self.custom_columns_metadata_dict = self.maingui.current_db.field_metadata.custom_field_metadata()
        self.initialize_last_used_vars_by_instance() # initializes:  self.last_current_column; self.startup_column; self.last_bookid.
        self.is_fatal_error = False
        self.original_long_text = ""
        errors = self.build_regular_expressions_for_guessing()
        if self.is_fatal_error:
            msg = "Fatal error in compiling Regular Expressions to guess at Plain/Markdown/HTML:\n\n" + str(errors)
            error_dialog(self.maingui, _(tool_name),_(msg), show=True)
            return None
        self.markdown_module_was_imported = False
        self.cancel_edits_requires_restart = False
        self.total_instances_active = 1
        self.is_closing = False
        self.is_proper_closing_event = False
        self.is_valid_long_text = False
        #-----------------------------------------------------

        self.mytitle = 'JS+:  Notes Viewer'
        if self.instance_number > 1:
            self.mytitle = self.mytitle + "  " + str(self.instance_number)
        self.setWindowTitle(self.mytitle)
        self.setWindowIcon(self.js_icon)
        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint )
        self.setSizeGripEnabled(True)
        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(11)
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.setLayout(self.layout_frame)
        self.layout_frame.setAlignment(Qt.AlignCenter)

        self.setLayout(self.layout_frame)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignCenter)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(600,600)
        self.scroll_area_frame.setFocusPolicy(Qt.FocusPolicy.NoFocus)

        self.layout_frame.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_frame

        # NOTE: the self.scroll_area_frame.setWidget(self.scroll_widget) is at the end of the init() AFTER all children have been created and assigned to a layout...
        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.scroll_widget.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_frame.addWidget(self.scroll_widget)            # causes automatic reparenting of QWidget to the parent of self.layout_frame, which is:  self .
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignCenter)
        #-----------------------------------------------------
        self.scroll_widget.setLayout(self.layout_top)                 # causes automatic reparenting of any widget later added below to the above parent
        #----------------------------------------
        self.layout_author = QHBoxLayout()
        self.layout_author.setSpacing(10)
        self.layout_author.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_author)
        #-----------------------------------------------------
        font.setPointSize(9)
        #-----------------------------------------------------
        self.move_library_view_index_up_pushbutton = QPushButton("&Up▲", self)  # 🞀🞂🞃🞁
        self.move_library_view_index_up_pushbutton.setMaximumWidth(40)
        self.move_library_view_index_up_pushbutton.setDefault(False)
        self.move_library_view_index_up_pushbutton.setFont(font)
        self.move_library_view_index_up_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to move the current Calibre Book Cursor up one row.")
        self.move_library_view_index_up_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.move_library_view_index_up_pushbutton.clicked.connect(self.move_library_view_index_up)
        self.layout_author.addWidget(self.move_library_view_index_up_pushbutton)
        #-----------------------------------------------------
        font.setPointSize(11)
        #-----------------------------------------------------
        self.author_label = QLabel("Author:")
        self.author_label.setFont(font)
        self.author_label.setMaximumWidth(650)
        self.author_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_author.addWidget(self.author_label)
        #-----------------------------------------------------
        font.setPointSize(9)
        #-----------------------------------------------------
        self.layout_title = QHBoxLayout()
        self.layout_title.setSpacing(10)
        self.layout_title.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_title)

        self.move_library_view_index_down_pushbutton = QPushButton("&Dn▼", self)
        self.move_library_view_index_down_pushbutton.setFont(font)
        self.move_library_view_index_down_pushbutton.setMaximumWidth(40)
        self.move_library_view_index_down_pushbutton.setDefault(False)
        self.move_library_view_index_down_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to move the current Calibre Book Cursor down one row.")
        self.move_library_view_index_down_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.move_library_view_index_down_pushbutton.clicked.connect(self.move_library_view_index_down)
        self.layout_title.addWidget(self.move_library_view_index_down_pushbutton)
        #-----------------------------------------------------
        font.setPointSize(11)
        #-----------------------------------------------------
        self.title_label = QLabel("Title:")
        self.title_label.setFont(font)
        self.title_label.setMaximumWidth(650)
        self.title_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_title.addWidget(self.title_label)
        #-----------------------------------------------------
        self.build_menu_for_main_window(font)
        self.build_menu_for_search_results(font)
        #-----------------------------------------------------
        font.setPointSize(9)
        #--------------------------------------------------
        self.layout_column = QHBoxLayout()
        self.layout_column.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_column)
        #--------------------------------------------------
        self.frame_1 = QFrame()
        self.frame_1.setFrameShape(QFrame.HLine)
        self.frame_1.setFrameShadow(QFrame.Sunken)
        self.frame_1.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_1)
        #--------------------------------------------------
        self.layout_radios = QHBoxLayout()
        self.layout_radios.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_radios)

        self.move_library_view_index_back_pushbutton = QPushButton("&Bk🡄", self)
        self.move_library_view_index_back_pushbutton.setFont(font)
        self.move_library_view_index_back_pushbutton.setMaximumWidth(40)
        self.move_library_view_index_back_pushbutton.setDefault(False)
        self.move_library_view_index_back_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to return (go back) to the previous book, wherever it might be.")
        self.move_library_view_index_back_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.move_library_view_index_back_pushbutton.clicked.connect(self.move_library_view_index_back)
        self.layout_radios.addWidget(self.move_library_view_index_back_pushbutton)

        self.custom_column_combobox = QComboBox()
        self.custom_column_combobox.setEditable(False)
        self.custom_column_combobox.setFrame(True)
        self.custom_column_combobox.setDuplicatesEnabled(True)
        self.custom_column_combobox.setMaxVisibleItems(10)
        self.custom_column_combobox.setMaximumWidth(200)
        self.custom_column_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.custom_column_combobox.setFont(font)
        self.custom_column_combobox.setPlaceholderText("#Columns")
        self.custom_column_combobox.setToolTip("<p style='white-space:wrap'>Long-Text/Comments Custom Columns.\n\nUse the Arrow Keys to move up and down the List of Custom Columns that are valid for this Notes Viewer to show.")
        self.layout_radios.addWidget(self.custom_column_combobox)

        self.custom_column_combobox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.block_combobox_signals = False

        font.setPointSize(9)

        self.show_custom_column_combobox_pushbutton = QPushButton("&CC", self)
        self.show_custom_column_combobox_pushbutton.setText("&CC")
        self.show_custom_column_combobox_pushbutton.setFont(font)
        self.show_custom_column_combobox_pushbutton.setMaximumWidth(30)
        self.show_custom_column_combobox_pushbutton.setDefault(False)
        self.show_custom_column_combobox_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to open up the list of all available Custom Columns for this Viewer.")
        self.show_custom_column_combobox_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.show_custom_column_combobox_pushbutton.clicked.connect(self.show_custom_column_combobox_popup)
        self.layout_radios.addWidget(self.show_custom_column_combobox_pushbutton)

        self.comments_datatype_list = []
        for custcol,custcol_dict in self.custom_columns_metadata_dict.items():
            if custcol_dict['datatype'] == "comments":
                self.comments_datatype_list.append(custcol)
        #END FOR

        self.comments_datatype_list.sort()

        self.get_saved_preferences()

        for custcol in self.comments_datatype_list:
            hide = self.determine_custom_column_customization_states(custcol,reason='KEEP')
            if hide:  #  True means "Hide"; False means "Keep".
                pass
            else:
                self.custom_column_combobox.addItem(custcol)
        #END FOR

        self.custom_column_combobox.setCurrentIndex(-1)

        self.radio_group_0_label = QLabel(" ")
        self.radio_group_0_label.setFont(font)
        self.radio_group_0_label.setMinimumWidth(2)
        self.radio_group_0_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_radios.addWidget(self.radio_group_0_label)

        self.lock_current_custom_column_checkbox = QCheckBox("Loc&k?")
        self.lock_current_custom_column_checkbox.setFont(font)
        self.lock_current_custom_column_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to 'lock' what the current Custom Column is regardless of what else you do?<br>The book is never 'locked'.")
        self.layout_radios.addWidget(self.lock_current_custom_column_checkbox)

        self.lock_current_custom_column_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.lock_current_custom_column = False
        self.lock_current_custom_column_state = False

        self.freeze_notes_viewer_checkbox = QCheckBox("&Freeze?")
        self.freeze_notes_viewer_checkbox.setFont(font)
        self.freeze_notes_viewer_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to 'Freeze' Notes Viewer where it is, and ignore whatever else you do in the Library until you 'Unfreeze' it'")
        self.layout_radios.addWidget(self.freeze_notes_viewer_checkbox)

        self.freeze_notes_viewer_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.event_freeze_notes_viewer_checkbox(None)

        self.freeze_current_custom_column = False

        self.radio_group_1_label = QLabel(" ")
        self.radio_group_1_label.setFont(font)
        self.radio_group_1_label.setMinimumWidth(90)
        self.radio_group_1_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_radios.addWidget(self.radio_group_1_label)

        self.plain_radio = QRadioButton('&Plain')
        self.plain_radio.setFont(font)
        self.layout_radios.addWidget(self.plain_radio)
        self.md_radio = QRadioButton('&Markdown')
        self.md_radio.setFont(font)
        self.layout_radios.addWidget(self.md_radio)
        self.html_radio = QRadioButton('H&TML')
        self.html_radio.setFont(font)
        self.layout_radios.addWidget(self.html_radio)

        self.format_radio_button_group = QButtonGroup(self.layout_radios)
        self.format_radio_button_group.setExclusive(True)
        self.format_radio_button_group.addButton(self.plain_radio)
        self.format_radio_button_group.addButton(self.md_radio)
        self.format_radio_button_group.addButton(self.html_radio)

        t = "<p style='white-space:wrap'>The text format to be used to show the current Custom Column's text."
        self.plain_radio.setToolTip(t)
        self.md_radio.setToolTip(t)
        self.html_radio.setToolTip(t)

        self.plain_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.md_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.html_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.radio_group_2_label = QLabel(" ")
        self.radio_group_2_label.setFont(font)
        self.radio_group_2_label.setMinimumWidth(55)
        self.radio_group_2_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_radios.addWidget(self.radio_group_2_label)

        help = self.return_help()
        t = "<p style='white-space:wrap'>View Mode or Edit Mode.<br>" + help

        self.view_radio = QRadioButton('&View')
        self.view_radio.setFont(font)
        self.view_radio.setToolTip(t)
        self.layout_radios.addWidget(self.view_radio)
        self.edit_radio = QRadioButton('&Edit')
        self.edit_radio.setFont(font)
        self.edit_radio.setToolTip(t)
        self.layout_radios.addWidget(self.edit_radio)

        self.view_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.edit_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.is_edit_mode = False

        self.view_edit_radio_button_group = QButtonGroup(self.layout_radios)
        self.view_edit_radio_button_group.setExclusive(True)
        self.view_edit_radio_button_group.addButton(self.view_radio)
        self.view_edit_radio_button_group.addButton(self.edit_radio)

        self.view_radio.setChecked(True)

        self.convert_format_to_html_pushbutton = QPushButton("To HTML", self)  # & shortcut removed due to conflict; button is normally hidden, and rarely used.
        self.convert_format_to_html_pushbutton.setFont(font)
        self.convert_format_to_html_pushbutton.setMaximumWidth(60)
        self.convert_format_to_html_pushbutton.setDefault(False)
        self.convert_format_to_html_pushbutton.setToolTip("<p style='white-space:wrap'>In Edit Mode, either convert the Plain Text to 'basic' HTML, \
        or convert or Markdown Text to its HTML equivalent.")
        self.convert_format_to_html_pushbutton.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.convert_format_to_html_pushbutton.clicked.connect(self.convert_comments_to_html)
        self.layout_radios.addWidget(self.convert_format_to_html_pushbutton)

        self.convert_format_to_html_pushbutton.hide()

        self.is_html_being_edited = False
        #--------------------------------------------------
        self.layout_note_editor = QHBoxLayout()
        self.layout_note_editor.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_note_editor)

        self.full_note_html_editor = None        #~ ...created on first use...thereafter hidden until time to show it
        #--------------------------------------------------
        self.frame_2 = QFrame()
        self.frame_2.setFrameShape(QFrame.HLine)
        self.frame_2.setFrameShadow(QFrame.Sunken)
        self.frame_2.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_2)
        #--------------------------------------------------
        self.layout_search = QHBoxLayout()
        self.layout_search.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_search)
        #--------------------------------------------------

        self.search_row_objects_to_hide_during_html_edit_list = []

        self.customize_notes_viewer_pushbutton = QPushButton("&Options", self)
        self.customize_notes_viewer_pushbutton.setFont(font)
        self.customize_notes_viewer_pushbutton.setMaximumWidth(50)
        self.customize_notes_viewer_pushbutton.setDefault(False)
        self.customize_notes_viewer_pushbutton.setToolTip("<p style='white-space:wrap'>Customize Notes Viewer.")
        self.customize_notes_viewer_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.customize_notes_viewer_pushbutton.clicked.connect(self.customize_notes_viewer)
        self.layout_search.addWidget(self.customize_notes_viewer_pushbutton)

        if self.instance_number == 1:
            self.search_row_objects_to_hide_during_html_edit_list.append(self.customize_notes_viewer_pushbutton)
        else:
            self.customize_notes_viewer_pushbutton.hide()
            self.customize_notes_viewer_pushbutton.setEnabled(False)

        self.layout_search.addStretch(1)

        self.search_regex_pushbutton = QPushButton("RegExp&.", self)
        self.search_regex_pushbutton.setFont(font)
        self.search_regex_pushbutton.setMaximumWidth(55)
        self.search_regex_pushbutton.setDefault(False)
        self.search_regex_pushbutton.setToolTip("<p style='white-space:wrap'>Regular Expression to use for 'Search'.")
        self.search_regex_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.search_regex_pushbutton.clicked.connect(self.edit_search_regular_expression )
        self.layout_search.addWidget(self.search_regex_pushbutton)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.search_regex_pushbutton)

        self.search_regex_qlineedit = QLineEdit(self)
        self.search_regex_qlineedit.setText(REGEX_DEFAULT)
        self.search_regex_qlineedit.setFont(font)
        self.search_regex_qlineedit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.search_regex_qlineedit.setMinimumWidth(150)
        self.search_regex_qlineedit.setMaximumWidth(150)
        self.search_regex_qlineedit.setToolTip("<p style='white-space:wrap'>Enter the words or phrase or regular expression to search for in the current Custom Column for all Books in this Library.\
        <br><br>Example: 'Fandom|dragon'<br>Example: 'mitochondria' <br>Example:  'sample size|toxin'\
        <br><br>The vertical bar symbol '|' means 'OR' in a regular expression.<br><br>Refer to the Calibre user manual for more about regular expressions.\
        <br><br>Special Function:  If you leave the Regular Expression blank (empty), then the very first sentence in every book's current Custom Column will be shown in the results list.\
        This is to facilitate browsing the Authors, Titles and current Custom Column (first sentence only) for all books in your Library, then selecting one that interests you, and 'jumping' directly to it.")
        self.layout_search.addWidget(self.search_regex_qlineedit)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.search_regex_qlineedit)

        self.search_regex_qlineedit.returnPressed.connect(self.do_regex_search_populate_standarditemmodel_items)

        self.do_search_push_button = QPushButton("Src&h", self)
        self.do_search_push_button.setFont(font)
        self.do_search_push_button.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.do_search_push_button.setToolTip("<p style='white-space:wrap'>Search the selected Custom Column using the selected Regular Expression.\
        <br><br>Special Function:  If you leave the Regular Expression blank (empty), then the very first sentence in every book's current Custom Column will be shown in the results list.\
        This is to facilitate browsing the Authors, Titles and current Custom Column (first sentence only) for all books in your Library, then selecting one that interests you, and 'jumping' directly to it.")
        self.do_search_push_button.setMaximumWidth(35)
        self.do_search_push_button.clicked.connect(self.do_regex_search_populate_standarditemmodel_items)
        self.layout_search.addWidget(self.do_search_push_button)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.do_search_push_button)

        self.matching_book_title_rows_combobox = QComboBox()
        self.matching_book_title_rows_combobox.setEditable(False)
        self.matching_book_title_rows_combobox.setPlaceholderText("Matching Title, Author, BookID & Snippet")
        self.matching_book_title_rows_combobox.setDuplicatesEnabled(False)
        self.matching_book_title_rows_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)   # AdjustToContents AdjustToMinimumContentsLengthWithIcon    AdjustToContentsOnFirstShow
        self.matching_book_title_rows_combobox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.matching_book_title_rows_combobox.setFrame(True)
        self.matching_book_title_rows_combobox.setFont(font)
        #~ self.matching_book_title_rows_combobox.setMaxVisibleItems(10)
        self.matching_book_title_rows_combobox.setMinimumWidth(400)  # Note: this has a direct impact on the width of the Popup() of the qtableview for the model of this qcombobox.
        self.matching_book_title_rows_combobox.setMaximumWidth(400)  # a value of 375 could cause ~25 characters to be cut off of the far right column, Snippets, compared to 400.
        t = "<p style='white-space:wrap'>Book Author, Title, ID, Snippet for Notes matching the Regular Expression.\
        <br><br>Selecting a drop-down results list book can cause an 'automatic jump' to that book if the 'automatic jump' checkbox is checked.\
        <br><br>To avoid an 'automatic jump', simply press the Escape key to exit the drop-down list of results.\
        <br><br>If the 'automatic jump' checkbox is not checked, simply by clicking the 'jump' button, you may jump to the book currently shown in the results list drop-down."
        self.matching_book_title_rows_combobox.setToolTip(t)
        self.layout_search.addWidget(self.matching_book_title_rows_combobox)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.matching_book_title_rows_combobox)

        self.matching_book_title_rows_combobox.currentTextChanged.connect(self.event_jump_to_matching_book)  # there is no returnPressed event.

        self.show_matching_book_push_button = QPushButton("&List", self)
        self.show_matching_book_push_button.setFont(font)
        self.show_matching_book_push_button.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.show_matching_book_push_button.setToolTip("<p style='white-space:wrap'>Show the list of matching Books to the left.")
        self.show_matching_book_push_button.setMaximumWidth(35)
        self.show_matching_book_push_button.clicked.connect(self.show_matching_book_combobox_popup)
        self.layout_search.addWidget(self.show_matching_book_push_button)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.show_matching_book_push_button)

        self.auto_jump_to_matching_book_checkbox = QCheckBox("&AJ")
        self.auto_jump_to_matching_book_checkbox.setFont(font)
        self.auto_jump_to_matching_book_checkbox.setToolTip("<p style='white-space:wrap'>Do you wish to 'automatically jump' to the new drop-down results list current book when it changes?")
        self.auto_jump_to_matching_book_checkbox.setMaximumWidth(35)
        self.auto_jump_to_matching_book_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.layout_search.addWidget(self.auto_jump_to_matching_book_checkbox)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.auto_jump_to_matching_book_checkbox)

        #~ -------------------------- convert to library-specific customization --------------------------delete code block after April 26, 2023...
        if 'GUI_TOOLS_NOTES_VIEWER_AUTO_JUMP_CHECKBOX_STATE' in prefs:
            state = prefs['GUI_TOOLS_NOTES_VIEWER_AUTO_JUMP_CHECKBOX_STATE']
            if bool(state) is True:
                self.autojump_state = True
            else:
                self.autojump_state = False
            del prefs['GUI_TOOLS_NOTES_VIEWER_AUTO_JUMP_CHECKBOX_STATE']
            prefs
        #~ -------------------------- convert to library-specific customization --------------------------delete code block after April 26, 2023...

        self.auto_jump_to_matching_book_checkbox.setChecked(self.autojump_state)

        self.jump_to_matching_book_push_button = QPushButton("&Jump", self)
        self.jump_to_matching_book_push_button.setFont(font)
        self.jump_to_matching_book_push_button.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.jump_to_matching_book_push_button.setToolTip("<p style='white-space:wrap'>Jump directly to the matching Book selected to the left.")
        self.jump_to_matching_book_push_button.setMaximumWidth(35)
        self.jump_to_matching_book_push_button.clicked.connect(self.jump_to_matching_book)
        self.layout_search.addWidget(self.jump_to_matching_book_push_button)

        self.search_row_objects_to_hide_during_html_edit_list.append(self.jump_to_matching_book_push_button)

        self.block_auto_jump_signals = False

        #--------------------------------------------------
        font.setPointSize(11)
        #--------------------------------------------------
        self.frame_3 = QFrame()
        self.frame_3.setFrameShape(QFrame.HLine)
        self.frame_3.setFrameShadow(QFrame.Sunken)
        self.frame_3.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_3)
        #--------------------------------------------------
        self.plain_qtextedit =  QTextEdit(self)
        self.plain_qtextedit.setReadOnly(True)
        self.plain_qtextedit.setFont(font)
        self.plain_qtextedit.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.layout_top.addWidget(self.plain_qtextedit)
        self.plain_qtextedit.clear()
        self.plain_qtextedit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.is_plain = True
        self.plain_qtextedit.show()
        self.plain_qtextedit.setEnabled(True)
        #--------------------------------------------------
        self.markdown_qtextedit =  QTextEdit(self)
        self.markdown_qtextedit.setReadOnly(True)
        self.markdown_qtextedit.setFont(font)
        self.markdown_qtextedit.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.layout_top.addWidget(self.markdown_qtextedit)
        self.markdown_qtextedit.clear()
        self.markdown_qtextedit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.is_markdown  = False
        self.markdown_qtextedit.hide()
        self.markdown_qtextedit.setEnabled(False)

        #--------------------------------------------------
        self.html_qtextbrowser =  QTextBrowser(self)
        self.html_qtextbrowser.setReadOnly(True)
        self.html_qtextbrowser.setFont(font)
        self.html_qtextbrowser.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.layout_top.addWidget(self.html_qtextbrowser)
        self.html_qtextbrowser.clear()
        self.html_qtextbrowser.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.is_html = False
        self.html_qtextbrowser.hide()
        self.html_qtextbrowser.setEnabled(False)
        self.html_qtextbrowser.setOpenExternalLinks(True)
        self.html_qtextbrowser.setOpenLinks(False)  # Internal Links means read the file contents within the widget itself!
        self.html_qtextbrowser.clearHistory()

        self.html_qtextbrowser.addAction(self.focus_keyboard_on_textbrowser_action)
        #--------------------------------------------------

        font.setPointSize(7)

        #--------------------------------------------------
        self.layout_bottom = QHBoxLayout()
        self.layout_bottom.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_bottom)
        #--------------------------------------------------
        self.current_bookid_label = QLabel("")
        self.current_bookid_label.setAlignment(Qt.AlignLeft)
        self.current_bookid_label.setFont(font)
        self.current_bookid_label.setToolTip("<p style='white-space:wrap'>The BookID of the current book.  To move the current cursor, which may be elsewhere, to that book:  Ctrl+G")
        self.current_bookid_label.setMaximumWidth(100)
        self.current_bookid_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_bottom.addWidget(self.current_bookid_label)
        #--------------------------------------------------

        font.setPointSize(9)

        #--------------------------------------------------
        self.bottom_buttonbox = QDialogButtonBox()
        self.layout_bottom.addWidget(self.bottom_buttonbox)

        self.push_button_save_edits_notes_viewer = QPushButton(" ", self)
        self.push_button_save_edits_notes_viewer.setFont(font)
        self.push_button_save_edits_notes_viewer.setText("&Save Edits")
        self.push_button_save_edits_notes_viewer.setToolTip("<p style='white-space:wrap'>Save any Edits made while in Edit Mode.")
        self.push_button_save_edits_notes_viewer.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_save_edits_notes_viewer.clicked.connect(self.save_edit_mode)
        self.bottom_buttonbox.addButton(self.push_button_save_edits_notes_viewer,QDialogButtonBox.AcceptRole)

        self.push_button_save_edits_notes_viewer.hide()

        self.push_button_refresh_notes_viewer = QPushButton(" ", self)
        self.push_button_refresh_notes_viewer.setFont(font)
        self.push_button_refresh_notes_viewer.setText("&Refresh Current Book")
        self.push_button_refresh_notes_viewer.setDefault(False)
        self.push_button_refresh_notes_viewer.setToolTip("<p style='white-space:wrap'>The text displayed will be refreshed from the currently selected book.\
                                                                                                                                <br><br>Clicking the mouse cursor on a book will automatically refresh the displayed text,\
                                                                                                                                as will moving the cursor via the keyboard to a new book or Column and then pressing the Enter/Return key.")
        self.push_button_refresh_notes_viewer.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_refresh_notes_viewer.clicked.connect(self.refresh_notes_viewer)
        self.bottom_buttonbox.addButton(self.push_button_refresh_notes_viewer,QDialogButtonBox.AcceptRole)

        self.refresh_button_is_cancel_edits = False

        self.push_button_save_settings_and_exit = QPushButton(" ", self)
        self.push_button_save_settings_and_exit.setFont(font)
        self.push_button_save_settings_and_exit.setText("E&xit Notes Viewer")
        self.push_button_save_settings_and_exit.setToolTip("<p style='white-space:wrap'>Save the current Notes Viewer window position and size, and then exit.")
        self.push_button_save_settings_and_exit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_save_settings_and_exit.clicked.connect(lambda:self.save_settings_and_exit(restart=False,source="ExitButton"))
        self.bottom_buttonbox.addButton(self.push_button_save_settings_and_exit,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(True)

        self.layout_bottom.addSpacing(100)

        font.setPointSize(7)

        self.push_button_create_child_instance = QPushButton("", self)
        self.push_button_create_child_instance.setIcon(self.js_icon)
        self.push_button_create_child_instance.setFont(font)
        self.push_button_create_child_instance.setMaximumSize(35,20)  #(minw, minh)
        #~ self.push_button_create_child_instance.setMaximumWidth(50)
        self.push_button_create_child_instance.setToolTip("<p style='white-space:wrap'>Create the next available child instance of Notes Viewer.  The current available to be created is shown. \
        <br><br>Library-specific Option.  Up to 3 instances of Notes Viewer may exist simultaneously, which includes the primary instance (#1).\
        <br><br>Instances #2 and #3 are 'children' of this primary instance (#1), and will be closed automatically when this primary instance (#1) exits.\
        <br><br>Only this primary instance (#1) may change Notes Viewer Options and create instances #2 and #3.\
        <br><br>Instances #2 and #3 are created with their 'Freeze' checkbox initially checked so that you may move on to another Note within this primary instance (#1) without losing your position in the Library List.")
        self.push_button_create_child_instance.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_create_child_instance.clicked.connect(self.create_nv_child_instance)
        self.layout_bottom.addWidget(self.push_button_create_child_instance)

        self.update_push_button_create_child_instance_text()

        if self.max_instances_allowed > 1:
            self.push_button_create_child_instance.show()
        else:
            self.push_button_create_child_instance.hide()

        if self.instance_number != 1:
            self.push_button_create_child_instance.hide()
            self.push_button_create_child_instance.setEnabled(False)

        font.setPointSize(9)

        #-----------------------------------------------------
        self.style_focus_on_widgets()
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        self.resize(self.sizeHint())
        #-----------------------------------------------------
        self.resize_dialog()
        #-----------------------------------------------------
        self.update()
        QApplication.instance().processEvents()
        #-----------------------------------------------------
        self.setTabOrder(self.move_library_view_index_up_pushbutton,self.move_library_view_index_down_pushbutton)
        self.setTabOrder(self.move_library_view_index_down_pushbutton,self.move_library_view_index_back_pushbutton)
        self.setTabOrder(self.move_library_view_index_back_pushbutton,self.custom_column_combobox)
        self.setTabOrder(self.custom_column_combobox,self.show_custom_column_combobox_pushbutton)
        self.setTabOrder(self.show_custom_column_combobox_pushbutton,self.lock_current_custom_column_checkbox)
        self.setTabOrder(self.lock_current_custom_column_checkbox,self.freeze_notes_viewer_checkbox)
        self.setTabOrder(self.freeze_notes_viewer_checkbox,self.plain_radio)
        self.setTabOrder(self.plain_radio,self.md_radio)
        self.setTabOrder(self.md_radio,self.html_radio)
        self.setTabOrder(self.html_radio,self.view_radio)
        self.setTabOrder(self.view_radio,self.edit_radio)
        self.setTabOrder(self.edit_radio,self.customize_notes_viewer_pushbutton)
        self.setTabOrder(self.customize_notes_viewer_pushbutton,self.search_regex_pushbutton)
        self.setTabOrder(self.search_regex_pushbutton,self.search_regex_qlineedit)
        self.setTabOrder(self.search_regex_qlineedit,self.do_search_push_button)
        self.setTabOrder(self.do_search_push_button,self.matching_book_title_rows_combobox)
        self.setTabOrder(self.matching_book_title_rows_combobox,self.show_matching_book_push_button)
        self.setTabOrder(self.show_matching_book_push_button,self.auto_jump_to_matching_book_checkbox)
        self.setTabOrder(self.auto_jump_to_matching_book_checkbox,self.jump_to_matching_book_push_button)
        #~ QTextEdit/QTextBrowser TabOrders are not static and are handled in an event...
        self.setTabOrder(self.push_button_save_edits_notes_viewer,self.push_button_refresh_notes_viewer)
        self.setTabOrder(self.push_button_refresh_notes_viewer,self.push_button_save_settings_and_exit)
        self.setTabOrder(self.push_button_save_settings_and_exit,self.push_button_create_child_instance)
        self.setTabOrder(self.push_button_create_child_instance,self.move_library_view_index_up_pushbutton)

        #-----------------------------------------------------
        self.block_radio_signals = True
        self.block_view_edit_radio_signals = True
        self.block_refresh_signals = True
        #-----------------------------------------------------

        global_default_column = prefs['GUI_TOOLS_NOTES_VIEWER_DEFAULT_CUSTOM_COLUMN_GLOBAL']
        if global_default_column.startswith("#"):
            if global_default_column in self.comments_datatype_list:
                self.default_column = global_default_column

        if self.default_column is not None:
            self.last_current_column = self.default_column
            self.startup_column = self.last_current_column
            self.custom_column_combobox.setCurrentText(self.default_column)
            if DEBUG: print("global_default_column has been set for use at startup: ", global_default_column)

        if self.instance_number == 1:
            try:
                if self.last_current_column.startswith("#"):
                    vpos = self.maingui.library_view.verticalScrollBar().value()  # if user opens NV before *any* book selected in view, at all; just ran Calibre and then with mouse scrolled down many pages...
                    self.startup_column = self.last_current_column
                    cix = self.maingui.library_view.currentIndex()
                    bookid = None
                    if cix.isValid():
                        bookid = self.maingui.library_view.model().id(cix)
                    else:
                        bookid = self.maingui.library_view.current_book()
                    idx = self.maingui.library_view.column_map.index(self.last_current_column)
                    current_column = self.maingui.library_view.column_map[idx]
                    idx = self.maingui.library_view.column_map.index(current_column)
                    current_column = self.maingui.library_view.column_map[idx]
                    if DEBUG: print("current_column per column_map: ", current_column )
                    if bookid is None:
                        row_number = 0
                        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
                    else:
                        row_number =  self.maingui.library_view.model().db.data.id_to_index(bookid)
                        if DEBUG: print("row_number of current book is: ", str(row_number))
                        if row_number is not None:
                            self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
                    self.custom_column_combobox.setCurrentText(current_column)
                    self.last_current_column = current_column
                    if DEBUG: print("new self.last_current_column: ", self.last_current_column )
                    self.maingui.library_view.verticalScrollBar().setValue(vpos)  # return as close as possible to where the user was when NV initialized
                else:
                    pass
            except Exception as e:
                if DEBUG: print("if self.instance_number == 1: vpos -- Exception in __init__ near end: ", str(e))
        #-----------------------------------------------------
        self.block_radio_signals = False
        self.block_view_edit_radio_signals = False
        self.block_refresh_signals = False
        self.block_jumping = False
        self.view_radio.setChecked(True)
        #-----------------------------------------------------
        if not self.last_bookid in self.all_book_ids_set:
            self.last_bookid = 0
        #-----------------------------------------------------
        if self.instance_number > 1:
            #~ #self.nv_prefs were vetted to ensure the library_id has not changed since they were last saved.
            #~ #if it had, self.last_bookid would be 0, and self.last_current_column would be "".
            if self.last_bookid > 0:
                if self.last_current_column.startswith("#"):
                    try:
                        if DEBUG: print("Child Instance: ", str(self.instance_number), ": Restore to self.last_bookid & self.last_current_column: ", self.last_current_column, str(self.last_bookid))
                        idx = self.maingui.library_view.column_map.index(self.last_current_column)
                        row_number = self.maingui.library_view.model().db.data.id_to_index(self.last_bookid)
                        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
                        self.set_text_background_color()
                        QApplication.instance().processEvents()
                        self.refresh_notes_viewer()
                    except Exception as e:
                        if DEBUG: print("\n\nException in:  Restore to self.last_bookid & self.last_current_column: ", self.last_current_column, str(self.last_bookid), "\n -- ", str(e))
                        msg = "Last Used Book does not currently exist.  Either you recently deleted it, or you are using a Virtual Library that does not contain it, or you did Library search, or you clicked a Virtual Library tab."
                        if DEBUG: print(msg)
                        info_dialog(self.maingui, _(tool_name),_(msg), show=True)
            self.lock_state = self.lock_current_custom_column_checkbox.isChecked()
            self.lock_current_custom_column_state = self.lock_state #freezing only
            self.freeze_notes_viewer_checkbox.setChecked(True)
            self.event_freeze_notes_viewer_checkbox(None)
        else:
            self.lock_current_custom_column_state = self.lock_current_custom_column_checkbox.isChecked()

        #-----------------------------------------------------
        #~ Now connect the special widgets
        #-----------------------------------------------------
        self.html_qtextbrowser.anchorClicked.connect(self.event_anchorclicked)

        self.plain_radio.toggled.connect(self.event_text_type_button_changed)
        self.md_radio.toggled.connect(self.event_text_type_button_changed)
        self.html_radio.toggled.connect(self.event_text_type_button_changed)
        self.view_radio.toggled.connect(self.event_view_edit_buttons_changed)
        self.edit_radio.toggled.connect(self.event_view_edit_buttons_changed)
        self.custom_column_combobox.currentIndexChanged.connect(self.event_custom_column_combobox_changed)
        self.lock_current_custom_column_checkbox.toggled.connect(self.event_lock_current_custom_column_checkbox)
        self.freeze_notes_viewer_checkbox.toggled.connect(self.event_freeze_notes_viewer_checkbox)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.view_radio.setChecked(True)
        self.startup_column = None
        self.goback_bookid = None
        self.maingui.library_view.selectionModel().currentChanged.connect(self.currentChangedEvent)
        if self.instance_number > 1:
            self.child_active_signal.connect(self.childActiveEvent)
        #-----------------------------------------------------
        self.plain_qtextedit.setEnabled(True)
        self.markdown_qtextedit.setEnabled(True)
        self.html_qtextbrowser.setEnabled(True)
        self.plain_qtextedit.setReadOnly(True)
        self.markdown_qtextedit.setReadOnly(True)
        self.html_qtextbrowser.setReadOnly(True)
        self.refresh_notes_viewer(source=OTHER)
        self.set_text_background_color()
        QApplication.instance().processEvents()
        #-----------------------------------------------------
        self.n_rows = self.maingui.library_view.model().rowCount(QModelIndex())
        self.model_created = False
        self.standarditemmodel_initialized = False
        self.set_user_tweaks()
        self.create_url_decoding_list()
        self.notes_viewer_instance_1_dialog = self
        self.notes_viewer_instance_2_dialog = None
        self.notes_viewer_instance_3_dialog = None
        #-----------------------------------------------------
        self.lock_current_custom_column_checkbox.setChecked(self.lock_state)  #library-specific preference
        self.lock_current_custom_column = self.lock_current_custom_column_checkbox.isChecked()
        self.lock_current_custom_column_state = self.lock_state #freezing only
        #-----------------------------------------------------
        self.search_current_item_setFocus()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_text_type_button_changed(self,event):
        if self.block_radio_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        self.long_text = self.original_long_text  #avoid recursive reformatting

        if self.plain_radio.isChecked():
            self.plain_qtextedit.setEnabled(True)
            self.is_plain = True
            self.is_markdown = False
            self.is_html = False
            plain_text = self.format_plain_text(self.long_text)
            self.plain_qtextedit.setPlainText(plain_text)
            self.plain_qtextedit.show()
            self.markdown_qtextedit.hide()
            self.html_qtextbrowser.hide()
            self.plain_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.markdown_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.html_qtextbrowser.setFocusProxy(self.plain_qtextedit)
            self.setTabOrder(self.jump_to_matching_book_push_button,self.plain_qtextedit)
            self.setTabOrder(self.plain_qtextedit,self.push_button_save_edits_notes_viewer)
            self.markdown_qtextedit.setEnabled(False)
            self.html_qtextbrowser.setEnabled(False)

        elif self.md_radio.isChecked():
            self.markdown_qtextedit.setEnabled(True)
            self.is_plain = False
            self.is_markdown = True
            self.is_html = False
            self.is_html_being_edited = False
            markdown_text = self.format_markdown_text(self.long_text)
            self.markdown_qtextedit.setMarkdown(markdown_text)   # https://doc.qt.io/qt-6/qtextedit.html#markdown-prop
            self.plain_qtextedit.hide()
            self.markdown_qtextedit.show()
            self.html_qtextbrowser.hide()
            self.markdown_qtextedit.setFocusProxy(self.markdown_qtextedit)
            self.plain_qtextedit.setFocusProxy(self.markdown_qtextedit)
            self.html_qtextbrowser.setFocusProxy(self.markdown_qtextedit)
            self.setTabOrder(self.jump_to_matching_book_push_button,self.markdown_qtextedit)
            self.setTabOrder(self.markdown_qtextedit,self.push_button_save_edits_notes_viewer)
            self.plain_qtextedit.setEnabled(False)
            self.html_qtextbrowser.setEnabled(False)

        elif self.html_radio.isChecked():
            self.html_qtextbrowser.setEnabled(True)
            self.is_plain = False
            self.is_markdown = False
            self.is_html = True
            self.is_html_being_edited = True
            html_text = self.format_plain_text(self.long_text)
            self.html_qtextbrowser.setHtml(html_text)                                 # https://doc.qt.io/qt-6/qtextedit.html#html-prop
            self.plain_qtextedit.hide()
            self.markdown_qtextedit.hide()
            self.html_qtextbrowser.show()
            self.html_qtextbrowser.setFocusProxy(self.html_qtextbrowser)
            self.plain_qtextedit.setFocusProxy(self.html_qtextbrowser)
            self.markdown_qtextedit.setFocusProxy(self.html_qtextbrowser)
            self.setTabOrder(self.jump_to_matching_book_push_button,self.html_qtextbrowser)
            self.setTabOrder(self.html_qtextbrowser,self.push_button_save_edits_notes_viewer)
            self.plain_qtextedit.setEnabled(False)
            self.markdown_qtextedit.setEnabled(False)
            self.html_qtextbrowser_setFocus()

        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_custom_column_combobox_changed(self,event):

        if self.block_combobox_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        if self.lock_current_custom_column:
            self.is_valid_long_text = True
            self.block_combobox_signals = True
            self.custom_column_combobox.setCurrentText(self.last_current_column)
            self.block_combobox_signals = False

            if not self.lock_current_custom_column_checkbox.isEnabled():
                self.lock_current_custom_column_checkbox.setEnabled(True)
                self.lock_current_custom_column_checkbox.setChecked(self.lock_state)

            idx = self.maingui.library_view.column_map.index(self.last_current_column)
            row_number = self.maingui.library_view.model().db.data.id_to_index(self.last_bookid)
            self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        current_column = self.custom_column_combobox.currentText()

        if current_column is None:
            return

        if not current_column.startswith("#"):
            return

        k = 'GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED'
        v = str(self.last_bookid)
        self.prefs_changes_dict[k] = v

        if not self.last_bookid > 0:
            return

        k = 'GUI_TOOLS_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED'
        v = current_column
        self.prefs_changes_dict[k] = v

        idx = self.maingui.library_view.column_map.index(current_column)
        current_column = self.maingui.library_view.column_map[idx]
        row_number = self.maingui.library_view.model().db.data.id_to_index(self.last_bookid)
        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
        self.custom_column_combobox.setCurrentText(current_column)
        self.refresh_notes_viewer(source=OTHER)
        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_view_edit_buttons_changed(self,event):
        if self.block_view_edit_radio_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        if self.view_radio.isChecked():
            self.is_edit_mode = False
        else:
            self.is_edit_mode = True

        if self.is_edit_mode:
            #~ from: View to: Edit Mode
            if self.is_html_being_edited:
                self.plain_qtextedit.setReadOnly(True)
            else:
                self.plain_radio.setChecked(True)
                self.plain_qtextedit.setReadOnly(False)
                self.cancel_edits_requires_restart = False

            self.push_button_save_edits_notes_viewer.show()
            self.push_button_refresh_notes_viewer.setText(" &Cancel Column Edits ")
            self.refresh_button_is_cancel_edits = True
            self.plain_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.markdown_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.html_qtextbrowser.setFocusProxy(self.plain_qtextedit)
            self.setTabOrder(self.edit_radio,self.plain_qtextedit)
            self.setTabOrder(self.plain_qtextedit,self.push_button_save_edits_notes_viewer)
            self.markdown_qtextedit.setReadOnly(True)
            self.html_qtextbrowser.setReadOnly(True)
            self.markdown_qtextedit.setEnabled(False)
            self.html_qtextbrowser.setEnabled(False)

            if self.is_html_being_edited:
                self.create_notes_editor()
            else:
                self.convert_format_to_html_pushbutton.show()
                self.plain_qtextedit.setFocus(Qt.OtherFocusReason)

        else:
            #~ from: Edit to: View Mode
            self.push_button_save_edits_notes_viewer.hide()
            self.push_button_refresh_notes_viewer.setText("&Refresh Current Book")
            self.refresh_button_is_cancel_edits = False
            self.plain_qtextedit.setEnabled(True)
            self.markdown_qtextedit.setEnabled(True)
            self.html_qtextbrowser.setEnabled(True)
            self.plain_qtextedit.setReadOnly(True)
            self.markdown_qtextedit.setReadOnly(True)
            self.html_qtextbrowser.setReadOnly(True)
            self.plain_radio.setChecked(True) # Qt can change tab orders by itself, so reset them...
            self.convert_format_to_html_pushbutton.hide()
            self.is_html_being_edited = False
            self.move_library_view_index_up_pushbutton.setFocus(Qt.OtherFocusReason)

        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_freeze_notes_viewer_checkbox(self,event):
        if self.freeze_notes_viewer_checkbox.isChecked():
            self.block_refresh_signals = True
            self.block_combobox_signals = True
            self.lock_current_custom_column_state = self.lock_current_custom_column_checkbox.isChecked()
            self.lock_current_custom_column = True
            self.custom_column_combobox.setEnabled(False)
            self.block_auto_jump_signals = True
            self.block_jumping = True
            self.freeze_notes_viewer_checkbox.setStyleSheet("QCheckBox { color: black; background-color: papayawhip;}")
        else:
            self.block_refresh_signals = False
            self.block_combobox_signals = False
            self.lock_current_custom_column = self.lock_current_custom_column_state
            self.custom_column_combobox.setEnabled(True)
            self.block_auto_jump_signals = False
            self.block_jumping = False
            if not self.is_dark_mode:
                self.freeze_notes_viewer_checkbox.setStyleSheet("QCheckBox { color: black; background-color: none; }")
            else:
                self.freeze_notes_viewer_checkbox.setStyleSheet("QCheckBox { color: white; background-color: black; }")
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_lock_current_custom_column_checkbox(self,event):
        if self.lock_current_custom_column_checkbox.isChecked():
            self.lock_current_custom_column = True
        else:
            self.lock_current_custom_column = False
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_anchorclicked(self,event):
        #~ Windows' path requires forward-slashes plus double-quotes to allow spaces in path as defined in js_dict:  {'http:': '"C:/Program Files (x86)/Microsoft/Edge/Application/msedge.exe"'}
        #~ But: the Tweak will be automatically corrected below if it needs to be.

        msg = \
        '''
        #~ Calibre > Preferences > Plugin Tweaks >   job_spy_notes_viewer_protocol_default_apps = {'http:': '"C:/Program Files (x86)/Microsoft/Edge/Application/msedge.exe"'}
        #~ Note that path to executable should be in Python path format (i.e., / not \), and Windows paths require double-quotes if there are any spaces in a path.
        #~ However, NV will automatically correct the application path for the above as necessary.
        '''
        if DEBUG: print(msg)

        if DEBUG: print("\nevent_anchorclicked: ", str(event))  # PyQt6.QtCore.QUrl('file:///X:%5CMyPluginsPy3%5C_ris_import_dir%5Cabstract.txt')
        #~ event_anchorclicked:  PyQt6.QtCore.QUrl("file:///D:/Calibre/Libraries/Test/Anat Admati/The Bankers' New Clothes (159)/data/")

        url = str(event)
        if DEBUG: print(url)
        url = url[19:  ]  #   "PyQt6.QtCore.QUrl('  OR PyQt6.QtCore.QUrl("
        if DEBUG: print(url)
        url = url.replace("')","")  # sometimes the QUrl is surrounded by single quotes:   if there is no real single quote (e.g. Bankers), the QUrl uses single quotes.
        if DEBUG: print(url)
        url = url.replace('")','')    # sometimes the QUrl is surrounded by double quotes:  if there is a real single quote (e.g. Banker's), the QUrl uses double quotes.
        if DEBUG: print(url)
        url = url.strip()
        if DEBUG: print("url to be opened: ", url)   #  url to be opened:  file:///X:%5CMyPluginsPy3%5C_ris_import_dir%5Cris_files%5C10.1371_journal.pcbi.1010869.ris
                                                                                #~ url to be opened:  file:///D:/Calibre/Libraries/Test/Anat Admati/The Bankers' New Clothes (159)/data/")

        #~ ------------------------------------------------------------------------------------------------------------------------------
        #~ sometimes the QUrl is surrounded by single quotes:
                #~ PyQt6.QtCore.QUrl('file:///X:%5CMyPluginsPy3%5C_ris_import_dir%5Cabstract.txt')
        #~ sometimes the QUrl is surrounded by double quotes:
                #~ grammatical single quote: PyQt6.QtCore.QUrl("file:///D:/Calibre/Libraries/Test/Anat Admati/The Bankers' New Clothes (159)/data/")
        #~ ------------------------------------------------------------------------------------------------------------------------------

        FILE = "file:"
        HTTP = "http:"
        HTTPS = "https:"
        ONENOTEHTTP = "onenote:http:"
        ONENOTEHTTPS = "onenote:https:"
        CALIBRESPY = "calibrespy:"

        #~ Note:  calibrespy: usage is not required in Tweaks; it is hardcoded here instead.  However, the URL in the html must be a valid 'command line' argument in the user's OS
        #~ For Windows, the requirement in a .bat file to have double-quotes around arguments with spaces must be honored, but this is problematic in Notes Viewer links.
        #~ If the Calibre Library path has spaces, and hence requires double-quotes, the following option, '--prefilter" ', is ignored.
        #~ So, either don't expect prefiltering if a Library path is specified in Windows, use Library paths with no spaces, or use the basic syntax that requires the user to select a Library each time.
        #~ <br>
        #~ <a href="calibrespy:calibre-debug -r CalibreSpy -- --prefilter">CalibreSpy:  -- --prefilter</a>
        #~ <br>
        #~ <a href="calibrespy:calibre-debug -r CalibreSpy -- --S:\Calibre\CalibreJobSpyTest1 --prefilter">CalibreSpy: -- --S:\Calibre\CalibreJobSpyTest1 --prefilter</a>
        #~ <br>

        app = None
        appurl = None
        tweak_found = False
        is_calibrespy = False

        #~ if DEBUG:
            #~ url = 'file:///X:%5CMyPluginsPy3%5C_ris_import_dir%5Cabstract.txt'
            #~ url = "file:///D:/Calibre/Libraries/Default/!Attachments/book%20%5B30%5D/"
            #~ url = "file:///D:/Calibre/Libraries/Default/!Attachments/book%20(30)"
            #~ url = "file:///D:/Calibre/Libraries/Default/!Attachments/book%2030/"

        for row in self.decode_url_encoded_list:  #~ example:  book%20%5B30%5D  decodes as:  book [30]
            from_,to_ = row
            url = url.replace(from_,to_)
        #END FOR
        if DEBUG: print("File url after replacement of url encoding: ", url)

        if FILE in url:
            if iswindows:
                if url.endswith(".exe") or url.endswith(".bat"):
                    try:
                        p_pid = subprocess.Popen(url, shell=True)
                        return
                    except Exception as e:
                        # webbrowser will next attempt to open it but automatically pass it to the correct app based on its file extension
                        if DEBUG: print("event_anchorclicked: p_pid = subprocess.Popen(url, shell=True): ", str(e))
                else:
                    pass  # webbrowser will open it but automatically pass it to the correct app based on its file extension
            else:
                try:
                    p_pid = subprocess.Popen(url, shell=True)
                    return
                except Exception as e:
                    if DEBUG: print("event_anchorclicked: p_pid = subprocess.Popen(url, shell=True): ", str(e))

        elif ONENOTEHTTP in url:
            if ONENOTEHTTP in self.js_tweaks_dict:  # Calibre > Preferences > Plugin Tweaks > ...
                app = self.js_tweaks_dict[ONENOTEHTTP]
                tweak_found = True
                url = url.replace("onenote:","")
                if DEBUG: print("ONENOTEHTTP used.")

        elif ONENOTEHTTPS in url:
            if ONENOTEHTTPS in self.js_tweaks_dict:
                app = self.js_tweaks_dict[ONENOTEHTTPS]
                tweak_found = True
                url = url.replace("onenote:","")
                if DEBUG: print("ONENOTEHTTPS used.")

        elif HTTPS in url:
            if HTTPS in self.js_tweaks_dict:  # Calibre > Preferences > Plugin Tweaks > ...
                app = self.js_tweaks_dict[HTTPS]
                tweak_found = True
                if DEBUG: print("HTTPS used.")

        elif HTTP in url:
            if HTTP in self.js_tweaks_dict:
                app = self.js_tweaks_dict[HTTP]
                tweak_found = True
                if DEBUG: print("HTTP used.")

        elif CALIBRESPY in url:
            url = url.replace("%5C","/")
            cmdline = url.replace(CALIBRESPY,"")
            tweak_found = False
            is_calibrespy = True
            if DEBUG: print("CALIBRESPY used: ", cmdline)

        if is_calibrespy:
            try:
                if DEBUG: print("CALIBRESPY:  event_anchorclicked: p_pid = subprocess.Popen(cmdline, shell=True): ", cmdline)
                return subprocess.Popen(cmdline, shell=True)
            except Exception as e:
                msg = "Error:  CALIBRESPY:  event_anchorclicked: p_pid = subprocess.Popen(cmdline, shell=True): " + cmdline + "   " + str(e)
                if DEBUG: print(msg)
                return  info_dialog(self.maingui, _(tool_name),_(msg), show=True)

        if tweak_found:
            if app is not None:
                if iswindows:   # Windows' path requires double-quotes to allow spaces in path...fix user error setting Tweak.
                    if not app.startswith('"'):
                        app = '"' + app
                    if not app.endswith('"'):
                        app = app + '"'
                app = app.replace(os.sep,"/")  # c:/ not c:\
                appurl = app + ' ' + url  # command-line format for executing app with filename as parameter
                if DEBUG: print("Corrected if needed:  app + ' ' + url = appurl: ", appurl)

        if tweak_found:
            try:
                if DEBUG: print("event_anchorclicked: p_pid = subprocess.Popen(appurl, shell=True): ", appurl)
                return subprocess.Popen(appurl, shell=True)
            except Exception as e:
                if DEBUG: print("Error:  event_anchorclicked: p_pid = subprocess.Popen(appurl, shell=True): ", appurl, str(e))
        else:
            if FILE in url:
                url = url.replace("file:///","")       #~ file:///C:/Users/MeMyselfandI/Desktop/New Text Document.txt
                url = url.replace(os.sep,"/")
                url = url.strip()
                #~ caution: do NOT add double-quotes surrounding the paths for the files being handled by webbrowser; they will fail the os.path.isfile test, and are unnecessary anyway.
                if (not os.path.isfile(url)) and (not os.path.isdir(url)):
                    msg = "File or Directory in url does not exist in stated path: " + url
                    if DEBUG: print(msg)
                    return info_dialog(self.maingui, _(tool_name),_(msg), show=True)
            else:
                pass

            try:
                if DEBUG: print("webbrowser.open(url, new=0): ", url)
                return webbrowser.open(url, new=0)
            except Exception as e:
                if DEBUG: print("Error:  event_anchorclicked: webbrowser.open(url, new=0): ", url, str(e))
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_jump_to_matching_book(self,event):
        # The self.last_matching_book_title_rows_combobox.currentChanged() event fires multiple times for every Search...never auto-jump because the Search changed the combo's contents.
        current = self.last_bookid
        if self.auto_jump_to_matching_book_checkbox.isChecked():
            if not self.block_auto_jump_signals:
                if self.instance_number > 1: #child
                    self.child_about_to_move()
                else:
                    self.instance_parent_about_to_move()
                self.freeze_notes_viewer_checkbox.setChecked(False)
                self.jump_to_matching_book()
                self.goback_bookid = current
    #---------------------------------------------------------------------------------------------------------------------------------------
    def currentChangedEvent(self,event):
        QApplication.instance().processEvents()
        if self.instance_number > 1: #child
            if self.freeze_notes_viewer_checkbox.isChecked():
                if DEBUG: print("child--currentChanged: self.freeze_notes_viewer_checkbox.isChecked() for instance: ", str(self.instance_number), " so nothing done.")
                return
            else:#child itself is being moved by user
                if DEBUG: print("child--currentChanged: self.child_active_signal.emit(1) then self.refresh_notes_viewer for instance ", str(self.instance_number))
                self.instance_parent.freeze_notes_viewer_checkbox.setChecked(True)
                #~ self.child_active_signal.emit(1)
                QApplication.instance().processEvents()
                #~ must give time for parent to freeze...
                return self.refresh_notes_viewer()
        else: #parent
            try:
                if self.freeze_notes_viewer_checkbox.isChecked():
                    if DEBUG: print("parent--currentChanged: self.freeze_notes_viewer_checkbox.isChecked() for instance: ", str(self.instance_number), " so nothing done.")
                    return
                else:  #parent itself is being moved by user
                    QApplication.instance().processEvents()
                    if DEBUG: print("parent--currentChanged: self.refresh_notes_viewer for instance: ", str(self.instance_number))
                    return self.refresh_notes_viewer()
            except Exception as e:
                if DEBUG: print("currentChangedEvent:  Exception in parent instance; changing libraries by clicking a calibre:// url? ", str(e))
    #---------------------------------------------------------------------------------------------------------------------------------------
    def childActiveEvent(self,value):
        #~ -----------------------------------------------------
        #~ Child must directly freeze parent immediately before parent gets moved by child-triggered currentChanged
        #~ -----------------------------------------------------
        #~ child_active_signal = pyqtSignal()
        #~ self.child_active_signal.connect(self.childActiveEvent)
        #~ self.child_active_signal.emit()
        #~ -----------------------------------------------------
        if DEBUG: print("\nchildActiveEvent: instance: ", str(self.instance_number), "\n")
        if self.instance_number > 1:
            self.instance_parent.freeze_notes_viewer_checkbox.setChecked(True)
            open_kids_dict = self.return_open_kids_dict()
            if self.instance_number == 2:
                if 3 in open_kids_dict:
                    sibling = open_kids_dict[3]
                    sibling.freeze_notes_viewer_checkbox.setChecked(True)
                    del sibling
            else:
                if 2 in open_kids_dict:
                    sibling = open_kids_dict[2]
                    sibling.freeze_notes_viewer_checkbox.setChecked(True)
                    del sibling
            QApplication.instance().processEvents()
            del open_kids_dict
    #---------------------------------------------------------------------------------------------------------------------------------------
    def child_about_to_move(self):
        self.child_active_signal.emit(1)
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def instance_parent_about_to_move(self):
        if DEBUG: print("\instance_parent_about_to_move\n")
        if self.instance_number > 1:
            return
        try:
            child = self.notes_viewer_instance_2_dialog
            child.freeze_notes_viewer_checkbox.setChecked(True)
            del child
        except:
            pass
        try:
            child = self.notes_viewer_instance_3_dialog
            child.freeze_notes_viewer_checkbox.setChecked(True)
            del child
        except:
            pass
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def closeEvent(self, event):
        if self.instance_number > 1:
            if DEBUG: print("\n\n-----------------------------\n\nChild: closeEvent: self.is_proper_closing_event: ", str(self.instance_number), self.is_proper_closing_event, "...ABOUT TO SELF.CLOSE()...")
            if not self.is_proper_closing_event:  # user clicked the big "X" in the window
                if DEBUG: print("did the user click the big 'X' in the window? Probably.")
                self.instance_parent.return_from_child_instance_to_close(self,self.instance_number,restart=False,restart_param=None)  # this instance's prefs changes will be handled differently
            self.close()
            if DEBUG: print("Child:  closeEvent:  AFTER SELF.CLOSE(): ",str(self.instance_number), "\n\n-----------------------------\n\n")
            return
    #---------------------------------------------------------------------------------------------------------------------------------------
    def keyPressEvent(self, event):
        # Confirm Escape Key to avoid exiting Notes Viewer erroneously.
        if event.key() == Qt.Key.Key_Escape:
            if not question_dialog(self, _(tool_name),('Did you intend to exit Notes Viewer?')):
                return
        QDialog.keyPressEvent(self, event)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index_up(self):
        if self.instance_number > 1: #child
            self.child_about_to_move()
        current = self.last_bookid
        self.move_library_view_index(-1)
        self.goback_bookid = current
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index_down(self):
        if self.instance_number > 1: #child
            self.child_about_to_move()
        current = self.last_bookid
        self.move_library_view_index(1)
        self.goback_bookid = current
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index_back(self):
        if self.goback_bookid is None:
            return
        if self.instance_number > 1: #child
            self.child_about_to_move()
        current = self.last_bookid
        self.move_library_view_index(0,self.goback_bookid)
        self.goback_bookid = current
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index(self,increment=0,bookid=None):
        if bookid is None or bookid == 0 :
            bookid = self.last_bookid
        if bookid == 0:
            return
        idx = self.maingui.library_view.column_map.index(self.last_current_column)
        current_column = self.maingui.library_view.column_map[idx]
        row_number = self.maingui.library_view.model().db.data.id_to_index(bookid)
        row_number = row_number + increment
        if row_number < 0:
            row_number = 0
        if row_number > self.n_rows:
            row_number = self.n_rows - 1
        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
        self.custom_column_combobox.setCurrentText(current_column)
        self.refresh_notes_viewer(source=OTHER)
        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def go_to_current_book(self):
        if self.instance_number > 1:
            self.child_about_to_move()
        else:
            self.instance_parent_about_to_move()

        self.freeze_notes_viewer_checkbox.setChecked(False)
        QApplication.instance().processEvents()
        bookid = self.last_bookid
        self.goback_bookid = bookid
        idx = self.maingui.library_view.column_map.index(self.last_current_column)
        current_column = self.maingui.library_view.column_map[idx]
        row_number = self.maingui.library_view.model().db.data.id_to_index(bookid)
        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
        self.custom_column_combobox.setCurrentText(current_column)

        self.refresh_notes_viewer(source=OTHER)
        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def refresh_notes_viewer_qtimer(self):
        QTimer.singleShot(600,self.refresh_notes_viewer_other)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def refresh_notes_viewer_other(self):
        self.hide_notes_editor()
        self.refresh_notes_viewer(source=OTHER)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def refresh_notes_viewer(self,source=CURRENTCHANGED):

        if self.edit_radio.isChecked():
            if not question_dialog(self, _(tool_name),('Did you intend to discard your changes?')):
                return

        #~ normal and routine scenario for html
        if self.block_refresh_signals:
            if self.is_html_being_edited:
                if self.refresh_button_is_cancel_edits:
                    if self.edit_radio.isChecked():
                        if self.html_radio.isChecked():
                            self.return_from_notes_viewer_to_cancel_html_edits_and_restart()
                            return

        #~ normal and routine scenario for markdown
        #~ backup scenario for html when user makes random clicks
        if self.cancel_edits_requires_restart:  # includes saving and restart
            if self.instance_number > 1:
                self.child_about_to_move()
            else:
                self.instance_parent_about_to_move()
            self.save_settings_and_exit(restart=True,source="refresh_notes_viewer")
            return

        if self.block_refresh_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        if self.lock_current_custom_column:
            self.block_combobox_signals = True
            self.custom_column_combobox.setCurrentText(self.last_current_column)
            self.block_combobox_signals = False
            # but: must allow book to change, keeping column locked

        if self.edit_radio.isChecked():
            self.block_refresh_signals = True
            self.view_radio.setChecked(True)
            self.block_refresh_signals = False
            return
        else:
            self.push_button_save_edits_notes_viewer.hide()

        if source == CURRENTCHANGED:
            if not self.last_bookid == 0:
                self.goback_bookid = self.last_bookid

        #~ --------------------------------------------------------------------------
        #~ --------------------------------------------------------------------------
        i = self.custom_column_combobox.currentIndex()
        if i == -1:
            if self.lock_current_custom_column_checkbox.isChecked():
                self.lock_state = True  #so can change it back later to what it was now
                self.lock_current_custom_column_checkbox.setChecked(False)
            #disable it so it cannot be checked until an existing combobox #column has been selected...
            self.lock_current_custom_column_checkbox.setEnabled(False)   #disable it.
        #~ --------------------------------------------------------------------------
        #~ --------------------------------------------------------------------------

        try:
            cix = self.maingui.library_view.currentIndex()
            if cix.isValid():
                if not self.lock_current_custom_column:
                    #---------------------------
                    current_column = self.get_current_column()
                    if DEBUG: print("current_column: ",current_column)
                    if current_column in self.custom_columns_metadata_dict:
                        custcol_dict = self.custom_columns_metadata_dict[current_column]
                        if custcol_dict['datatype'] == "comments":
                            self.is_valid_long_text = True
                        else:
                            self.is_valid_long_text = False
                    else:
                        self.is_valid_long_text = False
                    if not self.is_valid_long_text:
                        self.block_combobox_signals = True
                        self.custom_column_combobox.setCurrentIndex(-1)
                        self.block_combobox_signals = False
                    else:
                        self.block_combobox_signals = True
                        self.custom_column_combobox.setCurrentText(current_column)
                        self.block_combobox_signals = False
                    #---------------------------
                else:   # bookid is never locked
                    current_column = self.last_current_column

                if current_column == self.last_current_column:
                    self.column_has_changed = False
                else:
                    self.column_has_changed = True

                self.last_current_column = current_column

                bookid = self.maingui.library_view.model().id(cix)

                if bookid == self.last_bookid and self.goback_bookid == self.last_bookid:
                    self.book_has_changed = False
                else:
                    self.book_has_changed = True

                self.last_bookid = bookid

                bookid_str = "BookID: " + str(bookid)
                self.current_bookid_label.setText(bookid_str)

                if self.goback_bookid is None:
                    self.goback_bookid = bookid

                if self.last_current_column.startswith("#"):
                    self.update_nv_last_used_prefs_by_instance()

                #~ Note:  using mi = self.maingui.library_view.model().db.get_metadata avoids complications when changing libraries causing guidb changes, proper disconnection of events, and user not properly closing this dialog before changing the library.
                mi = self.maingui.library_view.model().db.get_metadata(bookid, index_is_id=True, get_user_categories=False)

                authorsx = mi.authors
                if isinstance(authorsx,tuple):
                    authors = list(authorsx)
                if isinstance(authorsx,list):
                    authors = ""
                    for a in authorsx:
                        authors = authors + a + " & "
                    #END FOR
                    authors = authors.strip()
                    if authors.endswith("&"):
                        authors = authors[0:-1]
                        authors = authors.strip()
                    authors = authors[0:MAX_AUTHOR_TITLE_LENGTH]
                title = mi.title
                title = title[0:MAX_AUTHOR_TITLE_LENGTH]
                bookid = str(bookid)
                if self.is_valid_long_text:
                    mi_dict = mi.get_user_metadata(current_column,make_copy=False)
                    long_text = mi_dict['#value#']
                    if long_text is None:
                        long_text = current_column + ":   [No Text]"
                else:
                    long_text = current_column + ":   [Not a Long-Text/Comments Custom Column]"

                #~ --------------------------------------------------------------------------
                #~ --------------------------------------------------------------------------
                if self.is_valid_long_text:
                    if not self.lock_current_custom_column_checkbox.isEnabled():
                        i = self.custom_column_combobox.currentIndex()
                        if i > -1:
                            self.lock_current_custom_column_checkbox.setEnabled(True)  #enable it.
                            self.lock_current_custom_column_checkbox.setChecked(self.lock_state) #whatever it previously was, or whatever customization says the default is.
                        else:
                            pass #already disabled
                else: # not self.is_valid_long_text
                    if (self.column_has_changed) and (not self.book_has_changed): # "bad" column is now current, but book is the same
                        if self.lock_current_custom_column_checkbox.isChecked():
                            self.lock_state = True  #so can change it back later to what it was now
                            self.lock_current_custom_column_checkbox.setChecked(False)
                        #disable it so it cannot be checked until an existing combobox "good" #column has been selected
                        self.lock_current_custom_column_checkbox.setEnabled(False)   #disable it.
                #~ --------------------------------------------------------------------------
                #~ --------------------------------------------------------------------------

                self.long_text = long_text
                self.original_long_text = long_text  # to later avoid recursive reformatting

                #---------------------------
                if len(authors) >= MAX_AUTHOR_TITLE_LENGTH:
                    authors = authors[0:MAX_AUTHOR_TITLE_LENGTH] + ellipsis
                self.author_label.setText(authors)
                if len(title) >= MAX_AUTHOR_TITLE_LENGTH:
                    title = title[0:MAX_AUTHOR_TITLE_LENGTH] + ellipsis
                self.title_label.setText(title)

                self.plain_qtextedit.hide()
                self.markdown_qtextedit.hide()
                self.html_qtextbrowser.hide()

                self.guess_text_type()

                if self.is_fatal_error:
                    return

                self.is_html = False
                self.is_markdown = False
                self.is_plain = False

                if self.best_guess == HTML:
                    self.is_html = True

                elif self.best_guess == MD:
                    self.is_markdown = True
                else:
                    self.is_plain = True

                if self.is_html:
                    self.is_html_being_edited = True
                else:
                    self.is_html_being_edited = False

                if self.is_markdown:
                    self.markdown_being_edited = True
                else:
                    self.markdown_being_edited = False

                self.block_radio_signals = True

                if self.is_plain:
                    plain_text = self.format_plain_text(self.long_text)
                    self.plain_qtextedit.setPlainText(plain_text)
                    self.plain_qtextedit.show()
                    self.plain_radio.setChecked(True)
                elif self.is_markdown :
                    markdown_text = self.format_markdown_text(self.long_text)
                    self.markdown_qtextedit.setMarkdown(markdown_text)
                    self.markdown_qtextedit.show()
                    self.md_radio.setChecked(True)
                elif self.is_html:
                    html_text = self.format_html_text(self.long_text)
                    self.html_qtextbrowser.setHtml(html_text)
                    self.html_qtextbrowser.clearHistory()
                    self.html_qtextbrowser.show()
                    self.html_radio.setChecked(True)

                self.block_radio_signals = False
                #---------------------------
            else:
                if DEBUG: print("cix is NOT valid...")
                return
            #--------------------------------------
            #--------------------------------------
        except Exception as e:
            if DEBUG: print("Exception in NotesViewerDialog:", str(e))
            return

        self.set_text_background_color()

        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def guess_text_type(self):

        self.is_plain_text = False
        self.is_markdown = False
        self.is_html = False

        self.best_guess = None
        type = self.determine_custom_column_customization_states(self.last_current_column,reason='GUESS')
        if type is not None:
            if type == "plain":
                self.best_guess = PLAIN
            elif type == "markdown":
                self.best_guess = MD
            elif type == "html":
                self.best_guess = HTML
            elif type == "guess":
                self.best_guess = None
            if DEBUG: print("Guessing Override from Customization: ", self.best_guess)

        if self.best_guess is not None:
            return

        n_markdown_score = 0
        n_html_score = 0

        n_md_regexes = len(self.markdown_regex_list)
        n_html_regexes = len(self.html_regex_list)

        self.long_text = self.original_long_text  #avoid recursive reformatting

        for r in self.markdown_regex_list:      # p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
            p,regex,fmt = r
            match = p.search(self.long_text)
            if match:
                if DEBUG: print(fmt, ": regex match: ", regex)
                n_markdown_score = n_markdown_score + 1
        #END FOR
        md_match_ratio = n_markdown_score/n_md_regexes
        if DEBUG: print("raw md_match_ratio: ", str(md_match_ratio))

        for r in self.html_regex_list:
            p,regex,fmt = r
            match = p.search(self.long_text)
            if match:
                if DEBUG: print(fmt, ": regex match: ", regex)
                n_html_score = n_html_score + 1
        #END FOR
        html_match_ratio = n_html_score/n_html_regexes
        if DEBUG: print("raw html_match_ratio: ", str(html_match_ratio))

        total_scores = n_markdown_score + n_html_score
        if total_scores > 0:
            md_match_ratio = (n_markdown_score/total_scores) * (md_match_ratio + html_match_ratio)
            html_match_ratio = (n_html_score/total_scores) * (md_match_ratio + html_match_ratio)

        if  md_match_ratio > html_match_ratio:
            self.best_guess = MD
        else:
            self.best_guess = HTML

        #~ but if both are very low, it is probably plain text.
        if md_match_ratio < 0.04  and html_match_ratio < 0.04:
            self.best_guess = PLAIN

        #~ but Calibre comments_to_text causes a 0 md and a 0.027 html.
        if md_match_ratio == 0  and html_match_ratio > 0.02:
            self.best_guess = HTML

        #~ but html regexes are highly specific, so if all things being equal, it is html not md.  override previous guess.
        if self.best_guess == MD:
            if n_html_score == n_markdown_score:
                self.best_guess = HTML
                if DEBUG: print("overridden to HTML based on n_html_score versus n_markdown_score: ", str(n_html_score), " versus ", str(n_markdown_score))

        if DEBUG: print("guessing:\n  MD: score & weighted ratio: ", str(n_markdown_score), str(md_match_ratio), "\n  HTML: score & weighted ratio: ", str(n_html_score), str(html_match_ratio), "\n  --->>> best guess: ", self.best_guess)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def format_plain_text(self,long_text):
        qte = QTextEdit()
        qte.setPlainText(long_text)
        long_text = qte.toPlainText()
        self.long_text = long_text
        del qte
        return long_text
    #---------------------------------------------------------------------------------------------------------------------------------------
    def format_markdown_text(self,long_text):
        qte = QTextEdit()
        qte.setMarkdown(long_text)
        long_text = qte.toMarkdown()
        self.long_text = long_text
        del qte
        return long_text
    #---------------------------------------------------------------------------------------------------------------------------------------
    def format_html_text(self,long_text):
        qte = QTextEdit()
        qte.setHtml(long_text)
        long_text = qte.toHtml()
        self.long_text = long_text
        del qte
        return long_text
    #---------------------------------------------------------------------------------------------------------------------------------------
    def get_current_column(self):
        if self.startup_column is None:
            current_column = None
            current_col = self.maingui.library_view.currentIndex().column()
            current_column = self.maingui.library_view.column_map[current_col]
            if DEBUG: print("current_col: ", str(current_col), "lookup/search name: ", str(current_column))
            if current_column.startswith("#"):
                self.custom_column_combobox.setCurrentText(current_column)
        else:
            if self.startup_column.startswith("#"):
                self.custom_column_combobox.setCurrentText(self.startup_column)
                current_column = self.startup_column
        return current_column
    #---------------------------------------------------------------------------------------------------------------------------------------
    def show_custom_column_combobox_popup(self):
        self.custom_column_combobox.showPopup()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_edit_mode(self):
        #~ editing mode for markdown and plain is done entirely in the plain text format qtextedit; editing html is done in the special html editor.

        if self.is_html_being_edited:
            self.long_text = self.full_note_html_editor.html
            self.hide_notes_editor()
            self.full_note_html_editor.html = ""
        else:
            edited_long_text = self.plain_qtextedit.toPlainText()
            self.long_text = self.format_plain_text(edited_long_text)
            del edited_long_text

        self.original_long_text = self.long_text

        book = self.last_bookid
        payload = []
        payload.append(book)
        mi = Metadata(_('Unknown'))
        custcol_dict = self.custom_columns_metadata_dict[self.last_current_column]
        custcol_dict['#value#'] = self.long_text
        mi.set_user_metadata(self.last_current_column, custcol_dict)
        id_map = {}
        id_map[book] = mi
        edit_metadata_action = self.maingui.iactions['Edit Metadata']

        if self.edit_radio.isChecked():
            self.block_refresh_signals = True
            self.view_radio.setChecked(True)  # required prior to callback executing
            self.block_refresh_signals = False
        else:
            self.block_refresh_signals = False
            self.push_button_save_edits_notes_viewer.hide()

        self.html_qtextbrowser.setEnabled(True)

        edit_metadata_action.apply_metadata_changes(id_map, callback=self.refresh_notes_viewer_qtimer())

        self.convert_format_to_html_pushbutton.hide()

        self.html_qtextbrowser.setReadOnly(True)

        if self.cancel_edits_requires_restart: # will restart regardless whether saved or canceled
            if self.instance_number > 1:
                self.child_about_to_move()
            else:
                self.instance_parent_about_to_move()

        self.refresh_notes_viewer(source=OTHER)

        del id_map
        del mi
        del custcol_dict
    #---------------------------------------------------------------------------------------------------------------------------------------
    def build_regular_expressions_for_guessing(self):

        markdown_regex_list = []   # http://chubakbidpaa.com/interesting/2021/09/28/regex-for-md.html
        markdown_regex_list.append('(#{1}\s)(.*)')  #header 1
        markdown_regex_list.append('(#{2}\s)(.*)')  #header 2
        markdown_regex_list.append('(#{3}\s)(.*)')  #header 3
        markdown_regex_list.append('(#{4}\s)(.*)')  #header 4
        markdown_regex_list.append('(#{5}\s)(.*)')  #header 5
        markdown_regex_list.append('(#{6}\s)(.*)')  #header 6
        markdown_regex_list.append('(\*|\_)+(\S+)(\*|\_)+')                        # boldItalicText
        markdown_regex_list.append("(\_|\*){1}(\w|\W|\d)+(\_|\*){1}")       # Italics
        markdown_regex_list.append("(\_|\*){3}(\w|\W|\d)+(\_|\*){3}")       # bold Italics
        markdown_regex_list.append("(\_|\*){2}(\w|\W|\d)+(\_|\*){2}")        # bold
        markdown_regex_list.append('(\[.*\])(\((http)(?:s)?(\:\/\/).*\))' )      # linkText
        markdown_regex_list.append('(^(\W{1})(\s)(.*)(?:$)?)+')                 # unordered list text
        markdown_regex_list.append('(^(\d+\.)(\s)(.*)(?:$)?)+')                  # numbered list
        markdown_regex_list.append('((^(\>{1})(\s)(.*)(?:$)?)+)')               # block quote
        markdown_regex_list.append("(\\'{1})(.*)(\\'{1})")                             # inline code
        markdown_regex_list.append("(\\'{3}\\n+)(.*)(\\n+\\'{3})")            # code block
        markdown_regex_list.append('(\=|\-|\*){3}')                                     # horizontal line
        markdown_regex_list.append('(\<{1})(\S+@\S+)(\>{1})')              # email text
        tables = "(((\|)([a-zA-Z\d+\s#!@'" + '"():;\\\/.\[\]\^<={$}>?(?!-))]+))+(\|))(?:\n)?((\|)(-+))+(\|)(\n)((\|)(\W+|\w+|\S+))+(\|$)'
        markdown_regex_list.append(tables)                                                 # tables
        markdown_regex_list.append(r'(?<=\[\[).*?(?=(?:\]\]|#|\|))')          # links
        s = '(\!)(\[(?:.*)?\])(\(.*(\.(\w{3}))(?:(\s\"' + "|\')(\w|\W|\d)+" + '(\"' + "|\'))?\))"    # imagefile
        markdown_regex_list.append(s)
        markdown_regex_list.append('(#{%d}\s)(.*)')                                  # HeaderAll  https://github.com/Chubek/md2docx/blob/master/patterns/patterns.go
        markdown_regex_list.append("(\_|\*){3}")                                       # ThreeUA
        markdown_regex_list.append("(\_|\*){2}")                                       #  TwoUA
        markdown_regex_list.append("(\_|\*){1}")                                       #  OneUA
        markdown_regex_list.append("((\|)(-+))+(\|)(\n)")                         #  TableSep

        html_regex_list = []
        html_regex_list.append(r' href=')
        html_regex_list.append(r'<!DOCTYPE html>')
        html_regex_list.append(r'<a|</a|<a href=')
        html_regex_list.append(r'<body|</body')
        html_regex_list.append(r'<br')
        html_regex_list.append(r'<cite|</cite')
        html_regex_list.append(r'<code|</code')
        html_regex_list.append(r'<div|</div')
        html_regex_list.append(r'<em|</em ')
        html_regex_list.append(r'<h1')
        html_regex_list.append(r'<h2')
        html_regex_list.append(r'<h3')
        html_regex_list.append(r'<h4')
        html_regex_list.append(r'<h5')
        html_regex_list.append(r'<h6')
        html_regex_list.append(r'<head|</head ')
        html_regex_list.append(r'<hr')
        html_regex_list.append(r'<html|</html')
        html_regex_list.append(r'<img src=')
        html_regex_list.append(r'<kbd')
        html_regex_list.append(r'<li|</li')
        html_regex_list.append(r'<meta')
        html_regex_list.append(r'<nobr')
        html_regex_list.append(r'<ol|</ol')
        html_regex_list.append(r'<pre|</pre')
        html_regex_list.append(r'<p|<p class="description"|</p>')
        html_regex_list.append(r'<script src="//')
        html_regex_list.append(r'<span|</span')
        html_regex_list.append(r'<strong|</strong')
        html_regex_list.append(r'<style|</style')
        html_regex_list.append(r'<sub|</sub')
        html_regex_list.append(r'<sup|</sup')
        html_regex_list.append(r'<td|</td')
        html_regex_list.append(r'<tr|</tr')
        html_regex_list.append(r'<tt|</tt')
        html_regex_list.append(r'<ul|</ul')
        html_regex_list.append(r'<u|</u')

        self.markdown_regex_list = []
        self.html_regex_list = []

        self.is_fatal_error = False
        errors = "No Errors"

        try:
            for regex in markdown_regex_list:
                p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
                fmt = "MD"
                r = p,regex,fmt
                self.markdown_regex_list.append(r)
            #END FOR
        except Exception as e:
            if DEBUG: print("markdown_regex_list: ", regex, "   ", str(e))
            self.markdown_regex_list.append(None)
            self.is_fatal_error = True
            errors = "markdown regex: " + regex + "     <<<---    " + str(e)

        try:
            for regex in html_regex_list:
                p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
                fmt = "HTML"
                r = p,regex,fmt
                self.html_regex_list.append(r)
            #END FOR
        except Exception as e:
            if DEBUG: print("html_regex_list: ", regex, "   ", str(e))
            self.html_regex_list.append(None)
            self.is_fatal_error = True
            errors = "html regex: " + regex + "     <<<---    " + str(e)

        return errors
    #---------------------------------------------------------------------------------------------------------------------------------------
    def do_regex_search_populate_standarditemmodel_items(self):

        self.block_auto_jump_signals = True

        self.create_model_for_search_results_combobox()

        regex = self.search_regex_qlineedit.text()

        if regex == REGEX_DEFAULT:
            self.default_regex = True
            regex = ".+$"   # "^.+[.]*"
        else:
            self.default_regex = False

        try:
            p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
        except Exception as e:
            self.block_auto_jump_signals = False
            msg = "Regular Expression failed to compile: ", regex, "  reason: ", str(e)
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        msg = None
        if self.last_current_column is None:
            msg = "Please select a #Custom Column before searching."
        if not self.last_current_column.startswith("#"):
            msg = "Please select a #Custom Column before searching."
        if not msg is None:
            self.block_auto_jump_signals = False
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        self.matching_book_title_rows_combobox.clear()  #will cause an event to jump...
        self.block_auto_jump_signals = True
        self.matching_book_title_rows_combobox.update()

        book_value_dict = {}

        dbcache = self.maingui.current_db.new_api
        all_book_ids_set = dbcache.all_book_ids()
        name = self.last_current_column
        if DEBUG: print("self.last_current_column: ", self.last_current_column, " number of books: ", str(len(all_book_ids_set)))
        for book in all_book_ids_set:
            value = dbcache.field_for(name, book_id=book, default_value=None)
            if isinstance(value,str):
                if self.default_regex:
                    value = value.strip()
                    value = value[0:100]
                book_value_dict[book] = value
                #~ if DEBUG: print(str(book),value.strip)
        #END FOR

        if len(book_value_dict) == 0:
            msg = "No books were found that have any values in the selected Custom Column: " +  self.last_current_column
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        partial_results_list = []
        book_id_list = []
        value = None
        for book,value in book_value_dict.items():
            match = p.search(value)
            if match is not None:
                snippet = match.group()
                #~ if DEBUG: print(str(snippet))
                if snippet is not None:
                    snippet = str(snippet)
                    snippet = snippet.strip()
                    r = book,snippet
                    partial_results_list.append(r)
                    book_id_list.append(book)
                    #~ if DEBUG: print("regex match: ", regex, "  ", str(book), "snippet: ", str(snippet))  # regex match:  fandom|dragon    963 snippet:  Dragon
                del snippet
            del match
        #END FOR
        del value

        if len(partial_results_list) == 0:
            self.block_auto_jump_signals = False
            msg = "No books were found that have any values matching your Search Regular Expression: " +  regex
            return info_dialog(self.maingui, _(tool_name),_(msg), show=True)

        author_sort_name_dict = {}
        name = 'authors'
        authorid_data_map_dict = dbcache.author_data()  # [author_id] = name, sort, link
        for authorid,data_dict in authorid_data_map_dict.items():
            #~ if DEBUG: print("authorid,data: ", str(authorid), str(data_dict))  #~ authorid,data:  76 {'name': 'Singh R', 'sort': 'R, Singh', 'link': ''}
            if 'sort' in data_dict:
                if 'name' in data_dict:
                    sort = data_dict['sort']
                    name = data_dict['name']
                    author_sort_name_dict[sort] = name
        #END FOR

        book_authorid_map_dict = dbcache.author_sort_strings_for_books(book_id_list)  # val_map[book_id] = tuple(adata[aid]['sort'] for aid in authors)

        book_author_dict = {}  #  [book] = name

        for book,data_list in book_authorid_map_dict.items():
            #~ if DEBUG: print("data_list: ", type(data_list))  #~ data_list:  <class 'tuple'>
            s = ""
            for sort in data_list:
                #~ if DEBUG: print(type(sort), str(sort))
                if sort in author_sort_name_dict:
                    name = author_sort_name_dict[sort]
                    s = s + name + " & "
            #END FOR
            name = s[0:-2]
            name = name.strip()  # author name
            #~ if DEBUG: print("author name: ", name)   #  author name:  Singh R & Kaushik S & Wang Y & Xiang Y & Novak I & Komatsu M & Tanaka K & Cuervo AM & Czaja MJ
            book_author_dict[book] = name
            del s
            del data_list
            del name
        #END FOR

        if len(book_author_dict) == 0:
            self.block_auto_jump_signals = False
            msg = "No books were found that have any Authors for your books: " +  regex
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        results_list = []

        for r in partial_results_list:
            book,snippet = r
            if book in book_author_dict:
                name = book_author_dict[book]
                if name:
                    row = book,name,snippet
                    results_list.append(row)
        #END FOR
        del partial_results_list
        del book_author_dict
        del book_authorid_map_dict
        del author_sort_name_dict

        try:
            book_id_list.sort()
            length_shortest_bookid = len(str(book_id_list[0]))
            n_highest_index = len(book_id_list) - 1
            length_longest_bookid = len(str(book_id_list[n_highest_index]))
            if length_longest_bookid > length_shortest_bookid:
                n_padding = length_longest_bookid - length_shortest_bookid
            else:
                n_padding = 0
        except Exception as e:
            n_padding = 0
            if DEBUG: print("padding bookid length error: ", str(e))

        padding_str = "0000000"
        padding_str = padding_str[0:n_padding]

        if DEBUG: print("padding_str has a length of: ", str(len(padding_str)))

        row = 0
        for r in results_list:
            book,name,snippet = r
            title = dbcache._field_for('title', book_id=book, default_value= 'Unknown')
            if title:
                title = title[0:100]
                title = title.strip()
                title = QStandardItem(title)
                name = name[0:100]  # author name
                name = name.strip()
                name = QStandardItem(name)

                book_str = str(book)
                book_str = "000000000" + book_str
                book_str = book_str[-10:]  # use the right 10 digits
                book = QStandardItem(book_str)

                if snippet.startswith("<"):
                    snippet = html2text(snippet, single_line_break=True)
                elif snippet.startswith("# "):   #https://www.markdownguide.org/cheat-sheet
                    snippet = snippet[2: ]
                elif snippet.startswith("## "):
                    snippet = snippet[3: ]
                elif snippet.startswith("### "):
                    snippet = snippet[4: ]
                elif snippet.startswith("#### "):
                    snippet = snippet[5: ]
                elif snippet.startswith("*"):
                    snippet = snippet[1: ]
                elif snippet.startswith("**"):
                    snippet = snippet[2: ]
                elif snippet.startswith(">"):
                    snippet = snippet[1: ]
                elif snippet.startswith("—"):
                    snippet = snippet[1: ]
                elif snippet.startswith("--"):
                    snippet = snippet[2: ]
                elif snippet.startswith("-"):
                    snippet = snippet[1: ]
                snippet = snippet.strip()
                if snippet.startswith("#"): #random artifact from html2text best-efforts stripping md to text
                    snippet = snippet[1: ]
                snippet = snippet[0:100]
                snippet = snippet.strip()
                snippet = QStandardItem(snippet)

                if self.current_sort_value_source is None:
                    self.current_sort_value_source = SORT_BY_BOOKID

                if self.current_sort_value_source == SORT_BY_TITLE:
                    sort = QStandardItem(title)
                elif self.current_sort_value_source == SORT_BY_AUTHOR:
                    sort = QStandardItem(name)
                elif self.current_sort_value_source == SORT_BY_BOOKID:
                    sort = QStandardItem(book_str)
                elif self.current_sort_value_source == SORT_BY_SNIPPET:
                    sort = QStandardItem(snippet)
                else:
                    continue

                self.standarditemmodel.setItem(row,0,sort)
                self.standarditemmodel.setItem(row,1,title)
                self.standarditemmodel.setItem(row,2,name)
                self.standarditemmodel.setItem(row,3,book)
                self.standarditemmodel.setItem(row,4,snippet)

                row = row + 1

        #END FOR

        self.standarditemmodel_initialized = True

        self.do_sort_on_search_results_sort_column(None,need_copy=False)    #defaults, whatever their source, will be used for this very first sort

        self.block_auto_jump_signals = True
        self.matching_book_title_rows_combobox.setCurrentIndex(0)
        self.block_auto_jump_signals = True
        self.matching_book_title_rows_combobox.update()
        QApplication.instance().processEvents()

        if DEBUG: print("number of items in combobox: ", str(self.matching_book_title_rows_combobox.count()))

        self.block_auto_jump_signals = False

        self.search_current_item_setFocus()

        if self.matching_book_title_rows_combobox.count() == 0:
            msg = "No matching book/author/title/snippets were found"
            return info_dialog(self.maingui, _(tool_name),_(msg), show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def edit_search_regular_expression(self):
        qid = QInputDialog()
        title = "Regular Expression for 'Search'"
        qid.setWindowTitle(title)
        s = "Specify the Regular Expression that should be used when the 'Search' pushbutton is clicked or activated."
        qid.setLabelText(s)
        regex = self.search_regex_qlineedit.text()
        qid.setTextValue(regex)
        qid.setOption(QInputDialog.InputDialogOption.UsePlainTextEditForTextInput)
        qid.setInputMethodHints(Qt.ImhMultiLine|Qt.ImhLatinOnly)
        qid.setOkButtonText("&Save Regular Expression")
        qid.setFixedSize(400, 100)
        qid.setToolTip("<p style='white-space:wrap'>Specify the Regular Expression that should be used when the 'Search' pushbutton is clicked or activated.")
        qid.show()
        if qid.exec_() == qid.Accepted:
            text = qid.textValue() # After clicking OK, get the input dialog content
            if isinstance(text,str):
                text = text.strip()
                self.search_regex_qlineedit.setText(text)
                self.search_regex_qlineedit.update()
        del qid
    #---------------------------------------------------------------------------------------------------------------------------------------
    def show_matching_book_combobox_popup(self):
        self.matching_book_title_rows_combobox.showPopup()
        if self.instance_number > 1:
            self.child_about_to_move()
        else:
            self.instance_parent_about_to_move()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def jump_to_matching_book(self):

        if self.block_jumping:
            return

        if not self.standarditemmodel_initialized:
            return

        if self.instance_number > 1: #child
            self.child_about_to_move()
        else:
            self.instance_parent_about_to_move()

        self.freeze_notes_viewer_checkbox.setChecked(False)

        current = self.last_bookid

        i = self.matching_book_title_rows_combobox.currentIndex()
        if DEBUG: print("combobox index: ", str(i))
        item = self.standarditemmodel.item(i,3)  #bookid is always 3
        if item is None:
            pass
        else:
            book = item.text().strip()
            if DEBUG: print("book: ", book)
            if book.isdigit():
                book = int(book)
                try:
                    self.move_library_view_index(increment=0,bookid=book)
                    self.goback_bookid = current
                    if DEBUG: print("jumped to book: ", str(book))
                except Exception as e:
                    msg = "Book does not currently exist.  Either you recently deleted it, or you are using a Virtual Library that does not contain it, or you did Library search, or you clicked a Virtual Library tab."
                    if DEBUG: print(msg)
                    return error_dialog(self.maingui, _(tool_name),_(msg), show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def set_text_background_color(self):
        if not self.is_dark_mode:
            self.plain_qtextedit.setTextBackgroundColor(color_white)
            self.markdown_qtextedit.setTextBackgroundColor(color_white)
            self.html_qtextbrowser.setTextBackgroundColor(color_white)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_custom_columns_listing_dialog_geometry(self):
        self.dialog_closing(None)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def convert_comments_to_html(self):
        if not self.is_edit_mode:
            return
        if self.markdown_being_edited:
            self.convert_markdown_comments_to_html()
        elif self.is_html_being_edited:
            return
        else:
            self.convert_plain_comments_to_html()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def convert_plain_comments_to_html(self):
        new_html = comments_to_html(self.long_text)
        if new_html.startswith('<p class="description">'):
            if new_html.endswith("</p>"):
                new_html = "<div>" + new_html + "</div>"
        self.long_text = self.format_plain_text(new_html)
        self.original_long_text = self.long_text
        self.plain_qtextedit.setPlainText(self.long_text)
        self.convert_format_to_html_pushbutton.hide()
        self.cancel_edits_requires_restart = True
    #---------------------------------------------------------------------------------------------------------------------------------------
    def convert_markdown_comments_to_html(self):
        if not self.markdown_module_was_imported:
            from .markdown import markdown  # https://python-markdown.github.io/reference/
            from markdown import Markdown
            self.md_module = Markdown()
            self.markdown_module_was_imported = True
            self.md_module.reset()
        else:
            try:
                self.md_module.reset()
            except:
                if DEBUG: print("self.md_module does not exist, but should...")
                self.markdown_module_was_imported = False  #user can try again...
                return

        new_html = self.md_module.convert(self.long_text)
        self.long_text = self.format_plain_text(new_html)
        self.original_long_text = self.long_text
        self.plain_qtextedit.setPlainText(self.long_text)
        self.convert_format_to_html_pushbutton.hide()
        self.cancel_edits_requires_restart = True
    #---------------------------------------------------------------------------------------------------------------------------------------
    def html_qtextbrowser_setFocus(self):
        self.html_qtextbrowser.setFocus()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def search_current_item_setFocus(self):
        self.matching_book_title_rows_combobox.setFocus()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def return_help(self):
        help_html = '''
        <p>In View Mode, keys are limited to navigation, and text may only be selected with the mouse:</p>
        <div class="table"><table class="generic">
        <thead><tr class="qt-style"><th>Keypresses</th><th>Action</th></tr></thead>
        <tr class="odd" valign="top"><td>Up</td><td>Moves one line up.</td></tr>
        <tr class="even" valign="top"><td>Down</td><td>Moves one line down.</td></tr>
        <tr class="odd" valign="top"><td>Left</td><td>Moves one character to the left.</td></tr>
        <tr class="even" valign="top"><td>Right</td><td>Moves one character to the right.</td></tr>
        <tr class="odd" valign="top"><td>PageUp</td><td>Moves one page up.</td></tr>
        <tr class="even" valign="top"><td>PageDown</td><td>Moves one page down.</td></tr>
        <tr class="odd" valign="top"><td>Home</td><td>Moves to the beginning of the text.</td></tr>
        <tr class="even" valign="top"><td>End</td><td>Moves to the end of the text.</td></tr>
        <tr class="odd" valign="top"><td>Alt+Wheel</td><td>Scrolls the page horizontally (the Wheel is the mouse wheel).</td></tr>
        <tr class="even" valign="top"><td>Ctrl+Wheel</td><td>Zooms the text.</td></tr>
        <tr class="odd" valign="top"><td>Ctrl+A</td><td>Selects all text.</td></tr>
        </table></div>
        '''
        return help_html
    #---------------------------------------------------------------------------------------------------------------------------------------
    def set_user_tweaks(self):
        self.js_tweaks_dict = None
        if not 'job_spy_notes_viewer_protocol_default_apps' in tweaks:
            return
        jsdict = tweaks['job_spy_notes_viewer_protocol_default_apps']
        if not isinstance(jsdict,dict): #currently returns a dict
            if DEBUG: print("jsdict was not a dict")
            jsdict,isvalid = self.convert_string_to_dict(jsdict)
            if not isvalid:
                del jsdict
                self.js_tweaks_dict = None
                msg = "Error in the Tweak 'job_spy_notes_viewer_protocol_default_apps' in Your Preferences > Tweaks > Plugin Tweaks value is invalid. <br><br>You should recustomize Job Spy Tweaks and Preferences for this GUI Tool before proceeding."
                return info_dialog(self.gui, 'JS+ GUI Tool: Tweak job_spy_notes_viewer_protocol_default_apps',msg).show()

        self.js_tweaks_dict = jsdict
    #---------------------------------------------------------------------------------------------------------------------------------------
    def convert_string_to_dict(self,sdict):
        import ast
        sdict = as_unicode(sdict)
        try:
            d = ast.literal_eval(sdict)
            del ast
            del sdict
            if not isinstance(d,dict):
                if DEBUG: print("[1] tweaks['job_spy_notes_viewer_protocol_default_apps'] is not formatted properly as a valid dictionary.")
                return d,False
            else:
               return d,True
        except:
            if DEBUG: print("[2] tweaks['job_spy_notes_viewer_protocol_default_apps'] is not formatted properly as a valid dictionary.")
            return d,False
    #---------------------------------------------------------------------------------------------------------------------------------------
    def customize_notes_viewer(self):
        try:
            self.notesviewer_customization_dialog.close()
        except:
            pass

        self.nv_state_dict = {}
        self.nv_state_dict['comments_datatype_list'] = self.comments_datatype_list

        from calibre_plugins.job_spy.notes_viewer_customization_dialog import NotesViewerCustomizationDialog

        self.notesviewer_customization_dialog = NotesViewerCustomizationDialog(self.maingui,self.nv_state_dict,self.js_icon,self.nv_prefs,self.convert_string_to_dict,self.return_from_notes_viewer_customization_to_save_prefs)
        self.notesviewer_customization_dialog.setAttribute(Qt.WA_DeleteOnClose)
        self.notesviewer_customization_dialog.show()

        del NotesViewerCustomizationDialog

        self.close()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def get_saved_preferences(self):

        #~ -------------------------------------------------------------
        #~ default the defaults and their related
        #~ -------------------------------------------------------------
        self.current_sort_value_source = SORT_BY_BOOKID
        self.sort_search_results_by_column = SORT_BY_BOOKID
        self.sort_search_results_by_title_ascending = True
        self.sort_search_results_by_author_ascending = True
        self.sort_search_results_by_bookid_ascending = True
        self.sort_search_results_by_snippet_ascending = True
        self.sort_search_results_by_title_last_state = -1
        self.sort_search_results_by_author_last_state = -1
        self.sort_search_results_by_bookid_last_state = -1
        self.sort_search_results_by_snippet_last_state = -1
        self.max_instances_allowed =  "1"
        #~ -------------------------------------------------------------

        library_options_dict = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_CUSTOMIZATION_BY_LIBRARY']

        library_options_dict,is_valid = self.convert_string_to_dict(library_options_dict)

        if not is_valid or not(isinstance(library_options_dict,dict)):
            msg = "Preferences dictionary loaded, but it is invalid.  Cannot use it."
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        self.library_options_dict = library_options_dict

        self.library_id = self.maingui.current_db.new_api.backend.library_id
        if DEBUG: print("library_id: ", self.library_id)
        k  = 'GUI_TOOLS_NOTES_VIEWER_LAST_LIBRARY_USED'
        v = self.library_id
        self.prefs_changes_dict[k] = v

        if self.library_id in self.library_options_dict:
            if DEBUG: print("self.library_id is in self.library_options_dict...")
            library_val = self.library_options_dict[self.library_id]
            if len(library_val) == 7:
                self.cc_settings_dict, self.default_column, self.lock_state, self.autojump_state, self.sort_column, self.sort_descending, self.max_instances_allowed = library_val
                self.max_instances_allowed = int(self.max_instances_allowed)
                if DEBUG: print("Current Version of library_val was found...")  # , str(library_val))
                if DEBUG: print("lock_state,autojump_state,sort_column, sort_descending as saved previously: ", self.lock_state, self.autojump_state)
            elif len(library_val) == 6:
                self.cc_settings_dict, self.default_column, self.lock_state, self.autojump_state, self.sort_column, self.sort_descending = library_val
                self.max_instances_allowed =  1
                if DEBUG: print("Current Version of library_val found...")  # , str(library_val))
                if DEBUG: print("lock_state,autojump_state,sort_column, sort_descending as saved previously: ", self.lock_state, self.autojump_state)
            elif len(library_val) == 4:
                self.cc_settings_dict, self.default_column, self.lock_state, self.autojump_state = library_val
                self.sort_column = "bookid"
                self.sort_descending = False
                self.max_instances_allowed =  1
                if DEBUG: print("Prior Version of library_val found...")  # , str(library_val))
                if DEBUG: print("lock_state and autojump_state as saved previously: ", self.lock_state, self.autojump_state)
                msg = "Library-specific defaults for 'Default Sort Search Results By Column?' and 'Sort Descending?' have not been customized yet via the 'Options' button.\
                        <br><br>Please customize this Library's 'Options' to set your permanent defaults for this Library."
                info_dialog(self.maingui, _(tool_name),_(msg), show=True)
            else:
                self.cc_settings_dict, self.default_column = library_val
                self.lock_state = False
                self.autojump_state = False
                self.sort_column = "bookid"
                self.sort_descending = False
                self.max_instances_allowed =  1
                if DEBUG: print("Legacy Version of library_val found...")   # , str(library_val))
                if DEBUG: print("No lock_state and autojump_state were saved previously; defaulted to False: ", self.lock_state, self.autojump_state)
                msg = "Library-specific defaults for 'Lock' and 'AutoJump' have not been customized yet via the 'Options' button.\
                        <br><br>Please customize this Library's 'Options' to set your permanent defaults for this Library."
                info_dialog(self.maingui, _(tool_name),_(msg), show=True)
        else:
            if DEBUG: print("self.library_id IS NOT in self.library_options_dict...")
            self.cc_settings_dict = {}
            self.default_column = None
            self.lock_state = False
            self.autojump_state = False
            self.sort_column = "bookid"
            self.sort_descending = False
            self.max_instances_allowed =  1
            if DEBUG: print("No Version whatsoever of library_val found...Library has never been Customized.")
            if DEBUG: print("\n\nNo library_val found whatsoever...everything defaulted to None or False.\n\n")

        if self.default_column is not None:
            if self.default_column in self.comments_datatype_list:
                pass
            else:
                self.default_column = None

        if self.sort_column == "title":
            self.sort_search_results_by_column = SORT_BY_TITLE
            if self.sort_descending:
                self.sort_search_results_by_title_ascending = False
            else:
                self.sort_search_results_by_title_ascending = True
        elif self.sort_column == "author":
            self.sort_search_results_by_column = SORT_BY_AUTHOR
            if self.sort_descending:
                self.sort_search_results_by_author_ascending = False
            else:
                self.sort_search_results_by_author_ascending = True
        elif self.sort_column == "bookid":
            self.sort_search_results_by_column = SORT_BY_BOOKID
            if self.sort_descending:
                self.sort_search_results_by_bookid_ascending = False
            else:
                self.sort_search_results_by_bookid_ascending = True
        elif self.sort_column == "snippet":
            self.sort_search_results_by_column = SORT_BY_SNIPPET
            if self.sort_descending:
                self.sort_search_results_by_snippet_ascending = False
            else:
                self.sort_search_results_by_snippet_ascending = True

        self.current_sort_value_source = self.sort_search_results_by_column
    #---------------------------------------------------------------------------------------------------------------------------------------
    def determine_custom_column_customization_states(self,cc,reason=None):
        if reason is None:
            return None

        if cc in self.cc_settings_dict:
            settings_dict = self.cc_settings_dict[cc]
            if reason == "KEEP":
                qcb_state = settings_dict['OPTION_QCHECKBOX']
                qcb_state = bool(qcb_state)   #  True means "Hide"; False means "Keep".
                return qcb_state

            if reason == "GUESS":
                qpr_state = settings_dict['OPTION_PLAIN_RADIO']
                qmr_state = settings_dict['OPTION_MD_RADIO']
                qhr_state = settings_dict['OPTION_HTML_RADIO']
                qgr_state = settings_dict['OPTION_GUESS_RADIO']

                qpr_state = bool(qpr_state)
                qmr_state = bool(qmr_state)
                qhr_state = bool(qhr_state)
                qgr_state = bool(qgr_state)

                if qpr_state is True:
                    return "plain"
                elif qmr_state is True:
                    return "markdown"
                elif qhr_state is True:
                    return "html"
                elif qgr_state is True:
                    return "guess"
                else:
                    return None
        else:
            return None
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_model_for_search_results_combobox(self):

        self.standarditemmodel = None
        del self.standarditemmodel
        self.matching_book_title_rows_combobox.clear()

        self.standarditemmodel = QStandardItemModel(self)
        self.standarditemmodel.setColumnCount(5)  # sort,title,author,bookid,snippet
        self.standarditemmodel.setRowCount(self.n_rows)

        self.hdr_label_list = \
        [SORT_VALUE_HEADING,\
        "Title ",\
        "Author ",\
        "BookID ",\
        "Snippet "]

        self.standarditemmodel.setHorizontalHeaderLabels(self.hdr_label_list)

        self.matching_book_title_rows_combobox.setModel(self.standarditemmodel)

        self.combotableview = QTableView(self.matching_book_title_rows_combobox, selectionBehavior=QAbstractItemView.SelectRows)
        self.combotableview.setSortingEnabled(True)
        self.combotableview.setSelectionMode(QAbstractItemView.SingleSelection)

        self.matching_book_title_rows_combobox.setView(self.combotableview)

        self.combotableview.verticalHeader().show()
        self.combotableview.horizontalHeader().show()

        self.combotableview.horizontalHeader().setMinimumSectionSize(50)  # Smallest is BookID ---> ResizeMode.ResizeToContents

        for i in range(self.combotableview.horizontalHeader().count()):
            self.combotableview.horizontalHeader().setSectionResizeMode(i, QHeaderView.ResizeMode.ResizeToContents)
        #END FOR

        #~ self.combotableview.horizontalHeader().setStretchLastSection(True)

        self.combotableview.horizontalHeader().setDefaultAlignment(Qt.AlignmentFlag.AlignLeft)

        self.combotableview.horizontalHeader().setContextMenuPolicy(Qt.ActionsContextMenu)
        self.combotableview.horizontalHeader().addAction(self.qaction_search_results)

        self.combotableview.verticalHeader().setContextMenuPolicy(Qt.ActionsContextMenu)
        self.combotableview.verticalHeader().addAction(self.qaction_search_results)

        self.combotableview.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.combotableview.customContextMenuRequested.connect(self.combotableview_customContextMenuRequested)  #see below.

        self.model_created = True
    #---------------------------------------------------------------------------------------------------------------------------------------
    def combotableview_customContextMenuRequested(self,point):
        #~ This compensates for the default behavior of the QTableView to treat a right-button mouse click the same as a left-button mouse click.
        #~ When the right button is released, this function is executed, but the QTableView popup is closed anyway.  That is the issue, and why this exists.
        #~ This function then re-opens the combobox list popup(), and then pops up the context menu at the same point as when the right button was released.
        #~ In summary, the focus of the mouse cursor in the QTableView is returned to exactly the point it was before it was lost due to the right button release.
        #~ Plus, the point of this exercise is that the Context Menu pops open also, but long after the right mouse button was released, closing the QTableView list.
        try:
            if DEBUG: print("combotableview_customContextMenuRequested:  Point Object=", str(point))
            self.show_matching_book_combobox_popup()
            self.menu.popup(self.combotableview.mapToGlobal(point))
        except Exception as e:
            if DEBUG: print("combotableview_customContextMenuRequested:  Exception: ", str(e))
    #---------------------------------------------------------------------------------------------------------------------------------------
    def build_menu_for_main_window(self,font):

        self.menu_main = QMenu(self)
        self.menu_main.clear()
        self.menu_main.setFont(font)
        self.menu_main.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.menu_main.setTearOffEnabled(True)
        self.menu_main.setSeparatorsCollapsible(False)

        self.menu_main.addSeparator()

        self.toggle_mark_for_current_book_action = QAction("Toggle 'Mark' for Current Book",None)
        self.toggle_mark_for_current_book_action.setShortcut(QKeySequence("Ctrl+M"))
        self.toggle_mark_for_current_book_action.triggered.connect(self.toggle_current_book)
        self.menu_main.addAction(self.toggle_mark_for_current_book_action)
        self.menu_main.addSeparator()
        self.menu_main.addSeparator()
        self.go_to_current_book_action = QAction("Go/Move cursor to the BookID",None)
        self.go_to_current_book_action.setShortcut(QKeySequence("Ctrl+G"))
        self.go_to_current_book_action.triggered.connect(self.go_to_current_book)
        self.menu_main.addAction(self.go_to_current_book_action)
        self.menu_main.addSeparator()
        self.menu_main.addSeparator()
        self.freeze_all_instances_action = QAction("Freeze All Instances",None)
        self.freeze_all_instances_action.setShortcut(QKeySequence("Ctrl+F"))
        self.freeze_all_instances_action.triggered.connect(self.freeze_all_instances)
        self.menu_main.addAction(self.freeze_all_instances_action)
        self.menu_main.addSeparator()
        self.menu_main.addSeparator()
        if self.instance_number == 1:
            self.create_child_instance_action = QAction("Create NV Child Instance",None)
            self.create_child_instance_action.setShortcut(QKeySequence("Ctrl+Z"))
            self.create_child_instance_action.triggered.connect(self.create_nv_child_instance)
            self.menu_main.addAction(self.create_child_instance_action)
            self.menu_main.addSeparator()
            self.menu_main.addSeparator()
        else:
            self.menu_main.addSeparator()
            self.menu_main.addSeparator()
        #END IF
        self.focus_keyboard_on_textbrowser_action = QAction("Set Keyboard Focus on the HTML Text Box [Only for HTML]",None)
        self.focus_keyboard_on_textbrowser_action.setShortcut(QKeySequence("Ctrl+N"))
        self.focus_keyboard_on_textbrowser_action.triggered.connect(self.html_qtextbrowser_setFocus)
        self.menu_main.addAction(self.focus_keyboard_on_textbrowser_action)
        self.menu_main.addSeparator()
        self.focus_keyboard_on_srch_current_item = QAction("Set Keyboard Focus on the 'Srch Current Item'",None)
        self.focus_keyboard_on_srch_current_item.setShortcut(QKeySequence("Ctrl+I"))
        self.focus_keyboard_on_srch_current_item.triggered.connect(self.search_current_item_setFocus)
        self.menu_main.addAction(self.focus_keyboard_on_srch_current_item)
        self.menu_main.addSeparator()

        self.qaction_menu_main = QAction('Notes Viewer: Miscellany', None)
        self.qaction_menu_main.setMenu(self.menu_main)

        self.scroll_widget.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.scroll_widget.customContextMenuRequested.connect(self.show_notes_viewer_context_menu)

        self.scroll_widget.addAction(self.focus_keyboard_on_textbrowser_action)
        self.scroll_widget.addAction(self.focus_keyboard_on_srch_current_item)
        self.scroll_widget.addAction(self.toggle_mark_for_current_book_action)
        self.scroll_widget.addAction(self.go_to_current_book_action)
        self.scroll_widget.addAction(self.freeze_all_instances_action)
        if self.instance_number == 1:
            self.scroll_widget.addAction(self.create_child_instance_action)
        #END IF
    #---------------------------------------------------------------------------------------------------------------------------------------
    def show_notes_viewer_context_menu(self, point):
        self.menu_main.popup(self.scroll_widget.mapToGlobal(point))
   #---------------------------------------------------------------------------------------------------------------------------------------
    def build_menu_for_search_results(self,font):

        self.menu_search_results = QMenu(self)
        self.menu_search_results.clear()
        self.menu_search_results.setFont(font)
        self.menu_search_results.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.menu_search_results.setTearOffEnabled(True)
        self.menu_search_results.setSeparatorsCollapsible(False)

        self.menu_search_results.addSeparator()

        #~ the actual column data sorted is always in column 0, the data for which was copied from the requested sort column prior to sorting.
        self.sort_search_results_by_title_action = QAction("Sort Search Results by Title",None)
        self.sort_search_results_by_title_action.setShortcut(QKeySequence("Ctrl+T"))
        self.sort_search_results_by_title_action.triggered.connect(lambda:self.do_sort_on_search_results_sort_column(1))
        self.menu_search_results.addAction(self.sort_search_results_by_title_action)
        self.sort_search_results_by_title_last_state = -1
        self.menu_search_results.addSeparator()
        self.sort_search_results_by_author_action = QAction("Sort Search Results by Author",None)
        self.sort_search_results_by_author_action.setShortcut(QKeySequence("Ctrl+A"))
        self.sort_search_results_by_author_action.triggered.connect(lambda:self.do_sort_on_search_results_sort_column(2))
        self.menu_search_results.addAction(self.sort_search_results_by_author_action)
        self.sort_search_results_by_author_last_state = -1
        self.menu_search_results.addSeparator()
        self.sort_search_results_by_book_action = QAction("Sort Search Results by BookID",None)
        self.sort_search_results_by_book_action.setShortcut(QKeySequence("Ctrl+B"))
        self.sort_search_results_by_book_action.triggered.connect(lambda:self.do_sort_on_search_results_sort_column(3))
        self.menu_search_results.addAction(self.sort_search_results_by_book_action)
        self.sort_search_results_by_book_last_state = -1
        self.menu_search_results.addSeparator()
        self.sort_search_results_by_snippet_action = QAction("Sort Search Results by Snippet",None)
        self.sort_search_results_by_snippet_action.setShortcut(QKeySequence("Ctrl+S"))
        self.sort_search_results_by_snippet_action.triggered.connect(lambda:self.do_sort_on_search_results_sort_column(4))
        self.menu_search_results.addAction(self.sort_search_results_by_snippet_action)
        self.sort_search_results_by_snippet_last_state = -1
        self.menu_search_results.addSeparator()
        self.menu_search_results.addSeparator()

        self.qaction_search_results = QAction('Search Results Menu:', None)
        self.qaction_search_results.setMenu(self.menu_search_results)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def toggle_current_book(self):
        if DEBUG: print("Toggle 'Mark' for Current Book (Ctrl+M): ", str(self.last_bookid))
        self.maingui.library_view.model().db.data.toggle_marked_ids({self.last_bookid,})
    #---------------------------------------------------------------------------------------------------------------------------------------
    def do_sort_on_search_results_sort_column(self,col=None,need_copy=True):

        if not self.standarditemmodel_initialized:
            #~ Important:  Ctrl+A shortcut is used in 2 places, on different widgets.  Both work fine separately, but collide here as described below.
            #~ Potential Error:  Shortcut of Ctrl+A activated on the main window of NV, and there are no Search Results (which uses Crtl+A for sorting the standarditems by Snippet)
            #~ If QTextEdit is in View Mode (with no focus possible), text may be selected ONLY with the mouse.  Either "Select All", or dragging the cursor around the text.
            #~                                                                                 The View Mode text may NOT be selected via Ctrl+A as it would be for Edit Mode.  Ctrl+A activates a standarditems search.
            #~ If Shortcut of Ctrl+A is activated while QTextEdit is in Edit Mode, and has focus, the shortcut does a normal "Select All Text" action.  The standarditems are irrelevant.
            if DEBUG: print("[sort_search_results_by_column]:  Shortcut of Ctrl+A will be ignored; QTextEdit must now be in View Mode, and no standard items in Search Results exist now.")
            return

        if DEBUG: print("do_sort_on_search_results_sort_column")

        #~ -------------------------------------------------------------------------------------------------------------------------------------------------------
        #~ Explanation of how Search Results sorting works:
        #~ -------------------------------------------------------------------------------------------------------------------------------------------------------
        #~ col 0:  sort column; copy of data in 'real' column (1 thru 4) that is then sorted in col 0 only; col 0 value always appears in qcombobox as the current value.
        #~ col 1:  title                       never sorted in situ
        #~ col 2:  author                   never sorted in situ
        #~ col 3:  bookid                   never sorted in situ         initial default is always column 3, bookid. customization may change this default.
        #~ col 4:  snippet                  never sorted in situ
        #~ self.current_sort_value_source = SORT_BY_BOOKID (initial default), but changes thereafter with each request to sort via a shortcut or menu selection.
        #~ -------------------------------------------------------------------------------------------------------------------------------------------------------
        #~ customization may specify the default column to sort:
        #~      self.sort_search_results_by_column = SORT_BY_BOOKID (initial default),  or whatever was customized for the current Library
        #~ customization may also specify whether the default column to sort will be sorted ascending or descending.
        #~      self.sort_search_results_by_title_ascending = True or False
        #~      self.sort_search_results_by_author_ascending = True or False
        #~      self.sort_search_results_by_bookid_ascending = True or False
        #~      self.sort_search_results_by_snippet_ascending = True or False
        #~ -------------------------------------------------------------------------------------------------------------------------------------------------------
        if col is None:
            self.current_sort_value_source = self.sort_search_results_by_column  #self.sort_search_results_by_column = SORT_BY_BOOKID
        else:
            self.current_sort_value_source = col                                                    #self.sort_search_results_by_column = col
        #~ ---------------------------------------------------------------------------
        #~ ---------------------------------------------------------------------------
        if self.current_sort_value_source == SORT_BY_TITLE:
            if need_copy:
                self.copy_search_results_to_col_0_from_real_column(1)  # col 0:  sort column; copy of 'real' column (1 thru 4) that is then sorted in col 0 only
            if self.sort_search_results_by_title_last_state == -1:  #initialization value when the context menu was first created with shortcuts
                if self.sort_search_results_by_title_ascending:       # customization may specify this; if not, it defaults to False when initialized.
                    self.sort_search_results_by_title_last_state = 1  #alternate sort direction with each call to sort, so "= 1" means "now use 0 this time"
                else:
                    self.sort_search_results_by_title_last_state = 0
        #~ ---------------------------------------------------------------------------
        elif self.current_sort_value_source == SORT_BY_AUTHOR:
            if need_copy:
                self.copy_search_results_to_col_0_from_real_column(2)
            if self.sort_search_results_by_author_last_state == -1:
                if self.sort_search_results_by_author_ascending:
                    self.sort_search_results_by_author_last_state = 1
                else:
                    self.sort_search_results_by_author_last_state = 0
        #~ ---------------------------------------------------------------------------
        elif self.current_sort_value_source == SORT_BY_BOOKID:
            if need_copy:
                self.copy_search_results_to_col_0_from_real_column(3)
            if self.sort_search_results_by_bookid_last_state == -1:
                if self.sort_search_results_by_bookid_ascending:
                    self.sort_search_results_by_bookid_last_state = 1
                else:
                    self.sort_search_results_by_bookid_last_state = 0
        #~ ---------------------------------------------------------------------------
        elif self.current_sort_value_source == SORT_BY_SNIPPET:
            if need_copy:
                self.copy_search_results_to_col_0_from_real_column(4)
            if self.sort_search_results_by_snippet_last_state == -1:
                if self.sort_search_results_by_snippet_ascending:
                    self.sort_search_results_by_snippet_last_state = 1
                else:
                    self.sort_search_results_by_snippet_last_state = 0
        #~ ---------------------------------------------------------------------------
        #~ ---------------------------------------------------------------------------
        if self.current_sort_value_source == 1:  # SORT_BY_TITLE
            self.hdr_label_list[0] = SORT_VALUE_HEADING + "Title"
            if self.sort_search_results_by_title_last_state == 1:
                self.combotableview.sortByColumn(0, Qt.AscendingOrder)
                self.sort_search_results_by_title_last_state = 0
            else:
                self.combotableview.sortByColumn(0, Qt.DescendingOrder)
                self.sort_search_results_by_title_last_state = 1
        #~ ---------------------------------------------------------------------------
        elif self.current_sort_value_source == 2:  # SORT_BY_AUTHOR
            self.hdr_label_list[0] = SORT_VALUE_HEADING + "Author"
            if self.sort_search_results_by_author_last_state == 1:
                self.combotableview.sortByColumn(0, Qt.AscendingOrder)
                self.sort_search_results_by_author_last_state = 0
            else:
                self.combotableview.sortByColumn(0, Qt.DescendingOrder)
                self.sort_search_results_by_author_last_state = 1
        #~ ---------------------------------------------------------------------------
        elif self.current_sort_value_source == 3:  # SORT_BY_BOOKID
            self.hdr_label_list[0] = SORT_VALUE_HEADING + "BookID"
            if self.sort_search_results_by_bookid_last_state == 1:
                self.combotableview.sortByColumn(0, Qt.AscendingOrder)
                self.sort_search_results_by_bookid_last_state = 0
            else:
                self.combotableview.sortByColumn(0, Qt.DescendingOrder)
                self.sort_search_results_by_bookid_last_state = 1
        #~ ---------------------------------------------------------------------------
        elif self.current_sort_value_source == 4:  # SORT_BY_SNIPPET
            self.hdr_label_list[0] = SORT_VALUE_HEADING + "Snippet"
            if self.sort_search_results_by_snippet_last_state == 1:
                self.combotableview.sortByColumn(0, Qt.AscendingOrder)
                self.sort_search_results_by_snippet_last_state = 0
            else:
                self.combotableview.sortByColumn(0, Qt.DescendingOrder)
                self.sort_search_results_by_snippet_last_state = 1

        self.standarditemmodel.setHorizontalHeaderLabels(self.hdr_label_list)
        self.combotableview.resizeColumnsToContents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def copy_search_results_to_col_0_from_real_column(self,source):

        if DEBUG:
            print("\ncopy_search_results_to_col_0_from_real_column:\nself.standarditemmodel.rowCount(): ", str(self.standarditemmodel.rowCount()))
            print("self.current_sort_value_source: ", str(source))

        row = 0
        while row < self.standarditemmodel.rowCount():
            item = self.standarditemmodel.item(row,source)
            if item is not None:
                #~ if DEBUG: print("source_value:          ", item.text())
                sort_item = item.clone()
                #~ if DEBUG: print("source_clone_value: ", sort_item.text())
                self.standarditemmodel.setItem(row,0,sort_item)
            del item
            row = row + 1
        #END WHILE
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_notes_editor(self):
        if self.full_note_html_editor is None:
            from calibre_plugins.job_spy.note_editor_dialog import Editor      # Calibre 4.0.0+ since QWebKit was deprecated...
            self.full_note_html_editor = Editor(parent=None, one_line_toolbar=True, toolbar_prefs_name=None)
            self.full_note_html_editor.html = self.original_long_text
            self.full_note_html_editor.setToolTip("<p style='white-space:wrap'>Current text of the current Note.")
            self.layout_note_editor.addWidget(self.full_note_html_editor)
            self.full_note_html_editor.show()
            del Editor
        else:
            self.full_note_html_editor.html = self.original_long_text
            self.full_note_html_editor.show()

        self.full_note_html_editor.setFocus(Qt.OtherFocusReason)

        self.cancel_edits_requires_restart = True

        self.plain_qtextedit.setReadOnly(True)
        self.markdown_qtextedit.setReadOnly(True)
        self.html_qtextbrowser.setReadOnly(True)

        self.plain_qtextedit.setEnabled(False)
        self.markdown_qtextedit.setEnabled(False)
        self.html_qtextbrowser.setEnabled(False)

        self.plain_qtextedit.hide()
        self.markdown_qtextedit.hide()
        self.html_qtextbrowser.hide()

        self.freeze_notes_viewer_checkbox.setChecked(True)
        self.freeze_notes_viewer_checkbox.hide()

        self.plain_radio.hide()
        self.md_radio.hide()
        self.html_radio.hide()

        self.view_radio.hide()

        for obj in self.search_row_objects_to_hide_during_html_edit_list:
            obj.hide()
        #END FOR
    #---------------------------------------------------------------------------------------------------------------------------------------
    def hide_notes_editor(self):
        if self.full_note_html_editor is not None:
            self.full_note_html_editor.hide()
        self.is_html_being_edited = False
        self.freeze_notes_viewer_checkbox.setChecked(False)
        self.freeze_notes_viewer_checkbox.show()
        self.plain_radio.show()
        self.md_radio.show()
        self.html_radio.show()
        self.view_radio.show()
        for obj in self.search_row_objects_to_hide_during_html_edit_list:
            obj.show()
        #END FOR
   #---------------------------------------------------------------------------------------------------------------------------------------
    def style_focus_on_widgets(self):

        if not self.is_dark_mode:
            ssht =   "QTextEdit:focus {border: 2px solid blue; } \
                        QTextBrowser:focus {border: 2px solid blue;}\
                        QComboBox:focus {border: 1px solid blue; background: white;}\
                        QLineEdit:focus {border: 1px solid blue; background: white;}\
                        QCheckBox:focus {border: 1px solid blue; background: white;}\
                        QPushButton:focus {border: 1px solid blue; background: white;}\
                        QRadioButton:focus {border: 1px solid blue; background: white;}\
                    "
        else:
            ssht =   "QTextEdit:focus {border: 2px solid lightBlue;} \
                        QTextBrowser:focus {border: 2px solid lightBlue;}\
                        QComboBox:focus {border: 1px solid lightBlue;}\
                        QLineEdit:focus {border: 1px solid lightBlue;}\
                        QCheckBox:focus {border: 1px solid lightBlue;}\
                        QPushButton:focus {border: 1px solid lightBlue;}\
                        QRadioButton:focus {border: 1px solid lightBlue;}\
                    "
        self.setStyleSheet(ssht)
   #---------------------------------------------------------------------------------------------------------------------------------------
    def create_url_decoding_list(self):
       #~ -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
        #~ newline ........................    space    	" 	    % 	    - 	      .       	< 	    > 	    \ 	        ^   	_ 	        `       	{       	|       	}        	~
        #~ %0A or %0D or %0D%0A 	%20 	%22 	%25 	%2D 	%2E 	%3C 	%3E 	%5C 	%5E 	%5F 	%60 	%7B 	%7C 	%7D 	%7E
        #~ -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
        #~ ! 	        #       	$ 	      &     	' 	    (            	)   	* 	    +           ,       	/ 	        :       	; 	    =       	? 	    @    	[            	]
        #~ %21 	%23 	%24 	%26 	%27 	%28 	%29 	%2A 	%2B 	%2C 	%2F 	%3A 	%3B 	%3D 	%3F 	%40 	%5B 	%5D
        #~ -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
        self.decode_url_encoded_list = []
        self.decode_url_encoded_list.append(('%20',' '))
        self.decode_url_encoded_list.append(('%21','!'))
        self.decode_url_encoded_list.append(('%22','"'))
        self.decode_url_encoded_list.append(('%23','#'))
        self.decode_url_encoded_list.append(('%24','$'))
        self.decode_url_encoded_list.append(('%25','%'))
        self.decode_url_encoded_list.append(('%26','&'))
        self.decode_url_encoded_list.append(('%27',"'"))
        self.decode_url_encoded_list.append(('%28','('))
        self.decode_url_encoded_list.append(('%29','))'))
        self.decode_url_encoded_list.append(('%2A','*'))
        self.decode_url_encoded_list.append(('%2B','+'))
        self.decode_url_encoded_list.append(('%2C',','))
        self.decode_url_encoded_list.append(('%2D','-'))
        self.decode_url_encoded_list.append(('%2E','.'))
        self.decode_url_encoded_list.append(('%2F','/'))
        self.decode_url_encoded_list.append(('%3A',':'))
        self.decode_url_encoded_list.append(('%3B',';'))
        self.decode_url_encoded_list.append(('%3C','<'))
        self.decode_url_encoded_list.append(('%3D','='))
        self.decode_url_encoded_list.append(('%3E','>'))
        self.decode_url_encoded_list.append(('%3F','?'))
        self.decode_url_encoded_list.append(('%40','@'))
        self.decode_url_encoded_list.append(('%5B','['))
        self.decode_url_encoded_list.append(('%5C','\\'))
        self.decode_url_encoded_list.append(('%5D',']'))
        self.decode_url_encoded_list.append(('%5E','^'))
        self.decode_url_encoded_list.append(('%5F','_'))
        self.decode_url_encoded_list.append(('%60','`'))
        self.decode_url_encoded_list.append(('%7B','{'))
        self.decode_url_encoded_list.append(('%7C','|'))
        self.decode_url_encoded_list.append(('%7D','}'))
        self.decode_url_encoded_list.append(('%7E','~'))
    #---------------------------------------------------------------------------------------------------------------------------------------
    def freeze_all_instances(self):
        if self.instance_number == 1:
            self.instance_parent_about_to_move()
            self.freeze_notes_viewer_checkbox.setChecked(True)
        elif self.instance_number == 2:
            self.child_about_to_move()
            self.freeze_notes_viewer_checkbox.setChecked(True)
        elif self.instance_number == 3:
            self.child_about_to_move()
            self.freeze_notes_viewer_checkbox.setChecked(True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_settings_and_exit(self,restart=False,source=None):
        #~ three sources:  'refresh_notes_viewer' and 'closeEvent' and None

        if DEBUG: print("save_settings_and_exit:  restart & source: ", restart, source)

        if source == "refresh_notes_viewer" or  source == "closeEvent" or source == "ExitButton":
            self.is_proper_closing_event = True
            if DEBUG: print("self.is_proper_closing_event: ", self.is_proper_closing_event)
        else:
            if DEBUG: print("Source is: ", source, " self.is_proper_closing_event: ", self.is_proper_closing_event)

        if self.instance_number > 1:
            if not self.is_proper_closing_event:
                self.instance_parent.update_push_button_create_child_instance_text()
                self.instance_parent.update()

        if self.edit_radio.isChecked():
            self.block_refresh_signals = True
            self.view_radio.setChecked(True)

        self.dialog_closing(None)

        try:
            self.maingui.library_view.selectionModel().currentChanged.disconnect(self.refresh_notes_viewer)
        except:
            pass

        if self.instance_number == 1:
            self.close_nv_child_instances()
            self.update_nv_last_used_prefs_by_instance()  #after all children have returned their generic (and specific) prefs and closed; generics in ui.py should be from #1
            if source == "refresh_notes_viewer":
                return self.return_from_notes_viewer_to_save_prefs(self.prefs_changes_dict,restart)
            elif source == "closeEvent":
                if not self.is_closing:
                    self.is_closing = True  # avoid recursive 'closeEvent' source
                return self.return_from_notes_viewer_to_save_prefs(self.prefs_changes_dict,restart)
            else:
                if not self.is_closing:
                    self.is_closing = True  # avoid recursive 'closeEvent' source
                return self.return_from_notes_viewer_to_save_prefs(self.prefs_changes_dict,restart)
        else: # child; source: ExitButton
            if DEBUG: print("self.save_settings_and_exit -- child:  self.return_from_child_instance_to_close(child): ", str(self.instance_number), str(self))
            if DEBUG: print("Source is: ", source)
            self.update_nv_last_used_prefs_by_instance()
            child = self
            restart_param = self.prefs_changes_dict  #regardless if restarting or not; need to save the current instance prefs for this child.
            return self.instance_parent.return_from_child_instance_to_close(child,self.instance_number,restart=restart,restart_param=restart_param)
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #~ Child Instances:
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_nv_child_instance(self,restart=False,restart_param=None):

        if self.instance_number > 1:    #   only instance #1, the parent, can create instances #2 and #3
            return
        elif self.max_instances_allowed == 1:
            msg = "Customization only allows one (1) instance of Notes Viewer.<br><br>Refer to 'Options'."
            return info_dialog(self.maingui, _(tool_name),_(msg), show=True)
        elif self.max_instances_allowed > 1:
            self.total_instances_active = 1  # primary
            try:
                child = self.notes_viewer_instance_2_dialog.instance_number
                self.notes_viewer_instance_2_dialog_exists = True
                self.total_instances_active = self.total_instances_active + 1
            except:
                self.notes_viewer_instance_2_dialog_exists = False
            try:
                child = self.notes_viewer_instance_3_dialog.instance_number
                self.notes_viewer_instance_3_dialog_exists = True
                self.total_instances_active = self.total_instances_active + 1
            except:
                self.notes_viewer_instance_3_dialog_exists = False

            self.update_push_button_create_child_instance_text()

            if DEBUG: print("self.notes_viewer_instance_2_dialog_exists: ", self.notes_viewer_instance_2_dialog_exists)
            if DEBUG: print("self.notes_viewer_instance_3_dialog_exists: ", self.notes_viewer_instance_3_dialog_exists)
            if DEBUG: print("self.total_instances_active: ", str(self.total_instances_active))

            if restart is True:
                if isinstance(restart_param,dict):  #only in restarts; usually restart_param is None.
                    if DEBUG: print("restart_param is a proper instance of a dict: ", str(restart_param))
                    child_prefs_changes_dict = restart_param
                    child_prefs = copy.deepcopy(self.nv_prefs)
                    for key,value in child_prefs_changes_dict.items():
                        if DEBUG: print("ui.py:  primaryinstance.create_nv_child_instance: ", key,value)
                        if key == 'GUI_TOOLS_NOTES_VIEWER_CUSTOMIZATION_BY_LIBRARY':
                            if DEBUG: print("key: ", key, " ignored for primaryinstance.create_nv_child_instance; only updated by NV-Customization")
                            continue  # only updated by notes_viewer_customization dialog
                        child_prefs[key] = value
                        if key in NOTES_VIEW_INSTANCE_SPECIFIC_PREFS_KEYS_SET:  # prefs keys that only pertain to child instances #2 and #3.
                            self.nv_prefs[key] = value  # must update instance-specific prefs just back from a child instance in the primary instance's self.nv_prefs for next child creation.
                            self.prefs_changes_dict[key] = value #must update child instance-specific prefs just back from a child instance in the primary instance's self.prefs_changes_dict which is returned to ui.py to update the "real" prefs, prefs.
                    #END FOR
                    del restart_param
                    del child_prefs_changes_dict
                else:
                    if DEBUG: print("isinstance(restart_param,dict) is False; returning.")
                    return
            else:
                child_prefs = copy.deepcopy(self.nv_prefs)  #normal scenario when starting NV from ui.py then creating a child instance.

            if self.total_instances_active >= self.max_instances_allowed:
                msg = "Maximum Notes Viewer instances have already been created: " + str(self.max_instances_allowed) + "<br><br>Refer to 'Options'."
                return  info_dialog(self.maingui, _(tool_name),_(msg), show=True)


            if not self.notes_viewer_instance_2_dialog_exists:
                self.freeze_notes_viewer_checkbox.setChecked(True)  # primary instance so it does not get changed by signal from self.maingui...
                instance_number = 2
                parent = self
                from calibre_plugins.job_spy.notes_viewer_dialog import NotesViewerDialog
                self.notes_viewer_instance_2_dialog = NotesViewerDialog(self.maingui,parent,self.js_icon,child_prefs,instance_number,
                                                                                        self.return_from_notes_viewer_to_save_prefs,
                                                                                        self.return_from_notes_viewer_customization_to_save_prefs,
                                                                                        self.return_from_notes_viewer_to_cancel_html_edits_and_restart)
                self.notes_viewer_instance_2_dialog.setAttribute(Qt.WA_DeleteOnClose)
                self.notes_viewer_instance_2_dialog.show()
                del NotesViewerDialog
                del child_prefs
                self.total_instances_active = self.total_instances_active + 1
                self.update_push_button_create_child_instance_text()
                if DEBUG: print("self.notes_viewer_instance_2_dialog has been created.")
            elif not self.notes_viewer_instance_3_dialog_exists:
                self.freeze_notes_viewer_checkbox.setChecked(True)  # primary instance so it does not get changed by signal from self.maingui...
                instance_number = 3
                parent = self
                from calibre_plugins.job_spy.notes_viewer_dialog import NotesViewerDialog
                self.notes_viewer_instance_3_dialog = NotesViewerDialog(self.maingui,parent,self.js_icon,child_prefs,instance_number,
                                                                                        self.return_from_notes_viewer_to_save_prefs,
                                                                                        self.return_from_notes_viewer_customization_to_save_prefs,
                                                                                        self.return_from_notes_viewer_to_cancel_html_edits_and_restart)
                self.notes_viewer_instance_3_dialog.setAttribute(Qt.WA_DeleteOnClose)
                self.notes_viewer_instance_3_dialog.show()
                del NotesViewerDialog
                del child_prefs
                self.total_instances_active = self.total_instances_active + 1
                self.update_push_button_create_child_instance_text()
                if DEBUG: print("self.notes_viewer_instance_3_dialog has been created.")
            else:
                self.update_push_button_create_child_instance_text()
                del child_prefs
                msg = "Maximum Notes Viewer instances have already been created.<br><br>Refer to 'Options'."
                if DEBUG: print(msg)
                return error_dialog(self.maingui, _(tool_name),_(msg), show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def return_open_kids_dict(self):
        open_kids_dict = {}
        if self.instance_parent.notes_viewer_instance_2_dialog is not None:
            open_kids_dict[2] = self.instance_parent.notes_viewer_instance_2_dialog
        if self.instance_parent.notes_viewer_instance_3_dialog is not None:
            open_kids_dict[3] = self.instance_parent.notes_viewer_instance_3_dialog
        return open_kids_dict
    #---------------------------------------------------------------------------------------------------------------------------------------
    def close_nv_child_instances(self):
        if self.instance_number > 1:     #  only instance #1 can close its children
            return

        try:
            child = self.notes_viewer_instance_2_dialog.instance_number
            self.notes_viewer_instance_2_dialog_exists = True
        except:
            self.notes_viewer_instance_2_dialog_exists = False
        try:
            child = self.notes_viewer_instance_3_dialog.instance_number
            self.notes_viewer_instance_3_dialog_exists = True
        except:
            self.notes_viewer_instance_3_dialog_exists = False

        if self.notes_viewer_instance_2_dialog_exists:
            try:
                try:
                    self.maingui.library_view.selectionModel().currentChanged.disconnect(self.notes_viewer_instance_2_dialog.refresh_notes_viewer)
                except Exception as e:
                    if DEBUG: print("close_nv_child_instances: instance_2: [A]", str(e))
                self.notes_viewer_instance_2_dialog.save_settings_and_exit()
                self.notes_viewer_instance_2_dialog = None
            except Exception as e:
                if DEBUG: print("close_nv_child_instances: instance_2: [B]", str(e))
                self.notes_viewer_instance_2_dialog = None

        if self.notes_viewer_instance_3_dialog_exists:
            try:
                try:
                    self.maingui.library_view.selectionModel().currentChanged.disconnect(self.notes_viewer_instance_3_dialog.refresh_notes_viewer)
                except  Exception as e:
                    if DEBUG: print("close_nv_child_instances: instance_3: [A]", str(e))
                self.notes_viewer_instance_3_dialog.save_settings_and_exit()
                self.notes_viewer_instance_3_dialog = None
            except Exception as e:
                if DEBUG: print("close_nv_child_instances: instance_3: [B]", str(e))
                self.notes_viewer_instance_3_dialog = None

        self.total_instances_active = 1
    #---------------------------------------------------------------------------------------------------------------------------------------
    def return_from_child_instance_to_close(self,child,child_instance_number,restart=False,restart_param=None):
        try:
            if DEBUG: print("return_from_child_instance_to_close:   child_instance_number, parent, self: ", str(child_instance_number), str(child), str(self))
            if child_instance_number == 2:
                self.notes_viewer_instance_2_dialog = None
                self.total_instances_active = self.total_instances_active - 1
            elif child_instance_number == 3:
                self.notes_viewer_instance_3_dialog = None
                self.total_instances_active = self.total_instances_active - 1
            if self.total_instances_active < 1:
                self.total_instances_active = 1
            self.update_push_button_create_child_instance_text()
            if DEBUG: print("return_from_child_instance_to_close -- self.total_instances_active: ", str(self.total_instances_active))
            if isinstance(restart_param,dict):  # self.prefs_changes_dict returned
                if restart:
                    if DEBUG: print("return_from_child_instance_to_close -- restart: ", restart, str(restart_param))
                    if DEBUG: print("prior to child.close() ")
                    child.close()  # also triggers qdialog.setAttribute(Qt.WA_DeleteOnClose)
                    if DEBUG: print("after child.close() ")
                    self.create_nv_child_instance(restart=restart,restart_param=restart_param)
                else:
                    #~ save the restart_param but do not restart now; in future, will restart with these just-saved prefs.
                    if DEBUG: print("return_from_child_instance_to_close -- NO restart: ", restart, str(restart_param))
                    child_prefs_changes_dict = restart_param
                    for key,value in child_prefs_changes_dict.items():
                        if key in NOTES_VIEW_INSTANCE_SPECIFIC_PREFS_KEYS_SET:  # prefs keys that only pertain to child instances #2 and #3.
                            self.nv_prefs[key] = value  # must update instance-specific prefs just back from a child instance in the primary instance's self.nv_prefs for next child creation.
                            self.prefs_changes_dict[key] = value #must update child instance-specific prefs just back from a child instance in the primary instance's self.prefs_changes_dict which is returned to ui.py to update the "real" prefs, prefs.
                            if DEBUG: print("return_from_child_instance_to_close -- self.prefs_changes_dict[k] = v: ", key, value)
                    #END FOR
                    if DEBUG: print("prior to child.close() ")
                    child.close()   # also triggers qdialog.setAttribute(Qt.WA_DeleteOnClose)
                    if DEBUG: print("after child.close() ")
            else:
                if DEBUG: print("self.return_from_child_instance_to_close -- isinstance(restart_param,dict) is False; cannot save instance prefs...")
            del restart_param
        except Exception as e:
            if DEBUG: print("self.return_from_child_instance_to_close: Exception: ", str(e))
    #---------------------------------------------------------------------------------------------------------------------------------------
    def update_push_button_create_child_instance_text(self):
        if self.instance_number != 1:
            return
        if self.max_instances_allowed == 1:
            return

        self.total_instances_active = 1

        try:
            child = self.notes_viewer_instance_2_dialog.instance_number
            self.total_instances_active = self.total_instances_active + 1
        except:
            pass
        try:
            child = self.notes_viewer_instance_3_dialog.instance_number
            self.total_instances_active = self.total_instances_active + 1
        except:
            pass

        self.available_child_instances = self.max_instances_allowed - self.total_instances_active
        if self.available_child_instances < 0:
            self.available_child_instances = 0
        self.push_button_create_child_instance.setText(str(self.available_child_instances))
        if self.available_child_instances == 0:
            self.push_button_create_child_instance.hide()
        else:
            self.push_button_create_child_instance.show()
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #~ Instance Specific Prefs/Variables
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def update_nv_last_used_prefs_by_instance(self):
        if self.instance_number == 1:
            k = 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_1_LAST_CUSTOM_COLUMN_VIEWED'
            v = self.last_current_column
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED'
            v = self.last_current_column
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_1_LAST_BOOKID_VIEWED'
            v = str(self.last_bookid)
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED'
            v = str(self.last_bookid)
            self.prefs_changes_dict[k] = v
            if DEBUG: print("Instance #1: ", k, v)
        elif self.instance_number == 2:
            k = 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_2_LAST_CUSTOM_COLUMN_VIEWED'
            v = self.last_current_column
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED'
            v = self.last_current_column
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_2_LAST_BOOKID_VIEWED'
            v = str(self.last_bookid)
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED'
            v = str(self.last_bookid)
            self.prefs_changes_dict[k] = v
            if DEBUG: print("Instance #2: ", k, v)
        elif self.instance_number == 3:
            k = 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_3_LAST_CUSTOM_COLUMN_VIEWED'
            v = self.last_current_column
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED'
            v = self.last_current_column
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_3_LAST_BOOKID_VIEWED'
            v = str(self.last_bookid)
            self.prefs_changes_dict[k] = v
            k = 'GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED'
            v = str(self.last_bookid)
            self.prefs_changes_dict[k] = v
            if DEBUG: print("Instance #3: ", k, v)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def initialize_last_used_vars_by_instance(self):

        #-----------------------------------------------------
        db = self.maingui.current_db.new_api
        frozen = db.all_book_ids()
        self.all_book_ids_set = set(frozen)
        del frozen
        #-----------------------------------------------------

        self.library_id = self.maingui.current_db.new_api.backend.library_id
        last_library_id_used = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_LAST_LIBRARY_USED']
        if self.library_id != last_library_id_used:
            #~ note:  the last used bookids by instance are remembered and reused only until the library changes.
            if self.instance_number == 1:
                self.last_bookid = 0
                self.last_current_column = ""
                self.startup_column = self.last_current_column
            else:
                self.last_bookid = self.instance_parent.last_bookid
                self.last_current_column = self.instance_parent.last_current_column
                self.startup_column = self.last_current_column
            if DEBUG: print("initialize_last_used_vars_by_instance: ", str(self.instance_number), ":  self.library_id != last_library_id_used:  self.last_current_column, self.startup_column, self.last_bookid", self.last_current_column, self.startup_column, str(self.last_bookid))
            return

        #~ ----------------------------------------------
        #~ generics are the default
        self.last_current_column = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED']

        lbid_ = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED']
        if DEBUG: print("ibid_ for 'GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED' : ", lbid_ )
        lbid_ = int(lbid_.strip())
        if not lbid_ in self.all_book_ids_set:
            lbid = 0
        else:
            lbid = lbid_
        #~ ----------------------------------------------

        if self.instance_number == 1:
            self.last_current_column = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_INSTANCE_1_LAST_CUSTOM_COLUMN_VIEWED']
            lbid_ = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_INSTANCE_1_LAST_BOOKID_VIEWED']
            if DEBUG: print("ibid_ for 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_1_LAST_BOOKID_VIEWED' : ", lbid_ )
            lbid_ = int(lbid_.strip())
            if not lbid_ in self.all_book_ids_set:
                pass # use generic
            else:
                lbid = lbid_  # instance specific overrides generic
                self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED'] = str(lbid)                    # parent's overrides saved generic prefs
                self.prefs_changes_dict['GUI_TOOLS_NOTES_VIEWER_LAST_BOOKID_VIEWED'] = str(lbid)   # parent's overrides saved generic prefs
        elif self.instance_number == 2:
            self.last_current_column = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_INSTANCE_2_LAST_CUSTOM_COLUMN_VIEWED']
            lbid_ = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_INSTANCE_2_LAST_BOOKID_VIEWED']
            if DEBUG: print("ibid_ for 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_2_LAST_BOOKID_VIEWED' : ", lbid_ )
            lbid_ = int(lbid_.strip())
            if not lbid_ in self.all_book_ids_set:
                pass # use generic
            else:
                lbid = lbid_  # instance specific overrides generic
        elif self.instance_number == 3:
            self.last_current_column = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_INSTANCE_3_LAST_CUSTOM_COLUMN_VIEWED']
            lbid_ = self.nv_prefs['GUI_TOOLS_NOTES_VIEWER_INSTANCE_3_LAST_BOOKID_VIEWED']
            if DEBUG: print("ibid_ for 'GUI_TOOLS_NOTES_VIEWER_INSTANCE_3_LAST_BOOKID_VIEWED' : ", lbid_ )
            lbid_ = int(lbid_.strip())
            if not lbid_ in self.all_book_ids_set:
                pass # use generic
            else:
                lbid = lbid_  # instance specific overrides generic

        self.startup_column = self.last_current_column
        self.last_bookid = lbid

        if not self.last_bookid in self.all_book_ids_set:
            self.last_bookid = 0

        if self.last_bookid == 0:
            self.last_bookid = self.maingui.library_view.current_book

        if DEBUG: print("initialize_last_used_vars_by_instance: ", str(self.instance_number), ":  self.last_current_column, self.startup_column, self.last_bookid", self.last_current_column, self.startup_column, str(self.last_bookid))

        self.update_nv_last_used_prefs_by_instance()  #sync saved prefs with initialized variables from prefs saved at another time
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
#END OF notes_viewer_dialog.py