# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.191"  #Qt6

from qt.core import (Qt, QDialog, QApplication,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout)

from calibre.constants import DEBUG
from calibre.gui2 import gprefs

from polyglot.builtins import as_unicode, iteritems, range

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(400, 200)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        #~ self.geom = None
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class CustomColumnsMatrixByLibraryDialog(SizePersistedDialog):

    def __init__(self, parent, library_cc_dict):
        unique_pref_name = 'Job_Spy:custom_columns_matrix_by_library_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        mytitle = 'JS+ GUI Tool:  Matrix of Custom Columns by Library:    '  + as_unicode(len(library_cc_dict)) + " Library Paths"
        self.setWindowTitle(_(mytitle))

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        self.library_cc_dict = library_cc_dict

        matrix_list = []

        #~ for library,cc_list in self.library_cc_dict.iteritems():
        for library,cc_list in iteritems(self.library_cc_dict):
            if cc_list is None:
                r = library,"","...Path Error...","","","","","",""
                matrix_list.append(r)
                del r
            else:
                for row in cc_list:
                    id,label,name,datatype,editable,display,is_multiple,normalized = row
                    r = library,id,label,name,datatype,editable,display,is_multiple,normalized
                    matrix_list.append(r)
                    del r
                #END FOR
        #END FOR

        self.n_total_rows = len(matrix_list)

        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("library path")
        column_label_list.append("label")
        column_label_list.append("name")
        column_label_list.append("datatype")
        column_label_list.append("id")
        column_label_list.append("editable")
        column_label_list.append("is_multiple")
        column_label_list.append("normalized")
        column_label_list.append("display")


        self.n_total_cols = 9

        self.matrix = QTableWidget(self.n_total_rows,self.n_total_cols)

        self.matrix.setSortingEnabled(False)

        self.matrix.setHorizontalHeaderLabels(column_label_list)

        self.deoptimize_column_widths()

        self.matrix.clearContents()
        #--------------------------------------------------

        r = 0
        for row in matrix_list:
            try:
                #~ if DEBUG: print(as_unicode(row))
                #---------------------------
                #---------------------------
                library,id,label,name,datatype,editable,display,is_multiple,normalized = row

                id = as_unicode(id)
                editable = as_unicode(editable)
                is_multiple = as_unicode(is_multiple)
                normalized = as_unicode(normalized)
                if not display:
                    display = ""
                else:
                    display = as_unicode(display)

                #---------------------------
                #---------------------------
                library_ = QTableWidgetItem(library)
                label_ = QTableWidgetItem(label)
                name_ = QTableWidgetItem(name)
                datatype_ = QTableWidgetItem(datatype)
                id_ = QTableWidgetItem(id)
                editable_ = QTableWidgetItem(editable)
                is_multiple_ = QTableWidgetItem(is_multiple)
                normalized_ = QTableWidgetItem(normalized)
                display_ = QTableWidgetItem(display)
                #---------------------------
                #---------------------------
                self.matrix.setItem(r,0,library_)
                self.matrix.setItem(r,1,label_)
                self.matrix.setItem(r,2,name_)
                self.matrix.setItem(r,3,datatype_)
                self.matrix.setItem(r,4,id_)
                self.matrix.setItem(r,5,editable_)
                self.matrix.setItem(r,6,is_multiple_)
                self.matrix.setItem(r,7,normalized_)
                self.matrix.setItem(r,8,display_)
                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("class CustomColumnsMatrixByLibraryDialog(SizePersistedDialog):", as_unicode(e))
                return
        #END FOR

        del matrix_list

        self.n_total_rows = r

        layout.addWidget(self.matrix)

        self.matrix.sortItems(0,Qt.AscendingOrder )

        self.matrix.setSortingEnabled(True)

        self.resize_all_columns()

        #-----------------------------------------------------

        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        layout.addWidget(self.bottom_buttonbox)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.setToolTip("<p style='white-space:wrap'>The matrix columns will be resized based on their longest row contents for each column.")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,QDialogButtonBox.AcceptRole)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.setToolTip("<p style='white-space:wrap'>The matrix columns will be resized to a fixed width regardless of their contents.")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,QDialogButtonBox.AcceptRole)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Export to Clipboard")
        self.push_button_copy_to_clipboard.setToolTip("<p style='white-space:wrap'>The matrix in its entirety will be copied to the Clipboard in a tab-delimited format.  Paste Special into a Spreadsheet, or Paste into a Text Document.")
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_log_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()

        self.optimize_column_widths()

        t = "<p style='white-space:wrap'>"
        t = t + "<b>Library Path:</b>  the 'remembered' Libraries that are shown under the Calibre Library icon."
        t = t + "<br><br><b>Label:</b>  the #name used in searches."
        t = t + "<br><br><b>Name:</b> the displayed column heading for the #name."
        t = t + "<br><br><b>Datatype:</b>  the type of data the custom column contains (text, comments, series, datetime, ratings, boolean, integers, floating point numerics, etc.) or whether it was built from other columns ('composite')."
        t = t + "<br><br><b>ID:</b>  the numeric identifier for this custom column.  Used in the metadata.db table names (e.g. custom_column_23 or books_custom_column_23_link). "
        t = t + "<br><br><b>Editable:</b> whether or not a custom column appears in 'Edit Metadata'.  Default is True.  It can be changed to False using the Job Spy GUI Tool 'Protect/Unprotect (Hide/Show) Custom Columns from Edit Metadata'. "
        t = t + "<br><br><b>Is_Multiple:</b> whether or not textual values are separated with a comma or ampersand.  If True, the default is comma unless 'like names' is specified in the Display.  "
        t = t + "<br><br><b>Normalized:</b> always True for the datatypes of Text, Enumeration, Series, Ratings.  The Tag Browser displays columns that are True.  Comments and the other datatypes are always False."
        t = t + "<br><br><b>Display:</b>  contains values specified by Calibre and/or the User that combined with Datatype tell Calibre how to display the custom column in the GUI and how to handle other functionality.  \
        Enumeration datatype valid values are stored in Display.  Comments are further defined as 'short text' or 'long text' within their Display.  Datetime and Integer/Float formats chosen by the User are stored in Display.  \
        The Description shown as a tooltip for each custom column heading is stored in its Display."

        self.matrix.setToolTip(t)

    #-----------------------------------------------------
    def resize_all_columns(self):
        self.matrix.resizeColumnsToContents()
        self.save_custom_columns_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.matrix.resizeColumnsToContents()
        self.save_custom_columns_listing_dialog_geometry()
    #-----------------------------------------------------
    def deoptimize_column_widths(self):
        self.matrix.setColumnWidth(0, 200)
        self.matrix.setColumnWidth(1, 200)
        self.matrix.setColumnWidth(2, 200)
        self.matrix.setColumnWidth(3, 200)
        self.matrix.setColumnWidth(4, 100)
        self.matrix.setColumnWidth(5, 100)
        self.matrix.setColumnWidth(6, 100)
        self.matrix.setColumnWidth(7, 100)
        self.matrix.setColumnWidth(8, 200)
    #-----------------------------------------------------
    def copy_log_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.matrix.selectAll()

        s = ''
        s = s + "\t".join([as_unicode(self.matrix.horizontalHeaderItem(i).text()) for i in range(0, self.n_total_cols)])
        s = s +  '\n'
        for r in range(0, self.n_total_rows):
            for c in range(0, self.n_total_cols):
                try:
                    s = s + as_unicode(self.matrix.item(r,c).text()) + "\t"
                except AttributeError:
                    s = s + "\t"
            s = s[:-1] + "\n"        #eliminate last '\t'
        self.clip.setText(s)

        self.save_custom_columns_listing_dialog_geometry()
    #-----------------------------------------------------
    def save_custom_columns_listing_dialog_geometry(self):
        self.dialog_closing(None)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#END OF custom_columns_matrix_by_library_dialog.py