# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.2.5"   #  Links

from qt.core import QMenu, QDialog, QIcon, QAction

import os, sys
from functools import partial
import zipfile
import subprocess

from calibre.gui2 import question_dialog, error_dialog, Dispatcher
from calibre.gui2.actions import InterfaceAction

from polyglot.builtins import as_unicode
from polyglot.queue import Queue

from calibre_plugins.author_book_count_hierarchy.common_utils import set_plugin_icon_resources, get_icon, create_menu_action_unique
from calibre_plugins.author_book_count_hierarchy.jobs import start_abc_hierarchy_threaded
from calibre_plugins.author_book_count_hierarchy.main import Create_ABC_Hierarchy

PLUGIN_ICONS = ['images/abchicon.png', 'images/abchsmallicon.png', 'images/readinstructionsicon.png']

class AuthorBookCountHierarchyUpdate(InterfaceAction):

    name = 'author_book_count_hierarchy'
    action_spec = ('ABCH', 'images/abchicon.png', 'Create ABC Hierarchy', None)
    action_type = 'global'
    accepts_drops = False
    auto_repeat = False
    priority = 9
    popup_type = 1

    #-----------------------------------------------------------------------------------------
    def genesis(self):
        self.menu = QMenu(self.gui)
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources )
        self.build_menus()
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.start_threaded_books_to_count)
 #-----------------------------------------------------------------------------------------
    def initialization_complete(self):
        self.maingui = self.gui
        self.guidb = self.gui.library_view.model().db
 #-----------------------------------------------------------------------------------------
    def library_changed(self, guidb):
        self.guidb = self.gui.library_view.model().db
    #-----------------------------------------------------------------------------------------
    def start_threaded_books_to_count(self):
        self.guidb = self.gui.library_view.model().db
        if question_dialog(self.gui, "Author Book Count Hierarchy", "Execute Author Book Count Hierarchy:  Continue?"):
            start_abc_hierarchy_threaded(self, self.gui, self.guidb, Dispatcher(self.abch_job_complete) )
            self.gui.status_bar.show_message(_('ABC Hierarchy Job Has Been Submitted'), 500)
        else:
            return
    #-----------------------------------------------------------------------------------------
    def abch_job_complete(self, job):
        if job.failed:
            error_dialog(self.gui, 'ABCH: Job Has Failed', 'You Must Restart Calibre Before Running ABCH in this Library.' , ' ' ).show()
            return
        self.force_refresh_of_cache()
        self.gui.status_bar.show_message(_('ABC Hierarchy Job Has Completed'), 10000)
    #-----------------------------------------------------------------------------------------
    def build_menus(self):
        m = self.menu
        m.clear()
        m.addSeparator()
        unique_name = "Author Book Count Hierarchy: Run ABC Job"
        create_menu_action_unique(self, m, 'Author Book Count Hierarchy', 'images/abchsmallicon.png',
                              triggered=partial(self.start_threaded_books_to_count),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        unique_name = "Author Book Count Hierarchy: Read Instructions"
        create_menu_action_unique(self, m, 'Read Instructions', 'images/readinstructionsicon.png',
                              triggered=partial(self.view_user_instructions),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        self.gui.keyboard.finalize()
    #-----------------------------------------------------------------------------------------
    def view_user_instructions(self):

        self.extract_documentation_from_zip() #every time to ensure latest copy is available after plugin upgrade

        if self.documentation_path is None:
            return

        try:
            p_pid = subprocess.Popen(self.documentation_path, shell=True)

            #Subprocess "p_pid"  is totally independent of Calibre,  and will never be checked or killed by this plugin...

        except:
            return error_dialog(self.gui, _('Documentation .pdf Not Found. Try reinstalling this plugin, and restarting Calibre.'),
                                       _('It is supposed to be: ' + self.documentation_path ), show=True)
    #-----------------------------------------------------------------------------------------
    def extract_documentation_from_zip(self):
        #extract the .pdf file from the zip so it can be executed directly.

        self.documentation_path = None

        self.plugin_path = as_unicode(self.plugin_path)

        zipfile_path = self.plugin_path

        destination_path = self.plugin_path
        destination_path = destination_path.replace("\Author Book Count Hierarchy.zip", "", 1)
        destination_path = destination_path.replace("/Author Book Count Hierarchy.zip", "", 1)

        zfile = zipfile.ZipFile(zipfile_path)

        dir_name = "abch_documentation"
        #~ dir_name = dir_name.encode("ascii", "strict")

        file_name = 'abch_instructions.pdf'
        #~ file_name = file_name.encode("ascii", "strict")

        file_name = os.path.join(dir_name, file_name )
        file_name = as_unicode(file_name)

        for name in zfile.namelist(): #all files in zip with full internal paths
            name = as_unicode(name)
            n = name.find(dir_name)
            if n >= 0:
                zfile.extract(name, destination_path)
                self.documentation_path = os.path.join(destination_path, file_name)
                self.documentation_path = as_unicode(self.documentation_path)
                #"C:\Users\DaltonST\AppData\Roaming\calibre\plugins\abch_documentation\abch_instructions.pdf"

  #-------------------------------------------------------------------------------------------------------------------------------------
    def force_refresh_of_cache(self):
        db = self.maingui.current_db.new_api
        frozen = db.all_book_ids()
        books = list(frozen)
        db.reload_from_db(clear_caches=False)
        self.maingui.library_view.model().refresh_ids(books)
        self.maingui.tags_view.recount()
    #-----------------------------------------------------------------------------------------
#END of ui.py