# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.85"  # Qt6

import os
from functools import partial

from qt.core import QApplication

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.gui2 import error_dialog,warning_dialog
from calibre.gui2.actions import InterfaceAction

from polyglot.builtins import  as_unicode,iteritems

from calibre_plugins.calibrespy.common_utils import set_plugin_icon_resources, get_icon, create_menu_action_unique
from calibre_plugins.calibrespy.calibrespy_dialog import *
from calibre_plugins.calibrespy.config import prefs

PLUGIN_ICONS = ['images/calibrespy.png','images/calibrespy_closed.png']

#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
class ActionCalibreSpy(InterfaceAction):

    name = 'CalibreSpy'
    action_spec = ('CalibreSpy', 'images/calibrespy.png', 'Light GUI for browsing, viewing, searching using VLs and many other search tools, and using many libraries simultaneously by multiple users without running the Calibre GUI itself.', None)
    action_type = 'global'
    accepts_drops = False
    auto_repeat = False
    priority = 9
    popup_type = 1
#---------------------------------------------------------------------------------------------------------------------------------------
    def genesis(self):
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.menu = QMenu(self.gui)
        self.menu.clear()
        self.menu.addSeparator()
        create_menu_action_unique(self, self.menu, 'CalibreSpy: normal read-only', 'images/calibrespy.png',
                              triggered=partial(self.normal_read_only))
        self.menu.addSeparator()
        create_menu_action_unique(self, self.menu, 'CalibreSpy: entirely read-only', 'images/calibrespy_closed.png',
                              triggered=partial(self.entirely_read_only))
        self.menu.addSeparator()
        create_menu_action_unique(self, self.menu, 'CalibreSpy: normal read-only [Prefilter]', 'images/calibrespy.png',
                              triggered=partial(self.normal_read_only_prefilter))
        self.menu.addSeparator()
        create_menu_action_unique(self, self.menu, 'CalibreSpy: entirely read-only [Prefilter]', 'images/calibrespy_closed.png',
                              triggered=partial(self.entirely_read_only_prefilter))
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.normal_read_only)
        self.gui.keyboard.finalize()

        self.menu.setToolTip("<p style='white-space:wrap'>CalibreSpy is <b><i>always</i></b> read-only for Books etc.\
                                                                                          <br><br>It can optionally also be read-only for CalibreSpy's own settings ('entirely read-only').\
                                                                                          <br><br>Network Libraries should always be viewed as 'entirely read-only'.\
                                                                                         <br><br>You should exit from the current CalibreSpy window <b><i>before</i></b> creating another to avoid inconvenient multi-user complications.\
                                                                                         <br><br>Only one (1) CalibreSpy window created from the Calibre GUI icon is allowed at any one (1) time.\
                                                                                         <br><br><i>CalibreSpy executed from a command file has no limitations on simultaneous multiple users.</i>\
                                                                                         <br><br>'Prefiltering' uses Regular Expressions to allow loading only specific books rather than the entire Library, greatly increasing performance.\
                                                                                         <br><br>Refer to the command file templates in '.../calibre/plugins/calibrespy_cli' for the available command file execution options.")
#---------------------------------------------------------------------------------------------------------------------------------------
    def initialization_complete(self):
        #~ for k,v in prefs.defaults.iteritems():
        for k,v in iteritems(prefs.defaults):
            if not k in prefs:
                prefs[k] = v
                prefs
        #END FOR
        self.extract_plugin_resources()
        self.remove_extra_files()
        self.ros = False
        if DEBUG: print("CalibreSpy: initialization_complete")
#---------------------------------------------------------------------------------------------------------------------------------------
    def normal_read_only(self):
        self.ros = False
        self.run_calibrespy_from_gui(source="normal",prefilter=False)
#---------------------------------------------------------------------------------------------------------------------------------------
    def entirely_read_only(self):
        self.ros = True
        self.run_calibrespy_from_gui(source="entirely",prefilter=False)
#---------------------------------------------------------------------------------------------------------------------------------------
    def normal_read_only_prefilter(self):
        self.ros = False
        self.run_calibrespy_from_gui(source="normal",prefilter=True)
#---------------------------------------------------------------------------------------------------------------------------------------
    def entirely_read_only_prefilter(self):
        self.ros = True
        self.run_calibrespy_from_gui(source="entirely",prefilter=True)
#---------------------------------------------------------------------------------------------------------------------------------------
    def run_calibrespy_from_gui(self,source=None,prefilter=False):

        is_valid = self.verify_current_location()
        if not is_valid:
            return error_dialog(None, _('CalibreSpy'),_('CalibreSpy resources directory is missing.  You must restart Calibre to repair CalibreSpy.'), show=True)

        try:
            self.calibrespy_dialog_gui.close()
            warning_dialog(None, _('CalibreSpy'),_("You should exit from the current CalibreSpy window <b><i>before</i></b> \
                                                                            creating another to avoid inconvenient multi-user complications.\
                                                                            <br><br>Only one (1) CalibreSpy window created from the Calibre GUI's icon is allowed at any one time.\
                                                                            <br><br><i>CalibreSpy executed from a command file have no limitations on simultaneous multiple users.</i>"), show=True)
        except:
            pass

        if prefs['CALIBRESPY_GLOBALLY_CONFIGURED'] == 0:
            return error_dialog(None, _('CalibreSpy'),_('CalibreSpy Global Default Preferences Have Not Yet Been Customized via Preferences > Plugins > Customize Plugins.'), show=True)

        if prefs['CALIBRESPY_ENABLE_GUI_EXECUTION'] == 0:
            return error_dialog(None, _('CalibreSpy'),_('CalibreSpy has not been enabled for execution from the GUI in Preferences > Plugins > Customize Plugins.\
                                                                                <br><br>Its icon can be removed from your ToolBars via Preferences > ToolBars & Menus.'), show=True)

        username = self.get_username()

        if source is None:
            self.ros = False

        self.calibrespy_dialog_gui = CalibreSpyDialog(execution_source=EXEC_SOURCE_GUI,cli_args_list=None,username=username,ros=self.ros,prefilter=prefilter)
        self.calibrespy_dialog_gui.setAttribute(Qt.WA_DeleteOnClose )
#---------------------------------------------------------------------------------------------------------------------------------------
    def extract_plugin_resources(self):

        try:
            destination_path = self.plugin_path
            destination_path = destination_path.replace("\CalibreSpy.zip", "")
            destination_path = destination_path.replace("/CalibreSpy.zip", "")

            dir_name = "calibrespy_cli"

            clidir = os.path.join(destination_path,dir_name)

            if isbytestring(clidir):
                clidir = clidir.decode(filesystem_encoding)
            clidir = clidir.replace(os.sep, '/')

            if os.path.isdir(clidir):
                pass
            else:
                if DEBUG: print("CalibreSpy CLI dir does NOT exist; it will be created: ", clidir)
                os.mkdir(clidir)

            prefs["CALIBRESPY_CLI_SUBDIRECTORY"] = clidir
            prefs

            zipfile_path = self.plugin_path
            import zipfile
            zfile = zipfile.ZipFile(zipfile_path)

            zfile.extractall(clidir)

            del zipfile
            del zfile

        except Exception as e:
            if DEBUG: print("extract_plugin_resources Exception: ", as_unicode(e))
#---------------------------------------------------------------------------------------------------------------------------------------
    def verify_current_location(self):
        #some users have deleted or moved the /calibre/plugin/calibrespy_cli directory while Calibre was still running, so the initialization section that always extracts files had already run...

        destination_path = self.plugin_path
        destination_path = destination_path.replace("\CalibreSpy.zip", "")
        destination_path = destination_path.replace("/CalibreSpy.zip", "")

        dir_name = "calibrespy_cli"

        clidir = os.path.join(destination_path,dir_name)

        if isbytestring(clidir):
            clidir = clidir.decode(filesystem_encoding)
        clidir = clidir.replace(os.sep, '/')

        force_extraction = False

        if not os.path.isdir(clidir):
            if DEBUG: print("CalibreSpy CLI dir does NOT exist; it will be created: ", clidir)
            os.mkdir(clidir)
            force_extraction = True
            if DEBUG: print("Extraction of zip being forced since calibrespy_cli is missing...")

        if os.path.isdir(clidir):
            if DEBUG: print("Current Location of CALIBRESPY_CLI_SUBDIRECTORY: ", as_unicode(clidir))
            #~ if DEBUG: print("Prior Location of CALIBRESPY_CLI_SUBDIRECTORY: ", as_unicode(prefs["CALIBRESPY_CLI_SUBDIRECTORY"]))
            calibrespy_icon_path = clidir + "/images/calibrespy.png"
            if not os.path.isfile(calibrespy_icon_path):
                force_extraction = True
                if DEBUG: print("Extraction of zip being forced since icons are missing...")
            if clidir != prefs["CALIBRESPY_CLI_SUBDIRECTORY"]:
                force_extraction = True
            if force_extraction:
                prefs["CALIBRESPY_CLI_SUBDIRECTORY"] = clidir
                prefs
                self.extract_plugin_resources()
                self.remove_extra_files()
                if DEBUG: print("User's .json file and/or Calibre's config dir and/or calibrespy_cli images have been altered by the user; CalbreSpy.zip resources were newly extracted to the current calibrespy_cli path.")
            return True
        else:
            return False
#---------------------------------------------------------------------------------------------------------------------------------------
    def library_changed(self,guidb):
        #~ not necessary since guidb is not used by CalibreSpy (ever), but users should use the CLI method...this nudges them to do so...
        try:
            self.calibrespy_dialog_gui.close()  # this is not the CLI-executed instance
        except:
            pass
#---------------------------------------------------------------------------------------------------------------------------------------
    def remove_extra_files(self):
        clidir = prefs["CALIBRESPY_CLI_SUBDIRECTORY"]
        files_list = os.listdir(clidir)
        for file in files_list:
            file = os.path.join(clidir, file)
            if isbytestring(file):
                file = file.decode(filesystem_encoding)
            file = file.replace(os.sep, '/')
            if os.path.isfile(file):
                if not file.endswith(".bat"):
                    try:
                        os.remove(file)
                    except Exception as e:
                        if DEBUG: print("cannot remove old file (locked?): ", file, "---> ", as_unicode(e))
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_username(self):
        try:
            username = os.path.expanduser('~')
        except Exception as e:
            if DEBUG: print("get_username error: ", as_unicode(e))
            username = "unknown user"
        return username
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#end of ui.py