# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.91"  # Qt.core

import apsw, ast, codecs, os, platform, subprocess, sys
from datetime import datetime, timedelta
from dateutil.parser import parse as dt_parse

from qt.core import pyqtSignal
from qt.core import (Qt, QDialog, QLabel, QFont, QFontMetrics, QWidget, QApplication, QComboBox, QToolTip,
                                       QIcon, QPixmap, QImage, QInputDialog, QAbstractItemView, QHeaderView,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox, QTimer, QKeySequence, QPoint,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout, QAction, QMenu, QMimeData, QUrl,
                                       QScrollArea, QWidget)

from calibre import isbytestring
from calibre.constants import filesystem_encoding, iswindows, DEBUG
from calibre.gui2 import error_dialog, question_dialog

from calibre.utils.config import prefs as mainprefs
DEFAULT_BOOK_FORMAT = mainprefs.get('output_format')
del mainprefs

from calibre.utils.date import format_date

from polyglot.builtins import as_unicode, iteritems, range, unicode_type

from calibre_plugins.calibrespy.config import ( DISPLAY_SIZE_TINY, DISPLAY_SIZE_SMALL, DISPLAY_SIZE_NORMAL,
                                                                             IDENTIFIERS_COLUMN, RATINGS_COLUMN,    #Version 1.0.43: Identifiers   #Version 1.0.52: Ratings
                                                                             MATRIX_MAXIMUM_COLUMNS_COUNT)

AUTHORS = 'authors'
TITLE = 'title'
SERIES = 'series'
SERIES_INDEX = 'series_index'
TAGS = 'tags'
PUBLISHER = 'publisher'
PUBLISHED = 'published'
LANGUAGES = 'languages'
ADDED = 'added'
MODIFIED = 'modified'
PATH = 'path'
CUSTOM_COLUMN_1 = "custom_column_1"
CUSTOM_COLUMN_2 = "custom_column_2"
CUSTOM_COLUMN_3 = "custom_column_3"
CUSTOM_COLUMN_4 = "custom_column_4"
CUSTOM_COLUMN_5 = "custom_column_5"
CUSTOM_COLUMN_6 = "custom_column_6"
CUSTOM_COLUMN_7 = "custom_column_7"
CUSTOM_COLUMN_8 = "custom_column_8"
CUSTOM_COLUMN_9 = "custom_column_9"
#~ ---------------------------------------------------
# newly available columns must be added below...
#~ ---------------------------------------------------
IDENTIFIERS = 'identifiers'   #Version 1.0.43
RATINGS = 'ratings'             #Version 1.0.52

#~ ---------------------------------------------------

AUTHORS_HEADING = "Author(s)"
TITLE_HEADING = "Title"
SERIES_HEADING = "Series"
INDEX_HEADING = "Index"
TAGS_HEADING = "Tags"
PUBLISHER_HEADING = "Publisher"
PUBLISHED_HEADING = "Published"
LANGUAGES_HEADING = "Lang"
ADDED_HEADING = "Added"
MODIFIED_HEADING = "Modified"
PATH_HEADING = "Path"
#~ ---------------------------------------------------
# newly available columns must be added below...
#~ ---------------------------------------------------
IDENTIFIERS_HEADING = "Identifiers"  #Version 1.0.43
RATINGS_HEADING = "Rating"     #Version 1.0.52

#~ ---------------------------------------------------

AUTHORS_SOURCE = "Authors:"
TITLE_SOURCE = "Title:"
SERIES_SOURCE = "Series:"
TAGS_SOURCE  = "Tags:"
PUBLISHER_SOURCE = "Publisher:"
LANGUAGES_SOURCE = "Languages:"
CUSTOM_COLUMN_SOURCE = "Custom Column"

DEFAULT_AUTHORS = "Author(s): ⁎"
DEFAULT_AUTHORS_NONE = ""
DEFAULT_AUTHORS_PLACEHOLDER = "〜"
DEFAULT_TITLE =  "Title: ⁎"
DEFAULT_TITLE_NONE = ""
DEFAULT_TITLE_PLACEHOLDER = "〜"
DEFAULT_SERIES =  "Series: ⁎"
DEFAULT_SERIES_NONE =  ""
DEFAULT_SERIES_PLACEHOLDER = "〜"
DEFAULT_TAGS = "Tags: ⁎"
DEFAULT_TAGS_NONE = ""
DEFAULT_TAGS_PLACEHOLDER = "〜"
DEFAULT_PUBLISHER = "Publisher: ⁎"
DEFAULT_PUBLISHER_NONE = ""
DEFAULT_PUBLISHER_PLACEHOLDER = "〜"

EXEC_SOURCE_CLI = "CLI"
EXEC_SOURCE_GUI = "GUI"

SOURCE_CONTEXT_MENU = "context_menu"
SOURCE_FILTER_USING_COLORS = "filter_using_colors"
SOURCE_FILTER_USING_HISTORY = "filter_using_history"
SOURCE_FILTER_AFTER_SEARCH = "filter_after_search"
SOURCE_INVERT_SELECTION = "invert_selection"

DEFAULT_FONT_WINDOWS = 'Segoe UI'
DEFAULT_FONT_OTHER = 'Sans Serif'

HISTORY_MAXIMUM_VALUES = 30
HISTORY_SEPARATOR = "|␢|"
HISTORY_SEPARATOR_LEGACY = "|^|"
HISTORY_HEADING_NAME = "Filtering History"

SEARCH_EXPRESSIONS_HEADING_VALUE = "^.+$"
SEARCH_TARGET_HEADING_NAME = "Search Column"
SEARCH_SEPARATOR = "|␢|"
SEARCH_MAXIMUM_VALUES = 30

MARKED_BOOK_SEPARATOR = "|␢|"
CYAN = "cyan"
RED = "red"
YELLOW = "yellow"
GREEN = "green"

CLIPBOARD_LISTENER_INTERVAL = 6000
TAB = "\t"
NEWLINE = "\n"
METADATA_STRING = "METADATA:"
HEADING_STRING = "HEADING:"

FTP_SEND_ACTION = "send"
FTP_DELETE_ACTION = "delete"

SAVE_COPY_TYPE_SINGLE = "single"
SAVE_COPY_TYPE_FILTERED = "filtered"

#---------------------------------------------------------------------------------------------------------------------------------------
class CalibreSpyDialogBase(QDialog):
    #-------------------------
    #~ CalibreSpy does *not* save any prefs to the global Calibre prefs, gprefs, since *many* Libraries can be opened by CalibreSpy at one time.
    #~ Calibre gprefs was not designed assuming that was possible and normal, so bad things could happen to gui.json, causing Calibre to reset it.
    #~ The use of the Qt .saveState() and .restoreState() bytearrays saved in metadata.db table _calibrespy_settings as a bytearray 'blob' compensates for that (and much more).
    #~ Qt exterior window dimensions are also saved at exit.
    #-------------------------
    def __init__(self, parent=None):
        QDialog.__init__(self, parent, Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint)
    def resize_dialog_to_preferences(self):
        size = QSize()
        size.setWidth(self.local_prefs['DIALOG_SIZE_WIDTH_LAST'])
        size.setHeight(self.local_prefs['DIALOG_SIZE_HEIGHT_LAST'])
        self.resize(size)
#---------------------------------------------------------------------------------------------------------------------------------------
class CalibreSpyDialog(CalibreSpyDialogBase):

    resized_signal = pyqtSignal()

    def __init__(self=None,execution_source=None,cli_args_list=None,username=None,qtapp=None,ros=None,prefilter=False):
        CalibreSpyDialogBase.__init__(self)
        #--------------------------------------------------
        #--------------------------------------------------
        self.cli_args_list = cli_args_list
        self.username = username
        self.username_timestamp = unicode_type(datetime.now())

        if ros is None:  #only passed by EXEC_SOURCE_GUI
            self.session_is_entirely_readonly = False
        elif ros == True:
            self.session_is_entirely_readonly = True
        else:
            self.session_is_entirely_readonly = False

        self.prefilter = prefilter  #allow pre-filtering of books from the selected metadata.db (e.g. for small tablets with big libraries)

        if execution_source == EXEC_SOURCE_CLI:
            self.is_cli = True
            self.skip_qinputdialog = False
            s = as_unicode(self.cli_args_list)
            if as_unicode("--ros") in s:   #read-only session re: CalibreSpy settings table (rest is always RO)
                self.session_is_entirely_readonly = True
            if as_unicode("--reset") in s:   #reset the CalibreSpy settings table to the current global prefs.default[]
                self.reset = True
            else:
                self.reset = False
            if as_unicode("--prefilter") in s:   #allow pre-filtering of books from the selected metadata.db (e.g. for small tablets with big libraries)
                self.prefilter = True
            else:
                self.prefilter = False
            if s.count(as_unicode("/")) == 0 and s.count(as_unicode("\\")) == 0:
                self.skip_qinputdialog = False
            elif self.cli_args_list is not None:
                if isinstance(self.cli_args_list,list):
                    s = as_unicode(self.cli_args_list)
                    if s.count(as_unicode("/")) == 0 and s.count(as_unicode("\\")) == 0:
                        self.skip_qinputdialog = False
                    else:
                            self.library_path = self.cli_args_list[1]
                            self.library_path = self.library_path.replace("--ros","").strip()
                            if self.library_path.startswith("--"):
                                self.library_path = self.library_path[2: ]
                            self.library_path_original = self.library_path
                            self.library_path =self.library_path.replace("\\","/")
                            if DEBUG: print("self.cli_args self.library_path: ", self.library_path)
                            if os.path.isdir(self.library_path):
                                self.skip_qinputdialog = True
                            else:
                                if DEBUG: print("self.cli_args self.library_path is Invalid; path does not exist; ignoring CLI arguments")
                                self.skip_qinputdialog = False
                    #END IF
                else:
                    self.skip_qinputdialog = False
            else:
                self.skip_qinputdialog = False
            import gc
            self.gc = gc
        else:
            self.is_cli = False
            self.skip_qinputdialog = False
            self.reset = False
        #--------------------------------------------------
        #--------------------------------------------------
        self.setWindowFlags(Qt.Window | Qt.WindowTitleHint | Qt.WindowMinMaxButtonsHint)
        #--------------------------------------------------
        #--------------------------------------------------
        from calibre_plugins.calibrespy.config import prefs
        if not self.skip_qinputdialog:
            from calibre.gui2 import gprefs as gprefsmain
            stats = gprefsmain.get('library_usage_stats', {})
            self.calibre_font = gprefsmain['font']
            del gprefsmain
            tmp = []
            for k,v in iteritems(stats):
                k = k.strip()
                r = v,k
                tmp.append(r)
            #END FOR
            del stats
            tmp.sort(reverse=True)
            items = []
            last_path = prefs['CALIBRESPY_LAST_LIBRARY_SELECTED']
            if  "/" in last_path:
                items.append(last_path)
                items.append("-------------------------")
            for r in tmp:
                v,k = r
                items.append(k)
            #END FOR
            del tmp
            title = "Calibre Library?"
            label = "Library            "
            self.library_path,ok = QInputDialog.getItem(None, title, label, items,0,False)
            if DEBUG: print("Selected:  ", as_unicode(self.library_path))
            if not ok:
                if self.is_cli:
                    self.reject()
                    sys.exit(0)
                else:
                    self.close()
                    return None
            if not self.library_path:
                if self.is_cli:
                    self.reject()
                    sys.exit(0)
                else:
                    self.close()
                    return None
            if not "/" in self.library_path:
                if self.is_cli:
                    self.reject()
                    sys.exit(0)
                else:
                    self.close()
                    return None
            self.library_path_original = self.library_path
            prefs['CALIBRESPY_LAST_LIBRARY_SELECTED']  = self.library_path
            prefs
            del items
        else:
            self.calibre_font = None

        is_network = self.check_for_network_metadata_db()
        if is_network:
            self.session_is_entirely_readonly = True

        self.hide_push_button_customize = False
        self.user_can_save_settings = False
        self.session_is_locked = True
        #--------------------------------------------------
        #--------------------------------------------------
        if DEBUG: self.elapsed_time_message(0,"CalibreSpy: Initializing")
        #--------------------------------------------------
        #--------------------------------------------------
        self.local_prefs = {}
        for k,v in iteritems(prefs):
            if not k in self.local_prefs:
                self.local_prefs[k] = v
        #END FOR
        self.local_prefs["CALIBRESPY_CLI_SUBDIRECTORY"] = prefs["CALIBRESPY_CLI_SUBDIRECTORY"]
        self.original_clidir = prefs["CALIBRESPY_CLI_SUBDIRECTORY"]
        del prefs
        self.filtering_history_available = False
        self.msg = None
        #~ ---------------------------------------------------------------------------
        self.get_metadatadb_local_prefs()
        if self.local_prefs['PREFERRED_OUTPUT_FORMAT'] == 0:
            self.local_prefs['PREFERRED_OUTPUT_FORMAT'] = DEFAULT_BOOK_FORMAT
        #~ ---------------------------------------------------------------------------
        self.check_startup_shutdown_status()
        #~ ---------------------------------------------------------------------------
        is_valid = self.get_calibrespy_icon_files()
        if not is_valid:
            self.session_is_entirely_readonly = True
            self.save_and_exit()
            if self.is_cli:
                self.reject()
                sys.exit(0)
            return
        if self.icon_calibrespy:
            self.setWindowIcon(self.icon_calibrespy)
        self.device_display_size = self.local_prefs['CALIBRESPY_DEVICE_DISPLAY_SIZE']
        self.build_fonts()
        self.build_stylesheets(qtapp=qtapp)
        self.verify_ftp_data_version()
        self.save_local_prefs()
        if self.prefilter:
            self.prefilter_books_control()
        else:
            self.get_table_books_data()
        self.initialize_scroll_area()
        self.create_filtering_widgets()
        self.create_matrix()
        self.load_matrix()
        self.create_bottom_widgets()
        self.finalize_scroll_area()
        self.finalize_matrix()
        self.load_filter_comboboxes()
        self.connect_events()
        self.get_library_name()
        self.build_matrix_context_menu()
        self.build_row_header_context_menu()
        self.build_column_header_context_menu()
        self.apply_filter_icons()
        self.enable_all_signals()
        self.do_final_inits()
        self.resize_dialog_to_preferences()
        self.show()
        #~ ---------------------------------------------------------------------------
        if DEBUG: self.elapsed_time_message(9,"CalibreSpy: Finished Initialization: elapsed ")
        #~ ---------------------------------------------------------------------------
        self.show_special_messages()
        #~ ---------------------------------------------------------------------------
        self.local_prefs['CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'] = 1        #true
        self.local_prefs['CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL'] = 0  #default for this session
        pref1 = 'CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'
        pref2 = 'CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL'
        self.save_local_prefs_specific(pref1,pref2)
        #~ ---------------------------------------------------------------------------
        if self.is_cli:
            self.gc.collect()
        #~ ---------------------------------------------------------------------------
        self.rebuild_row_to_bookid_dict()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# EVENT FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_current_cell_changed(self,currentRow,currentColumn,previousRow,previousColumn):
        self.current_row = currentRow
        self.current_column = currentColumn
        item = self.matrix.item(self.current_row,self.path_column)
        if not item:
            value = "...no formats found..."
        else:
            value = item.text()
        if value > " ":
            n = value.rfind("/")
            value = value[n+1: ]
        else:
            value = "...no formats found..."

        if currentColumn in self.column_number_name_dict:
            label = self.column_number_name_dict[currentColumn]
        else:
            label = "Unknown Column Name"

        value = value + "       [" + label + "]"
        self.change_bottom_message(value)
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_activated_action(self,index):
        if self.action_is_in_progress:
            return
        self.action_is_in_progress  = True
        self.open_current_book()
        self.action_is_in_progress  = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_matrix_customcontextmenurequested_action(self,pos):
        parent=self.sender()
        pos = QPoint(390, 60)  #override pos to force menu top to go to upper-middle of matrix one line down from horizontal heading so the bottom of the context menu is always wholly visible.
        pPos=parent.mapToGlobal(pos)
        self.menu.move(pPos)
        self.menu.show()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_row_header_customcontextmenurequested_action(self,pos):
        parent=self.sender()
        pPos=parent.mapToGlobal(pos)
        self.row_header_menu.move(pPos)
        self.row_header_menu.show()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_column_header_customcontextmenurequested_action(self,pos):
        parent=self.sender()
        pPos=parent.mapToGlobal(pos)
        self.column_header_menu.move(pPos)
        self.column_header_menu.show()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_authors_dropdown_arrow_clicked(self,event):
        self.authors_filter_combobox.showPopup()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_authors_changed(self,event):
        if self.all_signals_blocked:
            return
        if self.authors_signals_blocked:
            return
        self.current_author = self.authors_filter_combobox.currentText()
        self.authors_filter_combobox.hidePopup()
        self.filter_by_authors(source=AUTHORS)
        QApplication.instance().processEvents()
        self.authors_filter_is_applied = True
        self.filter_history_combobox.insertItem(1,AUTHORS_SOURCE + self.current_author)
        self.optimize_filter_history_combobox()
        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_title_dropdown_arrow_clicked(self,event):
        self.title_filter_combobox.showPopup()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_title_changed(self,event):
        if self.all_signals_blocked:
            return
        if self.title_signals_blocked:
            return
        self.current_title = self.title_filter_combobox.currentText()
        self.title_filter_combobox.hidePopup()
        self.filter_by_title(source=TITLE)
        QApplication.instance().processEvents()
        self.title_filter_is_applied = True
        self.filter_history_combobox.insertItem(1,TITLE_SOURCE + self.current_title)
        self.optimize_filter_history_combobox()
        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_series_dropdown_arrow_clicked(self,event):
        self.series_filter_combobox.showPopup()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_series_changed(self,event):
        if self.all_signals_blocked:
            return
        if self.series_signals_blocked:
            return
        self.current_series = self.series_filter_combobox.currentText()
        self.series_filter_combobox.hidePopup()
        self.filter_by_series(source=SERIES)
        QApplication.instance().processEvents()
        self.series_filter_is_applied = True
        self.filter_history_combobox.insertItem(1,SERIES_SOURCE + self.current_series)
        self.optimize_filter_history_combobox()
        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_tags_dropdown_arrow_clicked(self,event):
        self.tags_filter_combobox.showPopup()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_tags_changed(self,event):
        if self.all_signals_blocked:
            return
        if self.tags_signals_blocked:
            return
        self.current_tag = self.tags_filter_combobox.currentText()
        self.current_tag = self.current_tag.strip()
        self.tags_filter_combobox.hidePopup()
        self.filter_by_tags(source=TAGS)
        QApplication.instance().processEvents()
        self.tags_filter_is_applied = True
        self.optimize_filter_history_combobox()
        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_publisher_dropdown_arrow_clicked(self,event):
        self.publisher_filter_combobox.showPopup()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_publisher_changed(self,event):
        if self.all_signals_blocked:
            return
        if self.publisher_signals_blocked:
            return
        self.current_publisher = self.publisher_filter_combobox.currentText()
        self.publisher_filter_combobox.hidePopup()
        self.filter_by_publisher(source=PUBLISHER)
        QApplication.instance().processEvents()
        self.publisher_filter_is_applied = True
        self.filter_history_combobox.insertItem(1,PUBLISHER_SOURCE + self.current_publisher)
        self.optimize_filter_history_combobox()
        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_history_changed(self,event):
        if self.all_signals_blocked:
            return
        if self.filter_history_signals_blocked:
            return
        self.current_history = self.filter_history_combobox.currentText()
        self.filter_history_signals_blocked = True
        self.filter_history_combobox.setCurrentIndex(0)
        self.filter_history_signals_blocked = False
        self.filter_using_history()
        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_dialog_resized(self,event=None):
        self.local_prefs['DIALOG_SIZE_WIDTH_LAST'] = self.size().width()
        self.local_prefs['DIALOG_SIZE_HEIGHT_LAST'] = self.size().height()
#---------------------------------------------------------------------------------------------------------------------------------------
    def resizeEvent(self, event):
        self.resized_signal.emit()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# FILTERING FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def clear_all_filters(self):
        QTimer.singleShot(0,self.clear_all_filters_now)
#---------------------------------------------------------------------------------------------------------------------------------------
    def clear_all_filters_now(self):
        self.all_signals_blocked = True
        self.visible_rows_set.clear()
        for r in range(0,self.n_matrix_rows):
            self.matrix.setRowHidden(r,False)
            self.visible_rows_set.add(r)
        #END FOR
        self.matrix.update()
        self.first_visible_row = 0
        self.last_visible_row = self.n_matrix_rows - 1
        self.current_visible_row_count = self.n_matrix_rows
        self.reset_authors_filter_combobox()
        self.reset_title_filter_combobox()
        self.reset_series_filter_combobox()
        self.reset_tags_filter_combobox()
        self.reset_publisher_filter_combobox()
        self.current_author = DEFAULT_AUTHORS
        self.current_title = DEFAULT_TITLE
        self.current_series = DEFAULT_SERIES
        self.current_tag = DEFAULT_TAGS
        self.current_publisher = DEFAULT_PUBLISHER
        self.authors_filter_combobox.setCurrentIndex(0)
        self.title_filter_combobox.setCurrentIndex(0)
        self.series_filter_combobox.setCurrentIndex(0)
        self.tags_filter_combobox.setCurrentIndex(0)
        self.publisher_filter_combobox.setCurrentIndex(0)
        self.authors_filter_combobox.update()
        self.title_filter_combobox.update()
        self.series_filter_combobox.update()
        self.tags_filter_combobox.update()
        self.publisher_filter_combobox.update()
        self.all_signals_blocked = False
        self.matrix.setCurrentCell(0,0)
        self.authors_filter_is_applied = False
        self.title_filter_is_applied = False
        self.series_filter_is_applied = False
        self.tags_filter_is_applied = False
        self.publisher_filter_is_applied = False
        self.apply_filter_icons()
        self.authors_filtering_is_in_play = False
        self.title_filtering_is_in_play = False
        self.series_filtering_is_in_play = False
        self.tags_filtering_is_in_play = False
        self.publisher_filtering_is_in_play = False
        self.show_filter_combobox_counts()
        self.matrix.setFocus()
        if self.vl_is_active:
            self.vl_is_active = False
            self.push_button_virtual_libraries.setChecked(False)
            self.push_button_virtual_libraries.setDown(False)
            self.push_button_virtual_libraries.setToolTip(self.push_button_virtual_libraries_tooltip)
        QApplication.instance().processEvents()
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_using_history(self):
        if self.current_history.startswith(AUTHORS_SOURCE):
            value = self.current_history.replace(AUTHORS_SOURCE,"").strip()
            self.authors_filter_combobox.setCurrentText(value)
        elif self.current_history.startswith(TITLE_SOURCE):
            value = self.current_history.replace(TITLE_SOURCE,"").strip()
            self.title_filter_combobox.setCurrentText(value)
        elif self.current_history.startswith(SERIES_SOURCE):
            value = self.current_history.replace(SERIES_SOURCE,"").strip()
            self.series_filter_combobox.setCurrentText(value)
        elif self.current_history.startswith(TAGS_SOURCE):
            value = self.current_history.replace(TAGS_SOURCE,"").strip()
            self.tags_filter_combobox.setCurrentText(value)
        elif self.current_history.startswith(PUBLISHER_SOURCE):
            value = self.current_history.replace(PUBLISHER_SOURCE,"").strip()
            self.publisher_filter_combobox.setCurrentText(value)
        elif self.current_history.startswith(LANGUAGES_SOURCE):
            value = self.current_history.replace(LANGUAGES_SOURCE,"").strip()
            self.current_languages = value
            self.context_filter_languages_column_values(source=SOURCE_FILTER_USING_HISTORY)
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_by_authors(self,source=AUTHORS):

        if source == AUTHORS:  #author combobox event
            for r in range(0,self.n_matrix_rows):
                if self.current_author == DEFAULT_AUTHORS:
                    self.matrix.setRowHidden(r,False)
                    continue
                item = self.matrix.item(r,self.authors_column)
                if item is None:
                    authors = ""
                else:
                    authors = item.text()
                if authors != self.current_author:
                    self.matrix.setRowHidden(r,True)
                else:
                    if self.is_authors_filtering_in_play():   #author  combobox clicked second or third or fourth, not first...
                        pass
                    else:
                        self.matrix.setRowHidden(r,False)
            #END FOR
            if self.current_author == DEFAULT_AUTHORS:
                self.reset_authors_filter_combobox()
                self.optimize_authors_filter_combobox()
            else:
                self.authors_signals_blocked = True
                self.authors_filter_combobox.clear()
                self.authors_filter_combobox.addItem(self.current_author)
                self.authors_filter_combobox.setCurrentIndex(0)
                self.authors_filter_combobox.setItemIcon(0,self.icon_filter_applied)
                self.authors_signals_blocked = False
            #END IF
            self.authors_filtering_is_in_play = True
            self.filter_by_title(source=AUTHORS)    #always filter in the basic sequence of:  authors > title > series > tags > publisher and then wrap around if the user starts in the middle...
            self.filter_by_series(source=AUTHORS)
            self.filter_by_tags(source=AUTHORS)
            self.filter_by_publisher(source=AUTHORS)
            self.get_visible_row_count()
        else:  # either TITLE or SERIES or ...
            tmp_authors_list = []
            for r in range(0,self.n_matrix_rows):
                is_hidden = self.matrix.isRowHidden(r)
                if is_hidden:
                    continue
                else:
                    item = self.matrix.item(r,self.authors_column)
                    if item is None:
                        authors = ""
                    else:
                        authors = item.text()
                    if not authors in tmp_authors_list:
                        tmp_authors_list.append(authors)
                #END IF
            #END FOR
            self.authors_signals_blocked = True
            self.authors_filter_combobox.clear()
            tmp_authors_list.sort()
            n = len(tmp_authors_list)
            if n == 0:
                self.authors_filter_combobox.addItem(DEFAULT_AUTHORS_NONE)
            elif n == 1:
                self.authors_filter_combobox.addItem(tmp_authors_list[0])
            elif n == 2:
                if DEFAULT_AUTHORS in tmp_authors_list:
                    for t in tmp_authors_list:
                        if t != DEFAULT_AUTHORS:
                            self.authors_filter_combobox.addItem(t)
                    #END FOR
                else:
                    self.authors_filter_combobox.addItem(DEFAULT_AUTHORS_PLACEHOLDER)
                    for t in tmp_authors_list:
                        if t != DEFAULT_AUTHORS_PLACEHOLDER:
                            self.authors_filter_combobox.addItem(t)
                    #END FOR
            else:
                i = self.authors_filter_combobox.findText(DEFAULT_AUTHORS_PLACEHOLDER)
                if i < 0:
                    self.authors_filter_combobox.addItem(DEFAULT_AUTHORS_PLACEHOLDER)
                for t in tmp_authors_list:
                    self.authors_filter_combobox.addItem(t)
                #END FOR
            #END IF
            del tmp_authors_list
            self.optimize_authors_filter_combobox()
            self.authors_filter_combobox.setCurrentIndex(0)
            self.authors_filter_combobox.setItemIcon(0,self.icon_filter_applied)
            self.authors_filter_combobox.update()
            self.authors_filtering_is_in_play = True
            self.authors_signals_blocked = False
        #END IF

        self.show_filter_combobox_counts()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_by_title(self,source=TITLE):

        if source == TITLE:  #  title combobox event
            for r in range(0,self.n_matrix_rows):
                if self.current_title == DEFAULT_TITLE:
                    self.matrix.setRowHidden(r,False)
                    continue
                item = self.matrix.item(r,self.title_column)
                if item is None:
                    title = ""
                else:
                    title = item.text()
                if title != self.current_title:
                    self.matrix.setRowHidden(r,True)
                else:
                    if self.is_title_filtering_in_play():   #title  combobox clicked second or third or fourth, not first...
                        pass
                    else:
                        self.matrix.setRowHidden(r,False)
            #END FOR
            if self.current_title == DEFAULT_TITLE:
                self.reset_title_filter_combobox()
                self.optimize_title_filter_combobox()
            else:
                self.title_signals_blocked = True
                self.title_filter_combobox.clear()
                self.title_filter_combobox.addItem(self.current_title)
                self.title_filter_combobox.setCurrentIndex(0)
                self.title_filter_combobox.setItemIcon(0,self.icon_filter_applied)
                self.title_filter_combobox.update()
                self.title_signals_blocked = False
            #END IF
            self.title_filtering_is_in_play = True
            self.filter_by_series(source=TITLE)   #always filter in the basic sequence of:  authors > title > series > tags > publisher and then wrap around if the user starts in the middle...
            self.filter_by_tags(source=TITLE)
            self.filter_by_publisher(source=TITLE)
            self.filter_by_authors(source=TITLE)
            self.get_visible_row_count()
        else:  # either AUTHORS or SERIES or ...
            tmp_title_list = []
            for r in range(0,self.n_matrix_rows):
                is_hidden = self.matrix.isRowHidden(r)
                if is_hidden:
                    continue
                else:
                    item = self.matrix.item(r,self.title_column)
                    if item is None:
                        title = ""
                    else:
                        title = item.text()
                    if not title in tmp_title_list:
                        tmp_title_list.append(title)
            #END FOR
            self.title_signals_blocked = True
            self.title_filter_combobox.clear()
            tmp_title_list.sort()
            n = len(tmp_title_list)
            if n == 0:
                self.title_filter_combobox.addItem(DEFAULT_TITLE_NONE)
            elif n == 1:
                self.title_filter_combobox.addItem(tmp_title_list[0])
            elif n == 2:
                if DEFAULT_TITLE in tmp_title_list:
                    for t in tmp_title_list:
                        if t != DEFAULT_TITLE:
                            self.title_filter_combobox.addItem(t)
                    #END FOR
                else:
                    self.title_filter_combobox.addItem(DEFAULT_TITLE_PLACEHOLDER)
                    for t in tmp_title_list:
                        if t != DEFAULT_TITLE_PLACEHOLDER:
                            self.title_filter_combobox.addItem(t)
                    #END FOR
            else:
                i = self.title_filter_combobox.findText(DEFAULT_TITLE_PLACEHOLDER)
                if i < 0:
                    self.title_filter_combobox.addItem(DEFAULT_TITLE_PLACEHOLDER)
                for t in tmp_title_list:
                    self.title_filter_combobox.addItem(t)
                #END FOR
            #ENDIF
            del tmp_title_list
            self.optimize_title_filter_combobox()
            self.title_filter_combobox.setCurrentIndex(0)
            self.title_filter_combobox.setItemIcon(0,self.icon_filter_applied)
            self.title_filter_combobox.update()
            self.title_filtering_is_in_play = True
            self.title_signals_blocked = False
        #END IF

        self.show_filter_combobox_counts()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_by_series(self,source=SERIES):

        if source == SERIES:  #  series combobox event
            for r in range(0,self.n_matrix_rows):
                if self.current_series == DEFAULT_SERIES:
                    self.matrix.setRowHidden(r,False)
                    continue
                item = self.matrix.item(r,self.series_column)
                if item is None:
                    series = ""
                else:
                    series = item.text()
                if series != self.current_series:
                    self.matrix.setRowHidden(r,True)
                else:
                    if self.is_series_filtering_in_play():   #series combobox clicked second or third or fourth, not first...
                        pass
                    else:
                        self.matrix.setRowHidden(r,False)
            #END FOR
            if self.current_series == DEFAULT_SERIES:
                self.reset_series_filter_combobox()
                self.optimize_series_filter_combobox()
            else:
                self.series_signals_blocked = True
                self.series_filter_combobox.clear()
                self.series_filter_combobox.addItem(self.current_series)
                self.series_filter_combobox.setCurrentIndex(0)
                self.series_filter_combobox.setItemIcon(0,self.icon_filter_applied)
                self.series_filter_combobox.update()
                self.series_signals_blocked = False
            #END IF
            self.series_filtering_is_in_play = True
            self.filter_by_tags(source=SERIES)   #always filter in the basic sequence of:  authors > title > series > tags > publisher and then wrap around if the user starts in the middle...
            self.filter_by_publisher(source=SERIES)
            self.filter_by_authors(source=SERIES)
            self.filter_by_title(source=SERIES)
            self.get_visible_row_count()
        else:  # either AUTHORS or TITLE or ...
            tmp_series_list = []
            for r in range(0,self.n_matrix_rows):
                is_hidden = self.matrix.isRowHidden(r)
                if is_hidden:
                    continue
                else:
                    item = self.matrix.item(r,self.series_column)
                    if item is None:
                        series = ""
                    else:
                        series = item.text()
                    if not series in tmp_series_list:
                        tmp_series_list.append(series)
            #END FOR
            self.series_signals_blocked = True
            self.series_filter_combobox.clear()
            tmp_series_list.sort()
            n = len(tmp_series_list)
            if n == 0:
                self.series_filter_combobox.addItem(DEFAULT_SERIES_NONE)
            elif n == 1:
                self.series_filter_combobox.addItem(tmp_series_list[0])
            elif n == 2:
                if DEFAULT_SERIES in tmp_series_list:
                    for t in tmp_series_list:
                        if t != DEFAULT_SERIES:
                            self.series_filter_combobox.addItem(t)
                    #END FOR
                else:
                    self.series_filter_combobox.addItem(DEFAULT_SERIES_PLACEHOLDER)
                    for t in tmp_series_list:
                        if t != DEFAULT_SERIES_PLACEHOLDER:
                            self.series_filter_combobox.addItem(t)
                    #END FOR
            else:
                i = self.series_filter_combobox.findText(DEFAULT_SERIES_PLACEHOLDER)
                if i < 0:
                    self.series_filter_combobox.addItem(DEFAULT_SERIES_PLACEHOLDER)
                for t in tmp_series_list:
                    self.series_filter_combobox.addItem(t)
                #END FOR
            #END IF
            del tmp_series_list
            self.optimize_series_filter_combobox()
            self.series_filter_combobox.setCurrentIndex(0)
            self.series_filter_combobox.setItemIcon(0,self.icon_filter_applied)
            self.series_filter_combobox.update()
            self.series_filtering_is_in_play = True
            self.series_signals_blocked = False
        #END IF
        self.show_filter_combobox_counts()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_by_tags(self,source=TAGS):

        if source == TAGS:  #tags combobox event
            associated_tags_list = []  # Since 1 book can have many Tags, filterdown of 'Associated Tags' is possible. Example: history books have a combination of:  Factual:History; Factual:History-War; Factual:History-War-WWII; etc.
            for r in range(0,self.n_matrix_rows):
                if self.current_tag == DEFAULT_TAGS:
                    self.matrix.setRowHidden(r,False)
                    continue
                if self.matrix.isRowHidden(r):
                    continue  #do not unhide any rows that are already hidden
                item = self.matrix.item(r,self.tags_column)
                if item is None:
                    book_tags = ""
                else:
                    book_tags = item.text() # example: 3 tags in book_tags:  Fiction:History-European,Fiction:War,Fiction:Thriller
                book_tags = book_tags.strip()
                if self.current_tag == DEFAULT_TAGS_NONE:
                    if book_tags > "":
                        self.matrix.setRowHidden(r,True)
                        continue
                    else:
                        continue
                if not self.current_tag in book_tags:    # very quick acid test; tags are multiple and comma delimited, so equality cannot be checked (yet)...
                    self.matrix.setRowHidden(r,True)
                    continue
                else:  #partial matches due to "in" above must be excluded via granular comparison...
                    book_tags = book_tags + ","
                    s_split = book_tags.split(",")
                    del book_tags
                    match_found = False
                    for tag in s_split:
                        tag = tag.strip()
                        if tag == self.current_tag:
                            match_found = True
                            break
                    #END FOR
                    if not match_found:
                        self.matrix.setRowHidden(r,True)
                        continue
                    else:
                        for tag in s_split:
                            tag = tag.strip()
                            if tag > " ":
                                if tag != self.current_tag:
                                    associated_tags_list.append(tag)
                        #END FOR
                        if self.is_tags_filtering_in_play():   #tag combobox clicked second or third or fourth, not first...
                            pass
                        else:
                            self.matrix.setRowHidden(r,False)
            #END FOR
            self.filter_history_combobox.insertItem(1,TAGS_SOURCE + self.current_tag)  #self.current_tag is a "real" Tag, not a placeholder
            if self.current_tag == DEFAULT_TAGS:
                self.reset_tags_filter_combobox()
                self.optimize_tags_filter_combobox()
            else:
                self.tags_signals_blocked = True
                self.tags_filter_combobox.clear()
                self.tags_filter_combobox.addItem(self.current_tag)
                associated_tags_list = list(set(associated_tags_list))
                associated_tags_list.sort()
                for tag in associated_tags_list:
                    self.tags_filter_combobox.addItem(tag)
                #END FOR
                del associated_tags_list
                if self.tags_filter_combobox.count() > 1:
                    i = self.tags_filter_combobox.findText(DEFAULT_TAGS_PLACEHOLDER)
                    if i < 0:
                        self.tags_filter_combobox.insertItem(0,DEFAULT_TAGS_PLACEHOLDER)
                        self.tags_filter_combobox.setCurrentIndex(0)
                    else:
                        self.tags_filter_combobox.setCurrentIndex(i)
                else:
                    self.tags_filter_combobox.setCurrentIndex(0)
                self.current_tag = self.tags_filter_combobox.currentText()
                self.tags_filter_combobox.setItemIcon(0,self.icon_filter_applied)
                self.tags_filter_combobox.update()
                self.tags_signals_blocked = False
            #END IF
            self.tags_filtering_is_in_play = True
            self.filter_by_authors(source=TAGS)   #always filter in the basic sequence of:  authors > title > series > tags > publisher and then wrap around if the user starts in the middle...
            self.filter_by_title(source=TAGS)
            self.filter_by_series(source=TAGS)
            self.filter_by_publisher(source=TAGS)
            self.get_visible_row_count()
        else:  # either AUTHORS or TITLE or ...
            tmp_tags_list = []
            for r in range(0,self.n_matrix_rows):
                is_hidden = self.matrix.isRowHidden(r)
                if is_hidden:
                    continue
                else:
                    item = self.matrix.item(r,self.tags_column)
                    if item is None:
                        tags = ""
                    else:
                        tags = item.text()
                    tags = tags.strip()
                    if not tags in tmp_tags_list:
                        tmp_tags_list.append(tags)
                #END IF
            #END FOR
            self.tags_signals_blocked = True
            tmp_tags_split_list = []
            for tag in tmp_tags_list:
                origtag = tag
                tag = tag + ","
                s_split = tag.split(",")
                for t in s_split:
                    t = t.strip()
                    if origtag == "" or t > "":     #add no artifacts caused by s_split...
                        tmp_tags_split_list.append(t)
                #END FOR
                del s_split
            #END FOR
            del tmp_tags_list
            tmp_tags_split_list = list(set(tmp_tags_split_list))
            tmp_tags_split_list.sort()
            self.tags_filter_combobox.clear()
            n = len(tmp_tags_split_list)
            if n == 0:
                self.tags_filter_combobox.addItem(DEFAULT_TAGS_NONE)
            elif n == 1:
                self.tags_filter_combobox.addItem(tmp_tags_split_list[0])
            elif n == 2:
                if DEFAULT_TAGS in tmp_tags_split_list:
                    for t in tmp_tags_split_list:
                        if t != DEFAULT_TAGS:
                            self.tags_filter_combobox.addItem(t)
                    #END FOR
                else:
                    self.tags_filter_combobox.addItem(DEFAULT_TAGS_PLACEHOLDER)
                    for t in tmp_tags_split_list:
                        if t != DEFAULT_TAGS_PLACEHOLDER:
                            self.tags_filter_combobox.addItem(t)
                    #END FOR
            else:
                i = self.tags_filter_combobox.findText(DEFAULT_TAGS_PLACEHOLDER)
                if i < 0:
                    self.tags_filter_combobox.addItem(DEFAULT_TAGS_PLACEHOLDER)
                for t in tmp_tags_split_list:
                    if t != DEFAULT_TAGS:
                        self.tags_filter_combobox.addItem(t)
                #END FOR
            #END IF
            del tmp_tags_split_list
            self.optimize_tags_filter_combobox()
            self.tags_filter_combobox.setCurrentIndex(0)
            self.tags_filter_combobox.setItemIcon(0,self.icon_filter_applied)
            self.tags_filter_combobox.update()
            self.tags_filtering_is_in_play = True
            self.tags_signals_blocked = False
        #END IF

        self.show_filter_combobox_counts()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_by_publisher(self,source=PUBLISHER):

        if source == PUBLISHER:  #  publisher combobox event
            for r in range(0,self.n_matrix_rows):
                if self.current_publisher == DEFAULT_PUBLISHER:
                    self.matrix.setRowHidden(r,False)
                    continue
                item = self.matrix.item(r,self.publisher_column)
                if item is None:
                    publisher = ""
                else:
                    publisher = item.text()
                if publisher != self.current_publisher:
                    self.matrix.setRowHidden(r,True)
                else:
                    if self.is_publisher_filtering_in_play():   #publisher combobox clicked second or third or fourth, not first...
                        pass
                    else:
                        self.matrix.setRowHidden(r,False)
            #END FOR
            if self.current_publisher == DEFAULT_PUBLISHER:
                self.reset_publisher_filter_combobox()
                self.optimize_publisher_filter_combobox()
            else:
                self.publisher_signals_blocked = True
                self.publisher_filter_combobox.clear()
                self.publisher_filter_combobox.addItem(self.current_publisher)
                self.publisher_filter_combobox.setCurrentIndex(0)
                self.publisher_filter_combobox.setItemIcon(0,self.icon_filter_applied)
                self.publisher_filter_combobox.update()
                self.publisher_signals_blocked = False
            #END IF
            self.publisher_filtering_is_in_play = True
            self.filter_by_authors(source=PUBLISHER)  #always filter in the basic sequence of:  authors > title > series > tags > publisher and then wrap around if the user starts in the middle...
            self.filter_by_title(source=PUBLISHER)
            self.filter_by_series(source=PUBLISHER)
            self.filter_by_tags(source=PUBLISHER)
            self.get_visible_row_count()
        else:  # either AUTHORS or TITLE or ...
            tmp_publisher_list = []
            for r in range(0,self.n_matrix_rows):
                is_hidden = self.matrix.isRowHidden(r)
                if is_hidden:
                    continue
                else:
                    item = self.matrix.item(r,self.publisher_column)
                    if item:
                        publisher = item.text()
                        if not publisher in tmp_publisher_list:
                            tmp_publisher_list.append(publisher)
                #END FOR
            self.publisher_signals_blocked = True
            self.publisher_filter_combobox.clear()
            tmp_publisher_list.sort()
            n = len(tmp_publisher_list)
            if n == 0:
                self.publisher_filter_combobox.addItem(DEFAULT_PUBLISHER_NONE)
            elif n == 1:
                self.publisher_filter_combobox.addItem(tmp_publisher_list[0])
            elif n == 2:
                if DEFAULT_PUBLISHER in tmp_publisher_list:
                    for t in tmp_publisher_list:
                        if t != DEFAULT_PUBLISHER:
                            self.publisher_filter_combobox.addItem(t)
                    #END FOR
                else:
                    self.publisher_filter_combobox.addItem(DEFAULT_PUBLISHER_PLACEHOLDER)
                    for t in tmp_publisher_list:
                        if t != DEFAULT_PUBLISHER_PLACEHOLDER:
                            self.publisher_filter_combobox.addItem(t)
                    #END FOR
            else:
                i = self.title_filter_combobox.findText(DEFAULT_PUBLISHER_PLACEHOLDER)
                if i < 0:
                    self.publisher_filter_combobox.addItem(DEFAULT_PUBLISHER_PLACEHOLDER)
                for t in tmp_publisher_list:
                    self.publisher_filter_combobox.addItem(t)
                #END FOR
            #END IF
            del tmp_publisher_list
            self.optimize_publisher_filter_combobox()
            self.publisher_filter_combobox.setCurrentIndex(0)
            self.publisher_filter_combobox.setItemIcon(0,self.icon_filter_applied)
            self.publisher_filter_combobox.update()
            self.publisher_filtering_is_in_play = True
            self.publisher_signals_blocked = False
        #END IF

        self.show_filter_combobox_counts()
#---------------------------------------------------------------------------------------------------------------------------------------
    def reset_authors_filter_combobox(self):
        self.authors_signals_blocked = True
        self.authors_filter_combobox.clear()
        for row in self.authors_filter_combobox_full_list:
            self.authors_filter_combobox.addItem(row)
        #END FOR
        self.authors_filter_combobox.setCurrentIndex(0)
        self.authors_filter_combobox.update()
        self.authors_signals_blocked = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def reset_title_filter_combobox(self):
        self.title_signals_blocked = True
        self.title_filter_combobox.clear()
        for row in self.title_filter_combobox_full_list:
            self.title_filter_combobox.addItem(row)
        #END FOR
        self.title_filter_combobox.setCurrentIndex(0)
        self.title_filter_combobox.update()
        self.title_signals_blocked = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def reset_series_filter_combobox(self):
        self.series_signals_blocked = True
        self.series_filter_combobox.clear()
        for row in self.series_filter_combobox_full_list:
            self.series_filter_combobox.addItem(row)
        #END FOR
        self.series_filter_combobox.setCurrentIndex(0)
        self.series_filter_combobox.update()
        self.series_signals_blocked = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def reset_tags_filter_combobox(self):
        self.tags_signals_blocked = True
        self.tags_filter_combobox.clear()
        for row in self.tags_filter_combobox_full_list:
            self.tags_filter_combobox.addItem(row)
        #END FOR
        self.tags_filter_combobox.setCurrentIndex(0)
        self.tags_filter_combobox.update()
        self.tags_signals_blocked = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def reset_publisher_filter_combobox(self):
        self.publisher_signals_blocked = True
        self.publisher_filter_combobox.clear()
        for row in self.publisher_filter_combobox_full_list:
            self.publisher_filter_combobox.addItem(row)
        #END FOR
        self.publisher_filter_combobox.setCurrentIndex(0)
        self.publisher_filter_combobox.update()
        self.publisher_signals_blocked = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_authors_filter_combobox(self):
        #authors may never be blank in Calibre
        n = self.authors_filter_combobox.count()
        if n == 2:
            i0 = self.authors_filter_combobox.findText(DEFAULT_AUTHORS)
            i1 = self.authors_filter_combobox.findText(DEFAULT_AUTHORS_PLACEHOLDER)
            if i0 >= 0:
                self.authors_filter_combobox.removeItem(i0)
            if i1 >= 0:
                self.authors_filter_combobox.removeItem(i0)
            self.authors_filter_combobox.setCurrentIndex(0)
        elif n > 2:
            i = self.authors_filter_combobox.findText(DEFAULT_AUTHORS)
            if i >=0:
                self.authors_filter_combobox.removeItem(i)
            i = self.authors_filter_combobox.findText(DEFAULT_AUTHORS_PLACEHOLDER)
            if i >= 0:
                self.authors_filter_combobox.removeItem(i)
            i = self.authors_filter_combobox.findText(DEFAULT_AUTHORS_PLACEHOLDER)
            if i < 0:
                self.authors_filter_combobox.insertItem(0,DEFAULT_AUTHORS_PLACEHOLDER)
            self.authors_filter_combobox.setCurrentIndex(0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_title_filter_combobox(self):
        #titles may never be blank in Calibre
        n = self.title_filter_combobox.count()
        if n == 2:
            i0 = self.title_filter_combobox.findText(DEFAULT_TITLE)
            i1 = self.title_filter_combobox.findText(DEFAULT_TITLE_PLACEHOLDER)
            if i0 >= 0:
                self.title_filter_combobox.removeItem(i0)
            if i1 >= 0:
                self.title_filter_combobox.removeItem(i1)
            self.title_filter_combobox.setCurrentIndex(0)
        elif n > 2:
            i = self.title_filter_combobox.findText(DEFAULT_TITLE)
            if i >=0:
                self.title_filter_combobox.removeItem(i)
            i = self.title_filter_combobox.findText(DEFAULT_TITLE_PLACEHOLDER)
            if i >= 0:
                self.title_filter_combobox.removeItem(i)
            i = self.title_filter_combobox.findText(DEFAULT_TITLE_PLACEHOLDER)
            if i < 0:
                self.title_filter_combobox.insertItem(0,DEFAULT_TITLE_PLACEHOLDER)
            self.title_filter_combobox.setCurrentIndex(0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_series_filter_combobox(self):
        n = self.series_filter_combobox.count()
        if n == 1:
            i = self.series_filter_combobox.findText(DEFAULT_SERIES)
            if i == 0:
                self.series_filter_combobox.removeItem(i)
                self.series_filter_combobox.setItemText(0,DEFAULT_SERIES_PLACEHOLDER)
                self.series_filter_combobox.setCurrentIndex(0)
        elif n == 2:
            i = self.series_filter_combobox.findText(DEFAULT_SERIES)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.series_filter_combobox.removeItem(i)
                self.series_filter_combobox.setCurrentIndex(0)
            else:
                self.series_filter_combobox.insertItem(0,DEFAULT_SERIES_PLACEHOLDER)
                self.series_filter_combobox.setCurrentIndex(0)
        elif n > 2:
            i = self.series_filter_combobox.findText(DEFAULT_SERIES)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.series_filter_combobox.removeItem(i)
            i = self.series_filter_combobox.findText(DEFAULT_SERIES_PLACEHOLDER)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.series_filter_combobox.removeItem(i)
            i = self.series_filter_combobox.findText(DEFAULT_SERIES_PLACEHOLDER)
            if i < 0:
                self.series_filter_combobox.insertItem(0,DEFAULT_SERIES_PLACEHOLDER)
            self.series_filter_combobox.setCurrentIndex(0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_tags_filter_combobox(self):
        n = self.tags_filter_combobox.count()
        if n == 1:
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS)
            if i == 0:
                self.tags_filter_combobox.removeItem(i)
                self.tags_filter_combobox.setItemText(0,DEFAULT_TAGS_PLACEHOLDER)
                self.tags_filter_combobox.setCurrentIndex(0)
        elif n == 2:
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.tags_filter_combobox.removeItem(i)
                self.tags_filter_combobox.setCurrentIndex(0)
            else:
                self.tags_filter_combobox.insertItem(0,DEFAULT_TAGS_PLACEHOLDER)
                self.tags_filter_combobox.setCurrentIndex(0)
        elif n > 2:
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS)
            #~ if i >=0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.tags_filter_combobox.removeItem(i)
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS_PLACEHOLDER)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.tags_filter_combobox.removeItem(i)
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS_PLACEHOLDER)
            if i < 0:
                self.tags_filter_combobox.insertItem(0,DEFAULT_TAGS_PLACEHOLDER)
            self.tags_filter_combobox.setCurrentIndex(0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_publisher_filter_combobox(self):
        n = self.publisher_filter_combobox.count()
        if n == 1:
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER)
            if i == 0:
                self.publisher_filter_combobox.removeItem(i)
                self.publisher_filter_combobox.setItemText(0,DEFAULT_PUBLISHER_PLACEHOLDER)
                self.publisher_filter_combobox.setCurrentIndex(0)
        elif n == 2:
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.publisher_filter_combobox.removeItem(i)
                self.publisher_filter_combobox.setCurrentIndex(0)
            else:
                self.publisher_filter_combobox.insertItem(0,DEFAULT_PUBLISHER_PLACEHOLDER)
                self.publisher_filter_combobox.setCurrentIndex(0)
        elif n > 2:
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.publisher_filter_combobox.removeItem(i)
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER_PLACEHOLDER)
            #~ if i >= 0:  Qt in Calibre 4+ fails silently if i = 0 regardless of what the documentation says...
            if i > 0:
                self.publisher_filter_combobox.removeItem(i)
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER_PLACEHOLDER)
            if i < 0:
                self.publisher_filter_combobox.insertItem(0,DEFAULT_PUBLISHER_PLACEHOLDER)
            self.publisher_filter_combobox.setCurrentIndex(0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_filter_history_combobox(self):
        #remove previous search duplicates of the very newest value (index=1); index=0 always the name of the combobox itself.
        newest = self.filter_history_combobox.itemText(1)
        for i in range(2,self.filter_history_combobox.count()):
            t = self.filter_history_combobox.itemText(i)
            if t == newest:
                 self.filter_history_combobox.removeItem(i)
        #END FOR
        self.filter_history_combobox.update()
#---------------------------------------------------------------------------------------------------------------------------------------
    def apply_filter_icons(self):
        self.authors_filter_combobox.setItemIcon(0,self.icon_filter)
        self.title_filter_combobox.setItemIcon(0,self.icon_filter)
        self.series_filter_combobox.setItemIcon(0,self.icon_filter)
        self.tags_filter_combobox.setItemIcon(0,self.icon_filter)
        self.publisher_filter_combobox.setItemIcon(0,self.icon_filter)
        self.push_button_clear_all_filters.setIcon(self.icon_clear_filters)
        self.filter_history_combobox.setItemIcon(0,self.icon_filter)
#---------------------------------------------------------------------------------------------------------------------------------------
    def is_authors_filtering_in_play(self):
        if self.authors_filtering_is_in_play == True:
            return True
        else:
            return False
#---------------------------------------------------------------------------------------------------------------------------------------
    def is_title_filtering_in_play(self):
        if self.title_filtering_is_in_play == True:
            return True
        else:
            return False
#---------------------------------------------------------------------------------------------------------------------------------------
    def is_series_filtering_in_play(self):
        if self.series_filtering_is_in_play == True:
            return True
        else:
            return False
#---------------------------------------------------------------------------------------------------------------------------------------
    def is_tags_filtering_in_play(self):
        if self.tags_filtering_is_in_play == True:
            return True
        else:
            return False
#---------------------------------------------------------------------------------------------------------------------------------------
    def is_publisher_filtering_in_play(self):
        if self.publisher_filtering_is_in_play == True:
            return True
        else:
            return False
#---------------------------------------------------------------------------------------------------------------------------------------
    def invert_selection(self):
        for r in range(0,self.n_matrix_rows):
            if self.matrix.isRowHidden(r):
                self.matrix.setRowHidden(r,False)
            else:
                self.matrix.setRowHidden(r,True)
        #END FOR

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_INVERT_SELECTION)
        self.filter_by_title(source=SOURCE_INVERT_SELECTION)
        self.filter_by_series(source=SOURCE_INVERT_SELECTION)
        self.filter_by_tags(source=SOURCE_INVERT_SELECTION)
        self.filter_by_publisher(source=SOURCE_INVERT_SELECTION)

        self.matrix.setFocus()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# VIRTUAL LIBRARY FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def apply_virtual_libraries(self):

        if self.vl_is_active:
            self.push_button_virtual_libraries.setChecked(False)
            self.push_button_virtual_libraries.setDown(False)
            self.push_button_virtual_libraries.setToolTip(self.push_button_virtual_libraries_tooltip)
            self.vl_is_active = False
            self.clear_all_filters()
            return

        if self.virtual_libraries_dict is None:
            import json
            from calibre.utils.config import from_json

            from calibre_plugins.calibrespy.virtual_library_utils import (vl_search_criteria_parser_control,
                                                                                                            vl_search_criteria_parser_simple,
                                                                                                            vl_search_criteria_parser_complex,
                                                                                                            vl_search_criteria_parser_implied_and_handler,
                                                                                                            vl_search_criteria_parser_user_categories,
                                                                                                            vl_search_criteria_parser_saved_searches,
                                                                                                            vl_search_criteria_parser_another_vl,
                                                                                                            apply_vl_control,
                                                                                                            finalize_vl_sql,
                                                                                                            execute_vl_sql,
                                                                                                            vl_apsw_create_user_functions,
                                                                                                            vl_build_sql_dict_for_term,
                                                                                                            vl_build_target_table_names,
                                                                                                            vl_convert_term_to_target_value_condition,
                                                                                                            vl_parse_identifiers_term,
                                                                                                            vl_parse_tag_hierarchy_term,
                                                                                                            vl_parse_data_term,
                                                                                                            vl_parse_books_timestamp,
                                                                                                            vl_parse_counts,
                                                                                                            convert_vl_terms_to_sql,
                                                                                                            convert_vl_term_to_sql_snippet,
                                                                                                            vl_get_custom_column_technical_details,
                                                                                                            vl_build_standard_column_names_dict)

            self.vl_search_criteria_parser_control = vl_search_criteria_parser_control
            self.vl_search_criteria_parser_simple = vl_search_criteria_parser_simple
            self.vl_search_criteria_parser_complex = vl_search_criteria_parser_complex
            self.vl_search_criteria_parser_implied_and_handler = vl_search_criteria_parser_implied_and_handler
            self.vl_search_criteria_parser_user_categories = vl_search_criteria_parser_user_categories
            self.vl_search_criteria_parser_saved_searches = vl_search_criteria_parser_saved_searches
            self.vl_search_criteria_parser_another_vl = vl_search_criteria_parser_another_vl
            self.apply_vl_control = apply_vl_control
            self.finalize_vl_sql = finalize_vl_sql
            self.execute_vl_sql = execute_vl_sql
            self.vl_apsw_create_user_functions = vl_apsw_create_user_functions
            self.vl_build_sql_dict_for_term = vl_build_sql_dict_for_term
            self.vl_build_target_table_names = vl_build_target_table_names
            self.vl_convert_term_to_target_value_condition = vl_convert_term_to_target_value_condition
            self.vl_parse_identifiers_term = vl_parse_identifiers_term
            self.vl_parse_tag_hierarchy_term = vl_parse_tag_hierarchy_term
            self.vl_parse_data_term = vl_parse_data_term
            self.vl_parse_books_timestamp = vl_parse_books_timestamp
            self.vl_parse_counts = vl_parse_counts
            self.convert_vl_terms_to_sql = convert_vl_terms_to_sql
            self.convert_vl_term_to_sql_snippet = convert_vl_term_to_sql_snippet
            self.vl_get_custom_column_technical_details = vl_get_custom_column_technical_details
            self.vl_build_standard_column_names_dict = vl_build_standard_column_names_dict

            del vl_search_criteria_parser_control
            del vl_search_criteria_parser_simple
            del vl_search_criteria_parser_complex
            del vl_search_criteria_parser_implied_and_handler
            del vl_search_criteria_parser_user_categories
            del vl_search_criteria_parser_saved_searches
            del vl_search_criteria_parser_another_vl
            del apply_vl_control
            del finalize_vl_sql
            del execute_vl_sql
            del vl_apsw_create_user_functions
            del vl_build_sql_dict_for_term
            del vl_build_target_table_names
            del vl_convert_term_to_target_value_condition
            del vl_parse_identifiers_term
            del vl_parse_tag_hierarchy_term
            del vl_parse_data_term
            del vl_parse_books_timestamp
            del vl_parse_counts
            del convert_vl_terms_to_sql
            del convert_vl_term_to_sql_snippet
            del vl_get_custom_column_technical_details

            self.virtual_libraries_dict = {}
            self.virtual_libraries_list = []

            my_db,my_cursor,is_valid = self.apsw_connect_to_library()
            if not is_valid:
                 return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)
            mysql = "SELECT key,val FROM preferences WHERE key = 'virtual_libraries'   "
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            my_db.close()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                key,raw = row
                if not isinstance(raw, unicode_type):
                    raw = raw.decode(preferred_encoding)
                val = json.loads(raw, object_hook=from_json)
                #~ for k,v in val.iteritems():
                for k,v in iteritems(val):
                    unsupported = False
                    for c in self.unsupported_vl_terms_list:
                        if c in v:
                            if DEBUG: print("Unsupported VL: ", k, "   reason: ", c)
                            unsupported = True
                            break
                    #END FOR
                    if unsupported:
                        continue
                    unsupported = False
                    for c in self.unsupported_vl_terms_startswith_list:
                        if v.startswith(c):
                            if DEBUG: print("Unsupported VL: ", k, "   reason: ", c)
                            unsupported = True
                            break
                    #END FOR
                    if unsupported:
                        continue
                    if not ":" in v:
                        if DEBUG: print("Unsupported VL: ", k, "   reason:  No Colons -  ':'  ")
                        continue
                    if v.count("@") > 1:
                        if DEBUG: print("Unsupported VL: ", k, "   reason:  Only 1 @UserCategory criterion is allowed.")
                        continue
                    if v.count("search:") > 1:
                        if DEBUG: print("Unsupported VL: ", k, "   reason:  Only 1 Saved Search 'search:' criterion is allowed.")
                        continue
                    if ("@" in v) and ("search:" in v):
                        if DEBUG: print("Unsupported VL: ", k, "   reason:  Both '@' and 'search:' used in the same VL  ")
                        continue
                    self.virtual_libraries_dict[k] = v
                    self.virtual_libraries_list.append(k)
                    del k
                    del v
                #END FOR
                del key
                del raw
            #END FOR
            del tmp_rows
            del json
            del from_json

            self.virtual_libraries_list.sort()

            self.vl_build_standard_column_names_dict(self,DEBUG)

            if self.is_cli:
                self.gc.collect()

            self.datetime = datetime
            self.timedelta = timedelta

        else:
            pass

        if self.regex is None:
            import re
            self.regex = re
            del re

        title = "CalibreSpy: VLs"
        label = "Choose a Virtual Library            "
        self.current_vl_name,ok = QInputDialog.getItem(None, title, label, self.virtual_libraries_list,0,False)
        if DEBUG: print("------------------------------------------------------")
        if DEBUG: print("VL Selected:  ", self.current_vl_name)
        if not ok:
            self.current_vl_name = None
            self.current_vl_criteria = None
            self.push_button_virtual_libraries.setChecked(False)
            self.push_button_virtual_libraries.setDown(False)
            self.push_button_virtual_libraries.setToolTip(self.push_button_virtual_libraries_tooltip)
            self.vl_is_active = False
            return

        self.vl_is_active = True

        msg = "VL Selected:     " + self.current_vl_name
        self.change_bottom_message(msg)

        self.now = datetime.now()

        self.sql_dict = {}  #term = snippet

        self.current_vl_criteria = self.virtual_libraries_dict[self.current_vl_name]
        parsed_terms_list,is_valid,msg,type = self.vl_search_criteria_parser_control(self,DEBUG,self.current_vl_criteria)
        if not is_valid:
            msg2 = "VL Selected:     " + self.current_vl_name
            self.change_bottom_message(msg2)
            self.push_button_virtual_libraries.setChecked(False)
            self.push_button_virtual_libraries.setDown(False)
            self.push_button_virtual_libraries.setToolTip(self.push_button_virtual_libraries_tooltip)
            self.vl_is_active = False
            error_dialog(None, _('Apply Virtual Library'),_(msg), show=True)
        else:
            book_id_list,is_valid,msg = self.apply_vl_control(self,DEBUG,parsed_terms_list,type)
            if not is_valid:
                msg = "VL Selected:     " + self.current_vl_name
                self.change_bottom_message(msg)
                self.push_button_virtual_libraries.setChecked(False)
                self.push_button_virtual_libraries.setDown(False)
                self.push_button_virtual_libraries.setToolTip(self.push_button_virtual_libraries_tooltip)
                self.vl_is_active = False
                error_dialog(None, _(self.current_vl_name),_(msg), show=True)
            else:
                if DEBUG:  print("total number of books found: ", as_unicode(len(book_id_list)), "  for VL criteria: " + self.current_vl_criteria )
                book_id_set = set(book_id_list)
                del book_id_list
                self.display_sqlquery_search_results(book_id_set)
                t = "Current VL Name: <br><br>" + self.current_vl_name + \
                     "<br><br>Current VL Criteria: <br><br>" + self.current_vl_criteria + \
                     "<br><br>VL Books Originally Found:  " + as_unicode(len(book_id_set))
                self.push_button_virtual_libraries.setToolTip(t)
                del book_id_set

        self.matrix.setFocus()
        if self.first_visible_row is not None:
            if isinstance(self.first_visible_row,int):
                self.matrix.setCurrentCell(self.first_visible_row,0)

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def vl_apsw_user_function_regexp(self,regexpr,avalue):
        #http://www.sqlite.org/lang_expr.html:  The "X REGEXP Y" operator will be implemented as a call to "regexp(Y,X)"
        #---------------------------------------------------------------------------------------------------------------------------------------
        #mysql = 'SELECT id FROM custom_column_8 WHERE value REGEXP '^.+$'
        #---------------------------------------------------------------------------------------------------------------------------------------
        if regexpr:
            if avalue:
                try:
                    s_string = unicode_type(avalue)
                    re_string = unicode_type(regexpr)
                    self.regex.escape("\\")
                    p = self.regex.compile(re_string, self.regex.IGNORECASE|self.regex.DOTALL|self.regex.MULTILINE)
                    match = p.search(s_string)
                    if match:
                        del match
                        del p
                        return True
                    else:
                        del match
                        del p
                        return False
                except Exception as e:
                    print(as_unicode(e))
                    return False
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# OTHER ROUTINE FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_visible_row_count(self):
        QTimer.singleShot(0, self.count_visible_rows)                 # milliseconds
#---------------------------------------------------------------------------------------------------------------------------------------
    def count_visible_rows(self,source=None):
        self.visible_rows_set.clear()
        c = 0
        self.first_visible_row = None
        self.last_visible_row = 0
        for r in range(0,self.n_matrix_rows):
            if not self.matrix.isRowHidden(r):
                c = c + 1
                if self.first_visible_row is None:
                    self.first_visible_row = r
                self.last_visible_row = r
                self.visible_rows_set.add(r)
        #END FOR
        self.current_visible_row_count = c
        if not source:
            msg = "Filtered Rows: " + as_unicode(c)
            self.change_bottom_message(msg)
#---------------------------------------------------------------------------------------------------------------------------------------
    def show_filter_combobox_counts(self):
        self.count_visible_rows()
        if self.current_visible_row_count == 0:
            n = 0
        else:
            n = self.authors_filter_combobox.count()
            i = self.authors_filter_combobox.findText(DEFAULT_AUTHORS)
            if i == 0:
                n = n - 1
            i = self.authors_filter_combobox.findText(DEFAULT_AUTHORS_PLACEHOLDER)
            if i == 0:
                n = n - 1
            if n < 0:
                n = 0
        #ENDIF
        n = "<i>Authors:<b> " + as_unicode(n)
        self.authors_count_qlabel.setText(n)

        if self.current_visible_row_count == 0:
            n = 0
        else:
            n = self.title_filter_combobox.count()
            i = self.title_filter_combobox.findText(DEFAULT_TITLE)
            if i == 0:
                n = n - 1
            i = self.title_filter_combobox.findText(DEFAULT_TITLE_PLACEHOLDER)
            if i == 0:
                n = n - 1
            if n < 0:
                n = 0
        #ENDIF
        n = "<i>Titles:<b> " + as_unicode(n)
        self.title_count_qlabel.setText(n)

        if self.current_visible_row_count == 0:
            n = 0
        else:
            n = self.series_filter_combobox.count()
            i = self.series_filter_combobox.findText(DEFAULT_SERIES)
            if i == 0:
                n = n - 1
            i = self.series_filter_combobox.findText(DEFAULT_SERIES_PLACEHOLDER)
            if i == 0:
                n = n - 1
            i = self.series_filter_combobox.findText(DEFAULT_SERIES_NONE)
            if i == 0:
                n = n - 1
            if n < 0:
                n = 0
        #ENDIF
        n = "<i>Series:<b> " + as_unicode(n)
        self.series_count_qlabel.setText(n)

        if self.current_visible_row_count == 0:
            n = 0
        else:
            n = self.tags_filter_combobox.count()
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS)
            if i == 0:
                n = n - 1
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS_PLACEHOLDER)
            if i == 0:
                n = n - 1
            i = self.tags_filter_combobox.findText(DEFAULT_TAGS_NONE)
            if i == 0:
                n = n - 1
            if n < 0:
                n = 0
        #ENDIF
        n = "<i>Tags:<b> " + as_unicode(n)
        self.tags_count_qlabel.setText(n)

        if self.current_visible_row_count == 0:
            n = 0
        else:
            n = self.publisher_filter_combobox.count()
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER)
            if i == 0:
                n = n - 1
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER_PLACEHOLDER)
            if i == 0:
                n = n - 1
            i = self.publisher_filter_combobox.findText(DEFAULT_PUBLISHER_NONE)
            if i == 0:
                n = n - 1
            if n < 0:
                n = 0
        #ENDIF
        n = "<i>Publishers:<b> " + as_unicode(n)
        self.publisher_count_qlabel.setText(n)
#---------------------------------------------------------------------------------------------------------------------------------------
    def push_button_event_catcher(self):
        return
#---------------------------------------------------------------------------------------------------------------------------------------
    def change_bottom_message(self,msg):
        self.bottom_box_qlabel.setText(msg)
#---------------------------------------------------------------------------------------------------------------------------------------
    def resize_all_columns(self):
        self.matrix.resizeColumnsToContents()
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_column_widths(self):
        self.matrix.resizeColumnsToContents()
        self.matrix.setCurrentCell(self.current_row,0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def deoptimize_column_widths(self):
        self.matrix.setColumnWidth(self.authors_column, self.column_fixed_width_dict[self.authors_column])
        self.matrix.setColumnWidth(self.title_column, self.column_fixed_width_dict[self.title_column])
        self.matrix.setColumnWidth(self.series_column, self.column_fixed_width_dict[self.series_column])
        self.matrix.resizeColumnToContents(self.index_column)
        self.matrix.setColumnWidth(self.tags_column, self.column_fixed_width_dict[self.tags_column])
        self.matrix.setColumnWidth(self.publisher_column, self.column_fixed_width_dict[self.publisher_column])
        self.matrix.resizeColumnToContents(self.languages_column)
        self.matrix.resizeColumnToContents(self.published_column)
        self.matrix.resizeColumnToContents(self.added_column)
        self.matrix.resizeColumnToContents(self.modified_column)
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.path_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc1_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc2_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc3_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc4_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc5_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc6_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc7_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc8_column])
        self.matrix.setColumnWidth(self.path_column, self.column_fixed_width_dict[self.cc9_column])
        self.matrix.setCurrentCell(self.current_row,0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def optimize_current_column_width(self):
        current_column = self.matrix.currentColumn()
        self.matrix.resizeColumnToContents(current_column)
#---------------------------------------------------------------------------------------------------------------------------------------
    def deoptimize_current_column_width(self):
        current_column = self.matrix.currentColumn()
        n = self.column_fixed_width_dict[current_column]
        self.matrix.setColumnWidth(current_column,n)
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_matrix_header_state(self):
        if self.session_is_entirely_readonly:
            return
        if self.multiuser_is_in_play:  #do nothing
            return
        if not self.user_can_save_settings:
            return

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return
        my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _calibrespy_settings (prefkey,prefvalue,prefblob) VALUES (?,?,?)"
        my_cursor.execute(mysql,('CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE','this is a bytearray blob',(self.matrix.horizontalHeader().saveState())  ) )  # see restore_matrix_header_state for comments about this syntax.
        my_cursor.execute("commit")
        my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def restore_matrix_header_state(self):
        if self.session_is_entirely_readonly:
            return
        if self.multiuser_is_in_play:  #do nothing
            return #~ Multiple Users; matrix 'state' will not be restored for this user, since the other user may have invalidated the previously saved 'state'.

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return

        try:
            #~ MUST use this apsw execution iterator method so apsw dumps the blob almost directly into the Qt restoreState() function without unpacking intermediate tuples that cannot handle bytearrays.
            for x,y,z in my_cursor.execute("SELECT prefkey,prefvalue,prefblob FROM _calibrespy_settings WHERE prefkey = 'CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE' "):
                is_valid = self.matrix.horizontalHeader().restoreState(bytearray(z))
            #END FOR
        except:
            pass

        my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def purge_matrix_header_state(self):
        #~ must purge the header state if the columns have just been rearranged in customizing.
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return
        my_cursor.execute("begin")
        mysql = "DELETE FROM _calibrespy_settings WHERE prefkey = 'CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE'   "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_and_exit(self):

        try:
            self.dsr_dialog.close()
        except:
            pass
        try:
            self.calibrespysqlquery.close()
        except:
            pass
        try:
            self.library_browser_dialog.close()
        except:
            pass

        if self.session_is_entirely_readonly:
            self.hide()
            self.close()

        try:
            self.calibrespylibraryconfig.close()
        except:
            pass

        if not self.session_is_locked:
            if self.row_heights_were_mass_changed:
                self.exit_after_row_heights_mass_changed()
            else:
                self.hide()
                self.reset_all_row_heights()
                self.save_matrix_header_state()
                self.event_dialog_resized()
                self.save_local_prefs(exiting=True)
                self.close()
        else:
            self.hide()
            self.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def purge_state_and_exit_without_saving(self):
        if self.session_is_entirely_readonly:
            return
        try:
            self.calibrespylibraryconfig.close()
        except:
            pass
        try:
            self.dsr_dialog.close()
        except:
            pass
        try:
            self.calibrespysqlquery.close()
        except:
            pass
        try:
            self.library_browser_dialog.close()
        except:
            pass
        self.hide()
        self.reset_all_row_heights()
        self.purge_matrix_header_state()
        self.get_metadatadb_local_prefs()
        self.event_dialog_resized()
        self.save_local_prefs(exiting=True)
        self.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def exit_after_row_heights_mass_changed(self):
        try:
            self.dsr_dialog.close()
        except:
            pass
        try:
            self.calibrespysqlquery.close()
        except:
            pass
        try:
            self.library_browser_dialog.close()
        except:
            pass
        if self.session_is_entirely_readonly:
            self.hide()
            self.close()
        else:
            self.hide()
            self.reset_all_row_heights()
            #do *not* save the state...
            self.get_metadatadb_local_prefs()
            self.event_dialog_resized()
            self.save_local_prefs(exiting=True)
            self.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def check_for_network_metadata_db(self):
        #-------------------------------
        if "http" in self.library_path:
            return True
        #-------------------------------
        if "ftp" in self.library_path:
            return True
        #-------------------------------
        if self.library_path_original.startswith("\\\\"):    # "\\WDMyCloud\Calibre\autoadd" (in Windows)
            return True
        #-------------------------------
        if self.library_path.startswith("//"):
            return True
        #-------------------------------
        if "192.168." in self.library_path:
            return True
        #-------------------------------
        s_split = self.library_path.split(".")      # 172.16.1.0
        if len(s_split) > 3:
            t = 0
            for row in s_split:
                r = row.strip()
                if r.isdigit():
                    t = t + 1
            #END FOR
            if t > 3:
                del s_split
                return True
        del s_split
        #-------------------------------
        return False
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# MATRIX CONTEXT MENU SELECTION FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def ask_to_open_current_book(self):
        if question_dialog(None, "CalibreSpy", "Read the currently selected book?"):
            self.open_current_book()
#---------------------------------------------------------------------------------------------------------------------------------------
    def open_current_book(self):
        #~ open with the default application for the chosen format
        try:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            value = item.text()
            chosen_path = os.path.join(self.library_path,value)
            chosen_path = chosen_path.replace("\\","/")
            if iswindows:
                chosen_path = '"' + chosen_path + '"'
            if "/" in chosen_path:
                p_pid = subprocess.Popen(chosen_path, shell=True)
            else:
                pass
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def edit_current_book(self):
        if not self.path_to_desired_book_editor_program:
            return
        #~ open with the desired book editor program only
        try:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            value = item.text()
            value = value.strip()
            chosen_path = os.path.join(self.library_path,value)
            chosen_path = chosen_path.replace("\\","/")
            if iswindows:
                if not '"' in chosen_path:
                    chosen_path = '"' + chosen_path + '"'
                if not '"' in self.path_to_desired_book_editor_program:
                    self.path_to_desired_book_editor_program = '"' + self.path_to_desired_book_editor_program + '"'
            if "/" in chosen_path:
                path = self.path_to_desired_book_editor_program + " " + chosen_path
                p_pid = subprocess.Popen(path, shell=True)
            else:
                pass
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def view_current_book(self):
        if not self.path_to_desired_book_viewer_program:
            return
        #~ open with the desired viewer program only...unless it is a .mp3 or other audio file in self.audio_book_extensions_set
        try:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            value = item.text()
            value = value.strip()
            chosen_path = os.path.join(self.library_path,value)
            chosen_path = chosen_path.replace("\\","/")
            dir,file_extension = os.path.split(chosen_path)
            if iswindows:
                if not '"' in chosen_path:
                    chosen_path = '"' + chosen_path + '"'
                if not '"' in self.path_to_desired_book_viewer_program:
                    self.path_to_desired_book_viewer_program = '"' + self.path_to_desired_book_viewer_program + '"'
            if "/" in chosen_path:
                if not file_extension in self.audio_book_extensions_set:
                    path = self.path_to_desired_book_viewer_program + " " + chosen_path
                else:
                    pass  #let the OS pick the application handler...
                p_pid = subprocess.Popen(path, shell=True)
            else:
                pass
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def other_action_current_book(self):
        if not self.path_to_desired_book_other_program:
            return
        # open with user-defined program only
        try:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            value = item.text()
            value = value.strip()
            chosen_path = os.path.join(self.library_path,value)
            chosen_path = chosen_path.replace("\\","/")
            if iswindows:
                if not '"' in chosen_path:
                    chosen_path = '"' + chosen_path + '"'
                if not '"' in self.path_to_desired_book_other_program:
                    self.path_to_desired_book_other_program = '"' + self.path_to_desired_book_other_program + '"'
            if "/" in chosen_path:
                path = self.path_to_desired_book_other_program + " " + chosen_path
                p_pid = subprocess.Popen(path, shell=True)
            else:
                path = self.path_to_desired_book_other_program
                p_pid = subprocess.Popen(path, shell=True)
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def email_current_book(self):
        self.path_to_calibre_smtp_program_command_file = self.path_to_calibre_smtp_program_command_file.replace("\\","/")

        if not os.path.isfile(self.path_to_calibre_smtp_program_command_file):
            if DEBUG: print("'CALIBRESPY_COMMAND_FILE_PATH_EMAIL' is not a valid file path: ", self.path_to_calibre_smtp_program_command_file)
            return

        try:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            value = item.text()
            value = value.strip()
            chosen_path = os.path.join(self.library_path,value)
            chosen_path = chosen_path.replace("\\","/")
            path_to_calibre_smtp_program_command_file = self.path_to_calibre_smtp_program_command_file
            if iswindows:
                if not '"' in chosen_path:
                    chosen_path = '"' + chosen_path + '"'
                if not '"' in path_to_calibre_smtp_program_command_file:
                    path_to_calibre_smtp_program_command_file = '"' + path_to_calibre_smtp_program_command_file + '"'
            if "/" in chosen_path:
                path = path_to_calibre_smtp_program_command_file + " " + chosen_path
                p_pid = subprocess.Popen(path, shell=True)
                msg = "Current Book has been sent to calibre-smtp for emailing."
                self.change_bottom_message(msg)
            else:
                pass
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def open_format_path(self):
        #~ open file manager to the book's path
        try:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            if not item:
                return
            value = item.text()
            chosen_path = os.path.join(self.library_path,value)
            chosen_path = chosen_path.replace("\\","/")
            dir,format = os.path.split(chosen_path)
            if os.path.isdir(dir):
                if platform.system() == "Windows":
                    os.startfile(dir)
                elif platform.system() == "Darwin":
                    subprocess.Popen(["open", dir])
                else:
                    subprocess.Popen(["xdg-open", dir])
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_copy_single(self):
        self.save_copy(type=SAVE_COPY_TYPE_SINGLE)
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_copy_filtered(self):
        self.save_copy(type=SAVE_COPY_TYPE_FILTERED)
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_copy(self,type=None):

        book_list = []

        if type == SAVE_COPY_TYPE_SINGLE:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.path_column)
            if not item:
                return
            value = item.text()
            book_path = os.path.join(self.library_path,value)
            book_path = book_path.replace("\\","/")
            if not os.path.isfile(book_path):
                return
            book_list.append(book_path)
        elif type == SAVE_COPY_TYPE_FILTERED:
            for r in self.visible_rows_set:
                item = self.matrix.item(r,self.path_column)
                if not item:
                    continue
                value = item.text()
                book_path = os.path.join(self.library_path,value)
                book_path = book_path.replace("\\","/")
                if not os.path.isfile(book_path):
                    continue
                book_list.append(book_path)
            #END FOR
        else:
            return

        n = len(book_list)
        if n == 0:
            return
        elif n > 20:
            msg = "Save Copies of the displayed " + as_unicode(n) + " books?"
            if question_dialog(None, "CalibreSpy", msg):
                pass
            else:
                return

        book_list.sort()

        chosen_dir = self.choose_directory(default_dir=self.local_prefs['CALIBRESPY_SAVE_COPY_DIRECTORY'])
        if not chosen_dir:
            return

        if not os.path.isdir(chosen_dir):
            return

        self.local_prefs['CALIBRESPY_SAVE_COPY_DIRECTORY'] = chosen_dir

        if isbytestring(chosen_dir):
            chosen_dir = chosen_dir.decode(filesystem_encoding)
        chosen_dir = chosen_dir.replace(os.sep, '/')

        import shutil
        from shutil import copy

        for book_path in book_list:
            copy(book_path,chosen_dir)
        #END FOR

        del shutil
        del copy

        msg = "[" + as_unicode(len(book_list)) + " Book(s) copied to the selected directory]"
        self.change_bottom_message(msg)

        if platform.system() == "Windows":
            os.startfile(chosen_dir)
        elif platform.system() == "Darwin":
            subprocess.Popen(["open", chosen_dir])
        else:
            subprocess.Popen(["xdg-open", chosen_dir])

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def choose_directory(self,default_dir=None):
        if not os.path.isdir(default_dir):
            default_dir = ""
        from qt.core import  QFileDialog
        file_dialog = QFileDialog(self)
        file_dialog.setFileMode(QFileDialog.FileMode.Directory)
        file_dialog.setOption(QFileDialog.Option.ShowDirsOnly,True)
        file_dialog.setViewMode(QFileDialog.ViewMode.List)
        file_dialog.setDirectory(default_dir)
        dir = file_dialog.getExistingDirectory(self, 'Choose Target Directory', default_dir)
        file_dialog = None
        del file_dialog
        if not dir:
            return None
        else:
            return dir
#---------------------------------------------------------------------------------------------------------------------------------------
    def view_current_cover(self):
        if self.n_matrix_rows == 0:
            return
        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.path_column)
        value = item.text()
        value = value.strip()
        path = os.path.join(self.library_path,value)
        path = path.replace("\\","/")
        head,tail = os.path.split(path)
        path = os.path.join(head,"cover.jpg")
        path = path.replace("\\","/")
        if os.path.isfile(path):
            from calibre_plugins.calibrespy.image_viewer_dialog import ImageViewerDialog
            self.image_viewer = ImageViewerDialog(self.icon_calibrespy,path)
            self.image_viewer.setAttribute(Qt.WA_DeleteOnClose)
            self.image_viewer.show()
            del ImageViewerDialog
            if self.is_cli:
                self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def view_current_comments(self):
        if self.n_matrix_rows == 0:
            return
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:  #e.g. has no path.
            return
        bookid = self.row_to_bookid_dict[r]

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        comments = None
        mysql = "SELECT book,text,(SELECT title FROM books WHERE id = comments.book) FROM comments WHERE book = ?"
        for book,text,title in my_cursor.execute(mysql,([bookid])):
            if bookid == book:
                comments = text
        #END FOR
        my_db.close()

        if not comments:
            return

        from calibre_plugins.calibrespy.comments_viewer_single_dialog import CommentsViewerSingleDialog
        self.comments_viewer_single = CommentsViewerSingleDialog(self.icon_comments,title,comments,self.font,self.normal_fontsize,self.style_text)
        self.comments_viewer_single.setAttribute(Qt.WA_DeleteOnClose)
        self.comments_viewer_single.show()
        del comments
        del CommentsViewerSingleDialog
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def view_displayed_comments(self):
        if self.n_matrix_rows == 0:
            return

        bookids_list = []

        for r in self.visible_rows_set:
            if not r in self.row_to_bookid_dict:
                continue
            bookid = self.row_to_bookid_dict[r]
            bookids_list.append(int(bookid))
        #END FOR

        bookids_list = list(set(bookids_list))

        if len(bookids_list) == 0:
            return

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        comments_dict = {}

        mysql = "SELECT book,text,\
                       (SELECT title FROM books WHERE id = comments.book), \
                       (SELECT author_sort FROM books WHERE id = comments.book) \
                       FROM comments WHERE book = ?"
        for bookid in bookids_list:
            for book,text,title,author_sort in my_cursor.execute(mysql,([bookid])):
                if bookid == book:
                    data = bookid,author_sort,title,text
                    comments_dict[bookid] = data
                    break
            #END FOR
        #END FOR
        my_db.close()

        if len(comments_dict) == 0:
            return error_dialog(None, _('CalibreSpy'),_('No Comments exist for the indicated book(s).'), show=True)

        parent = None

        from calibre_plugins.calibrespy.comments_viewer_dialog import CommentsViewerDialog
        self.comments_viewer = CommentsViewerDialog(parent,self.icon_comments,comments_dict,bookids_list,self.font,self.normal_fontsize,self.style_text)
        self.comments_viewer.setAttribute(Qt.WA_DeleteOnClose)
        self.comments_viewer.show()
        del comments_dict
        del bookids_list
        del CommentsViewerDialog
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def library_browser_context(self):
        try:
            self.library_browser_dialog.close()
            del self.library_browser_dialog
        except:
            pass
        self.library_browser_is_active = False
        self.push_button_library_browser.setChecked(True)
        self.push_button_library_browser.setDown(True)
        self.library_browser(source="context")
#---------------------------------------------------------------------------------------------------------------------------------------
    def library_browser(self,source=None):

        if source is None:
            if self.library_browser_is_active and self.push_button_library_browser.isChecked():  #last closed properly
                self.push_button_library_browser.setChecked(False)
                self.push_button_library_browser.setDown(False)
                self.library_browser_is_active = False
                if self.is_cli:
                    self.gc.collect()
                return

        if self.n_matrix_rows == 0:
            return

        try:
            self.library_browser_dialog.close()
            del self.library_browser_dialog
        except:
            pass

        self.matrix.setFocus()

        self.library_browser_is_active = True

        search_engine_url = self.local_prefs['LIBRARY_BROWSER_SEARCH_ENGINE_URL']

        from calibre_plugins.calibrespy.library_browser_dialog import LibraryBrowserDialog
        self.library_browser_dialog = LibraryBrowserDialog(DEBUG,self.icon_calibrespy,self.font,self.normal_fontsize,self.style_text,
                                          self.push_button_library_browser,self.apsw_connect_to_library,
                                          self.row_to_bookid_dict,
                                          self.images_dir,self.icon_viewer,self.icon_book,self.icon_editor,self.icon_email,self.icon_directory,
                                          os,self.matrix,self.n_matrix_rows,self.n_matrix_columns,self.details_column_label_list,
                                          self.view_current_book,self.open_current_book,self.edit_current_book,self.email_current_book,
                                          self.save_copy_single, self.path_column,self.library_path, self.is_cli,self.library_browser_is_active,
                                          self.marked_bookids_color_dict,self.context_authors_link,search_engine_url)

        self.library_browser_dialog.setAttribute(Qt.WA_DeleteOnClose)
        self.library_browser_dialog.show()
        del LibraryBrowserDialog
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_current_authors(self):
        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.authors_column)
        authors = item.text()
        i = self.authors_filter_combobox.findText(authors)
        if i > -1:
            self.authors_filter_combobox.setCurrentIndex(i)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_current_title(self):
        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.title_column)
        title = item.text()
        i = self.title_filter_combobox.findText(title)
        if i > -1:
            self.title_filter_combobox.setCurrentIndex(i)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_current_series(self):
        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.series_column)
        series = item.text()
        i = self.series_filter_combobox.findText(series)
        if i > -1:
            self.series_filter_combobox.setCurrentIndex(i)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_current_tags(self):
        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.tags_column)
        tags = item.text()
        tags = tags + ","
        s_split = tags.split(",")
        for tag in s_split:
            tag = tag.strip()
            i = self.tags_filter_combobox.findText(tag)
            if i > -1:
                self.tags_filter_combobox.setCurrentIndex(i)
        #END FOR
        del s_split
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_current_publisher(self):
        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.publisher_column)
        publisher = item.text()
        i = self.publisher_filter_combobox.findText(publisher)
        if i > -1:
            self.publisher_filter_combobox.setCurrentIndex(i)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_languages_column_values(self,source=None):
        if not source:
            r = self.matrix.currentRow()
            item = self.matrix.item(r,self.languages_column)
            current_languages = item.text()
            self.filter_history_combobox.insertItem(1,LANGUAGES_SOURCE + current_languages)
        else:
            current_languages = self.current_languages

        for r in range(0, self.n_matrix_rows):
            item = self.matrix.item(r,self.languages_column)
            value = item.text()
            if value != current_languages:
                self.matrix.setRowHidden(r,True)
        #END FOR
        self.filter_by_authors(source=SOURCE_CONTEXT_MENU)
        self.filter_by_title(source=SOURCE_CONTEXT_MENU)
        self.filter_by_series(source=SOURCE_CONTEXT_MENU)
        self.filter_by_tags(source=SOURCE_CONTEXT_MENU)
        self.filter_by_publisher(source=SOURCE_CONTEXT_MENU)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_generic_custom_column_values(self):
        r = self.matrix.currentRow()
        cc_column = self.matrix.currentColumn()
        if not cc_column in self.cc_column_set:  #not a custom column's column
            return
        item = self.matrix.item(r,cc_column)
        if item:
            ccvalue = item.text()
            # no filter history for custom columns since everything about their existance or placement in the matrix is unstable, arbitrary, and easily changed at any time for any reason.
        else:
            return

        for r in range(0, self.n_matrix_rows):
            item = self.matrix.item(r,cc_column)
            if item:
                value = item.text()
                if value != ccvalue:
                    self.matrix.setRowHidden(r,True)
        #END FOR
        self.filter_by_authors(source=SOURCE_CONTEXT_MENU)
        self.filter_by_title(source=SOURCE_CONTEXT_MENU)
        self.filter_by_series(source=SOURCE_CONTEXT_MENU)
        self.filter_by_tags(source=SOURCE_CONTEXT_MENU)
        self.filter_by_publisher(source=SOURCE_CONTEXT_MENU)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_filter_generic_date_column_values(self):
        r = self.matrix.currentRow()
        date_column = self.matrix.currentColumn()
        if not date_column in self.date_column_set:  #not a date column's column
            return
        item = self.matrix.item(r,date_column)
        if item:
            datevalue = item.text()      # no history saved since not enough value added to occupy a "last 30 filter" spot.
        else:
            return

        for r in range(0, self.n_matrix_rows):
            item = self.matrix.item(r,date_column)
            if item:
                value = item.text()
                if value != datevalue:
                    self.matrix.setRowHidden(r,True)
        #END FOR
        self.filter_by_authors(source=SOURCE_CONTEXT_MENU)
        self.filter_by_title(source=SOURCE_CONTEXT_MENU)
        self.filter_by_series(source=SOURCE_CONTEXT_MENU)
        self.filter_by_tags(source=SOURCE_CONTEXT_MENU)
        self.filter_by_publisher(source=SOURCE_CONTEXT_MENU)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_clear_all_filters(self):
        self.clear_all_filters()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_copy_metadata_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        r = self.matrix.currentRow()
        if not isinstance(r,int):
            error_dialog(self, 'Cannot copy metadata','No book selected', show=True)
            return False,None

        s = HEADING_STRING + TAB

        h = ""
        for i in range(0, self.n_matrix_columns):
            try:
                item = self.matrix.horizontalHeaderItem(i)
            except:
                item = None
            if item is not None:
                t = item.text()
                #~ t = as_unicode(t)
                t = as_unicode(t)
            else:
                #~ t = as_unicode("")
                t = ""
            h = h + TAB + t
        #END FOR
        s = s + h
        del h

        s = s +  NEWLINE
        s = s + METADATA_STRING + TAB
        for c in range(0, self.n_matrix_columns):
            try:
                s = s + as_unicode(self.matrix.item(r,c).text()) + TAB
            except:
                s = s + TAB
        #END FOR
        s = s[:-1] + NEWLINE        #eliminate last '\t'
        if not  self.clip:
            self.clip = QApplication.clipboard()
        self.clip.setText(s)
        del s
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def hide_current_column(self):
        c = self.matrix.currentColumn()
        self.matrix.setColumnHidden(c,True)
#---------------------------------------------------------------------------------------------------------------------------------------
    def unhide_all_columns(self):
        for c in range(0,self.n_matrix_columns):
            self.matrix.setColumnHidden(c,False)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_sort_sequence_1(self):
        if self.sorts_locked:
            return
        self.sorts_locked = True
        msg = "[Sorting in Progress]"
        self.change_bottom_message(msg)
        self.sort_matrix(source="context")
        self.matrix.setCurrentCell(0,0)
        msg = "[Sorting Complete]"
        self.change_bottom_message(msg)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_sort_sequence_2(self):
        if self.sorts_locked:
            return
        self.sorts_locked = True
        msg = "[Sorting in Progress]"
        self.change_bottom_message(msg)
        c = self.matrix.currentColumn()
        self.matrix.sortItems(c,Qt.AscendingOrder)
        self.matrix.setCurrentCell(0,0)
        msg = "[Sorting Complete]"
        self.change_bottom_message(msg)
        self.rebuild_row_to_bookid_dict()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_sort_sequence_3(self):
        if self.sorts_locked:
            return
        self.sorts_locked = True
        msg = "[Sorting in Progress]"
        self.change_bottom_message(msg)
        c = self.matrix.currentColumn()
        self.matrix.sortItems(c,Qt.DescendingOrder)
        self.matrix.setCurrentCell(0,0)
        msg = "[Sorting Complete]"
        self.change_bottom_message(msg)
        self.rebuild_row_to_bookid_dict()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_ftp_book_send(self):
        QTimer.singleShot(0,self.context_ftp_book_send_now)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_ftp_book_send_now(self):
        self.context_ftp_book_action(action=FTP_SEND_ACTION)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_ftp_book_delete(self):
        QTimer.singleShot(0,self.context_ftp_book_delete_now)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_ftp_book_delete_now(self):
        self.context_ftp_book_action(action=FTP_DELETE_ACTION)
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_ftp_choose_active_device(self):
        #~ calibrespy_dialog self.local_prefs for FTP userid and password always remain encoded.  always.

        ftp_prefs = {}
        #~ for k,v in self.local_prefs.iteritems():
        for k,v in iteritems(self.local_prefs):
            if "FTP" in k:
                ftp_prefs[k] = v
        #END FOR

        from calibre_plugins.calibrespy.ftp_target_host_selection_dialog import FTPTargetHostSelectionDialog
        ftp_host_dialog = FTPTargetHostSelectionDialog(self,self.icon_ftp,self.font,self.style_text,ftp_prefs,self.return_new_local_pref_value)
        ftp_host_dialog.setAttribute(Qt.WA_DeleteOnClose )

        del ftp_prefs
        del FTPTargetHostSelectionDialog

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_ftp_book_action(self,action=None):

        if action is None:
            return

        r = self.matrix.currentRow()
        item = self.matrix.item(r,self.path_column)
        if item:
            path = item.text()
            if not "/" in path:
                msg = "[FTP Failure.  Path of Book is Invalid.]"
                self.change_bottom_message(msg)
                return

        msg = "[FTP Connecting...]"
        self.change_bottom_message(msg)

        QApplication.instance().processEvents()

        head,book_fname = os.path.split(path)
        book_fname = book_fname.replace("\\","/")

        book_path = os.path.join(self.library_path,path)
        book_path = book_path.replace("\\","/")

        host =  self.local_prefs['CALIBRESPY_FTP_HOST']
        port = self.local_prefs['CALIBRESPY_FTP_HOST_PORT']
        host_directory = self.local_prefs['CALIBRESPY_FTP_HOST_DIRECTORY']
        userid =  self.local_prefs['CALIBRESPY_FTP_USERID']
        password =  self.local_prefs['CALIBRESPY_FTP_PASSWORD']

        raw_in = userid
        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
        userid = codecs.decode(raw_in, 'rot13')
        raw_in = password
        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
        password = codecs.decode(raw_in, 'rot13')

        host = host.replace(os.sep,"/")
        host_directory = host_directory.replace(os.sep,"/")

        port = as_unicode(port).strip()
        port = int(port)

        if not self.ftp:
            from ftplib import FTP
            self.ftp = FTP
            del FTP

        ftp = self.ftp()

        try:
            ftp.connect(host,port)
        except Exception as e:
            msg = "Fatal error connecting to FTP host: " + as_unicode(host) + " port: " + as_unicode(port) + "  >>>  " + as_unicode(e)
            return error_dialog(self, _('FTP'),_(msg), show=True)

        msg = "[FTP Login...]"
        self.change_bottom_message(msg)

        QApplication.instance().processEvents()

        try:
            ftp.login(userid,password)
        except Exception as e:
            ftp.quit()
            msg = "Fatal error logging in to FTP host using specified credentials: " + as_unicode(e)
            return error_dialog(self, _('FTP'),_(msg), show=True)

        try:
            ftp.cwd(host_directory)
        except Exception as e:
            ftp.quit()
            msg = "Fatal error changing directory to: " + as_unicode(host_directory) + "  >>>  " + as_unicode(e)
            return error_dialog(self, _('FTP'),_(msg), show=True)

        if action == FTP_SEND_ACTION:
            msg = "[FTP uploading book to server...]"
            self.change_bottom_message(msg)
            book_path = book_path.replace( '/',os.sep)
            try:
                ftp.storbinary("STOR " + book_fname, open(book_path, "rb"), 1024)   # 8192 or 1024
                n_warning = 0
                msg = "[FTP Upload Completed]"
                if DEBUG: print(msg)
            except Exception as e:
                n_warning = 1
                msg = "[FTP Upload Error: <br>" + as_unicode(e) + "]"
                if DEBUG: print(msg)
        elif action == FTP_DELETE_ACTION:
            msg = "[FTP deleting book from server...]"
            self.change_bottom_message(msg)
            book_path = book_path.replace( '/',os.sep)
            try:
                ftp.delete(book_fname)
                n_warning = 0
                msg = "[FTP Deletion Completed]"
                if DEBUG: print(msg)
            except Exception as e:
                n_warning = 1
                msg = "[FTP Deletion Error: <br>" + as_unicode(e) + "]"
                if DEBUG: print(msg)
        else:
            msg = None

        if msg is not None:
            msg = msg.strip()
            self.change_bottom_message(msg)
            QApplication.instance().processEvents()

        try:
            ftp.quit()
        except:
            try:
                ftp.close()
            except:
                pass

        del ftp
        del msg

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_arbitrary_files(self):
        try:
            self.arbitrary_file_dialog.close()
        except:
            pass
        from calibre_plugins.calibrespy.arbitrary_file_dialog import ArbitraryFileDialog
        self.arbitrary_file_dialog = ArbitraryFileDialog(None,self.icon_gear,self.font,self.style_text)
        self.arbitrary_file_dialog.setModal(False)
        self.arbitrary_file_dialog.show()
        self.arbitrary_file_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del ArbitraryFileDialog
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_visualize_metadata(self):
        from calibre_plugins.calibrespy.visualize_metadata_dialog import VisualizeMetadataDialog
        dir = self.local_prefs['VISUALIZE_METADATA_DEFAULT_EXPORT_DIRECTORY']
        self.visualize_metadata_dialog = VisualizeMetadataDialog(None,self.library_path,self.icon_wrench_hammer,self.font,self.style_text,dir,self.return_new_local_pref_value)
        self.visualize_metadata_dialog.setModal(False)
        self.visualize_metadata_dialog.show()
        self.visualize_metadata_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del VisualizeMetadataDialog
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_formatspy(self):

        r = self.matrix.currentRow()

        if r in self.row_to_bookid_dict:
            bookid = self.row_to_bookid_dict[r]
        else:
            return

        item = self.matrix.item(r,self.path_column)
        if item:
            path = item.text()

        item = self.matrix.item(r,self.authors_column)
        if item:
            authors = item.text()

        item = self.matrix.item(r,self.title_column)
        if item:
            title = item.text()

        item = self.matrix.item(r,self.added_column)
        if item:
            added = item.text()

        from calibre_plugins.calibrespy.formatspy_dialog import FormatSpyDialog
        self.formatspy_dialog = FormatSpyDialog(self.icon_formatspy,self.library_path,path,bookid,authors,title,added,self.font,self.normal_fontsize,self.style_text)
        self.formatspy_dialog.setModal(False)
        self.formatspy_dialog.show()
        self.formatspy_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del FormatSpyDialog
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def context_authors_link(self):

        r = self.matrix.currentRow()

        if r in self.row_to_bookid_dict:
            bookid = self.row_to_bookid_dict[r]
        else:
            return

        if bookid in self.bookid_authors_link_dict:
            authors_link_list = self.bookid_authors_link_dict[bookid]
        else:
            return

        if not isinstance(authors_link_list,list):
            return

        if len(authors_link_list) == 0:
            return

        if len(authors_link_list) > 1:
            title = "Author Links"
            label = "Choose Author Link for Multiple Authors"
            link,ok = QInputDialog.getItem(None, title, label, authors_link_list,0,False)
            if not ok:
                return
        else:
            link = authors_link_list[0]

        link = link.strip()

        link = link.replace("\\","/")
        if link.startswith('"') and link.endswith('"'):  #Windows copy-path artifacts
            link = link[1:-1]

        if not link > " ":
            return

        if self.open_url is None:
            from calibre.gui2 import open_url
            self.open_url = open_url
            del open_url

        try:
            self.open_url(QUrl(link, QUrl.ParsingMode.TolerantMode))
            del link
        except Exception as e:
            msg = "Open URL Error for Author Link:  " + as_unicode(e)
            if DEBUG: print(msg)
            error_dialog(None, _('CalibreSpy'),_(msg), show=True)

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def calibre_links_menu_about_to_show(self):
        # see  https://manual.calibre-ebook.com/url_scheme.html
        #~ -----------------------------
        def create_calibre_link_action(icon,text,url):
            def add_action():
                if DEBUG: print("Calibre Link's url: ", url)
                QApplication.instance().clipboard().setText(url)
            self.calibre_links_menu.addAction(icon,text,add_action)
            self.calibre_links_menu.update()
        #~ -----------------------------

        r = self.matrix.currentRow()
        if r in self.row_to_bookid_dict:
            bookid = self.row_to_bookid_dict[r]
        else:
            return

        self.calibre_links_menu.clear()

        #~ -----------------------------
        #~ 'Show book in Calibre'
        #~ -----------------------------
        url = 'calibre://search/library_name?q=query'   #  __my_version__ = "1.0.91"  # calibre://search/CalibreZotero?q="id:13903" s/be ...q=id:13903
        url = url.replace("library_name",self.library_name)
        query = "id:" + as_unicode(bookid)
        url = url.replace("query",query)
        create_calibre_link_action(self.icon_book,_('Show book in Calibre'), url)
        #~ -----------------------------

        item = self.matrix.item(r,self.authors_column)
        if item is None:
            pass
        else:
            authors = item.text()
            if authors is not None:
                author_list = []
                authors = authors + " & "
                ssplit = authors.split("&")
                for r in ssplit:
                    r = r.strip()
                    if r > "":
                        author_list.append(r)
                        if DEBUG: print("author: ", r)
                #END FOR
                author_list.sort()
                if len(author_list) > 50:  #context menu won't fit on screen if too many authors...
                    author_list = author_list[0:50]
                #~ -----------------------------
                #~ 'Show books matching {} in Calibre'
                #~ -----------------------------
                for name in author_list:
                    #~ see  https://manual.calibre-ebook.com/url_scheme.html
                    text = 'Show books matching {} in Calibre'.format(name)
                    url = 'calibre://search/library_name?eq=query'
                    url = url.replace("library_name",self.library_name)
                    ssplit = name.split(",")  #CalibreSpy always uses the LN,FN format for authors...
                    n = len(ssplit)
                    if n == 0:
                        continue
                    elif n == 1:
                        fnln = ssplit[0].strip()
                    else:
                        fnln = ssplit[1].strip() + " " + ssplit[0].strip()
                    query = 'authors:"=NAME"'.replace("NAME",fnln)
                    query = query.encode("utf8").hex()  # hex is required in Windows 10 because otherwise, the double-quotes in the search term will be lost...making the search too broad...
                    url = url.replace("query",query)
                    create_calibre_link_action(self.icon_binoculars,text,url)
                #END FOR

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# VERTICAL HEADER CONTEXT MENU SELECTION FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def change_all_row_heights(self):
        #match the currently selected row's height.
        r = self.matrix.currentRow()
        h = self.matrix.rowHeight(r)
        if h > 60:
            h = 60
        for r in range(0,self.n_matrix_rows):
            self.matrix.setRowHeight(r, h)
        #END FOR
        self.row_heights_were_mass_changed = True
        self.matrix.update()
#---------------------------------------------------------------------------------------------------------------------------------------
    def reset_all_row_heights(self):
        #reset to the original row height just after matrix was loaded at startup.
        for r in range(0,self.n_matrix_rows):
            self.matrix.setRowHeight(r, self.original_row_height)
        #END FOR
        self.row_heights_were_mass_changed = False
        self.matrix.update()
#---------------------------------------------------------------------------------------------------------------------------------------
    def mark_current_book_cyan(self):
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:
            return
        item = self.matrix.item(r,0)
        item.setBackground(Qt.cyan)
        self.marked_rows_set.add(r)
        bookid = self.row_to_bookid_dict[r]
        self.marked_bookids_color_dict[bookid] = CYAN
        self.marked_bookids_set.add(bookid)
#---------------------------------------------------------------------------------------------------------------------------------------
    def mark_current_book_yellow(self):
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:
            return
        item = self.matrix.item(r,0)
        item.setBackground(Qt.yellow)
        self.marked_rows_set.add(r)
        bookid = self.row_to_bookid_dict[r]
        self.marked_bookids_color_dict[bookid] = YELLOW
        self.marked_bookids_set.add(bookid)
#---------------------------------------------------------------------------------------------------------------------------------------
    def mark_current_book_red(self):
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:
            return
        item = self.matrix.item(r,0)
        item.setBackground(Qt.red)
        self.marked_rows_set.add(r)
        bookid = self.row_to_bookid_dict[r]
        self.marked_bookids_color_dict[bookid] = RED
        self.marked_bookids_set.add(bookid)
#---------------------------------------------------------------------------------------------------------------------------------------
    def mark_current_book_green(self):
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:
            return
        item = self.matrix.item(r,0)
        item.setBackground(Qt.green)
        self.marked_rows_set.add(r)
        bookid = self.row_to_bookid_dict[r]
        self.marked_bookids_color_dict[bookid] = GREEN
        self.marked_bookids_set.add(bookid)
#---------------------------------------------------------------------------------------------------------------------------------------
    def unmark_current_book(self,r=None):
        if not r:
            r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:
            return
        item = self.matrix.item(r,0)
        if item:
            item.setBackground(self.original_row_background_color)
        self.marked_rows_set.discard(r)
        bookid = self.row_to_bookid_dict[r]
        if bookid in self.marked_bookids_color_dict:
            del self.marked_bookids_color_dict[bookid]
        self.marked_bookids_set.discard(bookid)
#---------------------------------------------------------------------------------------------------------------------------------------
    def unmark_all_books(self):
        tmp = self.marked_rows_set.copy()
        for r in tmp:
            self.unmark_current_book(r)
        #END FOR
        self.marked_rows_set.clear()
        self.marked_bookids_color_dict.clear()
        self.marked_bookids_set.clear()
        del tmp
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_marked_books(self):
        self.visible_rows_set.clear()
        for r in range(0,self.n_matrix_rows):
            if not r in self.marked_rows_set:
                self.matrix.setRowHidden(r,True)
            else:
                self.matrix.setRowHidden(r,False)
                self.visible_rows_set.add(r)
        #END FOR
        self.filter_by_authors(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_title(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_series(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_tags(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_publisher(source=SOURCE_FILTER_USING_COLORS)
        self.get_visible_row_count()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_marked_books_cyan(self):
        self.filter_marked_books_by_color(color=CYAN)
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_marked_books_green(self):
        self.filter_marked_books_by_color(color=GREEN)
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_marked_books_yellow(self):
        self.filter_marked_books_by_color(color=YELLOW)
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_marked_books_red(self):
        self.filter_marked_books_by_color(color=RED)
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_marked_books_by_color(self,color=None):
        self.visible_rows_set.clear()
        for r in range(0,self.n_matrix_rows):
            if not r in self.marked_rows_set:
                self.matrix.setRowHidden(r,True)
            else:
                bookid = self.row_to_bookid_dict[r]
                if self.marked_bookids_color_dict[bookid] == color:
                    self.matrix.setRowHidden(r,False)
                    self.visible_rows_set.add(r)
                else:
                    self.matrix.setRowHidden(r,True)
        #END FOR
        self.filter_by_authors(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_title(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_series(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_tags(source=SOURCE_FILTER_USING_COLORS)
        self.filter_by_publisher(source=SOURCE_FILTER_USING_COLORS)
        self.get_visible_row_count()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# HORIZONTAL HEADER CONTEXT MENU SELECTION FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_blank_column_values(self):
        c = self.matrix.currentColumn()
        for r in range(0, self.n_matrix_rows):
            item = self.matrix.item(r,c)
            if item:
                value = item.text()
                if not value > "":
                    self.matrix.setRowHidden(r,True)
        #END FOR
        self.filter_by_authors(source=SOURCE_CONTEXT_MENU)
        self.filter_by_title(source=SOURCE_CONTEXT_MENU)
        self.filter_by_series(source=SOURCE_CONTEXT_MENU)
        self.filter_by_tags(source=SOURCE_CONTEXT_MENU)
        self.filter_by_publisher(source=SOURCE_CONTEXT_MENU)
        self.get_visible_row_count()
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_nonblank_column_values(self):
        c = self.matrix.currentColumn()
        for r in range(0, self.n_matrix_rows):
            item = self.matrix.item(r,c)
            if item:
                value = item.text()
                if value > "":
                    self.matrix.setRowHidden(r,True)
        #END FOR
        self.filter_by_authors(source=SOURCE_CONTEXT_MENU)
        self.filter_by_title(source=SOURCE_CONTEXT_MENU)
        self.filter_by_series(source=SOURCE_CONTEXT_MENU)
        self.filter_by_tags(source=SOURCE_CONTEXT_MENU)
        self.filter_by_publisher(source=SOURCE_CONTEXT_MENU)
        self.get_visible_row_count()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# LIBRARY-SPECIFIC PREFERENCES FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def customize_calibrespy(self):
        self.save_local_prefs()
        from calibre_plugins.calibrespy.calibrespy_customization import CalibreSpyLibraryConfigWidget
        self.calibrespylibraryconfig = CalibreSpyLibraryConfigWidget(QApplication,self.icon_calibrespy,self.library_path,self.apsw_connect_to_library,self.change_local_prefs_after_customizing,self.style_text)
        self.calibrespylibraryconfig.show()
        self.calibrespylibraryconfig.setAttribute(Qt.WA_DeleteOnClose)
        del CalibreSpyLibraryConfigWidget
        self.rebuild_row_to_bookid_dict()  #just an efficient time to do this while the user's attention is elsewhere...
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def change_local_prefs_after_customizing(self,restart=False):
        #~ this function is passed to calibrespy_customization, and then executed by calibrespy_customization on its return with updated local prefs to calibrespy_dialog.
        self.calibrespylibraryconfig.close()
        del self.calibrespylibraryconfig

        if restart:  # column assignments have been changed...shut down CalibreSpy.
            try:
                self.library_browser_dialog.close()
            except:
                pass
            self.purge_state_and_exit_without_saving()
            self.hide()
            self.close()
        else:
            self.change_local_cosmetic_prefs_after_customizing()
#---------------------------------------------------------------------------------------------------------------------------------------
    def change_local_cosmetic_prefs_after_customizing(self):
        #~ Local preferences for everything cosmetic must be reapplied after returning from customiziing.
        #~ Metadata to load may have changed, but it is already loaded.  So, it will  just be hidden until the next restart of CalibreSpy.

        self.get_metadatadb_local_prefs()
        self.save_local_prefs()

        if self.n_matrix_rows == 0:
            if self.is_cli:
                self.gc.collect()
            return

        self.get_program_path_values()

        self.hide_show_optional_columns()

        self.build_fonts()

        self.matrix.setFont(self.font)

        self.matrix.horizontalHeader().setFont(self.font)
        self.matrix.verticalHeader().setFont(self.font)

        self.matrix.horizontalHeader().setResizeContentsPrecision(0)
        self.matrix.verticalHeader().setResizeContentsPrecision(0)

        n_added_row_height = int(self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE'])
        self.default_row_height = self.local_prefs['DEFAULT_MATRIX_ROW_HEIGHT']
        if self.default_row_height < 15:
            self.default_row_height = 15
        self.n_total_row_height  = self.default_row_height + n_added_row_height
        if self.n_total_row_height > 60:
            self.n_total_row_height = 60
        #END FOR

        self.original_row_height = self.default_row_height
        if self.original_row_height > 60:
            self.original_row_height = 60

        for c in range(0,self.n_matrix_columns):
            for r in range(0,self.n_matrix_rows):
                self.matrix.setRowHeight(r,self.n_total_row_height)
                item = self.matrix.item(r,c)
                if item is not None:
                    item.setFont(self.font)
            #END FOR
        #END FOR

        self.matrix.update()

        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        bc = self.local_prefs['CALIBRESPY_BACKGROUND_COLOR']
        ac = self.local_prefs['CALIBRESPY_ALTERNATING_COLOR']
        style_string = "QTableWidget, QHeaderView {alternate-background-color: [AC];background-color: [BC];color : [TC]; } "
        style_string = style_string.replace("[TC]",tc)
        style_string = style_string.replace("[AC]",bc)  # BC and AC reversed so darker color is used for matix row #0
        style_string = style_string.replace("[BC]",ac)
        self.matrix.setStyleSheet(style_string)

        self.matrix.update()

        item = self.matrix.item(0,0)
        self.original_row_background_color = item.background()

        self.font.setBold(True)
        self.bottom_box_qlabel.setFont(self.font)
        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        ss = "QLabel {color : [TC]; }".replace("[TC]",tc)
        self.bottom_box_qlabel.setStyleSheet(ss)
        self.font.setBold(False)

        self.authors_filter_combobox.setFont(self.font)
        self.title_filter_combobox.setFont(self.font)
        self.series_filter_combobox.setFont(self.font)
        self.tags_filter_combobox.setFont(self.font)
        self.publisher_filter_combobox.setFont(self.font)

        self.authors_count_qlabel.setFont(self.font)
        self.title_count_qlabel.setFont(self.font)
        self.series_count_qlabel.setFont(self.font)
        self.tags_count_qlabel.setFont(self.font)
        self.publisher_count_qlabel.setFont(self.font)

        self.push_button_optimize_column_widths.setFont(self.font)
        self.push_button_deoptimize_column_widths.setFont(self.font)
        self.push_button_open_current_book.setFont(self.font)
        self.push_button_view_current_book.setFont(self.font)
        self.push_button_other_current_book.setFont(self.font)
        self.push_button_edit_current_book.setFont(self.font)
        self.push_button_save_and_exit.setFont(self.font)

        self.push_button_clear_all_filters.setFont(self.font)
        self.filter_history_combobox.setFont(self.font)

        self.menu.setFont(self.font)
        self.column_header_menu.setFont(self.font)
        self.row_header_menu.setFont(self.font)

        QApplication.instance().processEvents()

        self.resize_dialog_to_preferences()

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_metadatadb_local_prefs(self):
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        if self.reset:
            self.session_is_entirely_readonly = False
            self.drop_metadatadb_settings_table(my_db,my_cursor)
            self.create_metadatadb_settings_table(my_db,my_cursor)
            self.save_local_prefs(my_db,my_cursor)
        else:
            try:
                mysql = "SELECT prefkey,prefvalue,prefblob FROM _calibrespy_settings"
                my_cursor.execute(mysql)
                tmp_rows = my_cursor.fetchall()
                if tmp_rows is None:
                    tmp_rows = []
                for row in tmp_rows:
                    key,value,geom = row
                    if key == "CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE":   #bytearray is a blob, and requires unique handling for both saving and retrieving...
                        continue
                    elif key == 'CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE':
                        n = int(value)   #all table values are unicode...
                        self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE']  = n
                    else:
                        if value == "0":       #all table values are unicode...
                            value = 0
                        elif value == "1":
                            value = 1
                        else:
                            if value.isdigit():   #e.g. column numbers...
                                value = int(value)
                        self.local_prefs[key] = value
                #END FOR
                del tmp_rows
            except Exception as e:
                if not self.session_is_entirely_readonly:
                    if DEBUG: print("Local Customization Preferences Table '_calibrespy_settings' will be created and populated with Global defaults if it does not already exist, or upgraded to V1.0.3 if it does.")
                    self.create_metadatadb_settings_table(my_db,my_cursor)
                    self.save_local_prefs(my_db,my_cursor)

        my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_local_prefs(self,my_db=None,my_cursor=None,exiting=False):

        if self.session_is_entirely_readonly:
            self.user_can_save_settings = False
            self.multiuser_is_in_play = True
            self.hide_push_button_customize = True
            self.save_multiuser_prefs = False
            return

        must_close_db = False

        if not my_db:
            my_db,my_cursor,is_valid = self.apsw_connect_to_library()
            must_close_db = True
            if not is_valid:
                 return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        self.check_multiuser_status(my_db,my_cursor)

        self.remove_orphaned_lock = False

        if not exiting:
            if not self.multiuser_is_in_play:
                self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'] = self.username_timestamp
                self.local_prefs['CALIBRESPY_LAST_UPDATED_USERNAME'] = self.username
                self.hide_push_button_customize = False
                self.save_multiuser_prefs = True
            else:
                self.save_multiuser_prefs = False
                self.hide_push_button_customize = True  # hide the pushbutton to access library-specific preferences..
        else:  #exiting, so no longer using table _calibrespy_settings...
            if not self.multiuser_is_in_play:
                self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'] = ""
                self.local_prefs['CALIBRESPY_LAST_UPDATED_USERNAME'] = ""
                self.save_multiuser_prefs = True
            else:
                self.save_multiuser_prefs = False
                # check for orphaned multiuser lock...
                then = datetime.strptime(self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'], "%Y-%m-%d %H:%M:%S.%f" )     #2018-09-10 08:15:37.294000
                now = datetime.now()
                elapsed = now - then
                elapsed = elapsed.total_seconds()
                n = self.local_prefs['CALIBRESPY_ORPHANED_MULTIUSER_LOCK_ELAPSED_HOURS']
                n = (int(n)) * (3600)
                if elapsed >= n:     # assume it is an orphaned lock, so ignore it
                    self.remove_orphaned_lock = True
                    if DEBUG: print("multiuser lock age in seconds: ", as_unicode(elapsed))
                else:  # might be many other viewers of the identical library at this identical point in time...do not change their specific multiuser-related values...
                    self.remove_orphaned_lock = False

        if exiting:
            if not self.multiuser_is_in_play:
                if self.user_can_save_settings:
                    self.local_prefs['CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'] = 1
                    self.local_prefs['CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL'] = 1

        if exiting:
            if not self.multiuser_is_in_play:
                if self.user_can_save_settings:
                    s = ""
                    for bookid in self.marked_bookids_set:
                        color = self.marked_bookids_color_dict[bookid]
                        bookid = as_unicode(bookid)
                        s = s + bookid + "," + color + MARKED_BOOK_SEPARATOR
                    #END FOR
                    self.local_prefs['CALIBRESPY_MARKED_ROW_SNAPSHOT'] = unicode_type(s)

        if self.filtering_history_available:
            if not self.multiuser_is_in_play:
                if self.user_can_save_settings:
                    self.save_filtering_history()

        if not self.multiuser_is_in_play:
            if self.user_can_save_settings:
                my_cursor.execute("begin")
                mysql = "INSERT OR REPLACE INTO _calibrespy_settings (prefkey,prefvalue,prefblob) VALUES (?,?,?)"
                #~ for key,value in self.local_prefs.iteritems():
                for key,value in iteritems(self.local_prefs):
                    if key == "CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE":   #bytearray is a blob requiring very special handling
                        continue
                    elif self.save_multiuser_prefs:
                        my_cursor.execute(mysql,(key,value,""))
                        continue
                    else:
                        if key == 'CALIBRESPY_LAST_UPDATED_DATETIME' or key == 'CALIBRESPY_LAST_UPDATED_USERNAME':
                            if self.remove_orphaned_lock:
                                if DEBUG: print("Orphaned Multiuser Lock will be removed: ", value)
                                value = ""
                                my_cursor.execute(mysql,(key,value,""))
                            else:
                                continue
                        else:
                            value = as_unicode(value)  # all values are unicode
                            my_cursor.execute(mysql,(key,value,""))
                            continue
                #END FOR
                my_cursor.execute("commit")

        if must_close_db:
            my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_local_prefs_specific(self,pref1=None,pref2=None,pref3=None,pref4=None):

        if self.session_is_entirely_readonly:
            return

        if not self.user_can_save_settings:
            return

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        must_close_db = True
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _calibrespy_settings (prefkey,prefvalue,prefblob) VALUES (?,?,?)"
        if pref1 is not None:
            if pref1 in self.local_prefs:
                value = self.local_prefs[pref1]
                my_cursor.execute(mysql,(pref1,value,""))
                if pref2 is not None:
                    if pref2 in self.local_prefs:
                        value = self.local_prefs[pref2]
                        my_cursor.execute(mysql,(pref2,value,""))
                        if pref3 is not None:
                            if pref3 in self.local_prefs:
                                value = self.local_prefs[pref3]
                                my_cursor.execute(mysql,(pref3,value,""))
                                if pref4 is not None:
                                    if pref4 in self.local_prefs:
                                        value = self.local_prefs[pref4]
                                        my_cursor.execute(mysql,(pref4,value,""))
        my_cursor.execute("commit")
        my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def return_new_local_pref_value(self,pref,value,update_db_now=False):
        #use: Visualize Metadata to return changed default metadata .csv export directory
        #use: FTP target device selection dialog:  return chosen FTP settings (host, port, etc.)

        self.local_prefs[pref] = value
        if update_db_now:
            self.save_local_prefs_specific(pref1=pref)
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_filtering_history(self):
        item = ""
        for i in range(1,self.filter_history_combobox.count()):
            v = self.filter_history_combobox.itemText(i).strip()
            if v:
                if v > "":
                    item = item + v + HISTORY_SEPARATOR
        #END FOR
        self.local_prefs['HISTORICAL_FILTERING_EVENTS'] = item
#---------------------------------------------------------------------------------------------------------------------------------------
    def create_metadatadb_settings_table(self,my_db,my_cursor):
        try:
            my_cursor.execute("begin")
            mysql_v103 = "CREATE TABLE _calibrespy_settings(prefkey TEXT NOT NULL COLLATE NOCASE PRIMARY KEY, prefvalue TEXT NOT NULL COLLATE NOCASE, prefblob BLOB, UNIQUE(prefkey))"
            my_cursor.execute(mysql_v103)
            my_cursor.execute("commit")
            if DEBUG: print("CalibreSpy settings table '_calibrespy_settings' has been newly created.")
        except Exception as e:
            try:
                my_cursor.execute("commit")
            except:
                pass
            self.upgrade_settings_table_if_needed(my_db,my_cursor)
#--------------------------------------------------
    def upgrade_settings_table_if_needed(self,my_db,my_cursor):
        try:
            mysql = "SELECT prefkey,prefvalue,prefblob FROM _calibrespy_settings"
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            del tmp_rows
            return
        except Exception as e:
            if DEBUG: print("Upgrade of table _calibrespy_settings structure from Version 1.02 to Version 1.03 is required...", as_unicode(e))
            mysql = "SELECT prefkey,prefvalue FROM _calibrespy_settings"
            my_cursor.execute(mysql)
            version_102_rows  = my_cursor.fetchall()
            my_cursor.execute("begin")
            mysql = "DROP TABLE IF EXISTS _calibrespy_settings "
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            if DEBUG: print("Version 1.0.2 table _calibrespy_settings was dropped.")
            my_cursor.execute("begin")
            mysql_v103 = "CREATE TABLE _calibrespy_settings(prefkey TEXT NOT NULL COLLATE NOCASE PRIMARY KEY, prefvalue TEXT NOT NULL COLLATE NOCASE, prefblob BLOB, UNIQUE(prefkey))"
            my_cursor.execute(mysql_v103)
            my_cursor.execute("commit")
            if DEBUG: print("CalibreSpy settings table '_calibrespy_settings' has been created.")
            my_cursor.execute("begin")
            for row in version_102_rows:
                prefkey,prefvalue = row
                prefblob = ""
                mysql = "INSERT OR REPLACE INTO _calibrespy_settings (prefkey,prefvalue,prefblob) VALUES (?,?,?)"
                my_cursor.execute(mysql,(prefkey,prefvalue,prefblob))
            #END FOR
            my_cursor.execute("commit")
            if DEBUG: print("Version 1.0.2 preferences have been added to the upgraded CalibreSpy preferences table '_calibrespy_settings'.  Conversion complete.")
#---------------------------------------------------------------------------------------------------------------------------------------
    def drop_metadatadb_settings_table(self,my_db,my_cursor):
        if not self.reset:
            return
        my_cursor.execute("begin")
        my_cursor.execute("DROP TABLE IF EXISTS _calibrespy_settings")
        my_cursor.execute("commit")
        self.reset = False
        if DEBUG: print("CalibreSpy settings table '_calibrespy_settings' has been dropped per '--reset' option.")
#---------------------------------------------------------------------------------------------------------------------------------------
# __INIT__ FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def build_fonts(self):

        if self.local_prefs['DEFAULT_FONT_FAMILY'] == 0:
            if self.calibre_font is not None:
                self.local_prefs['DEFAULT_FONT_FAMILY'] = self.calibre_font[0]
                self.local_prefs['DEFAULT_FONT_SIZE'] = self.calibre_font[1]
            else:
                if iswindows:
                    self.local_prefs['DEFAULT_FONT_FAMILY'] =  DEFAULT_FONT_WINDOWS
                    self.local_prefs['DEFAULT_FONT_SIZE'] = 9
                else:
                    self.local_prefs['DEFAULT_FONT_FAMILY'] =  DEFAULT_FONT_OTHER
                    self.local_prefs['DEFAULT_FONT_SIZE'] = 10

        self.font = QFont()
        self.font.setFamily(self.local_prefs['DEFAULT_FONT_FAMILY'])
        self.font.setBold(False)
        self.font.setPointSize(self.local_prefs['DEFAULT_FONT_SIZE'])
        self.normal_fontsize = self.local_prefs['DEFAULT_FONT_SIZE']

        tooltip_font = QFont()
        tooltip_font.setFamily(self.local_prefs['DEFAULT_FONT_FAMILY'])
        tooltip_font.setPointSize(self.local_prefs['DEFAULT_FONT_SIZE'] - 1)
        tooltip_font.setBold(False)
        QToolTip.setFont(tooltip_font)
        del tooltip_font
#---------------------------------------------------------------------------------------------------------------------------------------
    def build_stylesheets(self,qtapp=None):

        icon_down_arrow_path = os.path.join(self.images_dir,"down_arrow.png")  # 16x16 for qcombobox and qheaderview
        icon_down_arrow_path = icon_down_arrow_path.replace(os.sep,'/')
        icon_up_arrow_path = os.path.join(self.images_dir,"up_arrow.png")  # 16x16 for qheaderview
        icon_up_arrow_path = icon_up_arrow_path.replace(os.sep,'/')

        s = """QToolTip {
                    color: #000000;
                    background-color: #ffffe6;
                    border: 1px solid black; } \n
                    QMenu {
                    color: #000000;
                    background-color: white;
                    border: 2px solid white; }  \n
                QInputDialog {color: #000000; background-color: #fdfdfd ; }  \n
                QComboBox {
                    color: #000000;
                    background-color: white;
                    border: 1px solid gray;
                    border-radius: 3px;
                    padding: 1px 18px 1px 3px;
                    min-width: 12em; } \n
                QComboBox:editable { background: white; } \n
                QComboBox:on { /* shift the text when the popup opens */
                    padding-top: 3px;
                    padding-left: 4px; }\n
                QComboBox::drop-down {
                    color: #000000;
                    background-color: white;
                    subcontrol-origin: padding;
                    subcontrol-position: top right;
                    width: 30px;
                    border-left-width: 1px;
                    border-left-color: darkgray;
                    border-left-style: solid;
                    border-top-right-radius: 3px; /* same radius as the QComboBox */
                    border-bottom-right-radius: 3px; }\n
                QComboBox QAbstractItemView {
                    border: 1px solid darkgray;
                    selection-background-color: darkblue; }\n
                QComboBox:focus {
                    color: #000080; }\n
                QComboBox::down-arrow {
                """
        s = s + "image: url(" + icon_down_arrow_path + "); "
        s = s + """
                    width: 6px;
                    height: 6px; }\n
                QComboBox::down-arrow:on { /* shift the arrow down a little temporarily when popup is open */
                    top: 2px;
                    left: 2px; }\n
                QPushButton { background-color: #fdfdfd;}\n
                QPushButton:pressed { background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1,   stop:0 rgba(60, 186, 162, 255), stop:1 rgba(98, 211, 162, 255));}\n
                QPushButton:checked { border-style: inset; border: 1px solid #4C2600; border-bottom-width: 1px; border-right-width: 1px;border-top-width: 0px; border-left-width: 0px; background-color: rgb(255, 190, 127);}\n
                """
        s = s + """
                QHeaderView::section {
                    background-color: white;
                    color: black;
                    padding-left: 2px;
                    border: 1px solid #fdfdfd;}\n
                QHeaderView::section:hover {
                    color: white;
                    background-color: lightblue;}\n
                QHeaderView::section:pressed {
                    color: white;
                    background-color: blue;}\n
                /* style the sort indicators */
                QHeaderView::down-arrow {
                """
        s = s + "image: url(" + icon_down_arrow_path + "); "
        s = s + """
                    width: 10px;
                    height: 10px; }\n
                """
        s = s + """
                QHeaderView::up-arrow {
                """
        s = s + "image: url(" + icon_up_arrow_path + "); "
        s = s + """
                    width: 10px;
                    height: 10px; }\n
                """
        s = s + """
                QMenu { background-color: white; selection-background-color: darkblue; border: 1px solid darkgray; margin: 0px; }\n
                QMenu::item { padding: 2px 25px 2px 20px; border: 1px solid transparent; }\n
                QMenu::item:selected { border-color: midnightblue;  background: lightblue; }\n
                QMenu::separator { height: 1px; background: darkgray; margin-left: 5px; margin-right: 5px; }\n
                QMenu::indicator { width: 13px; height: 13px; }\n
                """
        s = s + """
                QSlider::groove:horizontal {
                    border: 1px solid #999999;
                    height: 8px; /* the groove expands to the size of the slider by default. by giving it a height, it has a fixed size */
                    background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #08972E, stop:1 #0DD743);
                    margin: 2px 0;}\n
                QSlider::handle:horizontal {
                    background: qlineargradient(x1:0, y1:0, x2:1, y2:1, stop:0 #2F9049, stop:1 #1F532D);
                    border: 1px solid #5c5c5c;
                    width: 18px;
                    margin: -2px 0; /* handle is placed by default on the contents rect of the groove. Expand outside the groove */
                    border-radius: 3px;}\n
                """
        q = """
                QTextEdit { color: [TC]; background-color: [BC];}\n
            """
        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        bc = self.local_prefs['CALIBRESPY_BACKGROUND_COLOR']
        q = q.replace("[TC]",tc)
        q = q.replace("[BC]",bc)

        s = s + q

        is_dark_theme = False

        if bc == "black":
            is_dark_theme = True

        if is_dark_theme:
            from calibre_plugins.calibrespy.ui_theme import QDarkPalette

            self.style_text = ""

            if (not self.is_cli) or (qtapp is None):
                dark_palette = QDarkPalette()
                self.setPalette(dark_palette)
                self.setStyleSheet(
                "QToolTip { color: #ffffff; background-color: #2a82da; border: 1px "
                "solid white; }")
            else:
                qtapp.setStyle("Fusion")
                dark_palette = QDarkPalette()
                qtapp.setPalette(dark_palette)
                qtapp.setStyleSheet(
                "QToolTip { color: #ffffff; background-color: #2a82da; border: 1px "
                "solid white; }")
        else:
            self.style_text = s     #also passed to all of the other CalibreSpy-created qdialogs

            if (not self.is_cli) or (qtapp is None):
                self.setStyleSheet(self.style_text)      #Standard Calibre's QApplication's default style cannot safely be changed for just CalibreSpy (only influenced to a degree).
            else:
                qtapp.setStyleSheet(self.style_text)  #CalibreSpy's QApplication's sole style.
#---------------------------------------------------------------------------------------------------------------------------------------
    def initialize_scroll_area(self):
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_dialog = QVBoxLayout()
        self.layout_dialog.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout_dialog)
        #--------------------------------------------------
        self.scroll_area_dialog = QScrollArea()
        self.scroll_area_dialog.setAlignment(Qt.AlignCenter)
        self.scroll_area_dialog.setWidgetResizable(True)
        self.scroll_area_dialog.ensureVisible(600,600)

        self.layout_dialog.addWidget(self.scroll_area_dialog)       # the scroll area is now the child of the parent of self.layout_dialog

        # NOTE: the self.scroll_area_dialog.setWidget(self.scroll_widget) is at the end of the init() AFTER all children have been created and assigned to a layout...
        #--------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_dialog.addWidget(self.scroll_widget)           # causes automatic reparenting of QWidget to the parent of self.layout_dialog, which is:  self .
        #--------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setAlignment(Qt.AlignCenter)

        self.scroll_widget.setLayout(self.layout_frame)                 # causes automatic reparenting of any widget later added below to the above parent
#---------------------------------------------------------------------------------------------------------------------------------------
    def create_filtering_widgets(self):
        #--------------------------------------------------
        self.layout_filters = QVBoxLayout()
        self.layout_filters.setAlignment(Qt.AlignCenter)
        self.layout_frame.addLayout(self.layout_filters)
        #--------------------------------------------------
        self.layout_filters_row1 = QHBoxLayout()
        self.layout_filters_row1.setAlignment(Qt.AlignCenter)
        self.layout_filters.addLayout(self.layout_filters_row1)
        #--------------------------------------------------
        self.font.setPointSize(self.normal_fontsize - 1)
        #--------------------------------------------------
        filter_minimum = 200
        filter_maximum = 200
        #--------------------------------------------------
        authors_tip = "<p style='white-space:wrap'>Filter for 'Author(s)', which is populated from Calibre's 'Author Sort'."

        self.authors_filter_combobox = QComboBox()
        self.authors_filter_combobox.setEditable(False)
        self.authors_filter_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)   # { AdjustToMinimumContentsLengthWithIcon }
        self.authors_filter_combobox.setFont(self.font)
        self.authors_filter_combobox.setMaxVisibleItems(20)
        self.authors_filter_combobox.setMinimumWidth(filter_minimum)
        self.authors_filter_combobox.setMaximumWidth(filter_maximum)
        self.authors_filter_combobox.setToolTip(authors_tip)
        self.layout_filters_row1.addWidget(self.authors_filter_combobox)

        self.authors_filter_combobox.addItem(DEFAULT_AUTHORS)
        self.authors_filter_combobox_full_list = []
        self.authors_filter_combobox_full_list.append(DEFAULT_AUTHORS)
        #--------------------------------------------------
        title_tip = "<p style='white-space:wrap'>Filter for 'Title'.        "

        self.title_filter_combobox = QComboBox()
        self.title_filter_combobox.setEditable(False)
        self.title_filter_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)
        self.title_filter_combobox.setFont(self.font)
        self.title_filter_combobox.setMaxVisibleItems(20)
        self.title_filter_combobox.setMinimumWidth(filter_minimum)
        self.title_filter_combobox.setMaximumWidth(filter_maximum)
        self.title_filter_combobox.setToolTip(title_tip)
        self.layout_filters_row1.addWidget(self.title_filter_combobox)

        self.title_filter_combobox.addItem(DEFAULT_TITLE)
        self.title_filter_combobox_full_list = []
        self.title_filter_combobox_full_list.append(DEFAULT_TITLE)
        #--------------------------------------------------
        #--------------------------------------------------
        series_tip = "<p style='white-space:wrap'>Filter for 'Series'.      "

        self.series_filter_combobox = QComboBox()
        self.series_filter_combobox.setEditable(False)
        self.series_filter_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)
        self.series_filter_combobox.setFont(self.font)
        self.series_filter_combobox.setMaxVisibleItems(20)
        self.series_filter_combobox.setMinimumWidth(filter_minimum)
        self.series_filter_combobox.setMaximumWidth(filter_maximum)
        self.series_filter_combobox.setToolTip(series_tip)
        self.layout_filters_row1.addWidget(self.series_filter_combobox)

        self.series_filter_combobox.addItem(DEFAULT_SERIES)
        self.series_filter_combobox.addItem(DEFAULT_SERIES_NONE)
        self.series_filter_combobox_full_list = []
        self.series_filter_combobox_full_list.append(DEFAULT_SERIES)
        self.series_filter_combobox_full_list.append(DEFAULT_SERIES_NONE)
        #--------------------------------------------------
        #--------------------------------------------------
        tags_tip = "<p style='white-space:wrap'>Filter for 'Tags'.      "

        self.tags_filter_combobox = QComboBox()
        self.tags_filter_combobox.setEditable(False)
        self.tags_filter_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)
        self.tags_filter_combobox.setFont(self.font)
        self.tags_filter_combobox.setMaxVisibleItems(20)
        self.tags_filter_combobox.setMinimumWidth(filter_minimum)
        self.tags_filter_combobox.setMaximumWidth(filter_maximum)
        self.tags_filter_combobox.setToolTip(tags_tip)
        self.layout_filters_row1.addWidget(self.tags_filter_combobox)

        self.tags_filter_combobox.addItem(DEFAULT_TAGS)
        self.tags_filter_combobox.addItem(DEFAULT_TAGS_NONE)
        self.tags_filter_combobox_full_list = []
        self.tags_filter_combobox_full_list.append(DEFAULT_TAGS)
        self.tags_filter_combobox_full_list.append(DEFAULT_TAGS_NONE)
        #--------------------------------------------------
        #--------------------------------------------------
        publisher_tip = "<p style='white-space:wrap'>Filter for 'publisher'.     "

        self.publisher_filter_combobox = QComboBox()
        self.publisher_filter_combobox.setEditable(False)
        self.publisher_filter_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)
        self.publisher_filter_combobox.setFont(self.font)
        self.publisher_filter_combobox.setMaxVisibleItems(20)
        self.publisher_filter_combobox.setMinimumWidth(filter_minimum)
        self.publisher_filter_combobox.setMaximumWidth(filter_maximum)
        self.publisher_filter_combobox.setToolTip(publisher_tip)
        self.layout_filters_row1.addWidget(self.publisher_filter_combobox)

        self.publisher_filter_combobox.addItem(DEFAULT_PUBLISHER)
        self.publisher_filter_combobox.addItem(DEFAULT_PUBLISHER_NONE)
        self.publisher_filter_combobox_full_list = []
        self.publisher_filter_combobox_full_list.append(DEFAULT_PUBLISHER)
        self.publisher_filter_combobox_full_list.append(DEFAULT_PUBLISHER_NONE)

        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_filters_row2 = QHBoxLayout()
        self.layout_filters_row2.setAlignment(Qt.AlignCenter)

        self.layout_filters.addLayout(self.layout_filters_row2)
        #--------------------------------------------------
        self.font.setPointSize(self.normal_fontsize - 1)
        self.font.setBold(True)

        self.authors_count_qlabel = QLabel("")
        self.authors_count_qlabel.setAlignment(Qt.AlignCenter)
        self.authors_count_qlabel.setMinimumWidth(filter_minimum)
        self.authors_count_qlabel.setMaximumWidth(300)
        self.layout_filters_row2.addWidget(self.authors_count_qlabel)

        self.title_count_qlabel = QLabel("")
        self.title_count_qlabel.setAlignment(Qt.AlignCenter)
        self.title_count_qlabel.setMinimumWidth(filter_minimum)
        self.title_count_qlabel.setMaximumWidth(300)
        self.layout_filters_row2.addWidget(self.title_count_qlabel)

        self.series_count_qlabel = QLabel("")
        self.series_count_qlabel.setAlignment(Qt.AlignCenter)
        self.series_count_qlabel.setMinimumWidth(filter_minimum)
        self.series_count_qlabel.setMaximumWidth(300)
        self.layout_filters_row2.addWidget(self.series_count_qlabel)

        self.tags_count_qlabel = QLabel("")
        self.tags_count_qlabel.setAlignment(Qt.AlignCenter)
        self.tags_count_qlabel.setMinimumWidth(filter_minimum)
        self.tags_count_qlabel.setMaximumWidth(300)
        self.layout_filters_row2.addWidget(self.tags_count_qlabel)

        self.publisher_count_qlabel = QLabel("")
        self.publisher_count_qlabel.setAlignment(Qt.AlignCenter)
        self.publisher_count_qlabel.setMinimumWidth(filter_minimum)
        self.publisher_count_qlabel.setMaximumWidth(300)
        self.layout_filters_row2.addWidget(self.publisher_count_qlabel)

        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)
        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_clear_filters = QHBoxLayout()
        self.layout_clear_filters.setAlignment(Qt.AlignCenter)
        self.layout_frame.addLayout(self.layout_clear_filters)

        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)

        self.push_button_clear_all_filters = QPushButton("Clear All Filters", self)
        self.push_button_clear_all_filters.setFont(self.font)
        self.push_button_clear_all_filters.setToolTip("<p style='white-space:wrap'>Remove all filters, show all possible rows, and return to the original sort order.")
        self.push_button_clear_all_filters.setMinimumWidth(filter_minimum - 50)
        self.push_button_clear_all_filters.setMaximumWidth(filter_maximum - 50)
        self.push_button_clear_all_filters.clicked.connect(self.clear_all_filters)
        self.layout_clear_filters.addWidget(self.push_button_clear_all_filters)

        self.filter_history_combobox = QComboBox()
        self.filter_history_combobox.setEditable(False)
        self.filter_history_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)
        self.filter_history_combobox.setFont(self.font)
        self.filter_history_combobox.setMaxVisibleItems(20)
        self.filter_history_combobox.setMaxCount(HISTORY_MAXIMUM_VALUES)
        self.filter_history_combobox.setMinimumWidth(filter_minimum)
        self.filter_history_combobox.setMaximumWidth(filter_maximum)
        t = "<p style='white-space:wrap'>Reapply historical filtering actions."
        self.filter_history_combobox.setToolTip(t)
        self.layout_clear_filters.addWidget(self.filter_history_combobox)

        self.clear_filters_spacer_1_label = QLabel("")
        self.clear_filters_spacer_1_label.setMinimumWidth(20)
        self.clear_filters_spacer_1_label.setMaximumWidth(20)
        self.layout_clear_filters.addWidget(self.clear_filters_spacer_1_label)

        self.search_expressions_combobox = QComboBox()
        self.search_expressions_combobox.setEditable(True)
        self.search_expressions_combobox.setDuplicatesEnabled(False)
        self.search_expressions_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.search_expressions_combobox.setFrame(True)
        self.search_expressions_combobox.setFont(self.font)
        self.search_expressions_combobox.setMaxVisibleItems(20)
        self.search_expressions_combobox.setMaxCount(HISTORY_MAXIMUM_VALUES)
        self.search_expressions_combobox.setMinimumWidth(filter_minimum - 75)
        self.search_expressions_combobox.setMaximumWidth(filter_maximum - 75)
        t = "<p style='white-space:wrap'>Regular Expressions for searching the selected column for currently displayed rows."
        self.search_expressions_combobox.setToolTip(t)
        self.layout_clear_filters.addWidget(self.search_expressions_combobox)

        self.search_targets_combobox = QComboBox()
        self.search_targets_combobox.setEditable(True)
        self.search_targets_combobox.setDuplicatesEnabled(False)
        self.search_targets_combobox.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLengthWithIcon)
        self.search_targets_combobox.setFrame(True)
        self.search_targets_combobox.setFont(self.font)
        self.search_targets_combobox.setMaxVisibleItems(20)
        self.search_targets_combobox.setMaxCount(HISTORY_MAXIMUM_VALUES)
        self.search_targets_combobox.setMinimumWidth(filter_minimum - 75)
        self.search_targets_combobox.setMaximumWidth(filter_maximum - 75)
        t = "<p style='white-space:wrap'>Column to be searched with the selected Regular Expression."
        self.search_targets_combobox.setToolTip(t)
        self.layout_clear_filters.addWidget(self.search_targets_combobox)

        self.push_button_do_search = QPushButton("", self)  # icon deferred until it becomes available
        self.push_button_do_search.setFont(self.font)
        self.push_button_do_search.setToolTip("<p style='white-space:wrap'>Search the selected column using the selected Regular Expression.")
        self.push_button_do_search.setMinimumWidth(35)
        self.push_button_do_search.setMaximumWidth(35)
        self.push_button_do_search.clicked.connect(self.do_search)
        self.layout_clear_filters.addWidget(self.push_button_do_search)

        self.clear_filters_spacer_2_label = QLabel("")
        self.clear_filters_spacer_2_label.setMinimumWidth(20)
        self.clear_filters_spacer_2_label.setMaximumWidth(20)
        self.layout_clear_filters.addWidget(self.clear_filters_spacer_2_label)

        self.push_button_virtual_libraries = QPushButton("VL", self)
        self.push_button_virtual_libraries.setFont(self.font)
        t = "<p style='white-space:wrap'>Apply Calibre Virtual Libraries within CalibreSpy.\
            <br><br>VLs from the Calibre GUI will be transformed into CalibreSpy read-only SQL Queries, then executed using the Calibre metadata.db database itself.\
            <br><br>Columns, whether Standard or Custom, do <b>not</b> have to be displayed by CalibreSpy for them to be used in a VL Query.\
            <br><br>Boolean operators supported:  AND; OR; NOT (and AND NOT).\
            <br><br>Implicit ANDs are supported for VL criteria with no round-brackets, '('.  You may not mix implicit ANDs together with explicit ANDs, ORs and NOTs.\
            <br><br>Relational operators supported:  =; !=; !=; &lt;; &lt;=; &gt;; &gt;=.\
            <br><br>Example: #orig_title:" + '"=My Orig Title"' + " and (#ddc:true or #lcc:true ) and (rating:&lt;=3 or rating:&gt;=6) and (#abc_hierarchy:true or #abc_numeric:true)\
            <br><br>Example: (((#ddc:true or #lcc:true ) and (rating:&lt;=3 or rating:&gt;=6)) not (#abc_hierarchy:false or #abc_numeric:false))\
            <br><br>Example: (pubdate:true or vl:my_vl_name1) and vl:my_vl_name2 and (#ddc:true or #lcc:true ) and not vl:my_vl_name3\
            <br><br>A VL may contain vl: terms that themselves contain other vl: terms.\
            <br><br>@User Category VL criteria supported: '@UserCategory:true'; '@UserCategory:false'; '@UserCategory.SubCategory:true';  '@UserCategory.SubCategory:false'\
            <br><br>Any '@UserCategory' term must be the sole criterion for any VL specifying a User Category.  \
            <br><br>The prefix 'search:' plus a 'Saved Search' name must be the sole criterion for any VL specifying a Saved Search.  Example VL:  search:mysavedsearch1    \
            <br><br>Only VLs that CalibreSpy believes it is able to transform will be offered as a choice in the VL dropdown selection menu.\
            <br><br>VL criteria that cannot be used since they are 'virtual' columns that do not really exist in the metadata:  'marked:' and 'ondevice:'.\
            <br> "
        self.push_button_virtual_libraries_tooltip = t
        self.push_button_virtual_libraries.setToolTip(t)
        self.push_button_virtual_libraries.setMinimumWidth(35)
        self.push_button_virtual_libraries.setMaximumWidth(35)
        self.push_button_virtual_libraries.clicked.connect(self.apply_virtual_libraries)
        self.layout_clear_filters.addWidget(self.push_button_virtual_libraries)

        self.push_button_virtual_libraries.setCheckable(True)

        self.push_button_do_duplicates_search = QPushButton("", self)  # icon deferred until it becomes available
        self.push_button_do_duplicates_search.setFont(self.font)
        self.push_button_do_duplicates_search.setMinimumWidth(35)
        self.push_button_do_duplicates_search.setMaximumWidth(35)
        self.push_button_do_duplicates_search.clicked.connect(self.do_duplicates_search)
        self.layout_clear_filters.addWidget(self.push_button_do_duplicates_search)
        self.push_button_do_duplicates_search.setToolTip("<p style='white-space:wrap'>Search for Duplicate Books Specifying Any 2 Columns (Standard or Custom).")

        self.push_button_do_interbook_search = QPushButton("", self)  # icon deferred until it becomes available
        self.push_button_do_interbook_search.setFont(self.font)
        self.push_button_do_interbook_search.setMinimumWidth(35)
        self.push_button_do_interbook_search.setMaximumWidth(35)
        self.push_button_do_interbook_search.clicked.connect(self.do_interbook_search)
        self.layout_clear_filters.addWidget(self.push_button_do_interbook_search)
        self.push_button_do_interbook_search.setToolTip("<p style='white-space:wrap'>Perform a Cross-Book (Inter-Book) Search that answers this type of query:                                           \
                                                                                                                            <br><br><b>'Find all of the books that have the same [COLUMN 1] and the same [COLUMN 2],\
                                                                                                                            ignoring all other columns.  I want to <i>compare each book to every other book</i>. \
                                                                                                                            I do NOT want to have to provide any specific value to search for. Let CalibreSpy do the work for me'.</b>\
                                                                                                                            <br><br>EXAMPLE:                                                                                                                                              \
                                                                                                                            <br>I have many books by many Authors both written in their original languages and translated to English, so I <b>cannot</b> specify any particular Author to search for. \
                                                                                                                            I want to display all books which have the same Author and same Custom Column 'original_title'. The Author is crucial for comparison,\
                                                                                                                            as the same Title/Custom Column 'original_title' can have multiple authors. This query should return only books for which there are translations.\
                                                                                                                            <br><br>Example of returned results:\
                                                                                                                            <br>Author ------------------ Title -------------------------  Custom Column 'original_title'                \
                                                                                                                            <br>\
                                                                                                                            <br>Jules Verne ----- Five weeks in ballon ----------------- Cinq semaines en ballon                          \
                                                                                                                            <br>Jules Verne ----- Cinq semaines en ballon ------------ Cinq semaines en ballon                          \
                                                                                                                            <br>Victor Hugo ---- Notre-Dame de Paris ---------------- Notre-Dame de Paris                               \
                                                                                                                            <br>Victor Hugo ---- The Hunchback of Notre-Dame ----- Notre-Dame de Paris                               \
                                                                                                                            ")


        self.push_button_do_contains_search = QPushButton("", self)  # icon deferred until it becomes available
        self.push_button_do_contains_search.setFont(self.font)
        self.push_button_do_contains_search.setMinimumWidth(35)
        self.push_button_do_contains_search.setMaximumWidth(35)
        self.push_button_do_contains_search.clicked.connect(self.do_contains_search)
        self.layout_clear_filters.addWidget(self.push_button_do_contains_search)
        self.push_button_do_contains_search.setToolTip("<p style='white-space:wrap'>Find metadata columns that 'contain' another metadata column's value (i.e., a mixture).\
                                                                                                                                      <br><br>Example:  Find all 'Titles' that have the 'Series' contained within them.")

        self.push_button_do_sqlquery_search = QPushButton("", self)  # icon deferred until it becomes available
        self.push_button_do_sqlquery_search.setFont(self.font)
        self.push_button_do_sqlquery_search.setMinimumWidth(35)
        self.push_button_do_sqlquery_search.setMaximumWidth(35)
        self.push_button_do_sqlquery_search.clicked.connect(self.do_sqlquery_search)
        self.layout_clear_filters.addWidget(self.push_button_do_sqlquery_search)
        self.push_button_do_sqlquery_search.setToolTip("<p style='white-space:wrap'>Perform a 'SQL Query' against the Calibre database.  Always 'read-only', of course.")

        self.push_button_drop_search_results = QPushButton("", self)   # icon deferred until it becomes available
        self.push_button_drop_search_results.setFont(self.font)
        t = "<p style='white-space:wrap'>'Drop Search Results': Drop the full paths for Calibre books that already exist in the current Calibre library anywhere within the DSR pop-up window.\
                                                              <br><br>Almost all search applications create search results that can be dragged from the application to the DSR pop-up window, and dropped.\
                                                               <br><br>In Windows, some search applications might need to be 'Run as Administrator' to be able to 'see' the DSR window as an active drop target."
        self.push_button_drop_search_results.setToolTip(t)
        self.push_button_drop_search_results.setMinimumWidth(35)
        self.push_button_drop_search_results.setMaximumWidth(35)
        self.push_button_drop_search_results.clicked.connect(self.drop_search_results)
        self.layout_clear_filters.addWidget(self.push_button_drop_search_results)

        self.push_button_invert_selection = QPushButton("", self)   # icon deferred until it becomes available
        self.push_button_invert_selection.setFont(self.font)
        self.push_button_invert_selection.setToolTip("<p style='white-space:wrap'>Invert the currently displayed rows (i.e., show what is currently hidden, and hide what is currently shown).")
        self.push_button_invert_selection.setMinimumWidth(35)
        self.push_button_invert_selection.setMaximumWidth(35)
        self.push_button_invert_selection.clicked.connect(self.invert_selection)
        self.layout_clear_filters.addWidget(self.push_button_invert_selection)

        self.listening_text_label = QLabel("")
        self.listening_text_label.setMinimumWidth(35)
        self.listening_text_label.setMaximumWidth(35)
        self.layout_clear_filters.addWidget(self.listening_text_label)
        self.listening_text_label.show()

        self.listening_icon_label = QLabel("")  # pixmap deferred until it becomes available
        self.listening_icon_label.setToolTip("<p style='white-space:wrap'>This CalibreSpy window is listening for 'Clipboard Search Queries'.")
        self.listening_icon_label.setMinimumWidth(35)
        self.listening_icon_label.setMaximumWidth(35)
        self.layout_clear_filters.addWidget(self.listening_icon_label)
        self.listening_icon_label.hide()

        self.authors_filtering_is_in_play = False
        self.title_filtering_is_in_play = False
        self.series_filtering_is_in_play = False
        self.tags_filtering_is_in_play = False
        self.publisher_filtering_is_in_play = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def create_matrix(self):

        self.set_column_numbers()
        self.get_table_custom_columns()

        #--------------------------------------------------
        #--------------------------------------------------
        self.n_matrix_rows = len(self.book_table_dict)
        #--------------------------------------------------
        mytitle = 'CalibreSpy      '+ "Library: " + self.library_path + "     Books: " + as_unicode(self.n_matrix_rows)
        self.setWindowTitle(mytitle)
        #--------------------------------------------------
        column_label_list = []
        c = self.local_prefs['AUTHORS_COLUMN']
        row = c,AUTHORS_HEADING
        column_label_list.append(row)
        c = self.local_prefs['TITLE_COLUMN']
        row = c,TITLE_HEADING
        column_label_list.append(row)
        c = self.local_prefs['SERIES_COLUMN']
        row = c,SERIES_HEADING
        column_label_list.append(row)
        c = self.local_prefs['INDEX_COLUMN']
        row = c,INDEX_HEADING
        column_label_list.append(row)
        c = self.local_prefs['TAGS_COLUMN']
        row = c,TAGS_HEADING
        column_label_list.append(row)
        c = self.local_prefs['PUBLISHED_COLUMN']
        row = c,PUBLISHED_HEADING
        column_label_list.append(row)
        c = self.local_prefs['PUBLISHER_COLUMN']
        row = c,PUBLISHER_HEADING
        column_label_list.append(row)
        c = self.local_prefs['LANGUAGES_COLUMN']
        row = c,LANGUAGES_HEADING
        column_label_list.append(row)
        c = self.local_prefs['ADDED_COLUMN']
        row = c,ADDED_HEADING
        column_label_list.append(row)
        c = self.local_prefs['MODIFIED_COLUMN']
        row = c,MODIFIED_HEADING
        column_label_list.append(row)
        c = self.local_prefs['PATH_COLUMN']
        row = c,PATH_HEADING
        column_label_list.append(row)
        #~ -------------------------------------------------------------------------------
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_1'] == 1:  #if not 1, then none.
            self.custom_columns_are_active = True
        else:
            self.custom_columns_are_active = False

        self.cc1_is_active = False
        self.cc2_is_active = False
        self.cc3_is_active = False
        self.cc4_is_active = False
        self.cc5_is_active = False
        self.cc6_is_active = False
        self.cc7_is_active = False
        self.cc8_is_active = False
        self.cc9_is_active = False

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_1'] == 1:
            self.cc1_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_2'] == 1:
            self.cc2_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_3'] == 1:
            self.cc3_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_4'] == 1:
            self.cc4_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_5'] == 1:
            self.cc5_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_6'] == 1:
            self.cc6_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_7'] == 1:
            self.cc7_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_8'] == 1:
            self.cc8_is_active = True
        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_9'] == 1:
            self.cc9_is_active = True
        #~ -------------------------------------------------------------------------------
        self.n_unused_custom_column_slots = 0

        if self.cc1_is_active:
            if 'CUSTOM_COLUMN_1_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_1_COLUMN']
                if CUSTOM_COLUMN_1 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_1]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]  #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc2_is_active:
            if 'CUSTOM_COLUMN_2_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_2_COLUMN']
                if CUSTOM_COLUMN_2 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_2]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]  #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc3_is_active:
            if 'CUSTOM_COLUMN_3_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_3_COLUMN']
                if CUSTOM_COLUMN_3 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_3]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc4_is_active:
            if 'CUSTOM_COLUMN_4_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_4_COLUMN']
                if CUSTOM_COLUMN_4 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_4]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc5_is_active:
            if 'CUSTOM_COLUMN_5_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_5_COLUMN']
                if CUSTOM_COLUMN_5 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_5]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc6_is_active:
            if 'CUSTOM_COLUMN_6_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_6_COLUMN']
                if CUSTOM_COLUMN_6 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_6]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc7_is_active:
            if 'CUSTOM_COLUMN_7_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_7_COLUMN']
                if CUSTOM_COLUMN_7 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_7]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc8_is_active:
            if 'CUSTOM_COLUMN_8_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_8_COLUMN']
                if CUSTOM_COLUMN_8 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_8]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1

        if self.cc8_is_active:
            if 'CUSTOM_COLUMN_9_COLUMN' in self.local_prefs:
                c = self.local_prefs['CUSTOM_COLUMN_9_COLUMN']
                if CUSTOM_COLUMN_9 in self.custom_column_assignment_dict:
                    label = self.custom_column_assignment_dict[CUSTOM_COLUMN_9]
                    if label in self.local_prefs:
                        label = self.local_prefs[label]   #    name of  cc #label for horizontal column header label
                        row = c,label
                        column_label_list.append(row)
                    else:
                        self.n_unused_custom_column_slots += 1
                else:
                    self.n_unused_custom_column_slots += 1
            else:
                self.n_unused_custom_column_slots += 1
        else:
            self.n_unused_custom_column_slots += 1
        #~ ----------------------------------------------------------------
        #~ ------------- add new columns below -----------------------
        non_original_calibrespy_column_dict = {}
        #~ --------------------------------
        if self.local_prefs['CALIBRESPY_LOAD_IDENTIFIERS'] == 1:    # Version 1.0.43 - Identifiers Added
            if self.identifiers_column == IDENTIFIERS_COLUMN:  #original default value from config.py
                self.identifiers_column = self.identifiers_column - self.n_unused_custom_column_slots   #! there is (or could be) a "gap"...
                if self.identifiers_column in self.cc_column_set:
                    self.cc_column_set.remove(self.identifiers_column)
                self.local_prefs['IDENTIFIERS_COLUMN'] = self.identifiers_column  #final value...
            c = self.local_prefs['IDENTIFIERS_COLUMN']  #final value...
            non_original_calibrespy_column_dict[c] = IDENTIFIERS_COLUMN  #so we can "close the gap" caused by custom columns...if there is one...
            row = c,IDENTIFIERS_HEADING
            column_label_list.append(row)
            self.identifiers_loaded = True
        else:
            self.identifiers_loaded = False
        #~ --------------------------------
        if self.local_prefs['CALIBRESPY_LOAD_RATINGS'] == 1:    # Version 1.0.52 - Ratings Added
            if self.ratings_column == RATINGS_COLUMN:  #original default value from config.py
                self.ratings_column = self.ratings_column - self.n_unused_custom_column_slots   #! there is (or could be) a "gap"...
                if self.ratings_column in self.cc_column_set:
                    self.cc_column_set.remove(self.ratings_column)
                self.local_prefs['RATINGS_COLUMN'] = self.ratings_column  #final value...
            c = self.local_prefs['RATINGS_COLUMN']  #final value...
            non_original_calibrespy_column_dict[c] = RATINGS_COLUMN  #so we can "close the gap" caused by custom columns...if there is one...
            row = c,RATINGS_HEADING
            column_label_list.append(row)
            self.ratings_loaded = True
        else:
            self.ratings_loaded = False
        #~ --------------------------------



        #~ ------------- add new columns above -----------------------
        #~ ----------------------------------------------------------------
        #~ -------------------------------------------------------------------------------
        #~ the customizing library-specific prefs dialog initializes using whatever is in self.local_prefs for column numbers.
        #~ successful validation of assigned columns in *that* dialog starts *here*.
        #~ there can be no 'gaps' in column numbers, starting with 0 and ending with MATRIX_MAXIMUM_COLUMNS_COUNT - 1, nor any duplicate column numbers.
        #~ the book matrix is created with exactly N columns, and is defined later in this same function:   self.matrix = QTableWidget(self.n_matrix_rows,self.n_matrix_columns)
        #~ any metadata with a column number > N will simply be ignored, as that column number will not exist in self.matrix.
        #~ custom columns are optional, so cause 'gaps' in the column numbering of active columns. there can be no 'gaps' in column numbers.
        #~ the beginning of that gap *must* be filled by self.identifiers_column (or future additional columns) if self.identifiers_column is active (data is being loaded for it).
        #~ remember: the default column numbers defined in config.sys may *never* be changed; to do so would corrupt all existing library-specific settings for CalibreSpy.
        #~ -------------------------------------------------------------------------------
        if not self.cc1_is_active:
            if self.local_prefs['CUSTOM_COLUMN_1_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_1_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_1_COLUMN']]  #use the Identifiers default column...no duplicates...
        if not self.cc2_is_active:
            if self.local_prefs['CUSTOM_COLUMN_2_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_2_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_2_COLUMN']]
        if not self.cc3_is_active:
            if self.local_prefs['CUSTOM_COLUMN_3_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_3_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_3_COLUMN']]
        if not self.cc4_is_active:
            if self.local_prefs['CUSTOM_COLUMN_4_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_4_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_4_COLUMN']]
        if not self.cc5_is_active:
            if self.local_prefs['CUSTOM_COLUMN_5_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_5_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_5_COLUMN']]
        if not self.cc6_is_active:
            if self.local_prefs['CUSTOM_COLUMN_6_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_6_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_6_COLUMN']]
        if not self.cc7_is_active:
            if self.local_prefs['CUSTOM_COLUMN_7_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_7_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_7_COLUMN']]
        if not self.cc8_is_active:
            if self.local_prefs['CUSTOM_COLUMN_8_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_8_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_8_COLUMN']]
        if not self.cc9_is_active:
            if self.local_prefs['CUSTOM_COLUMN_9_COLUMN'] in non_original_calibrespy_column_dict:
                self.local_prefs['CUSTOM_COLUMN_9_COLUMN'] = non_original_calibrespy_column_dict[self.local_prefs['CUSTOM_COLUMN_9_COLUMN']]
        #~ -------------------------------------------------------------------------------
        del non_original_calibrespy_column_dict
        #~ --------------------------------
        #~ --------------------------------
        self.column_number_name_dict = {}  #e.g.   0 = "Authors"
        column_label_list.sort()
        column_labels_list = []
        column_number_list = []
        msg = None
        last = -1
        for row in column_label_list:
            col,label = row
            if col != last + 1:
                msg = "Data Gap Found for Column Assignment: " + as_unicode(col) + "      name: " + label + \
                "<br><br>Please correct your Column Assignments in Customizing so the stated metadata may be displayed for you in CalibreSpy.\
                <br><br> Refer to the Customizing ToolTips for a detailed explanation of a 'Data Gap'. \
                <br><br><i>Some other metadata Columns may be mislabeled or missing entirely until this error is corrected.</i>"
            last = col
            if not col in column_number_list:
                column_labels_list.append(label)
                column_number_list.append(col)
                self.column_number_name_dict[col] = label
            else:
                if DEBUG:
                    msg = "SYSTEM ERROR: column numbers are corrupt.  Notify the developer of this plugin.<br><br>Duplicate column number:" + as_unicode(col) + "   name: " + label
                    print(msg)
        #END FOR
        self.details_column_label_list = column_label_list
        del column_label_list
        del column_number_list
        if msg is not None:
            error_dialog(None, _('CalibreSpy: Column Assignments of Metadata'),_(msg), show=True)
        del msg
        #--------------------------------------------------
        self.n_matrix_columns = MATRIX_MAXIMUM_COLUMNS_COUNT - self.n_unused_custom_column_slots
        #--------------------------------------------------
        self.current_row = 0
        self.current_column = 0
        #--------------------------------------------------
        #--------------------------------------------------
        self.matrix = QTableWidget(self.n_matrix_rows,self.n_matrix_columns)
        #--------------------------------------------------
        #--------------------------------------------------
        self.default_row_height = self.matrix.rowHeight(0)  # ~30    set before any other tweaks are applied to the row height
        if self.default_row_height < 15:
            self.default_row_height = 15
        self.local_prefs['DEFAULT_MATRIX_ROW_HEIGHT'] = self.default_row_height
        #--------------------------------------------------
        self.matrix.setFont(self.font)
        self.matrix.setCornerButtonEnabled(False)
        self.matrix.setSortingEnabled(False)
        self.matrix.setHorizontalHeaderLabels(column_labels_list)
        self.matrix.horizontalHeader().setVisible(True)
        self.matrix.verticalHeader().setVisible(True)
        self.layout_frame.addWidget(self.matrix)
#---------------------------------------------------------------------------------------------------------------------------------------
    def set_column_numbers(self):
        self.authors_column = self.local_prefs['AUTHORS_COLUMN']
        self.title_column = self.local_prefs['TITLE_COLUMN']
        self.series_column = self.local_prefs['SERIES_COLUMN']
        self.index_column = self.local_prefs['INDEX_COLUMN']
        self.tags_column = self.local_prefs['TAGS_COLUMN']
        self.published_column = self.local_prefs['PUBLISHED_COLUMN']
        self.publisher_column = self.local_prefs['PUBLISHER_COLUMN']
        self.languages_column = self.local_prefs['LANGUAGES_COLUMN']
        self.added_column = self.local_prefs['ADDED_COLUMN']
        self.modified_column = self.local_prefs['MODIFIED_COLUMN']
        self.path_column = self.local_prefs['PATH_COLUMN']
        self.cc1_column = self.local_prefs['CUSTOM_COLUMN_1_COLUMN']
        self.cc2_column = self.local_prefs['CUSTOM_COLUMN_2_COLUMN']
        self.cc3_column = self.local_prefs['CUSTOM_COLUMN_3_COLUMN']
        self.cc4_column = self.local_prefs['CUSTOM_COLUMN_4_COLUMN']
        self.cc5_column = self.local_prefs['CUSTOM_COLUMN_5_COLUMN']
        self.cc6_column = self.local_prefs['CUSTOM_COLUMN_6_COLUMN']
        self.cc7_column = self.local_prefs['CUSTOM_COLUMN_7_COLUMN']
        self.cc8_column = self.local_prefs['CUSTOM_COLUMN_8_COLUMN']
        self.cc9_column = self.local_prefs['CUSTOM_COLUMN_9_COLUMN']
        #~ ------------- add new columns below -----------------------
        self.identifiers_column = self.local_prefs['IDENTIFIERS_COLUMN']  # preliminary value only
        self.ratings_column = self.local_prefs['RATINGS_COLUMN']  # preliminary value only
        #~ ----------------------------------------------------------------

        self.date_column_set = set()
        self.date_column_set.add(self.published_column)
        self.date_column_set.add(self.added_column)
        self.date_column_set.add(self.modified_column)

        self.cc_column_set = set()
        self.cc_column_set.add(self.cc1_column)
        self.cc_column_set.add(self.cc2_column)
        self.cc_column_set.add(self.cc3_column)
        self.cc_column_set.add(self.cc4_column)
        self.cc_column_set.add(self.cc5_column)
        self.cc_column_set.add(self.cc6_column)
        self.cc_column_set.add(self.cc7_column)
        self.cc_column_set.add(self.cc8_column)
        self.cc_column_set.add(self.cc9_column)

        self.authors_column = int(self.authors_column)
        self.title_column = int(self.title_column)
        self.series_column = int(self.series_column)
        self.index_column = int(self.index_column)
        self.tags_column = int(self.tags_column)
        self.published_column = int(self.published_column)
        self.publisher_column = int(self.publisher_column)
        self.languages_column = int(self.languages_column)
        self.added_column = int(self.added_column)
        self.modified_column = int(self.modified_column)
        self.path_column = int(self.path_column)
        self.cc1_column = int(self.cc1_column)
        self.cc2_column = int(self.cc2_column)
        self.cc3_column = int(self.cc3_column)
        self.cc4_column = int(self.cc4_column)
        self.cc5_column = int(self.cc5_column)
        self.cc6_column = int(self.cc6_column)
        self.cc7_column = int(self.cc7_column)
        self.cc8_column = int(self.cc8_column)
        self.cc9_column = int(self.cc9_column)
        #~ ------------- add new columns below -----------------------
        self.identifiers_column = int(self.identifiers_column)  # preliminary value only...
        self.ratings_column = int(self.ratings_column)  # preliminary value only...
        #~ ----------------------------------------------------------------

        self.local_prefs['AUTHORS_COLUMN'] = self.authors_column
        self.local_prefs['TITLE_COLUMN'] = self.title_column
        self.local_prefs['SERIES_COLUMN'] = self.series_column
        self.local_prefs['INDEX_COLUMN'] = self.index_column
        self.local_prefs['TAGS_COLUMN'] = self.tags_column
        self.local_prefs['PUBLISHED_COLUMN'] =  self.published_column
        self.local_prefs['PUBLISHER_COLUMN'] = self.publisher_column
        self.local_prefs['LANGUAGES_COLUMN'] =  self.languages_column
        self.local_prefs['ADDED_COLUMN'] = self.added_column
        self.local_prefs['MODIFIED_COLUMN'] = self.modified_column
        self.local_prefs['PATH_COLUMN'] = self.path_column
        self.local_prefs['CUSTOM_COLUMN_1_COLUMN'] = self.cc1_column
        self.local_prefs['CUSTOM_COLUMN_2_COLUMN'] = self.cc2_column
        self.local_prefs['CUSTOM_COLUMN_3_COLUMN'] = self.cc3_column
        self.local_prefs['CUSTOM_COLUMN_4_COLUMN'] = self.cc4_column
        self.local_prefs['CUSTOM_COLUMN_5_COLUMN'] = self.cc5_column
        self.local_prefs['CUSTOM_COLUMN_6_COLUMN'] = self.cc6_column
        self.local_prefs['CUSTOM_COLUMN_7_COLUMN'] = self.cc7_column
        self.local_prefs['CUSTOM_COLUMN_8_COLUMN'] = self.cc8_column
        self.local_prefs['CUSTOM_COLUMN_9_COLUMN'] = self.cc9_column
        #~ ------------- add new columns below -----------------------
        self.local_prefs['IDENTIFIERS_COLUMN'] = self.identifiers_column  #preliminary value only...
        self.local_prefs['RATINGS_COLUMN'] = self.ratings_column  #preliminary value only...
        #~ ----------------------------------------------------------------

        self.save_local_prefs()
#---------------------------------------------------------------------------------------------------------------------------------------
    def load_matrix(self):
        is_valid = self.load_book_matrix()
        if not is_valid:
            print("self.load_book_matrix() failed...terminating...")
            sys.exit(0)
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def create_bottom_widgets(self):

        #--------------------------------------------------
        self.get_program_path_values()
        #--------------------------------------------------

        self.layout_bottom_boxes = QHBoxLayout()
        self.layout_bottom_boxes.setAlignment(Qt.AlignRight)  #helps tiny tablet screens...
        self.layout_frame.addLayout(self.layout_bottom_boxes)

        self.font.setBold(True)

        self.push_button_customize = QPushButton(self.icon_calibrespy,"", self)
        self.push_button_customize.setMaximumWidth(30)
        self.push_button_customize.setAutoDefault(False)
        self.push_button_customize.setDefault(False)
        self.push_button_customize.setFont(self.font)
        self.push_button_customize.setToolTip("<p style='white-space:wrap'>Library-specific Customization")
        self.push_button_customize.clicked.connect(self.customize_calibrespy)
        self.layout_bottom_boxes.addWidget(self.push_button_customize)

        self.no_customize_label = QLabel("")
        self.path_to_calibrespy_closed_icon_file = os.path.join(self.images_dir,"calibrespy_closed.png")
        self.path_to_calibrespy_closed_icon_file = self.path_to_calibrespy_closed_icon_file.replace(os.sep,'/')
        self.no_customize_label.setPixmap(QPixmap.fromImage(QImage(self.path_to_calibrespy_closed_icon_file).scaled(20, 20, Qt.KeepAspectRatio, Qt.SmoothTransformation) ) )
        self.no_customize_label.setMaximumWidth(35)
        t = "<p style='white-space:wrap'>No Library-specific Customization Allowed (Entirely Read-Only Mode).\
        <br><br>If this multi-user status is an error, then you may either (a) wait for the user-customized number of hours for orphaned locks to automatically reset,\
        or (b) run the command file and associated script that you have renamed and personalized for your OS and Library Names (using a text editor) that are found in user directory:   ...\calibre\plugins\calibrespy_cli\CalibreSpySettings__remove_multiuser_status_from_db\...\
        <br><br>Remember:  CalibreSpy <b><i>must</i></b> be properly exited using the Exit Button, or this will happen again.\
        <br><br>Do not simply close any command window or shut down your PC without first properly exiting from CalibreSpy."
        self.no_customize_label.setToolTip(t)
        self.layout_bottom_boxes.addWidget(self.no_customize_label)

        if self.session_is_entirely_readonly:
            self.hide_push_button_customize = True

        if self.hide_push_button_customize:   #multiuser
            self.push_button_customize.hide()
            self.no_customize_label.show()
            self.user_can_save_settings = False
            self.session_is_locked = True
        else:
            self.push_button_customize.show()
            self.no_customize_label.hide()
            self.user_can_save_settings = True
            self.session_is_locked = False

        self.bottom_box_qlabel = QLabel(" ")
        self.bottom_box_qlabel.setAlignment(Qt.AlignRight)
        self.bottom_box_qlabel.setToolTip("<p style='white-space:wrap'>Messages")

        if self.device_display_size ==  DISPLAY_SIZE_TINY:
            minimum_width = 200
            maximum_width = 600
            self.font.setPointSize(self.normal_fontsize - 2)
            self.bottom_box_qlabel.setFont(self.font)
        elif self.device_display_size == DISPLAY_SIZE_SMALL:
            minimum_width = 300
            maximum_width = 600
            self.font.setPointSize(self.normal_fontsize - 1)
            self.bottom_box_qlabel.setFont(self.font)
        elif self.device_display_size == DISPLAY_SIZE_NORMAL:
            minimum_width = 400
            maximum_width = 600
            self.font.setPointSize(self.normal_fontsize)
            self.bottom_box_qlabel.setFont(self.font)
        else:
            minimum_width = 400
            maximum_width = 600
            self.font.setPointSize(self.normal_fontsize)
            self.bottom_box_qlabel.setFont(self.font)
            self.local_prefs['CALIBRESPY_DEVICE_DISPLAY_SIZE'] = DISPLAY_SIZE_NORMAL
            self.save_local_prefs()

        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        ss = "QLabel {color : [TC]; }".replace("[TC]",tc)
        self.bottom_box_qlabel.setStyleSheet(ss)

        self.bottom_box_qlabel.setMinimumWidth(minimum_width)
        self.bottom_box_qlabel.setMaximumWidth(maximum_width)
        self.layout_bottom_boxes.addWidget(self.bottom_box_qlabel)

        self.bottom_buttonbox = QDialogButtonBox()
        self.layout_bottom_boxes.addWidget(self.bottom_buttonbox)

        self.push_button_event_catcher = QPushButton("", self)     # keeps self.matrix keyboard and mouse events from collaterally 'pushing' any real pushbuttons displayed below self.matrix.
        self.push_button_event_catcher.setAutoDefault(True)
        self.push_button_event_catcher.setDefault(True)
        self.push_button_event_catcher.setHidden(True)  #*
        self.push_button_event_catcher.setMinimumWidth(0)
        self.push_button_event_catcher.setMaximumWidth(0)
        self.bottom_buttonbox.addButton(self.push_button_event_catcher,QDialogButtonBox.AcceptRole)

        self.push_button_library_browser = QPushButton(self.icon_details,"", self)
        self.push_button_library_browser.setMaximumWidth(30)
        self.push_button_library_browser.setAutoDefault(False)
        self.push_button_library_browser.setDefault(False)
        self.push_button_library_browser.setFont(self.font)
        self.push_button_library_browser.setToolTip("<p style='white-space:wrap'>Open the 'Library Browser'.")
        self.push_button_library_browser.clicked.connect(self.library_browser)
        self.bottom_buttonbox.addButton(self.push_button_library_browser,QDialogButtonBox.AcceptRole)

        self.push_button_library_browser.setCheckable(True)

        self.push_button_view_current_cover = QPushButton(self.icon_cover,"", self)
        self.push_button_view_current_cover.setMaximumWidth(30)
        self.push_button_view_current_cover.setAutoDefault(False)
        self.push_button_view_current_cover.setDefault(False)
        self.push_button_view_current_cover.setFont(self.font)
        self.push_button_view_current_cover.setToolTip("<p style='white-space:wrap'>View the current book's cover.")
        self.push_button_view_current_cover.clicked.connect(self.view_current_cover)
        self.bottom_buttonbox.addButton(self.push_button_view_current_cover,QDialogButtonBox.AcceptRole)

        self.push_button_view_current_comments = QPushButton(self.icon_comments,"", self)
        self.push_button_view_current_comments.setMaximumWidth(30)
        self.push_button_view_current_comments.setAutoDefault(False)
        self.push_button_view_current_comments.setDefault(False)
        self.push_button_view_current_comments.setFont(self.font)
        self.push_button_view_current_comments.setToolTip("<p style='white-space:wrap'>View the current book's Comments.")
        self.push_button_view_current_comments.clicked.connect(self.view_current_comments)
        self.bottom_buttonbox.addButton(self.push_button_view_current_comments,QDialogButtonBox.AcceptRole)

        self.push_button_view_displayed_comments = QPushButton(self.icon_comments,"", self)
        self.push_button_view_displayed_comments.setMaximumWidth(30)
        self.push_button_view_displayed_comments.setAutoDefault(False)
        self.push_button_view_displayed_comments.setDefault(False)
        self.push_button_view_displayed_comments.setFont(self.font)
        self.push_button_view_displayed_comments.setToolTip("<p style='white-space:wrap'>View the displayed books' Comments.")
        self.push_button_view_displayed_comments.clicked.connect(self.view_displayed_comments)
        self.bottom_buttonbox.addButton(self.push_button_view_displayed_comments,QDialogButtonBox.AcceptRole)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("<>")
        self.push_button_optimize_column_widths.setAutoDefault(False)
        self.push_button_optimize_column_widths.setDefault(False)
        self.push_button_optimize_column_widths.setFont(self.font)
        self.push_button_optimize_column_widths.setToolTip("<p style='white-space:wrap'>The metadata columns will be resized based on their longest row contents for each column.")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,QDialogButtonBox.AcceptRole)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("><")
        self.push_button_deoptimize_column_widths.setAutoDefault(False)
        self.push_button_deoptimize_column_widths.setDefault(False)
        self.push_button_deoptimize_column_widths.setFont(self.font)
        self.push_button_deoptimize_column_widths.setToolTip("<p style='white-space:wrap'>The metadata columns will be resized to a fixed width regardless of their contents.")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,QDialogButtonBox.AcceptRole)

        self.push_button_open_current_book = QPushButton(" ", self)
        self.push_button_open_current_book.setText("Read")
        self.push_button_open_current_book.setAutoDefault(False)
        self.push_button_open_current_book.setDefault(False)
        self.push_button_open_current_book.setFont(self.font)
        self.push_button_open_current_book.setToolTip("<p style='white-space:wrap'>Open the currently selected book in your Default Reading Program for ebooks.")
        self.push_button_open_current_book.clicked.connect(self.open_current_book)
        self.bottom_buttonbox.addButton(self.push_button_open_current_book,QDialogButtonBox.AcceptRole)

        self.push_button_view_current_book = QPushButton(" ", self)
        self.push_button_view_current_book.setText("View")
        self.push_button_view_current_book.setAutoDefault(False)
        self.push_button_view_current_book.setDefault(False)
        self.push_button_view_current_book.setFont(self.font)
        self.push_button_view_current_book.clicked.connect(self.view_current_book)
        self.bottom_buttonbox.addButton(self.push_button_view_current_book,QDialogButtonBox.AcceptRole)

        if not self.path_to_desired_book_viewer_program:
            self.push_button_view_current_book.setEnabled(False)
            self.push_button_view_current_book.setToolTip("<p style='white-space:wrap'>CalibreSpy Customization specifies no valid Book Viewer Program Path; disabled.")
        else:
            self.push_button_view_current_book.setToolTip("<p style='white-space:wrap'>CalibreSpy Customization specifies viewing the currently selected book using: " + self.path_to_desired_book_viewer_program)

        self.push_button_other_current_book = QPushButton(" ", self)
        self.push_button_other_current_book.setText("Other")
        self.push_button_other_current_book.setAutoDefault(False)
        self.push_button_other_current_book.setDefault(False)
        self.push_button_other_current_book.setFont(self.font)
        self.push_button_other_current_book.clicked.connect(self.other_action_current_book)
        self.bottom_buttonbox.addButton(self.push_button_other_current_book,QDialogButtonBox.AcceptRole)

        if not self.path_to_desired_book_other_program:
            self.push_button_other_current_book.setEnabled(False)
            self.push_button_other_current_book.setToolTip("<p style='white-space:wrap'>CalibreSpy Customization has no valid Book Other Program Path; disabled.")
        else:
            self.push_button_other_current_book.setToolTip("<p style='white-space:wrap'>CalibreSpy Customization specifies opening the currently selected book using: " + self.path_to_desired_book_other_program)

        self.push_button_edit_current_book = QPushButton(" ", self)
        self.push_button_edit_current_book.setText("Edit")
        self.push_button_edit_current_book.setAutoDefault(False)
        self.push_button_edit_current_book.setDefault(False)
        self.push_button_edit_current_book.setFont(self.font)
        self.push_button_edit_current_book.clicked.connect(self.edit_current_book)
        self.bottom_buttonbox.addButton(self.push_button_edit_current_book,QDialogButtonBox.AcceptRole)

        if not self.path_to_desired_book_editor_program:
            self.push_button_edit_current_book.setEnabled(False)
            self.push_button_edit_current_book.setToolTip("<p style='white-space:wrap'>CalibreSpy Customization has no valid Book Editor Program Path; disabled.")
        else:
            self.push_button_edit_current_book.setToolTip("<p style='white-space:wrap'>CalibreSpy Customization specifies editing the currently selected book using: " + self.path_to_desired_book_editor_program)

        self.push_button_save_and_exit = QPushButton(" ", self)
        self.push_button_save_and_exit.setText("Exit")
        self.push_button_save_and_exit.setAutoDefault(False)
        self.push_button_save_and_exit.setDefault(False)
        self.push_button_save_and_exit.setFont(self.font)
        self.push_button_save_and_exit.setToolTip("<p style='white-space:wrap'>Save current CalibreSpy window settings (if not entirely Read-Only) and then exit.")
        self.push_button_save_and_exit.clicked.connect(self.save_and_exit)
        self.bottom_buttonbox.addButton(self.push_button_save_and_exit,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(False)

        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)
#---------------------------------------------------------------------------------------------------------------------------------------
    def finalize_scroll_area(self):
        self.scroll_widget.resize(self.sizeHint())
        self.scroll_area_dialog.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        self.scroll_area_dialog.resize(self.sizeHint())
#---------------------------------------------------------------------------------------------------------------------------------------
    def load_filter_comboboxes(self):
        self.authors_list = list(self.authors_set)
        del self.authors_set
        self.authors_list.sort()
        for t in self.authors_list:
            self.authors_filter_combobox.addItem(t)#default constants already added...
            self.authors_filter_combobox_full_list.append(t)#default constants already added...
        #END FOR
        self.authors_filter_combobox.setCurrentIndex(0)
        self.current_author = DEFAULT_AUTHORS
        self.authors_filter_combobox.activated.connect(self.event_filter_authors_dropdown_arrow_clicked)
        self.authors_filter_combobox.currentIndexChanged.connect(self.event_filter_authors_changed)
        #-----------------------------------------------------
        self.title_list = list(self.title_set)
        self.title_list.sort()
        for t in self.title_list:
            self.title_filter_combobox.addItem(t)#default constants already added...
            self.title_filter_combobox_full_list.append(t)#default constants already added...
        #END FOR
        self.title_list = self.title_list[0:3]  #for is-not-empty
        self.title_filter_combobox.setCurrentIndex(0)
        self.current_title = DEFAULT_TITLE
        self.title_filter_combobox.activated.connect(self.event_filter_title_dropdown_arrow_clicked)
        self.title_filter_combobox.currentIndexChanged.connect(self.event_filter_title_changed)
        #-----------------------------------------------------
        self.series_list = list(self.series_set)
        self.series_list.sort()
        for t in self.series_list:
            self.series_filter_combobox.addItem(t)#default constants already added...
            self.series_filter_combobox_full_list.append(t)#default constants already added...
        #END FOR
        self.series_list = self.series_list[0:3]  #for is-not-empty
        self.series_filter_combobox.setCurrentIndex(0)
        self.current_series = DEFAULT_SERIES
        self.series_filter_combobox.activated.connect(self.event_filter_series_dropdown_arrow_clicked)
        self.series_filter_combobox.currentIndexChanged.connect(self.event_filter_series_changed)
        #-----------------------------------------------------
        self.tags_list = list(self.tags_set)
        del self.tags_set
        tmp_tags_split_list = []
        for tag in self.tags_list:
            tag = tag + ","
            s_split = tag.split(",")
            for t in s_split:
                t = t.strip()
                if t > " ":
                    tmp_tags_split_list.append(t)
            #END FOR
        #END FOR
        tmp_tags_split_list.sort()
        for t in tmp_tags_split_list:
            self.tags_filter_combobox.addItem(t)#default constants already added...
            self.tags_filter_combobox_full_list.append(t)#default constants already added...
        #END FOR
        self.tags_list = tmp_tags_split_list[0:3]  #for is-not-empty
        del tmp_tags_split_list
        self.tags_filter_combobox.setCurrentIndex(0)
        self.current_tag = DEFAULT_TAGS
        self.tags_filter_combobox.activated.connect(self.event_filter_tags_dropdown_arrow_clicked)
        self.tags_filter_combobox.currentIndexChanged.connect(self.event_filter_tags_changed)
        #-----------------------------------------------------
        self.publisher_list = list(self.publisher_set)
        self.publisher_list.sort()
        del self.publisher_set
        for t in self.publisher_list:
            self.publisher_filter_combobox.addItem(t)  #default constants already added...
            self.publisher_filter_combobox_full_list.append(t)#default constants already added...
        #END FOR
        self.publisher_list = self.publisher_list[0:3]  #for is-not-empty
        self.publisher_filter_combobox.setCurrentIndex(0)
        self.current_publisher = DEFAULT_PUBLISHER
        self.publisher_filter_combobox.activated.connect(self.event_filter_publisher_dropdown_arrow_clicked)
        self.publisher_filter_combobox.currentIndexChanged.connect(self.event_filter_publisher_changed)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.show_filter_combobox_counts()
        #-----------------------------------------------------
        #-----------------------------------------------------
        history = self.local_prefs['HISTORICAL_FILTERING_EVENTS']
        if HISTORY_SEPARATOR_LEGACY in history:
            history = history.replace(HISTORY_SEPARATOR_LEGACY,HISTORY_SEPARATOR)
        history = history + HISTORY_SEPARATOR
        history_list = history.split(HISTORY_SEPARATOR)
        history_list = list(set(history_list))
        history_list.sort()
        history_list = history_list[0:HISTORY_MAXIMUM_VALUES]

        self.filter_history_combobox.insertItem(0,HISTORY_HEADING_NAME)  #always index 0

        for filter in history_list:
            filter = filter.strip()
            if filter > "":
                self.filter_history_combobox.addItem(filter)
        #END FOR

        self.filter_history_combobox.setCurrentText(HISTORY_HEADING_NAME)

        self.filtering_history_available = True

        self.filter_history_combobox.currentIndexChanged.connect(self.event_filter_history_changed)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.search_target_column_mapping_dict = {}

        self.search_targets_combobox.insertItem(0,SEARCH_TARGET_HEADING_NAME)  #always index 0
        c = self.local_prefs['AUTHORS_COLUMN']
        self.search_targets_combobox.addItem(AUTHORS)
        self.search_target_column_mapping_dict[AUTHORS] = c
        c = self.local_prefs['TITLE_COLUMN']
        self.search_targets_combobox.addItem(TITLE)
        self.search_target_column_mapping_dict[TITLE] = c
        c = self.local_prefs['SERIES_COLUMN']
        self.search_targets_combobox.addItem(SERIES)
        self.search_target_column_mapping_dict[SERIES] = c
        c = self.local_prefs['INDEX_COLUMN']
        self.search_targets_combobox.addItem(SERIES_INDEX)
        self.search_target_column_mapping_dict[SERIES_INDEX] = c
        c = self.local_prefs['TAGS_COLUMN']
        self.search_targets_combobox.addItem(TAGS)
        self.search_target_column_mapping_dict[TAGS] = c
        c = self.local_prefs['PUBLISHER_COLUMN']
        self.search_targets_combobox.addItem(PUBLISHER)
        self.search_target_column_mapping_dict[PUBLISHER] = c
        c = self.local_prefs['LANGUAGES_COLUMN']
        self.search_targets_combobox.addItem(LANGUAGES)
        self.search_target_column_mapping_dict[LANGUAGES] = c
        c = self.local_prefs['PUBLISHED_COLUMN']
        self.search_targets_combobox.addItem(PUBLISHED)
        self.search_target_column_mapping_dict[PUBLISHED] = c
        c = self.local_prefs['ADDED_COLUMN']
        self.search_targets_combobox.addItem(ADDED)
        self.search_target_column_mapping_dict[ADDED] = c
        c = self.local_prefs['MODIFIED_COLUMN']
        self.search_targets_combobox.addItem(MODIFIED)
        self.search_target_column_mapping_dict[MODIFIED] = c
        c = self.local_prefs['IDENTIFIERS_COLUMN']       #Version 1.0.43
        self.search_targets_combobox.addItem(IDENTIFIERS)
        self.search_target_column_mapping_dict[IDENTIFIERS] = c
        c = self.local_prefs['RATINGS_COLUMN']            #Version 1.0.52
        self.search_targets_combobox.addItem(RATINGS)
        self.search_target_column_mapping_dict[RATINGS] = c
        c = self.local_prefs['PATH_COLUMN']
        self.search_targets_combobox.addItem(PATH)
        self.search_target_column_mapping_dict[PATH] = c
        c = self.local_prefs['CUSTOM_COLUMN_1_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_1]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_2_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_2]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_3_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_3]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_4_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_4]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_5_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_5]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_6_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_6]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_7_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_7]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_8_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_8]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c
        c = self.local_prefs['CUSTOM_COLUMN_9_COLUMN']
        cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_9]
        if cc_label in self.custom_column_label_dict:
            id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            self.search_targets_combobox.addItem(name)
            self.search_target_column_mapping_dict[name] = c

        icon_binoculars_path = os.path.join(self.images_dir,"binoculars.png")
        icon_binoculars_path = icon_binoculars_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_binoculars_path)
        self.icon_binoculars = QIcon(pixmap)
        del pixmap

        for r in range(0,MATRIX_MAXIMUM_COLUMNS_COUNT + 1):
            self.search_targets_combobox.setItemIcon(r, self.icon_binoculars)
        #END FOR

        icon_search_path = os.path.join(self.images_dir,"search.png")
        icon_search_path = icon_search_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_search_path)
        self.icon_search = QIcon(pixmap)
        del pixmap

        self.push_button_do_search.setIcon(self.icon_search)

        self.virtual_libraries_dict = None

        icon_duplicates_search_path = os.path.join(self.images_dir,"duplicates.png")
        icon_duplicates_search_path = icon_duplicates_search_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_duplicates_search_path)
        self.icon_duplicates_search = QIcon(pixmap)
        del pixmap

        self.push_button_do_duplicates_search.setIcon(self.icon_duplicates_search)

        icon_interbook_search_path = os.path.join(self.images_dir,"interbook_search.png")
        icon_interbook_search_path = icon_interbook_search_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_interbook_search_path)
        self.icon_interbook_search = QIcon(pixmap)
        del pixmap

        self.push_button_do_interbook_search.setIcon(self.icon_interbook_search)

        icon_drop_search_results_path = os.path.join(self.images_dir,"drop.png")
        icon_drop_search_results_path = icon_drop_search_results_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_drop_search_results_path)
        self.icon_dsr = QIcon(pixmap)
        self.pixmap_dsr = pixmap
        del pixmap

        self.push_button_drop_search_results.setIcon(self.icon_dsr)

        icon_do_contains_search_path = os.path.join(self.images_dir,"mixture.png")
        icon_do_contains_search_path = icon_do_contains_search_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_do_contains_search_path)
        self.icon_mixture = QIcon(pixmap)
        del pixmap

        self.push_button_do_contains_search.setIcon(self.icon_mixture)

        icon_do_sql_query_search_path = os.path.join(self.images_dir,"sqlquery.png")
        icon_do_sql_query_search_path = icon_do_sql_query_search_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_do_sql_query_search_path)
        self.icon_sqlquery = QIcon(pixmap)
        del pixmap

        self.push_button_do_sqlquery_search.setIcon(self.icon_sqlquery)

        icon_invert_path = os.path.join(self.images_dir,"invert_selection.png")
        icon_invert_path = icon_invert_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_invert_path)
        self.icon_invert_selection = QIcon(pixmap)
        del pixmap

        self.push_button_invert_selection.setIcon(self.icon_invert_selection)

        icon_ear_path = os.path.join(self.images_dir,"ear.png")
        icon_ear_path = icon_ear_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(15,15))
        pixmap.load(icon_ear_path)
        self.listening_icon_label.setPixmap(pixmap)
        del pixmap

        #-----------------------------------------------------
        #-----------------------------------------------------
        self.load_search_regex_history()
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def connect_events(self):
        #-----------------------------------------------------
        #~ 1. When mouse left clicked -> Clicked(index) is emitted.
        #~ 2. When mouse left double clicked -> Clicked(index) and doubleClicked(index) and Activated(index) are emitted.
        #~ 3. When mouse right clicked -> Clicked(index) and customContextMenuRequested(index) are emitted.
        #~ 4. When touchscreen is tapped -> Clicked(index) and Pressed(index) are emitted.
        #~ 5. When touchscreen is double tapped -> Clicked(index) and doubleClicked(index) and Activated(index) are emitted.
        #~ 6. When keyboard Return/Enter pressed  -> Activated(index) is emitted.
        #-----------------------------------------------------
        self.matrix.customContextMenuRequested.connect(self.event_matrix_customcontextmenurequested_action) #3
        self.matrix.activated.connect(self.event_activated_action)  #6 (which overlaps with #2 for double click and #5 for double tap)
        self.matrix.verticalHeader().customContextMenuRequested.connect(self.event_row_header_customcontextmenurequested_action) #3
        self.matrix.horizontalHeader().customContextMenuRequested.connect(self.event_column_header_customcontextmenurequested_action) #3
        #-----------------------------------------------------
        self.action_is_in_progress = False
        self.matrix.currentCellChanged.connect(self.event_current_cell_changed)
        #-----------------------------------------------------
        self.resized_signal.connect(self.event_dialog_resized)
#---------------------------------------------------------------------------------------------------------------------------------------
    def enable_all_signals(self):
        self.all_signals_blocked = False
        self.authors_signals_blocked = False
        self.title_signals_blocked = False
        self.series_signals_blocked = False
        self.tags_signals_blocked = False
        self.publisher_signals_blocked = False
        self.filter_history_signals_blocked = False
        self.authors_filter_is_applied = False
        self.title_filter_is_applied = False
        self.series_filter_is_applied = False
        self.tags_filter_is_applied = False
        self.publisher_filter_is_applied = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def finalize_matrix(self):
        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        bc = self.local_prefs['CALIBRESPY_BACKGROUND_COLOR']   #BG1 (usually lighter)
        ac = self.local_prefs['CALIBRESPY_ALTERNATING_COLOR']    #BG2 (usually darker)

        style_string = "QTableWidget, QHeaderView {alternate-background-color: [AC];background-color: [BC];color : [TC]; border: 1px}"
        style_string = style_string.replace("[TC]",tc)
        style_string = style_string.replace("[BC]",ac)  #BG2 (usually darker)  row #0 uses background-color (which touches the QHeaderView 'white')
        style_string = style_string.replace("[AC]",bc)  #BG1 (usually lighter)  row #1 uses alternate-background-color
        self.matrix.setStyleSheet(style_string)

        self.matrix.setAlternatingRowColors(True)
        #-----------------------------------------------------
        self.matrix.setSelectionBehavior(QAbstractItemView.SelectRows)    # entire row
        self.matrix.setSelectionMode(QAbstractItemView.SingleSelection)        # single row
        self.matrix.setShowGrid(True)
        #-----------------------------------------------------
        self.matrix.setCurrentCell(0,0)  #selection mode is "entire row"...
        self.current_row = 0
        self.current_column = 0
        self.current_visible_row_count = self.n_matrix_rows
        self.last_visible_row = self.n_matrix_rows - 1

        self.column_fixed_width_dict = {}
        self.column_fixed_width_dict[self.authors_column] = 200
        self.column_fixed_width_dict[self.title_column] = 200
        self.column_fixed_width_dict[self.series_column] = 100
        self.column_fixed_width_dict[self.index_column] = 50
        self.column_fixed_width_dict[self.tags_column] = 100
        self.column_fixed_width_dict[self.published_column] = 60
        self.column_fixed_width_dict[self.publisher_column] = 100
        self.column_fixed_width_dict[self.languages_column] = 50
        self.column_fixed_width_dict[self.added_column] =  60
        self.column_fixed_width_dict[self.modified_column] = 60
        self.column_fixed_width_dict[self.path_column] = 200
        self.column_fixed_width_dict[self.cc1_column] = 100
        self.column_fixed_width_dict[self.cc2_column] = 100
        self.column_fixed_width_dict[self.cc3_column] = 100
        self.column_fixed_width_dict[self.cc4_column] = 100
        self.column_fixed_width_dict[self.cc5_column] = 100
        self.column_fixed_width_dict[self.cc6_column] = 100
        self.column_fixed_width_dict[self.cc7_column] = 100
        self.column_fixed_width_dict[self.cc8_column] = 100
        self.column_fixed_width_dict[self.cc9_column] = 100
        # add new columns just below.
        self.column_fixed_width_dict[self.identifiers_column] = 100      #Version 1.0.43
        self.column_fixed_width_dict[self.ratings_column] = 60      #Version 1.0.52
        #-----------------------------------------------------
        self.finalize_integer_columns()
        #-----------------------------------------------------
        self.deoptimize_column_widths()
        #-----------------------------------------------------
        self.original_row_height = self.matrix.rowHeight(0)
        if self.original_row_height > 80:
            self.original_row_height = 80

        self.matrix.horizontalHeader().setFixedHeight(25)
#---------------------------------------------------------------------------------------------------------------------------------------
    def finalize_integer_columns(self):
        if 1 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc1_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[1]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc1_column] = max
                self.matrix.setColumnWidth(self.cc1_column,max)
        if 2 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc2_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[2]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc2_column] = max
                self.matrix.setColumnWidth(self.cc2_column,max)
        if 3 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc3_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[3]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc3_column] = max
                self.matrix.setColumnWidth(self.cc3_column,max)
        if 4 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc4_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[4]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc4_column] = max
                self.matrix.setColumnWidth(self.cc4_column,max)
        if 5 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc5_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[5]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc5_column] = max
                self.matrix.setColumnWidth(self.cc5_column,max)
        if 6 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc6_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[6]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc6_column] = max
                self.matrix.setColumnWidth(self.cc6_column,max)
        if 7 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc7_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[7]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc7_column] = max
                self.matrix.setColumnWidth(self.cc7_column,max)
        if 8 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc8_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[8]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc8_column] = max
                self.matrix.setColumnWidth(self.cc8_column,max)
        if 9 in self.cc_integer_col_number_set:
            for r in range(0,self.n_matrix_rows):
                item = self.matrix.item(r,self.cc9_column)
                if item:
                    item.setTextAlignment(Qt.AlignRight)
            #END FOR
            max = self.cc_integer_max_length_dict[9]
            if max > 0:
                max = max + 1
                self.column_fixed_width_dict[self.cc9_column] = max
                self.matrix.setColumnWidth(self.cc9_column,max)
#---------------------------------------------------------------------------------------------------------------------------------------
    def init_sets_lists_dicts(self):
        self.authors_set = set()
        self.title_set = set()
        self.series_set = set()
        self.tags_set = set()
        self.publisher_set = set()
        #~ -----------------------------
        self.authors_list = []
        self.title_list = []
        self.series_list = []
        self.tags_list = []
        self.publisher_list = []
        #~ -----------------------------
        self.book_table_dict = {}
#---------------------------------------------------------------------------------------------------------------------------------------
    def prefilter_books_control(self):

        self.init_sets_lists_dicts()

        title = "Pre-Filter Books Using?"
        label = "Select Pre-Filtering Metadata                "
        items = []
        items.append("[1] Author")
        items.append("[2] Title")
        items.append("[3] Series ")
        items.append("[4] Tag")
        items.append("[5] Date Added")
        items.append("[6] Date Published")
        items.append("[7] Comments")
        items.append("[8] #Custom Column")
        items.append("Skip Prefiltering; Select Everything")
        self.prefilter_method,ok = QInputDialog.getItem(None, title, label, items,0,False)
        if DEBUG: print("Selected:  ", as_unicode(self.prefilter_method))
        if not ok:
            error_dialog(None, _('CalibreSpy'),_('Pre-Filtering Option Invalid.'), show=True)
            self.prefilter = False
            self.prefilter_method = None
            self.prefilter_regex = None
            return

        self.prefilter_column_is_custom = False

        if self.prefilter_method.startswith("[1]"):
            self.prefilter_column = "authors"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Authors                      "
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[2]"):
            self.prefilter_column = "title"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Title                             "
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[3]"):
            self.prefilter_column = "series"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Series Name                 "
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[4]"):
            self.prefilter_column = "tags"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Tags                              "
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[5]"):
            self.prefilter_column = "timestamp"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Date Added [YYYY-MM-DD] "
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[6]"):
            self.prefilter_column = "pubdate"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Date Published  [YYYY-MM-DD] "
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[7]"):
            self.prefilter_column = "comments"
            title = "Pre-Filter Books"
            label = "Specify Regular Expression for Comments (which are normally in HTML)"
            self.prefilter_regex,ok = QInputDialog.getText(None,title,label)
            if not ok:
                self.prefilter = False
        elif self.prefilter_method.startswith("[8]"):
            title = "Pre-Filter Books"
            label = "Specify [Line1] #customcolumn  [Line2] Regular Expression"
            multistr,ok = QInputDialog.getMultiLineText(None,title,label)
            if (not ok) or (not "\n" in multistr) or (not "#" in multistr):
                self.prefilter = False
            else:
                s_list = multistr.split("\n")
                if not len(s_list) > 1:   #allow user to hit enter needlessly at the end of line 2...
                    self.prefilter = False
                else:
                    self.prefilter_column = s_list[0].strip()
                    self.prefilter_regex = s_list[1].strip()
                    self.prefilter_column_is_custom = True
        elif self.prefilter_method.startswith("Skip"):
            if DEBUG: print("Skipping Prefiltering; loading entire Library...")
            self.prefilter = False
            if DEBUG: self.elapsed_time_message(0,"CalibreSpy: Initializing")        # reset start-time for loading the books since user input required above...
            self.get_table_books_data()
            return

        if not self.prefilter:
            error_dialog(None, _('CalibreSpy'),_('Pre-Filtering Specifications Invalid.'), show=True)
            self.prefilter_column = None
            self.prefilter_regex = None
            return

        if DEBUG: print("Pre-Filtering: ", self.prefilter_column, as_unicode(self.prefilter_regex))

        if DEBUG: self.elapsed_time_message(0,"CalibreSpy: Initializing")        # reset start-time for loading the books since user input required above...

        is_valid = self.prefilter_books()
        if not is_valid:
            error_dialog(None, _('CalibreSpy'),_('Pre-Filtering Specifications Invalid, or Nothing Found.'), show=True)
            self.prefilter_column = None
            self.prefilter_regex = None
#---------------------------------------------------------------------------------------------------------------------------------------
    def prefilter_books(self):

        mysql = self.build_prefilter_sql()

        if mysql is None:
            return False

        import re
        self.regex = re
        del re

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)
            return False

        my_db.createscalarfunction("regexp", self.apsw_user_function_regexp)

        my_cursor.execute(mysql,[self.prefilter_regex])
        tmp_rows = my_cursor.fetchall()

        my_db.close()

        if self.is_cli:
            self.gc.collect()

        if tmp_rows is None:
            return False

        n = len(tmp_rows)
        if n == 0:
            return False

        tmp_set = set()

        for row in tmp_rows:
            book,dummy = row
            tmp_set.add(book)
        #END FOR
        del tmp_rows

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        for book in tmp_set:
            mysql = "SELECT path,title,series_index,pubdate,id,timestamp,last_modified FROM books WHERE id = ? "
            my_cursor.execute(mysql,([book]))
            tmp_rows = my_cursor.fetchall()
            if tmp_rows is None:
                continue
            n = len(tmp_rows)
            if n == 0:
                continue
            for row in tmp_rows:
                path,title,series_index,pubdate,id,timestamp,modified = row
                if not path > " ":
                    path = as_unicode(id)
                pubdate = pubdate[0:10]
                if pubdate == "0101-01-01":
                    pubdate = ""
                if self.local_prefs['CALIBRESPY_LOAD_TIMESTAMP'] == 0:
                    timestamp = ""
                else:
                    timestamp = timestamp[0:10].strip()
                if self.local_prefs['CALIBRESPY_LOAD_MODIFIED'] == 0:
                    modified = ""
                else:
                    modified =  modified[0:10].strip()
                self.book_table_dict[id] = path,title,series_index,pubdate,timestamp,modified
                if not title in self.title_set:
                    self.title_set.add(title)
            #END FOR
            del tmp_rows
        #END FOR
        del tmp_set

        my_db.close()

        if self.is_cli:
            self.gc.collect()

        return True
#---------------------------------------------------------------------------------------------------------------------------------------
    def build_prefilter_sql(self):

        mysql = None

        if self.prefilter_column == AUTHORS:
            mysql = "SELECT book,'authors' FROM books_authors_link WHERE author IN (SELECT id FROM authors WHERE name REGEXP ? )"
        elif self.prefilter_column == TITLE:
            mysql = "SELECT id,'title' FROM books WHERE title REGEXP ? "
        elif self.prefilter_column == SERIES:
            mysql = "SELECT book,'series' FROM books_series_link WHERE series IN (SELECT id FROM series WHERE name REGEXP ? )"
        elif self.prefilter_column == TAGS:
            mysql = "SELECT book,'tags' FROM books_tags_link WHERE tag IN (SELECT id FROM tags WHERE name REGEXP ? )"
        elif self.prefilter_column == "timestamp":
            mysql = "SELECT id,'timestamp' FROM books WHERE timestamp REGEXP ? "
        elif self.prefilter_column == "pubdate":
            mysql = "SELECT id,'pubdate' FROM books WHERE pubdate REGEXP ? "
        elif self.prefilter_column == "comments":
            mysql = "SELECT book,'comments' FROM comments WHERE text REGEXP ? "
        elif self.prefilter_column_is_custom:
            self.get_prefilter_table_custom_columns()
            if not self.prefilter_column in self.prefilter_custom_column_label_dict:
                return None
            r = self.prefilter_custom_column_label_dict[self.prefilter_column]
            id,label,name,datatype,is_multiple,normalized,display = r
            if normalized == 0:
                mysql = "SELECT book,'cc[NN]' FROM custom_column_[NN] WHERE value REGEXP ? )"
                mysql = mysql.replace("[NN]",id)
            else:
                mysql = "SELECT book,'cc[NN]' FROM books_custom_column_[NN]_link WHERE value IN (SELECT id FROM custom_column_[NN] WHERE value REGEXP ? )"
                mysql = mysql.replace("[NN]",id)
        else:
            return None

        return mysql
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_prefilter_table_custom_columns(self):

        self.prefilter_custom_column_label_dict = {}

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("get_prefilter_table_custom_columns: apsw_connect_to_library failed.")
            return

        try:
            mysql = "SELECT id,label,name,datatype,is_multiple,normalized,display FROM custom_columns"
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                id,label,name,datatype,is_multiple,normalized,display = row
                if datatype != "composite" and datatype != "ratings":
                    label = "#" + label
                    id = as_unicode(id)
                    r = id,label,name,datatype,is_multiple,normalized,display
                    self.prefilter_custom_column_label_dict[label] = r
            #END FOR
            del tmp_rows
            my_db.close()
        except Exception as e:
            #~ if never had any custom columns...
            my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_table_books_data(self):

        self.init_sets_lists_dicts()

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        mysql = "SELECT path,title,series_index,pubdate,id,timestamp,last_modified FROM books "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        my_db.close()
        if tmp_rows is None:
            tmp_rows = []
        n = len(tmp_rows)
        if n == 0:
            return
        for row in tmp_rows:
            path,title,series_index,pubdate,id,timestamp,modified = row
            if not path > " ":
                path = as_unicode(id)
            pubdate = pubdate[0:10]
            if pubdate == "0101-01-01":
                pubdate = ""
            if self.local_prefs['CALIBRESPY_LOAD_TIMESTAMP'] == 0:
                timestamp = ""
            else:
                timestamp = timestamp[0:10].strip()
            if self.local_prefs['CALIBRESPY_LOAD_MODIFIED'] == 0:
                modified = ""
            else:
                modified =  modified[0:10].strip()
            self.book_table_dict[id] = path,title,series_index,pubdate,timestamp,modified
            if not title in self.title_set:
                self.title_set.add(title)
        #END FOR
        del tmp_rows
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def load_book_matrix(self):

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            error_dialog(none, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)
            return False

        #~ myflags = Qt.ItemFlag(Qt.ItemIsSelectable | Qt.ItemIsEnabled) #Qt5
        myflags = Qt.ItemFlag(Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsEnabled) #Qt6

        self.first_visible_row = 0
        self.current_row = 0

        n = self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE']
        self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE'] = int(n)
        n_added_row_height = int(n)
        self.default_row_height = self.local_prefs['DEFAULT_MATRIX_ROW_HEIGHT']
        if self.default_row_height < 30:
            self.default_row_height = 30
        self.n_total_row_height  = self.default_row_height + n_added_row_height
        if self.n_total_row_height > 60:
            self.n_total_row_height = 60

        self.languages_set = set()
        self.bookid_authors_link_dict = {}   #[bookid] = authors_link_list
        self.book_table_list = []

        self.cc_integer_col_number_set = set()     #[cc col number]    1 - 9
        self.cc_integer_max_length_dict = {}   #[cc col number] = longest string from original integer
        self.cc_integer_max_length_dict[1] = 0
        self.cc_integer_max_length_dict[2] = 0
        self.cc_integer_max_length_dict[3] = 0
        self.cc_integer_max_length_dict[4] = 0
        self.cc_integer_max_length_dict[5] = 0
        self.cc_integer_max_length_dict[6] = 0
        self.cc_integer_max_length_dict[7] = 0
        self.cc_integer_max_length_dict[8] = 0
        self.cc_integer_max_length_dict[9] = 0

        try:
            #~ for bookid,v in self.book_table_dict.iteritems():
            for bookid,v in iteritems(self.book_table_dict):
                path,title,series_index,pubdate,timestamp,modified = v
                mi_dict = self.get_other_metadata(my_db,my_cursor,bookid)
                #~ -----------------------------
                authors_list = mi_dict[AUTHORS]
                authors_sort = ""
                for sort in authors_list:
                    authors_sort = authors_sort + sort + " & "
                #END FOR
                authors_sort = authors_sort.strip()
                authors_sort = authors_sort[0:-1].strip()
                if not authors_sort in self.authors_set:   # sets ~ 20 times faster than lists for lookups
                    self.authors_set.add(authors_sort)
                #~ -----------------------------
                series = mi_dict['series']
                if series == DEFAULT_SERIES_NONE:
                    series_index = ""
                #~ -----------------------------
                if not series_index:
                    series_index = ""
                series_index = as_unicode(series_index)
                #~ -----------------------------
                book_file_name = mi_dict['book_file_name']
                if not book_file_name:
                    path = " (" + as_unicode(bookid) + ")/"
                else:
                    path = os.path.join(path,book_file_name)
                    path = path.replace(os.sep, '/')
                #~ -----------------------------
                publisher = mi_dict[PUBLISHER]
                #~ -----------------------------
                tags_list = mi_dict[TAGS]
                tags = ""
                for tag in tags_list:
                    tags = tags + tag + ", "
                #END FOR
                tags = tags.strip()
                if tags.endswith(","):
                    tags = tags[0:-1]
                #~ -----------------------------
                languages_list = mi_dict[LANGUAGES]
                languages = ""
                for language in languages_list:
                    languages = languages + language + ", "
                #END FOR
                languages = languages.strip()
                if languages.endswith(","):
                    languages = languages[0:-1]
                self.languages_set.add(languages)
                #~ -----------------------------
                identifiers_list = mi_dict[IDENTIFIERS]
                identifiers = ""
                if identifiers_list is not None:
                    for identifier in identifiers_list:
                        identifiers = identifiers + identifier + ", "
                    #END FOR
                    identifiers = identifiers.strip()
                    if identifiers.endswith(","):
                        identifiers = identifiers[0:-1]
                rating = mi_dict[RATINGS]
                #~ -----------------------------
                #~ -----------------------------
                cc1 = ""
                cc2 = ""
                cc3 = ""
                cc4 = ""
                cc5 = ""
                cc6 = ""
                cc7 = ""
                cc8 = ""
                cc9 = ""
                if self.custom_columns_are_active:
                    if self.cc1_is_active:
                        if CUSTOM_COLUMN_1 in mi_dict:
                            cc1 = mi_dict[CUSTOM_COLUMN_1]
                            cc1 = unicode_type(cc1)
                    if self.cc2_is_active:
                        if CUSTOM_COLUMN_2 in mi_dict:
                            cc2 = mi_dict[CUSTOM_COLUMN_2]
                            cc2 = unicode_type(cc2)
                    if self.cc3_is_active:
                        if CUSTOM_COLUMN_3 in mi_dict:
                            cc3 = mi_dict[CUSTOM_COLUMN_3]
                            cc3 = unicode_type(cc3)
                    if self.cc4_is_active:
                        if CUSTOM_COLUMN_4 in mi_dict:
                            cc4 = mi_dict[CUSTOM_COLUMN_4]
                            cc4 = unicode_type(cc4)
                    if self.cc5_is_active:
                        if CUSTOM_COLUMN_5 in mi_dict:
                            cc5 = mi_dict[CUSTOM_COLUMN_5]
                            cc5 = unicode_type(cc5)
                    if self.cc6_is_active:
                        if CUSTOM_COLUMN_6 in mi_dict:
                            cc6 = mi_dict[CUSTOM_COLUMN_6]
                            cc6 = unicode_type(cc6)
                    if self.cc7_is_active:
                        if CUSTOM_COLUMN_7 in mi_dict:
                            cc7 = mi_dict[CUSTOM_COLUMN_7]
                            cc7 = unicode_type(cc7)
                    if self.cc8_is_active:
                        if CUSTOM_COLUMN_8 in mi_dict:
                            cc8 = mi_dict[CUSTOM_COLUMN_8]
                            cc8 = unicode_type(cc8)
                    if self.cc9_is_active:
                        if CUSTOM_COLUMN_9 in mi_dict:
                            cc9 = mi_dict[CUSTOM_COLUMN_9]
                            cc9 = unicode_type(cc9)
                else:
                    pass
                #~ -----------------------------
                del mi_dict
                row = authors_sort,series,series_index,pubdate,title,path,tags,languages,publisher,cc1,cc2,cc3,cc4,cc5,cc6,cc7,cc8,cc9,timestamp,modified,identifiers,rating,bookid     #sort sequence
                self.book_table_list.append(row)
            #END FOR
            del self.book_table_dict
            if self.is_cli:
                self.gc.collect()
            self.book_table_list.sort()  #multicolumn sort implicit here...
        except Exception as e:
            my_db.close()
            if DEBUG: print("Exception in: load_book_matrix [1]:", as_unicode(e))
            return False

        self.languages_list = list(self.languages_set)
        del self.languages_set

        fontmetrics = QFontMetrics(self.font)

        #---------------------------
        #---------------------------
        if DEBUG: self.elapsed_time_message(1, "CalibreSpy: All Metadata Retrieved: elapsed  ")
        #---------------------------
        #---------------------------
        try:
            self.visible_rows_set = set()
            r = 0
            for row in self.book_table_list:       #in initial sort order

                authors_sort,series,series_index,pubdate,title,path,tags,languages,publisher,cc1,cc2,cc3,cc4,cc5,cc6,cc7,cc8,cc9,timestamp,modified,identifiers,rating,bookid = row

                authors_sort_ = QTableWidgetItem(authors_sort)
                title_ = QTableWidgetItem(title)
                series_ = QTableWidgetItem(series)
                series_index_ = QTableWidgetItem(series_index)
                tags_ = QTableWidgetItem(tags)
                pubdate_ = QTableWidgetItem(pubdate)
                publisher_ = QTableWidgetItem(publisher)
                languages_ = QTableWidgetItem(languages)
                added_ = QTableWidgetItem(timestamp)
                modified_ = QTableWidgetItem(modified)
                path_ = QTableWidgetItem(path)
                #add newly available columns below
                identifiers_ = QTableWidgetItem(identifiers)   #Version 1.0.43
                rating_ = QTableWidgetItem(rating)                #Version 1.0.52

                if self.custom_columns_are_active:
                    if self.cc1_is_active:
                        if 1 in self.cc_integer_col_number_set:
                            if cc1 == "":
                                cc1 = "0"
                            cc1_ = QTableWidgetItem()
                            cc1_.setData(Qt.DisplayRole,int(cc1))
                        else:
                            cc1_ = QTableWidgetItem(cc1)
                    if self.cc2_is_active:
                        if 2 in self.cc_integer_col_number_set:
                            if cc2 == "":
                                cc2 = "0"
                            cc2_ = QTableWidgetItem()
                            cc2_.setData(Qt.DisplayRole,int(cc2))
                        else:
                            cc2_ = QTableWidgetItem(cc2)
                    if self.cc3_is_active:
                        if 3 in self.cc_integer_col_number_set:
                            if cc3 == "":
                                cc3 = "0"
                            cc3_ = QTableWidgetItem()
                            cc3_.setData(Qt.DisplayRole,int(cc3))
                        else:
                            cc3_ = QTableWidgetItem(cc3)
                    if self.cc4_is_active:
                        if 4 in self.cc_integer_col_number_set:
                            if cc4 == "":
                                cc4 = "0"
                            cc4_ = QTableWidgetItem()
                            cc4_.setData(Qt.DisplayRole,int(cc4))
                        else:
                            cc4_ = QTableWidgetItem(cc4)
                    if self.cc5_is_active:
                        if 5 in self.cc_integer_col_number_set:
                            if cc5 == "":
                                cc5 = "0"
                            cc5_ = QTableWidgetItem()
                            cc5_.setData(Qt.DisplayRole,int(cc5))
                        else:
                            cc5_ = QTableWidgetItem(cc5)
                    if self.cc6_is_active:
                        if 6 in self.cc_integer_col_number_set:
                            if cc6 == "":
                                cc6 = "0"
                            cc6_ = QTableWidgetItem()
                            cc6_.setData(Qt.DisplayRole,int(cc6))
                        else:
                            cc6_ = QTableWidgetItem(cc6)
                    if self.cc7_is_active:
                        if 7 in self.cc_integer_col_number_set:
                            if cc7 == "":
                                cc7 = "0"
                            cc7_ = QTableWidgetItem()
                            cc7_.setData(Qt.DisplayRole,int(cc7))
                        else:
                            cc7_ = QTableWidgetItem(cc7)
                    if self.cc8_is_active:
                        if 8 in self.cc_integer_col_number_set:
                            if cc8 == "":
                                cc8 = "0"
                            cc8_ = QTableWidgetItem()
                            cc8_.setData(Qt.DisplayRole,int(cc8))
                        else:
                            cc8_ = QTableWidgetItem(cc8)
                    if self.cc9_is_active:
                        if 9 in self.cc_integer_col_number_set:
                            if cc9 == "":
                                cc9 = "0"
                            cc9_ = QTableWidgetItem()
                            cc9_.setData(Qt.DisplayRole,int(cc9))
                        else:
                            cc9_ = QTableWidgetItem(cc9)

                authors_sort_.setFont(self.font)
                title_.setFont(self.font)
                series_.setFont(self.font)
                series_index_.setFont(self.font)
                tags_.setFont(self.font)
                pubdate_.setFont(self.font)
                publisher_.setFont(self.font)
                languages_.setFont(self.font)
                added_.setFont(self.font)
                modified_.setFont(self.font)
                path_.setFont(self.font)
                #add newly available columns below
                identifiers_.setFont(self.font) #Version 1.0.43
                rating_.setFont(self.font) #Version 1.0.52
                if self.custom_columns_are_active:
                    if self.cc1_is_active:
                        cc1_.setFont(self.font)
                    if self.cc2_is_active:
                        cc2_.setFont(self.font)
                    if self.cc3_is_active:
                        cc3_.setFont(self.font)
                    if self.cc4_is_active:
                        cc4_.setFont(self.font)
                    if self.cc5_is_active:
                        cc5_.setFont(self.font)
                    if self.cc6_is_active:
                        cc6_.setFont(self.font)
                    if self.cc7_is_active:
                        cc7_.setFont(self.font)
                    if self.cc8_is_active:
                        cc8_.setFont(self.font)
                    if self.cc9_is_active:
                        cc9_.setFont(self.font)

                authors_sort_.setFlags(myflags)      # not enabled; read-only
                title_.setFlags(myflags)
                series_.setFlags(myflags)
                series_index_.setFlags(myflags)
                tags_.setFlags(myflags)
                pubdate_.setFlags(myflags)
                publisher_.setFlags(myflags)
                languages_.setFlags(myflags)
                added_.setFlags(myflags)
                modified_.setFlags(myflags)
                path_.setFlags(myflags)
                #add newly available columns below
                identifiers_.setFlags(myflags)  #Version 1.0.43
                rating_.setFlags(myflags)       #Version 1.0.52

                if self.custom_columns_are_active:
                    if self.cc1_is_active:
                        cc1_.setFlags(myflags)
                    if self.cc2_is_active:
                        cc2_.setFlags(myflags)
                    if self.cc3_is_active:
                        cc3_.setFlags(myflags)
                    if self.cc4_is_active:
                        cc4_.setFlags(myflags)
                    if self.cc5_is_active:
                        cc5_.setFlags(myflags)
                    if self.cc6_is_active:
                        cc6_.setFlags(myflags)
                    if self.cc7_is_active:
                        cc7_.setFlags(myflags)
                    if self.cc8_is_active:
                        cc8_.setFlags(myflags)
                    if self.cc9_is_active:
                        cc9_.setFlags(myflags)

                #---------------------------
                #---------------------------
                self.matrix.setItem(r,self.authors_column,authors_sort_)
                self.matrix.setItem(r,self.title_column,title_)
                self.matrix.setItem(r,self.series_column,series_)
                self.matrix.setItem(r,self.index_column,series_index_)
                self.matrix.setItem(r,self.tags_column,tags_)
                self.matrix.setItem(r,self.published_column,pubdate_)
                self.matrix.setItem(r,self.publisher_column,publisher_)
                self.matrix.setItem(r,self.languages_column,languages_)
                self.matrix.setItem(r,self.added_column,added_)
                self.matrix.setItem(r,self.modified_column,modified_)
                self.matrix.setItem(r,self.path_column,path_)
                #add newly available columns below
                self.matrix.setItem(r,self.identifiers_column,identifiers_)  #Version 1.0.43
                self.matrix.setItem(r,self.ratings_column,rating_)  #Version 1.0.52
                if self.custom_columns_are_active:
                    if self.cc1_is_active:
                        self.matrix.setItem(r,self.cc1_column,cc1_)
                    if self.cc2_is_active:
                        self.matrix.setItem(r,self.cc2_column,cc2_)
                    if self.cc3_is_active:
                        self.matrix.setItem(r,self.cc3_column,cc3_)
                    if self.cc4_is_active:
                        self.matrix.setItem(r,self.cc4_column,cc4_)
                    if self.cc5_is_active:
                        self.matrix.setItem(r,self.cc5_column,cc5_)
                    if self.cc6_is_active:
                        self.matrix.setItem(r,self.cc6_column,cc6_)
                    if self.cc7_is_active:
                        self.matrix.setItem(r,self.cc7_column,cc7_)
                    if self.cc8_is_active:
                        self.matrix.setItem(r,self.cc8_column,cc8_)
                    if self.cc9_is_active:
                        self.matrix.setItem(r,self.cc9_column,cc9_)

                self.matrix.setRowHeight(r,self.n_total_row_height)

                self.visible_rows_set.add(r)

                r = r + 1

                #--------------------------------------
                #--------------------------------------
            #END FOR
        except Exception as e:
            my_db.close()
            if DEBUG: print("Exception in: load_book_matrix[2]:", as_unicode(e))
            return False

        self.last_visible_row = r
        self.current_visible_row_count = r + 1

        my_db.close()
        #-----------------------------------------------------
        self.matrix.setSortingEnabled(True)
        #-----------------------------------------------------
        self.originally_loaded_0_0_author = None
        #-----------------------------------------------------
        self.sorts_locked = False
        self.sort_matrix(source="initial")
        #-----------------------------------------------------
        item = self.matrix.item(0,self.authors_column)
        if item:
            self.originally_loaded_0_0_author = item.text()
        #-----------------------------------------------------
        self.matrix.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self.matrix.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)

        self.matrix.setHorizontalScrollMode(QAbstractItemView.ScrollPerItem)

        self.matrix.verticalScrollBar().setStyleSheet("QScrollBar:vertical { width: 35px; }")

        self.matrix.verticalScrollBar().setToolTip("<p style='white-space:wrap'>Right-Click the scroll bar to use its context-menu.")

        self.matrix.setGridStyle(Qt.SolidLine)  #Qt6      Qt5: 1

        return True
#---------------------------------------------------------------------------------------------------------------------------------------
    def sort_matrix(self,source="None"):
        #default sort sequence from least significant (title) to most significant (authors) to accomodate Qt.  Assumes possibility of duplicate titles published at different dates plus other common scenarios.
        self.matrix.sortItems(self.title_column,Qt.AscendingOrder)
        self.matrix.sortItems(self.published_column,Qt.AscendingOrder)
        self.matrix.sortItems(self.index_column,Qt.AscendingOrder)
        self.matrix.sortItems(self.series_column,Qt.AscendingOrder)
        self.matrix.sortItems(self.authors_column,Qt.AscendingOrder)
        if source != "initial":
            self.rebuild_row_to_bookid_dict()
#---------------------------------------------------------------------------------------------------------------------------------------
    def rebuild_row_to_bookid_dict(self):
        QTimer.singleShot(0,self.rebuild_row_to_bookid_dict_now)
#---------------------------------------------------------------------------------------------------------------------------------------
    def rebuild_row_to_bookid_dict_now(self):

        self.sorts_locked = True

        if self.regex is None:
            import re
            self.regex = re
            del re

        bookid_re = "[(][0-9]+[)][/]"
        p = self.regex.compile(bookid_re)
        self.row_to_bookid_dict.clear()
        self.bookid_to_row_dict.clear()
        for r in range(0,self.n_matrix_rows):
            item = self.matrix.item(r,self.path_column)
            if item:
                path = item.text()
                match_book_id = p.search(path)
                if match_book_id:
                    s = match_book_id.group(0)
                    s = s.replace("(","")
                    s = s.replace(")","")
                    s = s.replace("/","")
                    s = s.strip()
                    if s.isdigit():
                        bookid = int(s)
                        self.row_to_bookid_dict[r] = bookid
                        self.bookid_to_row_dict[bookid] = r
                    else:
                        if DEBUG: print("Error: path does not contain the bookid...", as_unicode(path))
        #END FOR

        self.marked_rows_set.clear()

        for bookid in self.marked_bookids_set:
            color = self.marked_bookids_color_dict[bookid]
            r = self.bookid_to_row_dict[bookid]
            item = self.matrix.item(r,0)
            if item is None:
                continue
            if color == CYAN:
                item.setBackground(Qt.cyan)
            elif color == RED:
                item.setBackground(Qt.red)
            elif color == YELLOW:
                item.setBackground(Qt.yellow)
            elif color == GREEN:
                item.setBackground(Qt.green)
            else:
                item.setBackground(self.original_row_background_color)
            self.marked_rows_set.add(r)
        #END FOR
        self.sorts_locked = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_other_metadata(self,my_db,my_cursor,bookid):
        mi_dict = {}
        authors_list = self.get_authors(my_db,my_cursor,bookid)
        mi_dict[AUTHORS] = authors_list

        series = self.get_series(my_db,my_cursor,bookid)
        mi_dict['series'] = series
        if series > " ":
            if not series in self.series_set:    # sets ~ 20 times faster than lists for lookups
                self.series_set.add(series)

        book_file_name = self.get_book_file_name(my_db,my_cursor,bookid)
        mi_dict['book_file_name'] = book_file_name

        if self.local_prefs['CALIBRESPY_LOAD_PUBLISHER'] == 1:
            publisher = self.get_publisher(my_db,my_cursor,bookid)
            mi_dict[PUBLISHER] = publisher
            if publisher > " ":
                if not publisher in self.publisher_set:    # sets ~ 20 times faster than lists for lookups
                    self.publisher_set.add(publisher)
        else:
            mi_dict[PUBLISHER] = ""

        if self.local_prefs['CALIBRESPY_LOAD_TAGS'] == 1:

            tags_list = self.get_tags(my_db,my_cursor,bookid)

            mi_dict[TAGS] = tags_list
        else:
            mi_dict[TAGS] = ""

        if self.local_prefs['CALIBRESPY_LOAD_LANGUAGES'] == 1:
            languages_list = self.get_languages(my_db,my_cursor,bookid)
            mi_dict[LANGUAGES] = languages_list
        else:
            mi_dict[LANGUAGES] = ""

        if self.local_prefs['CALIBRESPY_LOAD_IDENTIFIERS'] == 1:     #Version 1.0.43
            identifiers_list = self.get_identifiers(my_db,my_cursor,bookid)
            mi_dict[IDENTIFIERS] = identifiers_list
        else:
            mi_dict[IDENTIFIERS] = None

        if self.local_prefs['CALIBRESPY_LOAD_RATINGS'] == 1:         #Version 1.0.52
            rating = self.get_rating(my_db,my_cursor,bookid)
            mi_dict[RATINGS] = rating
        else:
            mi_dict[RATINGS] = None

        if self.custom_columns_are_active:
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_1'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,1)
                mi_dict[CUSTOM_COLUMN_1] = cc
            else:
                mi_dict[CUSTOM_COLUMN_1] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_2'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,2)
                mi_dict[CUSTOM_COLUMN_2] = cc
            else:
                mi_dict[CUSTOM_COLUMN_2] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_3'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,3)
                mi_dict[CUSTOM_COLUMN_3] = cc
            else:
                mi_dict[CUSTOM_COLUMN_3] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_4'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,4)
                mi_dict[CUSTOM_COLUMN_4] = cc
            else:
                mi_dict[CUSTOM_COLUMN_4] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_5'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,5)
                if cc is not None:
                    mi_dict[CUSTOM_COLUMN_5] = cc
            else:
                mi_dict[CUSTOM_COLUMN_5] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_6'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,6)
                mi_dict[CUSTOM_COLUMN_6] = cc
            else:
                mi_dict[CUSTOM_COLUMN_6] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_7'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,7)
                mi_dict[CUSTOM_COLUMN_7] = cc
            else:
                mi_dict[CUSTOM_COLUMN_7] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_8'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,8)
                mi_dict[CUSTOM_COLUMN_8] = cc
            else:
                mi_dict[CUSTOM_COLUMN_8] = ""
            if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_9'] == 1:
                cc = self.get_custom_columns(my_db,my_cursor,bookid,9)
                mi_dict[CUSTOM_COLUMN_9] = cc
            else:
                mi_dict[CUSTOM_COLUMN_9] = ""

        return mi_dict
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_authors(self,my_db,my_cursor,bookid):
        authors_list = []
        authors_link_list = []
        mysql = "SELECT book,author FROM books_authors_link WHERE book = ?"
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return authors_list
        mysql = "SELECT id,sort,link FROM authors WHERE id = ?"
        for row in tmp_rows:
            book,authorid = row
            my_cursor.execute(mysql,([authorid]))
            tmp_rows2 = my_cursor.fetchall()
            if tmp_rows2 is None:
                tmp_rows2 = []
            if len(tmp_rows2) == 0:
                return authors_list
            for row2 in tmp_rows2:
                id,sort,link = row2
                sort = sort.strip()
                authors_list.append(sort)
                if link > "":
                    authors_link_list.append(link)
            #END FOR
        #END FOR
        del tmp_rows
        del tmp_rows2

        self.bookid_authors_link_dict[bookid] = authors_link_list
        del authors_link_list

        return authors_list
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_series(self,my_db,my_cursor,bookid):
        series = ""
        mysql = "SELECT book, (SELECT name FROM series WHERE series.id = books_series_link.series) \
                       FROM books_series_link WHERE books_series_link.book = ? "
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return series.strip()
        for row in tmp_rows:
            book,series = row
            break
        #END FOR
        del tmp_rows

        return series.strip()
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_publisher(self,my_db,my_cursor,bookid):
        publisher = ""
        mysql = "SELECT book, (SELECT name FROM publishers WHERE publishers.id = books_publishers_link.publisher) \
                       FROM books_publishers_link WHERE books_publishers_link.book = ? "
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return publisher.strip()
        for row in tmp_rows:
            book,publisher = row
        #END FOR
        del tmp_rows

        return publisher.strip()
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_tags(self,my_db,my_cursor,bookid):
        tags_list = []
        mysql = "SELECT book, (SELECT name FROM tags WHERE tags.id = books_tags_link.tag) \
                       FROM books_tags_link WHERE books_tags_link.book = ? "
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return tags_list
        for row in tmp_rows:
            book,tag = row
            tags_list.append(tag)
            self.tags_set.add(tag)
        #END FOR
        del tmp_rows

        return tags_list
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_languages(self,my_db,my_cursor,bookid):
        languages_list = []
        mysql = "SELECT book, (SELECT lang_code FROM languages WHERE languages.id = books_languages_link.lang_code) \
                       FROM books_languages_link WHERE books_languages_link.book = ? "
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return languages_list
        for row in tmp_rows:
            book,language = row
            languages_list.append(language)
        #END FOR
        del tmp_rows

        return languages_list
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_identifiers(self,my_db,my_cursor,bookid):
        identifiers_list = []
        mysql = "SELECT type,val FROM identifiers WHERE book = ? "
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return identifiers_list
        for row in tmp_rows:
            type,val = row
            r = type.strip() + ":" + val.strip()
            identifiers_list.append(r)
        #END FOR
        del tmp_rows

        identifiers_list.sort()

        return identifiers_list
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_rating(self,my_db,my_cursor,bookid):
        rating = ""
        mysql = "SELECT book, (SELECT rating FROM ratings WHERE ratings.id = books_ratings_link.rating) \
                       FROM books_ratings_link WHERE books_ratings_link.book = ? "
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        for row in tmp_rows:
            book,rating = row
        #END FOR
        del tmp_rows

        if rating == 1:
            rating = "✫"   #   == half-star (until Unicode 11+ migration has been completed)
        elif rating == 2:
            rating = "★"
        elif rating == 3:
            rating = "★✫"
        elif rating == 4:
            rating = "★★"
        elif rating == 5:
            rating = "★★✫"
        elif rating == 6:
            rating = "★★★"
        elif rating == 7:
            rating = "★★★✫"
        elif rating == 8:
            rating = "★★★★"
        elif rating == 9:
            rating = "★★★★✫"
        elif rating == 10:
            rating = "★★★★★"
        else:
            rating = ""

        return rating
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_custom_columns(self,my_db,my_cursor,bookid,n):
        #~ self.custom_column_label_dict[#label]   = id,#label,name,datatype,is_multiple,normalized,display

        try:
            if n == 1:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_1]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 2:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_2]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 3:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_3]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 4:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_4]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 5:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_5]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 6:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_6]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 7:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_7]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 8:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_8]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            elif n == 9:
                cc_label = self.custom_column_assignment_dict[CUSTOM_COLUMN_9]
                id,label,name,datatype,is_multiple,normalized,display = self.custom_column_label_dict[cc_label]
            else:
                return None

            normalized = int(normalized)

            cc_value = self.get_custom_column_data(my_db,my_cursor,bookid,id,datatype,is_multiple,normalized)

            if datatype == "datetime":
                cc_value = self.format_datetime_values(cc_label,cc_value)
            elif datatype == "int":
                self.cc_integer_col_number_set.add(n)
                length = len(cc_value)
                longest = self.cc_integer_max_length_dict[n]
                if length > longest:
                    self.cc_integer_max_length_dict[n] = length
        except Exception as e:
            if DEBUG: print("get_custom_columns exception: ", as_unicode(e))
            return None

        return cc_value
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_custom_column_data(self,my_db,my_cursor,bookid,id,datatype,is_multiple,normalized):
        id = as_unicode(id)

        if normalized == 1:
            if datatype == "series":
                mysql = "SELECT book,extra, \
                              (SELECT value FROM custom_column_[NN] WHERE id = books_custom_column_[NN]_link.value) \
                              FROM books_custom_column_[NN]_link WHERE book = ?".replace("[NN]",id)
            else:
                mysql = "SELECT book, \
                               (SELECT value FROM custom_column_[NN] WHERE id = books_custom_column_[NN]_link.value) \
                               FROM books_custom_column_[NN]_link WHERE book = ?".replace("[NN]",id)
        else:
            mysql = "SELECT book,value FROM custom_column_[NN] WHERE book = ?".replace("[NN]",id)

        tmp_list = []
        try:
            my_cursor.execute(mysql,([bookid]))
            tmp_rows = my_cursor.fetchall()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                if datatype == "series":
                    book,extra,value = row
                    value = value + " [" + as_unicode(extra) + "]"
                else:
                    book,value = row
                value = unicode_type(value)
                if datatype == "bool":
                    if value == "0":
                        value = "False"
                    else:
                        value = "True"
                if value > " ":
                    tmp_list.append(value.strip())
            #END FOR
        except Exception as e:
            if DEBUG: print("Error in 'get_custom_column_data': ", as_unicode(e), "  SQL: ", as_unicode(mysql))

        if len(tmp_list) == 0:
            cc_value = ""
        elif len(tmp_list) == 1: #either a simple text value, or is_multiple with only a single value
            cc_value = tmp_list[0].strip()
        else:  #is_multiple with multiple values
            cc_value = ""
            for value in tmp_list:
                cc_value = cc_value + value + ","
            #END FOR

        if cc_value.endswith(","):
            cc_value = cc_value[0:-1].strip()

        return cc_value
#---------------------------------------------------------------------------------------------------------------------------------------
    def format_datetime_values(self,cc_label,cc_value):
        #~ datetime format stored in sqlite:  2019-03-05 16:58:17.282000+00:00      "YYYY-MM-DD HH:MM:SS[+-]HH:MM"
        if not cc_value > " ":
            return ""
        if cc_value[0:2] == "01":
            return ""
        display_dict = self.custom_column_display_dict[cc_label]
        if not "date_format" in display_dict:
            return cc_value
        date_format = display_dict["date_format"]
        try:
            dtobject = dt_parse(cc_value, default=None, dayfirst=False)  #dt_parse comes from an open-source python date utils package used by Calibre
            cc_value = format_date(dtobject,date_format)  #format_date translates the user's custom column config date format string into a python date format string
        except Exception as e:
            if DEBUG: print("format_datetime_values exception: ", as_unicode(cc_value), as_unicode(date_format), as_unicode(e))
        return cc_value
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_table_custom_columns(self):

        cc1 = self.local_prefs['CUSTOM_COLUMN_1_SELECTED']
        cc2 = self.local_prefs['CUSTOM_COLUMN_2_SELECTED']
        cc3 = self.local_prefs['CUSTOM_COLUMN_3_SELECTED']
        cc4 = self.local_prefs['CUSTOM_COLUMN_4_SELECTED']
        cc5 = self.local_prefs['CUSTOM_COLUMN_5_SELECTED']
        cc6 = self.local_prefs['CUSTOM_COLUMN_6_SELECTED']
        cc7 = self.local_prefs['CUSTOM_COLUMN_7_SELECTED']
        cc8 = self.local_prefs['CUSTOM_COLUMN_8_SELECTED']
        cc9 = self.local_prefs['CUSTOM_COLUMN_9_SELECTED']

        if cc1.startswith("#"):
            s_split = cc1.split(">")
            cc1 = s_split[0].strip()
            cc1 = as_unicode(cc1)
            del s_split

        if cc2.startswith("#"):
            s_split = cc2.split(">")
            cc2 = s_split[0].strip()
            cc1 = as_unicode(cc2)
            del s_split

        if cc3.startswith("#"):
            s_split = cc3.split(">")
            cc3 = s_split[0].strip()
            cc1 = as_unicode(cc3)
            del s_split

        if cc4.startswith("#"):
            s_split = cc4.split(">")
            cc4 = s_split[0].strip()
            cc1 = as_unicode(cc4)
            del s_split

        if cc5.startswith("#"):
            s_split = cc5.split(">")
            cc5 = s_split[0].strip()
            cc1 = as_unicode(cc5)
            del s_split

        if cc6.startswith("#"):
            s_split = cc6.split(">")
            cc6 = s_split[0].strip()
            cc1 = as_unicode(cc6)
            del s_split

        if cc7.startswith("#"):
            s_split = cc7.split(">")
            cc7 = s_split[0].strip()
            cc1 = as_unicode(cc7)
            del s_split

        if cc8.startswith("#"):
            s_split = cc8.split(">")
            cc8 = s_split[0].strip()
            cc1 = as_unicode(cc8)
            del s_split

        if cc9.startswith("#"):
            s_split = cc9.split(">")
            cc9 = s_split[0].strip()
            cc1 = as_unicode(cc9)
            del s_split


        self.custom_column_assignment_dict = {}
        self.custom_column_assignment_dict[CUSTOM_COLUMN_1] = cc1
        self.custom_column_assignment_dict[CUSTOM_COLUMN_2] = cc2
        self.custom_column_assignment_dict[CUSTOM_COLUMN_3] = cc3
        self.custom_column_assignment_dict[CUSTOM_COLUMN_4] = cc4
        self.custom_column_assignment_dict[CUSTOM_COLUMN_5] = cc5
        self.custom_column_assignment_dict[CUSTOM_COLUMN_6] = cc6
        self.custom_column_assignment_dict[CUSTOM_COLUMN_7] = cc7
        self.custom_column_assignment_dict[CUSTOM_COLUMN_8] = cc8
        self.custom_column_assignment_dict[CUSTOM_COLUMN_9] = cc9

        cc_label_set = set()
        cc_label_set.add(cc1)
        cc_label_set.add(cc2)
        cc_label_set.add(cc3)
        cc_label_set.add(cc4)
        cc_label_set.add(cc5)
        cc_label_set.add(cc6)
        cc_label_set.add(cc7)
        cc_label_set.add(cc8)
        cc_label_set.add(cc9)

        self.custom_column_label_dict = {}
        self.custom_column_display_dict = {}

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("get_table_custom_columns: apsw_connect_to_library failed.")
            return

        try:
            mysql = "SELECT id,label,name,datatype,is_multiple,normalized,display FROM custom_columns"
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                id,label,name,datatype,is_multiple,normalized,display = row
                if datatype != "composite" and datatype != "ratings":
                    label = "#" + label
                    label = as_unicode(label)
                    if label in cc_label_set:
                        id = as_unicode(id)
                        r = id,label,name,datatype,is_multiple,normalized,display
                        self.custom_column_label_dict[label] = r
                        self.local_prefs[label] = name
                        if datatype == "datetime":
                            display = as_unicode(display)
                            display = ast.literal_eval(display)         #~ display:    {"date_format": "yyyy MMM dd", "description": ""}
                            if not isinstance(display,dict):
                                display = {}
                            self.custom_column_display_dict[label] = display
            #END FOR
            del tmp_rows
            del cc_label_set
            my_db.close()
        except Exception as e:
            #~ if never had any custom columns...
            my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_book_file_name(self,my_db,my_cursor,bookid):
        book_file_name = ""
        best_format = None
        mysql = "SELECT format,name FROM data WHERE book = ?"
        my_cursor.execute(mysql,([bookid]))
        tmp_rows = my_cursor.fetchall()
        if tmp_rows is None:
            tmp_rows = []
        if len(tmp_rows) == 0:
            book_file_name = None
            return book_file_name
        for row in tmp_rows:
            format,name = row
            format = format.lower()
            if format == self.local_prefs['PREFERRED_OUTPUT_FORMAT']:
                best_format = format
                book_file_name = name + "." + best_format
                break
        #END FOR
        if not best_format:
            for row in tmp_rows:
                format,name = row
                format = format.lower()
                book_file_name = name + "." + format
                break
            #END FOR
        del tmp_rows
        return book_file_name
#---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_connect_to_library(self):

        if isbytestring(self.library_path):
            self.library_path = self.library_path.decode(filesystem_encoding)
        self.library_path = self.library_path.replace(os.sep, '/')

        path = os.path.join(self.library_path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        self.library_metadatadb_path = path

        try:
            my_db = apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", as_unicode(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        #CalibreSpy is always read-only...except for library-specific preferences in its own physical settings table updated *instantly* (unlike metadata.db table 'preferences') when changes are indicated...
        mysql = "PRAGMA main.busy_timeout = 25000;"      #milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_program_path_values(self):
        self.path_to_desired_book_editor_program = self.local_prefs['CALIBRESPY_PROGRAM_PATH_EDITOR']
        self.path_to_desired_book_viewer_program = self.local_prefs['CALIBRESPY_PROGRAM_PATH_VIEWER']
        self.path_to_desired_book_other_program = self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER']
        self.path_to_calibre_smtp_program_command_file = self.local_prefs['CALIBRESPY_COMMAND_FILE_PATH_EMAIL']
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_calibrespy_icon_files(self):

        clidir = self.original_clidir
        self.local_prefs["CALIBRESPY_CLI_SUBDIRECTORY"] = self.original_clidir

        if isbytestring(clidir):
            clidir = clidir.decode(filesystem_encoding)
        clidir = clidir.replace(os.sep, '/')

        if not os.path.isdir(clidir):
            self.icon_calibrespy = None
            self.images_dir = ""
            if DEBUG: print("directory with resources does not exist: ", clidir, ".  User should execute the command file with .py script to reset multi-user locks and also fix OS UserName changes.")
            msg = "The directory for CalibreSpy resources (icons, etc.) '" + clidir + "' either does not exist, or there are insufficient security permissions to read it. \
            <br><br>To force CalibreSpy to recreate that critical directory, please uninstall CalibreSpy, restart Calibre, reinstall CalibreSpy, restart Calibre, then immediately execute CalibreSpy from its GUI icon. \
            <br><br>If you have changed your OS UserName, or are using a Library that was used by someone else until now, you should execute the command file with .py script to reset multi-user locks and also fix OS UserName changes."
            error_dialog(None, _('CalibreSpy'),_(msg), show=True)
            if DEBUG: print(msg)
            return False

        self.images_dir = os.path.join(clidir,"images")
        self.images_dir = os.path.join(clidir,"images")
        self.images_dir = self.images_dir.replace(os.sep,'/')

        self.path_to_calibrespy_icon_file = os.path.join(self.images_dir,"calibrespy.png")
        self.path_to_calibrespy_icon_file = self.path_to_calibrespy_icon_file.replace(os.sep,'/')

        if os.path.exists(self.path_to_calibrespy_icon_file):
            pixmap = QPixmap()
            pixmap.load(self.path_to_calibrespy_icon_file)
            self.icon_calibrespy = QIcon(pixmap)
            del pixmap
        else:
            self.icon_calibrespy = None

        icon_details_path = os.path.join(self.images_dir,"library_browser.png")
        icon_details_path = icon_details_path.replace(os.sep,'/')

        if os.path.exists(icon_details_path):
            pixmap = QPixmap(QSize(25,25))
            pixmap.load(icon_details_path)
            self.icon_details = QIcon(pixmap)
            del pixmap
        else:
            self.icon_details = None

        icon_cover_path = os.path.join(self.images_dir,"cover.png")
        icon_cover_path = icon_cover_path.replace(os.sep,'/')

        if os.path.exists(icon_cover_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_cover_path)
            self.icon_cover = QIcon(pixmap)
            del pixmap
        else:
            self.icon_cover = None

        icon_comments_path = os.path.join(self.images_dir,"comments.png")
        icon_comments_path = icon_comments_path.replace(os.sep,'/')

        if os.path.exists(icon_comments_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_comments_path)
            self.icon_comments = QIcon(pixmap)
            del pixmap
        else:
            self.icon_comments = None

        return True
#---------------------------------------------------------------------------------------------------------------------------------------
    def load_context_menu_icons(self):

        #~ --------------------------------------------------------
        #~ failsafe defaults...do not remove...
        #~ --------------------------------------------------------
        icon_book = self.icon_calibrespy.pixmap(QSize(20,20))
        icon_book = QIcon(icon_book)
        icon_viewer = icon_book
        icon_editor = icon_book
        #~ --------------------------------------------------------

        icon_book_path = os.path.join(self.images_dir,"book.png")
        icon_book_path = icon_book_path.replace(os.sep,'/')

        icon_viewer_path = os.path.join(self.images_dir,"viewer.png")
        icon_viewer_path = icon_viewer_path.replace(os.sep,'/')

        icon_editor_path = os.path.join(self.images_dir,"edit_book.png")
        icon_editor_path = icon_editor_path.replace(os.sep,'/')

        icon_gear_path = os.path.join(self.images_dir,"gear.png")
        icon_gear_path = icon_gear_path.replace(os.sep,'/')

        icon_filter_path = os.path.join(self.images_dir,"filter.png")
        icon_filter_path = icon_filter_path.replace(os.sep,'/')

        icon_clear_filters_path = os.path.join(self.images_dir,"clearfilters.png")
        icon_clear_filters_path = icon_clear_filters_path.replace(os.sep,'/')

        icon_filter_applied_path = os.path.join(self.images_dir,"filterapplied.png")
        icon_filter_applied_path = icon_filter_applied_path.replace(os.sep,'/')

        self.path_to_desired_book_other_icon = self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER_ICON']

        if self.path_to_desired_book_other_icon > " ":
            icon_other_path = self.path_to_desired_book_other_icon
        else:
            icon_other_path = icon_gear_path

        icon_contract_path = os.path.join(self.images_dir,"contract.png")
        icon_contract_path = icon_contract_path.replace(os.sep,'/')

        icon_expand_path = os.path.join(self.images_dir,"expand.png")
        icon_expand_path = icon_expand_path.replace(os.sep,'/')

        icon_visible_path = os.path.join(self.images_dir,"visible.png")
        icon_visible_path = icon_visible_path.replace(os.sep,'/')

        icon_invisible_path = os.path.join(self.images_dir,"invisible.png")
        icon_invisible_path = icon_invisible_path.replace(os.sep,'/')

        icon_clipboard_path = os.path.join(self.images_dir,"clipboard.png")
        icon_clipboard_path = icon_clipboard_path.replace(os.sep,'/')

        icon_directory_path = os.path.join(self.images_dir,"directory.png")
        icon_directory_path = icon_directory_path.replace(os.sep,'/')

        icon_email_path = os.path.join(self.images_dir,"email.png")
        icon_email_path = icon_email_path.replace(os.sep,'/')

        icon_sort_path = os.path.join(self.images_dir,"sort.png")
        icon_sort_path = icon_sort_path.replace(os.sep,'/')

        icon_ftp_path = os.path.join(self.images_dir,"ftp.png")
        icon_ftp_path = icon_ftp_path.replace(os.sep,'/')

        icon_wrench_hammer_path = os.path.join(self.images_dir,"wrench-hammer.png")
        icon_wrench_hammer_path = icon_wrench_hammer_path.replace(os.sep,'/')

        icon_piechart_path = os.path.join(self.images_dir,"piechart.png")
        icon_piechart_path = icon_piechart_path.replace(os.sep,'/')

        icon_formatspy_path = os.path.join(self.images_dir,"formatspy.png")
        icon_formatspy_path = icon_formatspy_path.replace(os.sep,'/')

        icon_marked_books_path = os.path.join(self.images_dir,"marked_books.png")
        icon_marked_books_path = icon_marked_books_path.replace(os.sep,'/')

        if os.path.exists(icon_book_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_book_path)
            icon_book = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_viewer_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_viewer_path)
            icon_viewer = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_editor_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_editor_path)
            icon_editor = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_other_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_other_path)
            icon_other = QIcon(pixmap)
            del pixmap
        else:
            icon_other = icon_book
            self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER_ICON'] = icon_book_path

        if os.path.exists(icon_gear_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_gear_path)
            icon_gear = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_filter_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_filter_path)
            icon_filter = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_clear_filters_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_clear_filters_path)
            icon_clear_filters = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_filter_applied_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_filter_applied_path)
            icon_filter_applied = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_contract_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_contract_path)
            icon_contract = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_expand_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_expand_path)
            icon_expand = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_visible_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_visible_path)
            icon_visible = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_invisible_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_invisible_path)
            icon_invisible = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_clipboard_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_clipboard_path)
            icon_clipboard = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_directory_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_directory_path)
            icon_directory = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_email_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_email_path)
            icon_email = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_sort_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_sort_path)
            icon_sort = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_ftp_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_ftp_path)
            icon_ftp = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_wrench_hammer_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_wrench_hammer_path)
            icon_wrench_hammer = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_piechart_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_piechart_path)
            icon_piechart = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_formatspy_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_formatspy_path)
            icon_formatspy = QIcon(pixmap)
            del pixmap

        if os.path.exists(icon_marked_books_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_marked_books_path)
            self.icon_marked_books = QIcon(pixmap)
            del pixmap

        self.icon_filter = icon_filter
        self.icon_clear_filters = icon_clear_filters
        self.icon_filter_applied = icon_filter_applied
        self.icon_gear = icon_gear
        self.icon_wrench_hammer = icon_wrench_hammer
        self.icon_ftp = icon_ftp
        self.icon_formatspy = icon_formatspy

        self.icon_book = icon_book      #Library Browser
        self.icon_viewer = icon_viewer  #Library Browser
        self.icon_editor = icon_editor   #Library Browser
        self.icon_email = icon_email     #Library Browser
        self.icon_directory = icon_directory #Library Browser

        return icon_book,icon_viewer,icon_editor,icon_other,icon_gear,icon_filter,icon_clear_filters,icon_contract,icon_expand,icon_visible,icon_invisible, icon_clipboard,icon_directory,icon_email,icon_sort,icon_ftp,icon_wrench_hammer, icon_piechart, icon_formatspy
#---------------------------------------------------------------------------------------------------------------------------------------
    def get_library_name(self):
        ssplit = self.library_path.split("/")
        n = len(ssplit)
        self.library_name = ssplit[n-1].strip()
        if DEBUG: print("self.library_name",self.library_name)
#---------------------------------------------------------------------------------------------------------------------------------------
    def build_matrix_context_menu(self):

        self.matrix.setContextMenuPolicy(Qt.CustomContextMenu)
        self.menu = QMenu("CalibreSpy",self.matrix)
        self.font.setPointSize(self.normal_fontsize - 1)
        self.menu.setFont(self.font)
        self.menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.menu.clear()
        self.menu.setTearOffEnabled(True)
        self.menu.setWindowTitle('CalibreSpy Main Menu')
        t = "<p style='white-space:wrap'>The specific application programs and their paths may be specified in CalibreSpy customization.\
                                                               <br><br>'Control+C' is inherited unchanged by CalibreSpy, and will copy a single cell (not row) to the clipboard. \
                                                               <br><br>'Copy Metadata to Clipboard' creates a <b><i>tab delimited</i></b> copy that is ready to 'paste special' into a spreadsheet, or just paste into text document.\
                                                               It also is used to perform multiple and simultaneous cross-CalibreSpy Search Queries when used with 'Clipboard Listening'. "
        self.menu.setToolTip(t)

        self.menu.addSeparator()

        icon_read,icon_view,icon_edit,icon_other,icon_gear,icon_filter,icon_clear_filters,icon_contract,icon_expand,icon_visible,icon_invisible, icon_clipboard,icon_directory,icon_email,icon_sort,icon_ftp,icon_wrench_hammer,icon_piechart, icon_formatspy = self.load_context_menu_icons()

        self.filter_menu = QMenu()
        self.filter_menu.setFont(self.font)
        self.filter_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.filter_menu.clear()
        self.filter_menu.setTearOffEnabled(True)
        self.filter_menu.setWindowTitle('Filtering')
        self.filter_menu.setTitle('Filtering')
        self.filter_menu.setIcon(icon_filter)

        self.filter_menu.setToolTip("<p style='white-space:wrap'>Filters may be applied or cleared as indicated.")

        self.filter_using_current_author_action = QAction(icon_filter,"Filter: Current Author",None)
        self.filter_using_current_author_action.setShortcut(QKeySequence("SHIFT+A"))
        self.filter_using_current_author_action.triggered.connect(self.context_filter_current_authors)
        self.filter_menu.addAction(self.filter_using_current_author_action)

        self.filter_using_current_title_action = QAction(icon_filter,"Filter: Current Title",None)
        self.filter_using_current_title_action.setShortcut(QKeySequence("SHIFT+B"))
        self.filter_using_current_title_action.triggered.connect(self.context_filter_current_title)
        self.filter_menu.addAction(self.filter_using_current_title_action)

        self.filter_using_current_series_action = QAction(icon_filter,"Filter: Current Series",None)
        self.filter_using_current_series_action.setShortcut(QKeySequence("SHIFT+S"))
        self.filter_using_current_series_action.triggered.connect(self.context_filter_current_series)
        self.filter_menu.addAction(self.filter_using_current_series_action)

        self.filter_using_current_tags_action = QAction(icon_filter,"Filter: Current Tags",None)
        self.filter_using_current_tags_action.setShortcut(QKeySequence("SHIFT+T"))
        self.filter_using_current_tags_action.triggered.connect(self.context_filter_current_tags)
        self.filter_menu.addAction(self.filter_using_current_tags_action)

        self.filter_using_current_publisher_action = QAction(icon_filter,"Filter: Current Publisher",None)
        self.filter_using_current_publisher_action.setShortcut(QKeySequence("SHIFT+P"))
        self.filter_using_current_publisher_action.triggered.connect(self.context_filter_current_publisher)
        self.filter_menu.addAction(self.filter_using_current_publisher_action)

        self.filter_using_languages_column_value_action = QAction(icon_filter,"Filter: Current Languages",None)
        self.filter_using_languages_column_value_action.setShortcut(QKeySequence("SHIFT+L"))
        self.filter_using_languages_column_value_action.triggered.connect(self.context_filter_languages_column_values)
        self.filter_menu.addAction(self.filter_using_languages_column_value_action)

        self.filter_using_custom_column_value_action = QAction(icon_filter,"Filter: Current Custom Column",None)
        self.filter_using_custom_column_value_action.setShortcut(QKeySequence("SHIFT+N"))
        self.filter_using_custom_column_value_action.triggered.connect(self.context_filter_generic_custom_column_values)
        self.filter_menu.addAction(self.filter_using_custom_column_value_action)

        self.filter_using_date_column_value_action = QAction(icon_filter,"Filter: Any Date Column",None)
        self.filter_using_date_column_value_action.setShortcut(QKeySequence("SHIFT+D"))
        self.filter_using_date_column_value_action.triggered.connect(self.context_filter_generic_date_column_values)
        self.filter_menu.addAction(self.filter_using_date_column_value_action)

        self.context_clear_all_filters_action = QAction(icon_clear_filters,"Clear All Filters",None)
        self.context_clear_all_filters_action.setShortcut(QKeySequence("SHIFT+C"))
        self.context_clear_all_filters_action.triggered.connect(self.context_clear_all_filters)
        self.filter_menu.addAction(self.context_clear_all_filters_action)

        self.menu.addMenu(self.filter_menu)

        self.menu.addSeparator()

        self.sort_menu = QMenu()
        self.sort_menu.setFont(self.font)
        self.sort_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.sort_menu.clear()
        self.sort_menu.setTearOffEnabled(True)
        self.sort_menu.setWindowTitle('Sorting')
        self.sort_menu.setTitle('Sorting')
        self.sort_menu.setIcon(icon_sort)

        self.sort_sequence_1_action = QAction(icon_sort,"Apply Default Sort Sequence",None)
        self.sort_sequence_1_action.triggered.connect(self.context_sort_sequence_1)
        self.sort_menu.addAction(self.sort_sequence_1_action)

        self.sort_sequence_2_action = QAction(icon_sort,"Sort Current Column ▲",None)
        self.sort_sequence_2_action.triggered.connect(self.context_sort_sequence_2)
        self.sort_menu.addAction(self.sort_sequence_2_action)

        self.sort_sequence_3_action = QAction(icon_sort,"Sort Current Column ▼",None)
        self.sort_sequence_3_action.triggered.connect(self.context_sort_sequence_3)
        self.sort_menu.addAction(self.sort_sequence_3_action)
        self.menu.addMenu(self.sort_menu)

        self.menu.addSeparator()

        self.read_action = QAction(icon_read,'Read',None)
        self.read_action.setShortcut(QKeySequence("Ctrl+R"))
        self.read_action.triggered.connect(self.open_current_book)
        self.menu.addAction(self.read_action)

        if self.path_to_desired_book_viewer_program:
            self.view_action = QAction(icon_view,'View', None)
            self.view_action.setShortcut(QKeySequence("Ctrl+V"))
            self.view_action.triggered.connect(self.view_current_book)
            self.menu.addAction(self.view_action)

        if self.path_to_desired_book_editor_program:
            self.edit_action = QAction(icon_edit,'Edit', None)
            self.edit_action.setShortcut(QKeySequence("Ctrl+E"))
            self.edit_action.triggered.connect(self.edit_current_book)
            self.menu.addAction(self.edit_action)

        if self.path_to_desired_book_other_program:
            head,fname = os.path.split(self.path_to_desired_book_other_program)
            fname = fname.replace(".exe","")
            self.other_action = QAction(icon_other,fname,None)
            self.other_action.setShortcut(QKeySequence("Ctrl+O"))
            self.other_action.triggered.connect(self.other_action_current_book)
            self.menu.addAction(self.other_action)
            self.push_button_other_current_book.setText(fname)  #default was set to 'Other'

        if self.path_to_calibre_smtp_program_command_file:
            self.email_action = QAction(icon_email,'Email',None)
            self.email_action.setShortcut(QKeySequence("Ctrl+M"))
            self.email_action.triggered.connect(self.email_current_book)
            self.menu.addAction(self.email_action)

        self.open_dir_action = QAction(icon_directory,"Open Path",None)
        self.open_dir_action.setShortcut(QKeySequence("Ctrl+D"))
        self.open_dir_action.triggered.connect(self.open_format_path)
        self.menu.addAction(self.open_dir_action)

        self.menu_save_copy = QMenu()
        self.menu_save_copy.setFont(self.font)
        self.menu_save_copy.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.menu_save_copy.clear()
        self.menu_save_copy.setTearOffEnabled(True)
        self.menu_save_copy.setWindowTitle('Save Copy of Book(s)')
        self.menu_save_copy.setTitle('Save Copy of Book(s)')
        self.menu_save_copy.setIcon(icon_directory)

        self.save_copy_single_action = QAction(icon_directory,"Save Copy:  Single Book",None)
        self.save_copy_single_action.setShortcut(QKeySequence("Ctrl+S"))
        self.save_copy_single_action.triggered.connect(self.save_copy_single)
        self.menu_save_copy.addAction(self.save_copy_single_action)

        self.save_copy_filtered_action = QAction(icon_directory,"Save Copy:  All Visible Books",None)
        self.save_copy_filtered_action.setShortcut(QKeySequence("Ctrl+F"))
        self.save_copy_filtered_action.triggered.connect(self.save_copy_filtered)
        self.menu_save_copy.addAction(self.save_copy_filtered_action)

        self.menu.addMenu(self.menu_save_copy)

        self.library_browser_action = QAction(self.icon_details,"Library Browser",None)
        self.library_browser_action.setShortcut(QKeySequence("Ctrl+B"))
        self.library_browser_action.triggered.connect(self.library_browser_context)
        self.menu.addAction(self.library_browser_action)

        self.view_cover_action = QAction(self.icon_cover,"View Cover",None)
        self.view_cover_action.setShortcut(QKeySequence("Ctrl+I"))
        self.view_cover_action.triggered.connect(self.view_current_cover)
        self.menu.addAction(self.view_cover_action)

        self.view_comments_single_action = QAction(self.icon_comments,"View Comments (Single)",None)
        self.view_comments_single_action.setShortcut(QKeySequence("Ctrl+H"))
        self.view_comments_single_action.triggered.connect(self.view_current_comments)
        self.menu.addAction(self.view_comments_single_action)

        self.view_comments_displayed_action = QAction(self.icon_comments,"View Comments (All Displayed)",None)
        self.view_comments_displayed_action.setShortcut(QKeySequence("Ctrl+A"))
        self.view_comments_displayed_action.triggered.connect(self.view_displayed_comments)
        self.menu.addAction(self.view_comments_displayed_action)

        self.view_author_links_action = QAction(icon_directory,"View Author Link",None)
        self.view_author_links_action.setShortcut(QKeySequence("Ctrl+L"))
        self.view_author_links_action.triggered.connect(self.context_authors_link)
        self.menu.addAction(self.view_author_links_action)

        self.calibre_links_menu = QMenu()
        self.calibre_links_menu.setFont(self.font)
        self.calibre_links_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.calibre_links_menu.clear()
        self.calibre_links_menu.setTearOffEnabled(True)
        self.calibre_links_menu.setTitle('Copy Calibre Links to Clipboard')
        self.calibre_links_menu.setIcon(self.icon_book)
        self.calibre_links_menu.aboutToShow.connect(self.calibre_links_menu_about_to_show)

        self.menu.addMenu(self.calibre_links_menu)

        self.menu.addSeparator()

        self.context_copy_metadata_to_clipboard_action = QAction(icon_clipboard,"Copy Metadata to Clipboard",None)
        self.context_copy_metadata_to_clipboard_action.setShortcut(QKeySequence("ALT+C"))   # CTRL+C for QTableWidgets defaults to "Copy the Current Cell Contents to the Clipboard"...
        self.context_copy_metadata_to_clipboard_action.triggered.connect(self.context_copy_metadata_to_clipboard)
        self.menu.addAction(self.context_copy_metadata_to_clipboard_action)

        self.clipboard_searches_menu = QMenu()
        self.clipboard_searches_menu.setFont(self.font)
        self.clipboard_searches_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.clipboard_searches_menu.clear()
        self.clipboard_searches_menu.setTearOffEnabled(True)
        self.clipboard_searches_menu.setWindowTitle('Clipboard Related Search Queries')
        self.clipboard_searches_menu.setTitle('Clipboard Related Queries')
        self.clipboard_searches_menu.setIcon(self.icon_binoculars)

        self.context_start_listening_to_clipboard_action = QAction(self.icon_binoculars,"Start Clipboard Listening for CalibreSpy Search Query",None)
        self.context_start_listening_to_clipboard_action.triggered.connect(self.start_clip_listener)
        self.clipboard_searches_menu.addAction(self.context_start_listening_to_clipboard_action)

        self.context_stop_listening_to_clipboard_action = QAction(self.icon_binoculars,"Stop Clipboard Listening for CalibreSpy Search Query",None)
        self.context_stop_listening_to_clipboard_action.triggered.connect(self.stop_clip_listener)
        self.clipboard_searches_menu.addAction(self.context_stop_listening_to_clipboard_action)

        self.clipboard_searches_menu.addSeparator()

        self.context_search_titles_using_title_list_in_clipboard_action = QAction(self.icon_binoculars,"Search Titles Using List of Titles in Clipboard",None)
        self.context_search_titles_using_title_list_in_clipboard_action.triggered.connect(self.execute_title_search_from_clipboard)
        self.clipboard_searches_menu.addAction(self.context_search_titles_using_title_list_in_clipboard_action)

        self.context_search_cc_using_text_in_clipboard_action = QAction(self.icon_binoculars,"Search Current Custom Column Using Value in Clipboard",None)
        self.context_search_cc_using_text_in_clipboard_action.triggered.connect(self.execute_custom_column_search_from_clipboard)
        self.clipboard_searches_menu.addAction(self.context_search_cc_using_text_in_clipboard_action)

        t = "<p style='white-space:wrap'>CalibreSpy has 3 methods of searches involving the system Clipboard:\
                                                               <br>---------------------------------------------------------------------\
                                                               <br>[Method #1] Search for books in as many other CalibreSpy Library 'windows' as you wish.\
                                                               <br>There are only 2 steps to doing this:\
                                                               <br>[Step 1] Turn 'Clipboard Listening' on in all of the other CalibreSpy windows in which you would like to be search targets;<br>\
                                                               <br>[Step 2] In the 'Search-From' CalibreSpy Library window, simply copy the metadata to the Clipboard (ALT+C)\
                                                               <br><br>Each of the Clipboard Listeners 'listen' every " + as_unicode(int(CLIPBOARD_LISTENER_INTERVAL/1000)) + " seconds for a new Clipboard Search Query.  \
                                                               Each Listener then executes filters using the Author(s) and Title copied from the Clipboard. \
                                                               The Search will try to match on Title even if the Author(s) are not found, and then on Series as a last resort. \
                                                               A message is displayed at the bottom of the target's window indicating what was or was not found.\
                                                               <br>---------------------------------------------------------------------\
                                                               <br>[Method #2] Search Titles Using List of Titles in Clipboard.<br>Use single textual values <i>or</i> textual lists separated by a CRLF or EOL ('return' or 'enter') or the '|' symbol.\
                                                               <br>---------------------------------------------------------------------\
                                                               <br>[Method #3] Search Current Custom Column Using Value in Clipboard.<br>Use single textual values (not lists).\
                                                               <br>---------------------------------------------------------------------"
        self.clipboard_searches_menu.setToolTip(t)

        self.menu.addMenu(self.clipboard_searches_menu)

        self.menu.addSeparator()

        self.context_optimize_current_column_action = QAction(icon_expand,"Optimize Current Column Width",None)
        self.context_optimize_current_column_action.setShortcut(QKeySequence("ALT+O"))
        self.context_optimize_current_column_action.triggered.connect(self.optimize_current_column_width)
        self.menu.addAction(self.context_optimize_current_column_action)

        self.context_deoptimize_current_column_action = QAction(icon_contract,"Deoptimize Current Column Width",None)
        self.context_deoptimize_current_column_action.setShortcut(QKeySequence("ALT+D"))
        self.context_deoptimize_current_column_action.triggered.connect(self.deoptimize_current_column_width)
        self.menu.addAction(self.context_deoptimize_current_column_action)

        self.menu.addSeparator()

        self.context_hide_current_column_action = QAction(icon_invisible,"Hide Current Column",None)
        self.context_hide_current_column_action.setShortcut(QKeySequence("ALT+H"))
        self.context_hide_current_column_action.triggered.connect(self.hide_current_column)
        self.menu.addAction(self.context_hide_current_column_action)

        self.context_unhide_current_column_action = QAction(icon_visible,"Unhide All Columns",None)
        self.context_unhide_current_column_action.setShortcut(QKeySequence("ALT+U"))
        self.context_unhide_current_column_action.triggered.connect(self.unhide_all_columns)
        self.menu.addAction(self.context_unhide_current_column_action)

        self.menu.addSeparator()

        self.miscellany_menu = QMenu()
        self.miscellany_menu.setFont(self.font)
        self.miscellany_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.miscellany_menu.clear()
        self.miscellany_menu.setTearOffEnabled(True)
        self.miscellany_menu.setWindowTitle('Miscellaneous Tools')
        self.miscellany_menu.setTitle('Miscellaneous Tools')
        self.miscellany_menu.setIcon(icon_wrench_hammer)

        self.miscellany_menu.setToolTip("<p style='white-space:wrap'>Miscellaneous tools and functions.")

        self.miscellany_ftp_choose_active_device_action = QAction(icon_ftp,"FTP Choose Active Target Host",None)
        self.miscellany_ftp_choose_active_device_action.triggered.connect(self.context_ftp_choose_active_device)
        self.miscellany_menu.addAction(self.miscellany_ftp_choose_active_device_action)

        self.miscellany_ftp_send_action = QAction(icon_ftp,"FTP Upload Book to Host",None)
        self.miscellany_ftp_send_action.triggered.connect(self.context_ftp_book_send)
        self.miscellany_menu.addAction(self.miscellany_ftp_send_action)

        self.miscellany_ftp_delete_action = QAction(icon_ftp,"FTP Delete Book from Host",None)
        self.miscellany_ftp_delete_action.triggered.connect(self.context_ftp_book_delete)
        self.miscellany_menu.addAction(self.miscellany_ftp_delete_action)

        self.miscellany_menu.addSeparator()

        self.miscellany_arbitrary_files_action = QAction(icon_wrench_hammer,"Select && Open Arbitrary Programs && Files",None)
        self.miscellany_arbitrary_files_action.triggered.connect(self.context_arbitrary_files)
        self.miscellany_menu.addAction(self.miscellany_arbitrary_files_action)

        self.miscellany_menu.addSeparator()

        self.miscellany_visualize_metadata_action = QAction(icon_piechart,"Visualize Metadata w/Export Option",None)
        self.miscellany_visualize_metadata_action.triggered.connect(self.context_visualize_metadata)
        self.miscellany_menu.addAction(self.miscellany_visualize_metadata_action)

        self.miscellany_menu.addSeparator()

        self.miscellany_formatspy_action = QAction(icon_formatspy,"FormatSpy",None)
        self.miscellany_formatspy_action.triggered.connect(self.context_formatspy)
        self.miscellany_menu.addAction(self.miscellany_formatspy_action)

        self.menu.addMenu(self.miscellany_menu)

        self.qaction = QAction(self.icon_calibrespy, 'CalibreSpy Main Menu', None)
        self.qaction.setMenu(self.menu)

        self.matrix.addAction(self.qaction)

        self.font.setPointSize(self.normal_fontsize)
#---------------------------------------------------------------------------------------------------------------------------------------
    def build_row_header_context_menu(self):
        self.matrix.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
        self.row_header_menu = QMenu("CalibreSpy",self.matrix.verticalHeader())
        self.row_header_menu.setIcon(self.icon_gear)
        self.font.setPointSize(self.normal_fontsize + 1)
        self.row_header_menu.setFont(self.font)
        self.row_header_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.font.setPointSize(self.normal_fontsize)
        self.row_header_menu.clear()
        self.row_header_menu.setWindowTitle('Row Heading Menu')
        self.row_header_menu.setTearOffEnabled(True)

        t = "<p style='white-space:wrap'>Books require a Path for in order to be Marked (colorized).  Format-less Books may not be marked in CalibreSpy.\
                                                              <br><br>Book Markings will be saved at exit <b>only</b> if you could otherwise save any CalibreSpy preferences and settings.\
                                                              <br><br>By first selecting a single row, and then right-clicking that same row's row-header, \
                                                               you may manually mass-change all row heights to equal the selected row's current \
                                                               height that was previously manually changed by dragging the row header 'down' or 'up'. "

        self.row_header_menu.setToolTip(t)

        self.context_change_all_row_heights_action = QAction(self.icon_gear,"Mass change all Row Heights",None)
        self.context_change_all_row_heights_action.setShortcut(QKeySequence("ALT+W"))
        self.context_change_all_row_heights_action.triggered.connect(self.change_all_row_heights)
        self.row_header_menu.addAction(self.context_change_all_row_heights_action)

        self.context_reset_all_row_heights_action = QAction(self.icon_gear,"Mass reset all Row Heights",None)
        self.context_reset_all_row_heights_action.setShortcut(QKeySequence("ALT+R"))
        self.context_reset_all_row_heights_action.triggered.connect(self.reset_all_row_heights)
        self.row_header_menu.addAction(self.context_reset_all_row_heights_action)

        self.row_header_menu.addSeparator()

        self.context_mark_current_book_cyan_action = QAction(self.icon_marked_books,"Mark Current Book-Cyan",None)
        self.context_mark_current_book_cyan_action.setShortcut(QKeySequence("ALT+M"))
        self.context_mark_current_book_cyan_action.triggered.connect(self.mark_current_book_cyan)
        self.row_header_menu.addAction(self.context_mark_current_book_cyan_action)

        self.context_mark_current_book_yellow_action = QAction(self.icon_marked_books,"Mark Current Book-Yellow",None)
        self.context_mark_current_book_yellow_action.setShortcut(QKeySequence("ALT+Y"))
        self.context_mark_current_book_yellow_action.triggered.connect(self.mark_current_book_yellow)
        self.row_header_menu.addAction(self.context_mark_current_book_yellow_action)

        self.context_mark_current_book_red_action = QAction(self.icon_marked_books,"Mark Current Book-Red",None)
        self.context_mark_current_book_red_action.setShortcut(QKeySequence("ALT+E"))
        self.context_mark_current_book_red_action.triggered.connect(self.mark_current_book_red)
        self.row_header_menu.addAction(self.context_mark_current_book_red_action)

        self.context_mark_current_book_green_action = QAction(self.icon_marked_books,"Mark Current Book-Green",None)
        self.context_mark_current_book_green_action.setShortcut(QKeySequence("ALT+G"))
        self.context_mark_current_book_green_action.triggered.connect(self.mark_current_book_green)
        self.row_header_menu.addAction(self.context_mark_current_book_green_action)

        self.row_header_menu.addSeparator()

        self.context_unmark_current_book_action = QAction(self.icon_marked_books,"Unmark Current Book",None)
        self.context_unmark_current_book_action.setShortcut(QKeySequence("ALT+X"))
        self.context_unmark_current_book_action.triggered.connect(self.unmark_current_book)
        self.row_header_menu.addAction(self.context_unmark_current_book_action)

        self.context_unmark_all_books_action = QAction(self.icon_marked_books,"Unmark All Books",None)
        self.context_unmark_all_books_action.setShortcut(QKeySequence("ALT+A"))
        self.context_unmark_all_books_action.triggered.connect(self.unmark_all_books)
        self.row_header_menu.addAction(self.context_unmark_all_books_action)

        self.row_header_menu.addSeparator()

        self.context_filter_marked_books_action = QAction(self.icon_marked_books,"Filter Marked Books",None)
        self.context_filter_marked_books_action.setShortcut(QKeySequence("ALT+F"))
        self.context_filter_marked_books_action.triggered.connect(self.filter_marked_books)
        self.row_header_menu.addAction(self.context_filter_marked_books_action)

        self.row_header_menu.addSeparator()

        self.context_filter_marked_books_cyan_action = QAction(self.icon_marked_books,"Filter Marked Books: Cyan",None)
        self.context_filter_marked_books_cyan_action.triggered.connect(self.filter_marked_books_cyan)
        self.row_header_menu.addAction(self.context_filter_marked_books_cyan_action)

        self.context_filter_marked_books_yellow_action = QAction(self.icon_marked_books,"Filter Marked Books: Yellow",None)
        self.context_filter_marked_books_yellow_action.triggered.connect(self.filter_marked_books_yellow)
        self.row_header_menu.addAction(self.context_filter_marked_books_yellow_action)

        self.context_filter_marked_books_green_action = QAction(self.icon_marked_books,"Filter Marked Books: Green",None)
        self.context_filter_marked_books_green_action.triggered.connect(self.filter_marked_books_green)
        self.row_header_menu.addAction(self.context_filter_marked_books_green_action)

        self.context_filter_marked_books_red_action = QAction(self.icon_marked_books,"Filter Marked Books: Red",None)
        self.context_filter_marked_books_red_action.triggered.connect(self.filter_marked_books_red)
        self.row_header_menu.addAction(self.context_filter_marked_books_red_action)

        self.row_header_qaction = QAction(self.icon_gear, 'Context Menu', None)
        self.row_header_qaction.setMenu(self.row_header_menu)

        self.matrix.verticalHeader().addAction(self.row_header_qaction)

        self.row_heights_were_mass_changed = False
#---------------------------------------------------------------------------------------------------------------------------------------
    def build_column_header_context_menu(self):
        self.matrix.horizontalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
        self.column_header_menu = QMenu("CalibreSpy",self.matrix.horizontalHeader())
        self.column_header_menu.setIcon(self.icon_filter)
        self.font.setPointSize(self.normal_fontsize + 1)
        self.column_header_menu.setFont(self.font)
        self.column_header_menu.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        self.font.setPointSize(self.normal_fontsize)
        self.column_header_menu.clear()
        self.column_header_menu.setWindowTitle('Column Heading Menu')
        self.column_header_menu.setTearOffEnabled(True)

        t = "<p style='white-space:wrap'>By first selecting a single column, and then right-clicking that same column's  column-header, you may hide all 'blank' or 'non-blank' column values."
        self.column_header_menu.setToolTip(t)

        self.context_filter_blank_column_values_action = QAction(self.icon_filter,"Filter-Away Blank Values",None)
        self.context_filter_blank_column_values_action.setShortcut(QKeySequence("ALT+B"))
        self.context_filter_blank_column_values_action.triggered.connect(self.filter_blank_column_values)
        self.column_header_menu.addAction(self.context_filter_blank_column_values_action)

        self.context_filter_nonblank_column_values_action = QAction(self.icon_filter,"Filter-Away Non-Blank Values",None)
        self.context_filter_nonblank_column_values_action.setShortcut(QKeySequence("ALT+N"))
        self.context_filter_nonblank_column_values_action.triggered.connect(self.filter_nonblank_column_values)
        self.column_header_menu.addAction(self.context_filter_nonblank_column_values_action)

        self.column_header_qaction = QAction(self.icon_gear, 'Context Menu', None)
        self.column_header_qaction.setMenu(self.column_header_menu)

        self.matrix.horizontalHeader().addAction(self.column_header_qaction)


        t = "<p style='white-space:wrap'>Right-Click the matrix, its column headings, and its row headings to use their respective context menus."
        self.matrix.setToolTip(t)

        if self.is_cli:
            s = "CalibreSpy was executed using a command file (CLI).<br><br>Right-Click the matrix, its column headings, and its row headings to use their respective context menus."
        else:
            s = "CalibreSpy was executed from the Calibre GUI plug-in icon.<br><br>Right-Click the matrix, its column headings, and row row headings to use their respective context menus."
        t = "<p style='white-space:wrap'>" + s
        self.setToolTip(t)
#---------------------------------------------------------------------------------------------------------------------------------------
    def do_final_inits(self):

        self.clip = None
        self.clip_listener = None
        self.clip_search_query = ""
        self.clip_search_query_previous = ""

        self.regex = None

        self.ftp = None

        self.is_on_top = False

        self.open_url = None

        self.audio_book_extensions_set = set()
        self.audio_book_extensions_set.add(".mp3")
        self.audio_book_extensions_set.add(".aac")
        self.audio_book_extensions_set.add(".aax")
        self.audio_book_extensions_set.add(".m4a")
        self.audio_book_extensions_set.add(".m4b")
        self.audio_book_extensions_set.add(".m4p")
        self.audio_book_extensions_set.add(".ogg")
        self.audio_book_extensions_set.add(".wma")
        self.audio_book_extensions_set.add(".flac")
        self.audio_book_extensions_set.add(".alac")

        self.restore_matrix_header_state()

        self.added_list = []
        item = self.matrix.item(0,self.added_column)
        if item:
            if item.text() > "":
                self.added_list.append('added')  # for is_list_empty

        self.modified_list = []
        item = self.matrix.item(0,self.modified_column)
        if item:
            if item.text() > "":
                self.modified_list.append('modified')  # for is_list_empty

        #  .restoreState() overwrites the following, so these *must* be done as late as possible.
        self.matrix.verticalHeader().setSectionResizeMode(QHeaderView.ResizeMode.Interactive)    #user must be able to change the widths manually
        self.matrix.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeMode.Interactive)
        self.matrix.horizontalHeader().setStretchLastSection(False)   #due to custom columns...
        self.matrix.horizontalHeader().setSectionsClickable(False)  #since row number changing via arbitrary sorting causes corruption of the mapping of row to bookid...

        self.hide_show_optional_columns()

        self.setSizeGripEnabled(True)

        self.matrix.setTabKeyNavigation(True)

        self.original_row_height = self.matrix.rowHeight(0)

        item = self.matrix.item(0,0)
        if item:
            self.original_row_background_color = item.background()

        self.row_to_bookid_dict = {}
        self.bookid_to_row_dict = {}
        self.marked_rows_set = set()
        self.marked_bookids_set = set()
        self.marked_bookids_color_dict = {}
        self.rebuild_row_to_bookid_dict_now()  # .restoreState() messes with the just-loaded row numbers, causing spurious markings to be restored below...

        s = self.local_prefs['CALIBRESPY_MARKED_ROW_SNAPSHOT']
        s = s + MARKED_BOOK_SEPARATOR
        s_split = s.split(MARKED_BOOK_SEPARATOR)

        for row in s_split:
            row = row.strip()
            if "," in row:
                r_split = row.split(",")
                bookid = r_split[0]
                color = r_split[1]
                bookid = bookid.strip()
                if bookid.isdigit():
                    bookid = int(bookid)
                    self.marked_bookids_set.add(bookid)
                    self.marked_bookids_color_dict[bookid] = color
        #END FOR
        del s_split
        self.local_prefs['CALIBRESPY_MARKED_ROW_SNAPSHOT'] = ""

        for bookid in self.marked_bookids_set:
            color = self.marked_bookids_color_dict[bookid]
            r = self.bookid_to_row_dict[bookid]
            item = self.matrix.item(r,0)
            if item is None:
                continue
            if color == CYAN:
                item.setBackground(Qt.cyan)
            elif color == RED:
                item.setBackground(Qt.red)
            elif color == YELLOW:
                item.setBackground(Qt.yellow)
            elif color == GREEN:
                item.setBackground(Qt.green)
            else:
                item.setBackground(self.original_row_background_color)
            self.marked_rows_set.add(r)

        self.vl_is_active = False

        self.library_browser_is_active = False

        self.unsupported_vl_terms_list = []
        #~ -----------------------------------------------------------------
        self.unsupported_vl_terms_list.append("ondevice:")  #db cache only
        self.unsupported_vl_terms_list.append("marked:")  #db cache only
        self.unsupported_vl_terms_list.append("selected:")  #invalid; user error
        self.unsupported_vl_terms_list.append("author_sort:true")  #nonsensical
        self.unsupported_vl_terms_list.append("author_sort:false")  #nonsensical
        self.unsupported_vl_terms_list.append("title_sort:true")  #nonsensical
        self.unsupported_vl_terms_list.append("title_sort:false")  #nonsensical
        self.unsupported_vl_terms_list.append(" title:true")  #nonsensical
        self.unsupported_vl_terms_list.append(" title:false")  #nonsensical
        #~ -----------------------------------------------------------------
        self.unsupported_vl_terms_startswith_list = []
        self.unsupported_vl_terms_startswith_list.append("title:true")  #nonsensical
        self.unsupported_vl_terms_startswith_list.append("title:false") #nonsensical

        self.matrix.setFocus()
        self.matrix.setCurrentCell(0,0)
#---------------------------------------------------------------------------------------------------------------------------------------
    def hide_show_optional_columns(self):

        if len(self.series_list) == 0 or self.local_prefs['CALIBRESPY_LOAD_SERIES'] == 0:
            self.matrix.setColumnHidden(self.series_column,True)
            self.matrix.setColumnHidden(self.index_column,True)
            self.series_filter_combobox.setVisible(False)
            self.series_count_qlabel.setVisible(False)
        else:
            self.matrix.setColumnHidden(self.series_column,False)
            self.matrix.setColumnHidden(self.index_column,False)
            self.series_filter_combobox.setVisible(True)
            self.series_count_qlabel.setVisible(True)

        if len(self.tags_list) == 0 or self.local_prefs['CALIBRESPY_LOAD_TAGS'] == 0:
            self.matrix.setColumnHidden(self.tags_column,True)
            self.tags_filter_combobox.setVisible(False)
            self.tags_count_qlabel.setVisible(False)
        else:
            self.matrix.setColumnHidden(self.tags_column,False)
            self.tags_filter_combobox.setVisible(True)
            self.tags_count_qlabel.setVisible(True)

        if len(self.publisher_list) == 0 or self.local_prefs['CALIBRESPY_LOAD_PUBLISHER'] == 0:
            self.matrix.setColumnHidden(self.publisher_column,True)
            self.publisher_filter_combobox.setVisible(False)
            self.publisher_count_qlabel.setVisible(False)
        else:
            self.matrix.setColumnHidden(self.publisher_column,False)
            self.publisher_filter_combobox.setVisible(True)
            self.publisher_count_qlabel.setVisible(True)

        if len(self.languages_list) == 0 or self.local_prefs['CALIBRESPY_LOAD_LANGUAGES'] == 0:
            self.matrix.setColumnHidden(self.languages_column,True)
        else:
            self.matrix.setColumnHidden(self.languages_column,False)

        if len(self.added_list) == 0 or self.local_prefs['CALIBRESPY_LOAD_TIMESTAMP'] == 0:
            self.matrix.setColumnHidden(self.added_column,True)
        else:
            self.matrix.setColumnHidden(self.added_column,False)

        if len(self.modified_list) == 0 or self.local_prefs['CALIBRESPY_LOAD_MODIFIED'] == 0:
            self.matrix.setColumnHidden(self.modified_column,True)
        else:
            self.matrix.setColumnHidden(self.modified_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_1'] == 0:
            self.matrix.setColumnHidden(self.cc1_column,True)
        else:
            self.matrix.setColumnHidden(self.cc1_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_2'] == 0:
            self.matrix.setColumnHidden(self.cc2_column,True)
        else:
            self.matrix.setColumnHidden(self.cc2_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_3'] == 0:
            self.matrix.setColumnHidden(self.cc3_column,True)
        else:
            self.matrix.setColumnHidden(self.cc3_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_4'] == 0:
            self.matrix.setColumnHidden(self.cc4_column,True)
        else:
            self.matrix.setColumnHidden(self.cc4_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_5'] == 0:
            self.matrix.setColumnHidden(self.cc5_column,True)
        else:
            self.matrix.setColumnHidden(self.cc5_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_6'] == 0:
            self.matrix.setColumnHidden(self.cc6_column,True)
        else:
            self.matrix.setColumnHidden(self.cc6_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_7'] == 0:
            self.matrix.setColumnHidden(self.cc7_column,True)
        else:
            self.matrix.setColumnHidden(self.cc7_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_8'] == 0:
            self.matrix.setColumnHidden(self.cc8_column,True)
        else:
            self.matrix.setColumnHidden(self.cc8_column,False)

        if self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_9'] == 0:
            self.matrix.setColumnHidden(self.cc9_column,True)
        else:
            self.matrix.setColumnHidden(self.cc9_column,False)

        #~ -------------- add newly available columns below ----------------------
        if self.local_prefs['CALIBRESPY_LOAD_IDENTIFIERS'] == 0:        # Version 1.0.43  Identifiers Added
            self.matrix.setColumnHidden(self.identifiers_column,True)
        else:
            if self.identifiers_loaded:
                self.matrix.setColumnHidden(self.identifiers_column,False)
            else:  #Customization changed to "load-it=True" after not originally loaded; cannot show until after restarted since no data.
                self.matrix.setColumnHidden(self.identifiers_column,True)

        if self.local_prefs['CALIBRESPY_LOAD_RATINGS'] == 0:             # Version 1.0.52  Ratings Added
            self.matrix.setColumnHidden(self.ratings_column,True)
        else:
            if self.ratings_loaded:
                self.matrix.setColumnHidden(self.ratings_column,False)
            else:  #Customization changed to "load-it=True" after not originally loaded; cannot show until after restarted since no data.
                self.matrix.setColumnHidden(self.ratings_column,True)
#---------------------------------------------------------------------------------------------------------------------------------------
    def elapsed_time_message(self,seq,msg):
        if seq == 0:
            self.start_time = datetime.now()
            return
        elif seq == 1:
            return
        elif seq == 9:
            elapsed = as_unicode(datetime.now() - self.start_time)
            elapsed = elapsed[2:-3]
            msg = "Total Start Time for " + as_unicode(self.n_matrix_rows) + " Books: " + as_unicode(elapsed)
            self.change_bottom_message(msg)
#---------------------------------------------------------------------------------------------------------------------------------------
    def show_special_messages(self):
        if self.msg is not None:
            if self.no_customize_label.isVisible():
                if self.simple_shutdown_is_sufficient:
                    pass  #normal scenario...first come, first served...
                else: #not normal at all.  crash...and lock must expire...
                    hrs = self.local_prefs['CALIBRESPY_ORPHANED_MULTIUSER_LOCK_ELAPSED_HOURS']
                    self.msg = "Customization blocked for up to: " + as_unicode(hrs) + " hours."
                    self.change_bottom_message(self.msg)
            else:
                self.push_button_customize.hide()
                self.no_customize_label.show()
                self.msg = self.msg + " Customization is briefly unavailable."
                self.change_bottom_message(self.msg)

        if self.multiuser_is_in_play:
            msg = "Entirely RO: CS Markings & Settings Will Not Be Saved"
            self.change_bottom_message(msg)
#---------------------------------------------------------------------------------------------------------------------------------------
    def keyPressEvent(self, event):
        if event.key() == Qt.Key_Escape:
            self.save_and_exit()
        else:
            QDialog.keyPressEvent(self, event)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# REGEX SEARCH FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def do_search(self):

        if self.regex is None:
            import re
            self.regex = re
            del re

        self.current_target = self.search_targets_combobox.currentText()
        if self.current_target == SEARCH_TARGET_HEADING_NAME:
            return
        if not self.current_target in self.search_target_column_mapping_dict:
            return

        self.target_column = self.search_target_column_mapping_dict[self.current_target]
        self.current_regex = self.search_expressions_combobox.currentText()

        i = self.search_expressions_combobox.findText(self.current_regex)
        if i < 0:
            self.search_expressions_combobox.insertItem(0,self.current_regex)

        try:
            p = self.regex.compile(self.current_regex, self.regex.IGNORECASE|self.regex.MULTILINE)
        except Exception as e:
            if DEBUG: print("ERROR:  p = re.compile()",as_unicode(e))
            msg = "Invalid RE: " + self.current_regex + "<br><br>" +  as_unicode(e)
            return error_dialog(self, 'Search Using Regular Expression',msg, show=True)

        for r in self.visible_rows_set:
            item = self.matrix.item(r,self.target_column)
            if item:
                match = p.search(item.text())
                if not match:
                    self.matrix.setRowHidden(r,True)
        #END FOR

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_title(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_series(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_tags(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_publisher(source=SOURCE_FILTER_AFTER_SEARCH )

        self.matrix.setFocus()

        self.save_search_regex_history()
#---------------------------------------------------------------------------------------------------------------------------------------
    def save_search_regex_history(self):
        item = ""
        for i in range(0,self.search_expressions_combobox.count()):
            v = self.search_expressions_combobox.itemText(i).strip()
            if v:
                if v > "":
                    item = item + v + SEARCH_SEPARATOR
        #END FOR
        self.local_prefs['HISTORICAL_SEARCH_EXPRESSIONS'] = item
#---------------------------------------------------------------------------------------------------------------------------------------
    def load_search_regex_history(self):

        icon_regex_path = os.path.join(self.images_dir,"regex.png")
        icon_regex_path = icon_regex_path.replace(os.sep,'/')
        pixmap = QPixmap(QSize(20,20))
        pixmap.load(icon_regex_path)
        self.icon_regex = QIcon(pixmap)
        del pixmap

        history = self.local_prefs['HISTORICAL_SEARCH_EXPRESSIONS']
        history = history + SEARCH_SEPARATOR
        history_list = history.split(SEARCH_SEPARATOR)
        history_list = history_list[0:SEARCH_MAXIMUM_VALUES]
        history_list = list(set(history_list))
        history_list.sort()

        self.search_expressions_combobox.insertItem(0,SEARCH_EXPRESSIONS_HEADING_VALUE)  #always index 0
        self.search_expressions_combobox.setItemIcon(0,self.icon_regex)

        i = 1
        for regex in history_list:
            regex = regex.strip()
            if regex > "" and regex != SEARCH_EXPRESSIONS_HEADING_VALUE:
                self.search_expressions_combobox.addItem(regex)
                self.search_expressions_combobox.setItemIcon(i,self.icon_regex)
                i = i + 1
        #END FOR
        del history
        del history_list
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# CLIPBOARD SEARCH QUERY FUNCTIONS:
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def start_clip_listener(self):
        if not self.clip:
            self.clip = QApplication.clipboard()
        self.clip_listener = QTimer(self)
        self.clip_listener.timeout.connect(self.check_clipboard_for_search_query)
        self.clip_listener.start(CLIPBOARD_LISTENER_INTERVAL)
        self.listening_icon_label.show()
        self.listening_text_label.hide()
#---------------------------------------------------------------------------------------------------------------------------------------
    def stop_clip_listener(self):
        if self.clip_listener is not None:
            self.clip_listener.stop()
            del self.clip_listener
        self.listening_text_label.show()
        self.listening_icon_label.hide()
#---------------------------------------------------------------------------------------------------------------------------------------
    def check_clipboard_for_search_query(self):
        self.clip_search_query = self.clip.text()
        if self.clip_search_query is not None:
            if isinstance(self.clip_search_query,unicode_type):
                if self.clip_search_query != self.clip_search_query_previous:
                    self.clip_search_query_previous = self.clip_search_query
                    s = as_unicode(self.clip_search_query)
                    if HEADING_STRING in s and METADATA_STRING in s :
                        self.execute_clipboard_search_query()
#---------------------------------------------------------------------------------------------------------------------------------------
    def execute_clipboard_search_query(self):
        msg = "[Clipboard Search: New Query in Progress...]"
        if DEBUG: print(msg)
        self.change_bottom_message(msg)
        s_split = self.clip_search_query.split("METADATA:")
        head = s_split[0]
        head_list = head.split(TAB)
        del head
        if head_list[0] == "HEADING:":  #always should...
            head_list = head_list[1: ]
        meta = s_split[1]
        del s_split
        metadata_list = meta.split(TAB)
        del meta
        self.column_search_query_data_mapping_dict = {}

        #~ if DEBUG:
            #~ print("........................head_list: ")
            #~ for r in head_list:
                #~ print(as_unicode(r))
            #~ #END FOR
            #~ print("........................metadata_list: ")
            #~ for r in metadata_list:
                #~ print(as_unicode(r))
            #~ #END FOR
            #~ print("................................................. ")

        if len(metadata_list) > 0:
            for c in range(0,len(head_list)):
                k = head_list[c]
                if not isinstance(k,unicode_type):
                    k = as_unicode(k)
                k = k.strip()
                try:
                    v = metadata_list[c]
                    v = v.strip()
                    if v.endswith(","):
                        v = v[0:-1]
                    self.column_search_query_data_mapping_dict[k.strip()] = v.strip()
                except:
                    self.column_search_query_data_mapping_dict[k.strip()] = ""
            #END FOR
        else:
            if DEBUG: print("metadata_list has no rows of data...")

        del metadata_list
        del head_list

        authors = self.column_search_query_data_mapping_dict[AUTHORS_HEADING]
        title = self.column_search_query_data_mapping_dict[TITLE_HEADING]
        series = self.column_search_query_data_mapping_dict[SERIES_HEADING]

        self.clear_all_filters()  #also then does a Garbage Collection if CLI.

        nothing_was_found = True

        i = self.authors_filter_combobox.findText(authors)
        if i > -1:
            self.authors_filter_combobox.setCurrentIndex(i)
            i = self.title_filter_combobox.findText(title)
            nothing_was_found = False
            if i > -1:
                self.title_filter_combobox.setCurrentIndex(i)
                msg = "[Clipboard Search:  Both Authors and Title were found]"
                self.change_bottom_message(msg)
            else:
                msg = "[Clipboard Search:  Only Authors were found]"
                self.change_bottom_message(msg)
        else:
            msg = "[Clipboard Search: " + authors + "  not found]"
            self.change_bottom_message(msg)
            i = self.title_filter_combobox.findText(title)
            if i > -1:
                self.title_filter_combobox.setCurrentIndex(i)
                msg = "[Clipboard Search:  only Title was found]"
                self.change_bottom_message(msg)
                nothing_was_found = False
            else:
                if series is not None:
                    if series != "":
                        i = self.series_filter_combobox.findText(series)
                        if i > -1:
                            self.series_filter_combobox.setCurrentIndex(i)
                            msg = "[Clipboard Search:  Series found (but not Authors and Title)]"
                            self.change_bottom_message(msg)
                            nothing_was_found = False
        #END IF
        if DEBUG: print(msg)
        if nothing_was_found:
            msg = '[Clipboard Search: nothing was found]'
            if DEBUG: print(msg)
            self.change_bottom_message(msg)
            self.visible_rows_set.clear()
            for r in range(0,self.n_matrix_rows):
                self.matrix.setRowHidden(r,True)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  SPECIAL SEARCHES FOR TITLE AND CUSTOM COLUMNS USING CLIPBOARD
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def execute_title_search_from_clipboard(self):
        self.execute_generic_search_from_clipboard(source=TITLE_SOURCE)
#---------------------------------------------------------------------------------------------------------------------------------------
    def execute_custom_column_search_from_clipboard(self):
        self.execute_generic_search_from_clipboard(source=CUSTOM_COLUMN_SOURCE)
#---------------------------------------------------------------------------------------------------------------------------------------
    def execute_generic_search_from_clipboard(self,source=None):
        if not source:
            return

        if source == TITLE_SOURCE:
            search_column = self.title_column
        else:
            search_column = self.matrix.currentColumn()
            if not search_column in self.cc_column_set:  #not a custom column's column
                return

        if not self.clip:
            self.clip = QApplication.clipboard()

        search_text = self.clip.text()

        if not isinstance(search_text,unicode_type):
            return

        if not search_text > " ":
            return

        search_text = search_text.replace("|",NEWLINE)  #textual lists separated by CRLF or EOL or the '|' symbol

        current_regex = ""
        search_list = search_text.split("\n")
        if isinstance(search_list,list):
            for line in search_list:
                line = line.strip()
                if line > " ":
                    current_regex = current_regex + line.strip() + "|"
            #END FOR
            current_regex = current_regex[0:-1]
        else:
            if orig_text > " ":
                current_regex = line.strip()
        del search_text
        del search_list

        if not current_regex > " ":
            return

        if self.regex is None:
            import re
            self.regex = re
            del re

        try:
            p = self.regex.compile(current_regex, self.regex.IGNORECASE|self.regex.MULTILINE)
        except Exception as e:
            if DEBUG: print("ERROR:  p = re.compile()",as_unicode(e))
            msg = "Invalid RE: " + current_regex + "<br><br>" +  as_unicode(e)
            return error_dialog(self, 'Search Titles or Custom Columns Using Regular Expression',msg, show=True)

        for r in range(0,self.n_matrix_rows):
            item = self.matrix.item(r,search_column)
            if item:
                match = p.search(item.text())
                if not match:
                    self.matrix.setRowHidden(r,True)
                else:
                    self.matrix.setRowHidden(r,False)
        #END FOR

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_title(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_series(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_tags(source=SOURCE_FILTER_AFTER_SEARCH )
        self.filter_by_publisher(source=SOURCE_FILTER_AFTER_SEARCH )
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  VERSION FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def verify_ftp_data_version(self):
        #encode legacy userid and password and resave (if possible)

        self.ftp_decoded = 0

        if self.local_prefs['CALIBRESPY_FTP_ENCODING_ENABLED'] == 0:  #never encoded previously

            raw_in = self.local_prefs['CALIBRESPY_FTP_USERID']
            raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
            u = codecs.encode(raw_in, 'rot13')
            self.local_prefs['CALIBRESPY_FTP_USERID'] = u

            raw_in = self.local_prefs['CALIBRESPY_FTP_PASSWORD']
            raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
            p = codecs.encode(raw_in, 'rot13')
            self.local_prefs['CALIBRESPY_FTP_PASSWORD'] = p

            if not self.session_is_entirely_readonly:
                if self.user_can_save_settings:
                    self.local_prefs['CALIBRESPY_FTP_ENCODING_ENABLED'] =1
                    self.save_local_prefs_specific(pref1='CALIBRESPY_FTP_ENCODING_ENABLED',pref2='CALIBRESPY_FTP_USERID',pref3='CALIBRESPY_FTP_PASSWORD')
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  MULTI-USER FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def check_multiuser_status(self,my_db,my_cursor):

        username = None
        userdatetime = None

        for x,y in my_cursor.execute("SELECT prefkey,prefvalue FROM _calibrespy_settings WHERE prefkey LIKE 'CALIBRESPY_LAST_UPDATED%' "):
            if x == 'CALIBRESPY_LAST_UPDATED_USERNAME':
                self.local_prefs['CALIBRESPY_LAST_UPDATED_USERNAME'] = y
                username = y
            elif x == 'CALIBRESPY_LAST_UPDATED_DATETIME':
                self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'] = y
                userdatetime = y
        #END FOR

        if (not username) or (not userdatetime):
            self.multiuser_is_in_play = False
        elif self.local_prefs['CALIBRESPY_LAST_UPDATED_USERNAME'] != self.username:
            if self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'] != "":
                if self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'] != self.username_timestamp:
                     self.multiuser_is_in_play = True
                else:
                    self.multiuser_is_in_play = False
            else:
                self.multiuser_is_in_play = False
        else:  #user has 2 instances of CalibreSpy looking at the identical Library.
            if self.local_prefs['CALIBRESPY_LAST_UPDATED_DATETIME'] != self.username_timestamp:
                self.multiuser_is_in_play = True
            else:
                self.multiuser_is_in_play = False

        if not self.user_can_save_settings:  #at startup always False
            if not self.multiuser_is_in_play:
                self.user_can_save_settings = True
        else:  #once allowed, shouldn't turn off for remainder of session.
            pass
#---------------------------------------------------------------------------------------------------------------------------------------
    def check_startup_shutdown_status(self):

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        must_close_db = True
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)
        self.check_multiuser_status(my_db,my_cursor)
        my_db.close()

        if not self.user_can_save_settings:  #per self.check_multiuser_status
            self.simple_shutdown_is_sufficient = True
            return

        if not 'CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL' in self.local_prefs:
            self.local_prefs['CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'] = 1

        if not'CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL' in self.local_prefs:
            self.local_prefs['CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL'] = 1

        if self.local_prefs['CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'] == 0:  #CalibreSpy crash due to programming bug (or user intentionally crashing it directly or indirectly)
            self.multiuser_is_in_play = False
            self.simple_shutdown_is_sufficient = True
            if DEBUG: print("CalibreSpy Last Startup was NOT Graceful")
        else:
            if self.local_prefs['CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL'] == 0:  #could be in current use by another person...
                if self.multiuser_is_in_play:  #normal scenario
                    self.simple_shutdown_is_sufficient = True
                else:  # not a multiuser issue, so not normal...
                    self.simple_shutdown_is_sufficient = False  # crash caused an orphan lock that needs to expire
                    if DEBUG: print("CalibreSpy Last Shutdown was NOT Graceful")
            else:
                self.simple_shutdown_is_sufficient = True

        if self.local_prefs['CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'] == 0:
            self.msg = "CalibreSpy's 'Last Startup' Was Not Successful. "
        elif self.local_prefs['CALIBRESPY_LAST_SHUTDOWN_WAS_GRACEFUL'] == 0:
            self.msg = "CalibreSpy's 'Last Shutdown' Was Not Normal. "

        if not self.multiuser_is_in_play:
            if self.user_can_save_settings:
                self.local_prefs['CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'] = 0  #default for this current session
                pref1 = 'CALIBRESPY_LAST_STARTUP_WAS_GRACEFUL'
                self.save_local_prefs_specific(pref1)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  DRAG-AND-DROP FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def drop_search_results(self):
        from calibre_plugins.calibrespy.drop_widget_dialog import DropSearchResultsDialog
        self.dsr_dialog = DropSearchResultsDialog(self.icon_dsr,self.pixmap_dsr,self.font,self.style_text,self.process_file_paths)
        self.dsr_dialog.setAttribute(Qt.WA_DeleteOnClose )
        self.dsr_dialog.show()
        del DropSearchResultsDialog
        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def process_file_paths(self,book_list):

        if len(book_list) == 0:
            return

        book_set = set(book_list)
        del book_list

        for r in range(0,self.n_matrix_rows):
            if r in self.row_to_bookid_dict:
                bookid = self.row_to_bookid_dict[r]
                if bookid in book_set:
                    self.matrix.setRowHidden(r,False)
                else:
                    self.matrix.setRowHidden(r,True)
            else:  #path column is empty...
                self.matrix.setRowHidden(r,True)
        #END FOR

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_title(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_series(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_tags(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_publisher(source=SOURCE_FILTER_AFTER_SEARCH)

        self.matrix.setFocus()

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  DUPLICATE BOOKS SEARCH FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def do_duplicates_search(self):
        # just an interbook search, but for the purpose of finding duplicate books.  keeps it simple for the user to understand.
        self.do_interbook_search(source="duplicates")
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  INTER-BOOK (CROSS-BOOK) SEARCH FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def do_interbook_search(self,source=None):
        #example:  find all books that have any other books with both the same author and the same #original_title.  compare every book to every other book.

        n = self.search_targets_combobox.count()
        column_1_list = []
        column_2_list = []
        for i in range(0,n):
            column = self.search_targets_combobox.itemText(i)
            if column == SEARCH_TARGET_HEADING_NAME:
                continue
            column_1_list.append(column)
            column_2_list.append(column)
        #END FOR

        if not source:
            title =   "Inter-Book Query"
        else:
            title = "Search: Duplicates"
        label = "Column #1  for All Books               "
        column_1,ok = QInputDialog.getItem(None, title, label, column_1_list,0,False)
        if not ok:
            del column_1_list
            del column_2_list
            return

        column_2_list.remove(column_1)

        if not source:
            title =   "Inter-Book Query"
        else:
            title = "Search: Duplicates"
        label = "Column #2  for All Books              "
        column_2,ok = QInputDialog.getItem(None, title, label, column_2_list,0,False)
        if not ok:
            del column_1_list
            del column_2_list
            return

        self.perform_interbook_search(column_1,column_2)

        self.matrix.setFocus()

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def perform_interbook_search(self,column_1,column_2):

        if not column_1 in self.search_target_column_mapping_dict:
            return

        if not column_2 in self.search_target_column_mapping_dict:
            return

        target_column1 = self.search_target_column_mapping_dict[column_1]
        target_column2 = self.search_target_column_mapping_dict[column_2]

        crossbook_working_dict = {}    # row  = val1,val2

        for r in range(0,self.n_matrix_rows):
            self.matrix.setRowHidden(r,True)  #hide all rows to start
            item1 = self.matrix.item(r,target_column1)
            if not item1:
                continue
            item2 = self.matrix.item(r,target_column2)
            if not item2:
                continue
            val1 = item1.text()
            val2 = item2.text()
            if val1 > " " and val2 > " ":
                crossbook_working_dict[r] = val1,val2
        #END FOR

        matches_set = set()

        i = 1
        #~ for k_a,v_a in crossbook_working_dict.iteritems():
        for k_a,v_a in iteritems(crossbook_working_dict):
            i = i + 1
            val1a,val2a = v_a
            #~ for k_b,v_b in crossbook_working_dict.iteritems():
            for k_b,v_b in iteritems(crossbook_working_dict):
                val1b,val2b = v_b
                if k_a == k_b:
                    continue
                if val1a == val1b and val2a == val2b:
                    match = k_a,k_b
                    matches_set.add(match)
                    match = k_b,k_a
                    if not match in matches_set:
                        matches_set.add(match)
            #END FOR
        #END FOR

        del crossbook_working_dict

        for match in matches_set:
            r1,r2 = match
            r1 = int(r1)
            r2 = int(r2)
            self.matrix.setRowHidden(r1,False)
            self.matrix.setRowHidden(r2,False)
        #END FOR

        del matches_set

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_title(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_series(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_tags(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_publisher(source=SOURCE_FILTER_AFTER_SEARCH)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  'CONTAINS' SEARCH FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def do_contains_search(self):

        n = self.search_targets_combobox.count()
        column_1_list = []
        column_2_list = []
        for i in range(0,n):
            column = self.search_targets_combobox.itemText(i)
            if column == SEARCH_TARGET_HEADING_NAME:
                continue
            column_1_list.append(column)
            column_2_list.append(column)
        #END FOR

        title =   "Contains Query"
        label = " Column #1               "
        column_1,ok = QInputDialog.getItem(None, title, label, column_1_list,0,False)
        if not ok:
            del column_1_list
            del column_2_list
            return

        column_2_list.remove(column_1)

        title =   "Contains Query"
        label = "Column #2             "
        column_2,ok = QInputDialog.getItem(None, title, label, column_2_list,0,False)
        if not ok:
            del column_1_list
            del column_2_list
            return

        self.perform_contains_search(column_1,column_2)

        self.matrix.setFocus()

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def perform_contains_search(self,column_1,column_2):

        if not column_1 in self.search_target_column_mapping_dict:
            return

        if not column_2 in self.search_target_column_mapping_dict:
            return

        target_column1 = self.search_target_column_mapping_dict[column_1]
        target_column2 = self.search_target_column_mapping_dict[column_2]

        for r in range(0,self.n_matrix_rows):
            self.matrix.setRowHidden(r,True)  #hide all rows to start
            item1 = self.matrix.item(r,target_column1)
            if not item1:
                continue
            item2 = self.matrix.item(r,target_column2)
            if not item2:
                continue
            val1 = item1.text()
            val2 = item2.text()
            if val1 > " " and val2 > " ":   # e.g.  Title    Series
                if val2 in val1:                   # e.g.  if Series in Title
                    self.matrix.setRowHidden(r,False)
        #END FOR

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_title(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_series(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_tags(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_publisher(source=SOURCE_FILTER_AFTER_SEARCH)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  SQL QUERY SEARCH FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def do_sqlquery_search(self):

        try:
            self.calibrespysqlquery.close()
        except:
            pass

        sql_prefs = {}
        #~ for k,v in self.local_prefs.iteritems():
        for k,v in iteritems(self.local_prefs):
            if k.startswith("SQL"):
                sql_prefs[k] = v
        #END FOR

        from calibre_plugins.calibrespy.sqlquery_search_dialog import CalibreSpySQLQuery
        self.calibrespysqlquery = CalibreSpySQLQuery(self.icon_sqlquery,self.font,self.style_text,self.library_metadatadb_path,self.display_sqlquery_search_results,sql_prefs,self.return_new_local_pref_value)
        self.calibrespysqlquery.setAttribute(Qt.WA_DeleteOnClose )
        self.calibrespysqlquery.show()
        del CalibreSpySQLQuery
        del sql_prefs

        self.matrix.setFocus()

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def display_sqlquery_search_results(self,bookid_set):

        for r in range(0,self.n_matrix_rows):
            if not r in self.row_to_bookid_dict:  #e.g. has no path, since has no formats, so has no bookid in Calibrespy...
                self.matrix.setRowHidden(r,True)
                continue
            bookid = self.row_to_bookid_dict[r]
            if bookid in bookid_set:
                self.matrix.setRowHidden(r,False)
            else:
                self.matrix.setRowHidden(r,True)
        #END FOR

        self.get_visible_row_count()

        self.filter_by_authors(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_title(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_series(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_tags(source=SOURCE_FILTER_AFTER_SEARCH)
        self.filter_by_publisher(source=SOURCE_FILTER_AFTER_SEARCH)

        if self.is_cli:
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#  SQLITE USER FUNCTIONS
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_user_function_regexp(self,regexpr,avalue):
        #http://www.sqlite.org/lang_expr.html:  The "X REGEXP Y" operator will be implemented as a call to "regexp(Y,X)"
        #---------------------------------------------------------------------------------------------------------------------------------------
        #mysql = 'SELECT id FROM custom_column_8 WHERE value REGEXP '^.+$'
        #---------------------------------------------------------------------------------------------------------------------------------------
        if regexpr:
            if avalue:
                try:
                    s_string = unicode_type(avalue)
                    re_string = unicode_type(regexpr)
                    self.regex.escape("\\")
                    p = self.regex.compile(re_string, self.regex.IGNORECASE|self.regex.DOTALL|self.regex.MULTILINE)
                    match = p.search(s_string)
                    if match:
                        #~ if DEBUG: print("apsw_user_function_regexp:  matched:", as_unicode(regexpr), as_unicode(avalue))
                        return True
                    else:
                        return False
                except Exception as e:
                    if DEBUG: print("apsw_user_function_regexp:  error: ", as_unicode(e), as_unicode(regexpr), as_unicode(avalue))
                    return False
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
# end of calibrespy_dialog.py