# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.89"  # Qt6 Customization

import apsw,os,sys
import codecs

from qt.core import (Qt, QDialog, QFont, QIcon, QGroupBox,QPushButton,
                                       QHBoxLayout, QVBoxLayout, QWidget, QSpinBox, pyqtSignal,
                                       QLabel, QLineEdit, QComboBox, QColor, QCheckBox, QCompleter,
                                       QScrollArea, QSize, QDialogButtonBox,
                                       QTableWidget, QTableWidgetItem, QAbstractItemView)

from calibre.constants import DEBUG,iswindows,isosx
from calibre.gui2 import gprefs, error_dialog, warning_dialog

from polyglot.builtins import as_unicode, iteritems, range, unicode_type

from calibre_plugins.calibrespy.config import (prefs,
                                                                            DISPLAY_SIZE_TINY, DISPLAY_SIZE_SMALL, DISPLAY_SIZE_NORMAL,
                                                                            AUTHORS_COLUMN, TITLE_COLUMN, SERIES_COLUMN, INDEX_COLUMN,
                                                                            TAGS_COLUMN, PUBLISHED_COLUMN, PUBLISHER_COLUMN, LANGUAGES_COLUMN,
                                                                            ADDED_COLUMN, MODIFIED_COLUMN, PATH_COLUMN,
                                                                            CUSTOM_COLUMN_1_COLUMN, CUSTOM_COLUMN_2_COLUMN, CUSTOM_COLUMN_3_COLUMN,
                                                                            CUSTOM_COLUMN_4_COLUMN, CUSTOM_COLUMN_5_COLUMN, CUSTOM_COLUMN_6_COLUMN,
                                                                            CUSTOM_COLUMN_7_COLUMN, CUSTOM_COLUMN_8_COLUMN, CUSTOM_COLUMN_9_COLUMN,
                                                                            IDENTIFIERS_COLUMN, RATINGS_COLUMN, MATRIX_MAXIMUM_COLUMNS_COUNT,    #Version 1.0.43: Identifiers  #Version 1.0.52: Ratings
                                                                            CUSTOM_COLUMN_1_COMBOBOX_HEADER, CUSTOM_COLUMN_2_COMBOBOX_HEADER,
                                                                            CUSTOM_COLUMN_3_COMBOBOX_HEADER, CUSTOM_COLUMN_4_COMBOBOX_HEADER,
                                                                            CUSTOM_COLUMN_5_COMBOBOX_HEADER, CUSTOM_COLUMN_6_COMBOBOX_HEADER,
                                                                            CUSTOM_COLUMN_7_COMBOBOX_HEADER, CUSTOM_COLUMN_8_COMBOBOX_HEADER,
                                                                            CUSTOM_COLUMN_9_COMBOBOX_HEADER)
#-----------------------------------------------------------------------------------------
class CalibreSpyLibraryConfigWidgetBase(QDialog):
    #---------------------------------------
    #~ CalibreSpy does *not* save any prefs to the global Calibre prefs, gprefs, since *many* Libraries can be opened by CalibreSpy at one time.
    #~ Calibre gprefs were *not* designed assuming that was normal, so bad things could happen to gui.json, causing Calibre to reset it.
    #---------------------------------------
    def __init__(self, parent):
        QDialog.__init__(self, parent)
    def resize_dialog_to_preferences(self):
        size = QSize()
        size.setWidth(self.local_prefs['CUSTOMIZE_SIZE_WIDTH_LAST'])
        size.setHeight(self.local_prefs['CUSTOMIZE_SIZE_HEIGHT_LAST'])
        self.resize(size)
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class CalibreSpyLibraryConfigWidget(CalibreSpyLibraryConfigWidgetBase):

    resized_signal = pyqtSignal()

    def __init__(self,QApplication,icon_image,library_path,apsw_connect_to_library,change_local_prefs_after_customizing,style_text):

        parent = None
        CalibreSpyLibraryConfigWidgetBase.__init__(self, parent)

        self.QApplication = QApplication

        self.setWindowFlags(Qt.Window | Qt.WindowTitleHint | Qt.WindowStaysOnTopHint)
        self.setModal(True)
        if icon_image:
            self.setWindowIcon(icon_image)
        self.setWindowTitle("CalibreSpy Library-Specific Preferences")

        self.library_path = library_path
        self.apsw_connect_to_library = apsw_connect_to_library
        self.change_local_prefs_after_customizing = change_local_prefs_after_customizing

        self.local_prefs = {}
        #~ for k,v in prefs.iteritems():
        for k,v in iteritems(prefs):
            if not k in self.local_prefs:
                self.local_prefs[k] = v
        #END FOR
        self.cc_comboxes_signals_blocked = True
        self.create_column_assignment_dict()
        self.get_metadatadb_local_prefs()

        #-----------------------------------------------------
        self.setStyleSheet(style_text)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.font = QFont()
        self.font.setFamily(self.local_prefs['DEFAULT_FONT_FAMILY'])
        self.font.setBold(False)
        self.font.setPointSize(self.local_prefs['DEFAULT_FONT_SIZE'])
        self.normal_fontsize = self.local_prefs['DEFAULT_FONT_SIZE']
        #-----------------------------------------------------
        #-----------------------------------------------------

        self.global_tip = "<p style='white-space:wrap'>\
              CalibreSpy preferences are Library-specific, \
              but when first created were originally defaulted from your 'Global Defaults'.\
             <br><br>Once CalibreSpy has been used for a specific Library, \
             that Library will no longer reference those Global Defaults, \
             but thereafter will reference and update the Library-specific preferences set here.\
             <br><br>Please read the ToolTips here by hovering your mouse."
        self.setToolTip(self.global_tip)
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout_frame)
        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignCenter)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(500,500)

        self.scroll_area_frame.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self.scroll_area_frame.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)

        self.layout_frame.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_frame
        #-----------------------------------------------------
        # NOTE: the self.scroll_area_frame.setWidget(self.scroll_widget) is at the end of the init() AFTER all children have been created and assigned to a layout...
        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_frame.addWidget(self.scroll_widget)           # causes automatic reparenting of QWidget to the parent of self.layout_frame, which is:  self .
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignCenter)
        #-----------------------------------------------------
        self.scroll_widget.setLayout(self.layout_top)                 # causes automatic reparenting of any widget later added below to the above parent
        #-----------------------------------------------------
        self.font.setPointSize(self.normal_fontsize + 2)
        self.font.setBold(True)
        self.global_defaults_message_qlabel = QLabel()
        self.global_defaults_message_qlabel.setTextFormat(Qt.RichText)
        self.global_defaults_message_qlabel.setText(self.library_path)
        self.global_defaults_message_qlabel.setAlignment(Qt.AlignCenter)
        self.global_defaults_message_qlabel.setToolTip(self.global_tip)
        self.global_defaults_message_qlabel.setFont(self.font)
        self.layout_top.addWidget(self.global_defaults_message_qlabel)
        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)

        self.paths_groupbox = QGroupBox("CalibreSpy Paths:")
        self.layout_top.addWidget(self.paths_groupbox)

        self.paths_layout = QVBoxLayout()
        self.paths_groupbox.setLayout(self.paths_layout)

        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)

        minwidth = 300
        maxwidth = 300

        #--------------------------------------------------
        self.layout_book_viewer = QHBoxLayout()
        self.layout_book_viewer.setAlignment(Qt.AlignLeft)
        self.paths_layout.addLayout(self.layout_book_viewer)

        self.book_viewer_qlabel = QLabel()
        self.book_viewer_qlabel.setTextFormat(Qt.RichText)
        self.book_viewer_qlabel.setText("Book Viewer Path: ")
        self.book_viewer_qlabel.setFont(self.font)
        self.book_viewer_qlabel.setMinimumWidth(minwidth - 90)
        self.book_viewer_qlabel.setMaximumWidth(maxwidth - 90)
        self.layout_book_viewer.addWidget(self.book_viewer_qlabel)

        self.path_book_viewer_qlineedit = QLineEdit()
        self.path_book_viewer_qlineedit.setText(self.local_prefs['CALIBRESPY_PROGRAM_PATH_VIEWER'])
        self.path_book_viewer_qlineedit.setFont(self.font)
        self.path_book_viewer_qlineedit.setToolTip("<p style='white-space:wrap'>This is the path to the application that you desire to be launched to 'View' the currently selected book in CalibreSpy.")
        self.path_book_viewer_qlineedit.setCursorPosition(0)
        self.path_book_viewer_qlineedit.setMinimumWidth(minwidth)
        self.path_book_viewer_qlineedit.setMaximumWidth(maxwidth + 100)
        self.layout_book_viewer.addWidget(self.path_book_viewer_qlineedit)
        #--------------------------------------------------
        self.layout_book_editor = QHBoxLayout()
        self.layout_book_editor.setAlignment(Qt.AlignLeft)
        self.paths_layout.addLayout(self.layout_book_editor)

        self.book_editor_qlabel = QLabel()
        self.book_editor_qlabel.setTextFormat(Qt.RichText)
        self.book_editor_qlabel.setText("Book Editor Path: ")
        self.book_editor_qlabel.setFont(self.font)
        self.book_editor_qlabel.setMinimumWidth(minwidth - 90)
        self.book_editor_qlabel.setMaximumWidth(maxwidth - 90)
        self.layout_book_editor.addWidget(self.book_editor_qlabel)

        self.path_book_editor_qlineedit = QLineEdit()
        self.path_book_editor_qlineedit.setText(self.local_prefs['CALIBRESPY_PROGRAM_PATH_EDITOR'])
        self.path_book_editor_qlineedit.setFont(self.font)
        self.path_book_editor_qlineedit.setToolTip("<p style='white-space:wrap'>This is the path to the application that you desire to be launched to 'Edit' the currently selected book in CalibreSpy.")
        self.path_book_editor_qlineedit.setCursorPosition(0)
        self.path_book_editor_qlineedit.setMinimumWidth(minwidth)
        self.path_book_editor_qlineedit.setMaximumWidth(maxwidth + 100)
        self.layout_book_editor.addWidget(self.path_book_editor_qlineedit)
        #--------------------------------------------------
        self.layout_book_other = QHBoxLayout()
        self.layout_book_other.setAlignment(Qt.AlignLeft)
        self.paths_layout.addLayout(self.layout_book_other)

        self.book_other_qlabel = QLabel()
        self.book_other_qlabel.setTextFormat(Qt.RichText)
        self.book_other_qlabel.setText("Book Other Program Path: ")
        self.book_other_qlabel.setFont(self.font)
        self.book_other_qlabel.setMinimumWidth(minwidth - 90)
        self.book_other_qlabel.setMaximumWidth(maxwidth - 90)
        self.layout_book_other.addWidget(self.book_other_qlabel)

        self.path_book_other_qlineedit = QLineEdit()
        self.path_book_other_qlineedit.setText(self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER'])
        self.path_book_other_qlineedit.setFont(self.font)
        self.path_book_other_qlineedit.setToolTip("<p style='white-space:wrap'>This is the path to any application that you desire to be launched to interact with the currently selected book in CalibreSpy.")
        self.path_book_other_qlineedit.setCursorPosition(0)
        self.path_book_other_qlineedit.setMinimumWidth(minwidth)
        self.path_book_other_qlineedit.setMaximumWidth(maxwidth + 100)
        self.layout_book_other.addWidget(self.path_book_other_qlineedit)
        #--------------------------------------------------
        self.layout_book_other_icon = QHBoxLayout()
        self.layout_book_other_icon.setAlignment(Qt.AlignLeft)
        self.paths_layout.addLayout(self.layout_book_other_icon)

        self.book_other_icon_qlabel = QLabel()
        self.book_other_icon_qlabel.setTextFormat(Qt.RichText)
        self.book_other_icon_qlabel.setText("Book Other Program Icon Path: ")
        self.book_other_icon_qlabel.setFont(self.font)
        self.book_other_icon_qlabel.setMinimumWidth(minwidth - 90)
        self.book_other_icon_qlabel.setMaximumWidth(maxwidth - 90)
        self.layout_book_other_icon.addWidget(self.book_other_icon_qlabel)

        self.path_book_other_icon_qlineedit = QLineEdit()
        self.path_book_other_icon_qlineedit.setText(self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER_ICON'])
        self.path_book_other_icon_qlineedit.setFont(self.font)
        self.path_book_other_icon_qlineedit.setToolTip("<p style='white-space:wrap'>This is the path to the icon file used for the Other Program.")
        self.path_book_other_icon_qlineedit.setCursorPosition(0)
        self.path_book_other_icon_qlineedit.setMinimumWidth(minwidth)
        self.path_book_other_icon_qlineedit.setMaximumWidth(maxwidth + 100)
        self.layout_book_other_icon.addWidget(self.path_book_other_icon_qlineedit)
        #--------------------------------------------------
        self.layout_smtp_command_file_path = QHBoxLayout()
        self.layout_smtp_command_file_path.setAlignment(Qt.AlignLeft)
        self.paths_layout.addLayout(self.layout_smtp_command_file_path)

        self.smtp_command_file_path_qlabel = QLabel()
        self.smtp_command_file_path_qlabel.setTextFormat(Qt.RichText)
        self.smtp_command_file_path_qlabel.setText("'Calibre-SMTP' Command File Path: ")
        self.smtp_command_file_path_qlabel.setFont(self.font)
        self.smtp_command_file_path_qlabel.setMinimumWidth(minwidth - 90)
        self.smtp_command_file_path_qlabel.setMaximumWidth(maxwidth - 90)
        self.layout_smtp_command_file_path.addWidget(self.smtp_command_file_path_qlabel)

        self.path_smtp_command_file_path_qlineedit = QLineEdit()
        self.path_smtp_command_file_path_qlineedit.setText(self.local_prefs['CALIBRESPY_COMMAND_FILE_PATH_EMAIL'])
        self.path_smtp_command_file_path_qlineedit.setFont(self.font)
        self.path_smtp_command_file_path_qlineedit.setToolTip("<p style='white-space:wrap'>This is the path to the command file (in Windows, .bat file) that executes CLI program calibre-smtp to email a book.")
        self.path_smtp_command_file_path_qlineedit.setCursorPosition(0)
        self.path_smtp_command_file_path_qlineedit.setMinimumWidth(minwidth)
        self.path_smtp_command_file_path_qlineedit.setMaximumWidth(maxwidth + 100)
        self.layout_smtp_command_file_path.addWidget(self.path_smtp_command_file_path_qlineedit)
        #--------------------------------------------------
        self.groupbox_matrix_colors = QGroupBox('CalibreSpy Row Colors && Height:')
        self.groupbox_matrix_colors.setToolTip("<p style='white-space:wrap'>This changes the colors of CalibreSpy, and the height of each row.")
        self.layout_top.addWidget(self.groupbox_matrix_colors)

        self.layout_matrix_groupbox = QVBoxLayout()
        self.groupbox_matrix_colors.setLayout(self.layout_matrix_groupbox)

        self.layout_matrix_colors = QHBoxLayout()
        self.layout_matrix_colors.setAlignment(Qt.AlignLeft)

        self.layout_matrix_groupbox.addLayout(self.layout_matrix_colors)

        self.text_color_label = QLabel("Text:")
        self.text_color_label.setToolTip("<p style='white-space:wrap'>These are 100% of the text colors available to choose from.  'Black' is the default neutral color.")
        self.layout_matrix_colors.addWidget(self.text_color_label)

        self.text_color_combobox = QComboBox()
        self.text_color_combobox.setEditable(False)
        self.text_color_combobox.setFont(self.font)
        self.text_color_combobox.setToolTip("<p style='white-space:wrap'>These are 100% of the text colors available to choose from.  'Black' is the default neutral color.")
        self.layout_matrix_colors.addWidget(self.text_color_combobox)

        self.background_color_label = QLabel("BG 1:")
        self.background_color_label.setToolTip("<p style='white-space:wrap'>These are 100% of the background colors available to choose from.  'Snow' is the default neutral color.")
        self.layout_matrix_colors.addWidget(self.background_color_label)

        self.background_color_combobox = QComboBox()
        self.background_color_combobox.setEditable(False)
        self.background_color_combobox.setFont(self.font)
        self.background_color_combobox.setToolTip("<p style='white-space:wrap'>These are 100% of the background colors available to choose from.  'Snow' is the default neutral color.")
        self.layout_matrix_colors.addWidget(self.background_color_combobox)

        self.alternating_color_label = QLabel("BG 2:")
        self.alternating_color_label.setToolTip("<p style='white-space:wrap'>These are 100% of the alternate background colors available to choose from.  'Gainsboro' is the default neutral color.")
        self.layout_matrix_colors.addWidget(self.alternating_color_label)

        self.alternating_color_combobox = QComboBox()
        self.alternating_color_combobox.setEditable(False)
        self.alternating_color_combobox.setFont(self.font)
        self.alternating_color_combobox.setToolTip("<p style='white-space:wrap'>These are 100% of the alternate background colors available to choose from.  'Gainsboro' is the default neutral color.")
        self.layout_matrix_colors.addWidget(self.alternating_color_combobox)

        color_list = QColor.colorNames()
        for color in color_list:
            self.text_color_combobox.addItem(color)
            self.background_color_combobox.addItem(color)
            self.alternating_color_combobox.addItem(color)
        #END FOR

        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        bc = self.local_prefs['CALIBRESPY_BACKGROUND_COLOR']
        ac = self.local_prefs['CALIBRESPY_ALTERNATING_COLOR']

        self.text_color_combobox.setCurrentText(tc)
        self.background_color_combobox.setCurrentText(bc)
        self.alternating_color_combobox.setCurrentText(ac)

        self.text_color_combobox.setEditable(False)
        self.background_color_combobox.setEditable(False)
        self.alternating_color_combobox.setEditable(False)

        style_text = "background-color: [BC]; color : [TC]"
        style_text = style_text.replace("[TC]",tc)
        style_text = style_text.replace("[BC]",bc)
        self.text_color_combobox.setStyleSheet(style_text)

        style_text = "background-color: [BC]; color : [TC]"
        style_text = style_text.replace("[TC]",tc)
        style_text = style_text.replace("[BC]",bc)
        self.background_color_combobox.setStyleSheet(style_text)

        style_text = "background-color: [BC]; color : [TC]"
        style_text = style_text.replace("[TC]",tc)
        style_text = style_text.replace("[BC]",ac)
        self.alternating_color_combobox.setStyleSheet(style_text)
        #--------------------------------------------------
        #--------------------------------------------------
        self.row_height_groupbox = QGroupBox("CalibreSpy Row Height Adjustment:")
        self.layout_top.addWidget(self.row_height_groupbox)

        self.row_height_layout = QHBoxLayout()
        self.row_height_groupbox.setLayout(self.row_height_layout)

        n = self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE']

        self.row_height_adjustment_qspinbox = QSpinBox()
        self.row_height_adjustment_qspinbox.setSuffix("          :Extra Row Height Spacing in Pixels (may be negative)")
        self.row_height_adjustment_qspinbox.setMaximumWidth(400)
        self.row_height_adjustment_qspinbox.setRange(-20,30)
        self.row_height_adjustment_qspinbox.setValue(int(n))  #will ignore setting value unless min/max already set...
        self.row_height_adjustment_qspinbox.setToolTip("<p style='white-space:wrap'>The matrix row height may be increased or decreased by this number of pixels.")
        self.row_height_layout.addWidget(self.row_height_adjustment_qspinbox)

        #--------------------------------------------------
        #--------------------------------------------------
        self.groupbox_metadata = QGroupBox('CalibreSpy Optional Metadata:')
        self.groupbox_metadata.setToolTip("<p style='white-space:wrap'>Select optional Calibre Standard Columns to load.  Note that Tags and Publisher might significantly increase CalibreSpy initialization time.")
        self.layout_top.addWidget(self.groupbox_metadata)

        self.layout_metadata_groupbox = QVBoxLayout()
        self.groupbox_metadata.setLayout(self.layout_metadata_groupbox)

        self.layout_metadata_line_1 = QHBoxLayout()
        self.layout_metadata_line_1.setAlignment(Qt.AlignCenter)
        self.layout_metadata_groupbox.addLayout(self.layout_metadata_line_1)

        self.load_series_qcheckbox = QCheckBox("Series")
        self.load_series_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_SERIES'])
        self.load_series_qcheckbox.setFont(self.font)
        self.load_series_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Series?  This could noticeably increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_series_qcheckbox)

        self.load_publisher_qcheckbox = QCheckBox("Publisher")
        self.load_publisher_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_PUBLISHER'])
        self.load_publisher_qcheckbox.setFont(self.font)
        self.load_publisher_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Publisher?  This could noticeably increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_publisher_qcheckbox)

        self.load_tags_qcheckbox = QCheckBox("Tags")
        self.load_tags_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_TAGS'])
        self.load_tags_qcheckbox.setFont(self.font)
        self.load_tags_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Tags?  This could significantly increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_tags_qcheckbox)

        self.load_languages_qcheckbox = QCheckBox("Languages")
        self.load_languages_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_LANGUAGES'])
        self.load_languages_qcheckbox.setFont(self.font)
        self.load_languages_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Languages?  This will slightly increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_languages_qcheckbox)

        self.load_timestamp_qcheckbox = QCheckBox("Date Added")
        self.load_timestamp_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_TIMESTAMP'])
        self.load_timestamp_qcheckbox.setFont(self.font)
        self.load_timestamp_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Date Added?  This will slightly increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_timestamp_qcheckbox)

        self.load_modified_qcheckbox = QCheckBox("Modified Date")
        self.load_modified_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_MODIFIED'])
        self.load_modified_qcheckbox.setFont(self.font)
        self.load_modified_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Modified Date?  This will slightly increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_modified_qcheckbox)

        self.load_identifiers_qcheckbox = QCheckBox("Identifiers")  #Version 1.0.43: Identifiers
        self.load_identifiers_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_IDENTIFIERS'])
        self.load_identifiers_qcheckbox.setFont(self.font)
        self.load_identifiers_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Identifiers?  This will slightly increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_identifiers_qcheckbox)

        self.load_ratings_qcheckbox = QCheckBox("Ratings")       #Version 1.0.52: Ratings
        self.load_ratings_qcheckbox.setChecked(self.local_prefs['CALIBRESPY_LOAD_RATINGS'])
        self.load_ratings_qcheckbox.setFont(self.font)
        self.load_ratings_qcheckbox.setToolTip("<p style='white-space:wrap'>Do you wish CalibreSpy to load Ratings?  This will slightly increase startup time.")
        self.layout_metadata_line_1.addWidget(self.load_ratings_qcheckbox)
        #~ --------------------------
        self.layout_metadata_line_2 = QHBoxLayout()
        self.layout_metadata_groupbox.addLayout(self.layout_metadata_line_2)

        self.custom_column_1_combobox = QComboBox()
        self.custom_column_1_combobox.setEditable(False)
        self.custom_column_1_combobox.setFont(self.font)
        self.custom_column_1_combobox.setMaximumWidth(200)
        self.custom_column_1_combobox.setToolTip("<p style='white-space:wrap'>CC#1:  Select a 1st custom column to load.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_2.addWidget(self.custom_column_1_combobox)

        self.custom_column_1_combobox.addItem(CUSTOM_COLUMN_1_COMBOBOX_HEADER)

        self.custom_column_2_combobox = QComboBox()
        self.custom_column_2_combobox.setEditable(False)
        self.custom_column_2_combobox.setFont(self.font)
        self.custom_column_2_combobox.setMaximumWidth(200)
        self.custom_column_2_combobox.setToolTip("<p style='white-space:wrap'>CC#2:  Select a 2nd custom column to load.<br><br>The 1st to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_2.addWidget(self.custom_column_2_combobox)

        self.custom_column_2_combobox.addItem(CUSTOM_COLUMN_2_COMBOBOX_HEADER)

        self.custom_column_3_combobox = QComboBox()
        self.custom_column_3_combobox.setEditable(False)
        self.custom_column_3_combobox.setFont(self.font)
        self.custom_column_3_combobox.setMaximumWidth(200)
        self.custom_column_3_combobox.setToolTip("<p style='white-space:wrap'>CC#3:  Select a 3d custom column to load.<br><br>The 1st and 2nd to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_2.addWidget(self.custom_column_3_combobox)

        self.custom_column_3_combobox.addItem(CUSTOM_COLUMN_3_COMBOBOX_HEADER)
        #~ --------------------------
        self.layout_metadata_line_3 = QHBoxLayout()
        self.layout_metadata_groupbox.addLayout(self.layout_metadata_line_3)

        self.custom_column_4_combobox = QComboBox()
        self.custom_column_4_combobox.setEditable(False)
        self.custom_column_4_combobox.setFont(self.font)
        self.custom_column_4_combobox.setMaximumWidth(200)
        self.custom_column_4_combobox.setToolTip("<p style='white-space:wrap'>CC#4:  Select a 4th custom column to load.<br><br>The 3d to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_3.addWidget(self.custom_column_4_combobox)

        self.custom_column_4_combobox.addItem(CUSTOM_COLUMN_4_COMBOBOX_HEADER)

        self.custom_column_5_combobox = QComboBox()
        self.custom_column_5_combobox.setEditable(False)
        self.custom_column_5_combobox.setFont(self.font)
        self.custom_column_5_combobox.setMaximumWidth(200)
        self.custom_column_5_combobox.setToolTip("<p style='white-space:wrap'>CC#5:  Select a 5th custom column to load.<br><br>The 4th to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_3.addWidget(self.custom_column_5_combobox)

        self.custom_column_5_combobox.addItem(CUSTOM_COLUMN_5_COMBOBOX_HEADER)

        self.custom_column_6_combobox = QComboBox()
        self.custom_column_6_combobox.setEditable(False)
        self.custom_column_6_combobox.setFont(self.font)
        self.custom_column_6_combobox.setMaximumWidth(200)
        self.custom_column_6_combobox.setToolTip("<p style='white-space:wrap'>CC#6:  Select a 6th custom column to load.<br><br>The 5th to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_3.addWidget(self.custom_column_6_combobox)

        self.custom_column_6_combobox.addItem(CUSTOM_COLUMN_6_COMBOBOX_HEADER)
        #~ --------------------------
        self.layout_metadata_line_4 = QHBoxLayout()
        self.layout_metadata_groupbox.addLayout(self.layout_metadata_line_4)

        self.custom_column_7_combobox = QComboBox()
        self.custom_column_7_combobox.setEditable(False)
        self.custom_column_7_combobox.setFont(self.font)
        self.custom_column_7_combobox.setMaximumWidth(200)
        self.custom_column_7_combobox.setToolTip("<p style='white-space:wrap'>CC#7:  Select a 7th custom column to load.<br><br>The 6th to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_4.addWidget(self.custom_column_7_combobox)

        self.custom_column_7_combobox.addItem(CUSTOM_COLUMN_7_COMBOBOX_HEADER)

        self.custom_column_8_combobox = QComboBox()
        self.custom_column_8_combobox.setEditable(False)
        self.custom_column_8_combobox.setFont(self.font)
        self.custom_column_8_combobox.setMaximumWidth(200)
        self.custom_column_8_combobox.setToolTip("<p style='white-space:wrap'>CC#8:  Select a 8th custom column to load.<br><br>The 7th to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_4.addWidget(self.custom_column_8_combobox)

        self.custom_column_8_combobox.addItem(CUSTOM_COLUMN_8_COMBOBOX_HEADER)

        self.custom_column_9_combobox = QComboBox()
        self.custom_column_9_combobox.setEditable(False)
        self.custom_column_9_combobox.setFont(self.font)
        self.custom_column_9_combobox.setMaximumWidth(200)
        self.custom_column_9_combobox.setToolTip("<p style='white-space:wrap'>CC#9:  Select a 9th custom column to load.<br><br>The 8th to load must already have been specified.<br><br>Caution: startup time may significantly increase.")
        self.layout_metadata_line_4.addWidget(self.custom_column_9_combobox)

        self.custom_column_9_combobox.addItem(CUSTOM_COLUMN_9_COMBOBOX_HEADER)
        #~ --------------------------
        self.get_custom_column_table()
        for row in self.custom_column_list:
            label,name,datatype = row
            item = "#" + label + ">" + name + ">" + datatype
            self.custom_column_1_combobox.addItem(item)
            self.custom_column_2_combobox.addItem(item)
            self.custom_column_3_combobox.addItem(item)
            self.custom_column_4_combobox.addItem(item)
            self.custom_column_5_combobox.addItem(item)
            self.custom_column_6_combobox.addItem(item)
            self.custom_column_7_combobox.addItem(item)
            self.custom_column_8_combobox.addItem(item)
            self.custom_column_9_combobox.addItem(item)
        #END FOR
        #~ --------------------------
        i = self.custom_column_1_combobox.findText(self.local_prefs['CUSTOM_COLUMN_1_SELECTED'])
        if i > -1:
            self.custom_column_1_combobox.setCurrentIndex(i)

        i = self.custom_column_2_combobox.findText(self.local_prefs['CUSTOM_COLUMN_2_SELECTED'])
        if i > -1:
            self.custom_column_2_combobox.setCurrentIndex(i)

        i = self.custom_column_3_combobox.findText(self.local_prefs['CUSTOM_COLUMN_3_SELECTED'])
        if i > -1:
            self.custom_column_3_combobox.setCurrentIndex(i)

        i = self.custom_column_4_combobox.findText(self.local_prefs['CUSTOM_COLUMN_4_SELECTED'])
        if i > -1:
            self.custom_column_4_combobox.setCurrentIndex(i)

        i = self.custom_column_5_combobox.findText(self.local_prefs['CUSTOM_COLUMN_5_SELECTED'])
        if i > -1:
            self.custom_column_5_combobox.setCurrentIndex(i)

        i = self.custom_column_6_combobox.findText(self.local_prefs['CUSTOM_COLUMN_6_SELECTED'])
        if i > -1:
            self.custom_column_6_combobox.setCurrentIndex(i)

        i = self.custom_column_7_combobox.findText(self.local_prefs['CUSTOM_COLUMN_7_SELECTED'])
        if i > -1:
            self.custom_column_7_combobox.setCurrentIndex(i)

        i = self.custom_column_8_combobox.findText(self.local_prefs['CUSTOM_COLUMN_8_SELECTED'])
        if i > -1:
            self.custom_column_8_combobox.setCurrentIndex(i)

        i = self.custom_column_9_combobox.findText(self.local_prefs['CUSTOM_COLUMN_9_SELECTED'])
        if i > -1:
            self.custom_column_9_combobox.setCurrentIndex(i)
        #~ --------------------------
        self.layout_metadata_line_5 = QHBoxLayout()
        self.layout_metadata_groupbox.addLayout(self.layout_metadata_line_5)

        self.push_button_reset_cc_comboboxes = QPushButton("Reset Custom Column Selections", self)
        self.push_button_reset_cc_comboboxes.setFont(self.font)
        self.push_button_reset_cc_comboboxes.setToolTip("<p style='white-space:wrap'>Resets the selections to 'Nothing Selected'.\
                                                                                           <br><br>Warning:  you must verify that your 'Metadata Matrix Column Order' is still correct after deactivating any Custom Columns.  Otherwise, bad things could happen.")
        self.push_button_reset_cc_comboboxes.clicked.connect(self.reset_cc_comboboxes)
        self.layout_metadata_line_5.addWidget(self.push_button_reset_cc_comboboxes)
        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        self.display_size_groupbox = QGroupBox("CalibreSpy Font, Preferred Format && Device Display Size:")
        self.layout_top.addWidget(self.display_size_groupbox)

        self.font_format_device_layout = QVBoxLayout()
        self.display_size_groupbox.setLayout(self.font_format_device_layout)

        self.layout_font_family = QHBoxLayout()
        self.layout_font_family.setAlignment(Qt.AlignLeft)
        self.font_format_device_layout.addLayout(self.layout_font_family)

        self.font_family_qlineedit = QLineEdit()
        self.font_family_qlineedit.setText(self.local_prefs['DEFAULT_FONT_FAMILY'])
        self.font_family_qlineedit.setMinimumWidth(250)
        self.font_family_qlineedit.setMaximumWidth(250)
        self.font_family_qlineedit.setFont(self.font)
        self.font_family_qlineedit.setToolTip("<p style='white-space:wrap'>This is font family to be used for this specific Library.<br><br>Be sure to select a font family appropriate for your operating system, and one which you know to be installed on your PC.  Otherwise, it will be ignored.")
        self.font_family_qlineedit.setCursorPosition(0)
        self.layout_font_family.addWidget(self.font_family_qlineedit)

        self.font_completer_list = ['Arial','Helvetica','Times New Roman','Times','Courier New','Courier','Verdana','Georgia','Palatino','Garamond','Bookman','Comic Sans MS','Trebuchet MS','Arial Black','Impact','Tahoma','Sans Serif','Monaco Mac','Lucida Console','Franklin Gothic Medium','Lucida Sans Unicode','Lucida Grande','Geneva','Sylfaen','URW Chancery L','URW Gothic L','Century Schoolbook L','Nimbus Mono L','URW Bookman L','Helvetica Neue','URW Palladio L','DejaVu Sans Mono','Gill Sans','Nimbus Sans L','Bitstream Charter','Segoe UI',]
        self.font_completer_list.sort()

        self.font_family_qcompleter = QCompleter(self.font_completer_list,self.font_family_qlineedit)
        self.font_family_qcompleter.setCompletionMode(QCompleter.CompletionMode.UnfilteredPopupCompletion)
        self.font_family_qcompleter.setCaseSensitivity(Qt.CaseInsensitive)

        self.font_family_qcompleter_popup = self.font_family_qcompleter.popup()
        self.font_family_qcompleter_popup.setStyleSheet("QAbstractItemView { color: black; background: white; selection-background-color: darkblue; }\n QScrollBar { width: 25px;}")

        self.font_family_qlineedit.setCompleter(self.font_family_qcompleter)

        self.font_size_qlineedit = QLineEdit()
        s = self.local_prefs['DEFAULT_FONT_SIZE']
        self.font_size_qlineedit.setText(unicode_type(s))
        self.font_size_qlineedit.setMinimumWidth(75)
        self.font_size_qlineedit.setMaximumWidth(75)
        self.font_size_qlineedit.setFont(self.font)
        self.font_size_qlineedit.setToolTip("<p style='white-space:wrap'>This is 'normal' font size to be used for this specific Library.")
        self.font_size_qlineedit.setCursorPosition(0)
        self.layout_font_family.addWidget(self.font_size_qlineedit)

        self.font_size_completer_list = []
        for i in range(5,20):
            i = unicode_type(i)
            self.font_size_completer_list.append(i)
        #END FOR

        self.font_size_qcompleter = QCompleter(self.font_size_completer_list,self.font_size_qlineedit)
        self.font_size_qcompleter.setCompletionMode(QCompleter.CompletionMode.UnfilteredPopupCompletion)
        self.font_size_qcompleter.setCaseSensitivity(Qt.CaseInsensitive)

        self.font_size_qcompleter_popup = self.font_size_qcompleter.popup()
        self.font_size_qcompleter_popup.setStyleSheet("QAbstractItemView { color: black; background: white; selection-background-color: darkblue; }\n QScrollBar { width: 25px;}")

        self.font_size_qlineedit.setCompleter(self.font_size_qcompleter)

        t = "<p style='white-space:wrap'>This format will be used <b>instead of</b> Calibre's 'Default Output Format' in 'Preferences > Behavior"

        self.local_default_output_format_combobox = QComboBox()
        self.local_default_output_format_combobox.setEditable(False)
        self.local_default_output_format_combobox.setFont(self.font)
        self.local_default_output_format_combobox.setMinimumWidth(100)
        self.local_default_output_format_combobox.setMaximumWidth(100)
        self.local_default_output_format_combobox.setToolTip(t)
        self.font_format_device_layout.addWidget(self.local_default_output_format_combobox)

        from calibre.customize.ui import available_output_formats

        output_formats_set = available_output_formats()  #all lower case
        output_formats_set.remove('oeb')
        output_formats_list = list(output_formats_set)
        output_formats_list.sort()
        for choice in output_formats_list:
            self.local_default_output_format_combobox.addItem(choice.upper())
        #END FOR
        del available_output_formats

        self.local_default_output_format_combobox.setCurrentText(self.local_prefs['PREFERRED_OUTPUT_FORMAT'].upper())

        self.display_size_combobox = QComboBox()
        self.display_size_combobox.setMaximumWidth(400)
        self.display_size_combobox.setEditable(False)
        self.display_size_combobox.setFont(self.font)
        self.display_size_combobox.setToolTip("<p style='white-space:wrap'>Tiny Tablets that are running CalibreSpy require some accomodation for their smaller size. ")
        self.font_format_device_layout.addWidget(self.display_size_combobox)

        self.display_size_combobox.addItem(DISPLAY_SIZE_NORMAL)
        self.display_size_combobox.addItem(DISPLAY_SIZE_SMALL)
        self.display_size_combobox.addItem(DISPLAY_SIZE_TINY)

        size = self.local_prefs['CALIBRESPY_DEVICE_DISPLAY_SIZE']
        self.display_size_combobox.setCurrentText(size)
        #--------------------------------------------------
        #--------------------------------------------------
        self.spacer_1_label = QLabel("")
        self.layout_top.addWidget(self.spacer_1_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.matrix_groupbox = QGroupBox("CalibreSpy Metadata Matrix Column Order:")
        self.layout_top.addWidget(self.matrix_groupbox)

        self.matrix_layout = QVBoxLayout()
        self.matrix_layout.setAlignment(Qt.AlignCenter)
        self.matrix_groupbox.setLayout(self.matrix_layout)

        self.build_matrix()

        self.matrix_layout.addWidget(self.matrix)

        self.push_button_validate_matrix = QPushButton("Validate Matrix Column Order", self)
        self.push_button_validate_matrix.setFont(self.font)
        self.push_button_validate_matrix.setToolTip("<p style='white-space:wrap'>Validate the Displayed Metadata Matrix Column Order.\
                                                                                <br><br>Warning:  you must verify that your 'Metadata Matrix Column Order' is still correct after deactivating any Custom Columns.  Otherwise, bad things could happen.")
        self.push_button_validate_matrix.clicked.connect(self.validate_matrix_items)
        self.layout_top.addWidget(self.push_button_validate_matrix)

        self.push_button_reset_matrix = QPushButton("Reset Matrix Column Order to Original Values", self)
        self.push_button_reset_matrix.setFont(self.font)
        self.push_button_reset_matrix.setToolTip("<p style='white-space:wrap'>Reset the Displayed Metadata Matrix Column Order to the values previously being used upon start-up.\
                                                                            <br><br>Warning:  you must verify that your 'Metadata Matrix Column Order' is still correct after deactivating any Custom Columns.  Otherwise, bad things could happen.")
        self.push_button_reset_matrix.clicked.connect(self.reset_matrix_items)
        self.layout_top.addWidget(self.push_button_reset_matrix)
        #--------------------------------------------------
        #--------------------------------------------------
        self.spacer_2_label = QLabel("")
        self.layout_top.addWidget(self.spacer_2_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_qspinbox = QHBoxLayout()
        self.layout_qspinbox.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_qspinbox)

        self.orphaned_multiuser_lock_elapsed_time_qspinbox = QSpinBox()
        self.orphaned_multiuser_lock_elapsed_time_qspinbox.setFont(self.font)
        self.orphaned_multiuser_lock_elapsed_time_qspinbox.setSuffix("    :Age (in hours) after MultiUser Locks will be deemed 'orphaned' and removed at the next Exit.")
        self.orphaned_multiuser_lock_elapsed_time_qspinbox.setMaximumWidth(600)
        self.orphaned_multiuser_lock_elapsed_time_qspinbox.setRange(2,24)
        n = self.local_prefs['CALIBRESPY_ORPHANED_MULTIUSER_LOCK_ELAPSED_HOURS']
        self.orphaned_multiuser_lock_elapsed_time_qspinbox.setValue(int(n))  #will ignore setting value unless min/max already set...
        t = "<p style='white-space:wrap'>The icon in the lower left corner of Calibrespy will be 'locked' if another user of this Library is already using it. \
                                                              <br><br>Normally, such locks are automatically removed when users exit from CalibreSpy.  \
                                                              However, if for any reason CalibreSpy were not to terminate 'normally', the locks would not be removed.  \
                                                              They would be 'orphaned'.<br><br>This setting will automatically delete such 'orphaned' locks after the stated number of hours of age. "
        self.orphaned_multiuser_lock_elapsed_time_qspinbox.setToolTip(t)
        self.layout_qspinbox.addWidget(self.orphaned_multiuser_lock_elapsed_time_qspinbox)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.spacer_3_label = QLabel("")
        self.layout_top.addWidget(self.spacer_3_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_ftp = QHBoxLayout()
        self.layout_ftp.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_ftp)

        self.ftp_host_label = QLabel("FTP:")
        self.ftp_host_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>These FTP parameters control the FTP GUI Tool."
        self.ftp_host_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_host_label)

        ftp_width = 100

        self.ftp_host_lineedit = QLineEdit(self)
        self.ftp_host_lineedit.setMinimumWidth(ftp_width)
        self.ftp_host_lineedit.setMaximumWidth(ftp_width)
        t = "<p style='white-space:wrap'>FTP Host.  Not TLS Encrypted."
        self.ftp_host_lineedit.setToolTip(t)
        self.ftp_host_lineedit.setText(self.local_prefs['CALIBRESPY_FTP_HOST'])
        self.layout_ftp.addWidget(self.ftp_host_lineedit)

        self.ftp_port_label = QLabel("Port:")
        self.ftp_port_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Port.  Usually 21.  Specify an integer."
        self.ftp_port_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_port_label)

        self.ftp_port_lineedit = QLineEdit(self)
        self.ftp_port_lineedit.setToolTip(t)
        self.ftp_port_lineedit.setMinimumWidth(50)
        self.ftp_port_lineedit.setMaximumWidth(50)
        port = self.local_prefs['CALIBRESPY_FTP_HOST_PORT']
        port = unicode_type(port)
        self.ftp_port_lineedit.setText(port)
        self.layout_ftp.addWidget(self.ftp_port_lineedit)

        self.ftp_directory_label = QLabel("Dir:")
        self.ftp_directory_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Host Directory into which the selected book formats will be uploaded.<br>Include a leading and trailing '/'.  <br>Example:  /myfolder/ "
        self.ftp_directory_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_directory_label)

        self.ftp_directory_lineedit = QLineEdit(self)
        self.ftp_directory_lineedit.setMinimumWidth(ftp_width)
        self.ftp_directory_lineedit.setMaximumWidth(ftp_width)
        self.ftp_directory_lineedit.setToolTip(t)
        self.ftp_directory_lineedit.setText(self.local_prefs['CALIBRESPY_FTP_HOST_DIRECTORY'])
        self.layout_ftp.addWidget(self.ftp_directory_lineedit)

        self.ftp_userid_label = QLabel("UserID:")
        self.ftp_userid_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Host User ID.  This is only very lightly encoded within CalibreSpy, and will be completely visible in each Library's metadata.db table _calibrespy_settings."
        self.ftp_userid_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_userid_label)

        self.ftp_userid_lineedit = QLineEdit(self)
        self.ftp_userid_lineedit.setMinimumWidth(ftp_width)
        self.ftp_userid_lineedit.setMaximumWidth(ftp_width)
        self.ftp_userid_lineedit.setToolTip(t)
        self.ftp_userid_lineedit.setText(self.local_prefs['CALIBRESPY_FTP_USERID'])
        self.layout_ftp.addWidget(self.ftp_userid_lineedit)

        self.ftp_password_label = QLabel("PW:")
        self.ftp_password_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Host Password.  This password is only very lightly encoded within CalibreSpy, and will be completely visible in each Library's metadata.db table _calibrespy_settings."
        self.ftp_password_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_password_label)

        self.ftp_password_lineedit = QLineEdit(self)
        self.ftp_password_lineedit.setText(self.local_prefs['CALIBRESPY_FTP_PASSWORD'])
        self.ftp_password_lineedit.setMinimumWidth(ftp_width)
        self.ftp_password_lineedit.setMaximumWidth(ftp_width)
        self.ftp_password_lineedit.setToolTip(t)

        self.layout_ftp.addWidget(self.ftp_password_lineedit)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.spacer_4_label = QLabel("")
        self.layout_top.addWidget(self.spacer_4_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_lb_search_engine_url = QHBoxLayout()
        self.layout_lb_search_engine_url.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_lb_search_engine_url)

        self.lb_search_engine_url_label = QLabel("Library Browser Metadata Link Search URL:  ")
        self.lb_search_engine_url_label.setFont(self.font)
        self.lb_search_engine_url_label.setAlignment(Qt.AlignRight)
        self.lb_search_engine_url_label.setMaximumWidth(400)
        t = "<p style='white-space:wrap'>The Library Browser Metadata Details contain links to the default internet browser search engine URL defined here.\
        <br><br>Example:  https://duckduckgo.com/?q= for which the Library Browser will automatically complete the search query beginning immediately after the 'q='. \
        <br><br>Example:  https://www.google.com/search?q=   for which the Library Browser will automatically complete the search query beginning immediately after the 'q='. \
        <br><br>Example:  https://www.bing.com/search?q= for which the Library Browser will automatically complete the search query beginning immediately after the 'q='. \
        <br><br>Example:  https://www.ask.com/web?q= for which the Library Browser will automatically complete the search query beginning immediately after the 'q='. \
        <br><br>Example:  https://www.wolframalpha.com/input/?i= for which the Library Browser will automatically complete the search query beginning immediately after the 'i='. \
        <br><br>Example:  https://search.yahoo.com/search?p= for which the Library Browser will automatically complete the search query beginning immediately after the 'p='. \
        <br><br>Only search URLs ending with a '=' may be used.  Be careful to add no extra quotes or other characters that would result in an invalid URL being created by the Library Browser."
        self.lb_search_engine_url_label.setToolTip(t)
        self.layout_lb_search_engine_url.addWidget(self.lb_search_engine_url_label)

        self.lb_search_engine_url_lineedit = QLineEdit(self)
        self.lb_search_engine_url_lineedit.setText(self.local_prefs['LIBRARY_BROWSER_SEARCH_ENGINE_URL'])
        self.lb_search_engine_url_lineedit.setFont(self.font)
        self.lb_search_engine_url_lineedit.setMinimumWidth(300)
        self.lb_search_engine_url_lineedit.setMaximumWidth(300)
        self.lb_search_engine_url_lineedit.setToolTip(t)

        self.layout_lb_search_engine_url.addWidget(self.lb_search_engine_url_lineedit)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.spacer_5_label = QLabel("")
        self.layout_top.addWidget(self.spacer_5_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_bottom_boxes = QHBoxLayout()
        self.layout_bottom_boxes.setAlignment(Qt.AlignRight)  #helps tiny tablet screens...
        self.layout_top.addLayout(self.layout_bottom_boxes)

        self.font.setBold(True)

        self.bottom_buttonbox = QDialogButtonBox()
        self.layout_bottom_boxes.addWidget(self.bottom_buttonbox)

        self.push_button_save_prefs = QPushButton("Save", self)
        self.push_button_save_prefs.setFont(self.font)
        self.push_button_save_prefs.setToolTip("<p style='white-space:wrap'>Save the current preferences.")
        self.push_button_save_prefs.clicked.connect(self.save_prefs)
        self.bottom_buttonbox.addButton(self.push_button_save_prefs,QDialogButtonBox.AcceptRole)
        #-----------------------------------------------------
        self.push_button_save_and_exit = QPushButton(" ", self)
        self.push_button_save_and_exit.setText("Save && Exit")
        self.push_button_save_and_exit.setAutoDefault(False)
        self.push_button_save_and_exit.setDefault(False)
        self.push_button_save_and_exit.setFont(self.font)
        self.push_button_save_and_exit.setToolTip("<p style='white-space:wrap'>Save the current preferences and the window size and then exit, returning the latest preferences to CalibreSpy.")
        self.push_button_save_and_exit.clicked.connect(self.save_and_exit)
        self.bottom_buttonbox.addButton(self.push_button_save_and_exit,QDialogButtonBox.AcceptRole)

        self.push_button_exit_prefs = QPushButton("Exit", self)
        self.push_button_exit_prefs.setFont(self.font)
        self.push_button_exit_prefs.setToolTip("<p style='white-space:wrap'> Previous 'Save' actions cannot be reversed, but you may exit without saving again.")
        self.push_button_exit_prefs.clicked.connect(self.exit_no_save)
        self.bottom_buttonbox.addButton(self.push_button_exit_prefs,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(True)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize_dialog_to_preferences()
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.text_color_combobox.currentIndexChanged.connect(self.event_combobox_colors_changed)
        self.background_color_combobox.currentIndexChanged.connect(self.event_combobox_colors_changed)
        self.alternating_color_combobox.currentIndexChanged.connect(self.event_combobox_colors_changed)
        #--------------------------------------------------
        self.custom_column_1_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_2_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_3_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_4_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_5_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_6_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_7_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_8_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)
        self.custom_column_9_combobox.currentIndexChanged.connect(self.event_combobox_custom_columns_changed)

        self.custom_column_specification_sequence_is_valid = False
        self.cc_comboxes_signals_blocked = False
        #--------------------------------------------------
        self.path_book_editor_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.path_book_viewer_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.path_book_other_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.path_book_other_icon_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.path_smtp_command_file_path_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)

        self.font_family_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.font_size_qlineedit.textEdited.connect(self.event_qlineedits_text_changed)

        self.ftp_host_lineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.ftp_port_lineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.ftp_directory_lineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.ftp_userid_lineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.ftp_password_lineedit.textEdited.connect(self.event_qlineedits_text_changed)
        self.lb_search_engine_url_lineedit.textEdited.connect(self.event_qlineedits_text_changed)

        self.resized_signal.connect(self.event_dialog_resized)
#--------------------------------------------------
#--------------------------------------------------
    def reset_cc_comboboxes(self):
        self.cc_comboxes_signals_blocked = True
        self.custom_column_1_combobox.setCurrentIndex(0)
        self.custom_column_2_combobox.setCurrentIndex(0)
        self.custom_column_3_combobox.setCurrentIndex(0)
        self.custom_column_4_combobox.setCurrentIndex(0)
        self.custom_column_5_combobox.setCurrentIndex(0)
        self.custom_column_6_combobox.setCurrentIndex(0)
        self.custom_column_7_combobox.setCurrentIndex(0)
        self.custom_column_8_combobox.setCurrentIndex(0)
        self.custom_column_9_combobox.setCurrentIndex(0)
        self.cc_comboxes_signals_blocked = False
#--------------------------------------------------
#--------------------------------------------------
    def create_column_assignment_dict(self):
        #~ default values only
        self.original_column_assignment_dict = {}
        self.original_column_assignment_dict[AUTHORS_COLUMN] = AUTHORS_COLUMN
        self.original_column_assignment_dict[TITLE_COLUMN] = TITLE_COLUMN
        self.original_column_assignment_dict[SERIES_COLUMN] = SERIES_COLUMN
        self.original_column_assignment_dict[INDEX_COLUMN] = INDEX_COLUMN
        self.original_column_assignment_dict[TAGS_COLUMN] = TAGS_COLUMN
        self.original_column_assignment_dict[PUBLISHED_COLUMN] = PUBLISHED_COLUMN
        self.original_column_assignment_dict[PUBLISHER_COLUMN] = PUBLISHER_COLUMN
        self.original_column_assignment_dict[LANGUAGES_COLUMN] = LANGUAGES_COLUMN
        self.original_column_assignment_dict[ADDED_COLUMN] = ADDED_COLUMN
        self.original_column_assignment_dict[MODIFIED_COLUMN] = MODIFIED_COLUMN
        self.original_column_assignment_dict[IDENTIFIERS_COLUMN] = IDENTIFIERS_COLUMN  #Version 1.0.43
        self.original_column_assignment_dict[RATINGS_COLUMN] = RATINGS_COLUMN   #Version 1.0.52
        self.original_column_assignment_dict[PATH_COLUMN] = PATH_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_1_COLUMN] = CUSTOM_COLUMN_1_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_2_COLUMN] = CUSTOM_COLUMN_2_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_3_COLUMN] = CUSTOM_COLUMN_3_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_4_COLUMN] = CUSTOM_COLUMN_4_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_5_COLUMN] = CUSTOM_COLUMN_5_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_6_COLUMN] = CUSTOM_COLUMN_6_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_7_COLUMN] = CUSTOM_COLUMN_7_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_8_COLUMN] = CUSTOM_COLUMN_8_COLUMN
        self.original_column_assignment_dict[CUSTOM_COLUMN_9_COLUMN] = CUSTOM_COLUMN_9_COLUMN
#--------------------------------------------------
#--------------------------------------------------
    def build_matrix(self,reset=False):

        if reset:
            try:
                self.matrix.setDisabled(True)
                self.matrix.setHidden(True)
                self.matrix_layout.removeWidget(self.matrix)
                self.matrix.close()
                self.matrix.destroy(True,True)
                self.matrix = None
                del self.matrix
                self.skip_restart_message = False
            except Exception as e:
                if DEBUG: print("Exception [1] in self.matrix.destroy() and del self.matrix: ", as_unicode(e))
                return

        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)    #Small & Tiny Tablet Optimization

        self.n_matrix_columns = MATRIX_MAXIMUM_COLUMNS_COUNT
        self.n_matrix_rows = 1

        self.matrix = QTableWidget(self.n_matrix_rows,self.n_matrix_columns)

        column_label_list = []  #here, the column headers are fixed, regardless of their assignment value.  It is the assignment values that are dynamic, not the label sequence left-to-right.
        label = "Author(s)"
        column_label_list.append(label)
        label = "Title"
        column_label_list.append(label)
        label = "Series"
        column_label_list.append(label)
        label = "Index"
        column_label_list.append(label)
        label = "Tags"
        column_label_list.append(label)
        label = "Published"
        column_label_list.append(label)
        label = "Publisher"
        column_label_list.append(label)
        label = "Lang"
        column_label_list.append(label)
        label = "Added"
        column_label_list.append(label)
        label = "Modified"
        column_label_list.append(label)
        label = "Path"
        column_label_list.append(label)
        label = "CC#1"
        column_label_list.append(label)
        label = "CC#2"
        column_label_list.append(label)
        label = "CC#3"
        column_label_list.append(label)
        label = "CC#4"
        column_label_list.append(label)
        label = "CC#5"
        column_label_list.append(label)
        label = "CC#6"
        column_label_list.append(label)
        label = "CC#7"
        column_label_list.append(label)
        label = "CC#8"
        column_label_list.append(label)
        label = "CC#9"
        column_label_list.append(label)
        label = "Identifiers"  #Version 1.0.43: Identifiers
        column_label_list.append(label)
        label = "Ratings"     #Version 1.0.52: Ratings
        column_label_list.append(label)

        self.matrix.setFont(self.font)
        self.matrix.horizontalHeader().setFont(self.font)
        self.matrix.verticalHeader().setFont(self.font)

        self.matrix.setCornerButtonEnabled(False)
        self.matrix.setSortingEnabled(False)
        self.matrix.setHorizontalHeaderLabels(column_label_list)
        self.matrix.setVerticalHeaderLabels([""])
        self.matrix.horizontalHeader().setVisible(True)
        self.matrix.verticalHeader().setVisible(True)

        self.authors_column = self.original_column_assignment_dict[AUTHORS_COLUMN]
        self.title_column = self.original_column_assignment_dict[TITLE_COLUMN]
        self.series_column = self.original_column_assignment_dict[SERIES_COLUMN]
        self.index_column = self.original_column_assignment_dict[INDEX_COLUMN]
        self.tags_column = self.original_column_assignment_dict[TAGS_COLUMN]
        self.published_column = self.original_column_assignment_dict[PUBLISHED_COLUMN]
        self.publisher_column = self.original_column_assignment_dict[PUBLISHER_COLUMN]
        self.languages_column = self.original_column_assignment_dict[LANGUAGES_COLUMN]
        self.added_column = self.original_column_assignment_dict[ADDED_COLUMN]
        self.modified_column = self.original_column_assignment_dict[MODIFIED_COLUMN]
        self.path_column = self.original_column_assignment_dict[PATH_COLUMN]
        self.cc1_column = self.original_column_assignment_dict[CUSTOM_COLUMN_1_COLUMN]
        self.cc2_column = self.original_column_assignment_dict[CUSTOM_COLUMN_2_COLUMN]
        self.cc3_column = self.original_column_assignment_dict[CUSTOM_COLUMN_3_COLUMN]
        self.cc4_column = self.original_column_assignment_dict[CUSTOM_COLUMN_4_COLUMN]
        self.cc5_column = self.original_column_assignment_dict[CUSTOM_COLUMN_5_COLUMN]
        self.cc6_column = self.original_column_assignment_dict[CUSTOM_COLUMN_6_COLUMN]
        self.cc7_column = self.original_column_assignment_dict[CUSTOM_COLUMN_7_COLUMN]
        self.cc8_column = self.original_column_assignment_dict[CUSTOM_COLUMN_8_COLUMN]
        self.cc9_column = self.original_column_assignment_dict[CUSTOM_COLUMN_9_COLUMN]
        self.identifiers_column = self.original_column_assignment_dict[IDENTIFIERS_COLUMN]   #Version 1.0.43
        self.ratings_column = self.original_column_assignment_dict[RATINGS_COLUMN]             #Version 1.0.52

        authors_sort_ = QTableWidgetItem(unicode_type(self.authors_column + 1))
        title_ = QTableWidgetItem(unicode_type(self.title_column + 1))
        series_ = QTableWidgetItem(unicode_type(self.series_column + 1))
        series_index_ = QTableWidgetItem(unicode_type(self.index_column + 1))
        tags_ = QTableWidgetItem(unicode_type(self.tags_column + 1))
        pubdate_ = QTableWidgetItem(unicode_type(self.published_column + 1))
        publisher_ = QTableWidgetItem(unicode_type(self.publisher_column + 1))
        languages_ = QTableWidgetItem(unicode_type(self.languages_column + 1))
        added_ = QTableWidgetItem(unicode_type(self.added_column + 1))
        modified_ = QTableWidgetItem(unicode_type(self.modified_column + 1))
        path_ = QTableWidgetItem(unicode_type(self.path_column + 1))
        cc1_ = QTableWidgetItem(unicode_type(self.cc1_column + 1))
        cc2_ = QTableWidgetItem(unicode_type(self.cc2_column + 1))
        cc3_ = QTableWidgetItem(unicode_type(self.cc3_column + 1))
        cc4_ = QTableWidgetItem(unicode_type(self.cc4_column + 1))
        cc5_ = QTableWidgetItem(unicode_type(self.cc5_column + 1))
        cc6_ = QTableWidgetItem(unicode_type(self.cc6_column + 1))
        cc7_ = QTableWidgetItem(unicode_type(self.cc7_column + 1))
        cc8_ = QTableWidgetItem(unicode_type(self.cc8_column + 1))
        cc9_ = QTableWidgetItem(unicode_type(self.cc9_column + 1))
        identifiers_ = QTableWidgetItem(unicode_type(self.identifiers_column + 1))  #Version 1.0.43
        ratings_ = QTableWidgetItem(unicode_type(self.ratings_column + 1))  #Version 1.0.52
        authors_sort_.setFont(self.font)
        title_.setFont(self.font)
        series_.setFont(self.font)
        series_index_.setFont(self.font)
        tags_.setFont(self.font)
        pubdate_.setFont(self.font)
        publisher_.setFont(self.font)
        languages_.setFont(self.font)
        added_.setFont(self.font)
        modified_.setFont(self.font)
        path_.setFont(self.font)
        cc1_.setFont(self.font)
        cc2_.setFont(self.font)
        cc3_.setFont(self.font)
        cc4_.setFont(self.font)
        cc5_.setFont(self.font)
        cc6_.setFont(self.font)
        cc7_.setFont(self.font)
        cc8_.setFont(self.font)
        cc9_.setFont(self.font)
        identifiers_.setFont(self.font) #Version 1.0.43
        ratings_.setFont(self.font)   #Version 1.0.52

        authors_sort_.setTextAlignment(Qt.AlignCenter)
        title_.setTextAlignment(Qt.AlignCenter)
        series_.setTextAlignment(Qt.AlignCenter)
        series_index_.setTextAlignment(Qt.AlignCenter)
        tags_.setTextAlignment(Qt.AlignCenter)
        pubdate_.setTextAlignment(Qt.AlignCenter)
        publisher_.setTextAlignment(Qt.AlignCenter)
        languages_.setTextAlignment(Qt.AlignCenter)
        added_.setTextAlignment(Qt.AlignCenter)
        modified_.setTextAlignment(Qt.AlignCenter)
        path_.setTextAlignment(Qt.AlignCenter)
        cc1_.setTextAlignment(Qt.AlignCenter)
        cc2_.setTextAlignment(Qt.AlignCenter)
        cc3_.setTextAlignment(Qt.AlignCenter)
        cc4_.setTextAlignment(Qt.AlignCenter)
        cc5_.setTextAlignment(Qt.AlignCenter)
        cc6_.setTextAlignment(Qt.AlignCenter)
        cc7_.setTextAlignment(Qt.AlignCenter)
        cc8_.setTextAlignment(Qt.AlignCenter)
        cc9_.setTextAlignment(Qt.AlignCenter)
        identifiers_.setTextAlignment(Qt.AlignCenter) #Version 1.0.43
        ratings_.setTextAlignment(Qt.AlignCenter)  #Version 1.0.52
        r = 0
        #here, the column headers are fixed, regardless of their assignment value.  It is the assignment values that are dynamic, not the label sequence left-to-right.
        self.matrix.setItem(r,AUTHORS_COLUMN,authors_sort_)
        self.matrix.setItem(r,TITLE_COLUMN,title_)
        self.matrix.setItem(r,SERIES_COLUMN,series_)
        self.matrix.setItem(r,INDEX_COLUMN,series_index_)
        self.matrix.setItem(r,TAGS_COLUMN,tags_)
        self.matrix.setItem(r,PUBLISHED_COLUMN,pubdate_)
        self.matrix.setItem(r,PUBLISHER_COLUMN,publisher_)
        self.matrix.setItem(r,LANGUAGES_COLUMN,languages_)
        self.matrix.setItem(r,IDENTIFIERS_COLUMN,identifiers_) #Version 1.0.43
        self.matrix.setItem(r,RATINGS_COLUMN,ratings_)  #Version 1.0.52
        self.matrix.setItem(r,ADDED_COLUMN,added_)
        self.matrix.setItem(r,MODIFIED_COLUMN,modified_)
        self.matrix.setItem(r,PATH_COLUMN,path_)
        self.matrix.setItem(r,CUSTOM_COLUMN_1_COLUMN,cc1_)
        self.matrix.setItem(r,CUSTOM_COLUMN_2_COLUMN,cc2_)
        self.matrix.setItem(r,CUSTOM_COLUMN_3_COLUMN,cc3_)
        self.matrix.setItem(r,CUSTOM_COLUMN_4_COLUMN,cc4_)
        self.matrix.setItem(r,CUSTOM_COLUMN_5_COLUMN,cc5_)
        self.matrix.setItem(r,CUSTOM_COLUMN_6_COLUMN,cc6_)
        self.matrix.setItem(r,CUSTOM_COLUMN_7_COLUMN,cc7_)
        self.matrix.setItem(r,CUSTOM_COLUMN_8_COLUMN,cc8_)
        self.matrix.setItem(r,CUSTOM_COLUMN_9_COLUMN,cc9_)

        self.matrix.setMinimumSize(600, 100)
        self.matrix.setMaximumSize(600, 100)
        self.matrix.resizeColumnsToContents()
        self.matrix.resizeRowsToContents()
        self.matrix.setCurrentCell(0,0)

        self.matrix.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectItems)    # single item
        self.matrix.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)        # single item
        self.matrix.setShowGrid(True)

        self.matrix.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self.matrix.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)

        self.matrix.horizontalHeader().setFixedHeight(35)

        max = MATRIX_MAXIMUM_COLUMNS_COUNT
        max = unicode_type(max)

        t =  "<p style='white-space:wrap'>These values are used to determine column order in CalibreSpy.\
                <br><br>You may change them as you please as long as the 'rules' are followed:\
                <br><br>Use only Integers from 1-" + max + ", inclusive.  No letters or punctuation symbols.  \
                An integer may be used only one (1) time.   No duplicates allowed.  \
                Every integer between 1 and " + max + ", inclusive, must be used one (1) time\
                <br><br>Caution: if you leave a 'data gap' in the column assignments, the end of that 'gap' will not be shown in CalibreSpy, because its column assignment will be ignored.\
                <br><br>Example of a 'data gap':  You load data for 3 Custom Columns, and set Path to be on their 'right' in the column assignments.  Later, you decide to eliminate Custom Column #3.  \
                However, you leave the column assignment for Path where it was before.  However, the column that was for the data in Custom Column #3 is now unused, leaving a 'gap' between \
                the column for Custom Column #2 and Path.  You will never see Path until you 'close the gap' by changing the column assignment for Path to a lower number.  Remember: duplicates are not allowed.\
                <br><br>If a newly repositioned column does appear after restarting CalibreSpy, simply right-click the book matrix and select 'Unhide All Columns'.\
                <br>"

        self.matrix.setToolTip(t)
        self.matrix_groupbox.setToolTip(t)

        self.font.setBold(False)
        self.font.setPointSize(self.normal_fontsize)

        self.resize_dialog_to_preferences()

        self.skip_restart_message = False
#--------------------------------------------------
#--------------------------------------------------
    def reset_matrix_items(self):
        self.build_matrix(reset=True)
        self.matrix_layout.addWidget(self.matrix)
        self.validate_matrix_items()
        self.restart_required = False
#--------------------------------------------------
#--------------------------------------------------
    def event_qlineedits_text_changed(self):
        self.local_prefs['CALIBRESPY_PROGRAM_PATH_EDITOR'] = self.path_book_editor_qlineedit.text().replace("\\","/").strip()
        self.local_prefs['CALIBRESPY_PROGRAM_PATH_VIEWER'] = self.path_book_viewer_qlineedit.text().replace("\\","/").strip()
        self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER'] = self.path_book_other_qlineedit.text().replace("\\","/").strip()
        self.local_prefs['CALIBRESPY_PROGRAM_PATH_OTHER_ICON'] = self.path_book_other_icon_qlineedit.text().replace("\\","/").strip()
        self.local_prefs['CALIBRESPY_COMMAND_FILE_PATH_EMAIL'] = self.path_smtp_command_file_path_qlineedit.text().replace("\\","/").strip()

        self.local_prefs['DEFAULT_FONT_FAMILY']  = self.font_family_qlineedit.text()
        size = self.font_size_qlineedit.text()
        size = size.strip()
        if size.isdigit():
            self.local_prefs['DEFAULT_FONT_SIZE']  = int(size)
        else:
            self.font_size_qlineedit.setText(as_unicode(self.local_prefs['DEFAULT_FONT_SIZE']))
            self.font_size_qlineedit.update()

        self.local_prefs['CALIBRESPY_FTP_HOST'] = self.ftp_host_lineedit.text().strip()
        self.local_prefs['CALIBRESPY_FTP_HOST_DIRECTORY'] = self.ftp_directory_lineedit.text().strip()
        self.local_prefs['CALIBRESPY_FTP_HOST_PORT'] = self.ftp_port_lineedit.text().strip()
        self.local_prefs['CALIBRESPY_FTP_USERID'] = self.ftp_userid_lineedit.text().strip()   #decoded
        self.local_prefs['CALIBRESPY_FTP_PASSWORD'] = self.ftp_password_lineedit.text().strip()  #decoded

        self.local_prefs['LIBRARY_BROWSER_SEARCH_ENGINE_URL'] = self.lb_search_engine_url_lineedit.text().strip()
#--------------------------------------------------
#--------------------------------------------------
    def event_combobox_colors_changed(self):
        self.local_prefs['CALIBRESPY_TEXT_COLOR'] = self.text_color_combobox.currentText()
        self.local_prefs['CALIBRESPY_BACKGROUND_COLOR'] = self.background_color_combobox.currentText()
        self.local_prefs['CALIBRESPY_ALTERNATING_COLOR'] =  self.alternating_color_combobox.currentText()

        tc = self.local_prefs['CALIBRESPY_TEXT_COLOR']
        bc = self.local_prefs['CALIBRESPY_BACKGROUND_COLOR']
        ac = self.local_prefs['CALIBRESPY_ALTERNATING_COLOR']

        style_text = "background-color: [BC]; color : [TC]"
        style_text = style_text.replace("[TC]",tc)
        style_text = style_text.replace("[BC]",bc)
        self.text_color_combobox.setStyleSheet(style_text)

        style_text = "background-color: [BC]; color : [TC]"
        style_text = style_text.replace("[TC]",tc)
        style_text = style_text.replace("[BC]",bc)
        self.background_color_combobox.setStyleSheet(style_text)

        style_text = "background-color: [BC]; color : [TC]"
        style_text = style_text.replace("[TC]",tc)
        style_text = style_text.replace("[BC]",ac)
        self.alternating_color_combobox.setStyleSheet(style_text)
#--------------------------------------------------
#--------------------------------------------------
    def event_combobox_custom_columns_changed(self,event=None):
        if self.cc_comboxes_signals_blocked:
            return

        cc1 = self.custom_column_1_combobox.currentText()
        cc2 = self.custom_column_2_combobox.currentText()
        cc3 = self.custom_column_3_combobox.currentText()
        cc4 = self.custom_column_4_combobox.currentText()
        cc5 = self.custom_column_5_combobox.currentText()
        cc6 = self.custom_column_6_combobox.currentText()
        cc7 = self.custom_column_7_combobox.currentText()
        cc8 = self.custom_column_8_combobox.currentText()
        cc9 = self.custom_column_9_combobox.currentText()

        self.there_is_cc1 = False
        self.there_is_cc2 = False
        self.there_is_cc3 = False
        self.there_is_cc4 = False
        self.there_is_cc5 = False
        self.there_is_cc6 = False
        self.there_is_cc7 = False
        self.there_is_cc8 = False
        self.there_is_cc9 = False

        self.local_prefs['CUSTOM_COLUMN_1_SELECTED'] = cc1.strip()
        self.local_prefs['CUSTOM_COLUMN_2_SELECTED'] = cc2.strip()
        self.local_prefs['CUSTOM_COLUMN_3_SELECTED'] = cc3.strip()
        self.local_prefs['CUSTOM_COLUMN_4_SELECTED'] = cc4.strip()
        self.local_prefs['CUSTOM_COLUMN_5_SELECTED'] = cc5.strip()
        self.local_prefs['CUSTOM_COLUMN_6_SELECTED'] = cc6.strip()
        self.local_prefs['CUSTOM_COLUMN_7_SELECTED'] = cc7.strip()
        self.local_prefs['CUSTOM_COLUMN_8_SELECTED'] = cc8.strip()
        self.local_prefs['CUSTOM_COLUMN_9_SELECTED'] = cc9.strip()

        if cc1.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_1'] = 0
            self.there_is_cc1 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_1'] = 1
            self.there_is_cc1 = True

        if cc2.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_2'] = 0
            self.there_is_cc2 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_2'] = 1
            self.there_is_cc2 = True

        if cc3.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_3'] = 0
            self.there_is_cc3 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_3'] = 1
            self.there_is_cc3 = True

        if cc4.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_4'] = 0
            self.there_is_cc4 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_4'] = 1
            self.there_is_cc4 = True

        if cc5.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_5'] = 0
            self.there_is_cc5 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_5'] = 1
            self.there_is_cc5 = True

        if cc6.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_6'] = 0
            self.there_is_cc6 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_6'] = 1
            self.there_is_cc6 = True

        if cc7.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_7'] = 0
            self.there_is_cc7 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_7'] = 1
            self.there_is_cc7 = True

        if cc8.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_8'] = 0
            self.there_is_cc8 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_8'] = 1
            self.there_is_cc8 = True

        if cc9.startswith("Custom Column"):
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_9'] = 0
            self.there_is_cc9 = False
        else:
            self.local_prefs['CALIBRESPY_LOAD_CUSTOM_COLUMN_9'] = 1
            self.there_is_cc9 = True

        if not self.there_is_cc1:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_2_combobox.setCurrentIndex(0)
            self.custom_column_3_combobox.setCurrentIndex(0)
            self.custom_column_4_combobox.setCurrentIndex(0)
            self.custom_column_5_combobox.setCurrentIndex(0)
            self.custom_column_6_combobox.setCurrentIndex(0)
            self.custom_column_7_combobox.setCurrentIndex(0)
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc2:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_3_combobox.setCurrentIndex(0)
            self.custom_column_4_combobox.setCurrentIndex(0)
            self.custom_column_5_combobox.setCurrentIndex(0)
            self.custom_column_6_combobox.setCurrentIndex(0)
            self.custom_column_7_combobox.setCurrentIndex(0)
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc3:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_4_combobox.setCurrentIndex(0)
            self.custom_column_5_combobox.setCurrentIndex(0)
            self.custom_column_6_combobox.setCurrentIndex(0)
            self.custom_column_7_combobox.setCurrentIndex(0)
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc4:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_5_combobox.setCurrentIndex(0)
            self.custom_column_6_combobox.setCurrentIndex(0)
            self.custom_column_7_combobox.setCurrentIndex(0)
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc5:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_6_combobox.setCurrentIndex(0)
            self.custom_column_7_combobox.setCurrentIndex(0)
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc6:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_7_combobox.setCurrentIndex(0)
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc7:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_8_combobox.setCurrentIndex(0)
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        elif not self.there_is_cc8:
            self.cc_comboxes_signals_blocked = True
            self.custom_column_9_combobox.setCurrentIndex(0)
            self.cc_comboxes_signals_blocked = False
        else:
            self.cc_comboxes_signals_blocked = False
#--------------------------------------------------
#--------------------------------------------------
    def event_dialog_resized(self,event=None):
        self.local_prefs['CUSTOMIZE_SIZE_WIDTH_LAST'] = self.size().width()
        self.local_prefs['CUSTOMIZE_SIZE_HEIGHT_LAST'] = self.size().height()
#--------------------------------------------------
#--------------------------------------------------
    def resizeEvent(self, event):
        self.resized_signal.emit()
#--------------------------------------------------
#--------------------------------------------------
    def keyPressEvent(self, event):
        if event.key() == Qt.Key_Escape:
            self.exit_no_save()
        else:
            QDialog.keyPressEvent(self, event)
#--------------------------------------------------
#--------------------------------------------------
    def get_metadatadb_local_prefs(self):
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        self.my_db = my_db
        self.my_cursor = my_cursor

        try:
            mysql = "SELECT prefkey,prefvalue,prefblob FROM _calibrespy_settings"
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                key,value,geom = row
                if key == "CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE":   #bytearray is a blob
                    continue  # only calibrespy_dialog handles this pref
                elif key == 'CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE':
                    self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE'] = int(value)
                    continue
                else:
                    value = as_unicode(value)  #all text table values should be unicode...
                    if value == "0":       #all table values are unicode...
                        value = 0
                    elif value == "1":
                        value = 1
                    elif value.isdigit():  #e.g. column numbers...see below...
                            value = int(value)

                    self.local_prefs[key] = value
                    if key == 'AUTHORS_COLUMN':
                        self.original_column_assignment_dict[AUTHORS_COLUMN] = value
                    elif key == 'TITLE_COLUMN':
                        self.original_column_assignment_dict[TITLE_COLUMN] = value
                    elif key == 'SERIES_COLUMN':
                        self.original_column_assignment_dict[SERIES_COLUMN] = value
                    elif key == 'INDEX_COLUMN':
                        self.original_column_assignment_dict[INDEX_COLUMN] = value
                    elif key == 'TAGS_COLUMN':
                        self.original_column_assignment_dict[TAGS_COLUMN] = value
                    elif key == 'PUBLISHED_COLUMN':
                        self.original_column_assignment_dict[PUBLISHED_COLUMN] = value
                    elif key == 'PUBLISHER_COLUMN':
                        self.original_column_assignment_dict[PUBLISHER_COLUMN] = value
                    elif key == 'LANGUAGES_COLUMN':
                        self.original_column_assignment_dict[LANGUAGES_COLUMN] = value
                    elif key == 'ADDED_COLUMN':
                        self.original_column_assignment_dict[ADDED_COLUMN] = value
                    elif key == 'MODIFIED_COLUMN':
                        self.original_column_assignment_dict[MODIFIED_COLUMN] = value
                    elif key == 'IDENTIFIERS_COLUMN':   #Version 1.0.43
                        self.original_column_assignment_dict[IDENTIFIERS_COLUMN] = value
                    elif key == 'RATINGS_COLUMN':   #Version 1.0.52
                        self.original_column_assignment_dict[RATINGS_COLUMN] = value
                    elif key == 'PATH_COLUMN':
                        self.original_column_assignment_dict[PATH_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_1_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_1_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_2_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_2_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_3_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_3_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_4_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_4_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_5_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_5_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_6_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_6_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_7_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_7_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_8_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_8_COLUMN] = value
                    elif key == 'CUSTOM_COLUMN_9_COLUMN':
                        self.original_column_assignment_dict[CUSTOM_COLUMN_9_COLUMN] = value
            #END FOR
            self.ftp_decoded = 0
            self.decode_ftp_data()
        except Exception as e:
            if DEBUG: print("Exception in SELECT prefkey,prefvalue,prefblob FROM _calibrespy_settings: ", as_unicode(e))
            self.create_metadatadb_settings_table()
            self.ftp_decoded = 0
            self.decode_ftp_data()
            self.save_local_prefs()
#--------------------------------------------------
#--------------------------------------------------
    def save_local_prefs(self):
        self.QApplication.instance().processEvents()
        #--------------------------------------------------
        self.event_qlineedits_text_changed()
        self.local_prefs['CALIBRESPY_DEVICE_DISPLAY_SIZE'] = self.display_size_combobox.currentText()
        self.local_prefs['CALIBRESPY_ROW_HEIGHT_CHANGE_VALUE'] = self.row_height_adjustment_qspinbox.value()
        self.local_prefs['PREFERRED_OUTPUT_FORMAT'] = self.local_default_output_format_combobox.currentText()
        self.local_prefs['CALIBRESPY_TEXT_COLOR'] = self.text_color_combobox.currentText()
        self.local_prefs['CALIBRESPY_BACKGROUND_COLOR'] = self.background_color_combobox.currentText()
        self.local_prefs['CALIBRESPY_ALTERNATING_COLOR'] =  self.alternating_color_combobox.currentText()
        self.local_prefs['CALIBRESPY_ORPHANED_MULTIUSER_LOCK_ELAPSED_HOURS'] = self.orphaned_multiuser_lock_elapsed_time_qspinbox.value()
        if self.load_series_qcheckbox.isChecked():
            self.local_prefs['CALIBRESPY_LOAD_SERIES'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_SERIES'] = 0
        if self.load_publisher_qcheckbox.isChecked():
            self.local_prefs['CALIBRESPY_LOAD_PUBLISHER'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_PUBLISHER'] = 0
        if self.load_tags_qcheckbox.isChecked():
            self.local_prefs['CALIBRESPY_LOAD_TAGS'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_TAGS'] = 0
        if self.load_identifiers_qcheckbox.isChecked():   #Version 1.0.43
            self.local_prefs['CALIBRESPY_LOAD_IDENTIFIERS'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_IDENTIFIERS'] = 0
        if self.load_ratings_qcheckbox.isChecked():     #Version 1.0.52
            self.local_prefs['CALIBRESPY_LOAD_RATINGS'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_RATINGS'] = 0
        if self.load_languages_qcheckbox.isChecked():
            self.local_prefs['CALIBRESPY_LOAD_LANGUAGES'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_LANGUAGES'] = 0
        if self.load_timestamp_qcheckbox.isChecked():
            self.local_prefs['CALIBRESPY_LOAD_TIMESTAMP'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_TIMESTAMP'] = 0
        if self.load_modified_qcheckbox.isChecked():
            self.local_prefs['CALIBRESPY_LOAD_MODIFIED'] = 1
        else:
            self.local_prefs['CALIBRESPY_LOAD_MODIFIED'] = 0
        self.local_prefs['CUSTOMIZE_SIZE_WIDTH_LAST'] = self.size().width()
        self.local_prefs['CUSTOMIZE_SIZE_HEIGHT_LAST'] = self.size().height()

        raw_out = self.local_prefs['CALIBRESPY_FTP_USERID']
        #~ raw_out = as_unicode(raw_out)
        raw_out = as_unicode(raw_out, encoding='utf-8', errors='replace')
        #~ raw_out = as_unicode(raw_out, encoding='ascii')
        self.local_prefs['CALIBRESPY_FTP_USERID'] = codecs.encode(raw_out, 'rot13')

        raw_out = self.local_prefs['CALIBRESPY_FTP_PASSWORD']
        #~ raw_out = as_unicode(raw_out)
        raw_out = as_unicode(raw_out, encoding='utf-8', errors='replace')
        #~ raw_out = as_unicode(raw_out, encoding='ascii')
        self.local_prefs['CALIBRESPY_FTP_PASSWORD'] = codecs.encode(raw_out, 'rot13')

        #---------------------------------------------------------------------
        #~ Note:  custom column prefs set already as part of validation
        #---------------------------------------------------------------------
        self.my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _calibrespy_settings (prefkey,prefvalue,prefblob) VALUES (?,?,?)"
        #~ for key,value in self.local_prefs.iteritems():
        for key,value in iteritems(self.local_prefs):
            if key != "CALIBRESPY_DIALOG_HORIZONTALHEADER_STATE":   # bytearrays are a blob...only calibrespy_dialog handles this.
                value = unicode_type(value)
                value = value.strip()
                self.my_cursor.execute(mysql,(key,value,""))
        #END FOR
        self.my_cursor.execute("commit")
#--------------------------------------------------
#--------------------------------------------------
    def get_custom_column_table(self):
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)
        self.custom_column_list = []
        try:
            mysql = "SELECT label,name,datatype FROM custom_columns"
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            my_db.close()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                 label,name,datatype = row
                 if datatype != "composite" and datatype != "ratings":
                    self.custom_column_list.append(row)
            #END FOR
            del tmp_rows
            self.custom_column_list.sort()
        except Exception as e:
            #~ if never had any custom columns...
            if DEBUG: print("get_custom_column_table: ", as_unicode(e))
            my_db.close()
#--------------------------------------------------
#--------------------------------------------------
    def create_metadatadb_settings_table(self):
        try:
            self.my_cursor.execute("begin")
            mysql_v103 = "CREATE TABLE IF NOT EXISTS _calibrespy_settings(prefkey TEXT NOT NULL COLLATE NOCASE PRIMARY KEY, prefvalue TEXT NOT NULL COLLATE NOCASE, prefblob BLOB, UNIQUE(prefkey))"
            self.my_cursor.execute(mysql_v103)
            self.my_cursor.execute("commit")
        except Exception as e:
            if DEBUG: print("Exception in create_metadatadb_settings_table: ", as_unicode(e))
            try:
                self.my_cursor.execute("commit")
            except:
                pass
            self.upgrade_settings_table_if_needed()
#--------------------------------------------------
#--------------------------------------------------
    def upgrade_settings_table_if_needed(self):
        try:
            mysql = "SELECT prefkey,prefvalue,prefblob FROM _calibrespy_settings"
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            return
        except Exception as e:
            if DEBUG: print("Upgrade of table  _calibrespy_settings from Version 1.02 to Version 1.03 is required...")
            mysql = "SELECT prefkey,prefvalue FROM _calibrespy_settings"
            self.my_cursor.execute(mysql)
            version_102_rows  = self.my_cursor.fetchall()
            self.my_cursor.execute("begin")
            mysql = "DROP TABLE IF EXISTS _calibrespy_settings "
            self.my_cursor.execute(mysql)
            self.my_cursor.execute("commit")
            if DEBUG: print("Version 1.0.2 table _calibrespy_settings was dropped.")
            self.my_cursor.execute("begin")
            mysql_v103 = "CREATE TABLE IF NOT EXISTS _calibrespy_settings(prefkey TEXT NOT NULL COLLATE NOCASE PRIMARY KEY, prefvalue TEXT NOT NULL COLLATE NOCASE, prefblob BLOB, UNIQUE(prefkey))"
            self.my_cursor.execute(mysql_v103)
            self.my_cursor.execute("commit")
            if DEBUG: print("Version 1.0.3 table _calibrespy_settings was created.")
            for row in version_102_rows:
                prefkey,prefvalue = row
                prefblob = ""
                mysql = "INSERT OR REPLACE INTO _calibrespy_settings (prefkey,prefvalue,prefblob) VALUES (?,?,?)"
                my_cursor.execute(mysql,(prefkey,prefvalue,prefblob))
            #END FOR
            self.my_cursor.execute("commit")
            if DEBUG: print("Version 1.0.3 table _calibrespy_settings rows have been added.  Conversion complete.")
#--------------------------------------------------
#--------------------------------------------------
    def exit_no_save(self):
        #~ user may have saved already...just don't save again...
        self.validate_matrix_items(source='save')
        if self.matrix_column_order_is_valid:
            pass
        else:
            return
        self.my_db.close()
        self.hide()
        if self.restart_required:
            self.change_local_prefs_after_customizing(restart=True)
        else:
            self.change_local_prefs_after_customizing(restart=False)
#--------------------------------------------------
#--------------------------------------------------
    def save_and_exit(self):
        self.validate_matrix_items(source='save')
        if self.matrix_column_order_is_valid:
            self.save_prefs()
        else:
            return
        self.my_db.close()
        self.hide()
        if self.restart_required:
            self.change_local_prefs_after_customizing(restart=True)
        else:
            self.change_local_prefs_after_customizing(restart=False)
#--------------------------------------------------
#--------------------------------------------------
    def save_prefs(self):
        self.validate_matrix_items(source='save')
        if self.matrix_column_order_is_valid:
            self.save_local_prefs()
#--------------------------------------------------
#--------------------------------------------------
    def validate_matrix_items(self,source=None):
        #~ do NOT change self.local_prefs here...

        self.matrix_column_order_is_valid = True
        reason = ""

        order_set = set()
        order_list = []
        for c in range(0,self.n_matrix_columns):
            item = self.matrix.item(0,c)
            value = item.text().strip()
            value = as_unicode(value)
            order_set.add(value)
            order_list.append(value)
        #END FOR
        if len(order_set) != len(order_list):
            self.matrix_column_order_is_valid  = False
            reason = reason + "<br>Duplicate Column Assignments."
        if as_unicode("") in order_set:
            self.matrix_column_order_is_valid  = False
            reason = reason + "<br>Blank Column Assignments are Invalid.  An integer from 1 to 22 is required."                                            #Version 1.0.52
        for pos in order_list:
            if pos != "":
                if not pos.isdigit():
                    self.matrix_column_order_is_valid  = False
                    reason = reason + "<br>Non-Numeric Assignments are Invalid.  An integer from 1 to 22 is required, not: " + pos                  #Version 1.0.52
                else:
                    pos = int(pos)
                    if not (pos >= 1 and pos <= self.n_matrix_columns):
                        self.matrix_column_order_is_valid  = False
                        reason = reason + "<br>Assignments must be an Integer between 1 and 22, inclusive: " + as_unicode(pos)                                       #Version 1.0.52
        #END FOR

        for pos in range(1,self.n_matrix_columns):
            pos = as_unicode(pos)
            if not pos in order_set:
                self.matrix_column_order_is_valid  = False
                reason = reason + "<br>Metadata Matrix Column Assignment is Missing:  #" + pos
        #END FOR

        if not self.matrix_column_order_is_valid:
            self.restart_required = False
            msg = "The displayed Metadata Matrix Column Assignments are not valid, and will not be saved.  The reasons:<br>" + reason
            error_dialog(self, _('CalibreSpy'),_(msg), show=True)
        else:
            self.matrix_column_order_is_valid = True
            self.new_column_assignment_dict = {}
            for c in range(0,self.n_matrix_columns):
                item = self.matrix.item(0,c)
                value = item.text().strip()
                value = int(value)
                self.new_column_assignment_dict[c] = value
            #END FOR
            self.compare_new_column_assignments_to_previous_assignments()
#--------------------------------------------------
#--------------------------------------------------
    def compare_new_column_assignments_to_previous_assignments(self):
        self.restart_required = False
        #~ for k,v in self.original_column_assignment_dict.iteritems():  # self.original_column_assignment_dict has *non-incremented* integers, fresh from table _calibrespy_settings at init.
        for k,v in iteritems(self.original_column_assignment_dict):
            if v != (self.new_column_assignment_dict[k] - 1):  # self.new_column_assignment_dict has user-friendly *incremented* integers.
                self.restart_required = True
        #END FOR

        #~ for k,v in self.new_column_assignment_dict.iteritems():
        for k,v in iteritems(self.new_column_assignment_dict):
            if k == AUTHORS_COLUMN:
                self.local_prefs['AUTHORS_COLUMN'] = v - 1
            elif k == TITLE_COLUMN:
                self.local_prefs['TITLE_COLUMN'] = v - 1
            elif k == SERIES_COLUMN:
                self.local_prefs['SERIES_COLUMN'] = v - 1
            elif k == INDEX_COLUMN:
                self.local_prefs['INDEX_COLUMN'] = v - 1
            elif k == TAGS_COLUMN:
                self.local_prefs['TAGS_COLUMN'] = v - 1
            elif k == PUBLISHED_COLUMN:
                self.local_prefs['PUBLISHED_COLUMN'] = v - 1
            elif k == PUBLISHER_COLUMN:
                self.local_prefs['PUBLISHER_COLUMN'] = v - 1
            elif k == LANGUAGES_COLUMN:
                self.local_prefs['LANGUAGES_COLUMN'] = v - 1
            elif k == ADDED_COLUMN:
                self.local_prefs['ADDED_COLUMN'] = v - 1
            elif k == MODIFIED_COLUMN:
                self.local_prefs['MODIFIED_COLUMN'] = v - 1
            elif k == IDENTIFIERS_COLUMN:                                         #Version 1.0.43
                self.local_prefs['IDENTIFIERS_COLUMN'] = v - 1
            elif k == RATINGS_COLUMN:                                               #Version 1.0.52
                self.local_prefs['RATINGS_COLUMN'] = v - 1
            elif k == PATH_COLUMN:
                self.local_prefs['PATH_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_1_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_1_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_2_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_2_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_3_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_3_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_4_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_4_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_5_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_5_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_6_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_6_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_7_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_7_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_8_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_8_COLUMN'] = v - 1
            elif k == CUSTOM_COLUMN_9_COLUMN:
                self.local_prefs['CUSTOM_COLUMN_9_COLUMN'] = v - 1
        #END FOR

        if self.restart_required:
            if not self.skip_restart_message:
                msg = "Caution: You have changed the Matrix Column Assignments, so CalibreSpy will automatically be exited as soon as you click 'Save and Exit'. "
                warning_dialog(self, _('CalibreSpy'), msg, show=True)
                self.skip_restart_message = True
#--------------------------------------------------
    def decode_ftp_data(self):
        if self.ftp_decoded == 1:
            return

        raw_in = self.local_prefs['CALIBRESPY_FTP_USERID']
        #~ raw_in = as_unicode(raw_in)
        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
        #~ raw_in = as_unicode(raw_in, encoding='ascii')
        userid = codecs.decode(raw_in, 'rot13')
        self.local_prefs['CALIBRESPY_FTP_USERID'] = userid

        raw_in = self.local_prefs['CALIBRESPY_FTP_PASSWORD']
        #~ raw_in = as_unicode(raw_in)
        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
        #~ raw_in = as_unicode(raw_in, encoding='ascii')
        password = codecs.decode(raw_in, 'rot13')
        self.local_prefs['CALIBRESPY_FTP_PASSWORD'] = password

        self.ftp_decoded = 1
#--------------------------------------------------
#END of config.py