# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.85"  # Qt6

import os,sys,subprocess
import ast

from qt.core import (Qt, QDialog, QVBoxLayout, QFont, QComboBox, QPushButton, QLabel, QFileDialog)
from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG

from polyglot.builtins import as_unicode, only_unicode_recursive, unicode_type

from calibre_plugins.calibrespy.config import prefs

class ArbitraryFileDialog(QDialog):

    def __init__(self,parent,icon,font,style_text):

        QDialog.__init__(self, parent)

        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint)

        self.setWindowTitle('CalibreSpy:  Select & Open Arbitrary Files')

        self.layout = QVBoxLayout()
        self.setLayout(self.layout)

        self.setStyleSheet(style_text)
        #~ self.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")

        font.setBold(False)
        font.setPointSize(10)

        self.setToolTip("<p style='white-space:wrap'>History of the arbitrary files that you have ever opened here, excluding those you have removed.  The last-selected file, if any, is always displayed.")

        minwidth = 500

        self.run_arbitrary_program_combobox = QComboBox()
        self.run_arbitrary_program_combobox.setEditable(False)
        self.run_arbitrary_program_combobox.setFrame(True)
        self.run_arbitrary_program_combobox.setDuplicatesEnabled(False)  # Note that it is always possible to programmatically insert duplicate items into the combobox.
        self.run_arbitrary_program_combobox.setMaxCount(50)
        self.run_arbitrary_program_combobox.setMaxVisibleItems(50)
        self.run_arbitrary_program_combobox.setMinimumWidth(minwidth)
        self.run_arbitrary_program_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)       # AdjustToContents = 0
        self.run_arbitrary_program_combobox.setFont(font)
        self.run_arbitrary_program_combobox.setToolTip("<p style='white-space:wrap'>The last-selected file.")
        self.layout.addWidget(self.run_arbitrary_program_combobox)

        history_list = prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST']
        history_list = as_unicode(history_list)
        history_tmp = ast.literal_eval(history_list)
        history_list = only_unicode_recursive(history_tmp)
        del history_tmp
        history_list = list(set(history_list))  #no duplicates
        history_list.sort()
        for row in history_list:
            if row > " ":
                self.run_arbitrary_program_combobox.addItem(row)
        #END FOR

        last_selected = prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_LAST_SELECTED']
        if last_selected in history_list:
            self.run_arbitrary_program_combobox.setCurrentText(last_selected)
        else:
            self.run_arbitrary_program_combobox.setCurrentIndex(-1)

        self.run_arbitrary_program_combobox.repaint()

        font.setPointSize(9)

        self.push_button_execute_selected_file_qpushbutton = QPushButton("Execute Selected File", self)
        self.push_button_execute_selected_file_qpushbutton.clicked.connect(self.execute_selected_file)
        self.push_button_execute_selected_file_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_execute_selected_file_qpushbutton.setFont(font)
        self.push_button_execute_selected_file_qpushbutton.setDefault(True)
        self.push_button_execute_selected_file_qpushbutton.setToolTip("<p style='white-space:wrap'>Open the file displayed above.")
        self.layout.addWidget(self.push_button_execute_selected_file_qpushbutton)

        self.push_button_execute_new_file_qpushbutton = QPushButton("Select New File to Execute", self)
        self.push_button_execute_new_file_qpushbutton.clicked.connect(self.select_and_open_files)
        self.push_button_execute_new_file_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_execute_new_file_qpushbutton.setFont(font)
        self.push_button_execute_new_file_qpushbutton.setDefault(False)
        self.push_button_execute_new_file_qpushbutton.setToolTip("<p style='white-space:wrap'>Select a new file to open.  The directory to be first displayed for selection is the directory of your last-selected file (shown above), which is also shown as the 'Current Directory' (below).")
        self.layout.addWidget(self.push_button_execute_new_file_qpushbutton)

        font.setPointSize(7)

        self.push_button_remove_item_qpushbutton = QPushButton("Remove Selected Item from List", self)
        self.push_button_remove_item_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_remove_item_qpushbutton.clicked.connect(self.remove_selected_item_from_list)
        self.push_button_remove_item_qpushbutton.setFont(font)
        self.push_button_remove_item_qpushbutton.setDefault(False)
        self.push_button_remove_item_qpushbutton.setToolTip("<p style='white-space:wrap'>Remove the selected item from your history.")
        self.layout.addWidget(self.push_button_remove_item_qpushbutton)

        self.spacer_1_label = QLabel("")
        self.layout.addWidget(self.spacer_1_label)

        self.current_directory_qlabel = QLabel("Current Directory:")
        self.current_directory_qlabel.setMinimumWidth(minwidth)
        self.current_directory_qlabel.setToolTip("<p style='white-space:wrap'>The directory of the file displayed above, which is the last file that was selected.")
        self.layout.addWidget(self.current_directory_qlabel)

        chosen_path = self.run_arbitrary_program_combobox.currentText()
        if chosen_path > " ":
            path = chosen_path.replace(os.sep, '/')
            default_dir,tail = os.path.split(path)
            if not default_dir:
                default_dir = ""
        else:
            default_dir = ""
        prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_CURRENT_DIRECTORY'] = unicode_type(default_dir)
        prefs
        s = "Current Directory:   " + default_dir
        self.current_directory_qlabel.setText(s)
        self.current_directory_qlabel.repaint()

        self.resize(self.sizeHint())

        self.run_arbitrary_program_combobox.currentIndexChanged.connect(self.event_current_text_changed)

    #-----------------------------------------------------
    def event_current_text_changed(self,event):
        chosen_path = self.run_arbitrary_program_combobox.currentText()
        self.save_all_items(chosen_path)
    #-----------------------------------------------------
    def execute_selected_file(self):
        try:
            chosen_path = self.run_arbitrary_program_combobox.currentText()
            if os.path.exists(chosen_path):
                p_pid = subprocess.Popen(chosen_path, shell=True)
                if DEBUG: print("Chosen path: ", chosen_path)
                self.save_all_items(chosen_path)
            else:
                self.select_and_open_files()
        except Exception as e:
            if DEBUG: print(bytes(e))
    #-----------------------------------------------------
    def select_and_open_files(self):
        try:
            chosen_path = self.choose_file_to_open()
            if not chosen_path:
                return
            else:
                p_pid = subprocess.Popen(chosen_path, shell=True)
                if DEBUG: print("Chosen path: ", chosen_path)
                self.save_all_items(chosen_path)
                self.repaint()
        except Exception as e:
            if DEBUG: print(bytes(e))
    #-----------------------------------------------------
    def choose_file_to_open(self):
        name = "choose_file_to_open"
        title = "CalibreSpy: Choose Arbitrary File to Open"

        path = self.run_arbitrary_program_combobox.currentText()

        if not path:
            path = prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_LAST_SELECTED']

        if not path > " ":
            path = prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_LAST_SELECTED']

        if not path > " ":
            if self.run_arbitrary_program_combobox.count() > 0:
                path = self.run_arbitrary_program_combobox.itemText(0)
            else:
                path = "/"

        if DEBUG: print("choose file to open path: ", path)

        path = path.replace(os.sep, '/')
        default_dir,tail = os.path.split(path)
        if not default_dir:
            default_dir = prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_CURRENT_DIRECTORY']
        else:
            prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_CURRENT_DIRECTORY'] = unicode_type(default_dir)
            prefs

        s = "Current Directory:   " + default_dir
        self.current_directory_qlabel.setText(s)
        self.current_directory_qlabel.repaint()
        if DEBUG: print(s)

        title = "Select Arbitrary File to Open"

        import_tuple = QFileDialog.getOpenFileName(None,"",default_dir,("All Files(*.*)") )
        if not import_tuple:
            return None
        path, dummy = import_tuple
        if not path:
            return None
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        if os.path.isfile(path):
            return path
        else:
            return None
    #-----------------------------------------------------
    def save_all_items(self,chosen_path):

        path = chosen_path.replace(os.sep, '/')
        default_dir,tail = os.path.split(path)

        prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_CURRENT_DIRECTORY'] = unicode_type(default_dir)
        s = "Current Directory:   " + default_dir
        self.current_directory_qlabel.setText(s)
        self.current_directory_qlabel.repaint()

        prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST_LAST_SELECTED'] = unicode_type(chosen_path)

        prefs

        #~ Note that it is always possible to programmatically insert duplicate items into the combobox.
        i = self.run_arbitrary_program_combobox.findText(chosen_path)
        if i < 0:
            self.run_arbitrary_program_combobox.addItem(chosen_path)
        self.run_arbitrary_program_combobox.setCurrentText(chosen_path)
        self.run_arbitrary_program_combobox.model().sort(0)

        self.run_arbitrary_program_combobox.repaint()

        history_list = []

        nitems = self.run_arbitrary_program_combobox.count()

        for i in range(0,nitems):
            row = self.run_arbitrary_program_combobox.itemText(i)
            if not row in history_list:
                if row > " ":
                    history_list.append(row)
        #END FOR

        history_list = only_unicode_recursive(history_list)

        history_list.sort()
        prefs['CALIBRESPY_ARBITRARY_PROGRAMS_HISTORY_LIST'] = unicode_type(history_list)
        prefs

        del history_list
    #-----------------------------------------------------
    def remove_selected_item_from_list(self):
        index = self.run_arbitrary_program_combobox.currentIndex()
        if index > -1:
            self.run_arbitrary_program_combobox.removeItem(index)
            chosen_path = self.run_arbitrary_program_combobox.currentText()
            self.save_all_items(chosen_path)
#END OF arbitrary_file_dialog.py