# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.191"  #Qt6

import os,sys,ast,re

from qt.core import (Qt, QDialog, QLabel,  QFont, QApplication,
                                       QIcon, QPixmap, QMargins, QScrollArea,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox,
                                       QSize, QVBoxLayout, QHBoxLayout, QCheckBox, QGroupBox,
                                       QComboBox, QPushButton, QFileDialog)

from calibre import isbytestring, sanitize_file_name_unicode
from calibre.constants import filesystem_encoding, DEBUG, config_dir
from calibre.gui2 import gprefs, pixmap_to_data
from calibre.ebooks.metadata import author_to_author_sort

from polyglot.builtins import as_unicode, iteritems, range, unicode_type

from calibre_plugins.job_spy.config import prefs

RE_ANYTHING = "^.+$"
IMAGE_TYPES_SUPPORTED = "Images(*.bmp *.gif *.ico *.icns *.jpeg *.jpg *.mng *.pbm *.pgm *.ppm *.png *.svg *.tga *.tif *.tiff *.wbmp *.webp *.xbm *.xpm *.xsvg)"   # http://doc.qt.io/qt-5/qimagereader.html#supportedImageFormats (not .jp2)

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(90, 90)

    def __init__(self, parent, unique_pref_value):
        QDialog.__init__(self, parent)
        self.unique_pref_value = unique_pref_value
        self.geom = gprefs.get(unique_pref_value, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):

        #~ if DEBUG: self.geom = None

        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_value] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class TagBrowserIconsDialog(SizePersistedDialog):

    def __init__(self, parent, maingui,nothingfound,filenotfound,tagbrowser_icon_tool_is_running):

        self.maingui = maingui
        self.guidb = self.maingui.library_view.model().db

        self.default_nothingfound = nothingfound    #nothing_found.png
        self.default_filenotfound = filenotfound         #filenotfound.png

        self.tagbrowser_icon_tool_is_running = tagbrowser_icon_tool_is_running

        self.ensure_icon_directory_exists()

        unique_pref_value = 'Job_Spy:tagbrowser_icons_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_value)

        mytitle = 'JS+ GUI Tool:  Tag Browser Icon Assignments'
        self.setWindowTitle(_(mytitle))

        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint | Qt.WindowStaysOnTopHint)              # http://doc.qt.io/qt-5/qt.html#WindowType-enum

        #~ --------------------------------------------------------------------
        #~ Retrieve Essential Data
        #~ --------------------------------------------------------------------
        self.tv = self.maingui.tags_view
        self.tvm = self.tv._model
        #~ --------------------------------------------------------------------
        self.state_map_dict = {}
        self.category_list = []
        #~ --------------------------------------------------------------------
        expanded_categories, state_map = self.tv.get_state()
        #~ for category,values in state_map.iteritems():
        for category,values in iteritems(state_map):
            if category.startswith("@") or category.startswith("#"):
                self.state_map_dict[category] = values
                self.category_list.append(category)
        #END FOR
        del expanded_categories
        del state_map
        #~ --------------------------------------------------------------------
        self.ifc_dict = {}
        #~ --------------------------------------------------------------------
        for category,values in iteritems(self.state_map_dict):
            if category.startswith("@") or category.startswith("#"):
                ifc = self.tvm.index_for_category(category)
                if ifc is not None:
                    self.ifc_dict[category] = ifc
        #END FOR
        #~ --------------------------------------------------------------------
        self.ifc_named_path_dict = {}
        #~ --------------------------------------------------------------------
        for category,ifc in iteritems(self.ifc_dict):
            if category.startswith("@") or category.startswith("#"):
                named_path = self.tvm.named_path_for_index(ifc)
                if ifc is not None:
                    self.ifc_named_path_dict[ifc] = named_path
        #END FOR
        #~ --------------------------------------------------------------------
        #~ --------------------------------------------------------------------
        #~ Qt
        #~ --------------------------------------------------------------------
        t = "<p style='white-space:wrap'>Assign custom icons to your Tag Browser Categories."
        t = t + "<br><br>You may assign them such that they either default by each unique combination of "
        t = t + "Source Custom Column and that Source's Value, or just use a 'default' icon for each Source Custom Column."
        t = t + "<br><br>Icons by Tag Browser Category (only) may be assigned using existing Tag Browser functionality.  Select a Category, then 'right-click' it to display its options."
        t = t + "<br><br>When Calibre starts, your Tag Browser icons specified by this JS GUI Tool can be automatically applied if you 'check' the specified option in this Tool."

        self.setToolTip(t)
        #--------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(10)
        #--------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.setLayout(self.layout_top)
        #--------------------------------------------------
        self.groupbox_comboboxes = QGroupBox('Tag Browser User Category Icon Assignments')
        self.groupbox_comboboxes.setToolTip(t)
        self.layout_top.addWidget(self.groupbox_comboboxes)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox = QVBoxLayout()
        self.groupbox_comboboxes.setLayout(self.vlayout_comboboxes_groupbox)
        #--------------------------------------------------
        self.hlayout_comboboxes_groupbox = QHBoxLayout()
        self.hlayout_comboboxes_groupbox.setAlignment(Qt.AlignLeft)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox.addLayout(self.hlayout_comboboxes_groupbox)
        #--------------------------------------------------
        #-----------------------
        minwidth = 150
        maxwidth = 150
        #-----------------------
        self.uc_category_combobox = QComboBox()
        self.uc_category_combobox.setEditable(False)
        self.uc_category_combobox.setFrame(True)
        self.uc_category_combobox.setDuplicatesEnabled(False)  # Note that it is always possible to programmatically insert duplicate items into the combobox.
        self.uc_category_combobox.setMaxVisibleItems(25)
        self.uc_category_combobox.setMinimumWidth(minwidth)
        self.uc_category_combobox.setMaximumWidth(maxwidth)
        self.uc_category_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)       # AdjustToContents = 0
        self.uc_category_combobox.setFont(font)
        self.uc_category_combobox.setToolTip("<p style='white-space:wrap'>Select a Tag Browser Category for which you would like to assign Tag Browser icons.<br><br>That Category will automatically be expanded in the Tag Browser so that you may see the results of your Icon assignments.")
        self.hlayout_comboboxes_groupbox.addWidget(self.uc_category_combobox)

        self.category_list.sort()
        for category in self.category_list:
            self.uc_category_combobox.addItem(category)
        #END FOR

        self.uc_category_combobox.setCurrentIndex(-1)
        #--------------------------------------------------
        self.uc_child_value_source_list = []

        self.uc_child_value_source_combobox = QComboBox()
        self.uc_child_value_source_combobox.setEditable(False)
        self.uc_child_value_source_combobox.setFrame(True)
        self.uc_child_value_source_combobox.setDuplicatesEnabled(False)  # Note that it is always possible to programmatically insert duplicate items into the combobox.
        self.uc_child_value_source_combobox.setMaxVisibleItems(25)
        self.uc_child_value_source_combobox.setMinimumWidth(minwidth)
        self.uc_child_value_source_combobox.setMaximumWidth(maxwidth)
        self.uc_child_value_source_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)       # AdjustToContents = 0
        self.uc_child_value_source_combobox.setFont(font)
        self.uc_child_value_source_combobox.setToolTip("<p style='white-space:wrap'>Select a data Source used by the selected Category.")
        self.hlayout_comboboxes_groupbox.addWidget(self.uc_child_value_source_combobox)

        #~ self.uc_child_value_source_combobox.addItem(QUESTION_MARK)
        #--------------------------------------------------
        self.uc_child_source_value_list = []

        self.uc_child_source_value_combobox = QComboBox()
        self.uc_child_source_value_combobox.setEditable(False)
        self.uc_child_source_value_combobox.setFrame(True)
        self.uc_child_source_value_combobox.setDuplicatesEnabled(False)  # Note that it is always possible to programmatically insert duplicate items into the combobox.
        self.uc_child_source_value_combobox.setMaxVisibleItems(25)
        self.uc_child_source_value_combobox.setMinimumWidth(minwidth)
        self.uc_child_source_value_combobox.setMaximumWidth(maxwidth)
        self.uc_child_source_value_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)       # AdjustToContents = 0
        self.uc_child_source_value_combobox.setFont(font)
        self.uc_child_source_value_combobox.setToolTip("<p style='white-space:wrap'>Select the Source's Value for which to assign an icon.")
        self.hlayout_comboboxes_groupbox.addWidget(self.uc_child_source_value_combobox)

        #~ self.uc_child_source_value_combobox.addItem(QUESTION_MARK)
        #~ --------------------------------------------------------------------
        self.uc_source_value_icons_list = []

        self.uc_child_source_value_icon_combobox = QComboBox()
        self.uc_child_source_value_icon_combobox.setEditable(True)
        self.uc_child_source_value_icon_combobox.setFrame(True)
        self.uc_child_source_value_icon_combobox.setDuplicatesEnabled(False)  # Note that it is always possible to programmatically insert duplicate items into the combobox.
        self.uc_child_source_value_icon_combobox.setMaxVisibleItems(25)
        self.uc_child_source_value_icon_combobox.setMinimumWidth(minwidth)
        self.uc_child_source_value_icon_combobox.setMaximumWidth(maxwidth)
        self.uc_child_source_value_icon_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)       # AdjustToContents = 0

        self.uc_child_source_value_icon_combobox.view().setTextElideMode(Qt.ElideLeft)    #popup()
        self.uc_child_source_value_icon_combobox.view().setMinimumWidth(500)               #popup()


        self.uc_child_source_value_icon_combobox.setFont(font)
        t = "<p style='white-space:wrap'>If there is a previously assigned Icon for the current Value on the left, it will be displayed here.\
                                                               <br><br>However, all previously assigned Icons for all currently available Values will be listed in this drop-down.\
                                                               <br><br>If you wish to use another Values Icon for the current Value, you may choose that Value's currently assigned Icon File, and assign it.\
                                                               It will be saved for the current Value under its own unique Value name.\
                                                               <br><br>Multiple Values never share the identical Icon File.\
                                                               <br><br>If you want most Values within a Source to have the identical Icon, you should assign their Source a Default Icon.  For exceptions, you may assign unique Icon Files as you please."
        self.uc_child_source_value_icon_combobox.setToolTip(t)
        self.hlayout_comboboxes_groupbox.addWidget(self.uc_child_source_value_icon_combobox)

        self.push_button_select_icon_file_qpushbutton = QPushButton("Browse", self)
        self.push_button_select_icon_file_qpushbutton.clicked.connect(self.select_icon_file)
        self.push_button_select_icon_file_qpushbutton.setMinimumWidth(75)
        self.push_button_select_icon_file_qpushbutton.setMaximumWidth(75)
        self.push_button_select_icon_file_qpushbutton.setFont(font)
        self.push_button_select_icon_file_qpushbutton.setDefault(True)
        t = "<p style='white-space:wrap'>Select the desired Icon File for the applicable value(s).\
                                                             <br><br>The file extension filter will be:  " +  IMAGE_TYPES_SUPPORTED  + " \
                                                             <br><br>If there are multiple images within the same image file, the <b>first</b> image will always be used.\
                                                             <br><br>If you select an image file but it appears to not be displayed, it will be because it is either corrupted, or it is an 'empty' image with nothing to see even when properly displayed.\
                                                              To test such an image file for validity, you must use an image-editing application, such as IrfanView."
        self.push_button_select_icon_file_qpushbutton.setToolTip(t)
        self.hlayout_comboboxes_groupbox.addWidget(self.push_button_select_icon_file_qpushbutton)

        self.push_button_execute_assignment_for_single_value_qpushbutton = QPushButton("Assign Icon to Single Value", self)
        self.push_button_execute_assignment_for_single_value_qpushbutton.clicked.connect(self.execute_assignment_for_single_value)
        self.push_button_execute_assignment_for_single_value_qpushbutton.setMinimumWidth(minwidth + 60)
        self.push_button_execute_assignment_for_single_value_qpushbutton.setFont(font)
        self.push_button_execute_assignment_for_single_value_qpushbutton.setDefault(False)
        self.push_button_execute_assignment_for_single_value_qpushbutton.setToolTip("<p style='white-space:wrap'>Assign the current Icon to the single, displayed current Value.")
        self.hlayout_comboboxes_groupbox.addWidget(self.push_button_execute_assignment_for_single_value_qpushbutton)

        #--------------------------------------------------
        self.hlayout_regex_groupbox = QHBoxLayout()
        self.hlayout_regex_groupbox.setAlignment(Qt.AlignLeft)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox.addLayout(self.hlayout_regex_groupbox)
        #--------------------------------------------------
        self.uc_child_source_value_regex_spacer_01_label = QLabel(" ")
        self.uc_child_source_value_regex_spacer_01_label.setMinimumWidth(minwidth)
        self.uc_child_source_value_regex_spacer_01_label.setMaximumWidth(maxwidth)
        self.hlayout_regex_groupbox.addWidget(self.uc_child_source_value_regex_spacer_01_label)

        self.uc_child_source_value_regex_spacer_02_label = QLabel(" ")                 # also holds the current Source's default icon image...
        self.uc_child_source_value_regex_spacer_02_label.setAlignment(Qt.AlignCenter)
        self.uc_child_source_value_regex_spacer_02_label.setMinimumWidth(minwidth)
        self.uc_child_source_value_regex_spacer_02_label.setMaximumWidth(maxwidth)
        self.uc_child_source_value_regex_spacer_02_label.setToolTip("<p style='white-space:wrap'>The Current Source's default icon image, if any.")
        self.hlayout_regex_groupbox.addWidget(self.uc_child_source_value_regex_spacer_02_label)

        self.uc_source_value_regex_list = []

        s = prefs['GUI_TOOLS_TAGBROWSER_ICONS_REGEX_COMPLETER_LIST']
        s = as_unicode(s)
        self.uc_source_value_regex_list = ast.literal_eval(s)
        if not isinstance(self.uc_source_value_regex_list,list):
            self.uc_source_value_regex_list = []

        if not RE_ANYTHING in self.uc_source_value_regex_list:
            self.uc_source_value_regex_list.append(RE_ANYTHING)

        self.uc_source_value_regex_list = list(set(self.uc_source_value_regex_list)) #no duplicates

        self.uc_source_value_regex_list.sort()

        font.setPointSize(8)

        self.uc_child_source_value_regex_combobox = QComboBox()
        self.uc_child_source_value_regex_combobox.setEditable(True)
        self.uc_child_source_value_regex_combobox.setFrame(True)
        self.uc_child_source_value_regex_combobox.setMinimumContentsLength(minwidth)
        self.uc_child_source_value_regex_combobox.setDuplicatesEnabled(False)  # Note that it is always possible to programmatically insert duplicate items into the combobox.
        self.uc_child_source_value_regex_combobox.setMaxVisibleItems(25)
        self.uc_child_source_value_regex_combobox.setMinimumWidth(minwidth)
        self.uc_child_source_value_regex_combobox.setMaximumWidth(maxwidth)
        self.uc_child_source_value_regex_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)       # AdjustToContents = 0
        self.uc_child_source_value_regex_combobox.setFont(font)
        t ="<p style='white-space:wrap'>Specify the Regular Expression to restrict the Values to only those that match your RE.  Only that restricted list of Values will then be available for further actions."
        self.uc_child_source_value_regex_combobox.setToolTip(t)

        self.hlayout_regex_groupbox.addWidget(self.uc_child_source_value_regex_combobox)

        for regex in self.uc_source_value_regex_list:
            self.uc_child_source_value_regex_combobox.addItem(regex)
        #END FOR

        font.setPointSize(10)

        self.uc_child_source_value_regex_spacer_03_label = QLabel(" ")       # also holds the current icon image...
        self.uc_child_source_value_regex_spacer_03_label.setAlignment(Qt.AlignCenter)
        self.uc_child_source_value_regex_spacer_03_label.setMinimumWidth(minwidth)
        self.uc_child_source_value_regex_spacer_03_label.setMaximumWidth(maxwidth)
        self.uc_child_source_value_regex_spacer_03_label.setToolTip("<p style='white-space:wrap'>The Current Value's specifically assigned icon image, if any.")
        self.hlayout_regex_groupbox.addWidget(self.uc_child_source_value_regex_spacer_03_label)

        self.uc_child_source_value_regex_spacer_04_label = QLabel(" ")
        self.uc_child_source_value_regex_spacer_04_label.setMinimumWidth(75)
        self.uc_child_source_value_regex_spacer_04_label.setMaximumWidth(75)
        self.hlayout_regex_groupbox.addWidget(self.uc_child_source_value_regex_spacer_04_label)

        self.push_button_execute_assignment_to_current_values_qpushbutton = QPushButton("Assign Icon to Current Values", self)
        self.push_button_execute_assignment_to_current_values_qpushbutton.clicked.connect(self.execute_assignment_for_all_current_values)
        self.push_button_execute_assignment_to_current_values_qpushbutton.setMinimumWidth(minwidth + 60)
        self.push_button_execute_assignment_to_current_values_qpushbutton.setFont(font)
        self.push_button_execute_assignment_to_current_values_qpushbutton.setDefault(False)
        self.push_button_execute_assignment_to_current_values_qpushbutton.setToolTip("<p style='white-space:wrap'>Assign Icon to all Values currently available in the drop-down list, potentially based on your Regular Expression restrictions, if any.")
        self.hlayout_regex_groupbox.addWidget(self.push_button_execute_assignment_to_current_values_qpushbutton)
        #~ --------------------------------------------------------------------

        font.setPointSize(8)

        #--------------------------------------------------
        self.hlayout_filter_groupbox = QHBoxLayout()
        self.hlayout_filter_groupbox.setAlignment(Qt.AlignLeft)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox.addLayout(self.hlayout_filter_groupbox)
        #--------------------------------------------------

        self.uc_child_source_value_filter_spacer_01_label = QLabel(" ")
        self.uc_child_source_value_filter_spacer_01_label.setMinimumWidth(minwidth)
        self.uc_child_source_value_filter_spacer_01_label.setMaximumWidth(maxwidth)
        self.hlayout_filter_groupbox.addWidget(self.uc_child_source_value_filter_spacer_01_label)

        self.push_button_assign_source_default_icon_qpushbutton = QPushButton("Assign Default Icon", self)
        self.push_button_assign_source_default_icon_qpushbutton.clicked.connect(self.assign_source_default_icon)
        self.push_button_assign_source_default_icon_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_assign_source_default_icon_qpushbutton.setMaximumWidth(maxwidth)
        self.push_button_assign_source_default_icon_qpushbutton.setFont(font)
        self.push_button_assign_source_default_icon_qpushbutton.setDefault(False)
        self.push_button_assign_source_default_icon_qpushbutton.setToolTip("<p style='white-space:wrap'>Select the Default Icon that you wish to be used for all of the Source's Values that have not otherwise been assigned a specific Icon of their own.")
        self.hlayout_filter_groupbox.addWidget(self.push_button_assign_source_default_icon_qpushbutton)

        self.push_button_execute_filter_using_regex_qpushbutton = QPushButton("Filter Values Using Regex", self)
        self.push_button_execute_filter_using_regex_qpushbutton.clicked.connect(self.execute_filter_using_regex)
        self.push_button_execute_filter_using_regex_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_execute_filter_using_regex_qpushbutton.setMaximumWidth(maxwidth)
        self.push_button_execute_filter_using_regex_qpushbutton.setFont(font)
        self.push_button_execute_filter_using_regex_qpushbutton.setDefault(False)
        self.push_button_execute_filter_using_regex_qpushbutton.setToolTip("<p style='white-space:wrap'>Filter all of the current Values currently listed in the Values drop-down above such that only those discrete Values matching the specified Regular Expression remain.")
        self.hlayout_filter_groupbox.addWidget(self.push_button_execute_filter_using_regex_qpushbutton)

        font.setPointSize(8)
        font.setBold(True)

        self.uc_child_source_value_filter_spacer_03_label = QLabel(" ")
        self.uc_child_source_value_filter_spacer_03_label.setFont(font)
        self.uc_child_source_value_filter_spacer_03_label.setAlignment(Qt.AlignCenter)
        self.uc_child_source_value_filter_spacer_03_label.setMinimumWidth(minwidth + minwidth + 75)
        self.uc_child_source_value_filter_spacer_03_label.setMaximumWidth(maxwidth + maxwidth + 75)
        self.hlayout_filter_groupbox.addWidget(self.uc_child_source_value_filter_spacer_03_label)

        font.setBold(False)
        font.setPointSize(10)

        #--------------------------------------------------
        self.hlayout_removal_single_groupbox = QHBoxLayout()
        self.hlayout_removal_single_groupbox.setAlignment(Qt.AlignLeft)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox.addLayout(self.hlayout_removal_single_groupbox)
        #--------------------------------------------------

        self.removal_spacer_01_label = QLabel("")
        self.removal_spacer_01_label.setMinimumWidth(minwidth)
        self.removal_spacer_01_label.setMaximumWidth(maxwidth)
        self.hlayout_removal_single_groupbox.addWidget(self.removal_spacer_01_label)

        font.setPointSize(8)

        self.push_button_remove_source_default_icon_qpushbutton = QPushButton("Remove Default Icon", self)
        self.push_button_remove_source_default_icon_qpushbutton.clicked.connect(self.remove_source_default_icon)
        self.push_button_remove_source_default_icon_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_remove_source_default_icon_qpushbutton.setMaximumWidth(maxwidth)
        self.push_button_remove_source_default_icon_qpushbutton.setFont(font)
        self.push_button_remove_source_default_icon_qpushbutton.setDefault(False)
        self.push_button_remove_source_default_icon_qpushbutton.setToolTip("<p style='white-space:wrap'>Remove the Source's previously assigned Default Icon.")
        self.hlayout_removal_single_groupbox.addWidget(self.push_button_remove_source_default_icon_qpushbutton)

        self.push_button_remove_current_regex_qpushbutton = QPushButton("Remove Current Regex", self)
        self.push_button_remove_current_regex_qpushbutton.clicked.connect(self.remove_current_regex)
        self.push_button_remove_current_regex_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_remove_current_regex_qpushbutton.setMaximumWidth(maxwidth)
        self.push_button_remove_current_regex_qpushbutton.setFont(font)
        self.push_button_remove_current_regex_qpushbutton.setDefault(False)
        self.push_button_remove_current_regex_qpushbutton.setToolTip("<p style='white-space:wrap'>Remove the currently displayed Regular Expression from the list of available REs shown in the drop-down.")
        self.hlayout_removal_single_groupbox.addWidget(self.push_button_remove_current_regex_qpushbutton)

        self.removal_spacer_03_label = QLabel("")
        self.removal_spacer_03_label.setMinimumWidth(minwidth)
        self.removal_spacer_03_label.setMaximumWidth(maxwidth)
        self.hlayout_removal_single_groupbox.addWidget(self.removal_spacer_03_label)

        self.push_button_remove_single_value_qpushbutton = QPushButton("Remove Icon for Single Value", self)
        self.push_button_remove_single_value_qpushbutton.clicked.connect(self.remove_single_value)
        self.push_button_remove_single_value_qpushbutton.setMinimumWidth(minwidth + 150)
        self.push_button_remove_single_value_qpushbutton.setFont(font)
        self.push_button_remove_single_value_qpushbutton.setDefault(False)
        self.push_button_remove_single_value_qpushbutton.setToolTip("<p style='white-space:wrap'>Remove the previously assigned Icon for the currently specified single Value.")
        self.hlayout_removal_single_groupbox.addWidget(self.push_button_remove_single_value_qpushbutton)

         #--------------------------------------------------
        self.hlayout_removal_all_current_groupbox = QHBoxLayout()
        self.hlayout_removal_all_current_groupbox.setAlignment(Qt.AlignLeft)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox.addLayout(self.hlayout_removal_all_current_groupbox)
        #--------------------------------------------------

        self.removal_all_spacer_01_label = QLabel("")
        self.removal_all_spacer_01_label.setMinimumWidth(minwidth)
        self.removal_all_spacer_01_label.setMaximumWidth(maxwidth)
        self.hlayout_removal_all_current_groupbox.addWidget(self.removal_all_spacer_01_label)

        self.removal_all_spacer_02_label = QLabel("")
        self.removal_all_spacer_02_label.setMinimumWidth(minwidth)
        self.removal_all_spacer_02_label.setMaximumWidth(maxwidth)
        self.hlayout_removal_all_current_groupbox.addWidget(self.removal_all_spacer_02_label)

        self.removal_all_spacer_03_label = QLabel("")
        self.removal_all_spacer_03_label.setMinimumWidth(minwidth)
        self.removal_all_spacer_03_label.setMaximumWidth(maxwidth)
        self.hlayout_removal_all_current_groupbox.addWidget(self.removal_all_spacer_03_label)

        self.removal_all_spacer_04_label = QLabel("")
        self.removal_all_spacer_04_label.setMinimumWidth(minwidth)
        self.removal_all_spacer_04_label.setMaximumWidth(maxwidth)
        self.hlayout_removal_all_current_groupbox.addWidget(self.removal_all_spacer_04_label)

        self.push_button_remove_current_values_qpushbutton = QPushButton("Remove Icons for Current Values", self)
        self.push_button_remove_current_values_qpushbutton.clicked.connect(self.remove_current_values)
        self.push_button_remove_current_values_qpushbutton.setMinimumWidth(minwidth + 150)
        self.push_button_remove_current_values_qpushbutton.setFont(font)
        self.push_button_remove_current_values_qpushbutton.setDefault(False)
        self.push_button_remove_current_values_qpushbutton.setToolTip("<p style='white-space:wrap'>Remove the previously assigned Icons for every Value currently available in the drop-down list.")
        self.hlayout_removal_all_current_groupbox.addWidget(self.push_button_remove_current_values_qpushbutton)

        font.setPointSize(10)

        #--------------------------------------------------
        self.hlayout_refresh_tagbrowser_groupbox = QHBoxLayout()
        self.hlayout_refresh_tagbrowser_groupbox.setAlignment(Qt.AlignLeft)
        #--------------------------------------------------
        self.vlayout_comboboxes_groupbox.addLayout(self.hlayout_refresh_tagbrowser_groupbox)
        #--------------------------------------------------

        self.push_button_refresh_tagbrowser_label = QLabel(" ")
        self.push_button_refresh_tagbrowser_label.setFont(font)
        self.push_button_refresh_tagbrowser_label.setAlignment(Qt.AlignCenter)
        wspan = (minwidth)
        self.push_button_refresh_tagbrowser_label.setMinimumWidth(wspan)
        self.push_button_refresh_tagbrowser_label.setMaximumWidth(wspan)
        self.hlayout_refresh_tagbrowser_groupbox.addWidget(self.push_button_refresh_tagbrowser_label)

        self.push_button_refresh_matrix_qpushbutton = QPushButton("Refresh Matrix", self)
        self.push_button_refresh_matrix_qpushbutton.clicked.connect(self.refresh_entire_matrix_manually)
        self.push_button_refresh_matrix_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_refresh_matrix_qpushbutton.setMaximumWidth(maxwidth)
        self.push_button_refresh_matrix_qpushbutton.setFont(font)
        self.push_button_refresh_matrix_qpushbutton.setDefault(False)
        self.push_button_refresh_matrix_qpushbutton.setToolTip("<p style='white-space:wrap'>Refresh the Matrix shown below to use the most current Assignments, if any.")
        self.hlayout_refresh_tagbrowser_groupbox.addWidget(self.push_button_refresh_matrix_qpushbutton)

        self.push_button_refresh_tagbrowser_qpushbutton = QPushButton("Refresh Tag Browser", self)
        self.push_button_refresh_tagbrowser_qpushbutton.clicked.connect(self.recount_tagbrowser)
        self.push_button_refresh_tagbrowser_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_refresh_tagbrowser_qpushbutton.setMaximumWidth(maxwidth)
        self.push_button_refresh_tagbrowser_qpushbutton.setFont(font)
        self.push_button_refresh_tagbrowser_qpushbutton.setDefault(False)
        self.push_button_refresh_tagbrowser_qpushbutton.setToolTip("<p style='white-space:wrap'>Refresh the Calibre Tag Browser to show the most current Assignments, if any.\
                                                                                                                                                             <br><br>Calibre <b>must</b> have been started with the 'Apply Icons at Calibre Startup' option <b>already</b> selected.")
        self.hlayout_refresh_tagbrowser_groupbox.addWidget(self.push_button_refresh_tagbrowser_qpushbutton)

        if not self.tagbrowser_icon_tool_is_running:
            self.push_button_refresh_tagbrowser_qpushbutton.setDisabled(True)

        self.autorun_checkbox_label = QLabel(" ")
        self.autorun_checkbox_label.setFont(font)
        self.autorun_checkbox_label.setAlignment(Qt.AlignCenter)
        wspan = (minwidth + 75)
        self.autorun_checkbox_label.setMinimumWidth(wspan)
        self.autorun_checkbox_label.setMaximumWidth(wspan)
        self.hlayout_refresh_tagbrowser_groupbox.addWidget(self.autorun_checkbox_label)

        self.autorun_checkbox = QCheckBox("Apply Icons at Calibre Startup")
        self.autorun_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to have this Tool apply your assigned icons at Calibre startup?<br><br>Restart Required for changes.")
        self.hlayout_refresh_tagbrowser_groupbox.addWidget(self.autorun_checkbox)

        if prefs['GUI_TOOLS_TAGBROWSER_ICONS_AUTO_SET_AT_STARTUP'] == unicode_type("True"):
            self.autorun_checkbox.setChecked(True)


        #~ --------------------------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        self.spacer_01_label = QLabel("")
        self.layout_top.addWidget(self.spacer_01_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.groupbox_matrix = QGroupBox('Matrix of Category Sources && Values')
        self.layout_top.addWidget(self.groupbox_matrix)
        #--------------------------------------------------
        self.vlayout_matrix_groupbox = QVBoxLayout()
        self.groupbox_matrix.setLayout(self.vlayout_matrix_groupbox)
        #--------------------------------------------------
        self.hlayout_matrix_groupbox = QHBoxLayout()
        self.hlayout_matrix_groupbox.setAlignment(Qt.AlignCenter)
        #--------------------------------------------------
        self.vlayout_matrix_groupbox.addLayout(self.hlayout_matrix_groupbox)
        #--------------------------------------------------
        self.icon_assignments_list = []
        self.init_matrix_table(1)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        self.layout_top.addWidget(self.bottom_buttonbox)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.setToolTip("<p style='white-space:wrap'>The listing columns will be resized based on their longest row contents for each column.")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,QDialogButtonBox.AcceptRole)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.setToolTip("<p style='white-space:wrap'>The listing columns will be resized to a fixed width regardless of their contents.")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,QDialogButtonBox.AcceptRole)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Export to Clipboard")
        self.push_button_copy_to_clipboard.setToolTip("<p style='white-space:wrap'>The Matrix in its entirety will be copied to the Clipboard in a tab-delimited format.  Paste Special into a Spreadsheet, or Paste into a Text Document.")
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_matrix_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()

        self.deoptimize_column_widths()

        self.block_events = True
        self.block_icon_file_events = False

        self.uc_category_combobox.currentIndexChanged.connect(self.event_current_text_changed_combo_category)
        self.uc_child_value_source_combobox.currentIndexChanged.connect(self.event_current_text_changed_combo_source)
        self.uc_child_source_value_combobox.currentIndexChanged.connect(self.event_current_text_changed_combo_value)
        self.uc_child_source_value_icon_combobox.currentIndexChanged.connect(self.event_current_text_changed_combo_value_icon_file)

        #~ self.uc_child_source_value_icon_combobox.activated.connect(self.event_source_value_icon_dropdown_arrow_clicked)

        self.uc_child_source_value_regex_combobox.currentIndexChanged.connect(self.event_current_text_changed_combo_value_regex)
        self.uc_child_source_value_regex_combobox.currentTextChanged.connect(self.event_current_text_changed_combo_value_regex)
        self.uc_child_source_value_regex_combobox.editTextChanged.connect(self.event_current_text_changed_combo_value_regex)

        self.autorun_checkbox.stateChanged.connect(self.event_autorun_checkbox_statechanged)

        self.block_events = False
        self.need_matrix_refresh = False
        self.need_values_combobox_refresh_only = False
        self.current_category = None
        self.current_source = None
        self.current_value = None
        self.current_icon_file = None

        self.custom_columns_metadata_dict = self.maingui.current_db.field_metadata.custom_field_metadata()  #for tag-like ampersand-separated like authors...shown in author-sort manner in tagbrowser...

        self.source_default_icon_dict = {}
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_current_text_changed_combo_category(self,event):
        msg = ""
        self.change_label_message(msg)
        if self.block_events:
            return
        self.current_category = self.uc_category_combobox.currentText()
        if self.current_category.startswith("@"):
            self.is_ampersand_uc = True
        else:
            self.is_ampersand_uc = False

        self.need_matrix_refresh = True
        self.need_values_combobox_refresh_only = False
        self.current_source = None
        self.refresh_matrix_control()
        self.save_preferences()

        c = self.uc_child_source_value_combobox.count()
        if c == 1:
            msg = as_unicode(c) + " Current Value Available for Assignment"
        else:
            msg = as_unicode(c) + " Current Values Available for Assignment"
        self.change_label_message(msg)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_current_text_changed_combo_source(self,event):
        msg = ""
        self.change_label_message(msg)

        if self.uc_child_value_source_combobox.count() == 0:
            return

        self.current_category = self.uc_category_combobox.currentText()
        self.current_source = self.uc_child_value_source_combobox.currentText()
        value = "default"
        icon_file_key = self.create_icon_file_key(self.current_source,value)
        icon_file = self.get_icon_file_name_for_current_value(icon_file_key)
        if icon_file:
            if os.path.isfile(icon_file):
                pixmap = self.create_pixmap_from_icon_file(icon_file)
                if pixmap:
                    self.uc_child_source_value_regex_spacer_02_label.setPixmap(pixmap)    #default icon image for current Source
                    self.source_default_icon_dict[self.current_source] = icon_file
                else:
                    self.uc_child_source_value_regex_spacer_02_label.setPixmap(self.default_nothingfound)
            else:
                self.uc_child_source_value_regex_spacer_02_label.setPixmap(self.default_nothingfound)
        else:
            self.uc_child_source_value_regex_spacer_02_label.setPixmap(self.default_nothingfound)
        self.uc_child_source_value_regex_spacer_02_label.repaint()

        if self.block_events:
            return

        self.current_source = self.uc_child_value_source_combobox.currentText()
        self.need_values_combobox_refresh_only = True
        self.refresh_matrix_control()

        c = self.uc_child_source_value_combobox.count()
        if c == 1:
            msg = as_unicode(c) + " Current Value Available for Assignment"
        else:
            msg = as_unicode(c) + " Current Values Available for Assignment"
        self.change_label_message(msg)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_current_text_changed_combo_value(self,event):
        msg = ""
        self.change_label_message(msg)

        if self.uc_child_source_value_combobox.count() == 0:
            return

        self.current_value = self.uc_child_source_value_combobox.currentText()
        self.current_icon_file = self.get_already_assigned_icon_file_for_value(self.current_source,self.current_value)
        if self.current_icon_file:
            self.block_icon_file_events = False
            self.event_current_text_changed_combo_value_icon_file(event=None,icon_file=self.current_icon_file)
        else:
            self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_nothingfound)
            self.uc_child_source_value_regex_spacer_03_label.repaint()
            self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_current_text_changed_combo_value_icon_file(self,event,icon_file=None):

        if self.block_icon_file_events:  #used by icon removal functions...
            return

        if not icon_file:
            self.current_icon_file = self.uc_child_source_value_icon_combobox.currentText()

        if self.current_icon_file:
            self.uc_child_source_value_icon_combobox.setEditText(self.current_icon_file)
            icon_image = self.create_icon_image_from_file_path(self.current_icon_file)
            sizes = icon_image.availableSizes()
            if sizes:
                if len(sizes) == 0:
                    pixmap = self.create_icon_image_from_file_path(self.current_icon_file,type="pixmap")
                    #~ if DEBUG: print("a1: ", as_unicode(type(pixmap)))  # <class 'qt.coreGui.QPixmap'>
                else:
                    pixmap = icon_image.pixmap(icon_image.availableSizes()[0])
                    #~ if DEBUG: print("b1: ", as_unicode(type(pixmap))) # <class 'qt.coreGui.QPixmap'>
            else:
                pixmap = self.create_icon_image_from_file_path(self.current_icon_file,type="pixmap")
                #~ if DEBUG: print("c1: ", as_unicode(type(pixmap)))   # <class 'qt.coreGui.QPixmap'>
            self.uc_child_source_value_regex_spacer_03_label.setPixmap(pixmap)
            self.uc_child_source_value_regex_spacer_03_label.repaint()
            del pixmap
            del icon_image
        else:
            self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_nothingfound)
            self.uc_child_source_value_regex_spacer_03_label.repaint()
            self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
    #-----------------------------------------------------
    #~ def event_source_value_icon_dropdown_arrow_clicked(self,event=None):
        #~ self.uc_child_source_value_icon_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        #~ self.uc_child_source_value_icon_combobox.showPopup()
    #-----------------------------------------------------
    def event_current_text_changed_combo_value_regex(self,event):
        if self.block_events:
            return
        self.current_regex = self.uc_child_source_value_regex_combobox.currentText()
        try:
            p = re.compile(self.current_regex)
            del p
            msg = ""
            self.change_label_message(msg)
        except Exception as e:
            if DEBUG: print("ERROR:  p = re.compile()",as_unicode(e))
            msg = "Invalid RE: " +  as_unicode(e)
            self.change_label_message(msg)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_autorun_checkbox_statechanged(self,event):
        self.save_preferences()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def execute_assignment_for_single_value(self,domatrixrefresh=None):
        if self.block_events:
            return

        if not domatrixrefresh:
            domatrixrefresh = True

        self.current_category = self.uc_category_combobox.currentText()
        self.current_source = self.uc_child_value_source_combobox.currentText()
        self.current_value = self.uc_child_source_value_combobox.currentText()
        self.current_icon_file = self.uc_child_source_value_icon_combobox.currentText()

        if not self.current_category:
            return False
        elif not self.current_source:
            return False
        elif not self.current_value:
            return False
        elif not self.current_icon_file:
            return False

        if not os.path.isfile(self.current_icon_file):
            msg = "Error in Assign Single Value: Icon File Does Not Exist"
            self.change_label_message(msg)
            return False

        new_path = self.save_icon_file_into_plugin_directory(self.current_source,self.current_value,self.current_icon_file)
        self.current_icon_file = new_path
        self.event_current_text_changed_combo_value(event=None)

        msg = "Assignment Applied for Single Value"
        self.change_label_message(msg)

        if domatrixrefresh:
            self.refresh_entire_matrix()

        return True
    #-----------------------------------------------------
    #-----------------------------------------------------
    def execute_assignment_for_all_current_values(self):
        if self.block_events:
            return

        issuccess = self.execute_assignment_for_single_value(domatrixrefresh=False)
        if not issuccess:
            return

        for row in self.uc_child_source_value_list:   #just filtered above...
            filtered_value = row
            new_path = self.save_icon_file_into_plugin_directory(self.current_source,filtered_value,self.current_icon_file)
        #END FOR

        n = len(self.uc_child_source_value_list)

        msg = as_unicode(n) + " Assignments Applied to Filtered Values"
        self.change_label_message(msg)

        self.refresh_entire_matrix()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def execute_filter_using_regex(self):
        if self.block_events:
            return

        if not self.current_category:
            return
        elif not self.current_source:
            return
        #~ elif not self.current_value:   #all rows previously filtered away...
            #~ return

        self.current_regex = self.uc_child_source_value_regex_combobox.currentText()

        try:
            p = re.compile(self.current_regex)
            del p
            msg = ""
            self.change_label_message(msg)
        except Exception as e:
            if DEBUG: print("ERROR:  p = re.compile()",as_unicode(e))
            msg = "Invalid RE: " +  as_unicode(e)
            self.change_label_message(msg)
            return

        if (not self.current_regex) or (self.current_regex == ""):
            msg = "No Regular Expression Specified"
            self.change_label_message(msg)

        if not self.current_regex in self.uc_source_value_regex_list:
            self.uc_source_value_regex_list.append(self.current_regex)
            s = as_unicode(self.uc_source_value_regex_list)
            self.uc_source_value_regex_list = list(set(self.uc_source_value_regex_list))
            self.uc_source_value_regex_list.sort()
            prefs['GUI_TOOLS_TAGBROWSER_ICONS_REGEX_COMPLETER_LIST'] = unicode_type(s)
            prefs
            self.block_events = True
            self.uc_child_source_value_regex_combobox.clear()
            for r in self.uc_source_value_regex_list:
                self.uc_child_source_value_regex_combobox.addItem(r)
            #END FOR
            self.uc_child_source_value_regex_combobox.setEditText(self.current_regex)
            self.block_events = False

        filtered_list = []
        match = None
        self.current_value = None

        try:
            p = re.compile(self.current_regex, re.IGNORECASE)
            for row in self.icon_assignments_list:
                category,source,value,icon_file = row
                if source == self.current_source:
                    match = p.search(value)
                    if match:
                        if not self.current_value:
                            self.current_value = value
                        filtered_list.append(row)
            #END FOR
        except Exception as e:
            msg = "Invalid RE: " +  as_unicode(e)
            self.change_label_message(msg)
            self.current_regex = None
            self.uc_child_source_value_regex_combobox.setEditText(self.current_regex)
            return

        self.block_events = True

        if len(filtered_list) == 0:
            self.uc_child_source_value_combobox.clear()
            self.uc_child_source_value_combobox.setCurrentIndex(-1)
            self.uc_child_source_value_icon_combobox.clear()
            self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
        else:
            filtered_list.sort()
            self.uc_child_source_value_list[:] = []
            self.uc_child_source_value_combobox.clear()
            for row in filtered_list:
                category,source,value,icon_file = row
                self.uc_child_source_value_combobox.addItem(value)
                self.uc_child_source_value_list.append(value)
            #END FOR
            self.current_source = self.uc_child_value_source_combobox.currentText()
            self.uc_child_source_value_combobox.setCurrentIndex(0)
            self.current_value = self.uc_child_source_value_combobox.currentText()
            self.refresh_combo_boxes(dosources=False,dovalues=False,doicons=True,data=filtered_list)
            self.block_events = False
            self.event_current_text_changed_combo_value(event=None)

        del filtered_list
        del match

        self.block_events = False

        n = self.uc_child_source_value_combobox.count()
        if n == 1:
            msg = as_unicode(n) + " Current Value remains after Filtering"
        else:
            msg = as_unicode(n) + " Current Values remain after Filtering"
        self.change_label_message(msg)

        return True
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_icon_assignments_list(self):

        del self.icon_assignments_list
        self.icon_assignments_list = []

        #~ ----------------
        if len(self.ifc_dict) != len(self.state_map_dict):   #  sub-categories are not in self.ifc_dict...
            if len(self.state_map_dict) > len(self.ifc_dict):
                pass
            else:
                if DEBUG: print("SEVERE ERROR 1a: Terminating...")
                if DEBUG: print("len(self.ifc_dict): ", as_unicode(len(self.ifc_dict)), " !=   len(self.state_map_dict: ", as_unicode(len(self.state_map_dict)) )
                if DEBUG:
                    for k,v in iteritems(self.ifc_dict):
                        print("for k,v in iteritems(self.ifc_dict): ", as_unicode(k),as_unicode(v))
                    for k,v in iteritems(self.state_map_dict):
                        if not k in self.ifc_dict:
                            print("ERROR: key in self.state_map_dict does not exist in  self.ifc_dict: ", k)
                return
        if len(self.ifc_named_path_dict) != len(self.state_map_dict):   #  sub-categories are not in self.ifc_dict...
            if len(self.state_map_dict) > len(self.ifc_named_path_dict):
                pass
            else:
                if DEBUG: print("SEVERE ERROR 1b: Terminating...")
                if DEBUG: print("len(self.ifc_named_path_dict): ", as_unicode(len(self.ifc_named_path_dict)), " !=   len(self.state_map_dict: ", as_unicode(len(self.state_map_dict)) )
                return
        #~ ----------------
        if not self.current_category in self.category_list:
            if DEBUG: print("SEVERE ERROR 2: Terminating...")
            return
        #~ ----------------

        if self.need_values_combobox_refresh_only:
            self.refresh_values_combobox_only()
        else:
            self.refresh_entire_icon_assignments_list()
            self.icon_assignments_list.sort()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_combo_boxes(self,dosources=False,dovalues=False,doicons=False,data=None):

        if not data:
            data = self.icon_assignments_list

        if dosources:
            self.uc_child_value_source_list[:] = []
        if dovalues:
            self.uc_child_source_value_list[:] = []
        if doicons:
            self.uc_source_value_icons_list[:] = []

        for row in data:
            category,source,value,icon_file = row
            if dosources:
                if not source in self.uc_child_value_source_list:
                    if source is not None:
                        self.uc_child_value_source_list.append(source)
            if dovalues:
                if not value in self.uc_child_source_value_list:
                    if value is not None:
                        self.uc_child_source_value_list.append(value)
            if doicons:
                if not icon_file in self.uc_source_value_icons_list:
                    if icon_file is not None:
                        self.uc_source_value_icons_list.append(icon_file)
        #END FOR

        if dosources:
            self.uc_child_value_source_list.sort()
            self.uc_child_value_source_combobox.clear()
            for source in self.uc_child_value_source_list:
                if source.startswith("#"):
                    self.uc_child_value_source_combobox.addItem(source)
            #END FOR
            self.uc_child_value_source_combobox.setCurrentIndex(0)
            self.current_source = self.uc_child_value_source_combobox.currentText()
            dummy = self.get_source_default_icon_file(self.current_source)  #builds dict too

        if dovalues:
            self.uc_child_source_value_list.sort()
            self.uc_child_source_value_combobox.clear()
            for value in self.uc_child_source_value_list:
                self.uc_child_source_value_combobox.addItem(value)
            #END FOR
            self.uc_child_source_value_combobox.setCurrentIndex(0)
            self.current_value = self.uc_child_source_value_combobox.currentText()

        if doicons:
            self.uc_source_value_icons_list.sort()
            self.uc_child_source_value_icon_combobox.clear()
            for icon_file in self.uc_source_value_icons_list:
                self.uc_child_source_value_icon_combobox.addItem(icon_file)
            #END FOR
            self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
            icon_file = self.get_already_assigned_icon_file_for_value(source,value)
            if icon_file:
                if not icon_file in self.uc_source_value_icons_list:
                    self.uc_child_source_value_icon_combobox.addItem(icon_file)
                    self.uc_child_source_value_icon_combobox.model().sort(0)
                    self.uc_source_value_icons_list.append(icon_file)
                self.uc_child_source_value_icon_combobox.setEditText(icon_file)
                self.current_icon_file = icon_file
                pixmap = self.create_pixmap_from_icon_file(icon_file)
                self.uc_child_source_value_regex_spacer_03_label.setPixmap(pixmap)
                self.uc_child_source_value_regex_spacer_03_label.repaint()
                del pixmap
            else:
                self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
                self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_nothingfound)
                self.uc_child_source_value_regex_spacer_03_label.repaint()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def create_pixmap_from_icon_file(self,icon_file):
        pixmap = None
        icon_image = self.create_icon_image_from_file_path(icon_file)
        sizes = icon_image.availableSizes()
        if sizes:
            if len(sizes) == 0:
                pixmap = self.create_icon_image_from_file_path(icon_file,type="pixmap")
                #~ if DEBUG: print("a: ", as_unicode(type(pixmap)))  # <class 'qt.coreGui.QPixmap'>
            else:
                pixmap = icon_image.pixmap(icon_image.availableSizes()[0])
                #~ if DEBUG: print("b: ", as_unicode(type(pixmap))) # <class 'qt.coreGui.QPixmap'>
        else:
            pixmap = self.create_icon_image_from_file_path(icon_file,type="pixmap")
            #~ if DEBUG: print("c: ", as_unicode(type(pixmap)))   # <class 'qt.coreGui.QPixmap'>
        del icon_image
        del sizes
        return pixmap
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_matrix_control(self):

        if self.need_values_combobox_refresh_only:
            self.refresh_values_combobox_only()
            self.need_values_combobox_refresh_only = False
        else:
            self.block_events = True
            self.refresh_icon_assignments_list()
            self.refresh_combo_boxes(dosources=True,dovalues=False,doicons=False,data=None)
            self.refresh_values_combobox_only()
            self.refresh_matrix_table()
            self.need_matrix_refresh = False
            self.need_values_combobox_refresh_only = False
            self.block_events = False

        self.save_preferences()
        self.save_tagbrowser_icons_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def init_matrix_table(self,n_rows):

        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("Category")
        column_label_list.append("Source")
        column_label_list.append("Value")
        column_label_list.append("Assigned Icon File")
        column_label_list.append("Icon")

        #--------------------------------------------------
        self.matrix = QTableWidget(n_rows,5)
        #--------------------------------------------------

        self.matrix.setSortingEnabled(False)

        self.matrix.setHorizontalHeaderLabels(column_label_list)

        self.matrix.setColumnWidth(0, 150)
        self.matrix.setColumnWidth(1, 150)
        self.matrix.setColumnWidth(2, 150)
        self.matrix.setColumnWidth(3, 150)
        self.matrix.setColumnWidth(4,   50)

        #--------------------------------------------------
        self.hlayout_matrix_groupbox.addWidget(self.matrix)

        self.resize_all_columns()

        t = "<p style='white-space:wrap'>The Current Path for your user-assigned icon files is: <i>" + self.tagbrowser_icon_file_dir + "</i>"
        t = t + "<br><br>You <b>may not manually add</b> any files whatsoever to that directory, or Calibre failures could occur.  Delete, but do <b>not</b> add any icon files manually.  Only add new icons using this GUI Tool.  Otherwise, bad things could happen. "
        t = t + "<br><br>When you back up your Calibre Libraries, it would also be a good idea to back up Calibre's entire user-specific configuration directory, which is: <i>" + config_dir + "</i>"

        self.matrix.setToolTip(t)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_matrix_table(self):

        try:
            self.matrix.setDisabled(True)
            self.matrix.setHidden(True)
            self.hlayout_matrix_groupbox.removeWidget(self.matrix)
            self.matrix.close()
            self.matrix.destroy(True,True)
            self.matrix = None
            del self.matrix
        except Exception as e:
            if DEBUG: print("Exception [1] in self.matrix.destroy() and del self.matrix: ", as_unicode(e))

        n_rows = len(self.icon_assignments_list)

        #-------------------------------
        self.init_matrix_table(n_rows)
        #-------------------------------

        self.matrix.repaint()

        self.matrix.setSortingEnabled(False)

        r = 0
        for row in self.icon_assignments_list:
            try:
                #~ if DEBUG: print("refresh_matrix_table row: ", as_unicode(row))  #refresh_matrix_table row:  (u'@MyColors', u'#mytaglike', u'yellow', None)
                #---------------------------
                #---------------------------
                category,source,value,icon_file = row
                #---------------------------
                #---------------------------
                if icon_file:
                    icon = self.create_icon_image_from_file_path(icon_file,type="icon")
                    if not icon:
                        icon = QIcon(self.default_nothingfound)
                else:
                    icon = QIcon(self.default_nothingfound)

                if icon_file:
                    p = as_unicode(icon_file)
                    n = p.find("tagbrowser_icons")
                    icon_file = p[n: ]  #shorten what is displayed to what is useful
                    icon_file = icon_file.replace("tagbrowser_icons","")
                    icon_file = icon_file[1: ]

                #---------------------------
                #---------------------------
                category_ = QTableWidgetItem(category)
                source_ = QTableWidgetItem(source)
                value_ = QTableWidgetItem(value)
                icon_file_ = QTableWidgetItem(icon_file)
                icon_ = QTableWidgetItem(icon," ")

                #---------------------------
                #---------------------------
                self.matrix.setItem(r,0,category_)
                self.matrix.setItem(r,1,source_)
                self.matrix.setItem(r,2,value_)
                self.matrix.setItem(r,3,icon_file_)
                self.matrix.setItem(r,4,icon_)
                #--------------------------------------
                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("=========>>>>>>>>Exception in refresh_matrix_table():", as_unicode(e))
                break
        #END FOR

        self.n_total_rows = r

        self.matrix.setSortingEnabled(True)

        self.resize_all_columns()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_entire_icon_assignments_list(self):

        source_dict = {}

        if self.is_ampersand_uc:
            use_source_dict = True
            dummy, state_map = self.tv.get_state()
            #~ for category,v_dict in state_map.iteritems():
            for category,v_dict in iteritems(state_map):
                if category == self.current_category:
                    if "." in category:   # this tool does *not* handle sub-categories...
                        if DEBUG: print("Sub-Category being skipped: ", category)
                        return
                    #~ if DEBUG: print("state_map: ", as_unicode(category), ">>>", as_unicode(v_dict))  #~ state_map:  @MyColors >>> {(u'black', u'#mytaglike'): 0, (u'yellow', u'#mytaglike'): 0, (u'blue', u'#mytaglike'): 0, (u'green', u'#mytaglike'): 0}
                    #~ for k,v in v_dict.iteritems():
                    for k,v in iteritems(v_dict):
                        k_list = list(k)
                        value = k_list[0]
                        source = k_list[1]
                        if source.startswith("#"):
                            source_dict[value] = source
                            if source in self.custom_columns_metadata_dict:
                                data_dict = self.custom_columns_metadata_dict[source]
                                is_names = "is_names': True"
                                if is_names in as_unicode(data_dict):            #......'display': {u'is_names': True..........
                                    authors_list = []
                                    authors_list.append(value)    #== string_to_authors(authors_string)
                                    author_sort = self.guidb.new_api.author_sort_from_authors(authors_list, key_func=lambda x: x)
                                    source_dict[author_sort] = source
                                    del authors_list
                                    #~ if DEBUG: print("source: ", as_unicode(source), "  value of values' author-sort: ", as_unicode(author_sort))      #~ source:  #real_authors   value of values' author-sort:  Pottinger, Julia
                                else:
                                    pass
                                del data_dict
                                dummy = self.get_source_default_icon_file(source)  #builds dict too
                            else:
                                if DEBUG: print("Source Custom Column is NOT in the data_dict...Error: ", source)
                        else:
                            pass  # Standard Calibre columns such as Tags, Series, etc. whose tag browser icons Calibre deals with itself.

                        #~ if DEBUG: print("source: ", as_unicode(source), "  value: ", as_unicode(value))

                    #END FOR
                else:
                    pass
            #END FOR
            state_map = None
            dummy = None
            k_list = None
            del state_map
            del dummy
            del k_list
        else:
            cc_source = self.current_category
            use_source_dict = False
            dummy = self.get_source_default_icon_file(cc_source)  #builds dict too

        if "." in self.current_category:   # this tool does *not* handle sub-categories...
            if DEBUG: print("Sub-Category being skipped: ", category)
            return

        try:
            ifc = self.tvm.index_for_category(self.current_category)  #@MyColors   or  #abc_hierarchy
            named_path = self.tvm.named_path_for_index(ifc)
            index = self.tvm.index_for_named_path(named_path)     # [u'@MyColors:MyColors']
            if index is None:
                if DEBUG: print("index is None:  self.current_category: ", self.current_category, "  named_path: ", as_unicode(named_path) )
                return
            if not index.isValid():
                if DEBUG: print("SEVERE ERROR not index.isValid(): Terminating...")
                return
        except Exception as e:
            if DEBUG: print("Exception in  refresh_entire_icon_assignments_list: ", as_unicode(e))
            if DEBUG: print("self.current_category: ", self.current_category)
            return

        for r in range(self.tvm.rowCount(index)):
            child = index.child(r, 0)
            data = self.tvm.data(child,role=Qt.DisplayRole)
            value = data
            #~ if DEBUG: print("display data: ", as_unicode(data))
            if use_source_dict:
                if value in source_dict:
                    source = source_dict[value]
                else:
                    continue  # source = "Standard Column"
            else:
                source = cc_source
            icon_file = self.get_already_assigned_icon_file_for_value(source,value)
            if not icon_file:
                if source in self.source_default_icon_dict:
                    icon_file = self.source_default_icon_dict[source]
            row = self.current_category,source,value,icon_file                                      # in matrix table:   category,source,value,icon_file = row
            self.icon_assignments_list.append(row)
        #END FOR
        self.icon_assignments_list.sort()
        try:
            self.tv.expand_node_and_children(index)  # Calibre5.3+
        except:
            try:
                self.tv.expand_node_and_descendants(index)  # Calibre4.6 - 5.2
            except Exception as e:
                if DEBUG: print("TagBrowser Tool: both  expand_node_and_children(index) AND expand_node_and_descendants(index) failed...", as_unicode(e))
                pass
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_entire_matrix_manually(self):
        if self.block_events:
            return
        self.refresh_entire_matrix()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_entire_matrix(self):
        if not self.current_category:
            return

        if not self.block_events:
            self.block_events = True
            must_unblock = True
        else:
            must_unblock = False

        del self.icon_assignments_list
        self.icon_assignments_list = []
        self.refresh_entire_icon_assignments_list()
        self.refresh_matrix_table()

        if must_unblock:
            self.block_events = False
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_values_combobox_only(self):
        # icon_file is a function of value, so it will always be refreshed when value is refreshed.

        self.current_source = self.uc_child_value_source_combobox.currentText()

        if not self.current_source:  #whenever the selected category changes...
            if len(self.icon_assignments_list) > 0:
                row = self.icon_assignments_list[0]
                category,source,value,icon_file = row
                self.current_source = source
            else:
                return  #if a category (e.g. #fast) has no values whatsoever because it is a new custom column that has never been used...

        current_values_for_current_source_list = []
        for row in self.icon_assignments_list:
            category,source,value,icon_file = row
            if source == self.current_source:
                current_values_for_current_source_list.append(row)
        #END FOR
        self.refresh_combo_boxes(dosources=False,dovalues=True,doicons=True,data=current_values_for_current_source_list)
        self.event_current_text_changed_combo_value(event=None)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_already_assigned_icon_file_for_value(self,source,value):
        #~ also used by ui.py
        icon_file_key = self.create_icon_file_key(source,value)
        icon_file = self.get_icon_file_name_for_current_value(icon_file_key)
        if os.path.isfile(icon_file):
            return icon_file
        else:
            return None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def create_icon_file_key(self,source,value):
        #~ also used by ui.py
        icon_file_key = "_" + source + "_" + value + "_"
        icon_file_key = sanitize_file_name_unicode(icon_file_key)
        return icon_file_key
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_icon_file_name_for_current_value(self,icon_file_key):
        #~ also used by ui.py
        fname = icon_file_key + ".png"
        icon_file = os.path.join(self.tagbrowser_icon_file_dir,fname)
        icon_file = icon_file.replace(os.sep, '/')
        return icon_file
    #-----------------------------------------------------
    #-----------------------------------------------------
    def select_icon_file(self):
        if not self.current_category:
            return

        path = self.select_value_icon_file_dialog()
        if not path:
            self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
            return
        icon_file = path
        if not icon_file in self.uc_source_value_icons_list:
            self.uc_child_source_value_icon_combobox.addItem(icon_file)
            self.uc_child_source_value_icon_combobox.model().sort(0)
            self.uc_source_value_icons_list.append(icon_file)
        self.uc_child_source_value_icon_combobox.setEditText(icon_file)
        self.current_icon_file = icon_file
        try:
            icon_image = self.create_icon_image_from_file_path(self.current_icon_file,type="icon")
            #~ if DEBUG: print("icon_image: ", as_unicode(type(icon_image)))  # <class 'qt.coreGui.QIcon'>
            try:
                sizes = icon_image.availableSizes()
                if sizes:
                    if len(sizes) == 0:
                        pixmap = self.create_icon_image_from_file_path(self.current_icon_file,type="pixmap")
                        #~ if DEBUG: print("a1: ", as_unicode(type(pixmap)))  # <class 'qt.coreGui.QPixmap'>
                    else:
                        pixmap = icon_image.pixmap(icon_image.availableSizes()[0])
                        #~ if DEBUG: print("b1: ", as_unicode(type(pixmap))) # <class 'qt.coreGui.QPixmap'>
                else:
                    pixmap = self.create_icon_image_from_file_path(self.current_icon_file,type="pixmap")
                    #~ if DEBUG: print("c1: ", as_unicode(type(pixmap)))   # <class 'qt.coreGui.QPixmap'>
            except Exception as e:
                if DEBUG: print("========ERROR: select_icon_file 0: ", self.current_icon_file, as_unicode(e))
                self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_filenotfound)
                self.uc_child_source_value_regex_spacer_03_label.repaint()
                return

            self.uc_child_source_value_regex_spacer_03_label.setPixmap(pixmap)
            self.uc_child_source_value_regex_spacer_03_label.repaint()

            del pixmap
            del icon_image

        except Exception as e:
            if DEBUG: print("=====ERROR: select_icon_file 1: ", self.current_icon_file, as_unicode(e))
            self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_filenotfound)
            self.uc_child_source_value_regex_spacer_03_label.repaint()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def select_value_icon_file_dialog(self):

        title = "Select Icon File to Assign to Value(s)"
        default_dir = prefs['GUI_TOOLS_TAGBROWSER_ICONS_DEFAULT_SEARCH_DIRECTORY']
        import_tuple = QFileDialog.getOpenFileName(None,"",default_dir,(IMAGE_TYPES_SUPPORTED) )
        if not import_tuple:
            return None
        path, dummy = import_tuple
        if not path:
            return None
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        if os.path.isfile(path):
            default_dir,tail = os.path.split(path)
            prefs['GUI_TOOLS_TAGBROWSER_ICONS_DEFAULT_SEARCH_DIRECTORY'] = unicode_type(default_dir)
            prefs
            msg = default_dir
            self.change_label_message(msg)
            return path
        else:
            return None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_icon_file_into_plugin_directory(self,source,value,icon_file):
        #user manually selects the source icon file to be used here
        selected_path = self.current_icon_file
        dir = self.tagbrowser_icon_file_dir
        icon_file_key = self.create_icon_file_key(source,value)
        fname = icon_file_key + '.png'
        path = os.path.join(dir,fname)
        new_path = path.replace(os.sep,"/")
        icon_image = self.create_icon_image_from_file_path(selected_path,type="pixmap")
        self.save_icon_image_as_file_into_plugin_directory(icon_image,new_path)
        return new_path
    #-----------------------------------------------------
    #-----------------------------------------------------
    def create_icon_image_from_file_path(self,path,type="icon"):
        #~ also used by ui.py
        path = path.replace(os.sep,"/")
        #~ if DEBUG: print("create_icon_image_from_file_path", type, "  ", path)
        if type == "icon":
            p = QIcon(path).pixmap(QSize(20, 20))  # same size as Tag Browser uses...
            icon_image = QIcon(p)                            # QIcon object
            del p
        elif type == "pixmap":
            icon_image = QIcon(path).pixmap(QSize(20, 20))  # QPixMap object
            icon_image.save(path,format = 'PNG',quality = -1)
        else:
            icon_image = None
        return icon_image
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_icon_image_as_file_into_plugin_directory(self,icon_image,path):
        icon_image.save(path)
        del icon_image
        msg = "Icon Assignment Saved"
        self.change_label_message(msg)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def ensure_icon_directory_exists(self):
        #~ also used by ui.py
        dir = os.path.join(config_dir,'plugins')
        dir =  os.path.join(dir, 'tagbrowser_icons')
        if not os.path.exists(dir):
            os.makedirs(dir)
            if DEBUG: print("directory created: ", dir)
        self.tagbrowser_icon_file_dir = dir
    #-----------------------------------------------------
    #-----------------------------------------------------
    def change_label_message(self,msg):
        self.uc_child_source_value_filter_spacer_03_label.setText(msg)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def remove_single_value(self):
        if self.block_events:
            return

        if not self.current_category:
            return
        elif not self.current_source:
            return
        elif not self.current_value:
            return
        elif not self.current_icon_file:
            return

        self.block_events = True
        self.block_icon_file_events = True

        if os.path.isfile(self.current_icon_file):
            os.remove(self.current_icon_file)
        self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
        self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_nothingfound)
        self.uc_child_source_value_regex_spacer_03_label.repaint()

        self.refresh_entire_matrix()

        self.block_icon_file_events = False
        self.block_events = False
    #-----------------------------------------------------
    #-----------------------------------------------------
    def remove_current_values(self):
        if self.block_events:
            return

        if not self.current_category:
            self.current_category = self.uc_category_combobox.currentText()
        elif not self.current_source:
            self.current_source = self.uc_child_value_source_combobox.currentText()
        elif not self.current_value:
            self.current_value = self.uc_child_source_value_combobox.currentText()

        if not self.current_category:
            return
        elif not self.current_source:
            return
        elif not self.current_value:
            return

        self.block_events = True
        self.block_icon_file_events = True

        if self.current_icon_file:
            if os.path.isfile(self.current_icon_file):
                os.remove(self.current_icon_file)

        source = self.current_source
        count = self.uc_child_source_value_combobox.count()
        for i in range(0,count):
            value = self.uc_child_source_value_combobox.itemText(i)
            if value:
                icon_file = self.get_already_assigned_icon_file_for_value(source,value)
                if icon_file:
                    if os.path.isfile(icon_file):
                        os.remove(icon_file)
                        #~ if DEBUG: print("icon_file removed: ", icon_file)
        #END FOR

        self.uc_child_source_value_icon_combobox.clear()

        self.uc_child_source_value_icon_combobox.setCurrentIndex(-1)
        self.uc_child_source_value_regex_spacer_03_label.setPixmap(self.default_nothingfound)
        self.uc_child_source_value_regex_spacer_03_label.repaint()

        self.refresh_entire_matrix()

        self.block_icon_file_events = False
        self.block_events = False
    #-----------------------------------------------------
    #-----------------------------------------------------
    def recount_tagbrowser(self):
        if self.block_events:
            return
        if not self.tagbrowser_icon_tool_is_running:
            return
        self.tv.recount()
        self.save_preferences()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def resize_all_columns(self):
        self.deoptimize_column_widths()
        self.save_tagbrowser_icons_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.matrix.resizeColumnsToContents()
        self.save_tagbrowser_icons_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def deoptimize_column_widths(self):
        self.matrix.setColumnWidth(0, 150)
        self.matrix.setColumnWidth(1, 150)
        self.matrix.setColumnWidth(2, 400)
        self.matrix.setColumnWidth(3, 140)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def copy_matrix_to_clipboard(self):
        if self.block_events:
            return

        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.matrix.selectAll()

        s = ''
        s = s + "\t".join([as_unicode(self.matrix.horizontalHeaderItem(i).text()) for i in range(0, 4)])   # only copy 4 columns (5th is a pixmap)
        s = s +  '\n'
        for r in range(0, self.n_total_rows):
            for c in range(0, 4):
                try:
                    s = s + as_unicode(self.matrix.item(r,c).text()) + "\t"
                except AttributeError:
                    s = s + "\t"
            s = s[:-1] + "\n"        #eliminate last '\t'
        self.clip.setText(s)

        self.save_tagbrowser_icons_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_tagbrowser_icons_dialog_geometry(self):
        self.dialog_closing(None)
        self.save_preferences()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_preferences(self):

        if self.autorun_checkbox.isChecked():
            prefs['GUI_TOOLS_TAGBROWSER_ICONS_AUTO_SET_AT_STARTUP'] = unicode_type("True")
        else:
            prefs['GUI_TOOLS_TAGBROWSER_ICONS_AUTO_SET_AT_STARTUP'] = unicode_type("False")

        if not RE_ANYTHING in self.uc_source_value_regex_list:
            self.uc_source_value_regex_list.append(RE_ANYTHING)
        self.uc_source_value_regex_list = list(set(self.uc_source_value_regex_list))
        s = as_unicode(self.uc_source_value_regex_list)
        prefs['GUI_TOOLS_TAGBROWSER_ICONS_REGEX_COMPLETER_LIST'] = unicode_type(s)

        prefs
    #-----------------------------------------------------
    #-----------------------------------------------------
    def remove_current_regex(self):
        if not self.current_category:
            return
        elif not self.current_source:
            return

        del_regex = self.uc_child_source_value_regex_combobox.currentText()
        if del_regex is None or del_regex == "":
            msg = "No Regular Expression Specified to Remove"
            self.change_label_message(msg)
            return

        if del_regex == "":
            msg = "Regex of '' cannot be removed.  Sorry."
            self.change_label_message(msg)
            return

        if del_regex.strip() == RE_ANYTHING:
            msg = "Regex of '^.+$' cannot be removed.  Sorry."
            self.change_label_message(msg)
            return

        self.block_events = True

        new_list = []
        for r in self.uc_source_value_regex_list:
            if not r == del_regex:
                r = r.strip()
                if not r in new_list:
                    new_list.append(r)
        #END FOR
        new_list.sort()
        self.uc_source_value_regex_list[:] = []
        for r in new_list:
            self.uc_source_value_regex_list.append(r)
        #END FOR

        if not RE_ANYTHING in self.uc_source_value_regex_list:
            self.uc_source_value_regex_list.append(RE_ANYTHING)

        if not "" in self.uc_source_value_regex_list:
            self.uc_source_value_regex_list.append("")

        self.uc_source_value_regex_list.sort()

        s = as_unicode(self.uc_source_value_regex_list)
        prefs['GUI_TOOLS_TAGBROWSER_ICONS_REGEX_COMPLETER_LIST'] = unicode_type(s)
        prefs

        self.uc_child_source_value_regex_combobox.clear()
        for r in self.uc_source_value_regex_list:
            self.uc_child_source_value_regex_combobox.addItem(r)
        #END FOR

        self.uc_child_source_value_regex_combobox.setCurrentIndex(-1)

        self.current_regex = None

        self.block_events = False

        msg = "Regex Removed: " + del_regex
        self.change_label_message(msg)

        del del_regex
    #-----------------------------------------------------
    #-----------------------------------------------------
    def assign_source_default_icon(self):
        if self.block_events:
            return

        self.current_source = self.uc_child_value_source_combobox.currentText()
        if not self.current_source:
            return

        title = "Select Default Icon File to Assign to Source"
        default_dir = prefs['GUI_TOOLS_TAGBROWSER_ICONS_DEFAULT_SEARCH_DIRECTORY']
        import_tuple = QFileDialog.getOpenFileName(None,"",default_dir,(IMAGE_TYPES_SUPPORTED) )
        if not import_tuple:
            return None
        path, dummy = import_tuple
        if not path:
            return None
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        if os.path.isfile(path):
            pixmap = self.create_pixmap_from_icon_file(path)
            if pixmap:
                self.uc_child_source_value_regex_spacer_02_label.setPixmap(pixmap)    #default icon image for current Source
                self.uc_child_source_value_regex_spacer_02_label.repaint()
                dir = self.tagbrowser_icon_file_dir
                icon_file_key = self.create_icon_file_key(self.current_source,"default")
                fname = icon_file_key + '.png'
                path = os.path.join(dir,fname)
                path = path.replace(os.sep,"/")
                self.save_icon_image_as_file_into_plugin_directory(pixmap,path)
                msg = "Source's Default Icon Assigned"
                self.change_label_message(msg)
                self.refresh_entire_matrix()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def remove_source_default_icon(self):
        if self.block_events:
            return

        self.current_source = self.uc_child_value_source_combobox.currentText()
        if not self.current_source:
            return
        dir = self.tagbrowser_icon_file_dir
        icon_file_key = self.create_icon_file_key(self.current_source,"default")
        fname = icon_file_key + '.png'
        path = os.path.join(dir,fname)
        path = path.replace(os.sep,"/")
        if os.path.isfile(path):
            os.remove(path)
            self.uc_child_source_value_regex_spacer_02_label.setPixmap(self.default_nothingfound)
            self.uc_child_source_value_regex_spacer_02_label.repaint()
            msg = "Source's Default Icon Removed"
            self.change_label_message(msg)
            self.refresh_entire_matrix()
        else:
            msg = "Nothing to Remove"
            self.change_label_message(msg)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_source_default_icon_file(self,source):
        value = "default"
        self.source_default_icon_dict[source] = None
        icon_file_key = self.create_icon_file_key(source,value)
        icon_file = self.get_icon_file_name_for_current_value(icon_file_key)
        if icon_file:
            if os.path.isfile(icon_file):
                pixmap = self.create_pixmap_from_icon_file(icon_file)
                if pixmap:
                    self.source_default_icon_dict[source] = icon_file
                    return icon_file
                else:
                    return None
        else:
            return None
    #-----------------------------------------------------
    #-----------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#END OF tagbrowser_icons_dialog.py