# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2023 DaltonST'
__my_version__ = "1.0.201"  # New  "JS+:GUI Tool:  RIS Tags: Create Individual Custom Columns"

import copy

from qt.core import (Qt, QVBoxLayout, QWidget, QLineEdit, QGroupBox,
                                    QDialog, QPushButton, QFont, QScrollArea, QGridLayout,
                                    QCheckBox, QApplication, QHBoxLayout, QSize)

from calibre.constants import DEBUG
from calibre.gui2 import gprefs

from polyglot.builtins import iteritems

TEXT_DEFAULT_WIDTH =  700
TEXT_DEFAULT_HEIGHT = 700
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(300, 500)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        #~ if DEBUG: self.geom = None
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result=None):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class RISCustomColumnsSelectDialog(SizePersistedDialog):

    def __init__(self,guidb,created_already_list,tag_cc_mapping_dict,js_icon):

        parent = None
        unique_pref_name = 'Job_Spy:RISCustomColumnsSelectDialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.guidb = guidb
        self.created_already_set = set(created_already_list)
        self.tag_cc_mapping_dict = tag_cc_mapping_dict

        from calibre.gui2.ui import get_gui
        self.maingui = get_gui()

        self.setWindowFlags(Qt.Window | Qt.WindowTitleHint | Qt.WindowMinMaxButtonsHint)
        self.setWindowIcon(js_icon)
        title = "JS+:GUI Tool:  RIS Tags: Create Individual Custom Columns"
        self.setWindowTitle(title)
        self.setSizeGripEnabled(True)
        self.setModal(True)

        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(10)
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignLeft)
        self.setLayout(self.layout_top)

        self.layout_top.addSpacing(2)

        self.layout_top_pushbuttons = QHBoxLayout()
        self.layout_top_pushbuttons.setSpacing(0)
        self.layout_top_pushbuttons.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_top_pushbuttons)

        self.clear_all_checkboxes_pushbutton = QPushButton("Clear All", self)
        self.clear_all_checkboxes_pushbutton.clicked.connect(self.clear_all_checkboxes)
        self.clear_all_checkboxes_pushbutton.setMaximumWidth(300)
        self.layout_top_pushbuttons.addWidget(self.clear_all_checkboxes_pushbutton)

        self.layout_top_pushbuttons.addSpacing(25)

        self.check_all_checkboxes_pushbutton = QPushButton("Select All", self)
        self.check_all_checkboxes_pushbutton.clicked.connect(self.check_all_checkboxes)
        self.check_all_checkboxes_pushbutton.setMaximumWidth(300)
        self.layout_top_pushbuttons.addWidget(self.check_all_checkboxes_pushbutton)

        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignLeft)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(700,700)

        self.layout_top.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_top

        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_top.addWidget(self.scroll_widget)           # causes automatic reparenting of QWidget to the parent of self.layout_top, which is:  self .
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setSpacing(0)
        self.layout_frame.setAlignment(Qt.AlignCenter)

        self.scroll_widget.setLayout(self.layout_frame)        # causes automatic reparenting of any widget later added to self.layout_frame to the parent of self.layout_frame, which is:  QWidget .

        #-----------------------------------------------------
        self.custom_columns_groupbox = QGroupBox('Possible RIS Custom Columns to Create:')
        self.custom_columns_groupbox.setMaximumWidth(800)
        self.custom_columns_groupbox.setToolTip("<p style='white-space:wrap'>Select the Custom Columns that you wish to create.'")
        self.layout_frame.addWidget(self.custom_columns_groupbox)

        self.custom_columns_layout = QGridLayout()
        self.custom_columns_groupbox.setLayout(self.custom_columns_layout)
        #-----------------------------------------------------
        #-----------------------------------------------------
        font.setPointSize(8)
        font.setBold(True)
        #-----------------------------------------------------
        #-----------------------------------------------------
        if DEBUG: print("is_valid = self.create_custom_column_choices(font)")
        self.is_valid,self.error_msg = self.create_custom_column_choices(font)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------

        #-----------------------------------------------------
        #-----------------------------------------------------
        self.layout_frame.addSpacing(10)

        self.layout_save_pushbuttons = QHBoxLayout()
        self.layout_save_pushbuttons.setSpacing(0)
        self.layout_save_pushbuttons.setAlignment(Qt.AlignCenter)
        self.layout_frame.addLayout(self.layout_save_pushbuttons)

        self.exit_pushbutton = QPushButton("Save && Exit", self)
        self.exit_pushbutton.clicked.connect(self.exit_dialog)
        self.exit_pushbutton.setMaximumWidth(300)
        self.layout_save_pushbuttons.addWidget(self.exit_pushbutton)

        self.resize_dialog()

    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_custom_column_choices(self,font):

        self.qt_qcheckbox_objects_dict = {}  #  tag = qcb
        self.qt_qlineedit_objects_dict = {}     #  tag = qle

        r = 0

        try:
            for tag,cc in iteritems(self.tag_cc_mapping_dict):   #   {"A1": "#ris_first_authors", "A2": "#ris_secondary_authors",...}
                if DEBUG: print(tag,cc)
                if not cc.startswith("#"):
                    continue

                if not "ris_" in cc:
                    continue

                try:
                    template_qcheckbox = QCheckBox(tag)
                    template_qcheckbox.setTristate(False)
                    template_qcheckbox.setToolTip("<p style='white-space:wrap'>Create this Tag's specific RIS Custom Column?")
                except  Exception as e:
                    msg = "template_qcheckbox error: " + str(e)
                    if DEBUG: print(msg)
                    return False,msg

                try:
                    template_qlineedit = QLineEdit(self)
                    template_qlineedit.setFont(font)
                    template_qlineedit.setToolTip("<p style='white-space:wrap'>This is a possible RIS Custom Column that a specific RIS Tag may be mapped to upon updating metadata.")
                except  Exception as e:
                    msg = "template_qlineedit error: " + str(e)
                    if DEBUG: print(msg)
                    return False,msg

                qcb = template_qcheckbox
                if cc in self.created_already_set:  #  e.g.  "#ris_date"
                    qcb.setChecked(False)
                    qcb.setEnabled(False)
                else:
                    qcb.setChecked(True)
                    qcb.setEnabled(True)
                self.qt_qcheckbox_objects_dict[tag] = qcb

                qle = template_qlineedit
                qle.setText(cc)
                qle.setEnabled(False)  # user cannot change this value...

                self.qt_qlineedit_objects_dict[tag] = qle

                self.custom_columns_layout.addWidget(qcb,r,0)
                self.custom_columns_layout.addWidget(qle,r,1)

                r = r + 1
            #END FOR
        except  Exception as e:
            if DEBUG: print("create_custom_column_choices -- error: ", str(e))
            return False,str(e)

        return True,None
    #---------------------------------------------------------------------------------------------------------------------------------------
    def clear_all_checkboxes(self):
        for tag, qcb in iteritems(self.qt_qcheckbox_objects_dict):
            if qcb.isEnabled():
                qcb.setChecked(False)
                qcb.update()
        #END FOR
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def check_all_checkboxes(self):
        for tag, qcb in iteritems(self.qt_qcheckbox_objects_dict):
            if qcb.isEnabled():
                qcb.setChecked(True)
                qcb.update()
        #END FOR
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def return_tag_cc_list(self):

        selected_cc_list = []  # list of 2-tuples...

        if not self.is_valid:
            from calibre.gui2 import error_dialog
            msg = "There was a fatal error.  No custom columns will be created.  Reason: " + self.error_msg
            error_dialog(self.gui, _('JS+ GUI Tool'),_(msg), show=True)
            return selected_cc_list

        try:
            for tag, qcb in iteritems(self.qt_qcheckbox_objects_dict):
                if qcb.isChecked():
                    qle = self.qt_qlineedit_objects_dict[tag]
                    cc = qle.text()
                    row = tag,cc
                    selected_cc_list.append(row)
                    if DEBUG: print("return_tag_cc_list - Selected: ", tag, cc)
            #END FOR
        except Exception as e:
            from calibre.gui2 import error_dialog
            msg = "There was a fatal error in returning the selected custom columns to create.<br><br>No custom columns will be created.<br><br>Reason: " + self.error_msg
            error_dialog(self.gui, _('JS+ GUI Tool'),_(msg), show=True)
            selected_cc_list = []
            return selected_cc_list

        return selected_cc_list
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def exit_dialog(self):
        self.dialog_closing()
        #~ instead of closing it with self.close(), close it with self.accept().
        #~ Once the dialog.exec() event loop returns to ui.py, execute this dialog's attribute:   return_tag_cc_list()
        self.accept()
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------

#~ # end of ris_tags_custom_columns_select_dialog.py