# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.218"  #Notes Viewer: Enhancements for URI Links & Dark Mode

import re, sre_constants
from functools import partial

from qt.core import (QAction, QApplication, QButtonGroup, QCheckBox, QColor, QComboBox,
                                    QDialog, QDialogButtonBox, QFont, QFrame, QGridLayout, QGroupBox,
                                    QHBoxLayout, QIcon, QLabel, QLineEdit, QMargins, QModelIndex,
                                    QPushButton, QRadioButton,QScrollArea, QSize, QTextBrowser, QTextEdit,
                                    QTextOption, QTimer, QVBoxLayout, QWidget, Qt)

from calibre.constants import DEBUG, ismacos, iswindows
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import gprefs, error_dialog, info_dialog
from calibre.gui2.palette import windows_is_system_dark_mode_enabled, linux_is_system_dark_mode_enabled
from calibre.library.comments import comments_to_html

MD = 'md'
HTML = 'html'
PLAIN = 'plain'
MAX_AUTHOR_TITLE_LENGTH = 85
color_white = QColor("white")
tool_name = "Notes Viewer"
ellipsis = '…'
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(200, 200)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent, Qt.WindowSystemMenuHint|Qt.WindowMinimizeButtonHint )
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
class NotesViewerDialog(SizePersistedDialog):

    def __init__(self,maingui,parent,js_icon,prefs,return_from_notes_viewer_to_save_prefs):
        unique_pref_name = 'Job_Spy:notesviewer_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.maingui = maingui
        self.gui = maingui
        self.guidb = self.maingui.library_view.model().db
        self.original_guidb = self.guidb
        self.js_icon = js_icon
        self.nv_prefs = prefs
        self.prefs_set = set()
        self.return_from_notes_viewer_to_save_prefs = return_from_notes_viewer_to_save_prefs

        self.is_dark_mode = False
        if iswindows:
            if windows_is_system_dark_mode_enabled():
                self.is_dark_mode = True
        elif not iswindows and not ismacos:
            if linux_is_system_dark_mode_enabled():
                self.is_dark_mode = True

        self.custom_columns_metadata_dict = self.maingui.current_db.field_metadata.custom_field_metadata()
        self.last_current_column = prefs['GUI_TOOLS_CONVERT_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED']
        self.startup_column = self.last_current_column
        self.last_bookid = 0
        self.is_fatal_error = False
        self.original_long_text = ""
        errors = self.build_regular_expressions_for_guessing()
        if self.is_fatal_error:
            msg = "Fatal error in compiling Regular Expressions to guess at Plain/Markdown/HTML:\n\n" + str(errors)
            error_dialog(self.maingui, _(tool_name),_(msg), show=True)
            return None

        mytitle = 'JS+ GUI Tool:  Notes Viewer'
        self.setWindowTitle(_(mytitle))
        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint )
        self.setSizeGripEnabled(True)
        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(11)
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.setLayout(self.layout_frame)
        self.layout_frame.setAlignment(Qt.AlignCenter)

        self.setLayout(self.layout_frame)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignCenter)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(600,600)

        self.layout_frame.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_frame

        # NOTE: the self.scroll_area_frame.setWidget(self.scroll_widget) is at the end of the init() AFTER all children have been created and assigned to a layout...
        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_frame.addWidget(self.scroll_widget)            # causes automatic reparenting of QWidget to the parent of self.layout_frame, which is:  self .
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignCenter)
        #-----------------------------------------------------
        self.scroll_widget.setLayout(self.layout_top)                 # causes automatic reparenting of any widget later added below to the above parent
        #----------------------------------------
        self.layout_author = QHBoxLayout()
        self.layout_author.setSpacing(10)
        self.layout_author.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_author)

        font.setPointSize(9)

        self.move_library_view_index_up_pushbutton = QPushButton("&Up▲", self)  # 🞀🞂🞃🞁
        self.move_library_view_index_up_pushbutton.setMaximumWidth(40)
        self.move_library_view_index_up_pushbutton.setDefault(False)
        self.move_library_view_index_up_pushbutton.setFont(font)
        self.move_library_view_index_up_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to move the current Calibre Book Cursor up one row.")
        self.move_library_view_index_up_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.move_library_view_index_up_pushbutton.clicked.connect(self.move_library_view_index_up)
        self.layout_author.addWidget(self.move_library_view_index_up_pushbutton)

        font.setPointSize(11)

        self.author_label = QLabel("Author:")
        self.author_label.setFont(font)
        self.author_label.setMaximumWidth(650)
        self.author_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_author.addWidget(self.author_label)

        font.setPointSize(9)
        #--------------------------------------------------

        self.layout_title = QHBoxLayout()
        self.layout_title.setSpacing(10)
        self.layout_title.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_title)

        self.move_library_view_index_down_pushbutton = QPushButton("&Dn▼", self)
        self.move_library_view_index_down_pushbutton.setFont(font)
        self.move_library_view_index_down_pushbutton.setMaximumWidth(40)
        self.move_library_view_index_down_pushbutton.setDefault(False)
        self.move_library_view_index_down_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to move the current Calibre Book Cursor down one row.")
        self.move_library_view_index_down_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.move_library_view_index_down_pushbutton.clicked.connect(self.move_library_view_index_down)
        self.layout_title.addWidget(self.move_library_view_index_down_pushbutton)

        font.setPointSize(11)

        self.title_label = QLabel("Title:")
        self.title_label.setFont(font)
        self.title_label.setMaximumWidth(650)
        self.title_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_title.addWidget(self.title_label)

        font.setPointSize(9)
        #--------------------------------------------------
        self.layout_column = QHBoxLayout()
        self.layout_column.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_column)
        #--------------------------------------------------
        self.frame_1 = QFrame()
        self.frame_1.setFrameShape(QFrame.HLine)
        self.frame_1.setFrameShadow(QFrame.Sunken)
        self.frame_1.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_1)
        #--------------------------------------------------
        self.layout_radios = QHBoxLayout()
        self.layout_radios.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_radios)

        self.custom_column_combobox = QComboBox()
        self.custom_column_combobox.setEditable(False)
        self.custom_column_combobox.setFrame(True)
        self.custom_column_combobox.setDuplicatesEnabled(True)
        self.custom_column_combobox.setMaxVisibleItems(10)
        self.custom_column_combobox.setMaximumWidth(200)
        self.custom_column_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.custom_column_combobox.setFont(font)
        self.custom_column_combobox.setPlaceholderText("#Columns")
        self.custom_column_combobox.setToolTip("<p style='white-space:wrap'>Long-Text/Comments Custom Columns.\n\nUse the Arrow Keys to move up and down the List of Custom Columns that are valid for this Notes Viewer to show.")
        self.layout_radios.addWidget(self.custom_column_combobox)

        self.custom_column_combobox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.block_combobox_signals = False

        font.setPointSize(9)

        self.show_custom_column_combobox_pushbutton = QPushButton("&CC", self)
        self.show_custom_column_combobox_pushbutton.setText("&CC")
        self.show_custom_column_combobox_pushbutton.setFont(font)
        self.show_custom_column_combobox_pushbutton.setMaximumWidth(30)
        self.show_custom_column_combobox_pushbutton.setDefault(False)
        self.show_custom_column_combobox_pushbutton.setToolTip("<p style='white-space:wrap'>Push this button to open up the list of all available Custom Columns for this Viewer.")
        self.show_custom_column_combobox_pushbutton.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.show_custom_column_combobox_pushbutton.clicked.connect(self.show_custom_column_combobox_popup)
        self.layout_radios.addWidget(self.show_custom_column_combobox_pushbutton)

        comments_datatype_list = []
        for custcol,custcol_dict in self.custom_columns_metadata_dict.items():
            if custcol_dict['datatype'] == "comments":
                comments_datatype_list.append(custcol)
        #END FOR
        comments_datatype_list.sort()
        for custcol in comments_datatype_list:
            self.custom_column_combobox.addItem(custcol)
        #END FOR

        self.custom_column_combobox.setCurrentIndex(-1)

        self.radio_group_0_label = QLabel(" ")
        self.radio_group_0_label.setFont(font)
        self.radio_group_0_label.setMinimumWidth(2)
        self.radio_group_0_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_radios.addWidget(self.radio_group_0_label)

        self.lock_current_custom_column_checkbox = QCheckBox("Loc&k?")
        self.lock_current_custom_column_checkbox.setFont(font)
        self.lock_current_custom_column_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to 'lock' what the current Custom Column is regardless of what else you do?<br>The book is never 'locked'.")
        self.layout_radios.addWidget(self.lock_current_custom_column_checkbox)

        self.lock_current_custom_column_checkbox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.lock_current_custom_column = False

        self.radio_group_1_label = QLabel(" ")
        self.radio_group_1_label.setFont(font)
        self.radio_group_1_label.setMinimumWidth(45)
        self.radio_group_1_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_radios.addWidget(self.radio_group_1_label)

        self.plain_radio = QRadioButton('&Plain')
        self.plain_radio.setFont(font)
        self.layout_radios.addWidget(self.plain_radio)
        self.md_radio = QRadioButton('&Markdown')
        self.md_radio.setFont(font)
        self.layout_radios.addWidget(self.md_radio)
        self.html_radio = QRadioButton('H&TML')
        self.html_radio.setFont(font)
        self.layout_radios.addWidget(self.html_radio)

        self.format_radio_button_group = QButtonGroup(self.layout_radios)
        self.format_radio_button_group.setExclusive(True)
        self.format_radio_button_group.addButton(self.plain_radio)
        self.format_radio_button_group.addButton(self.md_radio)
        self.format_radio_button_group.addButton(self.html_radio)

        t = "<p style='white-space:wrap'>The text format to be used to show the current Custom Column's text."
        self.plain_radio.setToolTip(t)
        self.md_radio.setToolTip(t)
        self.html_radio.setToolTip(t)

        self.plain_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.md_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.html_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.radio_group_2_label = QLabel(" ")
        self.radio_group_2_label.setFont(font)
        self.radio_group_2_label.setMinimumWidth(55)
        self.radio_group_2_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_radios.addWidget(self.radio_group_2_label)

        help = self.return_help()
        t = "<p style='white-space:wrap'>View Mode or Edit Mode.<br>" + help

        self.view_radio = QRadioButton('&View')
        self.view_radio.setFont(font)
        self.view_radio.setToolTip(t)
        self.layout_radios.addWidget(self.view_radio)
        self.edit_radio = QRadioButton('&Edit')
        self.edit_radio.setFont(font)
        self.edit_radio.setToolTip(t)
        self.layout_radios.addWidget(self.edit_radio)

        self.view_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.edit_radio.setFocusPolicy(Qt.FocusPolicy.WheelFocus)

        self.is_edit_mode = False

        self.view_edit_radio_button_group = QButtonGroup(self.layout_radios)
        self.view_edit_radio_button_group.setExclusive(True)
        self.view_edit_radio_button_group.addButton(self.view_radio)
        self.view_edit_radio_button_group.addButton(self.edit_radio)

        self.view_radio.setChecked(True)

        self.convert_plain_to_html_pushbutton_pushbutton = QPushButton("&HTML", self)
        self.convert_plain_to_html_pushbutton_pushbutton.setFont(font)
        self.convert_plain_to_html_pushbutton_pushbutton.setMaximumWidth(50)
        self.convert_plain_to_html_pushbutton_pushbutton.setDefault(False)
        self.convert_plain_to_html_pushbutton_pushbutton.setToolTip("<p style='white-space:wrap'>Convert the Plain Text in Edit Mode to basic HTML using Calibre's 'comments-to-html' tool.")
        self.convert_plain_to_html_pushbutton_pushbutton.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.convert_plain_to_html_pushbutton_pushbutton.clicked.connect(self.convert_plain_comments_to_html)
        self.layout_radios.addWidget(self.convert_plain_to_html_pushbutton_pushbutton)

        self.convert_plain_to_html_pushbutton_pushbutton.hide()

        #--------------------------------------------------
        self.frame_2 = QFrame()
        self.frame_2.setFrameShape(QFrame.HLine)
        self.frame_2.setFrameShadow(QFrame.Sunken)
        self.frame_2.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_2)
        #--------------------------------------------------
        self.layout_search = QHBoxLayout()
        self.layout_search.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_search)
        #--------------------------------------------------

        self.search_regex_label = QLabel()
        self.search_regex_label.setTextFormat(Qt.RichText)
        self.search_regex_label.setText("RegExpr")
        self.search_regex_label.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.search_regex_label.setFont(font)
        self.layout_search.addWidget(self.search_regex_label)

        self.search_regex_qlineedit = QLineEdit(self)
        self.search_regex_qlineedit.setText("")
        self.search_regex_qlineedit.setFont(font)
        if DEBUG:
            self.search_regex_qlineedit.setText("economics|universal|fandom|dragon|author")
        self.search_regex_qlineedit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.search_regex_qlineedit.setMinimumWidth(200)
        self.search_regex_qlineedit.setToolTip("<p style='white-space:wrap'>Enter the words or phrase or regular expression to search for in the current Custom Column for all Books in this Library.\
        <br><br>Example: 'Fandom|dragon'<br>Example: 'mitochondria' <br>Example:  'sample size|toxin'\
        <br><br>The vertical bar symbol '|' means 'OR' in a regular expression.<br><br>Refer to the Calibre user manual for more about regular expressions.")
        self.layout_search.addWidget(self.search_regex_qlineedit)
        self.search_regex_label.setBuddy(self.search_regex_qlineedit)

        self.do_search_push_button = QPushButton("Src&h", self)
        self.do_search_push_button.setFont(font)
        self.do_search_push_button.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.do_search_push_button.setToolTip("<p style='white-space:wrap'>Search the selected Custom Column using the selected Regular Expression.")
        self.do_search_push_button.setMaximumWidth(35)
        self.do_search_push_button.clicked.connect(self.do_regex_search)
        self.layout_search.addWidget(self.do_search_push_button)

        self.matching_book_title_rows_combobox = QComboBox()
        self.matching_book_title_rows_combobox.setEditable(False)
        self.matching_book_title_rows_combobox.setPlaceholderText("Matching Title,Author,BookID & Snippet")
        self.matching_book_title_rows_combobox.setDuplicatesEnabled(False)
        self.matching_book_title_rows_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.matching_book_title_rows_combobox.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.matching_book_title_rows_combobox.setFrame(True)
        self.matching_book_title_rows_combobox.setFont(font)
        self.matching_book_title_rows_combobox.setMaxVisibleItems(10)
        self.matching_book_title_rows_combobox.setMinimumWidth(200)
        t = "<p style='white-space:wrap'>Book Author, Title, ID, Snippet for Notes matching the Regular Expression."
        self.matching_book_title_rows_combobox.setToolTip(t)
        self.layout_search.addWidget(self.matching_book_title_rows_combobox)

        self.show_matching_book_push_button = QPushButton("&List", self)
        self.show_matching_book_push_button.setFont(font)
        self.show_matching_book_push_button.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.show_matching_book_push_button.setToolTip("<p style='white-space:wrap'>Show the list of matching Books to the left.")
        self.show_matching_book_push_button.setMaximumWidth(35)
        self.show_matching_book_push_button.clicked.connect(self.show_matching_book_combobox_popup)
        self.layout_search.addWidget(self.show_matching_book_push_button)

        self.jump_to_matching_book_push_button = QPushButton("&Jump", self)
        self.jump_to_matching_book_push_button.setFont(font)
        self.jump_to_matching_book_push_button.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.jump_to_matching_book_push_button.setToolTip("<p style='white-space:wrap'>Jump directly to the matching Book selected to the left.")
        self.jump_to_matching_book_push_button.setMaximumWidth(35)
        self.jump_to_matching_book_push_button.clicked.connect(self.jump_to_matching_book)
        self.layout_search.addWidget(self.jump_to_matching_book_push_button)

        #--------------------------------------------------
        font.setPointSize(11)
        #--------------------------------------------------
        self.frame_3 = QFrame()
        self.frame_3.setFrameShape(QFrame.HLine)
        self.frame_3.setFrameShadow(QFrame.Sunken)
        self.frame_3.setFocusPolicy(Qt.FocusPolicy.NoFocus)
        self.layout_top.addWidget(self.frame_3)
        #--------------------------------------------------
        self.plain_qtextedit =  QTextEdit(self)
        self.plain_qtextedit.setReadOnly(True)
        self.plain_qtextedit.setFont(font)
        self.plain_qtextedit.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.layout_top.addWidget(self.plain_qtextedit)
        self.plain_qtextedit.clear()
        self.plain_qtextedit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.is_plain = True
        self.plain_qtextedit.show()
        self.plain_qtextedit.setEnabled(True)
        #--------------------------------------------------
        self.markdown_qtextedit =  QTextEdit(self)
        self.markdown_qtextedit.setReadOnly(True)
        self.markdown_qtextedit.setFont(font)
        self.markdown_qtextedit.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.layout_top.addWidget(self.markdown_qtextedit)
        self.markdown_qtextedit.clear()
        self.markdown_qtextedit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.is_markdown  = False
        self.markdown_qtextedit.hide()
        self.markdown_qtextedit.setEnabled(False)
        #--------------------------------------------------
        self.html_qtextbrowser =  QTextBrowser(self)
        self.html_qtextbrowser.setReadOnly(True)
        self.html_qtextbrowser.setFont(font)
        self.html_qtextbrowser.setWordWrapMode(QTextOption.WrapMode.WordWrap)
        self.layout_top.addWidget(self.html_qtextbrowser)
        self.html_qtextbrowser.clear()
        self.html_qtextbrowser.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.is_html = False
        self.html_qtextbrowser.hide()
        self.html_qtextbrowser.setEnabled(False)
        self.html_qtextbrowser.setOpenExternalLinks(True)
        self.html_qtextbrowser.setOpenLinks(True)
        self.html_qtextbrowser.clearHistory()
        #--------------------------------------------------

        font.setPointSize(9)

        self.bottom_buttonbox = QDialogButtonBox()
        self.layout_top.addWidget(self.bottom_buttonbox)

        self.push_button_save_edits_notes_viewer = QPushButton(" ", self)
        self.push_button_save_edits_notes_viewer.setFont(font)
        self.push_button_save_edits_notes_viewer.setText("&Save Edits")
        self.push_button_save_edits_notes_viewer.setToolTip("<p style='white-space:wrap'>Save any Edits made while in Edit Mode.")
        self.push_button_save_edits_notes_viewer.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_save_edits_notes_viewer.clicked.connect(self.save_edit_mode)
        self.bottom_buttonbox.addButton(self.push_button_save_edits_notes_viewer,QDialogButtonBox.AcceptRole)

        self.push_button_save_edits_notes_viewer.hide()

        self.push_button_refresh_notes_viewer = QPushButton(" ", self)
        self.push_button_refresh_notes_viewer.setFont(font)
        self.push_button_refresh_notes_viewer.setText("&Refresh Current Book")
        self.push_button_refresh_notes_viewer.setDefault(False)
        self.push_button_refresh_notes_viewer.setToolTip("<p style='white-space:wrap'>The text displayed will be refreshed from the currently selected book.\
                                                                                                                                <br><br>Clicking the mouse cursor on a book will automatically refresh the displayed text,\
                                                                                                                                as will moving the cursor via the keyboard to a new book or Column and then pressing the Enter/Return key.")
        self.push_button_refresh_notes_viewer.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_refresh_notes_viewer.clicked.connect(self.refresh_notes_viewer)
        self.bottom_buttonbox.addButton(self.push_button_refresh_notes_viewer,QDialogButtonBox.AcceptRole)

        self.push_button_save_settings_and_exit = QPushButton(" ", self)
        self.push_button_save_settings_and_exit.setFont(font)
        self.push_button_save_settings_and_exit.setText("E&xit Notes Viewer")
        self.push_button_save_settings_and_exit.setToolTip("<p style='white-space:wrap'>Save the current Notes Viewer window position and size, and then exit.")
        self.push_button_save_settings_and_exit.setFocusPolicy(Qt.FocusPolicy.WheelFocus)
        self.push_button_save_settings_and_exit.clicked.connect(self.save_settings_and_exit)
        self.bottom_buttonbox.addButton(self.push_button_save_settings_and_exit,QDialogButtonBox.AcceptRole)

        self.bottom_buttonbox.setCenterButtons(True)
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        self.resize(self.sizeHint())
        #-----------------------------------------------------
        self.resize_dialog()
        #-----------------------------------------------------
        self.update()
        QApplication.instance().processEvents()
        #-----------------------------------------------------
        self.setTabOrder(self.move_library_view_index_up_pushbutton,self.move_library_view_index_down_pushbutton)
        self.setTabOrder(self.move_library_view_index_down_pushbutton,self.custom_column_combobox)
        self.setTabOrder(self.custom_column_combobox,self.show_custom_column_combobox_pushbutton)
        self.setTabOrder(self.show_custom_column_combobox_pushbutton,self.lock_current_custom_column_checkbox)
        self.setTabOrder(self.lock_current_custom_column_checkbox,self.plain_radio)
        self.setTabOrder(self.plain_radio,self.md_radio)
        self.setTabOrder(self.md_radio,self.html_radio)
        self.setTabOrder(self.html_radio,self.view_radio)
        self.setTabOrder(self.view_radio,self.edit_radio)
        self.setTabOrder(self.edit_radio,self.search_regex_qlineedit)
        self.setTabOrder(self.search_regex_qlineedit,self.do_search_push_button)
        self.setTabOrder(self.do_search_push_button,self.matching_book_title_rows_combobox)
        self.setTabOrder(self.matching_book_title_rows_combobox,self.show_matching_book_push_button)
        self.setTabOrder(self.show_matching_book_push_button,self.jump_to_matching_book_push_button)
        #~ QTextEdit TabOrders are not static and are handled in an event...
        self.setTabOrder(self.push_button_save_edits_notes_viewer,self.push_button_refresh_notes_viewer)
        self.setTabOrder(self.push_button_refresh_notes_viewer,self.push_button_save_settings_and_exit)
        self.setTabOrder(self.push_button_save_settings_and_exit,self.move_library_view_index_up_pushbutton)

        #-----------------------------------------------------
        self.block_radio_signals = True
        self.block_view_edit_radio_signals = True
        self.block_refresh_signals = True
        #-----------------------------------------------------

        try:
            if self.last_current_column.startswith("#"):
                vpos = self.maingui.library_view.verticalScrollBar().value()  # if user opens NV before *any* book selected in view, at all; just ran Calibre and then with mouse scrolled down many pages...
                self.startup_column = self.last_current_column
                if DEBUG: print("__init__ near end: ",self.last_current_column, "  vpos: ", str(vpos)  )
                cix = self.maingui.library_view.currentIndex()
                bookid = None
                if cix.isValid():
                    bookid = self.maingui.library_view.model().id(cix)
                else:
                    bookid = self.maingui.library_view.current_book()
                idx = self.maingui.library_view.column_map.index(self.last_current_column)
                current_column = self.maingui.library_view.column_map[idx]
                idx = self.maingui.library_view.column_map.index(current_column)
                current_column = self.maingui.library_view.column_map[idx]
                if DEBUG: print("current_column per column_map: ", current_column )
                if bookid is None:
                    row_number = None
                else:
                    row_number =  self.maingui.library_view.model().db.data.id_to_index(bookid)
                    if DEBUG: print("row_number of current book is: ", str(row_number))
                    if row_number is not None:
                        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
                self.custom_column_combobox.setCurrentText(current_column)
                self.last_current_column = current_column
                if DEBUG: print("new self.last_current_column: ", self.last_current_column )
                self.maingui.library_view.verticalScrollBar().setValue(vpos)  # return as close as possible to where the user was when NV initialized
            else:
                pass
        except Exception as e:
            if DEBUG: print("Exception in __init__ at end: ", str(e))

        #-----------------------------------------------------
        self.block_radio_signals = False
        self.block_view_edit_radio_signals = False
        self.block_refresh_signals = False
        #-----------------------------------------------------

        self.plain_radio.toggled.connect(self.event_text_type_button_changed)
        self.md_radio.toggled.connect(self.event_text_type_button_changed)
        self.html_radio.toggled.connect(self.event_text_type_button_changed)
        self.view_radio.toggled.connect(self.event_view_edit_buttons_changed)
        self.edit_radio.toggled.connect(self.event_view_edit_buttons_changed)
        self.custom_column_combobox.currentIndexChanged.connect(self.event_custom_column_combobox_changed)
        self.lock_current_custom_column_checkbox.toggled.connect(self.event_lock_current_custom_column_checkbox)
        #-----------------------------------------------------
        self.startup_column = None
        self.maingui.library_view.selectionModel().currentChanged.connect(self.refresh_notes_viewer)
        #-----------------------------------------------------
        self.refresh_notes_viewer()
        self.plain_radio.setChecked(True)
        self.edit_radio.setChecked(True)
        self.view_radio.setChecked(True)
        #-----------------------------------------------------
        self.n_rows = self.maingui.library_view.model().rowCount(QModelIndex())
        #-----------------------------------------------------

    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_text_type_button_changed(self,event):
        if self.block_radio_signals:
            return

        self.long_text = self.original_long_text  #avoid recursive reformatting

        if self.plain_radio.isChecked():
            self.plain_qtextedit.setEnabled(True)
            self.is_plain = True
            self.is_markdown = False
            self.is_html = False
            plain_text = self.format_plain_text(self.long_text)
            self.plain_qtextedit.setPlainText(plain_text)
            self.plain_qtextedit.show()
            self.markdown_qtextedit.hide()
            self.html_qtextbrowser.hide()
            self.plain_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.markdown_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.html_qtextbrowser.setFocusProxy(self.plain_qtextedit)
            self.setTabOrder(self.jump_to_matching_book_push_button,self.plain_qtextedit)
            self.setTabOrder(self.plain_qtextedit,self.push_button_save_edits_notes_viewer)
            self.markdown_qtextedit.setEnabled(False)
            self.html_qtextbrowser.setEnabled(False)

        elif self.md_radio.isChecked():
            self.markdown_qtextedit.setEnabled(True)
            self.is_plain = False
            self.is_markdown = True
            self.is_html = False
            markdown_text = self.format_markdown_text(self.long_text)
            self.markdown_qtextedit.setMarkdown(markdown_text)   # https://doc.qt.io/qt-6/qtextedit.html#markdown-prop
            self.plain_qtextedit.hide()
            self.markdown_qtextedit.show()
            self.html_qtextbrowser.hide()
            self.markdown_qtextedit.setFocusProxy(self.markdown_qtextedit)
            self.plain_qtextedit.setFocusProxy(self.markdown_qtextedit)
            self.html_qtextbrowser.setFocusProxy(self.markdown_qtextedit)
            self.setTabOrder(self.jump_to_matching_book_push_button,self.markdown_qtextedit)
            self.setTabOrder(self.markdown_qtextedit,self.push_button_save_edits_notes_viewer)
            self.plain_qtextedit.setEnabled(False)
            self.html_qtextbrowser.setEnabled(False)

        elif self.html_radio.isChecked():
            self.html_qtextbrowser.setEnabled(True)
            self.is_plain = False
            self.is_markdown = False
            self.is_html = True
            html_text = self.format_plain_text(self.long_text)
            self.html_qtextbrowser.setHtml(html_text)                                 # https://doc.qt.io/qt-6/qtextedit.html#html-prop
            self.plain_qtextedit.hide()
            self.markdown_qtextedit.hide()
            self.html_qtextbrowser.show()
            self.html_qtextbrowser.setFocusProxy(self.html_qtextbrowser)
            self.plain_qtextedit.setFocusProxy(self.html_qtextbrowser)
            self.markdown_qtextedit.setFocusProxy(self.html_qtextbrowser)
            self.setTabOrder(self.jump_to_matching_book_push_button,self.html_qtextbrowser)
            self.setTabOrder(self.html_qtextbrowser,self.push_button_save_edits_notes_viewer)
            self.plain_qtextedit.setEnabled(False)
            self.markdown_qtextedit.setEnabled(False)

        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_custom_column_combobox_changed(self,event):
        if self.block_combobox_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return
        if self.lock_current_custom_column:
            self.block_combobox_signals = True
            self.custom_column_combobox.setCurrentText(self.last_current_column)
            self.block_combobox_signals = False
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return
        current_column = self.custom_column_combobox.currentText()
        if current_column is None:
            return
        if not current_column.startswith("#"):
            return
        if not self.last_bookid > 0:
            return
        idx = self.maingui.library_view.column_map.index(current_column)
        current_column = self.maingui.library_view.column_map[idx]
        row_number = self.maingui.library_view.model().db.data.id_to_index(self.last_bookid)
        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
        self.custom_column_combobox.setCurrentText(current_column)
        self.refresh_notes_viewer()
        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_view_edit_buttons_changed(self,event):
        if self.block_view_edit_radio_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        if self.view_radio.isChecked():
            self.is_edit_mode = False
        else:
            self.is_edit_mode = True

        if self.is_edit_mode:
            #~ from: View (Any Type)     to: Edit Mode (Always Plain)
            self.plain_radio.setChecked(True)
            self.plain_qtextedit.setReadOnly(False)
            self.push_button_save_edits_notes_viewer.show()
            self.push_button_refresh_notes_viewer.setText(" &Cancel Column Edits ")
            self.plain_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.markdown_qtextedit.setFocusProxy(self.plain_qtextedit)
            self.html_qtextbrowser.setFocusProxy(self.plain_qtextedit)
            self.setTabOrder(self.edit_radio,self.plain_qtextedit)
            self.setTabOrder(self.plain_qtextedit,self.push_button_save_edits_notes_viewer)
            self.markdown_qtextedit.setReadOnly(True)
            self.html_qtextbrowser.setReadOnly(True)
            self.markdown_qtextedit.setEnabled(False)
            self.html_qtextbrowser.setEnabled(False)
            self.convert_plain_to_html_pushbutton_pushbutton.show()
        else:
            #~ from: Edit (Always Plain) to: View Mode (Still Plain)
            self.push_button_save_edits_notes_viewer.hide()
            self.push_button_refresh_notes_viewer.setText("&Refresh Current Book")
            self.plain_qtextedit.setEnabled(True)
            self.markdown_qtextedit.setEnabled(True)
            self.html_qtextbrowser.setEnabled(True)
            self.plain_qtextedit.setReadOnly(True)
            self.markdown_qtextedit.setReadOnly(True)
            self.html_qtextbrowser.setReadOnly(True)
            self.plain_radio.setChecked(True) # Qt can change tab orders by itself, so reset them...
            self.convert_plain_to_html_pushbutton_pushbutton.hide()

        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_lock_current_custom_column_checkbox(self,event):
        if self.lock_current_custom_column_checkbox.isChecked():
            self.lock_current_custom_column = True
        else:
            self.lock_current_custom_column = False
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index_up(self):
        self.move_library_view_index(-1)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index_down(self):
        self.move_library_view_index(1)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def move_library_view_index(self,increment=0,bookid=None):
        if bookid is None:
            bookid = self.last_bookid
        idx = self.maingui.library_view.column_map.index(self.last_current_column)
        current_column = self.maingui.library_view.column_map[idx]
        row_number = self.maingui.library_view.model().db.data.id_to_index(bookid)
        row_number = row_number + increment
        if row_number < 0:
            row_number = 0
        if row_number > self.n_rows:
            row_number = self.n_rows - 1
        self.maingui.library_view.select_cell(row_number=row_number, logical_column=idx)
        self.custom_column_combobox.setCurrentText(current_column)
        self.refresh_notes_viewer()
        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def refresh_notes_viewer_qtimer(self):
        QTimer.singleShot(600,self.refresh_notes_viewer)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def refresh_notes_viewer(self):

        if self.block_refresh_signals:
            self.set_text_background_color()
            QApplication.instance().processEvents()
            return

        if self.lock_current_custom_column:
            self.block_combobox_signals = True
            self.custom_column_combobox.setCurrentText(self.last_current_column)
            self.block_combobox_signals = False
            # but: must allow book to change, keeping column locked

        if self.edit_radio.isChecked():
            self.block_refresh_signals = True
            self.view_radio.setChecked(True)
            self.block_refresh_signals = False
            return
        else:
            self.push_button_save_edits_notes_viewer.hide()

        try:
            cix = self.maingui.library_view.currentIndex()
            if cix.isValid():
                if not self.lock_current_custom_column:
                    #---------------------------
                    current_column = self.get_current_column()
                    if DEBUG: print("current_column: ",current_column)
                    if current_column in self.custom_columns_metadata_dict:
                        custcol_dict = self.custom_columns_metadata_dict[current_column]
                        if custcol_dict['datatype'] == "comments":
                            self.is_valid_long_text = True
                        else:
                            self.is_valid_long_text = False
                    else:
                        self.is_valid_long_text = False
                    if not self.is_valid_long_text:
                        self.block_combobox_signals = True
                        self.custom_column_combobox.setCurrentIndex(-1)
                        self.block_combobox_signals = False
                    else:
                        self.block_combobox_signals = True
                        self.custom_column_combobox.setCurrentText(current_column)
                        self.block_combobox_signals = False
                    #---------------------------
                else:   # bookid is never locked
                    current_column = self.last_current_column

                bookid = self.maingui.library_view.model().id(cix)
                self.last_bookid = bookid
                self.last_current_column = current_column

                if self.last_current_column.startswith("#"):
                    r = 'GUI_TOOLS_CONVERT_NOTES_VIEWER_LAST_CUSTOM_COLUMN_USED', self.last_current_column
                    self.prefs_set.add(r)

                #~ Note:  using mi = self.maingui.library_view.model().db.get_metadata avoids complications when changing libraries causing guidb changes, proper disconnection of events, and user not properly closing this dialog before changing the library.
                mi = self.maingui.library_view.model().db.get_metadata(bookid, index_is_id=True, get_user_categories=False)

                authorsx = mi.authors
                if isinstance(authorsx,tuple):
                    authors = list(authorsx)
                if isinstance(authorsx,list):
                    authors = ""
                    for a in authorsx:
                        authors = authors + a + " & "
                    #END FOR
                    authors = authors.strip()
                    if authors.endswith("&"):
                        authors = authors[0:-1]
                        authors = authors.strip()
                    authors = authors[0:MAX_AUTHOR_TITLE_LENGTH]
                title = mi.title
                title = title[0:MAX_AUTHOR_TITLE_LENGTH]
                bookid = str(bookid)
                if self.is_valid_long_text:
                    mi_dict = mi.get_user_metadata(current_column,make_copy=False)
                    long_text = mi_dict['#value#']
                    if long_text is None:
                        long_text = current_column + ":   [No Text]"
                else:
                    long_text = current_column + ":   [Not a Long-Text/Comments Custom Column]"

                self.long_text = long_text
                self.original_long_text = long_text  # to later avoid recursive reformatting

                #---------------------------
                if len(authors) >= MAX_AUTHOR_TITLE_LENGTH:
                    authors = authors[0:MAX_AUTHOR_TITLE_LENGTH] + ellipsis
                self.author_label.setText(authors)
                if len(title) >= MAX_AUTHOR_TITLE_LENGTH:
                    title = title[0:MAX_AUTHOR_TITLE_LENGTH] + ellipsis
                self.title_label.setText(title)

                self.plain_qtextedit.hide()
                self.markdown_qtextedit.hide()
                self.html_qtextbrowser.hide()

                self.guess_text_type()

                if self.is_fatal_error:
                    return

                self.is_html = False
                self.is_markdown = False
                self.is_plain = False

                if self.best_guess == HTML:
                    self.is_html = True
                elif self.best_guess == MD:
                    self.is_markdown = True
                else:
                    self.is_plain = True

                self.block_radio_signals = True

                if self.is_plain:
                    plain_text = self.format_plain_text(self.long_text)
                    self.plain_qtextedit.setPlainText(plain_text)
                    self.plain_qtextedit.show()
                    self.plain_radio.setChecked(True)
                elif self.is_markdown :
                    markdown_text = self.format_markdown_text(self.long_text)
                    self.markdown_qtextedit.setMarkdown(markdown_text)
                    self.markdown_qtextedit.show()
                    self.md_radio.setChecked(True)
                elif self.is_html:
                    html_text = self.format_html_text(self.long_text)
                    self.html_qtextbrowser.setHtml(html_text)
                    self.html_qtextbrowser.clearHistory()
                    self.html_qtextbrowser.show()
                    self.html_radio.setChecked(True)

                self.block_radio_signals = False
                #---------------------------
            else:
                if DEBUG: print("cix is NOT valid...")
                return
            #--------------------------------------
            #--------------------------------------
        except Exception as e:
            if DEBUG: print("Exception in NotesViewerDialog:", str(e))
            return

        self.set_text_background_color()

        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def guess_text_type(self):

        self.is_plain_text = False
        self.is_markdown = False
        self.is_html = False

        n_markdown_score = 0
        n_html_score = 0

        n_md_regexes = len(self.markdown_regex_list)
        n_html_regexes = len(self.html_regex_list)

        self.long_text = self.original_long_text  #avoid recursive reformatting

        for r in self.markdown_regex_list:      # p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
            p,regex,fmt = r
            match = p.search(self.long_text)
            if match:
                if DEBUG: print(fmt, ": regex match: ", regex)
                n_markdown_score = n_markdown_score + 1
        #END FOR
        md_match_ratio = n_markdown_score/n_md_regexes
        if DEBUG: print("raw md_match_ratio: ", str(md_match_ratio))

        for r in self.html_regex_list:
            p,regex,fmt = r
            match = p.search(self.long_text)
            if match:
                if DEBUG: print(fmt, ": regex match: ", regex)
                n_html_score = n_html_score + 1
        #END FOR
        html_match_ratio = n_html_score/n_html_regexes
        if DEBUG: print("raw html_match_ratio: ", str(html_match_ratio))

        total_scores = n_markdown_score + n_html_score
        if total_scores > 0:
            md_match_ratio = (n_markdown_score/total_scores) * (md_match_ratio + html_match_ratio)
            html_match_ratio = (n_html_score/total_scores) * (md_match_ratio + html_match_ratio)

        if  md_match_ratio > html_match_ratio:
            self.best_guess = MD
        else:
            self.best_guess = HTML

        #~ but if both are very low, it is probably plain text.
        if md_match_ratio < 0.04  and html_match_ratio < 0.04:
            self.best_guess = PLAIN

        #~ but Calibre comments_to_text causes a 0 md and a 0.027 html.
        if md_match_ratio == 0  and html_match_ratio > 0.02:
            self.best_guess = HTML

        if DEBUG: print("guessing:\n  MD: score & weighted ratio: ", str(n_markdown_score), str(md_match_ratio), "\n  HTML: score & weighted ratio: ", str(n_html_score), str(html_match_ratio), "\n  --->>> best guess: ", self.best_guess)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def format_plain_text(self,long_text):
        qte = QTextEdit()
        qte.setPlainText(long_text)
        long_text = qte.toPlainText()
        self.long_text = long_text
        del qte
        return long_text
    #---------------------------------------------------------------------------------------------------------------------------------------
    def format_markdown_text(self,long_text):
        qte = QTextEdit()
        qte.setMarkdown(long_text)
        long_text = qte.toMarkdown()
        self.long_text = long_text
        del qte
        return long_text
    #---------------------------------------------------------------------------------------------------------------------------------------
    def format_html_text(self,long_text):
        qte = QTextEdit()
        qte.setHtml(long_text)
        long_text = qte.toHtml()
        self.long_text = long_text
        del qte
        return long_text
    #---------------------------------------------------------------------------------------------------------------------------------------
    def get_current_column(self):
        if self.startup_column is None:
            current_column = None
            current_col = self.maingui.library_view.currentIndex().column()
            current_column = self.maingui.library_view.column_map[current_col]
            if DEBUG: print("current_col: ", str(current_col), "lookup/search name: ", str(current_column))
            if current_column.startswith("#"):
                self.custom_column_combobox.setCurrentText(current_column)
        else:
            if self.startup_column.startswith("#"):
                self.custom_column_combobox.setCurrentText(self.startup_column)
                current_column = self.startup_column
        return current_column
    #---------------------------------------------------------------------------------------------------------------------------------------
    def show_custom_column_combobox_popup(self):
        self.custom_column_combobox.showPopup()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_edit_mode(self):
        #~ editing mode for markdown and html, plus of course plain, is done entirely in the plain text format qtextedit, since editing md and html are not wysiwyg.

        edited_long_text = self.plain_qtextedit.toPlainText()
        self.long_text = self.format_plain_text(edited_long_text)
        self.original_long_text = self.long_text

        book = self.last_bookid
        payload = []
        payload.append(book)
        mi = Metadata(_('Unknown'))
        custcol_dict = self.custom_columns_metadata_dict[self.last_current_column]
        custcol_dict['#value#'] = self.long_text
        mi.set_user_metadata(self.last_current_column, custcol_dict)
        id_map = {}
        id_map[book] = mi
        edit_metadata_action = self.maingui.iactions['Edit Metadata']

        if self.edit_radio.isChecked():
            self.block_refresh_signals = True
            self.view_radio.setChecked(True)  # required prior to callback executing
            self.block_refresh_signals = False
        else:
            self.block_refresh_signals = False
            self.push_button_save_edits_notes_viewer.hide()

        edit_metadata_action.apply_metadata_changes(id_map, callback=self.refresh_notes_viewer_qtimer())

        self.convert_plain_to_html_pushbutton_pushbutton.hide()

        del id_map
        del mi
        del edited_long_text
        del custcol_dict
    #---------------------------------------------------------------------------------------------------------------------------------------
    def build_regular_expressions_for_guessing(self):

        markdown_regex_list = []   # http://chubakbidpaa.com/interesting/2021/09/28/regex-for-md.html
        markdown_regex_list.append('(#{1}\s)(.*)')  #header 1
        markdown_regex_list.append('(#{2}\s)(.*)')  #header 2
        markdown_regex_list.append('(#{3}\s)(.*)')  #header 3
        markdown_regex_list.append('(#{4}\s)(.*)')  #header 4
        markdown_regex_list.append('(#{5}\s)(.*)')  #header 5
        markdown_regex_list.append('(#{6}\s)(.*)')  #header 6
        markdown_regex_list.append('(\*|\_)+(\S+)(\*|\_)+')                        # boldItalicText
        markdown_regex_list.append("(\_|\*){1}(\w|\W|\d)+(\_|\*){1}")       # Italics
        markdown_regex_list.append("(\_|\*){3}(\w|\W|\d)+(\_|\*){3}")       # bold Italics
        markdown_regex_list.append("(\_|\*){2}(\w|\W|\d)+(\_|\*){2}")        # bold
        markdown_regex_list.append('(\[.*\])(\((http)(?:s)?(\:\/\/).*\))' )      # linkText
        markdown_regex_list.append('(^(\W{1})(\s)(.*)(?:$)?)+')                 # unordered list text
        markdown_regex_list.append('(^(\d+\.)(\s)(.*)(?:$)?)+')                  # numbered list
        markdown_regex_list.append('((^(\>{1})(\s)(.*)(?:$)?)+)')               # block quote
        markdown_regex_list.append("(\\'{1})(.*)(\\'{1})")                             # inline code
        markdown_regex_list.append("(\\'{3}\\n+)(.*)(\\n+\\'{3})")            # code block
        markdown_regex_list.append('(\=|\-|\*){3}')                                     # horizontal line
        markdown_regex_list.append('(\<{1})(\S+@\S+)(\>{1})')              # email text
        tables = "(((\|)([a-zA-Z\d+\s#!@'" + '"():;\\\/.\[\]\^<={$}>?(?!-))]+))+(\|))(?:\n)?((\|)(-+))+(\|)(\n)((\|)(\W+|\w+|\S+))+(\|$)'
        markdown_regex_list.append(tables)                                                 # tables
        markdown_regex_list.append(r'(?<=\[\[).*?(?=(?:\]\]|#|\|))')          # links
        s = '(\!)(\[(?:.*)?\])(\(.*(\.(\w{3}))(?:(\s\"' + "|\')(\w|\W|\d)+" + '(\"' + "|\'))?\))"    # imagefile
        markdown_regex_list.append(s)
        markdown_regex_list.append('(#{%d}\s)(.*)')                                  # HeaderAll  https://github.com/Chubek/md2docx/blob/master/patterns/patterns.go
        markdown_regex_list.append("(\_|\*){3}")                                       # ThreeUA
        markdown_regex_list.append("(\_|\*){2}")                                       #  TwoUA
        markdown_regex_list.append("(\_|\*){1}")                                       #  OneUA
        markdown_regex_list.append("((\|)(-+))+(\|)(\n)")                         #  TableSep

        html_regex_list = []
        html_regex_list.append(r' href=')
        html_regex_list.append(r'<!DOCTYPE html>')
        html_regex_list.append(r'<a|</a|<a href=')
        html_regex_list.append(r'<body|</body')
        html_regex_list.append(r'<br')
        html_regex_list.append(r'<cite|</cite')
        html_regex_list.append(r'<code|</code')
        html_regex_list.append(r'<div|</div')
        html_regex_list.append(r'<em|</em ')
        html_regex_list.append(r'<h1')
        html_regex_list.append(r'<h2')
        html_regex_list.append(r'<h3')
        html_regex_list.append(r'<h4')
        html_regex_list.append(r'<h5')
        html_regex_list.append(r'<h6')
        html_regex_list.append(r'<head|</head ')
        html_regex_list.append(r'<hr')
        html_regex_list.append(r'<html|</html')
        html_regex_list.append(r'<img src=')
        html_regex_list.append(r'<kbd')
        html_regex_list.append(r'<li|</li')
        html_regex_list.append(r'<meta')
        html_regex_list.append(r'<nobr')
        html_regex_list.append(r'<ol|</ol')
        html_regex_list.append(r'<pre|</pre')
        html_regex_list.append(r'<p|<p class="description"|</p>')
        html_regex_list.append(r'<script src="//')
        html_regex_list.append(r'<span|</span')
        html_regex_list.append(r'<strong|</strong')
        html_regex_list.append(r'<style|</style')
        html_regex_list.append(r'<sub|</sub')
        html_regex_list.append(r'<sup|</sup')
        html_regex_list.append(r'<td|</td')
        html_regex_list.append(r'<tr|</tr')
        html_regex_list.append(r'<tt|</tt')
        html_regex_list.append(r'<ul|</ul')
        html_regex_list.append(r'<u|</u')

        self.markdown_regex_list = []
        self.html_regex_list = []

        self.is_fatal_error = False
        errors = "No Errors"

        try:
            for regex in markdown_regex_list:
                p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
                fmt = "MD"
                r = p,regex,fmt
                self.markdown_regex_list.append(r)
            #END FOR
        except Exception as e:
            if DEBUG: print("markdown_regex_list: ", regex, "   ", str(e))
            self.markdown_regex_list.append(None)
            self.is_fatal_error = True
            errors = "markdown regex: " + regex + "     <<<---    " + str(e)

        try:
            for regex in html_regex_list:
                p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
                fmt = "HTML"
                r = p,regex,fmt
                self.html_regex_list.append(r)
            #END FOR
        except Exception as e:
            if DEBUG: print("html_regex_list: ", regex, "   ", str(e))
            self.html_regex_list.append(None)
            self.is_fatal_error = True
            errors = "html regex: " + regex + "     <<<---    " + str(e)

        return errors
    #---------------------------------------------------------------------------------------------------------------------------------------
    def do_regex_search(self):
        regex = self.search_regex_qlineedit.text()
        try:
            p = re.compile(regex, re.IGNORECASE|re.MULTILINE)
        except Exception as e:
            msg = "Regular Expression failed to compile: ", regex, "  reason: ", str(e)
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        msg = None
        if self.last_current_column is None:
            msg = "Please select a #Custom Column before searching."
        if not self.last_current_column.startswith("#"):
            msg = "Please select a #Custom Column before searching."
        if not msg is None:
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        self.matching_book_title_rows_combobox.clear()
        self.matching_book_title_rows_combobox.update()

        book_value_dict = {}

        dbcache = self.maingui.current_db.new_api
        all_book_ids_set = dbcache.all_book_ids()
        name = self.last_current_column
        if DEBUG: print("self.last_current_column: ", self.last_current_column, " number of books: ", str(len(all_book_ids_set)))
        for book in all_book_ids_set:
            value = dbcache.field_for(name, book_id=book, default_value=None)
            if isinstance(value,str):
                book_value_dict[book] = value
                if DEBUG: print(str(book),value.strip)
        #END FOR

        if len(book_value_dict) == 0:
            msg = "No books were found that have any values in the selected Custom Column: " +  self.last_current_column
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        partial_results_list = []
        book_id_list = []
        value = None
        for book,value in book_value_dict.items():
            match = p.search(value)
            if match is not None:
                snippet = match.group()
                if DEBUG: print(str(snippet))
                if snippet is not None:
                    snippet = str(snippet)
                    snippet = snippet.strip()
                    r = book,snippet
                    partial_results_list.append(r)
                    book_id_list.append(book)
                    if DEBUG: print("regex match: ", regex, "  ", str(book), "snippet: ", str(snippet))  # regex match:  fandom|dragon    963 snippet:  Dragon
                del snippet
            del match
        #END FOR
        del value

        if len(partial_results_list) == 0:
            msg = "No books were found that have any values matching your Search Regular Expression: " +  regex
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        author_sort_name_dict = {}
        name = 'authors'
        authorid_data_map_dict = dbcache.author_data()  # [author_id] = name, sort, link
        for authorid,data_dict in authorid_data_map_dict.items():
            #~ if DEBUG: print("authorid,data: ", str(authorid), str(data_dict))  #~ authorid,data:  76 {'name': 'Singh R', 'sort': 'R, Singh', 'link': ''}
            if 'sort' in data_dict:
                if 'name' in data_dict:
                    sort = data_dict['sort']
                    name = data_dict['name']
                    author_sort_name_dict[sort] = name
        #END FOR

        book_authorid_map_dict = dbcache.author_sort_strings_for_books(book_id_list)  # val_map[book_id] = tuple(adata[aid]['sort'] for aid in authors)
        if DEBUG:
            for book,data_list in book_authorid_map_dict.items():
                print(str(book))   #963
                for r in data_list:
                    print(str(r))   #~ R, Singh
                #END FOR
            #END FOR

        book_author_dict = {}  #  [book] = name

        for book,data_list in book_authorid_map_dict.items():
            if DEBUG: print("data_list: ", type(data_list))  #~ data_list:  <class 'tuple'>
            s = ""
            for sort in data_list:
                if DEBUG: print(type(sort), str(sort))
                if sort in author_sort_name_dict:
                    name = author_sort_name_dict[sort]
                    s = s + name + " & "
            #END FOR
            name = s[0:-2]
            name = name.strip()
            if DEBUG: print("author name: ", name)   #  author name:  Singh R & Kaushik S & Wang Y & Xiang Y & Novak I & Komatsu M & Tanaka K & Cuervo AM & Czaja MJ
            book_author_dict[book] = name
            del s
            del data_list
            del name
        #END FOR

        if len(book_author_dict) == 0:
            msg = "No books were found that have any Authors for your books: " +  regex
            return error_dialog(self.maingui, _(tool_name),_(msg), show=True)

        results_list = []

        for r in partial_results_list:
            book,snippet = r
            if book in book_author_dict:
                name = book_author_dict[book]
                if name:
                    row = book,name,snippet
                    results_list.append(row)
        #END FOR
        del partial_results_list
        del book_author_dict
        del book_authorid_map_dict
        del author_sort_name_dict

        for r in results_list:
            book,name,snippet = r
            title = dbcache._field_for('title', book_id=book, default_value= 'Unknown')
            if title:
                s = name[0:20] + "•" + title + "•" + str(book) + "•" + snippet[0:100]
                self.matching_book_title_rows_combobox.addItem(s)
                #~ if DEBUG:
                    #~ i = 0
                    #~ while i < 10000:
                        #~ self.matching_book_title_rows_combobox.addItem(s)
                        #~ i = i + 1
                    #~ #END WHILE
                #~ if DEBUG: print(s)   # Singh R & Kaushik S & Wang Y & Xiang Y & Novak I & Komatsu M & Tanaka K & Cuervo AM & Czaja MJ•Autophagy Regulates Lipid Metabolism.•963•Dragon
        #END FOR

        self.matching_book_title_rows_combobox.setCurrentIndex(0)
        self.matching_book_title_rows_combobox.update()
        QApplication.instance().processEvents()

        if DEBUG: print("number of items in combobox: ", str(self.matching_book_title_rows_combobox.count()))

        if self.matching_book_title_rows_combobox.count() == 0:
            msg = "No matching book/author/title/snippets were found"
            return info_dialog(self.maingui, _(tool_name),_(msg), show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def show_matching_book_combobox_popup(self):
        self.matching_book_title_rows_combobox.showPopup()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def jump_to_matching_book(self):
        text = self.matching_book_title_rows_combobox.currentText()
        s_split = text.split("•")
        if len(s_split) == 4:
            book = s_split[2]
            book = book.strip()
            if book.isdigit():
                book = int(book)
                try:
                    self.move_library_view_index(increment=0,bookid=book)
                    if DEBUG: print("jumped to book: ", str(book))
                except Exception as e:
                    msg = "Book does not currently exist.  Either you recently deleted it, or you are using a Virtual Library that does not contain it."
                    if DEBUG: print(msg)
                    return error_dialog(self.maingui, _(tool_name),_(msg), show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def set_text_background_color(self):
        if not self.is_dark_mode:
            self.plain_qtextedit.setTextBackgroundColor(color_white)
            self.markdown_qtextedit.setTextBackgroundColor(color_white)
            self.html_qtextbrowser.setTextBackgroundColor(color_white)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_custom_columns_listing_dialog_geometry(self):
        self.dialog_closing(None)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def convert_plain_comments_to_html(self):
        if not self.is_edit_mode:
            return
        if DEBUG:
            if self.long_text != self.original_long_text:
                print("\nself.long_text != self.original_long_text\n")
        new_html = comments_to_html(self.long_text)
        self.long_text = self.format_plain_text(new_html)
        self.original_long_text = self.long_text
        self.plain_qtextedit.setPlainText(self.long_text)
        self.save_edit_mode()
        self.html_radio.setChecked(True)
        self.set_text_background_color()
        QApplication.instance().processEvents()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_settings_and_exit(self):

        if self.edit_radio.isChecked():
            self.block_refresh_signals = True
            self.view_radio.setChecked(True)
            self.block_refresh_signals = False

        self.dialog_closing(None)

        try:
            self.maingui.library_view.selectionModel().currentChanged.disconnect(self.refresh_notes_viewer)
        except:
            pass

        self.return_from_notes_viewer_to_save_prefs(self.prefs_set)

        self.close()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def return_help(self):
        help_html = '''
        <p>In View Mode, keys are limited to navigation, and text may only be selected with the mouse:</p>
        <div class="table"><table class="generic">
        <thead><tr class="qt-style"><th>Keypresses</th><th>Action</th></tr></thead>
        <tr class="odd" valign="top"><td>Up</td><td>Moves one line up.</td></tr>
        <tr class="even" valign="top"><td>Down</td><td>Moves one line down.</td></tr>
        <tr class="odd" valign="top"><td>Left</td><td>Moves one character to the left.</td></tr>
        <tr class="even" valign="top"><td>Right</td><td>Moves one character to the right.</td></tr>
        <tr class="odd" valign="top"><td>PageUp</td><td>Moves one page up.</td></tr>
        <tr class="even" valign="top"><td>PageDown</td><td>Moves one page down.</td></tr>
        <tr class="odd" valign="top"><td>Home</td><td>Moves to the beginning of the text.</td></tr>
        <tr class="even" valign="top"><td>End</td><td>Moves to the end of the text.</td></tr>
        <tr class="odd" valign="top"><td>Alt+Wheel</td><td>Scrolls the page horizontally (the Wheel is the mouse wheel).</td></tr>
        <tr class="even" valign="top"><td>Ctrl+Wheel</td><td>Zooms the text.</td></tr>
        <tr class="odd" valign="top"><td>Ctrl+A</td><td>Selects all text.</td></tr>
        </table></div>
        '''
        return help_html
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
#END OF notes_viewer_dialog.py