# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__my_version__ = "1.0.206"  # NBIB to RIS Converter/Exploder to Auto-Add

import os
import re
from time import sleep
from qt.core import QTimer
from calibre.constants import DEBUG
from calibre.gui2 import info_dialog
from polyglot.builtins import as_unicode, iteritems

from calibre.gui2.ui import get_gui
maingui = get_gui()

KEYVAL_SEPARATOR = "⎨"      # U+23A8  unique characters required for unique line splitting symbol  e.g. "pmid⎨35835100".split(KEYVAL_SEPARATOR)

nbib_tags_not_officially_multiple = []  # more are than are not.
nbib_tags_not_officially_multiple.append("AB")
nbib_tags_not_officially_multiple.append("BTI")
nbib_tags_not_officially_multiple.append("CTI")
nbib_tags_not_officially_multiple.append("DP")
nbib_tags_not_officially_multiple.append("IP")
nbib_tags_not_officially_multiple.append("ISBN")
nbib_tags_not_officially_multiple.append("JID")
nbib_tags_not_officially_multiple.append("JT")
nbib_tags_not_officially_multiple.append("LA")
nbib_tags_not_officially_multiple.append("MID")
nbib_tags_not_officially_multiple.append("PB")
nbib_tags_not_officially_multiple.append("PG")
nbib_tags_not_officially_multiple.append("PL")
nbib_tags_not_officially_multiple.append("PMC")
nbib_tags_not_officially_multiple.append("PMCR")
nbib_tags_not_officially_multiple.append("PMID")
nbib_tags_not_officially_multiple.append("TA")
nbib_tags_not_officially_multiple.append("TI")
nbib_tags_not_officially_multiple.append("TT")
nbib_tags_not_officially_multiple.append("VI")
nbib_tags_not_officially_multiple.append("VTI")
NBIB_TAGS_NOT_OFFICIALLY_MULTIPLE = nbib_tags_not_officially_multiple

TOOL_NAME = "JS+ GUI Tool: NBIB to RIS Converter/Exploder to Auto-Add"
#------------------------------------------------------------------------------------------------------
def convert_nbib_to_ris_control(qappcurrent, gui, nbib_set_list_dict, n_nbib_sets, converted_nbib_to_ris_auto_add_path):

    if len(nbib_set_list_dict) == 0:
        if DEBUG: print("Error: len(nbib_set_list_dict) == 0:")
        return False

    NBIB_KEYS_MAPPING_DICT,NBIB_SPECIFIC_RIS_TAG_LIST = get_nbib_keys_mapping_dict()
    NBIB_TAG_NAME_DICT = get_nbib_tag_name_dict()

    final_is_valid = True
    msg = None

    n_processed = 0
    n_uniqueness_counter = 0
    n_expected_output_files = len(nbib_set_list_dict)

    output_files_to_write_list = []  #so can use qtimer.singleshot to stagger them so Calibre does not HourGlass/Hang forever if there are a lot of files in the Auto-Add directory to add.

    for unique_name,nbib_set in iteritems(nbib_set_list_dict):    # each k == a nbib_set created in ui.py...
        #unique_name is unique for each output .ris file, each having a single nbib_set.

        if len(nbib_set) == 0:
            if DEBUG: print("empty nbib_set")
            continue

        n_processed = n_processed + 1

        nbib_items_dict = parse_nbib_set_list(NBIB_KEYS_MAPPING_DICT,NBIB_TAG_NAME_DICT,NBIB_TAGS_NOT_OFFICIALLY_MULTIPLE,nbib_set)

        ris_string_list = convert_nbib_items_dict_to_ris(NBIB_KEYS_MAPPING_DICT, nbib_items_dict)

        if len(ris_string_list) == 0:
            if DEBUG: print("ERROR: ris_string_list is empty...")
            continue

        n_uniqueness_counter = n_uniqueness_counter + 1

        unique_namex = str(unique_name) + str(n_uniqueness_counter)

        file_name = unique_namex + ".ris"

        if DEBUG: print("Unique in Auto-Add directory, RIS .ris file name will be: ", file_name)
        output_ris_full_path = os.path.join(converted_nbib_to_ris_auto_add_path,file_name)
        output_ris_full_path = output_ris_full_path.replace(os.sep,"/")
        if DEBUG: print(">>>>>auto-add:  output_ris_full_path: ", output_ris_full_path)
        r = output_ris_full_path,ris_string_list
        output_files_to_write_list.append(r)
    #END FOR

    n_total = len(output_files_to_write_list)

    if n_total != n_expected_output_files:
        msg = "PROGRAM ERROR:  the number of expected output .ris files is: " + str(n_expected_output_files) + " but output_files_to_write_list has only: " + str(n_total) + ".   ABORTING."
        if DEBUG: print(msg)
        final_is_valid = False
        return final_is_valid,msg

    n_total_done = 0
    n = 0
    for r in output_files_to_write_list:
        output_ris_full_path,ris_string_list = r
        n = n + 1
        n_total_done = n_total_done + 1
        is_valid,msg = write_new_ris_file_to_calibre_autoadd_directory(output_ris_full_path,ris_string_list)
        if not is_valid:
            final_is_valid = False
            if DEBUG: print("NOT is_valid; breaking & returning: ", output_ris_full_path, str(ris_string_list))
            break
        if n >= 2:
            n_percent_finished = round(100 * (n_total_done / n_total))
            n = 0
            qappcurrent.processEvents()
            sleep(4)  # assumes Calibre can easily add (without fully hanging the GUI) ~1 new book via auto-add per 2 seconds
            msg = "JS: New RIS 'Books' are being Auto-Added (two at a time): Progress: " + str(n_percent_finished) + "%"
            maingui.status_bar.show_message(msg)
            if DEBUG: print(msg)
            qappcurrent.processEvents()
    #END FOR

    if n_nbib_sets != n_processed:
        if DEBUG: print("n_nbib_sets != n_processed: ", str(n_nbib_sets)," != ", str(n_processed))
        final_is_valid = False
    else:
        if DEBUG: print("n_nbib_sets == n_processed: ", str(n_nbib_sets)," == ", str(n_processed))

    return final_is_valid,msg
#------------------------------------------------------------------------------------------------------
def parse_nbib_set_list(NBIB_KEYS_MAPPING_DICT,NBIB_TAG_NAME_DICT,NBIB_TAGS_NOT_OFFICIALLY_MULTIPLE,nbib_set):

    tag_value_list = []

    for line in nbib_set:
        if line.strip() > "":
            s_split = line.split(KEYVAL_SEPARATOR)
            if len(s_split) == 2:
                nbib_key = s_split[0].strip()
                nbib_value = s_split[1].strip()
                r = nbib_key,nbib_value
                tag_value_list.append(r)
        else:
            if DEBUG: print("skipping line in nbib_set_list: ", line)
        continue
    #END FOR
    del nbib_set

    #~ pre-process to combine identical multiple NBIB Tags so their target long-text custom column has "is_multiple" equivalent functionality just for NBIB.
    #~ However, the Extract RIS Citations (ERC) filetype plugin has "real" RIS Tag functionality for is-multiple Tags such as KW; so, this is only for Non-RIS Tags (new NBIB RIS Tags).

    accumulate_tag_values_dict = {}  # pmid line must *always* be the first line. python3 keeps the original dict order.
    val_list = []

    for r in tag_value_list:
        accumulate_tag = False
        nbib_key,nbib_value = r
        if not nbib_key in nbib_tags_not_officially_multiple:  #e.g. pmid and some others, but certainly not even most.
            if nbib_key in NBIB_KEYS_MAPPING_DICT:  #validate validity of the nbib_key
                accumulate_tag = True
            else:
                msg = "PROGRAM ERROR: UNMAPPED NBIB TAG TO RIS TAG:  " + nbib_key + "<br><br>Communicate this to the Job Spy Plugin Developer. \
                <br><br>In the meantime, the missing NBIB Tag mapping will be defaulted so as to ensure .ris file integrity."
                if DEBUG: print(msg)
                info_dialog(maingui, TOOL_NAME,msg).show()
        else:
            accumulate_tag = False

        if accumulate_tag:
            if not nbib_key in accumulate_tag_values_dict:
                val_list = []
                val_list.append(nbib_value)
                accumulate_tag_values_dict[nbib_key] = val_list
            else:
                val_list = accumulate_tag_values_dict[nbib_key]
                val_list.append(nbib_value)
                accumulate_tag_values_dict[nbib_key] = val_list
            if DEBUG: print("In NBIB_SPECIFIC_RIS_TAG_LIST: ", nbib_key)
        else:  #often an original RIS Tag, not an NBIB-specific one, so let the ERC filetype plugin handle original RIS Tag multiples per its own configuration that is RIS-specific (only).
            val_list = []
            val_list.append(nbib_value)
            accumulate_tag_values_dict[nbib_key] = val_list
            if DEBUG: print("Original RIS Tag: ", nbib_key)
    #END FOR
    del val_list
    del tag_value_list

    tag_value_list = []

    for nbib_key,val_list in iteritems(accumulate_tag_values_dict):
        orig_val_list_string1 = "".join(val_list)        # for ISBN
        orig_val_list_string2 = " & ".join(val_list)  # for ISSN
        if len(val_list) == 1 and nbib_key != "IS":  # ISSN(s) or an ISBN
            nbib_value = val_list[0]
        else:
            if nbib_key != "AU" and nbib_key != "IS":  # Authors are handled natively by ERC for Calibre Authors.
                if nbib_key in NBIB_TAG_NAME_DICT:
                    name = NBIB_TAG_NAME_DICT[nbib_key]
                    name =  name + ": "
                    val = ""
                    for s in val_list:
                        s = s.strip()
                        val = val + name + s.strip() + "  &   "
                    #END FOR
                    nbib_value = val.strip()
                    del val
                    if nbib_value.endswith("  &"):
                        nbib_value = nbib_value[0:-3].strip()
            elif nbib_key == "AU":    # "AU":  # Authors are handled natively by ERC if passed separators of " and ".
                val = ""
                for s in val_list:
                    val = val + s + " and  "
                #END FOR
                val = val.strip()
                val = val[0:-3].strip()
                nbib_value = val
            elif nbib_key == "IS":
                if "978" in orig_val_list_string1 or "979" in orig_val_list_string1:
                    #ISBN
                    nbib_value = orig_val_list_string1
                    if nbib_value.count("97") > 1:
                        nbib_value = nbib_value.replace("97","||97")
                        s_split = nbib_value.split("||")
                        first = s_split[0].strip()
                        if "97" in first:
                            nbib_value = first
                        else:
                            nbib_value = s_split[1].strip()
                else:
                    #ISSN
                    if nbib_key in NBIB_TAG_NAME_DICT:
                        name = NBIB_TAG_NAME_DICT[nbib_key]
                        name =  name + ": "
                    else:
                        name = "issn: "
                    new_value = name + orig_val_list_string2  # 2470-1343 (Electronic)  &  2470-1343 (Linking)
                    r = "GN",new_value.strip()
                    tag_value_list.append(r)
                    if DEBUG: print("final accumulated nbib_value for 'IS':  'GN' created for original NBIB multiple values: ",new_value)
                    if not "-" in nbib_value:
                        nbib_value = nbib_value.replace(" ","-")
                        nbib_value = nbib_value + " "
                    if not "-" in nbib_value:
                        nbib_value = nbib_value[0:4] + "-" + nbib_value[4: ] + " "
                    s_split = nbib_value.split(" ")
                    nbib_value = s_split[0].strip()
                if DEBUG: print("final accumulated nbib_value for isbn/issn tag  'IS': ",nbib_value)
            else:
                if DEBUG: print("ERROR: not AU and not IS??: ", nbib_key)
        r = nbib_key,nbib_value.strip()
        tag_value_list.append(r)
        if DEBUG: print("final accumulated nbib_value: ",nbib_value)
    #END FOR
    del accumulate_tag_values_dict

    nbib_items_dict = {}

    for r in tag_value_list:
        nbib_key,nbib_value = r
        nbib_value = nbib_value.replace("}","")
        nbib_items_dict[nbib_key] = nbib_value
        if DEBUG: print("final: ", r)
    #END FOR
    del tag_value_list

    if DEBUG:
        print("\n\nListing of contents of nbib_items_dict:  nbib_key,nbib_value:  ")
        for k,v in iteritems(nbib_items_dict):
            print(k,v)
        #END FOR
        print("\n\n")

    return nbib_items_dict
#------------------------------------------------------------------------------------------------------
def convert_nbib_items_dict_to_ris(NBIB_KEYS_MAPPING_DICT, nbib_items_dict):

    TAGS_TO_TITLECASE = ['TI','T1','T2','T3','TT','ST','J2','JF','CT','CY']
    TAGS_TO_CHANGE_AND_TO_AMPERSAND = ['AU']
    #~ TAGS_ARE_CALIBRE_TAGS = []

    ris_tag_value_list = []

    special_pubdate_dict = {}

    for nbib_key, nbib_value in iteritems(nbib_items_dict):
        if DEBUG: print(nbib_key, nbib_value)
        if nbib_key in NBIB_KEYS_MAPPING_DICT:
            ris_tag = NBIB_KEYS_MAPPING_DICT[nbib_key]
        else:
            ris_tag = "M1"  # miscellaneous -- default ris_tag to use to default it to Comments
        #ENDIF
        r = ris_tag,nbib_value
        ris_tag_value_list.append(r)
    #END FOR

    ris_tag_value_list.sort()

    ris_string_list = []

    for r in ris_tag_value_list:
        ris_tag,nbib_value = r
        if ris_tag == 'TY':  # must be the very first row...
            s = f'{ris_tag}  - {nbib_value}'    # format is:  XX__-_xxxx; otherwise, considered corrupt .ris file.
            ris_string_list.append(s)
            break
    #END FOR

    for r in ris_tag_value_list:
        ris_tag,nbib_value = r

        if ris_tag == 'TY':
            continue

        if ris_tag in TAGS_TO_CHANGE_AND_TO_AMPERSAND:  #authors, who if have ' and ' must change to the Calibre standard ' & ' separator so each author gets its own Calibre author-id.
            nbib_value = nbib_value.replace(' and ',' & ')

        if ris_tag in TAGS_TO_TITLECASE:
            nbib_value = nbib_value.title()
            if DEBUG: print("ris_tag titlecased: ", ris_tag, nbib_value)

        #~ if ris_tag in TAGS_ARE_CALIBRE_TAGS:
            #~ s_split = nbib_value.split(" & ")
            #~ new_val = ""
            #~ for s in s_split:
                #~ s = s.strip()
                #~ if s > "":
                    #~ new_val = s + ","
            #~ #END FOR
            #~ nbib_value = new_val[0:-1].strip()
            #~ if DEBUG: print("ris_tag: value prepared to be a Calibre Tag: ", ris_tag, nbib_value)

        if ris_tag == "C8":  #Location ID, a.k.a. DOI &/or PII (often both simultaneously)
            #~ Location ID: S0092-8674(22)00787-5 [pii]  &   Location ID: 10.1016/j.cell.2022.06.035 [doi]
            new_nbib_value = None
            if "&" in nbib_value:
                s_split = nbib_value.split("&")
                if len(s_split) >= 2:
                    p1 = s_split[0].strip()
                    p2 = s_split[1].strip()
                    if "doi" in p1:
                        new_nbib_value = p1
                    elif "doi" in p2:
                        new_nbib_value = p2
                    else:
                        pass
                else:
                    p1 = nbib_value
                    if "doi" in p1:
                        new_nbib_value = p1
                    else:
                        pass
            else:
                p1 = nbib_value
                if "doi" in p1:
                    new_nbib_value = p1
                else:
                    pass
            if new_nbib_value is not None:
                new_nbib_value = new_nbib_value.replace("Location ID:","")
                new_nbib_value = new_nbib_value.replace("[doi]","")
                new_nbib_value = new_nbib_value.strip()
                new_ris_tag = "DO"  #DOI    maps to Calibre:  Identifier["doi"] = 10.1016/j.cell.2022.06.03
                s = f'{new_ris_tag}  - {new_nbib_value}'
                ris_string_list.append(s)
            else:
                pass
        s = f'{ris_tag}  - {nbib_value}'
        ris_string_list.append(s)
    #END FOR

    s = 'ER  -'   # very last, and *must* have 2 spaces after ER and before the - ...
    ris_string_list.append(s)

    if DEBUG: print("".join(ris_string_list))

    return ris_string_list
#------------------------------------------------------------------------------------------------------
def write_new_ris_file_to_calibre_autoadd_directory(output_ris_full_path,ris_string_list):
    is_valid = True
    msg = "Not available"

    f = open(output_ris_full_path, 'wt', encoding='utf-8')
    try:
        QTimer.singleShot(0, lambda: ris_write_line_qtimer(f,ris_string_list))
        is_valid = True
    except Exception as e:
        is_valid = False
        msg = "ERROR:   Qtimer.singleshot(0, lambda: ris_write_line_qtimer(f,ris_string_list)) " + str(e)

    return is_valid,msg
#------------------------------------------------------------------------------------------------------
def ris_write_line_qtimer(f,ris_string_list):
    try:
        for line in ris_string_list:
            if line > "":
                line = line + "\n"
                f.write(line)
        #END FOR
    except Exception as e:
        if DEBUG: print("write_new_ris_file_qtimer: Error: ", str(e))

    f.close()
    del f
    return
#------------------------------------------------------------------------------------------------------
def get_nbib_keys_mapping_dict():
    #~ important:  many long-text custom columns support functionally "is_multiple" NBIB Tags via the text format "NBIB Tag Name:Value",
    #~ and ALWAYS APPENDING NEW LINES TO ANY EXISTING LINES for the identical long-text custom column.
    #~ ['C1','people_other','#ris_people_other',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C2','article_other','#ris_article_other',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C3','legal','#ris_legal',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C4','dates_other','#ris_dates_other',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C5','correction_republication','#ris_republication',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C6','miscellany','#ris_miscellany',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C7','grant_contract','#ris_grant_contract',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C8','location_id','#ris_location_id',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C9','other','#ris_other',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['C0','phst_date','#ris_phst_date',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['U0','pmc_id','#ris_pmc_id',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['U1','pmc_release','#ris_pmc_release',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['U2','ec_rn_number','#ris_ec_rn_number',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['U3','subset','#ris_subset',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['U4','composite_source','#ris_composite_source',LONGTEXT_COMMENTS_NORMAL_NOT_HTML],
    #~ ['U5','journal_id','#ris_journal_id',LONGTEXT_COMMENTS_NORMAL_NOT_HTML]]

    #~ Note: 'CP' was previously remapped to PMID for the BIB conversion tool, so it is not NBIB specific, or it would be listed above. It is used by both NBIB and BIB originating from PubMed.

    #~ Reference:  https://www.nlm.nih.gov/bsd/mms/medlineelements.html

    nbib_keys_mapping_dict = {\
                                                'AID':'C2',
                                                'CIN':'C2',
                                                'CON':'C2',
                                                'CP':'C2',
                                                'DDIN':'C2',
                                                'DRIN':'C2',
                                                'EFR':'C2',
                                                'EIN':'C2',
                                                'OID':'C2',
                                                'ORI':'C2',
                                                'PST':'C2',
                                                'RF':'C2',
                                                'SPIN':'C2',
                                                'STAT':'C2',
                                                'UIN':'C2',
                                                'UOF':'C2',
                                                'SO':'U4',
                                                'CRDT':'C4',
                                                'CTDT':'C4',
                                                'DCOM':'C4',
                                                'DEP':'C4',
                                                'DRDT':'C4',
                                                'EDAT':'C4',
                                                'LR':'C4',
                                                'MHDA':'C4',
                                                'RN':'U2',
                                                'GR':'C7',
                                                'JID':'U5',
                                                'CI':'C3',
                                                'COIS':'C3',
                                                'ECF':'C3',
                                                'ECI':'C3',
                                                'OCI':'C3',
                                                'LID':'C8',
                                                'GS':'C6',
                                                'NM':'C6',
                                                'OT':'C6',
                                                'OTO':'C6',
                                                'SFM':'C6',
                                                'OAB':'C9',
                                                'OABL':'C9',
                                                'AD':'C1',
                                                'AUID':'C1',
                                                'CN':'C1',
                                                'FAU':'C1',
                                                'FED':'C1',
                                                'FIR':'C1',
                                                'FPS':'C1',
                                                'IR':'C1',
                                                'IRAD':'C1',
                                                'OWN':'C1',
                                                'PS':'C1',
                                                'PHST':'CO',
                                                'PMC':'U0',
                                                'PMCR':'U1',
                                                'CRF':'C5',
                                                'CRI':'C5',
                                                'RIN':'C5',
                                                'ROF':'C5',
                                                'RPF':'C5',
                                                'RPI':'C5',
                                                'RRF':'C5',
                                                'RRI':'C5',
                                                'SB':'U3',
                                                'AB':'AB',
                                                'AU':'AU',
                                                'BTI':'BT',
                                                'PMID':'CP',
                                                'SI':'DB',
                                                'ED':'ED',
                                                'EN':'ET',
                                                'MID':'ID',
                                                'IP':'IS',
                                                'TA':'JA',
                                                'JT':'JF',
                                                'MH':'KW',   #MH-MeSHTerms become Calibre Tags, so MH values must be split properly at the " & " prior to handing over to Calibre.
                                                'LA':'LA',
                                                'GN':'N1',
                                                'PB':'PB',
                                                'PL':'PP',
                                                'DP':'PY',
                                                'IS':'SN',
                                                'ISBN':'SN',
                                                'PG':'SP',     # PG =  SP (start page) - EP (end page)
                                                'VTI':'T2',
                                                'CTI':'T3',
                                                'TI':'TI',
                                                'TT':'TT',
                                                'PT':'TY',
                                                'VI':'VL'\
                                                }

    nbib_specific_ris_tag_list = []  #only these values will be accumulated into their specific #ris_... long-text custom columns, simulating "is-multiple" in a long-text custom column.
    nbib_specific_ris_tag_list.append("C0")
    nbib_specific_ris_tag_list.append("C1")
    nbib_specific_ris_tag_list.append("C2")
    nbib_specific_ris_tag_list.append("C3")
    nbib_specific_ris_tag_list.append("C4")
    nbib_specific_ris_tag_list.append("C5")
    nbib_specific_ris_tag_list.append("C6")
    nbib_specific_ris_tag_list.append("C7")
    nbib_specific_ris_tag_list.append("C8")
    nbib_specific_ris_tag_list.append("C9")
    nbib_specific_ris_tag_list.append("U1")
    nbib_specific_ris_tag_list.append("U2")
    nbib_specific_ris_tag_list.append("U3")
    nbib_specific_ris_tag_list.append("U4")
    nbib_specific_ris_tag_list.append("U5")

    return nbib_keys_mapping_dict,nbib_specific_ris_tag_list
#------------------------------------------------------------------------------------------------------
def get_nbib_tag_name_dict():
    nbib_tag_name_dict = {\
    'AB':'Abstract',
    'AD':'Affiliation',
    'AID':'Article',
    'AU':'Author',
    'AUID':'Author',
    'BTI':'Book',
    'CI':'Copyright',
    'CIN':'Comment',
    'CN':'Corporate',
    'COI':'Conflict',
    'CON':'Comment',
    'CP':'Chapter',
    'CRDT':'Create',
    'CRF':'Corrected',
    'CRI':'Corrected',
    'CTDT':'Contribution',
    'CTI':'Collection',
    'DCOM':'Completion',
    'DDIN':'Dataset',
    'DRIN':'Dataset',
    'DEP':'Date',
    'DP':'Publication',
    'DRDT':'Date',
    'ECF':'Expression',
    'ECI':'Expression',
    'EDAT':'Entry Date',
    'EFR':'Erratum For',
    'EIN':'Erratum In',
    'ED':'Editor',
    'EN':'Edition',
    'FAU':'Full Author Name',
    'FED':'Full Editor Name',
    'FIR':'Full Investigator Name',
    'FPS':'Full Personal Name',
    'GN':'General Note',
    'GR':'Grant Number',
    'GS':'Gene Symbol',
    'IP':'Issue',
    'IR':'Investigator',
    'IRAD':'Investigator Affiliation',
    'IS':'ISSN',
    'ISBN':'ISBN',
    'JID':'NLM Unique ID',
    'JT':'Full Journal Title',
    'LA':'Language',
    'LID':'Location ID',
    'LR':'Modification Date',
    'MH':'MeSH Terms',
    'MHDA':'MeSH Date',
    'MID':'Manuscript Identifier',
    'NM':'Substance Name',
    'OAB':'Other Abstract',
    'OABL':'Other Abstract Language',
    'OCI':'Other Copyright Information',
    'OID':'Other ID Identification',
    'ORI':'Original Report In',
    'OT':'Other Term',
    'OTO':'Other Term Owner',
    'OWN':'Owner',
    'PB':'Publisher',
    'PG':'Pagination',
    'PHST':'Publication History Status Date',
    'PL':'Place of Publication',
    'PMC':'PubMed Central Identifier',
    'PMCR':'PMC Release',
    'PMID':'PubMed Unique Identifier',
    'PS':'Personal Name as Subject',
    'PST':'Publication Status',
    'PT':'Publication Type',
    'RF':'Number of References',
    'RIN':'Retraction In',
    'RN':'EC/RN Number',
    'ROF':'Retraction Of',
    'RPF':'Republished From',
    'RPI':'Republished In',
    'RRI':'Retracted and Republished In',
    'RRF':'Retracted and Republished From',
    'SB':'Subset',
    'SFM':'Space Flight Mission',
    'SI':'Secondary Source ID',
    'SO':'Source',
    'SPIN':'Summary For Patients In',
    'STAT':'Status',
    'TA':'Journal Title',
    'TI':'Title',
    'TT':'Transliterated Title',
    'UIN':'Update In',
    'UOF':'Update Of',
    'VI':'Volume',
    'VTI':'Volume Title'\
    }
    return nbib_tag_name_dict
#------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------